/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.workspaces.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a directory that is used with Amazon WorkSpaces.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WorkspaceDirectory implements SdkPojo, Serializable,
        ToCopyableBuilder<WorkspaceDirectory.Builder, WorkspaceDirectory> {
    private static final SdkField<String> DIRECTORY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkspaceDirectory::directoryId)).setter(setter(Builder::directoryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DirectoryId").build()).build();

    private static final SdkField<String> ALIAS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkspaceDirectory::alias)).setter(setter(Builder::alias))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Alias").build()).build();

    private static final SdkField<String> DIRECTORY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkspaceDirectory::directoryName)).setter(setter(Builder::directoryName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DirectoryName").build()).build();

    private static final SdkField<String> REGISTRATION_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkspaceDirectory::registrationCode)).setter(setter(Builder::registrationCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RegistrationCode").build()).build();

    private static final SdkField<List<String>> SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(WorkspaceDirectory::subnetIds))
            .setter(setter(Builder::subnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> DNS_IP_ADDRESSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(WorkspaceDirectory::dnsIpAddresses))
            .setter(setter(Builder::dnsIpAddresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsIpAddresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CUSTOMER_USER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkspaceDirectory::customerUserName)).setter(setter(Builder::customerUserName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerUserName").build()).build();

    private static final SdkField<String> IAM_ROLE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkspaceDirectory::iamRoleId)).setter(setter(Builder::iamRoleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamRoleId").build()).build();

    private static final SdkField<String> DIRECTORY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkspaceDirectory::directoryTypeAsString)).setter(setter(Builder::directoryType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DirectoryType").build()).build();

    private static final SdkField<String> WORKSPACE_SECURITY_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkspaceDirectory::workspaceSecurityGroupId)).setter(setter(Builder::workspaceSecurityGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkspaceSecurityGroupId").build())
            .build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkspaceDirectory::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<DefaultWorkspaceCreationProperties> WORKSPACE_CREATION_PROPERTIES_FIELD = SdkField
            .<DefaultWorkspaceCreationProperties> builder(MarshallingType.SDK_POJO)
            .getter(getter(WorkspaceDirectory::workspaceCreationProperties))
            .setter(setter(Builder::workspaceCreationProperties))
            .constructor(DefaultWorkspaceCreationProperties::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkspaceCreationProperties")
                    .build()).build();

    private static final SdkField<List<String>> IP_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(WorkspaceDirectory::ipGroupIds))
            .setter(setter(Builder::ipGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ipGroupIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<WorkspaceAccessProperties> WORKSPACE_ACCESS_PROPERTIES_FIELD = SdkField
            .<WorkspaceAccessProperties> builder(MarshallingType.SDK_POJO)
            .getter(getter(WorkspaceDirectory::workspaceAccessProperties)).setter(setter(Builder::workspaceAccessProperties))
            .constructor(WorkspaceAccessProperties::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkspaceAccessProperties").build())
            .build();

    private static final SdkField<String> TENANCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(WorkspaceDirectory::tenancyAsString)).setter(setter(Builder::tenancy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tenancy").build()).build();

    private static final SdkField<SelfservicePermissions> SELFSERVICE_PERMISSIONS_FIELD = SdkField
            .<SelfservicePermissions> builder(MarshallingType.SDK_POJO)
            .getter(getter(WorkspaceDirectory::selfservicePermissions)).setter(setter(Builder::selfservicePermissions))
            .constructor(SelfservicePermissions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SelfservicePermissions").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DIRECTORY_ID_FIELD,
            ALIAS_FIELD, DIRECTORY_NAME_FIELD, REGISTRATION_CODE_FIELD, SUBNET_IDS_FIELD, DNS_IP_ADDRESSES_FIELD,
            CUSTOMER_USER_NAME_FIELD, IAM_ROLE_ID_FIELD, DIRECTORY_TYPE_FIELD, WORKSPACE_SECURITY_GROUP_ID_FIELD, STATE_FIELD,
            WORKSPACE_CREATION_PROPERTIES_FIELD, IP_GROUP_IDS_FIELD, WORKSPACE_ACCESS_PROPERTIES_FIELD, TENANCY_FIELD,
            SELFSERVICE_PERMISSIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String directoryId;

    private final String alias;

    private final String directoryName;

    private final String registrationCode;

    private final List<String> subnetIds;

    private final List<String> dnsIpAddresses;

    private final String customerUserName;

    private final String iamRoleId;

    private final String directoryType;

    private final String workspaceSecurityGroupId;

    private final String state;

    private final DefaultWorkspaceCreationProperties workspaceCreationProperties;

    private final List<String> ipGroupIds;

    private final WorkspaceAccessProperties workspaceAccessProperties;

    private final String tenancy;

    private final SelfservicePermissions selfservicePermissions;

    private WorkspaceDirectory(BuilderImpl builder) {
        this.directoryId = builder.directoryId;
        this.alias = builder.alias;
        this.directoryName = builder.directoryName;
        this.registrationCode = builder.registrationCode;
        this.subnetIds = builder.subnetIds;
        this.dnsIpAddresses = builder.dnsIpAddresses;
        this.customerUserName = builder.customerUserName;
        this.iamRoleId = builder.iamRoleId;
        this.directoryType = builder.directoryType;
        this.workspaceSecurityGroupId = builder.workspaceSecurityGroupId;
        this.state = builder.state;
        this.workspaceCreationProperties = builder.workspaceCreationProperties;
        this.ipGroupIds = builder.ipGroupIds;
        this.workspaceAccessProperties = builder.workspaceAccessProperties;
        this.tenancy = builder.tenancy;
        this.selfservicePermissions = builder.selfservicePermissions;
    }

    /**
     * <p>
     * The directory identifier.
     * </p>
     * 
     * @return The directory identifier.
     */
    public String directoryId() {
        return directoryId;
    }

    /**
     * <p>
     * The directory alias.
     * </p>
     * 
     * @return The directory alias.
     */
    public String alias() {
        return alias;
    }

    /**
     * <p>
     * The name of the directory.
     * </p>
     * 
     * @return The name of the directory.
     */
    public String directoryName() {
        return directoryName;
    }

    /**
     * <p>
     * The registration code for the directory. This is the code that users enter in their Amazon WorkSpaces client
     * application to connect to the directory.
     * </p>
     * 
     * @return The registration code for the directory. This is the code that users enter in their Amazon WorkSpaces
     *         client application to connect to the directory.
     */
    public String registrationCode() {
        return registrationCode;
    }

    /**
     * Returns true if the SubnetIds property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasSubnetIds() {
        return subnetIds != null && !(subnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The identifiers of the subnets used with the directory.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSubnetIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The identifiers of the subnets used with the directory.
     */
    public List<String> subnetIds() {
        return subnetIds;
    }

    /**
     * Returns true if the DnsIpAddresses property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDnsIpAddresses() {
        return dnsIpAddresses != null && !(dnsIpAddresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IP addresses of the DNS servers for the directory.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDnsIpAddresses()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The IP addresses of the DNS servers for the directory.
     */
    public List<String> dnsIpAddresses() {
        return dnsIpAddresses;
    }

    /**
     * <p>
     * The user name for the service account.
     * </p>
     * 
     * @return The user name for the service account.
     */
    public String customerUserName() {
        return customerUserName;
    }

    /**
     * <p>
     * The identifier of the IAM role. This is the role that allows Amazon WorkSpaces to make calls to other services,
     * such as Amazon EC2, on your behalf.
     * </p>
     * 
     * @return The identifier of the IAM role. This is the role that allows Amazon WorkSpaces to make calls to other
     *         services, such as Amazon EC2, on your behalf.
     */
    public String iamRoleId() {
        return iamRoleId;
    }

    /**
     * <p>
     * The directory type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #directoryType}
     * will return {@link WorkspaceDirectoryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #directoryTypeAsString}.
     * </p>
     * 
     * @return The directory type.
     * @see WorkspaceDirectoryType
     */
    public WorkspaceDirectoryType directoryType() {
        return WorkspaceDirectoryType.fromValue(directoryType);
    }

    /**
     * <p>
     * The directory type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #directoryType}
     * will return {@link WorkspaceDirectoryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #directoryTypeAsString}.
     * </p>
     * 
     * @return The directory type.
     * @see WorkspaceDirectoryType
     */
    public String directoryTypeAsString() {
        return directoryType;
    }

    /**
     * <p>
     * The identifier of the security group that is assigned to new WorkSpaces.
     * </p>
     * 
     * @return The identifier of the security group that is assigned to new WorkSpaces.
     */
    public String workspaceSecurityGroupId() {
        return workspaceSecurityGroupId;
    }

    /**
     * <p>
     * The state of the directory's registration with Amazon WorkSpaces.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link WorkspaceDirectoryState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the directory's registration with Amazon WorkSpaces.
     * @see WorkspaceDirectoryState
     */
    public WorkspaceDirectoryState state() {
        return WorkspaceDirectoryState.fromValue(state);
    }

    /**
     * <p>
     * The state of the directory's registration with Amazon WorkSpaces.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link WorkspaceDirectoryState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the directory's registration with Amazon WorkSpaces.
     * @see WorkspaceDirectoryState
     */
    public String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The default creation properties for all WorkSpaces in the directory.
     * </p>
     * 
     * @return The default creation properties for all WorkSpaces in the directory.
     */
    public DefaultWorkspaceCreationProperties workspaceCreationProperties() {
        return workspaceCreationProperties;
    }

    /**
     * Returns true if the IpGroupIds property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasIpGroupIds() {
        return ipGroupIds != null && !(ipGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The identifiers of the IP access control groups associated with the directory.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasIpGroupIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The identifiers of the IP access control groups associated with the directory.
     */
    public List<String> ipGroupIds() {
        return ipGroupIds;
    }

    /**
     * <p>
     * The devices and operating systems that users can use to access WorkSpaces.
     * </p>
     * 
     * @return The devices and operating systems that users can use to access WorkSpaces.
     */
    public WorkspaceAccessProperties workspaceAccessProperties() {
        return workspaceAccessProperties;
    }

    /**
     * <p>
     * Specifies whether the directory is dedicated or shared. To use Bring Your Own License (BYOL), this value must be
     * set to <code>DEDICATED</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your Own Windows
     * Desktop Images</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tenancy} will
     * return {@link Tenancy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tenancyAsString}.
     * </p>
     * 
     * @return Specifies whether the directory is dedicated or shared. To use Bring Your Own License (BYOL), this value
     *         must be set to <code>DEDICATED</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your Own
     *         Windows Desktop Images</a>.
     * @see Tenancy
     */
    public Tenancy tenancy() {
        return Tenancy.fromValue(tenancy);
    }

    /**
     * <p>
     * Specifies whether the directory is dedicated or shared. To use Bring Your Own License (BYOL), this value must be
     * set to <code>DEDICATED</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your Own Windows
     * Desktop Images</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tenancy} will
     * return {@link Tenancy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tenancyAsString}.
     * </p>
     * 
     * @return Specifies whether the directory is dedicated or shared. To use Bring Your Own License (BYOL), this value
     *         must be set to <code>DEDICATED</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your Own
     *         Windows Desktop Images</a>.
     * @see Tenancy
     */
    public String tenancyAsString() {
        return tenancy;
    }

    /**
     * <p>
     * The default self-service permissions for WorkSpaces in the directory.
     * </p>
     * 
     * @return The default self-service permissions for WorkSpaces in the directory.
     */
    public SelfservicePermissions selfservicePermissions() {
        return selfservicePermissions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(directoryId());
        hashCode = 31 * hashCode + Objects.hashCode(alias());
        hashCode = 31 * hashCode + Objects.hashCode(directoryName());
        hashCode = 31 * hashCode + Objects.hashCode(registrationCode());
        hashCode = 31 * hashCode + Objects.hashCode(subnetIds());
        hashCode = 31 * hashCode + Objects.hashCode(dnsIpAddresses());
        hashCode = 31 * hashCode + Objects.hashCode(customerUserName());
        hashCode = 31 * hashCode + Objects.hashCode(iamRoleId());
        hashCode = 31 * hashCode + Objects.hashCode(directoryTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(workspaceSecurityGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(workspaceCreationProperties());
        hashCode = 31 * hashCode + Objects.hashCode(ipGroupIds());
        hashCode = 31 * hashCode + Objects.hashCode(workspaceAccessProperties());
        hashCode = 31 * hashCode + Objects.hashCode(tenancyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(selfservicePermissions());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WorkspaceDirectory)) {
            return false;
        }
        WorkspaceDirectory other = (WorkspaceDirectory) obj;
        return Objects.equals(directoryId(), other.directoryId()) && Objects.equals(alias(), other.alias())
                && Objects.equals(directoryName(), other.directoryName())
                && Objects.equals(registrationCode(), other.registrationCode()) && Objects.equals(subnetIds(), other.subnetIds())
                && Objects.equals(dnsIpAddresses(), other.dnsIpAddresses())
                && Objects.equals(customerUserName(), other.customerUserName()) && Objects.equals(iamRoleId(), other.iamRoleId())
                && Objects.equals(directoryTypeAsString(), other.directoryTypeAsString())
                && Objects.equals(workspaceSecurityGroupId(), other.workspaceSecurityGroupId())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(workspaceCreationProperties(), other.workspaceCreationProperties())
                && Objects.equals(ipGroupIds(), other.ipGroupIds())
                && Objects.equals(workspaceAccessProperties(), other.workspaceAccessProperties())
                && Objects.equals(tenancyAsString(), other.tenancyAsString())
                && Objects.equals(selfservicePermissions(), other.selfservicePermissions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("WorkspaceDirectory").add("DirectoryId", directoryId()).add("Alias", alias())
                .add("DirectoryName", directoryName()).add("RegistrationCode", registrationCode()).add("SubnetIds", subnetIds())
                .add("DnsIpAddresses", dnsIpAddresses()).add("CustomerUserName", customerUserName())
                .add("IamRoleId", iamRoleId()).add("DirectoryType", directoryTypeAsString())
                .add("WorkspaceSecurityGroupId", workspaceSecurityGroupId()).add("State", stateAsString())
                .add("WorkspaceCreationProperties", workspaceCreationProperties()).add("IpGroupIds", ipGroupIds())
                .add("WorkspaceAccessProperties", workspaceAccessProperties()).add("Tenancy", tenancyAsString())
                .add("SelfservicePermissions", selfservicePermissions()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DirectoryId":
            return Optional.ofNullable(clazz.cast(directoryId()));
        case "Alias":
            return Optional.ofNullable(clazz.cast(alias()));
        case "DirectoryName":
            return Optional.ofNullable(clazz.cast(directoryName()));
        case "RegistrationCode":
            return Optional.ofNullable(clazz.cast(registrationCode()));
        case "SubnetIds":
            return Optional.ofNullable(clazz.cast(subnetIds()));
        case "DnsIpAddresses":
            return Optional.ofNullable(clazz.cast(dnsIpAddresses()));
        case "CustomerUserName":
            return Optional.ofNullable(clazz.cast(customerUserName()));
        case "IamRoleId":
            return Optional.ofNullable(clazz.cast(iamRoleId()));
        case "DirectoryType":
            return Optional.ofNullable(clazz.cast(directoryTypeAsString()));
        case "WorkspaceSecurityGroupId":
            return Optional.ofNullable(clazz.cast(workspaceSecurityGroupId()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "WorkspaceCreationProperties":
            return Optional.ofNullable(clazz.cast(workspaceCreationProperties()));
        case "ipGroupIds":
            return Optional.ofNullable(clazz.cast(ipGroupIds()));
        case "WorkspaceAccessProperties":
            return Optional.ofNullable(clazz.cast(workspaceAccessProperties()));
        case "Tenancy":
            return Optional.ofNullable(clazz.cast(tenancyAsString()));
        case "SelfservicePermissions":
            return Optional.ofNullable(clazz.cast(selfservicePermissions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WorkspaceDirectory, T> g) {
        return obj -> g.apply((WorkspaceDirectory) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WorkspaceDirectory> {
        /**
         * <p>
         * The directory identifier.
         * </p>
         * 
         * @param directoryId
         *        The directory identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder directoryId(String directoryId);

        /**
         * <p>
         * The directory alias.
         * </p>
         * 
         * @param alias
         *        The directory alias.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alias(String alias);

        /**
         * <p>
         * The name of the directory.
         * </p>
         * 
         * @param directoryName
         *        The name of the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder directoryName(String directoryName);

        /**
         * <p>
         * The registration code for the directory. This is the code that users enter in their Amazon WorkSpaces client
         * application to connect to the directory.
         * </p>
         * 
         * @param registrationCode
         *        The registration code for the directory. This is the code that users enter in their Amazon WorkSpaces
         *        client application to connect to the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder registrationCode(String registrationCode);

        /**
         * <p>
         * The identifiers of the subnets used with the directory.
         * </p>
         * 
         * @param subnetIds
         *        The identifiers of the subnets used with the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(Collection<String> subnetIds);

        /**
         * <p>
         * The identifiers of the subnets used with the directory.
         * </p>
         * 
         * @param subnetIds
         *        The identifiers of the subnets used with the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(String... subnetIds);

        /**
         * <p>
         * The IP addresses of the DNS servers for the directory.
         * </p>
         * 
         * @param dnsIpAddresses
         *        The IP addresses of the DNS servers for the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpAddresses(Collection<String> dnsIpAddresses);

        /**
         * <p>
         * The IP addresses of the DNS servers for the directory.
         * </p>
         * 
         * @param dnsIpAddresses
         *        The IP addresses of the DNS servers for the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpAddresses(String... dnsIpAddresses);

        /**
         * <p>
         * The user name for the service account.
         * </p>
         * 
         * @param customerUserName
         *        The user name for the service account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerUserName(String customerUserName);

        /**
         * <p>
         * The identifier of the IAM role. This is the role that allows Amazon WorkSpaces to make calls to other
         * services, such as Amazon EC2, on your behalf.
         * </p>
         * 
         * @param iamRoleId
         *        The identifier of the IAM role. This is the role that allows Amazon WorkSpaces to make calls to other
         *        services, such as Amazon EC2, on your behalf.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamRoleId(String iamRoleId);

        /**
         * <p>
         * The directory type.
         * </p>
         * 
         * @param directoryType
         *        The directory type.
         * @see WorkspaceDirectoryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkspaceDirectoryType
         */
        Builder directoryType(String directoryType);

        /**
         * <p>
         * The directory type.
         * </p>
         * 
         * @param directoryType
         *        The directory type.
         * @see WorkspaceDirectoryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkspaceDirectoryType
         */
        Builder directoryType(WorkspaceDirectoryType directoryType);

        /**
         * <p>
         * The identifier of the security group that is assigned to new WorkSpaces.
         * </p>
         * 
         * @param workspaceSecurityGroupId
         *        The identifier of the security group that is assigned to new WorkSpaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workspaceSecurityGroupId(String workspaceSecurityGroupId);

        /**
         * <p>
         * The state of the directory's registration with Amazon WorkSpaces.
         * </p>
         * 
         * @param state
         *        The state of the directory's registration with Amazon WorkSpaces.
         * @see WorkspaceDirectoryState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkspaceDirectoryState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the directory's registration with Amazon WorkSpaces.
         * </p>
         * 
         * @param state
         *        The state of the directory's registration with Amazon WorkSpaces.
         * @see WorkspaceDirectoryState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkspaceDirectoryState
         */
        Builder state(WorkspaceDirectoryState state);

        /**
         * <p>
         * The default creation properties for all WorkSpaces in the directory.
         * </p>
         * 
         * @param workspaceCreationProperties
         *        The default creation properties for all WorkSpaces in the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workspaceCreationProperties(DefaultWorkspaceCreationProperties workspaceCreationProperties);

        /**
         * <p>
         * The default creation properties for all WorkSpaces in the directory.
         * </p>
         * This is a convenience that creates an instance of the {@link DefaultWorkspaceCreationProperties.Builder}
         * avoiding the need to create one manually via {@link DefaultWorkspaceCreationProperties#builder()}.
         *
         * When the {@link Consumer} completes, {@link DefaultWorkspaceCreationProperties.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #workspaceCreationProperties(DefaultWorkspaceCreationProperties)}.
         * 
         * @param workspaceCreationProperties
         *        a consumer that will call methods on {@link DefaultWorkspaceCreationProperties.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #workspaceCreationProperties(DefaultWorkspaceCreationProperties)
         */
        default Builder workspaceCreationProperties(
                Consumer<DefaultWorkspaceCreationProperties.Builder> workspaceCreationProperties) {
            return workspaceCreationProperties(DefaultWorkspaceCreationProperties.builder()
                    .applyMutation(workspaceCreationProperties).build());
        }

        /**
         * <p>
         * The identifiers of the IP access control groups associated with the directory.
         * </p>
         * 
         * @param ipGroupIds
         *        The identifiers of the IP access control groups associated with the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipGroupIds(Collection<String> ipGroupIds);

        /**
         * <p>
         * The identifiers of the IP access control groups associated with the directory.
         * </p>
         * 
         * @param ipGroupIds
         *        The identifiers of the IP access control groups associated with the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipGroupIds(String... ipGroupIds);

        /**
         * <p>
         * The devices and operating systems that users can use to access WorkSpaces.
         * </p>
         * 
         * @param workspaceAccessProperties
         *        The devices and operating systems that users can use to access WorkSpaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workspaceAccessProperties(WorkspaceAccessProperties workspaceAccessProperties);

        /**
         * <p>
         * The devices and operating systems that users can use to access WorkSpaces.
         * </p>
         * This is a convenience that creates an instance of the {@link WorkspaceAccessProperties.Builder} avoiding the
         * need to create one manually via {@link WorkspaceAccessProperties#builder()}.
         *
         * When the {@link Consumer} completes, {@link WorkspaceAccessProperties.Builder#build()} is called immediately
         * and its result is passed to {@link #workspaceAccessProperties(WorkspaceAccessProperties)}.
         * 
         * @param workspaceAccessProperties
         *        a consumer that will call methods on {@link WorkspaceAccessProperties.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #workspaceAccessProperties(WorkspaceAccessProperties)
         */
        default Builder workspaceAccessProperties(Consumer<WorkspaceAccessProperties.Builder> workspaceAccessProperties) {
            return workspaceAccessProperties(WorkspaceAccessProperties.builder().applyMutation(workspaceAccessProperties).build());
        }

        /**
         * <p>
         * Specifies whether the directory is dedicated or shared. To use Bring Your Own License (BYOL), this value must
         * be set to <code>DEDICATED</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your Own
         * Windows Desktop Images</a>.
         * </p>
         * 
         * @param tenancy
         *        Specifies whether the directory is dedicated or shared. To use Bring Your Own License (BYOL), this
         *        value must be set to <code>DEDICATED</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your
         *        Own Windows Desktop Images</a>.
         * @see Tenancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Tenancy
         */
        Builder tenancy(String tenancy);

        /**
         * <p>
         * Specifies whether the directory is dedicated or shared. To use Bring Your Own License (BYOL), this value must
         * be set to <code>DEDICATED</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your Own
         * Windows Desktop Images</a>.
         * </p>
         * 
         * @param tenancy
         *        Specifies whether the directory is dedicated or shared. To use Bring Your Own License (BYOL), this
         *        value must be set to <code>DEDICATED</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/workspaces/latest/adminguide/byol-windows-images.html">Bring Your
         *        Own Windows Desktop Images</a>.
         * @see Tenancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Tenancy
         */
        Builder tenancy(Tenancy tenancy);

        /**
         * <p>
         * The default self-service permissions for WorkSpaces in the directory.
         * </p>
         * 
         * @param selfservicePermissions
         *        The default self-service permissions for WorkSpaces in the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selfservicePermissions(SelfservicePermissions selfservicePermissions);

        /**
         * <p>
         * The default self-service permissions for WorkSpaces in the directory.
         * </p>
         * This is a convenience that creates an instance of the {@link SelfservicePermissions.Builder} avoiding the
         * need to create one manually via {@link SelfservicePermissions#builder()}.
         *
         * When the {@link Consumer} completes, {@link SelfservicePermissions.Builder#build()} is called immediately and
         * its result is passed to {@link #selfservicePermissions(SelfservicePermissions)}.
         * 
         * @param selfservicePermissions
         *        a consumer that will call methods on {@link SelfservicePermissions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #selfservicePermissions(SelfservicePermissions)
         */
        default Builder selfservicePermissions(Consumer<SelfservicePermissions.Builder> selfservicePermissions) {
            return selfservicePermissions(SelfservicePermissions.builder().applyMutation(selfservicePermissions).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String directoryId;

        private String alias;

        private String directoryName;

        private String registrationCode;

        private List<String> subnetIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> dnsIpAddresses = DefaultSdkAutoConstructList.getInstance();

        private String customerUserName;

        private String iamRoleId;

        private String directoryType;

        private String workspaceSecurityGroupId;

        private String state;

        private DefaultWorkspaceCreationProperties workspaceCreationProperties;

        private List<String> ipGroupIds = DefaultSdkAutoConstructList.getInstance();

        private WorkspaceAccessProperties workspaceAccessProperties;

        private String tenancy;

        private SelfservicePermissions selfservicePermissions;

        private BuilderImpl() {
        }

        private BuilderImpl(WorkspaceDirectory model) {
            directoryId(model.directoryId);
            alias(model.alias);
            directoryName(model.directoryName);
            registrationCode(model.registrationCode);
            subnetIds(model.subnetIds);
            dnsIpAddresses(model.dnsIpAddresses);
            customerUserName(model.customerUserName);
            iamRoleId(model.iamRoleId);
            directoryType(model.directoryType);
            workspaceSecurityGroupId(model.workspaceSecurityGroupId);
            state(model.state);
            workspaceCreationProperties(model.workspaceCreationProperties);
            ipGroupIds(model.ipGroupIds);
            workspaceAccessProperties(model.workspaceAccessProperties);
            tenancy(model.tenancy);
            selfservicePermissions(model.selfservicePermissions);
        }

        public final String getDirectoryId() {
            return directoryId;
        }

        @Override
        public final Builder directoryId(String directoryId) {
            this.directoryId = directoryId;
            return this;
        }

        public final void setDirectoryId(String directoryId) {
            this.directoryId = directoryId;
        }

        public final String getAlias() {
            return alias;
        }

        @Override
        public final Builder alias(String alias) {
            this.alias = alias;
            return this;
        }

        public final void setAlias(String alias) {
            this.alias = alias;
        }

        public final String getDirectoryName() {
            return directoryName;
        }

        @Override
        public final Builder directoryName(String directoryName) {
            this.directoryName = directoryName;
            return this;
        }

        public final void setDirectoryName(String directoryName) {
            this.directoryName = directoryName;
        }

        public final String getRegistrationCode() {
            return registrationCode;
        }

        @Override
        public final Builder registrationCode(String registrationCode) {
            this.registrationCode = registrationCode;
            return this;
        }

        public final void setRegistrationCode(String registrationCode) {
            this.registrationCode = registrationCode;
        }

        public final Collection<String> getSubnetIds() {
            return subnetIds;
        }

        @Override
        public final Builder subnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetIds(String... subnetIds) {
            subnetIds(Arrays.asList(subnetIds));
            return this;
        }

        public final void setSubnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
        }

        public final Collection<String> getDnsIpAddresses() {
            return dnsIpAddresses;
        }

        @Override
        public final Builder dnsIpAddresses(Collection<String> dnsIpAddresses) {
            this.dnsIpAddresses = DnsIpAddressesCopier.copy(dnsIpAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsIpAddresses(String... dnsIpAddresses) {
            dnsIpAddresses(Arrays.asList(dnsIpAddresses));
            return this;
        }

        public final void setDnsIpAddresses(Collection<String> dnsIpAddresses) {
            this.dnsIpAddresses = DnsIpAddressesCopier.copy(dnsIpAddresses);
        }

        public final String getCustomerUserName() {
            return customerUserName;
        }

        @Override
        public final Builder customerUserName(String customerUserName) {
            this.customerUserName = customerUserName;
            return this;
        }

        public final void setCustomerUserName(String customerUserName) {
            this.customerUserName = customerUserName;
        }

        public final String getIamRoleId() {
            return iamRoleId;
        }

        @Override
        public final Builder iamRoleId(String iamRoleId) {
            this.iamRoleId = iamRoleId;
            return this;
        }

        public final void setIamRoleId(String iamRoleId) {
            this.iamRoleId = iamRoleId;
        }

        public final String getDirectoryTypeAsString() {
            return directoryType;
        }

        @Override
        public final Builder directoryType(String directoryType) {
            this.directoryType = directoryType;
            return this;
        }

        @Override
        public final Builder directoryType(WorkspaceDirectoryType directoryType) {
            this.directoryType(directoryType == null ? null : directoryType.toString());
            return this;
        }

        public final void setDirectoryType(String directoryType) {
            this.directoryType = directoryType;
        }

        public final String getWorkspaceSecurityGroupId() {
            return workspaceSecurityGroupId;
        }

        @Override
        public final Builder workspaceSecurityGroupId(String workspaceSecurityGroupId) {
            this.workspaceSecurityGroupId = workspaceSecurityGroupId;
            return this;
        }

        public final void setWorkspaceSecurityGroupId(String workspaceSecurityGroupId) {
            this.workspaceSecurityGroupId = workspaceSecurityGroupId;
        }

        public final String getStateAsString() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(WorkspaceDirectoryState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final DefaultWorkspaceCreationProperties.Builder getWorkspaceCreationProperties() {
            return workspaceCreationProperties != null ? workspaceCreationProperties.toBuilder() : null;
        }

        @Override
        public final Builder workspaceCreationProperties(DefaultWorkspaceCreationProperties workspaceCreationProperties) {
            this.workspaceCreationProperties = workspaceCreationProperties;
            return this;
        }

        public final void setWorkspaceCreationProperties(
                DefaultWorkspaceCreationProperties.BuilderImpl workspaceCreationProperties) {
            this.workspaceCreationProperties = workspaceCreationProperties != null ? workspaceCreationProperties.build() : null;
        }

        public final Collection<String> getIpGroupIds() {
            return ipGroupIds;
        }

        @Override
        public final Builder ipGroupIds(Collection<String> ipGroupIds) {
            this.ipGroupIds = IpGroupIdListCopier.copy(ipGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipGroupIds(String... ipGroupIds) {
            ipGroupIds(Arrays.asList(ipGroupIds));
            return this;
        }

        public final void setIpGroupIds(Collection<String> ipGroupIds) {
            this.ipGroupIds = IpGroupIdListCopier.copy(ipGroupIds);
        }

        public final WorkspaceAccessProperties.Builder getWorkspaceAccessProperties() {
            return workspaceAccessProperties != null ? workspaceAccessProperties.toBuilder() : null;
        }

        @Override
        public final Builder workspaceAccessProperties(WorkspaceAccessProperties workspaceAccessProperties) {
            this.workspaceAccessProperties = workspaceAccessProperties;
            return this;
        }

        public final void setWorkspaceAccessProperties(WorkspaceAccessProperties.BuilderImpl workspaceAccessProperties) {
            this.workspaceAccessProperties = workspaceAccessProperties != null ? workspaceAccessProperties.build() : null;
        }

        public final String getTenancyAsString() {
            return tenancy;
        }

        @Override
        public final Builder tenancy(String tenancy) {
            this.tenancy = tenancy;
            return this;
        }

        @Override
        public final Builder tenancy(Tenancy tenancy) {
            this.tenancy(tenancy == null ? null : tenancy.toString());
            return this;
        }

        public final void setTenancy(String tenancy) {
            this.tenancy = tenancy;
        }

        public final SelfservicePermissions.Builder getSelfservicePermissions() {
            return selfservicePermissions != null ? selfservicePermissions.toBuilder() : null;
        }

        @Override
        public final Builder selfservicePermissions(SelfservicePermissions selfservicePermissions) {
            this.selfservicePermissions = selfservicePermissions;
            return this;
        }

        public final void setSelfservicePermissions(SelfservicePermissions.BuilderImpl selfservicePermissions) {
            this.selfservicePermissions = selfservicePermissions != null ? selfservicePermissions.build() : null;
        }

        @Override
        public WorkspaceDirectory build() {
            return new WorkspaceDirectory(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
