/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.workmailmessageflow;

import java.nio.file.Path;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.services.workmailmessageflow.model.GetRawMessageContentRequest;
import software.amazon.awssdk.services.workmailmessageflow.model.GetRawMessageContentResponse;
import software.amazon.awssdk.services.workmailmessageflow.model.PutRawMessageContentRequest;
import software.amazon.awssdk.services.workmailmessageflow.model.PutRawMessageContentResponse;

/**
 * Service client for accessing Amazon WorkMail Message Flow asynchronously. This can be created using the static
 * {@link #builder()} method.
 *
 * <p>
 * The WorkMail Message Flow API provides access to email messages as they are being sent and received by a WorkMail
 * organization.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public interface WorkMailMessageFlowAsyncClient extends SdkClient {
    String SERVICE_NAME = "workmailmessageflow";

    /**
     * Create a {@link WorkMailMessageFlowAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static WorkMailMessageFlowAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link WorkMailMessageFlowAsyncClient}.
     */
    static WorkMailMessageFlowAsyncClientBuilder builder() {
        return new DefaultWorkMailMessageFlowAsyncClientBuilder();
    }

    /**
     * <p>
     * Retrieves the raw content of an in-transit email message, in MIME format.
     * </p>
     *
     * @param getRawMessageContentRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The raw content of the email message, in MIME format.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested email message is not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkMailMessageFlowException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample WorkMailMessageFlowAsyncClient.GetRawMessageContent
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workmailmessageflow-2019-05-01/GetRawMessageContent"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> getRawMessageContent(GetRawMessageContentRequest getRawMessageContentRequest,
            AsyncResponseTransformer<GetRawMessageContentResponse, ReturnT> asyncResponseTransformer) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the raw content of an in-transit email message, in MIME format.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetRawMessageContentRequest.Builder} avoiding the
     * need to create one manually via {@link GetRawMessageContentRequest#builder()}
     * </p>
     *
     * @param getRawMessageContentRequest
     *        A {@link Consumer} that will call methods on {@link GetRawMessageContentRequest.Builder} to create a
     *        request.
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The raw content of the email message, in MIME format.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested email message is not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkMailMessageFlowException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample WorkMailMessageFlowAsyncClient.GetRawMessageContent
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workmailmessageflow-2019-05-01/GetRawMessageContent"
     *      target="_top">AWS API Documentation</a>
     */
    default <ReturnT> CompletableFuture<ReturnT> getRawMessageContent(
            Consumer<GetRawMessageContentRequest.Builder> getRawMessageContentRequest,
            AsyncResponseTransformer<GetRawMessageContentResponse, ReturnT> asyncResponseTransformer) {
        return getRawMessageContent(GetRawMessageContentRequest.builder().applyMutation(getRawMessageContentRequest).build(),
                asyncResponseTransformer);
    }

    /**
     * <p>
     * Retrieves the raw content of an in-transit email message, in MIME format.
     * </p>
     *
     * @param getRawMessageContentRequest
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The raw content of the email message, in MIME format.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested email message is not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkMailMessageFlowException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample WorkMailMessageFlowAsyncClient.GetRawMessageContent
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workmailmessageflow-2019-05-01/GetRawMessageContent"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetRawMessageContentResponse> getRawMessageContent(
            GetRawMessageContentRequest getRawMessageContentRequest, Path destinationPath) {
        return getRawMessageContent(getRawMessageContentRequest, AsyncResponseTransformer.toFile(destinationPath));
    }

    /**
     * <p>
     * Retrieves the raw content of an in-transit email message, in MIME format.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetRawMessageContentRequest.Builder} avoiding the
     * need to create one manually via {@link GetRawMessageContentRequest#builder()}
     * </p>
     *
     * @param getRawMessageContentRequest
     *        A {@link Consumer} that will call methods on {@link GetRawMessageContentRequest.Builder} to create a
     *        request.
     * @param destinationPath
     *        {@link Path} to file that response contents will be written to. The file must not exist or this method
     *        will throw an exception. If the file is not writable by the current user then an exception will be thrown.
     *        The service documentation for the response content is as follows '
     *        <p>
     *        The raw content of the email message, in MIME format.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested email message is not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkMailMessageFlowException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample WorkMailMessageFlowAsyncClient.GetRawMessageContent
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workmailmessageflow-2019-05-01/GetRawMessageContent"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetRawMessageContentResponse> getRawMessageContent(
            Consumer<GetRawMessageContentRequest.Builder> getRawMessageContentRequest, Path destinationPath) {
        return getRawMessageContent(GetRawMessageContentRequest.builder().applyMutation(getRawMessageContentRequest).build(),
                destinationPath);
    }

    /**
     * <p>
     * Updates the raw content of an in-transit email message, in MIME format.
     * </p>
     * <p>
     * This example describes how to update in-transit email message. For more information and examples for using this
     * API, see <a href="https://docs.aws.amazon.com/workmail/latest/adminguide/update-with-lambda.html"> Updating
     * message content with AWS Lambda</a>.
     * </p>
     * <note>
     * <p>
     * Updates to an in-transit message only appear when you call <code>PutRawMessageContent</code> from an AWS Lambda
     * function configured with a synchronous <a
     * href="https://docs.aws.amazon.com/workmail/latest/adminguide/lambda.html#synchronous-rules"> Run Lambda</a> rule.
     * If you call <code>PutRawMessageContent</code> on a delivered or sent message, the message remains unchanged, even
     * though <a
     * href="https://docs.aws.amazon.com/workmail/latest/APIReference/API_messageflow_GetRawMessageContent.html"
     * >GetRawMessageContent</a> returns an updated message.
     * </p>
     * </note>
     *
     * @param putRawMessageContentRequest
     * @return A Java Future containing the result of the PutRawMessageContent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested email message is not found.</li>
     *         <li>InvalidContentLocationException WorkMail could not access the updated email content. Possible
     *         reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         You made the request in a region other than your S3 bucket region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/bucket-owner-condition.html">S3 bucket
     *         owner</a> is not the same as the calling AWS account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You have an incomplete or missing S3 bucket policy. For more information about policies, see <a
     *         href="https://docs.aws.amazon.com/workmail/latest/adminguide/update-with-lambda.html"> Updating message
     *         content with AWS Lambda </a> in the <i>WorkMail Administrator Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>MessageRejectedException The requested email could not be updated due to an error in the MIME
     *         content. Check the error message for more information about what caused the error.</li>
     *         <li>MessageFrozenException The requested email is not eligible for update. This is usually the case for a
     *         redirected email.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkMailMessageFlowException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample WorkMailMessageFlowAsyncClient.PutRawMessageContent
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workmailmessageflow-2019-05-01/PutRawMessageContent"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<PutRawMessageContentResponse> putRawMessageContent(
            PutRawMessageContentRequest putRawMessageContentRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the raw content of an in-transit email message, in MIME format.
     * </p>
     * <p>
     * This example describes how to update in-transit email message. For more information and examples for using this
     * API, see <a href="https://docs.aws.amazon.com/workmail/latest/adminguide/update-with-lambda.html"> Updating
     * message content with AWS Lambda</a>.
     * </p>
     * <note>
     * <p>
     * Updates to an in-transit message only appear when you call <code>PutRawMessageContent</code> from an AWS Lambda
     * function configured with a synchronous <a
     * href="https://docs.aws.amazon.com/workmail/latest/adminguide/lambda.html#synchronous-rules"> Run Lambda</a> rule.
     * If you call <code>PutRawMessageContent</code> on a delivered or sent message, the message remains unchanged, even
     * though <a
     * href="https://docs.aws.amazon.com/workmail/latest/APIReference/API_messageflow_GetRawMessageContent.html"
     * >GetRawMessageContent</a> returns an updated message.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutRawMessageContentRequest.Builder} avoiding the
     * need to create one manually via {@link PutRawMessageContentRequest#builder()}
     * </p>
     *
     * @param putRawMessageContentRequest
     *        A {@link Consumer} that will call methods on {@link PutRawMessageContentRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the PutRawMessageContent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested email message is not found.</li>
     *         <li>InvalidContentLocationException WorkMail could not access the updated email content. Possible
     *         reasons:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         You made the request in a region other than your S3 bucket region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/bucket-owner-condition.html">S3 bucket
     *         owner</a> is not the same as the calling AWS account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You have an incomplete or missing S3 bucket policy. For more information about policies, see <a
     *         href="https://docs.aws.amazon.com/workmail/latest/adminguide/update-with-lambda.html"> Updating message
     *         content with AWS Lambda </a> in the <i>WorkMail Administrator Guide</i>.
     *         </p>
     *         </li></li>
     *         <li>MessageRejectedException The requested email could not be updated due to an error in the MIME
     *         content. Check the error message for more information about what caused the error.</li>
     *         <li>MessageFrozenException The requested email is not eligible for update. This is usually the case for a
     *         redirected email.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>WorkMailMessageFlowException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample WorkMailMessageFlowAsyncClient.PutRawMessageContent
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/workmailmessageflow-2019-05-01/PutRawMessageContent"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<PutRawMessageContentResponse> putRawMessageContent(
            Consumer<PutRawMessageContentRequest.Builder> putRawMessageContentRequest) {
        return putRawMessageContent(PutRawMessageContentRequest.builder().applyMutation(putRawMessageContentRequest).build());
    }
}
