/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transfer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the details for a connector object. The connector object is used for AS2 outbound processes, to connect the
 * Transfer Family customer with the trading partner.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class As2ConnectorConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<As2ConnectorConfig.Builder, As2ConnectorConfig> {
    private static final SdkField<String> LOCAL_PROFILE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LocalProfileId").getter(getter(As2ConnectorConfig::localProfileId))
            .setter(setter(Builder::localProfileId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocalProfileId").build()).build();

    private static final SdkField<String> PARTNER_PROFILE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PartnerProfileId").getter(getter(As2ConnectorConfig::partnerProfileId))
            .setter(setter(Builder::partnerProfileId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PartnerProfileId").build()).build();

    private static final SdkField<String> MESSAGE_SUBJECT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MessageSubject").getter(getter(As2ConnectorConfig::messageSubject))
            .setter(setter(Builder::messageSubject))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageSubject").build()).build();

    private static final SdkField<String> COMPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Compression").getter(getter(As2ConnectorConfig::compressionAsString))
            .setter(setter(Builder::compression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Compression").build()).build();

    private static final SdkField<String> ENCRYPTION_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EncryptionAlgorithm").getter(getter(As2ConnectorConfig::encryptionAlgorithmAsString))
            .setter(setter(Builder::encryptionAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionAlgorithm").build())
            .build();

    private static final SdkField<String> SIGNING_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SigningAlgorithm").getter(getter(As2ConnectorConfig::signingAlgorithmAsString))
            .setter(setter(Builder::signingAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SigningAlgorithm").build()).build();

    private static final SdkField<String> MDN_SIGNING_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MdnSigningAlgorithm").getter(getter(As2ConnectorConfig::mdnSigningAlgorithmAsString))
            .setter(setter(Builder::mdnSigningAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MdnSigningAlgorithm").build())
            .build();

    private static final SdkField<String> MDN_RESPONSE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MdnResponse").getter(getter(As2ConnectorConfig::mdnResponseAsString))
            .setter(setter(Builder::mdnResponse))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MdnResponse").build()).build();

    private static final SdkField<String> BASIC_AUTH_SECRET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BasicAuthSecretId").getter(getter(As2ConnectorConfig::basicAuthSecretId))
            .setter(setter(Builder::basicAuthSecretId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BasicAuthSecretId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOCAL_PROFILE_ID_FIELD,
            PARTNER_PROFILE_ID_FIELD, MESSAGE_SUBJECT_FIELD, COMPRESSION_FIELD, ENCRYPTION_ALGORITHM_FIELD,
            SIGNING_ALGORITHM_FIELD, MDN_SIGNING_ALGORITHM_FIELD, MDN_RESPONSE_FIELD, BASIC_AUTH_SECRET_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String localProfileId;

    private final String partnerProfileId;

    private final String messageSubject;

    private final String compression;

    private final String encryptionAlgorithm;

    private final String signingAlgorithm;

    private final String mdnSigningAlgorithm;

    private final String mdnResponse;

    private final String basicAuthSecretId;

    private As2ConnectorConfig(BuilderImpl builder) {
        this.localProfileId = builder.localProfileId;
        this.partnerProfileId = builder.partnerProfileId;
        this.messageSubject = builder.messageSubject;
        this.compression = builder.compression;
        this.encryptionAlgorithm = builder.encryptionAlgorithm;
        this.signingAlgorithm = builder.signingAlgorithm;
        this.mdnSigningAlgorithm = builder.mdnSigningAlgorithm;
        this.mdnResponse = builder.mdnResponse;
        this.basicAuthSecretId = builder.basicAuthSecretId;
    }

    /**
     * <p>
     * A unique identifier for the AS2 local profile.
     * </p>
     * 
     * @return A unique identifier for the AS2 local profile.
     */
    public final String localProfileId() {
        return localProfileId;
    }

    /**
     * <p>
     * A unique identifier for the partner profile for the connector.
     * </p>
     * 
     * @return A unique identifier for the partner profile for the connector.
     */
    public final String partnerProfileId() {
        return partnerProfileId;
    }

    /**
     * <p>
     * Used as the <code>Subject</code> HTTP header attribute in AS2 messages that are being sent with the connector.
     * </p>
     * 
     * @return Used as the <code>Subject</code> HTTP header attribute in AS2 messages that are being sent with the
     *         connector.
     */
    public final String messageSubject() {
        return messageSubject;
    }

    /**
     * <p>
     * Specifies whether the AS2 file is compressed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #compression} will
     * return {@link CompressionEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #compressionAsString}.
     * </p>
     * 
     * @return Specifies whether the AS2 file is compressed.
     * @see CompressionEnum
     */
    public final CompressionEnum compression() {
        return CompressionEnum.fromValue(compression);
    }

    /**
     * <p>
     * Specifies whether the AS2 file is compressed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #compression} will
     * return {@link CompressionEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #compressionAsString}.
     * </p>
     * 
     * @return Specifies whether the AS2 file is compressed.
     * @see CompressionEnum
     */
    public final String compressionAsString() {
        return compression;
    }

    /**
     * <p>
     * The algorithm that is used to encrypt the file.
     * </p>
     * <note>
     * <p>
     * You can only specify <code>NONE</code> if the URL for your connector uses HTTPS. This ensures that no traffic is
     * sent in clear text.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #encryptionAlgorithm} will return {@link EncryptionAlg#UNKNOWN_TO_SDK_VERSION}. The raw value returned by
     * the service is available from {@link #encryptionAlgorithmAsString}.
     * </p>
     * 
     * @return The algorithm that is used to encrypt the file.</p> <note>
     *         <p>
     *         You can only specify <code>NONE</code> if the URL for your connector uses HTTPS. This ensures that no
     *         traffic is sent in clear text.
     *         </p>
     * @see EncryptionAlg
     */
    public final EncryptionAlg encryptionAlgorithm() {
        return EncryptionAlg.fromValue(encryptionAlgorithm);
    }

    /**
     * <p>
     * The algorithm that is used to encrypt the file.
     * </p>
     * <note>
     * <p>
     * You can only specify <code>NONE</code> if the URL for your connector uses HTTPS. This ensures that no traffic is
     * sent in clear text.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #encryptionAlgorithm} will return {@link EncryptionAlg#UNKNOWN_TO_SDK_VERSION}. The raw value returned by
     * the service is available from {@link #encryptionAlgorithmAsString}.
     * </p>
     * 
     * @return The algorithm that is used to encrypt the file.</p> <note>
     *         <p>
     *         You can only specify <code>NONE</code> if the URL for your connector uses HTTPS. This ensures that no
     *         traffic is sent in clear text.
     *         </p>
     * @see EncryptionAlg
     */
    public final String encryptionAlgorithmAsString() {
        return encryptionAlgorithm;
    }

    /**
     * <p>
     * The algorithm that is used to sign the AS2 messages sent with the connector.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #signingAlgorithm}
     * will return {@link SigningAlg#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #signingAlgorithmAsString}.
     * </p>
     * 
     * @return The algorithm that is used to sign the AS2 messages sent with the connector.
     * @see SigningAlg
     */
    public final SigningAlg signingAlgorithm() {
        return SigningAlg.fromValue(signingAlgorithm);
    }

    /**
     * <p>
     * The algorithm that is used to sign the AS2 messages sent with the connector.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #signingAlgorithm}
     * will return {@link SigningAlg#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #signingAlgorithmAsString}.
     * </p>
     * 
     * @return The algorithm that is used to sign the AS2 messages sent with the connector.
     * @see SigningAlg
     */
    public final String signingAlgorithmAsString() {
        return signingAlgorithm;
    }

    /**
     * <p>
     * The signing algorithm for the MDN response.
     * </p>
     * <note>
     * <p>
     * If set to DEFAULT (or not set at all), the value for <code>SigningAlgorithm</code> is used.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #mdnSigningAlgorithm} will return {@link MdnSigningAlg#UNKNOWN_TO_SDK_VERSION}. The raw value returned by
     * the service is available from {@link #mdnSigningAlgorithmAsString}.
     * </p>
     * 
     * @return The signing algorithm for the MDN response.</p> <note>
     *         <p>
     *         If set to DEFAULT (or not set at all), the value for <code>SigningAlgorithm</code> is used.
     *         </p>
     * @see MdnSigningAlg
     */
    public final MdnSigningAlg mdnSigningAlgorithm() {
        return MdnSigningAlg.fromValue(mdnSigningAlgorithm);
    }

    /**
     * <p>
     * The signing algorithm for the MDN response.
     * </p>
     * <note>
     * <p>
     * If set to DEFAULT (or not set at all), the value for <code>SigningAlgorithm</code> is used.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #mdnSigningAlgorithm} will return {@link MdnSigningAlg#UNKNOWN_TO_SDK_VERSION}. The raw value returned by
     * the service is available from {@link #mdnSigningAlgorithmAsString}.
     * </p>
     * 
     * @return The signing algorithm for the MDN response.</p> <note>
     *         <p>
     *         If set to DEFAULT (or not set at all), the value for <code>SigningAlgorithm</code> is used.
     *         </p>
     * @see MdnSigningAlg
     */
    public final String mdnSigningAlgorithmAsString() {
        return mdnSigningAlgorithm;
    }

    /**
     * <p>
     * Used for outbound requests (from an Transfer Family server to a partner AS2 server) to determine whether the
     * partner response for transfers is synchronous or asynchronous. Specify either of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SYNC</code>: The system expects a synchronous MDN response, confirming that the file was transferred
     * successfully (or not).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code>: Specifies that no MDN response is required.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mdnResponse} will
     * return {@link MdnResponse#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mdnResponseAsString}.
     * </p>
     * 
     * @return Used for outbound requests (from an Transfer Family server to a partner AS2 server) to determine whether
     *         the partner response for transfers is synchronous or asynchronous. Specify either of the following
     *         values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SYNC</code>: The system expects a synchronous MDN response, confirming that the file was
     *         transferred successfully (or not).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code>: Specifies that no MDN response is required.
     *         </p>
     *         </li>
     * @see MdnResponse
     */
    public final MdnResponse mdnResponse() {
        return MdnResponse.fromValue(mdnResponse);
    }

    /**
     * <p>
     * Used for outbound requests (from an Transfer Family server to a partner AS2 server) to determine whether the
     * partner response for transfers is synchronous or asynchronous. Specify either of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SYNC</code>: The system expects a synchronous MDN response, confirming that the file was transferred
     * successfully (or not).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code>: Specifies that no MDN response is required.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mdnResponse} will
     * return {@link MdnResponse#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mdnResponseAsString}.
     * </p>
     * 
     * @return Used for outbound requests (from an Transfer Family server to a partner AS2 server) to determine whether
     *         the partner response for transfers is synchronous or asynchronous. Specify either of the following
     *         values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SYNC</code>: The system expects a synchronous MDN response, confirming that the file was
     *         transferred successfully (or not).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code>: Specifies that no MDN response is required.
     *         </p>
     *         </li>
     * @see MdnResponse
     */
    public final String mdnResponseAsString() {
        return mdnResponse;
    }

    /**
     * <p>
     * Provides Basic authentication support to the AS2 Connectors API. To use Basic authentication, you must provide
     * the name or Amazon Resource Name (ARN) of a secret in Secrets Manager.
     * </p>
     * <p>
     * The default value for this parameter is <code>null</code>, which indicates that Basic authentication is not
     * enabled for the connector.
     * </p>
     * <p>
     * If the connector should use Basic authentication, the secret needs to be in the following format:
     * </p>
     * <p>
     * <code>{ "Username": "user-name", "Password": "user-password" }</code>
     * </p>
     * <p>
     * Replace <code>user-name</code> and <code>user-password</code> with the credentials for the actual user that is
     * being authenticated.
     * </p>
     * <p>
     * Note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You are storing these credentials in Secrets Manager, <i>not passing them directly</i> into this API.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you are using the API, SDKs, or CloudFormation to configure your connector, then you must create the secret
     * before you can enable Basic authentication. However, if you are using the Amazon Web Services management console,
     * you can have the system create the secret for you.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you have previously enabled Basic authentication for a connector, you can disable it by using the
     * <code>UpdateConnector</code> API call. For example, if you are using the CLI, you can run the following command
     * to remove Basic authentication:
     * </p>
     * <p>
     * <code>update-connector --connector-id my-connector-id --as2-config 'BasicAuthSecretId=""'</code>
     * </p>
     * 
     * @return Provides Basic authentication support to the AS2 Connectors API. To use Basic authentication, you must
     *         provide the name or Amazon Resource Name (ARN) of a secret in Secrets Manager.</p>
     *         <p>
     *         The default value for this parameter is <code>null</code>, which indicates that Basic authentication is
     *         not enabled for the connector.
     *         </p>
     *         <p>
     *         If the connector should use Basic authentication, the secret needs to be in the following format:
     *         </p>
     *         <p>
     *         <code>{ "Username": "user-name", "Password": "user-password" }</code>
     *         </p>
     *         <p>
     *         Replace <code>user-name</code> and <code>user-password</code> with the credentials for the actual user
     *         that is being authenticated.
     *         </p>
     *         <p>
     *         Note the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You are storing these credentials in Secrets Manager, <i>not passing them directly</i> into this API.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you are using the API, SDKs, or CloudFormation to configure your connector, then you must create the
     *         secret before you can enable Basic authentication. However, if you are using the Amazon Web Services
     *         management console, you can have the system create the secret for you.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you have previously enabled Basic authentication for a connector, you can disable it by using the
     *         <code>UpdateConnector</code> API call. For example, if you are using the CLI, you can run the following
     *         command to remove Basic authentication:
     *         </p>
     *         <p>
     *         <code>update-connector --connector-id my-connector-id --as2-config 'BasicAuthSecretId=""'</code>
     */
    public final String basicAuthSecretId() {
        return basicAuthSecretId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(localProfileId());
        hashCode = 31 * hashCode + Objects.hashCode(partnerProfileId());
        hashCode = 31 * hashCode + Objects.hashCode(messageSubject());
        hashCode = 31 * hashCode + Objects.hashCode(compressionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionAlgorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(signingAlgorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(mdnSigningAlgorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(mdnResponseAsString());
        hashCode = 31 * hashCode + Objects.hashCode(basicAuthSecretId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof As2ConnectorConfig)) {
            return false;
        }
        As2ConnectorConfig other = (As2ConnectorConfig) obj;
        return Objects.equals(localProfileId(), other.localProfileId())
                && Objects.equals(partnerProfileId(), other.partnerProfileId())
                && Objects.equals(messageSubject(), other.messageSubject())
                && Objects.equals(compressionAsString(), other.compressionAsString())
                && Objects.equals(encryptionAlgorithmAsString(), other.encryptionAlgorithmAsString())
                && Objects.equals(signingAlgorithmAsString(), other.signingAlgorithmAsString())
                && Objects.equals(mdnSigningAlgorithmAsString(), other.mdnSigningAlgorithmAsString())
                && Objects.equals(mdnResponseAsString(), other.mdnResponseAsString())
                && Objects.equals(basicAuthSecretId(), other.basicAuthSecretId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("As2ConnectorConfig").add("LocalProfileId", localProfileId())
                .add("PartnerProfileId", partnerProfileId()).add("MessageSubject", messageSubject())
                .add("Compression", compressionAsString()).add("EncryptionAlgorithm", encryptionAlgorithmAsString())
                .add("SigningAlgorithm", signingAlgorithmAsString()).add("MdnSigningAlgorithm", mdnSigningAlgorithmAsString())
                .add("MdnResponse", mdnResponseAsString()).add("BasicAuthSecretId", basicAuthSecretId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LocalProfileId":
            return Optional.ofNullable(clazz.cast(localProfileId()));
        case "PartnerProfileId":
            return Optional.ofNullable(clazz.cast(partnerProfileId()));
        case "MessageSubject":
            return Optional.ofNullable(clazz.cast(messageSubject()));
        case "Compression":
            return Optional.ofNullable(clazz.cast(compressionAsString()));
        case "EncryptionAlgorithm":
            return Optional.ofNullable(clazz.cast(encryptionAlgorithmAsString()));
        case "SigningAlgorithm":
            return Optional.ofNullable(clazz.cast(signingAlgorithmAsString()));
        case "MdnSigningAlgorithm":
            return Optional.ofNullable(clazz.cast(mdnSigningAlgorithmAsString()));
        case "MdnResponse":
            return Optional.ofNullable(clazz.cast(mdnResponseAsString()));
        case "BasicAuthSecretId":
            return Optional.ofNullable(clazz.cast(basicAuthSecretId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<As2ConnectorConfig, T> g) {
        return obj -> g.apply((As2ConnectorConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, As2ConnectorConfig> {
        /**
         * <p>
         * A unique identifier for the AS2 local profile.
         * </p>
         * 
         * @param localProfileId
         *        A unique identifier for the AS2 local profile.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localProfileId(String localProfileId);

        /**
         * <p>
         * A unique identifier for the partner profile for the connector.
         * </p>
         * 
         * @param partnerProfileId
         *        A unique identifier for the partner profile for the connector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partnerProfileId(String partnerProfileId);

        /**
         * <p>
         * Used as the <code>Subject</code> HTTP header attribute in AS2 messages that are being sent with the
         * connector.
         * </p>
         * 
         * @param messageSubject
         *        Used as the <code>Subject</code> HTTP header attribute in AS2 messages that are being sent with the
         *        connector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageSubject(String messageSubject);

        /**
         * <p>
         * Specifies whether the AS2 file is compressed.
         * </p>
         * 
         * @param compression
         *        Specifies whether the AS2 file is compressed.
         * @see CompressionEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CompressionEnum
         */
        Builder compression(String compression);

        /**
         * <p>
         * Specifies whether the AS2 file is compressed.
         * </p>
         * 
         * @param compression
         *        Specifies whether the AS2 file is compressed.
         * @see CompressionEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CompressionEnum
         */
        Builder compression(CompressionEnum compression);

        /**
         * <p>
         * The algorithm that is used to encrypt the file.
         * </p>
         * <note>
         * <p>
         * You can only specify <code>NONE</code> if the URL for your connector uses HTTPS. This ensures that no traffic
         * is sent in clear text.
         * </p>
         * </note>
         * 
         * @param encryptionAlgorithm
         *        The algorithm that is used to encrypt the file.</p> <note>
         *        <p>
         *        You can only specify <code>NONE</code> if the URL for your connector uses HTTPS. This ensures that no
         *        traffic is sent in clear text.
         *        </p>
         * @see EncryptionAlg
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionAlg
         */
        Builder encryptionAlgorithm(String encryptionAlgorithm);

        /**
         * <p>
         * The algorithm that is used to encrypt the file.
         * </p>
         * <note>
         * <p>
         * You can only specify <code>NONE</code> if the URL for your connector uses HTTPS. This ensures that no traffic
         * is sent in clear text.
         * </p>
         * </note>
         * 
         * @param encryptionAlgorithm
         *        The algorithm that is used to encrypt the file.</p> <note>
         *        <p>
         *        You can only specify <code>NONE</code> if the URL for your connector uses HTTPS. This ensures that no
         *        traffic is sent in clear text.
         *        </p>
         * @see EncryptionAlg
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionAlg
         */
        Builder encryptionAlgorithm(EncryptionAlg encryptionAlgorithm);

        /**
         * <p>
         * The algorithm that is used to sign the AS2 messages sent with the connector.
         * </p>
         * 
         * @param signingAlgorithm
         *        The algorithm that is used to sign the AS2 messages sent with the connector.
         * @see SigningAlg
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SigningAlg
         */
        Builder signingAlgorithm(String signingAlgorithm);

        /**
         * <p>
         * The algorithm that is used to sign the AS2 messages sent with the connector.
         * </p>
         * 
         * @param signingAlgorithm
         *        The algorithm that is used to sign the AS2 messages sent with the connector.
         * @see SigningAlg
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SigningAlg
         */
        Builder signingAlgorithm(SigningAlg signingAlgorithm);

        /**
         * <p>
         * The signing algorithm for the MDN response.
         * </p>
         * <note>
         * <p>
         * If set to DEFAULT (or not set at all), the value for <code>SigningAlgorithm</code> is used.
         * </p>
         * </note>
         * 
         * @param mdnSigningAlgorithm
         *        The signing algorithm for the MDN response.</p> <note>
         *        <p>
         *        If set to DEFAULT (or not set at all), the value for <code>SigningAlgorithm</code> is used.
         *        </p>
         * @see MdnSigningAlg
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MdnSigningAlg
         */
        Builder mdnSigningAlgorithm(String mdnSigningAlgorithm);

        /**
         * <p>
         * The signing algorithm for the MDN response.
         * </p>
         * <note>
         * <p>
         * If set to DEFAULT (or not set at all), the value for <code>SigningAlgorithm</code> is used.
         * </p>
         * </note>
         * 
         * @param mdnSigningAlgorithm
         *        The signing algorithm for the MDN response.</p> <note>
         *        <p>
         *        If set to DEFAULT (or not set at all), the value for <code>SigningAlgorithm</code> is used.
         *        </p>
         * @see MdnSigningAlg
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MdnSigningAlg
         */
        Builder mdnSigningAlgorithm(MdnSigningAlg mdnSigningAlgorithm);

        /**
         * <p>
         * Used for outbound requests (from an Transfer Family server to a partner AS2 server) to determine whether the
         * partner response for transfers is synchronous or asynchronous. Specify either of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SYNC</code>: The system expects a synchronous MDN response, confirming that the file was transferred
         * successfully (or not).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code>: Specifies that no MDN response is required.
         * </p>
         * </li>
         * </ul>
         * 
         * @param mdnResponse
         *        Used for outbound requests (from an Transfer Family server to a partner AS2 server) to determine
         *        whether the partner response for transfers is synchronous or asynchronous. Specify either of the
         *        following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SYNC</code>: The system expects a synchronous MDN response, confirming that the file was
         *        transferred successfully (or not).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code>: Specifies that no MDN response is required.
         *        </p>
         *        </li>
         * @see MdnResponse
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MdnResponse
         */
        Builder mdnResponse(String mdnResponse);

        /**
         * <p>
         * Used for outbound requests (from an Transfer Family server to a partner AS2 server) to determine whether the
         * partner response for transfers is synchronous or asynchronous. Specify either of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SYNC</code>: The system expects a synchronous MDN response, confirming that the file was transferred
         * successfully (or not).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code>: Specifies that no MDN response is required.
         * </p>
         * </li>
         * </ul>
         * 
         * @param mdnResponse
         *        Used for outbound requests (from an Transfer Family server to a partner AS2 server) to determine
         *        whether the partner response for transfers is synchronous or asynchronous. Specify either of the
         *        following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SYNC</code>: The system expects a synchronous MDN response, confirming that the file was
         *        transferred successfully (or not).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code>: Specifies that no MDN response is required.
         *        </p>
         *        </li>
         * @see MdnResponse
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MdnResponse
         */
        Builder mdnResponse(MdnResponse mdnResponse);

        /**
         * <p>
         * Provides Basic authentication support to the AS2 Connectors API. To use Basic authentication, you must
         * provide the name or Amazon Resource Name (ARN) of a secret in Secrets Manager.
         * </p>
         * <p>
         * The default value for this parameter is <code>null</code>, which indicates that Basic authentication is not
         * enabled for the connector.
         * </p>
         * <p>
         * If the connector should use Basic authentication, the secret needs to be in the following format:
         * </p>
         * <p>
         * <code>{ "Username": "user-name", "Password": "user-password" }</code>
         * </p>
         * <p>
         * Replace <code>user-name</code> and <code>user-password</code> with the credentials for the actual user that
         * is being authenticated.
         * </p>
         * <p>
         * Note the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * You are storing these credentials in Secrets Manager, <i>not passing them directly</i> into this API.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you are using the API, SDKs, or CloudFormation to configure your connector, then you must create the
         * secret before you can enable Basic authentication. However, if you are using the Amazon Web Services
         * management console, you can have the system create the secret for you.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you have previously enabled Basic authentication for a connector, you can disable it by using the
         * <code>UpdateConnector</code> API call. For example, if you are using the CLI, you can run the following
         * command to remove Basic authentication:
         * </p>
         * <p>
         * <code>update-connector --connector-id my-connector-id --as2-config 'BasicAuthSecretId=""'</code>
         * </p>
         * 
         * @param basicAuthSecretId
         *        Provides Basic authentication support to the AS2 Connectors API. To use Basic authentication, you must
         *        provide the name or Amazon Resource Name (ARN) of a secret in Secrets Manager.</p>
         *        <p>
         *        The default value for this parameter is <code>null</code>, which indicates that Basic authentication
         *        is not enabled for the connector.
         *        </p>
         *        <p>
         *        If the connector should use Basic authentication, the secret needs to be in the following format:
         *        </p>
         *        <p>
         *        <code>{ "Username": "user-name", "Password": "user-password" }</code>
         *        </p>
         *        <p>
         *        Replace <code>user-name</code> and <code>user-password</code> with the credentials for the actual user
         *        that is being authenticated.
         *        </p>
         *        <p>
         *        Note the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        You are storing these credentials in Secrets Manager, <i>not passing them directly</i> into this API.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you are using the API, SDKs, or CloudFormation to configure your connector, then you must create
         *        the secret before you can enable Basic authentication. However, if you are using the Amazon Web
         *        Services management console, you can have the system create the secret for you.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you have previously enabled Basic authentication for a connector, you can disable it by using the
         *        <code>UpdateConnector</code> API call. For example, if you are using the CLI, you can run the
         *        following command to remove Basic authentication:
         *        </p>
         *        <p>
         *        <code>update-connector --connector-id my-connector-id --as2-config 'BasicAuthSecretId=""'</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder basicAuthSecretId(String basicAuthSecretId);
    }

    static final class BuilderImpl implements Builder {
        private String localProfileId;

        private String partnerProfileId;

        private String messageSubject;

        private String compression;

        private String encryptionAlgorithm;

        private String signingAlgorithm;

        private String mdnSigningAlgorithm;

        private String mdnResponse;

        private String basicAuthSecretId;

        private BuilderImpl() {
        }

        private BuilderImpl(As2ConnectorConfig model) {
            localProfileId(model.localProfileId);
            partnerProfileId(model.partnerProfileId);
            messageSubject(model.messageSubject);
            compression(model.compression);
            encryptionAlgorithm(model.encryptionAlgorithm);
            signingAlgorithm(model.signingAlgorithm);
            mdnSigningAlgorithm(model.mdnSigningAlgorithm);
            mdnResponse(model.mdnResponse);
            basicAuthSecretId(model.basicAuthSecretId);
        }

        public final String getLocalProfileId() {
            return localProfileId;
        }

        public final void setLocalProfileId(String localProfileId) {
            this.localProfileId = localProfileId;
        }

        @Override
        public final Builder localProfileId(String localProfileId) {
            this.localProfileId = localProfileId;
            return this;
        }

        public final String getPartnerProfileId() {
            return partnerProfileId;
        }

        public final void setPartnerProfileId(String partnerProfileId) {
            this.partnerProfileId = partnerProfileId;
        }

        @Override
        public final Builder partnerProfileId(String partnerProfileId) {
            this.partnerProfileId = partnerProfileId;
            return this;
        }

        public final String getMessageSubject() {
            return messageSubject;
        }

        public final void setMessageSubject(String messageSubject) {
            this.messageSubject = messageSubject;
        }

        @Override
        public final Builder messageSubject(String messageSubject) {
            this.messageSubject = messageSubject;
            return this;
        }

        public final String getCompression() {
            return compression;
        }

        public final void setCompression(String compression) {
            this.compression = compression;
        }

        @Override
        public final Builder compression(String compression) {
            this.compression = compression;
            return this;
        }

        @Override
        public final Builder compression(CompressionEnum compression) {
            this.compression(compression == null ? null : compression.toString());
            return this;
        }

        public final String getEncryptionAlgorithm() {
            return encryptionAlgorithm;
        }

        public final void setEncryptionAlgorithm(String encryptionAlgorithm) {
            this.encryptionAlgorithm = encryptionAlgorithm;
        }

        @Override
        public final Builder encryptionAlgorithm(String encryptionAlgorithm) {
            this.encryptionAlgorithm = encryptionAlgorithm;
            return this;
        }

        @Override
        public final Builder encryptionAlgorithm(EncryptionAlg encryptionAlgorithm) {
            this.encryptionAlgorithm(encryptionAlgorithm == null ? null : encryptionAlgorithm.toString());
            return this;
        }

        public final String getSigningAlgorithm() {
            return signingAlgorithm;
        }

        public final void setSigningAlgorithm(String signingAlgorithm) {
            this.signingAlgorithm = signingAlgorithm;
        }

        @Override
        public final Builder signingAlgorithm(String signingAlgorithm) {
            this.signingAlgorithm = signingAlgorithm;
            return this;
        }

        @Override
        public final Builder signingAlgorithm(SigningAlg signingAlgorithm) {
            this.signingAlgorithm(signingAlgorithm == null ? null : signingAlgorithm.toString());
            return this;
        }

        public final String getMdnSigningAlgorithm() {
            return mdnSigningAlgorithm;
        }

        public final void setMdnSigningAlgorithm(String mdnSigningAlgorithm) {
            this.mdnSigningAlgorithm = mdnSigningAlgorithm;
        }

        @Override
        public final Builder mdnSigningAlgorithm(String mdnSigningAlgorithm) {
            this.mdnSigningAlgorithm = mdnSigningAlgorithm;
            return this;
        }

        @Override
        public final Builder mdnSigningAlgorithm(MdnSigningAlg mdnSigningAlgorithm) {
            this.mdnSigningAlgorithm(mdnSigningAlgorithm == null ? null : mdnSigningAlgorithm.toString());
            return this;
        }

        public final String getMdnResponse() {
            return mdnResponse;
        }

        public final void setMdnResponse(String mdnResponse) {
            this.mdnResponse = mdnResponse;
        }

        @Override
        public final Builder mdnResponse(String mdnResponse) {
            this.mdnResponse = mdnResponse;
            return this;
        }

        @Override
        public final Builder mdnResponse(MdnResponse mdnResponse) {
            this.mdnResponse(mdnResponse == null ? null : mdnResponse.toString());
            return this;
        }

        public final String getBasicAuthSecretId() {
            return basicAuthSecretId;
        }

        public final void setBasicAuthSecretId(String basicAuthSecretId) {
            this.basicAuthSecretId = basicAuthSecretId;
        }

        @Override
        public final Builder basicAuthSecretId(String basicAuthSecretId) {
            this.basicAuthSecretId = basicAuthSecretId;
            return this;
        }

        @Override
        public As2ConnectorConfig build() {
            return new As2ConnectorConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
