/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.textract.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information on the metrics used to evalute the peformance of a given adapter version. Includes data for
 * baseline model performance and individual adapter version perfromance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AdapterVersionEvaluationMetric implements SdkPojo, Serializable,
        ToCopyableBuilder<AdapterVersionEvaluationMetric.Builder, AdapterVersionEvaluationMetric> {
    private static final SdkField<EvaluationMetric> BASELINE_FIELD = SdkField
            .<EvaluationMetric> builder(MarshallingType.SDK_POJO).memberName("Baseline")
            .getter(getter(AdapterVersionEvaluationMetric::baseline)).setter(setter(Builder::baseline))
            .constructor(EvaluationMetric::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Baseline").build()).build();

    private static final SdkField<EvaluationMetric> ADAPTER_VERSION_FIELD = SdkField
            .<EvaluationMetric> builder(MarshallingType.SDK_POJO).memberName("AdapterVersion")
            .getter(getter(AdapterVersionEvaluationMetric::adapterVersion)).setter(setter(Builder::adapterVersion))
            .constructor(EvaluationMetric::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdapterVersion").build()).build();

    private static final SdkField<String> FEATURE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FeatureType").getter(getter(AdapterVersionEvaluationMetric::featureTypeAsString))
            .setter(setter(Builder::featureType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeatureType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BASELINE_FIELD,
            ADAPTER_VERSION_FIELD, FEATURE_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final EvaluationMetric baseline;

    private final EvaluationMetric adapterVersion;

    private final String featureType;

    private AdapterVersionEvaluationMetric(BuilderImpl builder) {
        this.baseline = builder.baseline;
        this.adapterVersion = builder.adapterVersion;
        this.featureType = builder.featureType;
    }

    /**
     * <p>
     * The F1 score, precision, and recall metrics for the baseline model.
     * </p>
     * 
     * @return The F1 score, precision, and recall metrics for the baseline model.
     */
    public final EvaluationMetric baseline() {
        return baseline;
    }

    /**
     * <p>
     * The F1 score, precision, and recall metrics for the baseline model.
     * </p>
     * 
     * @return The F1 score, precision, and recall metrics for the baseline model.
     */
    public final EvaluationMetric adapterVersion() {
        return adapterVersion;
    }

    /**
     * <p>
     * Indicates the feature type being analyzed by a given adapter version.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #featureType} will
     * return {@link FeatureType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #featureTypeAsString}.
     * </p>
     * 
     * @return Indicates the feature type being analyzed by a given adapter version.
     * @see FeatureType
     */
    public final FeatureType featureType() {
        return FeatureType.fromValue(featureType);
    }

    /**
     * <p>
     * Indicates the feature type being analyzed by a given adapter version.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #featureType} will
     * return {@link FeatureType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #featureTypeAsString}.
     * </p>
     * 
     * @return Indicates the feature type being analyzed by a given adapter version.
     * @see FeatureType
     */
    public final String featureTypeAsString() {
        return featureType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(baseline());
        hashCode = 31 * hashCode + Objects.hashCode(adapterVersion());
        hashCode = 31 * hashCode + Objects.hashCode(featureTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AdapterVersionEvaluationMetric)) {
            return false;
        }
        AdapterVersionEvaluationMetric other = (AdapterVersionEvaluationMetric) obj;
        return Objects.equals(baseline(), other.baseline()) && Objects.equals(adapterVersion(), other.adapterVersion())
                && Objects.equals(featureTypeAsString(), other.featureTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AdapterVersionEvaluationMetric").add("Baseline", baseline())
                .add("AdapterVersion", adapterVersion()).add("FeatureType", featureTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Baseline":
            return Optional.ofNullable(clazz.cast(baseline()));
        case "AdapterVersion":
            return Optional.ofNullable(clazz.cast(adapterVersion()));
        case "FeatureType":
            return Optional.ofNullable(clazz.cast(featureTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Baseline", BASELINE_FIELD);
        map.put("AdapterVersion", ADAPTER_VERSION_FIELD);
        map.put("FeatureType", FEATURE_TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AdapterVersionEvaluationMetric, T> g) {
        return obj -> g.apply((AdapterVersionEvaluationMetric) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AdapterVersionEvaluationMetric> {
        /**
         * <p>
         * The F1 score, precision, and recall metrics for the baseline model.
         * </p>
         * 
         * @param baseline
         *        The F1 score, precision, and recall metrics for the baseline model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baseline(EvaluationMetric baseline);

        /**
         * <p>
         * The F1 score, precision, and recall metrics for the baseline model.
         * </p>
         * This is a convenience method that creates an instance of the {@link EvaluationMetric.Builder} avoiding the
         * need to create one manually via {@link EvaluationMetric#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EvaluationMetric.Builder#build()} is called immediately and its
         * result is passed to {@link #baseline(EvaluationMetric)}.
         * 
         * @param baseline
         *        a consumer that will call methods on {@link EvaluationMetric.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #baseline(EvaluationMetric)
         */
        default Builder baseline(Consumer<EvaluationMetric.Builder> baseline) {
            return baseline(EvaluationMetric.builder().applyMutation(baseline).build());
        }

        /**
         * <p>
         * The F1 score, precision, and recall metrics for the baseline model.
         * </p>
         * 
         * @param adapterVersion
         *        The F1 score, precision, and recall metrics for the baseline model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder adapterVersion(EvaluationMetric adapterVersion);

        /**
         * <p>
         * The F1 score, precision, and recall metrics for the baseline model.
         * </p>
         * This is a convenience method that creates an instance of the {@link EvaluationMetric.Builder} avoiding the
         * need to create one manually via {@link EvaluationMetric#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EvaluationMetric.Builder#build()} is called immediately and its
         * result is passed to {@link #adapterVersion(EvaluationMetric)}.
         * 
         * @param adapterVersion
         *        a consumer that will call methods on {@link EvaluationMetric.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #adapterVersion(EvaluationMetric)
         */
        default Builder adapterVersion(Consumer<EvaluationMetric.Builder> adapterVersion) {
            return adapterVersion(EvaluationMetric.builder().applyMutation(adapterVersion).build());
        }

        /**
         * <p>
         * Indicates the feature type being analyzed by a given adapter version.
         * </p>
         * 
         * @param featureType
         *        Indicates the feature type being analyzed by a given adapter version.
         * @see FeatureType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FeatureType
         */
        Builder featureType(String featureType);

        /**
         * <p>
         * Indicates the feature type being analyzed by a given adapter version.
         * </p>
         * 
         * @param featureType
         *        Indicates the feature type being analyzed by a given adapter version.
         * @see FeatureType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FeatureType
         */
        Builder featureType(FeatureType featureType);
    }

    static final class BuilderImpl implements Builder {
        private EvaluationMetric baseline;

        private EvaluationMetric adapterVersion;

        private String featureType;

        private BuilderImpl() {
        }

        private BuilderImpl(AdapterVersionEvaluationMetric model) {
            baseline(model.baseline);
            adapterVersion(model.adapterVersion);
            featureType(model.featureType);
        }

        public final EvaluationMetric.Builder getBaseline() {
            return baseline != null ? baseline.toBuilder() : null;
        }

        public final void setBaseline(EvaluationMetric.BuilderImpl baseline) {
            this.baseline = baseline != null ? baseline.build() : null;
        }

        @Override
        public final Builder baseline(EvaluationMetric baseline) {
            this.baseline = baseline;
            return this;
        }

        public final EvaluationMetric.Builder getAdapterVersion() {
            return adapterVersion != null ? adapterVersion.toBuilder() : null;
        }

        public final void setAdapterVersion(EvaluationMetric.BuilderImpl adapterVersion) {
            this.adapterVersion = adapterVersion != null ? adapterVersion.build() : null;
        }

        @Override
        public final Builder adapterVersion(EvaluationMetric adapterVersion) {
            this.adapterVersion = adapterVersion;
            return this;
        }

        public final String getFeatureType() {
            return featureType;
        }

        public final void setFeatureType(String featureType) {
            this.featureType = featureType;
        }

        @Override
        public final Builder featureType(String featureType) {
            this.featureType = featureType;
            return this;
        }

        @Override
        public final Builder featureType(FeatureType featureType) {
            this.featureType(featureType == null ? null : featureType.toString());
            return this;
        }

        @Override
        public AdapterVersionEvaluationMetric build() {
            return new AdapterVersionEvaluationMetric(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
