/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.textract.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class StartDocumentTextDetectionRequest extends TextractRequest implements
        ToCopyableBuilder<StartDocumentTextDetectionRequest.Builder, StartDocumentTextDetectionRequest> {
    private static final SdkField<DocumentLocation> DOCUMENT_LOCATION_FIELD = SdkField
            .<DocumentLocation> builder(MarshallingType.SDK_POJO)
            .getter(getter(StartDocumentTextDetectionRequest::documentLocation)).setter(setter(Builder::documentLocation))
            .constructor(DocumentLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentLocation").build()).build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StartDocumentTextDetectionRequest::clientRequestToken)).setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestToken").build())
            .build();

    private static final SdkField<String> JOB_TAG_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StartDocumentTextDetectionRequest::jobTag)).setter(setter(Builder::jobTag))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobTag").build()).build();

    private static final SdkField<NotificationChannel> NOTIFICATION_CHANNEL_FIELD = SdkField
            .<NotificationChannel> builder(MarshallingType.SDK_POJO)
            .getter(getter(StartDocumentTextDetectionRequest::notificationChannel)).setter(setter(Builder::notificationChannel))
            .constructor(NotificationChannel::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationChannel").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOCUMENT_LOCATION_FIELD,
            CLIENT_REQUEST_TOKEN_FIELD, JOB_TAG_FIELD, NOTIFICATION_CHANNEL_FIELD));

    private final DocumentLocation documentLocation;

    private final String clientRequestToken;

    private final String jobTag;

    private final NotificationChannel notificationChannel;

    private StartDocumentTextDetectionRequest(BuilderImpl builder) {
        super(builder);
        this.documentLocation = builder.documentLocation;
        this.clientRequestToken = builder.clientRequestToken;
        this.jobTag = builder.jobTag;
        this.notificationChannel = builder.notificationChannel;
    }

    /**
     * <p>
     * The location of the document to be processed.
     * </p>
     * 
     * @return The location of the document to be processed.
     */
    public DocumentLocation documentLocation() {
        return documentLocation;
    }

    /**
     * <p>
     * The idempotent token that's used to identify the start request. If you use the same token with multiple
     * <code>StartDocumentTextDetection</code> requests, the same <code>JobId</code> is returned. Use
     * <code>ClientRequestToken</code> to prevent the same job from being accidentally started more than once. For more
     * information, see <a href="https://docs.aws.amazon.com/textract/latest/dg/api-async.html">Calling Amazon Textract
     * Asynchronous Operations</a>.
     * </p>
     * 
     * @return The idempotent token that's used to identify the start request. If you use the same token with multiple
     *         <code>StartDocumentTextDetection</code> requests, the same <code>JobId</code> is returned. Use
     *         <code>ClientRequestToken</code> to prevent the same job from being accidentally started more than once.
     *         For more information, see <a href="https://docs.aws.amazon.com/textract/latest/dg/api-async.html">Calling
     *         Amazon Textract Asynchronous Operations</a>.
     */
    public String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * <p>
     * An identifier that you specify that's included in the completion notification published to the Amazon SNS topic.
     * For example, you can use <code>JobTag</code> to identify the type of document that the completion notification
     * corresponds to (such as a tax form or a receipt).
     * </p>
     * 
     * @return An identifier that you specify that's included in the completion notification published to the Amazon SNS
     *         topic. For example, you can use <code>JobTag</code> to identify the type of document that the completion
     *         notification corresponds to (such as a tax form or a receipt).
     */
    public String jobTag() {
        return jobTag;
    }

    /**
     * <p>
     * The Amazon SNS topic ARN that you want Amazon Textract to publish the completion status of the operation to.
     * </p>
     * 
     * @return The Amazon SNS topic ARN that you want Amazon Textract to publish the completion status of the operation
     *         to.
     */
    public NotificationChannel notificationChannel() {
        return notificationChannel;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(documentLocation());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(jobTag());
        hashCode = 31 * hashCode + Objects.hashCode(notificationChannel());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StartDocumentTextDetectionRequest)) {
            return false;
        }
        StartDocumentTextDetectionRequest other = (StartDocumentTextDetectionRequest) obj;
        return Objects.equals(documentLocation(), other.documentLocation())
                && Objects.equals(clientRequestToken(), other.clientRequestToken()) && Objects.equals(jobTag(), other.jobTag())
                && Objects.equals(notificationChannel(), other.notificationChannel());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("StartDocumentTextDetectionRequest").add("DocumentLocation", documentLocation())
                .add("ClientRequestToken", clientRequestToken()).add("JobTag", jobTag())
                .add("NotificationChannel", notificationChannel()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DocumentLocation":
            return Optional.ofNullable(clazz.cast(documentLocation()));
        case "ClientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "JobTag":
            return Optional.ofNullable(clazz.cast(jobTag()));
        case "NotificationChannel":
            return Optional.ofNullable(clazz.cast(notificationChannel()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StartDocumentTextDetectionRequest, T> g) {
        return obj -> g.apply((StartDocumentTextDetectionRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends TextractRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, StartDocumentTextDetectionRequest> {
        /**
         * <p>
         * The location of the document to be processed.
         * </p>
         * 
         * @param documentLocation
         *        The location of the document to be processed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentLocation(DocumentLocation documentLocation);

        /**
         * <p>
         * The location of the document to be processed.
         * </p>
         * This is a convenience that creates an instance of the {@link DocumentLocation.Builder} avoiding the need to
         * create one manually via {@link DocumentLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link DocumentLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #documentLocation(DocumentLocation)}.
         * 
         * @param documentLocation
         *        a consumer that will call methods on {@link DocumentLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #documentLocation(DocumentLocation)
         */
        default Builder documentLocation(Consumer<DocumentLocation.Builder> documentLocation) {
            return documentLocation(DocumentLocation.builder().applyMutation(documentLocation).build());
        }

        /**
         * <p>
         * The idempotent token that's used to identify the start request. If you use the same token with multiple
         * <code>StartDocumentTextDetection</code> requests, the same <code>JobId</code> is returned. Use
         * <code>ClientRequestToken</code> to prevent the same job from being accidentally started more than once. For
         * more information, see <a href="https://docs.aws.amazon.com/textract/latest/dg/api-async.html">Calling Amazon
         * Textract Asynchronous Operations</a>.
         * </p>
         * 
         * @param clientRequestToken
         *        The idempotent token that's used to identify the start request. If you use the same token with
         *        multiple <code>StartDocumentTextDetection</code> requests, the same <code>JobId</code> is returned.
         *        Use <code>ClientRequestToken</code> to prevent the same job from being accidentally started more than
         *        once. For more information, see <a
         *        href="https://docs.aws.amazon.com/textract/latest/dg/api-async.html">Calling Amazon Textract
         *        Asynchronous Operations</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * An identifier that you specify that's included in the completion notification published to the Amazon SNS
         * topic. For example, you can use <code>JobTag</code> to identify the type of document that the completion
         * notification corresponds to (such as a tax form or a receipt).
         * </p>
         * 
         * @param jobTag
         *        An identifier that you specify that's included in the completion notification published to the Amazon
         *        SNS topic. For example, you can use <code>JobTag</code> to identify the type of document that the
         *        completion notification corresponds to (such as a tax form or a receipt).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobTag(String jobTag);

        /**
         * <p>
         * The Amazon SNS topic ARN that you want Amazon Textract to publish the completion status of the operation to.
         * </p>
         * 
         * @param notificationChannel
         *        The Amazon SNS topic ARN that you want Amazon Textract to publish the completion status of the
         *        operation to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationChannel(NotificationChannel notificationChannel);

        /**
         * <p>
         * The Amazon SNS topic ARN that you want Amazon Textract to publish the completion status of the operation to.
         * </p>
         * This is a convenience that creates an instance of the {@link NotificationChannel.Builder} avoiding the need
         * to create one manually via {@link NotificationChannel#builder()}.
         *
         * When the {@link Consumer} completes, {@link NotificationChannel.Builder#build()} is called immediately and
         * its result is passed to {@link #notificationChannel(NotificationChannel)}.
         * 
         * @param notificationChannel
         *        a consumer that will call methods on {@link NotificationChannel.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notificationChannel(NotificationChannel)
         */
        default Builder notificationChannel(Consumer<NotificationChannel.Builder> notificationChannel) {
            return notificationChannel(NotificationChannel.builder().applyMutation(notificationChannel).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends TextractRequest.BuilderImpl implements Builder {
        private DocumentLocation documentLocation;

        private String clientRequestToken;

        private String jobTag;

        private NotificationChannel notificationChannel;

        private BuilderImpl() {
        }

        private BuilderImpl(StartDocumentTextDetectionRequest model) {
            super(model);
            documentLocation(model.documentLocation);
            clientRequestToken(model.clientRequestToken);
            jobTag(model.jobTag);
            notificationChannel(model.notificationChannel);
        }

        public final DocumentLocation.Builder getDocumentLocation() {
            return documentLocation != null ? documentLocation.toBuilder() : null;
        }

        @Override
        public final Builder documentLocation(DocumentLocation documentLocation) {
            this.documentLocation = documentLocation;
            return this;
        }

        public final void setDocumentLocation(DocumentLocation.BuilderImpl documentLocation) {
            this.documentLocation = documentLocation != null ? documentLocation.build() : null;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        public final String getJobTag() {
            return jobTag;
        }

        @Override
        public final Builder jobTag(String jobTag) {
            this.jobTag = jobTag;
            return this;
        }

        public final void setJobTag(String jobTag) {
            this.jobTag = jobTag;
        }

        public final NotificationChannel.Builder getNotificationChannel() {
            return notificationChannel != null ? notificationChannel.toBuilder() : null;
        }

        @Override
        public final Builder notificationChannel(NotificationChannel notificationChannel) {
            this.notificationChannel = notificationChannel;
            return this;
        }

        public final void setNotificationChannel(NotificationChannel.BuilderImpl notificationChannel) {
            this.notificationChannel = notificationChannel != null ? notificationChannel.build() : null;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public StartDocumentTextDetectionRequest build() {
            return new StartDocumentTextDetectionRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
