/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.textract.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A <code>Block</code> represents items that are recognized in a document within a group of pixels close to each other.
 * The information returned in a <code>Block</code> object depends on the type of operation. In text detection for
 * documents (for example <a>DetectDocumentText</a>), you get information about the detected words and lines of text. In
 * text analysis (for example <a>AnalyzeDocument</a>), you can also get information about the fields, tables, and
 * selection elements that are detected in the document.
 * </p>
 * <p>
 * An array of <code>Block</code> objects is returned by both synchronous and asynchronous operations. In synchronous
 * operations, such as <a>DetectDocumentText</a>, the array of <code>Block</code> objects is the entire set of results.
 * In asynchronous operations, such as <a>GetDocumentAnalysis</a>, the array is returned over one or more responses.
 * </p>
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works.html">How Amazon
 * Textract Works</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Block implements SdkPojo, Serializable, ToCopyableBuilder<Block.Builder, Block> {
    private static final SdkField<String> BLOCK_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Block::blockTypeAsString)).setter(setter(Builder::blockType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BlockType").build()).build();

    private static final SdkField<Float> CONFIDENCE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .getter(getter(Block::confidence)).setter(setter(Builder::confidence))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Confidence").build()).build();

    private static final SdkField<String> TEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Block::text)).setter(setter(Builder::text))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Text").build()).build();

    private static final SdkField<Integer> ROW_INDEX_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Block::rowIndex)).setter(setter(Builder::rowIndex))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RowIndex").build()).build();

    private static final SdkField<Integer> COLUMN_INDEX_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Block::columnIndex)).setter(setter(Builder::columnIndex))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ColumnIndex").build()).build();

    private static final SdkField<Integer> ROW_SPAN_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Block::rowSpan)).setter(setter(Builder::rowSpan))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RowSpan").build()).build();

    private static final SdkField<Integer> COLUMN_SPAN_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Block::columnSpan)).setter(setter(Builder::columnSpan))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ColumnSpan").build()).build();

    private static final SdkField<Geometry> GEOMETRY_FIELD = SdkField.<Geometry> builder(MarshallingType.SDK_POJO)
            .getter(getter(Block::geometry)).setter(setter(Builder::geometry)).constructor(Geometry::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Geometry").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).getter(getter(Block::id))
            .setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<List<Relationship>> RELATIONSHIPS_FIELD = SdkField
            .<List<Relationship>> builder(MarshallingType.LIST)
            .getter(getter(Block::relationships))
            .setter(setter(Builder::relationships))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Relationships").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Relationship> builder(MarshallingType.SDK_POJO)
                                            .constructor(Relationship::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ENTITY_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(Block::entityTypesAsStrings))
            .setter(setter(Builder::entityTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EntityTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SELECTION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Block::selectionStatusAsString)).setter(setter(Builder::selectionStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SelectionStatus").build()).build();

    private static final SdkField<Integer> PAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Block::page)).setter(setter(Builder::page))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Page").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BLOCK_TYPE_FIELD,
            CONFIDENCE_FIELD, TEXT_FIELD, ROW_INDEX_FIELD, COLUMN_INDEX_FIELD, ROW_SPAN_FIELD, COLUMN_SPAN_FIELD, GEOMETRY_FIELD,
            ID_FIELD, RELATIONSHIPS_FIELD, ENTITY_TYPES_FIELD, SELECTION_STATUS_FIELD, PAGE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String blockType;

    private final Float confidence;

    private final String text;

    private final Integer rowIndex;

    private final Integer columnIndex;

    private final Integer rowSpan;

    private final Integer columnSpan;

    private final Geometry geometry;

    private final String id;

    private final List<Relationship> relationships;

    private final List<String> entityTypes;

    private final String selectionStatus;

    private final Integer page;

    private Block(BuilderImpl builder) {
        this.blockType = builder.blockType;
        this.confidence = builder.confidence;
        this.text = builder.text;
        this.rowIndex = builder.rowIndex;
        this.columnIndex = builder.columnIndex;
        this.rowSpan = builder.rowSpan;
        this.columnSpan = builder.columnSpan;
        this.geometry = builder.geometry;
        this.id = builder.id;
        this.relationships = builder.relationships;
        this.entityTypes = builder.entityTypes;
        this.selectionStatus = builder.selectionStatus;
        this.page = builder.page;
    }

    /**
     * <p>
     * The type of text item that's recognized. In operations for text detection, the following types are returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document page.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script characters that
     * aren't separated by spaces.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     * </p>
     * </li>
     * </ul>
     * <p>
     * In text analysis operations, the following types are returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document page.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's detected on a
     * document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET object is a KEY
     * <code>Block</code> object or a VALUE <code>Block</code> object.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script characters
     * that aren't separated by spaces.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two or more
     * rows or columns, with a cell span of one row and one column each.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the text in the
     * cell.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box that's
     * detected on a document page. Use the value of <code>SelectionStatus</code> to determine the status of the
     * selection element.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #blockType} will
     * return {@link BlockType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #blockTypeAsString}.
     * </p>
     * 
     * @return The type of text item that's recognized. In operations for text detection, the following types are
     *         returned:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document
     *         page.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script characters
     *         that aren't separated by spaces.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         In text analysis operations, the following types are returned:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document page.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's
     *         detected on a document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET object
     *         is a KEY <code>Block</code> object or a VALUE <code>Block</code> object.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script
     *         characters that aren't separated by spaces.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two or
     *         more rows or columns, with a cell span of one row and one column each.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the text
     *         in the cell.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box
     *         that's detected on a document page. Use the value of <code>SelectionStatus</code> to determine the status
     *         of the selection element.
     *         </p>
     *         </li>
     * @see BlockType
     */
    public BlockType blockType() {
        return BlockType.fromValue(blockType);
    }

    /**
     * <p>
     * The type of text item that's recognized. In operations for text detection, the following types are returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document page.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script characters that
     * aren't separated by spaces.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     * </p>
     * </li>
     * </ul>
     * <p>
     * In text analysis operations, the following types are returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document page.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's detected on a
     * document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET object is a KEY
     * <code>Block</code> object or a VALUE <code>Block</code> object.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script characters
     * that aren't separated by spaces.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two or more
     * rows or columns, with a cell span of one row and one column each.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the text in the
     * cell.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box that's
     * detected on a document page. Use the value of <code>SelectionStatus</code> to determine the status of the
     * selection element.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #blockType} will
     * return {@link BlockType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #blockTypeAsString}.
     * </p>
     * 
     * @return The type of text item that's recognized. In operations for text detection, the following types are
     *         returned:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document
     *         page.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script characters
     *         that aren't separated by spaces.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         In text analysis operations, the following types are returned:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document page.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's
     *         detected on a document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET object
     *         is a KEY <code>Block</code> object or a VALUE <code>Block</code> object.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script
     *         characters that aren't separated by spaces.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two or
     *         more rows or columns, with a cell span of one row and one column each.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the text
     *         in the cell.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box
     *         that's detected on a document page. Use the value of <code>SelectionStatus</code> to determine the status
     *         of the selection element.
     *         </p>
     *         </li>
     * @see BlockType
     */
    public String blockTypeAsString() {
        return blockType;
    }

    /**
     * <p>
     * The confidence score that Amazon Textract has in the accuracy of the recognized text and the accuracy of the
     * geometry points around the recognized text.
     * </p>
     * 
     * @return The confidence score that Amazon Textract has in the accuracy of the recognized text and the accuracy of
     *         the geometry points around the recognized text.
     */
    public Float confidence() {
        return confidence;
    }

    /**
     * <p>
     * The word or line of text that's recognized by Amazon Textract.
     * </p>
     * 
     * @return The word or line of text that's recognized by Amazon Textract.
     */
    public String text() {
        return text;
    }

    /**
     * <p>
     * The row in which a table cell is located. The first row position is 1. <code>RowIndex</code> isn't returned by
     * <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
     * </p>
     * 
     * @return The row in which a table cell is located. The first row position is 1. <code>RowIndex</code> isn't
     *         returned by <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
     */
    public Integer rowIndex() {
        return rowIndex;
    }

    /**
     * <p>
     * The column in which a table cell appears. The first column position is 1. <code>ColumnIndex</code> isn't returned
     * by <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
     * </p>
     * 
     * @return The column in which a table cell appears. The first column position is 1. <code>ColumnIndex</code> isn't
     *         returned by <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
     */
    public Integer columnIndex() {
        return columnIndex;
    }

    /**
     * <p>
     * The number of rows that a table cell spans. Currently this value is always 1, even if the number of rows spanned
     * is greater than 1. <code>RowSpan</code> isn't returned by <code>DetectDocumentText</code> and
     * <code>GetDocumentTextDetection</code>.
     * </p>
     * 
     * @return The number of rows that a table cell spans. Currently this value is always 1, even if the number of rows
     *         spanned is greater than 1. <code>RowSpan</code> isn't returned by <code>DetectDocumentText</code> and
     *         <code>GetDocumentTextDetection</code>.
     */
    public Integer rowSpan() {
        return rowSpan;
    }

    /**
     * <p>
     * The number of columns that a table cell spans. Currently this value is always 1, even if the number of columns
     * spanned is greater than 1. <code>ColumnSpan</code> isn't returned by <code>DetectDocumentText</code> and
     * <code>GetDocumentTextDetection</code>.
     * </p>
     * 
     * @return The number of columns that a table cell spans. Currently this value is always 1, even if the number of
     *         columns spanned is greater than 1. <code>ColumnSpan</code> isn't returned by
     *         <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
     */
    public Integer columnSpan() {
        return columnSpan;
    }

    /**
     * <p>
     * The location of the recognized text on the image. It includes an axis-aligned, coarse bounding box that surrounds
     * the text, and a finer-grain polygon for more accurate spatial information.
     * </p>
     * 
     * @return The location of the recognized text on the image. It includes an axis-aligned, coarse bounding box that
     *         surrounds the text, and a finer-grain polygon for more accurate spatial information.
     */
    public Geometry geometry() {
        return geometry;
    }

    /**
     * <p>
     * The identifier for the recognized text. The identifier is only unique for a single operation.
     * </p>
     * 
     * @return The identifier for the recognized text. The identifier is only unique for a single operation.
     */
    public String id() {
        return id;
    }

    /**
     * Returns true if the Relationships property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRelationships() {
        return relationships != null && !(relationships instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of child blocks of the current block. For example, a LINE object has child blocks for each WORD block
     * that's part of the line of text. There aren't Relationship objects in the list for relationships that don't
     * exist, such as when the current block has no child blocks. The list size can be the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * 0 - The block has no child blocks.
     * </p>
     * </li>
     * <li>
     * <p>
     * 1 - The block has child blocks.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRelationships()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of child blocks of the current block. For example, a LINE object has child blocks for each WORD
     *         block that's part of the line of text. There aren't Relationship objects in the list for relationships
     *         that don't exist, such as when the current block has no child blocks. The list size can be the
     *         following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         0 - The block has no child blocks.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         1 - The block has child blocks.
     *         </p>
     *         </li>
     */
    public List<Relationship> relationships() {
        return relationships;
    }

    /**
     * <p>
     * The type of entity. The following can be returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>KEY</i> - An identifier for a field on the document.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>VALUE</i> - The field text.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
     * <code>GetDocumentTextDetection</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEntityTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The type of entity. The following can be returned:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>KEY</i> - An identifier for a field on the document.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>VALUE</i> - The field text.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
     *         <code>GetDocumentTextDetection</code>.
     */
    public List<EntityType> entityTypes() {
        return EntityTypesCopier.copyStringToEnum(entityTypes);
    }

    /**
     * Returns true if the EntityTypes property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasEntityTypes() {
        return entityTypes != null && !(entityTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type of entity. The following can be returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>KEY</i> - An identifier for a field on the document.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>VALUE</i> - The field text.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
     * <code>GetDocumentTextDetection</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEntityTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The type of entity. The following can be returned:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>KEY</i> - An identifier for a field on the document.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>VALUE</i> - The field text.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
     *         <code>GetDocumentTextDetection</code>.
     */
    public List<String> entityTypesAsStrings() {
        return entityTypes;
    }

    /**
     * <p>
     * The selection status of a selection element, such as an option button or check box.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #selectionStatus}
     * will return {@link SelectionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #selectionStatusAsString}.
     * </p>
     * 
     * @return The selection status of a selection element, such as an option button or check box.
     * @see SelectionStatus
     */
    public SelectionStatus selectionStatus() {
        return SelectionStatus.fromValue(selectionStatus);
    }

    /**
     * <p>
     * The selection status of a selection element, such as an option button or check box.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #selectionStatus}
     * will return {@link SelectionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #selectionStatusAsString}.
     * </p>
     * 
     * @return The selection status of a selection element, such as an option button or check box.
     * @see SelectionStatus
     */
    public String selectionStatusAsString() {
        return selectionStatus;
    }

    /**
     * <p>
     * The page on which a block was detected. <code>Page</code> is returned by asynchronous operations. Page values
     * greater than 1 are only returned for multipage documents that are in PDF format. A scanned image (JPEG/PNG), even
     * if it contains multiple document pages, is considered to be a single-page document. The value of
     * <code>Page</code> is always 1. Synchronous operations don't return <code>Page</code> because every input document
     * is considered to be a single-page document.
     * </p>
     * 
     * @return The page on which a block was detected. <code>Page</code> is returned by asynchronous operations. Page
     *         values greater than 1 are only returned for multipage documents that are in PDF format. A scanned image
     *         (JPEG/PNG), even if it contains multiple document pages, is considered to be a single-page document. The
     *         value of <code>Page</code> is always 1. Synchronous operations don't return <code>Page</code> because
     *         every input document is considered to be a single-page document.
     */
    public Integer page() {
        return page;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(blockTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(confidence());
        hashCode = 31 * hashCode + Objects.hashCode(text());
        hashCode = 31 * hashCode + Objects.hashCode(rowIndex());
        hashCode = 31 * hashCode + Objects.hashCode(columnIndex());
        hashCode = 31 * hashCode + Objects.hashCode(rowSpan());
        hashCode = 31 * hashCode + Objects.hashCode(columnSpan());
        hashCode = 31 * hashCode + Objects.hashCode(geometry());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(relationships());
        hashCode = 31 * hashCode + Objects.hashCode(entityTypesAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(selectionStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(page());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Block)) {
            return false;
        }
        Block other = (Block) obj;
        return Objects.equals(blockTypeAsString(), other.blockTypeAsString()) && Objects.equals(confidence(), other.confidence())
                && Objects.equals(text(), other.text()) && Objects.equals(rowIndex(), other.rowIndex())
                && Objects.equals(columnIndex(), other.columnIndex()) && Objects.equals(rowSpan(), other.rowSpan())
                && Objects.equals(columnSpan(), other.columnSpan()) && Objects.equals(geometry(), other.geometry())
                && Objects.equals(id(), other.id()) && Objects.equals(relationships(), other.relationships())
                && Objects.equals(entityTypesAsStrings(), other.entityTypesAsStrings())
                && Objects.equals(selectionStatusAsString(), other.selectionStatusAsString())
                && Objects.equals(page(), other.page());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Block").add("BlockType", blockTypeAsString()).add("Confidence", confidence())
                .add("Text", text()).add("RowIndex", rowIndex()).add("ColumnIndex", columnIndex()).add("RowSpan", rowSpan())
                .add("ColumnSpan", columnSpan()).add("Geometry", geometry()).add("Id", id())
                .add("Relationships", relationships()).add("EntityTypes", entityTypesAsStrings())
                .add("SelectionStatus", selectionStatusAsString()).add("Page", page()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BlockType":
            return Optional.ofNullable(clazz.cast(blockTypeAsString()));
        case "Confidence":
            return Optional.ofNullable(clazz.cast(confidence()));
        case "Text":
            return Optional.ofNullable(clazz.cast(text()));
        case "RowIndex":
            return Optional.ofNullable(clazz.cast(rowIndex()));
        case "ColumnIndex":
            return Optional.ofNullable(clazz.cast(columnIndex()));
        case "RowSpan":
            return Optional.ofNullable(clazz.cast(rowSpan()));
        case "ColumnSpan":
            return Optional.ofNullable(clazz.cast(columnSpan()));
        case "Geometry":
            return Optional.ofNullable(clazz.cast(geometry()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Relationships":
            return Optional.ofNullable(clazz.cast(relationships()));
        case "EntityTypes":
            return Optional.ofNullable(clazz.cast(entityTypesAsStrings()));
        case "SelectionStatus":
            return Optional.ofNullable(clazz.cast(selectionStatusAsString()));
        case "Page":
            return Optional.ofNullable(clazz.cast(page()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Block, T> g) {
        return obj -> g.apply((Block) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Block> {
        /**
         * <p>
         * The type of text item that's recognized. In operations for text detection, the following types are returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document page.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script characters
         * that aren't separated by spaces.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         * </p>
         * </li>
         * </ul>
         * <p>
         * In text analysis operations, the following types are returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document page.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's detected on
         * a document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET object is a KEY
         * <code>Block</code> object or a VALUE <code>Block</code> object.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script
         * characters that aren't separated by spaces.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two or more
         * rows or columns, with a cell span of one row and one column each.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the text in
         * the cell.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box that's
         * detected on a document page. Use the value of <code>SelectionStatus</code> to determine the status of the
         * selection element.
         * </p>
         * </li>
         * </ul>
         * 
         * @param blockType
         *        The type of text item that's recognized. In operations for text detection, the following types are
         *        returned:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document
         *        page.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script
         *        characters that aren't separated by spaces.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        In text analysis operations, the following types are returned:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document
         *        page.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's
         *        detected on a document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET
         *        object is a KEY <code>Block</code> object or a VALUE <code>Block</code> object.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script
         *        characters that aren't separated by spaces.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two
         *        or more rows or columns, with a cell span of one row and one column each.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the
         *        text in the cell.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box
         *        that's detected on a document page. Use the value of <code>SelectionStatus</code> to determine the
         *        status of the selection element.
         *        </p>
         *        </li>
         * @see BlockType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BlockType
         */
        Builder blockType(String blockType);

        /**
         * <p>
         * The type of text item that's recognized. In operations for text detection, the following types are returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document page.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script characters
         * that aren't separated by spaces.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         * </p>
         * </li>
         * </ul>
         * <p>
         * In text analysis operations, the following types are returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document page.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's detected on
         * a document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET object is a KEY
         * <code>Block</code> object or a VALUE <code>Block</code> object.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script
         * characters that aren't separated by spaces.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two or more
         * rows or columns, with a cell span of one row and one column each.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the text in
         * the cell.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box that's
         * detected on a document page. Use the value of <code>SelectionStatus</code> to determine the status of the
         * selection element.
         * </p>
         * </li>
         * </ul>
         * 
         * @param blockType
         *        The type of text item that's recognized. In operations for text detection, the following types are
         *        returned:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document
         *        page.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script
         *        characters that aren't separated by spaces.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        In text analysis operations, the following types are returned:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document
         *        page.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's
         *        detected on a document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET
         *        object is a KEY <code>Block</code> object or a VALUE <code>Block</code> object.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script
         *        characters that aren't separated by spaces.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two
         *        or more rows or columns, with a cell span of one row and one column each.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the
         *        text in the cell.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box
         *        that's detected on a document page. Use the value of <code>SelectionStatus</code> to determine the
         *        status of the selection element.
         *        </p>
         *        </li>
         * @see BlockType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BlockType
         */
        Builder blockType(BlockType blockType);

        /**
         * <p>
         * The confidence score that Amazon Textract has in the accuracy of the recognized text and the accuracy of the
         * geometry points around the recognized text.
         * </p>
         * 
         * @param confidence
         *        The confidence score that Amazon Textract has in the accuracy of the recognized text and the accuracy
         *        of the geometry points around the recognized text.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confidence(Float confidence);

        /**
         * <p>
         * The word or line of text that's recognized by Amazon Textract.
         * </p>
         * 
         * @param text
         *        The word or line of text that's recognized by Amazon Textract.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder text(String text);

        /**
         * <p>
         * The row in which a table cell is located. The first row position is 1. <code>RowIndex</code> isn't returned
         * by <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param rowIndex
         *        The row in which a table cell is located. The first row position is 1. <code>RowIndex</code> isn't
         *        returned by <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rowIndex(Integer rowIndex);

        /**
         * <p>
         * The column in which a table cell appears. The first column position is 1. <code>ColumnIndex</code> isn't
         * returned by <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param columnIndex
         *        The column in which a table cell appears. The first column position is 1. <code>ColumnIndex</code> 
         *        isn't returned by <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder columnIndex(Integer columnIndex);

        /**
         * <p>
         * The number of rows that a table cell spans. Currently this value is always 1, even if the number of rows
         * spanned is greater than 1. <code>RowSpan</code> isn't returned by <code>DetectDocumentText</code> and
         * <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param rowSpan
         *        The number of rows that a table cell spans. Currently this value is always 1, even if the number of
         *        rows spanned is greater than 1. <code>RowSpan</code> isn't returned by <code>DetectDocumentText</code>
         *        and <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rowSpan(Integer rowSpan);

        /**
         * <p>
         * The number of columns that a table cell spans. Currently this value is always 1, even if the number of
         * columns spanned is greater than 1. <code>ColumnSpan</code> isn't returned by <code>DetectDocumentText</code>
         * and <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param columnSpan
         *        The number of columns that a table cell spans. Currently this value is always 1, even if the number of
         *        columns spanned is greater than 1. <code>ColumnSpan</code> isn't returned by
         *        <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder columnSpan(Integer columnSpan);

        /**
         * <p>
         * The location of the recognized text on the image. It includes an axis-aligned, coarse bounding box that
         * surrounds the text, and a finer-grain polygon for more accurate spatial information.
         * </p>
         * 
         * @param geometry
         *        The location of the recognized text on the image. It includes an axis-aligned, coarse bounding box
         *        that surrounds the text, and a finer-grain polygon for more accurate spatial information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder geometry(Geometry geometry);

        /**
         * <p>
         * The location of the recognized text on the image. It includes an axis-aligned, coarse bounding box that
         * surrounds the text, and a finer-grain polygon for more accurate spatial information.
         * </p>
         * This is a convenience that creates an instance of the {@link Geometry.Builder} avoiding the need to create
         * one manually via {@link Geometry#builder()}.
         *
         * When the {@link Consumer} completes, {@link Geometry.Builder#build()} is called immediately and its result is
         * passed to {@link #geometry(Geometry)}.
         * 
         * @param geometry
         *        a consumer that will call methods on {@link Geometry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #geometry(Geometry)
         */
        default Builder geometry(Consumer<Geometry.Builder> geometry) {
            return geometry(Geometry.builder().applyMutation(geometry).build());
        }

        /**
         * <p>
         * The identifier for the recognized text. The identifier is only unique for a single operation.
         * </p>
         * 
         * @param id
         *        The identifier for the recognized text. The identifier is only unique for a single operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * A list of child blocks of the current block. For example, a LINE object has child blocks for each WORD block
         * that's part of the line of text. There aren't Relationship objects in the list for relationships that don't
         * exist, such as when the current block has no child blocks. The list size can be the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * 0 - The block has no child blocks.
         * </p>
         * </li>
         * <li>
         * <p>
         * 1 - The block has child blocks.
         * </p>
         * </li>
         * </ul>
         * 
         * @param relationships
         *        A list of child blocks of the current block. For example, a LINE object has child blocks for each WORD
         *        block that's part of the line of text. There aren't Relationship objects in the list for relationships
         *        that don't exist, such as when the current block has no child blocks. The list size can be the
         *        following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        0 - The block has no child blocks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        1 - The block has child blocks.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relationships(Collection<Relationship> relationships);

        /**
         * <p>
         * A list of child blocks of the current block. For example, a LINE object has child blocks for each WORD block
         * that's part of the line of text. There aren't Relationship objects in the list for relationships that don't
         * exist, such as when the current block has no child blocks. The list size can be the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * 0 - The block has no child blocks.
         * </p>
         * </li>
         * <li>
         * <p>
         * 1 - The block has child blocks.
         * </p>
         * </li>
         * </ul>
         * 
         * @param relationships
         *        A list of child blocks of the current block. For example, a LINE object has child blocks for each WORD
         *        block that's part of the line of text. There aren't Relationship objects in the list for relationships
         *        that don't exist, such as when the current block has no child blocks. The list size can be the
         *        following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        0 - The block has no child blocks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        1 - The block has child blocks.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relationships(Relationship... relationships);

        /**
         * <p>
         * A list of child blocks of the current block. For example, a LINE object has child blocks for each WORD block
         * that's part of the line of text. There aren't Relationship objects in the list for relationships that don't
         * exist, such as when the current block has no child blocks. The list size can be the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * 0 - The block has no child blocks.
         * </p>
         * </li>
         * <li>
         * <p>
         * 1 - The block has child blocks.
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link List<Relationship>.Builder} avoiding the need to
         * create one manually via {@link List<Relationship>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Relationship>.Builder#build()} is called immediately and its
         * result is passed to {@link #relationships(List<Relationship>)}.
         * 
         * @param relationships
         *        a consumer that will call methods on {@link List<Relationship>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #relationships(List<Relationship>)
         */
        Builder relationships(Consumer<Relationship.Builder>... relationships);

        /**
         * <p>
         * The type of entity. The following can be returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>KEY</i> - An identifier for a field on the document.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>VALUE</i> - The field text.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         * <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param entityTypes
         *        The type of entity. The following can be returned:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>KEY</i> - An identifier for a field on the document.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>VALUE</i> - The field text.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         *        <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityTypesWithStrings(Collection<String> entityTypes);

        /**
         * <p>
         * The type of entity. The following can be returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>KEY</i> - An identifier for a field on the document.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>VALUE</i> - The field text.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         * <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param entityTypes
         *        The type of entity. The following can be returned:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>KEY</i> - An identifier for a field on the document.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>VALUE</i> - The field text.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         *        <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityTypesWithStrings(String... entityTypes);

        /**
         * <p>
         * The type of entity. The following can be returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>KEY</i> - An identifier for a field on the document.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>VALUE</i> - The field text.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         * <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param entityTypes
         *        The type of entity. The following can be returned:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>KEY</i> - An identifier for a field on the document.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>VALUE</i> - The field text.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         *        <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityTypes(Collection<EntityType> entityTypes);

        /**
         * <p>
         * The type of entity. The following can be returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>KEY</i> - An identifier for a field on the document.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>VALUE</i> - The field text.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         * <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param entityTypes
         *        The type of entity. The following can be returned:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>KEY</i> - An identifier for a field on the document.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>VALUE</i> - The field text.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         *        <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityTypes(EntityType... entityTypes);

        /**
         * <p>
         * The selection status of a selection element, such as an option button or check box.
         * </p>
         * 
         * @param selectionStatus
         *        The selection status of a selection element, such as an option button or check box.
         * @see SelectionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SelectionStatus
         */
        Builder selectionStatus(String selectionStatus);

        /**
         * <p>
         * The selection status of a selection element, such as an option button or check box.
         * </p>
         * 
         * @param selectionStatus
         *        The selection status of a selection element, such as an option button or check box.
         * @see SelectionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SelectionStatus
         */
        Builder selectionStatus(SelectionStatus selectionStatus);

        /**
         * <p>
         * The page on which a block was detected. <code>Page</code> is returned by asynchronous operations. Page values
         * greater than 1 are only returned for multipage documents that are in PDF format. A scanned image (JPEG/PNG),
         * even if it contains multiple document pages, is considered to be a single-page document. The value of
         * <code>Page</code> is always 1. Synchronous operations don't return <code>Page</code> because every input
         * document is considered to be a single-page document.
         * </p>
         * 
         * @param page
         *        The page on which a block was detected. <code>Page</code> is returned by asynchronous operations. Page
         *        values greater than 1 are only returned for multipage documents that are in PDF format. A scanned
         *        image (JPEG/PNG), even if it contains multiple document pages, is considered to be a single-page
         *        document. The value of <code>Page</code> is always 1. Synchronous operations don't return
         *        <code>Page</code> because every input document is considered to be a single-page document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder page(Integer page);
    }

    static final class BuilderImpl implements Builder {
        private String blockType;

        private Float confidence;

        private String text;

        private Integer rowIndex;

        private Integer columnIndex;

        private Integer rowSpan;

        private Integer columnSpan;

        private Geometry geometry;

        private String id;

        private List<Relationship> relationships = DefaultSdkAutoConstructList.getInstance();

        private List<String> entityTypes = DefaultSdkAutoConstructList.getInstance();

        private String selectionStatus;

        private Integer page;

        private BuilderImpl() {
        }

        private BuilderImpl(Block model) {
            blockType(model.blockType);
            confidence(model.confidence);
            text(model.text);
            rowIndex(model.rowIndex);
            columnIndex(model.columnIndex);
            rowSpan(model.rowSpan);
            columnSpan(model.columnSpan);
            geometry(model.geometry);
            id(model.id);
            relationships(model.relationships);
            entityTypesWithStrings(model.entityTypes);
            selectionStatus(model.selectionStatus);
            page(model.page);
        }

        public final String getBlockTypeAsString() {
            return blockType;
        }

        @Override
        public final Builder blockType(String blockType) {
            this.blockType = blockType;
            return this;
        }

        @Override
        public final Builder blockType(BlockType blockType) {
            this.blockType(blockType == null ? null : blockType.toString());
            return this;
        }

        public final void setBlockType(String blockType) {
            this.blockType = blockType;
        }

        public final Float getConfidence() {
            return confidence;
        }

        @Override
        public final Builder confidence(Float confidence) {
            this.confidence = confidence;
            return this;
        }

        public final void setConfidence(Float confidence) {
            this.confidence = confidence;
        }

        public final String getText() {
            return text;
        }

        @Override
        public final Builder text(String text) {
            this.text = text;
            return this;
        }

        public final void setText(String text) {
            this.text = text;
        }

        public final Integer getRowIndex() {
            return rowIndex;
        }

        @Override
        public final Builder rowIndex(Integer rowIndex) {
            this.rowIndex = rowIndex;
            return this;
        }

        public final void setRowIndex(Integer rowIndex) {
            this.rowIndex = rowIndex;
        }

        public final Integer getColumnIndex() {
            return columnIndex;
        }

        @Override
        public final Builder columnIndex(Integer columnIndex) {
            this.columnIndex = columnIndex;
            return this;
        }

        public final void setColumnIndex(Integer columnIndex) {
            this.columnIndex = columnIndex;
        }

        public final Integer getRowSpan() {
            return rowSpan;
        }

        @Override
        public final Builder rowSpan(Integer rowSpan) {
            this.rowSpan = rowSpan;
            return this;
        }

        public final void setRowSpan(Integer rowSpan) {
            this.rowSpan = rowSpan;
        }

        public final Integer getColumnSpan() {
            return columnSpan;
        }

        @Override
        public final Builder columnSpan(Integer columnSpan) {
            this.columnSpan = columnSpan;
            return this;
        }

        public final void setColumnSpan(Integer columnSpan) {
            this.columnSpan = columnSpan;
        }

        public final Geometry.Builder getGeometry() {
            return geometry != null ? geometry.toBuilder() : null;
        }

        @Override
        public final Builder geometry(Geometry geometry) {
            this.geometry = geometry;
            return this;
        }

        public final void setGeometry(Geometry.BuilderImpl geometry) {
            this.geometry = geometry != null ? geometry.build() : null;
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final Collection<Relationship.Builder> getRelationships() {
            return relationships != null ? relationships.stream().map(Relationship::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder relationships(Collection<Relationship> relationships) {
            this.relationships = RelationshipListCopier.copy(relationships);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relationships(Relationship... relationships) {
            relationships(Arrays.asList(relationships));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relationships(Consumer<Relationship.Builder>... relationships) {
            relationships(Stream.of(relationships).map(c -> Relationship.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRelationships(Collection<Relationship.BuilderImpl> relationships) {
            this.relationships = RelationshipListCopier.copyFromBuilder(relationships);
        }

        public final Collection<String> getEntityTypesAsStrings() {
            return entityTypes;
        }

        @Override
        public final Builder entityTypesWithStrings(Collection<String> entityTypes) {
            this.entityTypes = EntityTypesCopier.copy(entityTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entityTypesWithStrings(String... entityTypes) {
            entityTypesWithStrings(Arrays.asList(entityTypes));
            return this;
        }

        @Override
        public final Builder entityTypes(Collection<EntityType> entityTypes) {
            this.entityTypes = EntityTypesCopier.copyEnumToString(entityTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entityTypes(EntityType... entityTypes) {
            entityTypes(Arrays.asList(entityTypes));
            return this;
        }

        public final void setEntityTypesWithStrings(Collection<String> entityTypes) {
            this.entityTypes = EntityTypesCopier.copy(entityTypes);
        }

        public final String getSelectionStatusAsString() {
            return selectionStatus;
        }

        @Override
        public final Builder selectionStatus(String selectionStatus) {
            this.selectionStatus = selectionStatus;
            return this;
        }

        @Override
        public final Builder selectionStatus(SelectionStatus selectionStatus) {
            this.selectionStatus(selectionStatus == null ? null : selectionStatus.toString());
            return this;
        }

        public final void setSelectionStatus(String selectionStatus) {
            this.selectionStatus = selectionStatus;
        }

        public final Integer getPage() {
            return page;
        }

        @Override
        public final Builder page(Integer page) {
            this.page = page;
            return this;
        }

        public final void setPage(Integer page) {
            this.page = page;
        }

        @Override
        public Block build() {
            return new Block(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
