/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.shield.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A summary of information about the attack.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SummarizedAttackVector implements SdkPojo, Serializable,
        ToCopyableBuilder<SummarizedAttackVector.Builder, SummarizedAttackVector> {
    private static final SdkField<String> VECTOR_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SummarizedAttackVector::vectorType)).setter(setter(Builder::vectorType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VectorType").build()).build();

    private static final SdkField<List<SummarizedCounter>> VECTOR_COUNTERS_FIELD = SdkField
            .<List<SummarizedCounter>> builder(MarshallingType.LIST)
            .getter(getter(SummarizedAttackVector::vectorCounters))
            .setter(setter(Builder::vectorCounters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VectorCounters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SummarizedCounter> builder(MarshallingType.SDK_POJO)
                                            .constructor(SummarizedCounter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VECTOR_TYPE_FIELD,
            VECTOR_COUNTERS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String vectorType;

    private final List<SummarizedCounter> vectorCounters;

    private SummarizedAttackVector(BuilderImpl builder) {
        this.vectorType = builder.vectorType;
        this.vectorCounters = builder.vectorCounters;
    }

    /**
     * <p>
     * The attack type, for example, SNMP reflection or SYN flood.
     * </p>
     * 
     * @return The attack type, for example, SNMP reflection or SYN flood.
     */
    public String vectorType() {
        return vectorType;
    }

    /**
     * Returns true if the VectorCounters property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasVectorCounters() {
        return vectorCounters != null && !(vectorCounters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of counters that describe the details of the attack.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasVectorCounters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of counters that describe the details of the attack.
     */
    public List<SummarizedCounter> vectorCounters() {
        return vectorCounters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vectorType());
        hashCode = 31 * hashCode + Objects.hashCode(vectorCounters());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SummarizedAttackVector)) {
            return false;
        }
        SummarizedAttackVector other = (SummarizedAttackVector) obj;
        return Objects.equals(vectorType(), other.vectorType()) && Objects.equals(vectorCounters(), other.vectorCounters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SummarizedAttackVector").add("VectorType", vectorType()).add("VectorCounters", vectorCounters())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VectorType":
            return Optional.ofNullable(clazz.cast(vectorType()));
        case "VectorCounters":
            return Optional.ofNullable(clazz.cast(vectorCounters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SummarizedAttackVector, T> g) {
        return obj -> g.apply((SummarizedAttackVector) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SummarizedAttackVector> {
        /**
         * <p>
         * The attack type, for example, SNMP reflection or SYN flood.
         * </p>
         * 
         * @param vectorType
         *        The attack type, for example, SNMP reflection or SYN flood.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vectorType(String vectorType);

        /**
         * <p>
         * The list of counters that describe the details of the attack.
         * </p>
         * 
         * @param vectorCounters
         *        The list of counters that describe the details of the attack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vectorCounters(Collection<SummarizedCounter> vectorCounters);

        /**
         * <p>
         * The list of counters that describe the details of the attack.
         * </p>
         * 
         * @param vectorCounters
         *        The list of counters that describe the details of the attack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vectorCounters(SummarizedCounter... vectorCounters);

        /**
         * <p>
         * The list of counters that describe the details of the attack.
         * </p>
         * This is a convenience that creates an instance of the {@link List<SummarizedCounter>.Builder} avoiding the
         * need to create one manually via {@link List<SummarizedCounter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<SummarizedCounter>.Builder#build()} is called immediately
         * and its result is passed to {@link #vectorCounters(List<SummarizedCounter>)}.
         * 
         * @param vectorCounters
         *        a consumer that will call methods on {@link List<SummarizedCounter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vectorCounters(List<SummarizedCounter>)
         */
        Builder vectorCounters(Consumer<SummarizedCounter.Builder>... vectorCounters);
    }

    static final class BuilderImpl implements Builder {
        private String vectorType;

        private List<SummarizedCounter> vectorCounters = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SummarizedAttackVector model) {
            vectorType(model.vectorType);
            vectorCounters(model.vectorCounters);
        }

        public final String getVectorType() {
            return vectorType;
        }

        @Override
        public final Builder vectorType(String vectorType) {
            this.vectorType = vectorType;
            return this;
        }

        public final void setVectorType(String vectorType) {
            this.vectorType = vectorType;
        }

        public final Collection<SummarizedCounter.Builder> getVectorCounters() {
            return vectorCounters != null ? vectorCounters.stream().map(SummarizedCounter::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder vectorCounters(Collection<SummarizedCounter> vectorCounters) {
            this.vectorCounters = SummarizedCounterListCopier.copy(vectorCounters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vectorCounters(SummarizedCounter... vectorCounters) {
            vectorCounters(Arrays.asList(vectorCounters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vectorCounters(Consumer<SummarizedCounter.Builder>... vectorCounters) {
            vectorCounters(Stream.of(vectorCounters).map(c -> SummarizedCounter.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setVectorCounters(Collection<SummarizedCounter.BuilderImpl> vectorCounters) {
            this.vectorCounters = SummarizedCounterListCopier.copyFromBuilder(vectorCounters);
        }

        @Override
        public SummarizedAttackVector build() {
            return new SummarizedAttackVector(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
