/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.shield;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.shield.model.AccessDeniedException;
import software.amazon.awssdk.services.shield.model.AccessDeniedForDependencyException;
import software.amazon.awssdk.services.shield.model.AssociateDrtLogBucketRequest;
import software.amazon.awssdk.services.shield.model.AssociateDrtLogBucketResponse;
import software.amazon.awssdk.services.shield.model.AssociateDrtRoleRequest;
import software.amazon.awssdk.services.shield.model.AssociateDrtRoleResponse;
import software.amazon.awssdk.services.shield.model.CreateProtectionRequest;
import software.amazon.awssdk.services.shield.model.CreateProtectionResponse;
import software.amazon.awssdk.services.shield.model.CreateSubscriptionRequest;
import software.amazon.awssdk.services.shield.model.CreateSubscriptionResponse;
import software.amazon.awssdk.services.shield.model.DeleteProtectionRequest;
import software.amazon.awssdk.services.shield.model.DeleteProtectionResponse;
import software.amazon.awssdk.services.shield.model.DescribeAttackRequest;
import software.amazon.awssdk.services.shield.model.DescribeAttackResponse;
import software.amazon.awssdk.services.shield.model.DescribeDrtAccessRequest;
import software.amazon.awssdk.services.shield.model.DescribeDrtAccessResponse;
import software.amazon.awssdk.services.shield.model.DescribeEmergencyContactSettingsRequest;
import software.amazon.awssdk.services.shield.model.DescribeEmergencyContactSettingsResponse;
import software.amazon.awssdk.services.shield.model.DescribeProtectionRequest;
import software.amazon.awssdk.services.shield.model.DescribeProtectionResponse;
import software.amazon.awssdk.services.shield.model.DescribeSubscriptionRequest;
import software.amazon.awssdk.services.shield.model.DescribeSubscriptionResponse;
import software.amazon.awssdk.services.shield.model.DisassociateDrtLogBucketRequest;
import software.amazon.awssdk.services.shield.model.DisassociateDrtLogBucketResponse;
import software.amazon.awssdk.services.shield.model.DisassociateDrtRoleRequest;
import software.amazon.awssdk.services.shield.model.DisassociateDrtRoleResponse;
import software.amazon.awssdk.services.shield.model.GetSubscriptionStateRequest;
import software.amazon.awssdk.services.shield.model.GetSubscriptionStateResponse;
import software.amazon.awssdk.services.shield.model.InternalErrorException;
import software.amazon.awssdk.services.shield.model.InvalidOperationException;
import software.amazon.awssdk.services.shield.model.InvalidPaginationTokenException;
import software.amazon.awssdk.services.shield.model.InvalidParameterException;
import software.amazon.awssdk.services.shield.model.InvalidResourceException;
import software.amazon.awssdk.services.shield.model.LimitsExceededException;
import software.amazon.awssdk.services.shield.model.ListAttacksRequest;
import software.amazon.awssdk.services.shield.model.ListAttacksResponse;
import software.amazon.awssdk.services.shield.model.ListProtectionsRequest;
import software.amazon.awssdk.services.shield.model.ListProtectionsResponse;
import software.amazon.awssdk.services.shield.model.LockedSubscriptionException;
import software.amazon.awssdk.services.shield.model.NoAssociatedRoleException;
import software.amazon.awssdk.services.shield.model.OptimisticLockException;
import software.amazon.awssdk.services.shield.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.shield.model.ResourceNotFoundException;
import software.amazon.awssdk.services.shield.model.ShieldException;
import software.amazon.awssdk.services.shield.model.UpdateEmergencyContactSettingsRequest;
import software.amazon.awssdk.services.shield.model.UpdateEmergencyContactSettingsResponse;
import software.amazon.awssdk.services.shield.model.UpdateSubscriptionRequest;
import software.amazon.awssdk.services.shield.model.UpdateSubscriptionResponse;
import software.amazon.awssdk.services.shield.transform.AssociateDrtLogBucketRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.AssociateDrtRoleRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.CreateProtectionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.CreateSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DeleteProtectionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeAttackRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeDrtAccessRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeEmergencyContactSettingsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeProtectionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DisassociateDrtLogBucketRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DisassociateDrtRoleRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.GetSubscriptionStateRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.ListAttacksRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.ListProtectionsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.UpdateEmergencyContactSettingsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.UpdateSubscriptionRequestMarshaller;

/**
 * Internal implementation of {@link ShieldClient}.
 *
 * @see ShieldClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultShieldClient implements ShieldClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultShieldClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Authorizes the DDoS Response team (DRT) to access the specified Amazon S3 bucket containing your AWS WAF logs.
     * You can associate up to 10 Amazon S3 buckets with your subscription.
     * </p>
     * <p>
     * To use the services of the DRT and make an <code>AssociateDRTLogBucket</code> request, you must be subscribed to
     * the <a href="https://aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>.
     * </p>
     *
     * @param associateDrtLogBucketRequest
     * @return Result of the AssociateDRTLogBucket operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws NoAssociatedRoleException
     *         The ARN of the role that you specifed does not exist.
     * @throws LimitsExceededException
     *         Exception that indicates that the operation would exceed a limit.</p>
     *         <p>
     *         <code>Type</code> is the type of limit that would be exceeded.
     *         </p>
     *         <p>
     *         <code>Limit</code> is the threshold that would be exceeded.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid.
     * @throws AccessDeniedForDependencyException
     *         In order to grant the necessary access to the DDoS Response Team, the user submitting
     *         <code>AssociateDRTRole</code> must have the <code>iam:PassRole</code> permission. This error indicates
     *         the user did not have the appropriate permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a User
     *         Permissions to Pass a Role to an AWS Service</a>.
     * @throws OptimisticLockException
     *         Exception that indicates that the protection state has been modified by another client. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.AssociateDRTLogBucket
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/AssociateDRTLogBucket" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AssociateDrtLogBucketResponse associateDRTLogBucket(AssociateDrtLogBucketRequest associateDrtLogBucketRequest)
            throws InternalErrorException, InvalidOperationException, NoAssociatedRoleException, LimitsExceededException,
            InvalidParameterException, AccessDeniedForDependencyException, OptimisticLockException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateDrtLogBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateDrtLogBucketResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AssociateDrtLogBucketRequest, AssociateDrtLogBucketResponse>()
                .withOperationName("AssociateDRTLogBucket").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(associateDrtLogBucketRequest)
                .withMarshaller(new AssociateDrtLogBucketRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Authorizes the DDoS Response team (DRT), using the specified role, to access your AWS account to assist with DDoS
     * attack mitigation during potential attacks. This enables the DRT to inspect your AWS WAF configuration and create
     * or update AWS WAF rules and web ACLs.
     * </p>
     * <p>
     * You can associate only one <code>RoleArn</code> with your subscription. If you submit an
     * <code>AssociateDRTRole</code> request for an account that already has an associated role, the new
     * <code>RoleArn</code> will replace the existing <code>RoleArn</code>.
     * </p>
     * <p>
     * Prior to making the <code>AssociateDRTRole</code> request, you must attach the <a href=
     * "https://console.aws.amazon.com/iam/home?#/policies/arn:aws:iam::aws:policy/service-role/AWSShieldDRTAccessPolicy"
     * >AWSShieldDRTAccessPolicy</a> managed policy to the role you will specify in the request. For more information
     * see <a
     * href=" https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_manage-attach-detach.html">Attaching and
     * Detaching IAM Policies</a>. The role must also trust the service principal <code> drt.shield.amazonaws.com</code>
     * . For more information, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html">IAM JSON
     * Policy Elements: Principal</a>.
     * </p>
     * <p>
     * The DRT will have access only to your AWS WAF and Shield resources. By submitting this request, you authorize the
     * DRT to inspect your AWS WAF and Shield configuration and create and update AWS WAF rules and web ACLs on your
     * behalf. The DRT takes these actions only if explicitly authorized by you.
     * </p>
     * <p>
     * You must have the <code>iam:PassRole</code> permission to make an <code>AssociateDRTRole</code> request. For more
     * information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a
     * User Permissions to Pass a Role to an AWS Service</a>.
     * </p>
     * <p>
     * To use the services of the DRT and make an <code>AssociateDRTRole</code> request, you must be subscribed to the
     * <a href="https://aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>.
     * </p>
     *
     * @param associateDrtRoleRequest
     * @return Result of the AssociateDRTRole operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid.
     * @throws AccessDeniedForDependencyException
     *         In order to grant the necessary access to the DDoS Response Team, the user submitting
     *         <code>AssociateDRTRole</code> must have the <code>iam:PassRole</code> permission. This error indicates
     *         the user did not have the appropriate permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a User
     *         Permissions to Pass a Role to an AWS Service</a>.
     * @throws OptimisticLockException
     *         Exception that indicates that the protection state has been modified by another client. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.AssociateDRTRole
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/AssociateDRTRole" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AssociateDrtRoleResponse associateDRTRole(AssociateDrtRoleRequest associateDrtRoleRequest)
            throws InternalErrorException, InvalidOperationException, InvalidParameterException,
            AccessDeniedForDependencyException, OptimisticLockException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateDrtRoleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AssociateDrtRoleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AssociateDrtRoleRequest, AssociateDrtRoleResponse>()
                .withOperationName("AssociateDRTRole").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(associateDrtRoleRequest)
                .withMarshaller(new AssociateDrtRoleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Enables AWS Shield Advanced for a specific AWS resource. The resource can be an Amazon CloudFront distribution,
     * Elastic Load Balancing load balancer, AWS Global Accelerator accelerator, Elastic IP Address, or an Amazon Route
     * 53 hosted zone.
     * </p>
     * <p>
     * You can add protection to only a single resource with each CreateProtection request. If you want to add
     * protection to multiple resources at once, use the <a href="https://console.aws.amazon.com/waf/">AWS WAF
     * console</a>. For more information see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/getting-started-ddos.html">Getting Started with AWS
     * Shield Advanced</a> and <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/configure-new-protection.html">Add AWS Shield
     * Advanced Protection to more AWS Resources</a>.
     * </p>
     *
     * @param createProtectionRequest
     * @return Result of the CreateProtection operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidResourceException
     *         Exception that indicates that the resource is invalid. You might not have access to the resource, or the
     *         resource might not exist.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws LimitsExceededException
     *         Exception that indicates that the operation would exceed a limit.</p>
     *         <p>
     *         <code>Type</code> is the type of limit that would be exceeded.
     *         </p>
     *         <p>
     *         <code>Limit</code> is the threshold that would be exceeded.
     * @throws ResourceAlreadyExistsException
     *         Exception indicating the specified resource already exists.
     * @throws OptimisticLockException
     *         Exception that indicates that the protection state has been modified by another client. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.CreateProtection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/CreateProtection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateProtectionResponse createProtection(CreateProtectionRequest createProtectionRequest)
            throws InternalErrorException, InvalidResourceException, InvalidOperationException, LimitsExceededException,
            ResourceAlreadyExistsException, OptimisticLockException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProtectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateProtectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateProtectionRequest, CreateProtectionResponse>()
                .withOperationName("CreateProtection").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createProtectionRequest)
                .withMarshaller(new CreateProtectionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Activates AWS Shield Advanced for an account.
     * </p>
     * <p>
     * As part of this request you can specify <code>EmergencySettings</code> that automaticaly grant the DDoS response
     * team (DRT) needed permissions to assist you during a suspected DDoS attack. For more information see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/authorize-DRT.html">Authorize the DDoS Response Team
     * to Create Rules and Web ACLs on Your Behalf</a>.
     * </p>
     * <p>
     * To use the services of the DRT, you must be subscribed to the <a
     * href="https://aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>.
     * </p>
     * <p>
     * When you initally create a subscription, your subscription is set to be automatically renewed at the end of the
     * existing subscription period. You can change this by submitting an <code>UpdateSubscription</code> request.
     * </p>
     *
     * @param createSubscriptionRequest
     * @return Result of the CreateSubscription operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceAlreadyExistsException
     *         Exception indicating the specified resource already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.CreateSubscription
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/CreateSubscription" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateSubscriptionResponse createSubscription(CreateSubscriptionRequest createSubscriptionRequest)
            throws InternalErrorException, ResourceAlreadyExistsException, AwsServiceException, SdkClientException,
            ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSubscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateSubscriptionRequest, CreateSubscriptionResponse>()
                .withOperationName("CreateSubscription").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createSubscriptionRequest)
                .withMarshaller(new CreateSubscriptionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes an AWS Shield Advanced <a>Protection</a>.
     * </p>
     *
     * @param deleteProtectionRequest
     * @return Result of the DeleteProtection operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist.
     * @throws OptimisticLockException
     *         Exception that indicates that the protection state has been modified by another client. You can retry the
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DeleteProtection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DeleteProtection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteProtectionResponse deleteProtection(DeleteProtectionRequest deleteProtectionRequest)
            throws InternalErrorException, ResourceNotFoundException, OptimisticLockException, AwsServiceException,
            SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProtectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteProtectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteProtectionRequest, DeleteProtectionResponse>()
                .withOperationName("DeleteProtection").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteProtectionRequest)
                .withMarshaller(new DeleteProtectionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes the details of a DDoS attack.
     * </p>
     *
     * @param describeAttackRequest
     * @return Result of the DescribeAttack operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws AccessDeniedException
     *         Exception that indicates the specified <code>AttackId</code> does not exist, or the requester does not
     *         have the appropriate permissions to access the <code>AttackId</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DescribeAttack
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeAttack" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeAttackResponse describeAttack(DescribeAttackRequest describeAttackRequest) throws InternalErrorException,
            AccessDeniedException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAttackResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeAttackResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeAttackRequest, DescribeAttackResponse>()
                .withOperationName("DescribeAttack").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeAttackRequest)
                .withMarshaller(new DescribeAttackRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the current role and list of Amazon S3 log buckets used by the DDoS Response team (DRT) to access your
     * AWS account while assisting with attack mitigation.
     * </p>
     *
     * @param describeDrtAccessRequest
     * @return Result of the DescribeDRTAccess operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DescribeDRTAccess
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeDRTAccess" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeDrtAccessResponse describeDRTAccess(DescribeDrtAccessRequest describeDrtAccessRequest)
            throws InternalErrorException, ResourceNotFoundException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDrtAccessResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeDrtAccessResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeDrtAccessRequest, DescribeDrtAccessResponse>()
                .withOperationName("DescribeDRTAccess").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeDrtAccessRequest)
                .withMarshaller(new DescribeDrtAccessRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the email addresses that the DRT can use to contact you during a suspected attack.
     * </p>
     *
     * @param describeEmergencyContactSettingsRequest
     * @return Result of the DescribeEmergencyContactSettings operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DescribeEmergencyContactSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeEmergencyContactSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEmergencyContactSettingsResponse describeEmergencyContactSettings(
            DescribeEmergencyContactSettingsRequest describeEmergencyContactSettingsRequest) throws InternalErrorException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEmergencyContactSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEmergencyContactSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeEmergencyContactSettingsRequest, DescribeEmergencyContactSettingsResponse>()
                        .withOperationName("DescribeEmergencyContactSettings").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeEmergencyContactSettingsRequest)
                        .withMarshaller(new DescribeEmergencyContactSettingsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the details of a <a>Protection</a> object.
     * </p>
     *
     * @param describeProtectionRequest
     * @return Result of the DescribeProtection operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DescribeProtection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeProtection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeProtectionResponse describeProtection(DescribeProtectionRequest describeProtectionRequest)
            throws InternalErrorException, InvalidParameterException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeProtectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeProtectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeProtectionRequest, DescribeProtectionResponse>()
                .withOperationName("DescribeProtection").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeProtectionRequest)
                .withMarshaller(new DescribeProtectionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides details about the AWS Shield Advanced subscription for an account.
     * </p>
     *
     * @param describeSubscriptionRequest
     * @return Result of the DescribeSubscription operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DescribeSubscription
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeSubscription" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeSubscriptionResponse describeSubscription(DescribeSubscriptionRequest describeSubscriptionRequest)
            throws InternalErrorException, ResourceNotFoundException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSubscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeSubscriptionRequest, DescribeSubscriptionResponse>()
                .withOperationName("DescribeSubscription").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeSubscriptionRequest)
                .withMarshaller(new DescribeSubscriptionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes the DDoS Response team's (DRT) access to the specified Amazon S3 bucket containing your AWS WAF logs.
     * </p>
     * <p>
     * To make a <code>DisassociateDRTLogBucket</code> request, you must be subscribed to the <a
     * href="https://aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>. However, if you are
     * not subscribed to one of these support plans, but had been previously and had granted the DRT access to your
     * account, you can submit a <code>DisassociateDRTLogBucket</code> request to remove this access.
     * </p>
     *
     * @param disassociateDrtLogBucketRequest
     * @return Result of the DisassociateDRTLogBucket operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws NoAssociatedRoleException
     *         The ARN of the role that you specifed does not exist.
     * @throws AccessDeniedForDependencyException
     *         In order to grant the necessary access to the DDoS Response Team, the user submitting
     *         <code>AssociateDRTRole</code> must have the <code>iam:PassRole</code> permission. This error indicates
     *         the user did not have the appropriate permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a User
     *         Permissions to Pass a Role to an AWS Service</a>.
     * @throws OptimisticLockException
     *         Exception that indicates that the protection state has been modified by another client. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DisassociateDRTLogBucket
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DisassociateDRTLogBucket"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateDrtLogBucketResponse disassociateDRTLogBucket(
            DisassociateDrtLogBucketRequest disassociateDrtLogBucketRequest) throws InternalErrorException,
            InvalidOperationException, NoAssociatedRoleException, AccessDeniedForDependencyException, OptimisticLockException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateDrtLogBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateDrtLogBucketResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DisassociateDrtLogBucketRequest, DisassociateDrtLogBucketResponse>()
                        .withOperationName("DisassociateDRTLogBucket").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(disassociateDrtLogBucketRequest)
                        .withMarshaller(new DisassociateDrtLogBucketRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes the DDoS Response team's (DRT) access to your AWS account.
     * </p>
     * <p>
     * To make a <code>DisassociateDRTRole</code> request, you must be subscribed to the <a
     * href="https://aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>. However, if you are
     * not subscribed to one of these support plans, but had been previously and had granted the DRT access to your
     * account, you can submit a <code>DisassociateDRTRole</code> request to remove this access.
     * </p>
     *
     * @param disassociateDrtRoleRequest
     * @return Result of the DisassociateDRTRole operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws OptimisticLockException
     *         Exception that indicates that the protection state has been modified by another client. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.DisassociateDRTRole
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DisassociateDRTRole" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DisassociateDrtRoleResponse disassociateDRTRole(DisassociateDrtRoleRequest disassociateDrtRoleRequest)
            throws InternalErrorException, InvalidOperationException, OptimisticLockException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateDrtRoleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateDrtRoleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DisassociateDrtRoleRequest, DisassociateDrtRoleResponse>()
                .withOperationName("DisassociateDRTRole").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(disassociateDrtRoleRequest)
                .withMarshaller(new DisassociateDrtRoleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the <code>SubscriptionState</code>, either <code>Active</code> or <code>Inactive</code>.
     * </p>
     *
     * @param getSubscriptionStateRequest
     * @return Result of the GetSubscriptionState operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.GetSubscriptionState
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/GetSubscriptionState" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetSubscriptionStateResponse getSubscriptionState(GetSubscriptionStateRequest getSubscriptionStateRequest)
            throws InternalErrorException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSubscriptionStateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSubscriptionStateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetSubscriptionStateRequest, GetSubscriptionStateResponse>()
                .withOperationName("GetSubscriptionState").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getSubscriptionStateRequest)
                .withMarshaller(new GetSubscriptionStateRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns all ongoing DDoS attacks or all DDoS attacks during a specified time period.
     * </p>
     *
     * @param listAttacksRequest
     * @return Result of the ListAttacks operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid.
     * @throws InvalidOperationException
     *         Exception that indicates that the operation would not cause any change to occur.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.ListAttacks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListAttacks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAttacksResponse listAttacks(ListAttacksRequest listAttacksRequest) throws InternalErrorException,
            InvalidParameterException, InvalidOperationException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAttacksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAttacksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListAttacksRequest, ListAttacksResponse>()
                .withOperationName("ListAttacks").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listAttacksRequest)
                .withMarshaller(new ListAttacksRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists all <a>Protection</a> objects for the account.
     * </p>
     *
     * @param listProtectionsRequest
     * @return Result of the ListProtections operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist.
     * @throws InvalidPaginationTokenException
     *         Exception that indicates that the NextToken specified in the request is invalid. Submit the request using
     *         the NextToken value that was returned in the response.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.ListProtections
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListProtections" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListProtectionsResponse listProtections(ListProtectionsRequest listProtectionsRequest) throws InternalErrorException,
            ResourceNotFoundException, InvalidPaginationTokenException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProtectionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListProtectionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListProtectionsRequest, ListProtectionsResponse>()
                .withOperationName("ListProtections").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listProtectionsRequest)
                .withMarshaller(new ListProtectionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the details of the list of email addresses that the DRT can use to contact you during a suspected attack.
     * </p>
     *
     * @param updateEmergencyContactSettingsRequest
     * @return Result of the UpdateEmergencyContactSettings operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid.
     * @throws OptimisticLockException
     *         Exception that indicates that the protection state has been modified by another client. You can retry the
     *         request.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.UpdateEmergencyContactSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/UpdateEmergencyContactSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateEmergencyContactSettingsResponse updateEmergencyContactSettings(
            UpdateEmergencyContactSettingsRequest updateEmergencyContactSettingsRequest) throws InternalErrorException,
            InvalidParameterException, OptimisticLockException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEmergencyContactSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateEmergencyContactSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpdateEmergencyContactSettingsRequest, UpdateEmergencyContactSettingsResponse>()
                        .withOperationName("UpdateEmergencyContactSettings").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(updateEmergencyContactSettingsRequest)
                        .withMarshaller(new UpdateEmergencyContactSettingsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the details of an existing subscription. Only enter values for parameters you want to change. Empty
     * parameters are not updated.
     * </p>
     *
     * @param updateSubscriptionRequest
     * @return Result of the UpdateSubscription operation returned by the service.
     * @throws InternalErrorException
     *         Exception that indicates that a problem occurred with the service infrastructure. You can retry the
     *         request.
     * @throws LockedSubscriptionException
     *         You are trying to update a subscription that has not yet completed the 1-year commitment. You can change
     *         the <code>AutoRenew</code> parameter during the last 30 days of your subscription. This exception
     *         indicates that you are attempting to change <code>AutoRenew</code> prior to that period.
     * @throws ResourceNotFoundException
     *         Exception indicating the specified resource does not exist.
     * @throws InvalidParameterException
     *         Exception that indicates that the parameters passed to the API are invalid.
     * @throws OptimisticLockException
     *         Exception that indicates that the protection state has been modified by another client. You can retry the
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ShieldException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ShieldClient.UpdateSubscription
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/UpdateSubscription" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateSubscriptionResponse updateSubscription(UpdateSubscriptionRequest updateSubscriptionRequest)
            throws InternalErrorException, LockedSubscriptionException, ResourceNotFoundException, InvalidParameterException,
            OptimisticLockException, AwsServiceException, SdkClientException, ShieldException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateSubscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateSubscriptionRequest, UpdateSubscriptionResponse>()
                .withOperationName("UpdateSubscription").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateSubscriptionRequest)
                .withMarshaller(new UpdateSubscriptionRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ShieldException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OptimisticLockException")
                                .exceptionBuilderSupplier(OptimisticLockException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LockedSubscriptionException")
                                .exceptionBuilderSupplier(LockedSubscriptionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceException")
                                .exceptionBuilderSupplier(InvalidResourceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOperationException")
                                .exceptionBuilderSupplier(InvalidOperationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalErrorException")
                                .exceptionBuilderSupplier(InternalErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException")
                                .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitsExceededException")
                                .exceptionBuilderSupplier(LimitsExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoAssociatedRoleException")
                                .exceptionBuilderSupplier(NoAssociatedRoleException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedForDependencyException")
                                .exceptionBuilderSupplier(AccessDeniedForDependencyException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
