/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about the threat detected in a security finding and the file paths that were affected by the
 * threat.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Threat implements SdkPojo, Serializable, ToCopyableBuilder<Threat.Builder, Threat> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Threat::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Severity").getter(getter(Threat::severity)).setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Severity").build()).build();

    private static final SdkField<Integer> ITEM_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ItemCount").getter(getter(Threat::itemCount)).setter(setter(Builder::itemCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ItemCount").build()).build();

    private static final SdkField<List<FilePaths>> FILE_PATHS_FIELD = SdkField
            .<List<FilePaths>> builder(MarshallingType.LIST)
            .memberName("FilePaths")
            .getter(getter(Threat::filePaths))
            .setter(setter(Builder::filePaths))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FilePaths").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<FilePaths> builder(MarshallingType.SDK_POJO)
                                            .constructor(FilePaths::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, SEVERITY_FIELD,
            ITEM_COUNT_FIELD, FILE_PATHS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String severity;

    private final Integer itemCount;

    private final List<FilePaths> filePaths;

    private Threat(BuilderImpl builder) {
        this.name = builder.name;
        this.severity = builder.severity;
        this.itemCount = builder.itemCount;
        this.filePaths = builder.filePaths;
    }

    /**
     * <p>
     * The name of the threat.
     * </p>
     * <p>
     * Length Constraints: Minimum of 1 length. Maximum of 128 length.
     * </p>
     * 
     * @return The name of the threat. </p>
     *         <p>
     *         Length Constraints: Minimum of 1 length. Maximum of 128 length.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The severity of the threat.
     * </p>
     * <p>
     * Length Constraints: Minimum of 1 length. Maximum of 128 length.
     * </p>
     * 
     * @return The severity of the threat. </p>
     *         <p>
     *         Length Constraints: Minimum of 1 length. Maximum of 128 length.
     */
    public final String severity() {
        return severity;
    }

    /**
     * <p>
     * This total number of items in which the threat has been detected.
     * </p>
     * 
     * @return This total number of items in which the threat has been detected.
     */
    public final Integer itemCount() {
        return itemCount;
    }

    /**
     * For responses, this returns true if the service returned a value for the FilePaths property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFilePaths() {
        return filePaths != null && !(filePaths instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Provides information about the file paths that were affected by the threat.
     * </p>
     * <p>
     * Array Members: Minimum number of 1 item. Maximum number of 5 items.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFilePaths} method.
     * </p>
     * 
     * @return Provides information about the file paths that were affected by the threat. </p>
     *         <p>
     *         Array Members: Minimum number of 1 item. Maximum number of 5 items.
     */
    public final List<FilePaths> filePaths() {
        return filePaths;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(severity());
        hashCode = 31 * hashCode + Objects.hashCode(itemCount());
        hashCode = 31 * hashCode + Objects.hashCode(hasFilePaths() ? filePaths() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Threat)) {
            return false;
        }
        Threat other = (Threat) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(severity(), other.severity())
                && Objects.equals(itemCount(), other.itemCount()) && hasFilePaths() == other.hasFilePaths()
                && Objects.equals(filePaths(), other.filePaths());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Threat").add("Name", name()).add("Severity", severity()).add("ItemCount", itemCount())
                .add("FilePaths", hasFilePaths() ? filePaths() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Severity":
            return Optional.ofNullable(clazz.cast(severity()));
        case "ItemCount":
            return Optional.ofNullable(clazz.cast(itemCount()));
        case "FilePaths":
            return Optional.ofNullable(clazz.cast(filePaths()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Severity", SEVERITY_FIELD);
        map.put("ItemCount", ITEM_COUNT_FIELD);
        map.put("FilePaths", FILE_PATHS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Threat, T> g) {
        return obj -> g.apply((Threat) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Threat> {
        /**
         * <p>
         * The name of the threat.
         * </p>
         * <p>
         * Length Constraints: Minimum of 1 length. Maximum of 128 length.
         * </p>
         * 
         * @param name
         *        The name of the threat. </p>
         *        <p>
         *        Length Constraints: Minimum of 1 length. Maximum of 128 length.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The severity of the threat.
         * </p>
         * <p>
         * Length Constraints: Minimum of 1 length. Maximum of 128 length.
         * </p>
         * 
         * @param severity
         *        The severity of the threat. </p>
         *        <p>
         *        Length Constraints: Minimum of 1 length. Maximum of 128 length.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder severity(String severity);

        /**
         * <p>
         * This total number of items in which the threat has been detected.
         * </p>
         * 
         * @param itemCount
         *        This total number of items in which the threat has been detected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder itemCount(Integer itemCount);

        /**
         * <p>
         * Provides information about the file paths that were affected by the threat.
         * </p>
         * <p>
         * Array Members: Minimum number of 1 item. Maximum number of 5 items.
         * </p>
         * 
         * @param filePaths
         *        Provides information about the file paths that were affected by the threat. </p>
         *        <p>
         *        Array Members: Minimum number of 1 item. Maximum number of 5 items.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filePaths(Collection<FilePaths> filePaths);

        /**
         * <p>
         * Provides information about the file paths that were affected by the threat.
         * </p>
         * <p>
         * Array Members: Minimum number of 1 item. Maximum number of 5 items.
         * </p>
         * 
         * @param filePaths
         *        Provides information about the file paths that were affected by the threat. </p>
         *        <p>
         *        Array Members: Minimum number of 1 item. Maximum number of 5 items.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filePaths(FilePaths... filePaths);

        /**
         * <p>
         * Provides information about the file paths that were affected by the threat.
         * </p>
         * <p>
         * Array Members: Minimum number of 1 item. Maximum number of 5 items.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securityhub.model.FilePaths.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.securityhub.model.FilePaths#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securityhub.model.FilePaths.Builder#build()} is called immediately and
         * its result is passed to {@link #filePaths(List<FilePaths>)}.
         * 
         * @param filePaths
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securityhub.model.FilePaths.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filePaths(java.util.Collection<FilePaths>)
         */
        Builder filePaths(Consumer<FilePaths.Builder>... filePaths);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String severity;

        private Integer itemCount;

        private List<FilePaths> filePaths = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Threat model) {
            name(model.name);
            severity(model.severity);
            itemCount(model.itemCount);
            filePaths(model.filePaths);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getSeverity() {
            return severity;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        @Override
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        public final Integer getItemCount() {
            return itemCount;
        }

        public final void setItemCount(Integer itemCount) {
            this.itemCount = itemCount;
        }

        @Override
        public final Builder itemCount(Integer itemCount) {
            this.itemCount = itemCount;
            return this;
        }

        public final List<FilePaths.Builder> getFilePaths() {
            List<FilePaths.Builder> result = FilePathListCopier.copyToBuilder(this.filePaths);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFilePaths(Collection<FilePaths.BuilderImpl> filePaths) {
            this.filePaths = FilePathListCopier.copyFromBuilder(filePaths);
        }

        @Override
        public final Builder filePaths(Collection<FilePaths> filePaths) {
            this.filePaths = FilePathListCopier.copy(filePaths);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filePaths(FilePaths... filePaths) {
            filePaths(Arrays.asList(filePaths));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filePaths(Consumer<FilePaths.Builder>... filePaths) {
            filePaths(Stream.of(filePaths).map(c -> FilePaths.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Threat build() {
            return new Threat(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
