/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that defines how Security Hub is configured. The configuration policy includes whether Security Hub is
 * enabled or disabled, a list of enabled security standards, a list of enabled or disabled security controls, and a
 * list of custom parameter values for specified controls. If you provide a list of security controls that are enabled
 * in the configuration policy, Security Hub disables all other controls (including newly released controls). If you
 * provide a list of security controls that are disabled in the configuration policy, Security Hub enables all other
 * controls (including newly released controls).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SecurityHubPolicy implements SdkPojo, Serializable,
        ToCopyableBuilder<SecurityHubPolicy.Builder, SecurityHubPolicy> {
    private static final SdkField<Boolean> SERVICE_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ServiceEnabled").getter(getter(SecurityHubPolicy::serviceEnabled))
            .setter(setter(Builder::serviceEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceEnabled").build()).build();

    private static final SdkField<List<String>> ENABLED_STANDARD_IDENTIFIERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EnabledStandardIdentifiers")
            .getter(getter(SecurityHubPolicy::enabledStandardIdentifiers))
            .setter(setter(Builder::enabledStandardIdentifiers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnabledStandardIdentifiers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<SecurityControlsConfiguration> SECURITY_CONTROLS_CONFIGURATION_FIELD = SdkField
            .<SecurityControlsConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("SecurityControlsConfiguration")
            .getter(getter(SecurityHubPolicy::securityControlsConfiguration))
            .setter(setter(Builder::securityControlsConfiguration))
            .constructor(SecurityControlsConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityControlsConfiguration")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SERVICE_ENABLED_FIELD,
            ENABLED_STANDARD_IDENTIFIERS_FIELD, SECURITY_CONTROLS_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Boolean serviceEnabled;

    private final List<String> enabledStandardIdentifiers;

    private final SecurityControlsConfiguration securityControlsConfiguration;

    private SecurityHubPolicy(BuilderImpl builder) {
        this.serviceEnabled = builder.serviceEnabled;
        this.enabledStandardIdentifiers = builder.enabledStandardIdentifiers;
        this.securityControlsConfiguration = builder.securityControlsConfiguration;
    }

    /**
     * <p>
     * Indicates whether Security Hub is enabled in the policy.
     * </p>
     * 
     * @return Indicates whether Security Hub is enabled in the policy.
     */
    public final Boolean serviceEnabled() {
        return serviceEnabled;
    }

    /**
     * For responses, this returns true if the service returned a value for the EnabledStandardIdentifiers property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasEnabledStandardIdentifiers() {
        return enabledStandardIdentifiers != null && !(enabledStandardIdentifiers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list that defines which security standards are enabled in the configuration policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnabledStandardIdentifiers} method.
     * </p>
     * 
     * @return A list that defines which security standards are enabled in the configuration policy.
     */
    public final List<String> enabledStandardIdentifiers() {
        return enabledStandardIdentifiers;
    }

    /**
     * <p>
     * An object that defines which security controls are enabled in the configuration policy. The enablement status of
     * a control is aligned across all of the enabled standards in an account.
     * </p>
     * 
     * @return An object that defines which security controls are enabled in the configuration policy. The enablement
     *         status of a control is aligned across all of the enabled standards in an account.
     */
    public final SecurityControlsConfiguration securityControlsConfiguration() {
        return securityControlsConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(serviceEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnabledStandardIdentifiers() ? enabledStandardIdentifiers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(securityControlsConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SecurityHubPolicy)) {
            return false;
        }
        SecurityHubPolicy other = (SecurityHubPolicy) obj;
        return Objects.equals(serviceEnabled(), other.serviceEnabled())
                && hasEnabledStandardIdentifiers() == other.hasEnabledStandardIdentifiers()
                && Objects.equals(enabledStandardIdentifiers(), other.enabledStandardIdentifiers())
                && Objects.equals(securityControlsConfiguration(), other.securityControlsConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SecurityHubPolicy").add("ServiceEnabled", serviceEnabled())
                .add("EnabledStandardIdentifiers", hasEnabledStandardIdentifiers() ? enabledStandardIdentifiers() : null)
                .add("SecurityControlsConfiguration", securityControlsConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ServiceEnabled":
            return Optional.ofNullable(clazz.cast(serviceEnabled()));
        case "EnabledStandardIdentifiers":
            return Optional.ofNullable(clazz.cast(enabledStandardIdentifiers()));
        case "SecurityControlsConfiguration":
            return Optional.ofNullable(clazz.cast(securityControlsConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ServiceEnabled", SERVICE_ENABLED_FIELD);
        map.put("EnabledStandardIdentifiers", ENABLED_STANDARD_IDENTIFIERS_FIELD);
        map.put("SecurityControlsConfiguration", SECURITY_CONTROLS_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SecurityHubPolicy, T> g) {
        return obj -> g.apply((SecurityHubPolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SecurityHubPolicy> {
        /**
         * <p>
         * Indicates whether Security Hub is enabled in the policy.
         * </p>
         * 
         * @param serviceEnabled
         *        Indicates whether Security Hub is enabled in the policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceEnabled(Boolean serviceEnabled);

        /**
         * <p>
         * A list that defines which security standards are enabled in the configuration policy.
         * </p>
         * 
         * @param enabledStandardIdentifiers
         *        A list that defines which security standards are enabled in the configuration policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabledStandardIdentifiers(Collection<String> enabledStandardIdentifiers);

        /**
         * <p>
         * A list that defines which security standards are enabled in the configuration policy.
         * </p>
         * 
         * @param enabledStandardIdentifiers
         *        A list that defines which security standards are enabled in the configuration policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabledStandardIdentifiers(String... enabledStandardIdentifiers);

        /**
         * <p>
         * An object that defines which security controls are enabled in the configuration policy. The enablement status
         * of a control is aligned across all of the enabled standards in an account.
         * </p>
         * 
         * @param securityControlsConfiguration
         *        An object that defines which security controls are enabled in the configuration policy. The enablement
         *        status of a control is aligned across all of the enabled standards in an account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityControlsConfiguration(SecurityControlsConfiguration securityControlsConfiguration);

        /**
         * <p>
         * An object that defines which security controls are enabled in the configuration policy. The enablement status
         * of a control is aligned across all of the enabled standards in an account.
         * </p>
         * This is a convenience method that creates an instance of the {@link SecurityControlsConfiguration.Builder}
         * avoiding the need to create one manually via {@link SecurityControlsConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SecurityControlsConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #securityControlsConfiguration(SecurityControlsConfiguration)}.
         * 
         * @param securityControlsConfiguration
         *        a consumer that will call methods on {@link SecurityControlsConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #securityControlsConfiguration(SecurityControlsConfiguration)
         */
        default Builder securityControlsConfiguration(
                Consumer<SecurityControlsConfiguration.Builder> securityControlsConfiguration) {
            return securityControlsConfiguration(SecurityControlsConfiguration.builder()
                    .applyMutation(securityControlsConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Boolean serviceEnabled;

        private List<String> enabledStandardIdentifiers = DefaultSdkAutoConstructList.getInstance();

        private SecurityControlsConfiguration securityControlsConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(SecurityHubPolicy model) {
            serviceEnabled(model.serviceEnabled);
            enabledStandardIdentifiers(model.enabledStandardIdentifiers);
            securityControlsConfiguration(model.securityControlsConfiguration);
        }

        public final Boolean getServiceEnabled() {
            return serviceEnabled;
        }

        public final void setServiceEnabled(Boolean serviceEnabled) {
            this.serviceEnabled = serviceEnabled;
        }

        @Override
        public final Builder serviceEnabled(Boolean serviceEnabled) {
            this.serviceEnabled = serviceEnabled;
            return this;
        }

        public final Collection<String> getEnabledStandardIdentifiers() {
            if (enabledStandardIdentifiers instanceof SdkAutoConstructList) {
                return null;
            }
            return enabledStandardIdentifiers;
        }

        public final void setEnabledStandardIdentifiers(Collection<String> enabledStandardIdentifiers) {
            this.enabledStandardIdentifiers = EnabledStandardIdentifierListCopier.copy(enabledStandardIdentifiers);
        }

        @Override
        public final Builder enabledStandardIdentifiers(Collection<String> enabledStandardIdentifiers) {
            this.enabledStandardIdentifiers = EnabledStandardIdentifierListCopier.copy(enabledStandardIdentifiers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder enabledStandardIdentifiers(String... enabledStandardIdentifiers) {
            enabledStandardIdentifiers(Arrays.asList(enabledStandardIdentifiers));
            return this;
        }

        public final SecurityControlsConfiguration.Builder getSecurityControlsConfiguration() {
            return securityControlsConfiguration != null ? securityControlsConfiguration.toBuilder() : null;
        }

        public final void setSecurityControlsConfiguration(SecurityControlsConfiguration.BuilderImpl securityControlsConfiguration) {
            this.securityControlsConfiguration = securityControlsConfiguration != null ? securityControlsConfiguration.build()
                    : null;
        }

        @Override
        public final Builder securityControlsConfiguration(SecurityControlsConfiguration securityControlsConfiguration) {
            this.securityControlsConfiguration = securityControlsConfiguration;
            return this;
        }

        @Override
        public SecurityHubPolicy build() {
            return new SecurityHubPolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
