/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the indicators observed in an Amazon GuardDuty Extended Threat Detection attack sequence.
 * Indicators include a set of signals, which can be API activities or findings that GuardDuty uses to detect an attack
 * sequence finding. GuardDuty generates an attack sequence finding when multiple signals align to a potentially
 * suspicious activity. To receive GuardDuty attack sequence findings in Security Hub, you must have GuardDuty and
 * GuardDuty S3 Protection enabled. For more information, see <a
 * href="https://docs.aws.amazon.com/guardduty/latest/ug/guardduty-extended-threat-detection.html">GuardDuty Extended
 * Threat Detection </a> in the <i>Amazon GuardDuty User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Indicator implements SdkPojo, Serializable, ToCopyableBuilder<Indicator.Builder, Indicator> {
    private static final SdkField<String> KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Key")
            .getter(getter(Indicator::key)).setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Key").build()).build();

    private static final SdkField<List<String>> VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Values")
            .getter(getter(Indicator::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(Indicator::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(Indicator::type)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_FIELD, VALUES_FIELD,
            TITLE_FIELD, TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String key;

    private final List<String> values;

    private final String title;

    private final String type;

    private Indicator(BuilderImpl builder) {
        this.key = builder.key;
        this.values = builder.values;
        this.title = builder.title;
        this.type = builder.type;
    }

    /**
     * <p>
     * The name of the indicator that’s present in the attack sequence finding.
     * </p>
     * 
     * @return The name of the indicator that’s present in the attack sequence finding.
     */
    public final String key() {
        return key;
    }

    /**
     * For responses, this returns true if the service returned a value for the Values property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Values associated with each indicator key. For example, if the indicator key is <code>SUSPICIOUS_NETWORK</code>,
     * then the value will be the name of the network. If the indicator key is <code>ATTACK_TACTIC</code>, then the
     * value will be one of the MITRE tactics.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValues} method.
     * </p>
     * 
     * @return Values associated with each indicator key. For example, if the indicator key is
     *         <code>SUSPICIOUS_NETWORK</code>, then the value will be the name of the network. If the indicator key is
     *         <code>ATTACK_TACTIC</code>, then the value will be one of the MITRE tactics.
     */
    public final List<String> values() {
        return values;
    }

    /**
     * <p>
     * The title describing the indicator.
     * </p>
     * 
     * @return The title describing the indicator.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The type of indicator.
     * </p>
     * 
     * @return The type of indicator.
     */
    public final String type() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(hasValues() ? values() : null);
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(type());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Indicator)) {
            return false;
        }
        Indicator other = (Indicator) obj;
        return Objects.equals(key(), other.key()) && hasValues() == other.hasValues() && Objects.equals(values(), other.values())
                && Objects.equals(title(), other.title()) && Objects.equals(type(), other.type());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Indicator").add("Key", key()).add("Values", hasValues() ? values() : null).add("Title", title())
                .add("Type", type()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Key":
            return Optional.ofNullable(clazz.cast(key()));
        case "Values":
            return Optional.ofNullable(clazz.cast(values()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Type":
            return Optional.ofNullable(clazz.cast(type()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Key", KEY_FIELD);
        map.put("Values", VALUES_FIELD);
        map.put("Title", TITLE_FIELD);
        map.put("Type", TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Indicator, T> g) {
        return obj -> g.apply((Indicator) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Indicator> {
        /**
         * <p>
         * The name of the indicator that’s present in the attack sequence finding.
         * </p>
         * 
         * @param key
         *        The name of the indicator that’s present in the attack sequence finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * Values associated with each indicator key. For example, if the indicator key is
         * <code>SUSPICIOUS_NETWORK</code>, then the value will be the name of the network. If the indicator key is
         * <code>ATTACK_TACTIC</code>, then the value will be one of the MITRE tactics.
         * </p>
         * 
         * @param values
         *        Values associated with each indicator key. For example, if the indicator key is
         *        <code>SUSPICIOUS_NETWORK</code>, then the value will be the name of the network. If the indicator key
         *        is <code>ATTACK_TACTIC</code>, then the value will be one of the MITRE tactics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * Values associated with each indicator key. For example, if the indicator key is
         * <code>SUSPICIOUS_NETWORK</code>, then the value will be the name of the network. If the indicator key is
         * <code>ATTACK_TACTIC</code>, then the value will be one of the MITRE tactics.
         * </p>
         * 
         * @param values
         *        Values associated with each indicator key. For example, if the indicator key is
         *        <code>SUSPICIOUS_NETWORK</code>, then the value will be the name of the network. If the indicator key
         *        is <code>ATTACK_TACTIC</code>, then the value will be one of the MITRE tactics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);

        /**
         * <p>
         * The title describing the indicator.
         * </p>
         * 
         * @param title
         *        The title describing the indicator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The type of indicator.
         * </p>
         * 
         * @param type
         *        The type of indicator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder type(String type);
    }

    static final class BuilderImpl implements Builder {
        private String key;

        private List<String> values = DefaultSdkAutoConstructList.getInstance();

        private String title;

        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(Indicator model) {
            key(model.key);
            values(model.values);
            title(model.title);
            type(model.type);
        }

        public final String getKey() {
            return key;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final Collection<String> getValues() {
            if (values instanceof SdkAutoConstructList) {
                return null;
            }
            return values;
        }

        public final void setValues(Collection<String> values) {
            this.values = NonEmptyStringListCopier.copy(values);
        }

        @Override
        public final Builder values(Collection<String> values) {
            this.values = NonEmptyStringListCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public Indicator build() {
            return new Indicator(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
