/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides details about an Amazon MQ message broker. A message broker allows software applications and components to
 * communicate using various programming languages, operating systems, and formal messaging protocols.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AwsAmazonMqBrokerDetails implements SdkPojo, Serializable,
        ToCopyableBuilder<AwsAmazonMqBrokerDetails.Builder, AwsAmazonMqBrokerDetails> {
    private static final SdkField<String> AUTHENTICATION_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AuthenticationStrategy").getter(getter(AwsAmazonMqBrokerDetails::authenticationStrategy))
            .setter(setter(Builder::authenticationStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthenticationStrategy").build())
            .build();

    private static final SdkField<Boolean> AUTO_MINOR_VERSION_UPGRADE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AutoMinorVersionUpgrade").getter(getter(AwsAmazonMqBrokerDetails::autoMinorVersionUpgrade))
            .setter(setter(Builder::autoMinorVersionUpgrade))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoMinorVersionUpgrade").build())
            .build();

    private static final SdkField<String> BROKER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BrokerArn").getter(getter(AwsAmazonMqBrokerDetails::brokerArn)).setter(setter(Builder::brokerArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BrokerArn").build()).build();

    private static final SdkField<String> BROKER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BrokerName").getter(getter(AwsAmazonMqBrokerDetails::brokerName)).setter(setter(Builder::brokerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BrokerName").build()).build();

    private static final SdkField<String> DEPLOYMENT_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeploymentMode").getter(getter(AwsAmazonMqBrokerDetails::deploymentMode))
            .setter(setter(Builder::deploymentMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeploymentMode").build()).build();

    private static final SdkField<AwsAmazonMqBrokerEncryptionOptionsDetails> ENCRYPTION_OPTIONS_FIELD = SdkField
            .<AwsAmazonMqBrokerEncryptionOptionsDetails> builder(MarshallingType.SDK_POJO).memberName("EncryptionOptions")
            .getter(getter(AwsAmazonMqBrokerDetails::encryptionOptions)).setter(setter(Builder::encryptionOptions))
            .constructor(AwsAmazonMqBrokerEncryptionOptionsDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionOptions").build()).build();

    private static final SdkField<String> ENGINE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EngineType").getter(getter(AwsAmazonMqBrokerDetails::engineType)).setter(setter(Builder::engineType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineType").build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EngineVersion").getter(getter(AwsAmazonMqBrokerDetails::engineVersion))
            .setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<String> HOST_INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HostInstanceType").getter(getter(AwsAmazonMqBrokerDetails::hostInstanceType))
            .setter(setter(Builder::hostInstanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HostInstanceType").build()).build();

    private static final SdkField<String> BROKER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BrokerId").getter(getter(AwsAmazonMqBrokerDetails::brokerId)).setter(setter(Builder::brokerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BrokerId").build()).build();

    private static final SdkField<AwsAmazonMqBrokerLdapServerMetadataDetails> LDAP_SERVER_METADATA_FIELD = SdkField
            .<AwsAmazonMqBrokerLdapServerMetadataDetails> builder(MarshallingType.SDK_POJO).memberName("LdapServerMetadata")
            .getter(getter(AwsAmazonMqBrokerDetails::ldapServerMetadata)).setter(setter(Builder::ldapServerMetadata))
            .constructor(AwsAmazonMqBrokerLdapServerMetadataDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LdapServerMetadata").build())
            .build();

    private static final SdkField<AwsAmazonMqBrokerLogsDetails> LOGS_FIELD = SdkField
            .<AwsAmazonMqBrokerLogsDetails> builder(MarshallingType.SDK_POJO).memberName("Logs")
            .getter(getter(AwsAmazonMqBrokerDetails::logs)).setter(setter(Builder::logs))
            .constructor(AwsAmazonMqBrokerLogsDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Logs").build()).build();

    private static final SdkField<AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails> MAINTENANCE_WINDOW_START_TIME_FIELD = SdkField
            .<AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails> builder(MarshallingType.SDK_POJO)
            .memberName("MaintenanceWindowStartTime")
            .getter(getter(AwsAmazonMqBrokerDetails::maintenanceWindowStartTime))
            .setter(setter(Builder::maintenanceWindowStartTime))
            .constructor(AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaintenanceWindowStartTime").build())
            .build();

    private static final SdkField<Boolean> PUBLICLY_ACCESSIBLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("PubliclyAccessible").getter(getter(AwsAmazonMqBrokerDetails::publiclyAccessible))
            .setter(setter(Builder::publiclyAccessible))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PubliclyAccessible").build())
            .build();

    private static final SdkField<List<String>> SECURITY_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroups")
            .getter(getter(AwsAmazonMqBrokerDetails::securityGroups))
            .setter(setter(Builder::securityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STORAGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StorageType").getter(getter(AwsAmazonMqBrokerDetails::storageType)).setter(setter(Builder::storageType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageType").build()).build();

    private static final SdkField<List<String>> SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SubnetIds")
            .getter(getter(AwsAmazonMqBrokerDetails::subnetIds))
            .setter(setter(Builder::subnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<AwsAmazonMqBrokerUsersDetails>> USERS_FIELD = SdkField
            .<List<AwsAmazonMqBrokerUsersDetails>> builder(MarshallingType.LIST)
            .memberName("Users")
            .getter(getter(AwsAmazonMqBrokerDetails::users))
            .setter(setter(Builder::users))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Users").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AwsAmazonMqBrokerUsersDetails> builder(MarshallingType.SDK_POJO)
                                            .constructor(AwsAmazonMqBrokerUsersDetails::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AUTHENTICATION_STRATEGY_FIELD,
            AUTO_MINOR_VERSION_UPGRADE_FIELD, BROKER_ARN_FIELD, BROKER_NAME_FIELD, DEPLOYMENT_MODE_FIELD,
            ENCRYPTION_OPTIONS_FIELD, ENGINE_TYPE_FIELD, ENGINE_VERSION_FIELD, HOST_INSTANCE_TYPE_FIELD, BROKER_ID_FIELD,
            LDAP_SERVER_METADATA_FIELD, LOGS_FIELD, MAINTENANCE_WINDOW_START_TIME_FIELD, PUBLICLY_ACCESSIBLE_FIELD,
            SECURITY_GROUPS_FIELD, STORAGE_TYPE_FIELD, SUBNET_IDS_FIELD, USERS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String authenticationStrategy;

    private final Boolean autoMinorVersionUpgrade;

    private final String brokerArn;

    private final String brokerName;

    private final String deploymentMode;

    private final AwsAmazonMqBrokerEncryptionOptionsDetails encryptionOptions;

    private final String engineType;

    private final String engineVersion;

    private final String hostInstanceType;

    private final String brokerId;

    private final AwsAmazonMqBrokerLdapServerMetadataDetails ldapServerMetadata;

    private final AwsAmazonMqBrokerLogsDetails logs;

    private final AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails maintenanceWindowStartTime;

    private final Boolean publiclyAccessible;

    private final List<String> securityGroups;

    private final String storageType;

    private final List<String> subnetIds;

    private final List<AwsAmazonMqBrokerUsersDetails> users;

    private AwsAmazonMqBrokerDetails(BuilderImpl builder) {
        this.authenticationStrategy = builder.authenticationStrategy;
        this.autoMinorVersionUpgrade = builder.autoMinorVersionUpgrade;
        this.brokerArn = builder.brokerArn;
        this.brokerName = builder.brokerName;
        this.deploymentMode = builder.deploymentMode;
        this.encryptionOptions = builder.encryptionOptions;
        this.engineType = builder.engineType;
        this.engineVersion = builder.engineVersion;
        this.hostInstanceType = builder.hostInstanceType;
        this.brokerId = builder.brokerId;
        this.ldapServerMetadata = builder.ldapServerMetadata;
        this.logs = builder.logs;
        this.maintenanceWindowStartTime = builder.maintenanceWindowStartTime;
        this.publiclyAccessible = builder.publiclyAccessible;
        this.securityGroups = builder.securityGroups;
        this.storageType = builder.storageType;
        this.subnetIds = builder.subnetIds;
        this.users = builder.users;
    }

    /**
     * <p>
     * The authentication strategy used to secure the broker. The default is <code>SIMPLE</code>.
     * </p>
     * 
     * @return The authentication strategy used to secure the broker. The default is <code>SIMPLE</code>.
     */
    public final String authenticationStrategy() {
        return authenticationStrategy;
    }

    /**
     * <p>
     * Whether automatically upgrade new minor versions for brokers, as new versions are released and supported by
     * Amazon MQ. Automatic upgrades occur during the scheduled maintenance window of the broker or after a manual
     * broker reboot.
     * </p>
     * 
     * @return Whether automatically upgrade new minor versions for brokers, as new versions are released and supported
     *         by Amazon MQ. Automatic upgrades occur during the scheduled maintenance window of the broker or after a
     *         manual broker reboot.
     */
    public final Boolean autoMinorVersionUpgrade() {
        return autoMinorVersionUpgrade;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the broker.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the broker.
     */
    public final String brokerArn() {
        return brokerArn;
    }

    /**
     * <p>
     * The broker's name.
     * </p>
     * 
     * @return The broker's name.
     */
    public final String brokerName() {
        return brokerName;
    }

    /**
     * <p>
     * The broker's deployment mode.
     * </p>
     * 
     * @return The broker's deployment mode.
     */
    public final String deploymentMode() {
        return deploymentMode;
    }

    /**
     * <p>
     * Encryption options for the broker. Doesn’t apply to RabbitMQ brokers.
     * </p>
     * 
     * @return Encryption options for the broker. Doesn’t apply to RabbitMQ brokers.
     */
    public final AwsAmazonMqBrokerEncryptionOptionsDetails encryptionOptions() {
        return encryptionOptions;
    }

    /**
     * <p>
     * The type of broker engine.
     * </p>
     * 
     * @return The type of broker engine.
     */
    public final String engineType() {
        return engineType;
    }

    /**
     * <p>
     * The version of the broker engine.
     * </p>
     * 
     * @return The version of the broker engine.
     */
    public final String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * The broker's instance type.
     * </p>
     * 
     * @return The broker's instance type.
     */
    public final String hostInstanceType() {
        return hostInstanceType;
    }

    /**
     * <p>
     * The unique ID that Amazon MQ generates for the broker.
     * </p>
     * 
     * @return The unique ID that Amazon MQ generates for the broker.
     */
    public final String brokerId() {
        return brokerId;
    }

    /**
     * <p>
     * The metadata of the Lightweight Directory Access Protocol (LDAP) server used to authenticate and authorize
     * connections to the broker. This is an optional failover server.
     * </p>
     * 
     * @return The metadata of the Lightweight Directory Access Protocol (LDAP) server used to authenticate and
     *         authorize connections to the broker. This is an optional failover server.
     */
    public final AwsAmazonMqBrokerLdapServerMetadataDetails ldapServerMetadata() {
        return ldapServerMetadata;
    }

    /**
     * <p>
     * Turns on Amazon CloudWatch logging for brokers.
     * </p>
     * 
     * @return Turns on Amazon CloudWatch logging for brokers.
     */
    public final AwsAmazonMqBrokerLogsDetails logs() {
        return logs;
    }

    /**
     * <p>
     * The scheduled time period (UTC) during which Amazon MQ begins to apply pending updates or patches to the broker.
     * </p>
     * 
     * @return The scheduled time period (UTC) during which Amazon MQ begins to apply pending updates or patches to the
     *         broker.
     */
    public final AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails maintenanceWindowStartTime() {
        return maintenanceWindowStartTime;
    }

    /**
     * <p>
     * Permits connections from applications outside of the VPC that hosts the broker's subnets.
     * </p>
     * 
     * @return Permits connections from applications outside of the VPC that hosts the broker's subnets.
     */
    public final Boolean publiclyAccessible() {
        return publiclyAccessible;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroups property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroups() {
        return securityGroups != null && !(securityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of rules (one minimum, 125 maximum) that authorize connections to brokers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroups} method.
     * </p>
     * 
     * @return The list of rules (one minimum, 125 maximum) that authorize connections to brokers.
     */
    public final List<String> securityGroups() {
        return securityGroups;
    }

    /**
     * <p>
     * The broker's storage type.
     * </p>
     * 
     * @return The broker's storage type.
     */
    public final String storageType() {
        return storageType;
    }

    /**
     * For responses, this returns true if the service returned a value for the SubnetIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSubnetIds() {
        return subnetIds != null && !(subnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of groups that define which subnets and IP ranges the broker can use from different Availability Zones.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubnetIds} method.
     * </p>
     * 
     * @return The list of groups that define which subnets and IP ranges the broker can use from different Availability
     *         Zones.
     */
    public final List<String> subnetIds() {
        return subnetIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the Users property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasUsers() {
        return users != null && !(users instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of all broker usernames for the specified broker. Doesn't apply to RabbitMQ brokers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUsers} method.
     * </p>
     * 
     * @return The list of all broker usernames for the specified broker. Doesn't apply to RabbitMQ brokers.
     */
    public final List<AwsAmazonMqBrokerUsersDetails> users() {
        return users;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(authenticationStrategy());
        hashCode = 31 * hashCode + Objects.hashCode(autoMinorVersionUpgrade());
        hashCode = 31 * hashCode + Objects.hashCode(brokerArn());
        hashCode = 31 * hashCode + Objects.hashCode(brokerName());
        hashCode = 31 * hashCode + Objects.hashCode(deploymentMode());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionOptions());
        hashCode = 31 * hashCode + Objects.hashCode(engineType());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hostInstanceType());
        hashCode = 31 * hashCode + Objects.hashCode(brokerId());
        hashCode = 31 * hashCode + Objects.hashCode(ldapServerMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(logs());
        hashCode = 31 * hashCode + Objects.hashCode(maintenanceWindowStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(publiclyAccessible());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroups() ? securityGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(storageType());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetIds() ? subnetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasUsers() ? users() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AwsAmazonMqBrokerDetails)) {
            return false;
        }
        AwsAmazonMqBrokerDetails other = (AwsAmazonMqBrokerDetails) obj;
        return Objects.equals(authenticationStrategy(), other.authenticationStrategy())
                && Objects.equals(autoMinorVersionUpgrade(), other.autoMinorVersionUpgrade())
                && Objects.equals(brokerArn(), other.brokerArn()) && Objects.equals(brokerName(), other.brokerName())
                && Objects.equals(deploymentMode(), other.deploymentMode())
                && Objects.equals(encryptionOptions(), other.encryptionOptions())
                && Objects.equals(engineType(), other.engineType()) && Objects.equals(engineVersion(), other.engineVersion())
                && Objects.equals(hostInstanceType(), other.hostInstanceType()) && Objects.equals(brokerId(), other.brokerId())
                && Objects.equals(ldapServerMetadata(), other.ldapServerMetadata()) && Objects.equals(logs(), other.logs())
                && Objects.equals(maintenanceWindowStartTime(), other.maintenanceWindowStartTime())
                && Objects.equals(publiclyAccessible(), other.publiclyAccessible())
                && hasSecurityGroups() == other.hasSecurityGroups() && Objects.equals(securityGroups(), other.securityGroups())
                && Objects.equals(storageType(), other.storageType()) && hasSubnetIds() == other.hasSubnetIds()
                && Objects.equals(subnetIds(), other.subnetIds()) && hasUsers() == other.hasUsers()
                && Objects.equals(users(), other.users());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AwsAmazonMqBrokerDetails").add("AuthenticationStrategy", authenticationStrategy())
                .add("AutoMinorVersionUpgrade", autoMinorVersionUpgrade()).add("BrokerArn", brokerArn())
                .add("BrokerName", brokerName()).add("DeploymentMode", deploymentMode())
                .add("EncryptionOptions", encryptionOptions()).add("EngineType", engineType())
                .add("EngineVersion", engineVersion()).add("HostInstanceType", hostInstanceType()).add("BrokerId", brokerId())
                .add("LdapServerMetadata", ldapServerMetadata()).add("Logs", logs())
                .add("MaintenanceWindowStartTime", maintenanceWindowStartTime()).add("PubliclyAccessible", publiclyAccessible())
                .add("SecurityGroups", hasSecurityGroups() ? securityGroups() : null).add("StorageType", storageType())
                .add("SubnetIds", hasSubnetIds() ? subnetIds() : null).add("Users", hasUsers() ? users() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AuthenticationStrategy":
            return Optional.ofNullable(clazz.cast(authenticationStrategy()));
        case "AutoMinorVersionUpgrade":
            return Optional.ofNullable(clazz.cast(autoMinorVersionUpgrade()));
        case "BrokerArn":
            return Optional.ofNullable(clazz.cast(brokerArn()));
        case "BrokerName":
            return Optional.ofNullable(clazz.cast(brokerName()));
        case "DeploymentMode":
            return Optional.ofNullable(clazz.cast(deploymentMode()));
        case "EncryptionOptions":
            return Optional.ofNullable(clazz.cast(encryptionOptions()));
        case "EngineType":
            return Optional.ofNullable(clazz.cast(engineType()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "HostInstanceType":
            return Optional.ofNullable(clazz.cast(hostInstanceType()));
        case "BrokerId":
            return Optional.ofNullable(clazz.cast(brokerId()));
        case "LdapServerMetadata":
            return Optional.ofNullable(clazz.cast(ldapServerMetadata()));
        case "Logs":
            return Optional.ofNullable(clazz.cast(logs()));
        case "MaintenanceWindowStartTime":
            return Optional.ofNullable(clazz.cast(maintenanceWindowStartTime()));
        case "PubliclyAccessible":
            return Optional.ofNullable(clazz.cast(publiclyAccessible()));
        case "SecurityGroups":
            return Optional.ofNullable(clazz.cast(securityGroups()));
        case "StorageType":
            return Optional.ofNullable(clazz.cast(storageType()));
        case "SubnetIds":
            return Optional.ofNullable(clazz.cast(subnetIds()));
        case "Users":
            return Optional.ofNullable(clazz.cast(users()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AuthenticationStrategy", AUTHENTICATION_STRATEGY_FIELD);
        map.put("AutoMinorVersionUpgrade", AUTO_MINOR_VERSION_UPGRADE_FIELD);
        map.put("BrokerArn", BROKER_ARN_FIELD);
        map.put("BrokerName", BROKER_NAME_FIELD);
        map.put("DeploymentMode", DEPLOYMENT_MODE_FIELD);
        map.put("EncryptionOptions", ENCRYPTION_OPTIONS_FIELD);
        map.put("EngineType", ENGINE_TYPE_FIELD);
        map.put("EngineVersion", ENGINE_VERSION_FIELD);
        map.put("HostInstanceType", HOST_INSTANCE_TYPE_FIELD);
        map.put("BrokerId", BROKER_ID_FIELD);
        map.put("LdapServerMetadata", LDAP_SERVER_METADATA_FIELD);
        map.put("Logs", LOGS_FIELD);
        map.put("MaintenanceWindowStartTime", MAINTENANCE_WINDOW_START_TIME_FIELD);
        map.put("PubliclyAccessible", PUBLICLY_ACCESSIBLE_FIELD);
        map.put("SecurityGroups", SECURITY_GROUPS_FIELD);
        map.put("StorageType", STORAGE_TYPE_FIELD);
        map.put("SubnetIds", SUBNET_IDS_FIELD);
        map.put("Users", USERS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AwsAmazonMqBrokerDetails, T> g) {
        return obj -> g.apply((AwsAmazonMqBrokerDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AwsAmazonMqBrokerDetails> {
        /**
         * <p>
         * The authentication strategy used to secure the broker. The default is <code>SIMPLE</code>.
         * </p>
         * 
         * @param authenticationStrategy
         *        The authentication strategy used to secure the broker. The default is <code>SIMPLE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authenticationStrategy(String authenticationStrategy);

        /**
         * <p>
         * Whether automatically upgrade new minor versions for brokers, as new versions are released and supported by
         * Amazon MQ. Automatic upgrades occur during the scheduled maintenance window of the broker or after a manual
         * broker reboot.
         * </p>
         * 
         * @param autoMinorVersionUpgrade
         *        Whether automatically upgrade new minor versions for brokers, as new versions are released and
         *        supported by Amazon MQ. Automatic upgrades occur during the scheduled maintenance window of the broker
         *        or after a manual broker reboot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoMinorVersionUpgrade(Boolean autoMinorVersionUpgrade);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the broker.
         * </p>
         * 
         * @param brokerArn
         *        The Amazon Resource Name (ARN) of the broker.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder brokerArn(String brokerArn);

        /**
         * <p>
         * The broker's name.
         * </p>
         * 
         * @param brokerName
         *        The broker's name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder brokerName(String brokerName);

        /**
         * <p>
         * The broker's deployment mode.
         * </p>
         * 
         * @param deploymentMode
         *        The broker's deployment mode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentMode(String deploymentMode);

        /**
         * <p>
         * Encryption options for the broker. Doesn’t apply to RabbitMQ brokers.
         * </p>
         * 
         * @param encryptionOptions
         *        Encryption options for the broker. Doesn’t apply to RabbitMQ brokers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionOptions(AwsAmazonMqBrokerEncryptionOptionsDetails encryptionOptions);

        /**
         * <p>
         * Encryption options for the broker. Doesn’t apply to RabbitMQ brokers.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link AwsAmazonMqBrokerEncryptionOptionsDetails.Builder} avoiding the need to create one manually via
         * {@link AwsAmazonMqBrokerEncryptionOptionsDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AwsAmazonMqBrokerEncryptionOptionsDetails.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #encryptionOptions(AwsAmazonMqBrokerEncryptionOptionsDetails)}.
         * 
         * @param encryptionOptions
         *        a consumer that will call methods on {@link AwsAmazonMqBrokerEncryptionOptionsDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionOptions(AwsAmazonMqBrokerEncryptionOptionsDetails)
         */
        default Builder encryptionOptions(Consumer<AwsAmazonMqBrokerEncryptionOptionsDetails.Builder> encryptionOptions) {
            return encryptionOptions(AwsAmazonMqBrokerEncryptionOptionsDetails.builder().applyMutation(encryptionOptions).build());
        }

        /**
         * <p>
         * The type of broker engine.
         * </p>
         * 
         * @param engineType
         *        The type of broker engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineType(String engineType);

        /**
         * <p>
         * The version of the broker engine.
         * </p>
         * 
         * @param engineVersion
         *        The version of the broker engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * The broker's instance type.
         * </p>
         * 
         * @param hostInstanceType
         *        The broker's instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostInstanceType(String hostInstanceType);

        /**
         * <p>
         * The unique ID that Amazon MQ generates for the broker.
         * </p>
         * 
         * @param brokerId
         *        The unique ID that Amazon MQ generates for the broker.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder brokerId(String brokerId);

        /**
         * <p>
         * The metadata of the Lightweight Directory Access Protocol (LDAP) server used to authenticate and authorize
         * connections to the broker. This is an optional failover server.
         * </p>
         * 
         * @param ldapServerMetadata
         *        The metadata of the Lightweight Directory Access Protocol (LDAP) server used to authenticate and
         *        authorize connections to the broker. This is an optional failover server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ldapServerMetadata(AwsAmazonMqBrokerLdapServerMetadataDetails ldapServerMetadata);

        /**
         * <p>
         * The metadata of the Lightweight Directory Access Protocol (LDAP) server used to authenticate and authorize
         * connections to the broker. This is an optional failover server.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link AwsAmazonMqBrokerLdapServerMetadataDetails.Builder} avoiding the need to create one manually via
         * {@link AwsAmazonMqBrokerLdapServerMetadataDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AwsAmazonMqBrokerLdapServerMetadataDetails.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #ldapServerMetadata(AwsAmazonMqBrokerLdapServerMetadataDetails)}.
         * 
         * @param ldapServerMetadata
         *        a consumer that will call methods on {@link AwsAmazonMqBrokerLdapServerMetadataDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ldapServerMetadata(AwsAmazonMqBrokerLdapServerMetadataDetails)
         */
        default Builder ldapServerMetadata(Consumer<AwsAmazonMqBrokerLdapServerMetadataDetails.Builder> ldapServerMetadata) {
            return ldapServerMetadata(AwsAmazonMqBrokerLdapServerMetadataDetails.builder().applyMutation(ldapServerMetadata)
                    .build());
        }

        /**
         * <p>
         * Turns on Amazon CloudWatch logging for brokers.
         * </p>
         * 
         * @param logs
         *        Turns on Amazon CloudWatch logging for brokers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logs(AwsAmazonMqBrokerLogsDetails logs);

        /**
         * <p>
         * Turns on Amazon CloudWatch logging for brokers.
         * </p>
         * This is a convenience method that creates an instance of the {@link AwsAmazonMqBrokerLogsDetails.Builder}
         * avoiding the need to create one manually via {@link AwsAmazonMqBrokerLogsDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AwsAmazonMqBrokerLogsDetails.Builder#build()} is called
         * immediately and its result is passed to {@link #logs(AwsAmazonMqBrokerLogsDetails)}.
         * 
         * @param logs
         *        a consumer that will call methods on {@link AwsAmazonMqBrokerLogsDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logs(AwsAmazonMqBrokerLogsDetails)
         */
        default Builder logs(Consumer<AwsAmazonMqBrokerLogsDetails.Builder> logs) {
            return logs(AwsAmazonMqBrokerLogsDetails.builder().applyMutation(logs).build());
        }

        /**
         * <p>
         * The scheduled time period (UTC) during which Amazon MQ begins to apply pending updates or patches to the
         * broker.
         * </p>
         * 
         * @param maintenanceWindowStartTime
         *        The scheduled time period (UTC) during which Amazon MQ begins to apply pending updates or patches to
         *        the broker.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maintenanceWindowStartTime(AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails maintenanceWindowStartTime);

        /**
         * <p>
         * The scheduled time period (UTC) during which Amazon MQ begins to apply pending updates or patches to the
         * broker.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails.Builder} avoiding the need to create one manually
         * via {@link AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #maintenanceWindowStartTime(AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails)}.
         * 
         * @param maintenanceWindowStartTime
         *        a consumer that will call methods on
         *        {@link AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #maintenanceWindowStartTime(AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails)
         */
        default Builder maintenanceWindowStartTime(
                Consumer<AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails.Builder> maintenanceWindowStartTime) {
            return maintenanceWindowStartTime(AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails.builder()
                    .applyMutation(maintenanceWindowStartTime).build());
        }

        /**
         * <p>
         * Permits connections from applications outside of the VPC that hosts the broker's subnets.
         * </p>
         * 
         * @param publiclyAccessible
         *        Permits connections from applications outside of the VPC that hosts the broker's subnets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publiclyAccessible(Boolean publiclyAccessible);

        /**
         * <p>
         * The list of rules (one minimum, 125 maximum) that authorize connections to brokers.
         * </p>
         * 
         * @param securityGroups
         *        The list of rules (one minimum, 125 maximum) that authorize connections to brokers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(Collection<String> securityGroups);

        /**
         * <p>
         * The list of rules (one minimum, 125 maximum) that authorize connections to brokers.
         * </p>
         * 
         * @param securityGroups
         *        The list of rules (one minimum, 125 maximum) that authorize connections to brokers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(String... securityGroups);

        /**
         * <p>
         * The broker's storage type.
         * </p>
         * 
         * @param storageType
         *        The broker's storage type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageType(String storageType);

        /**
         * <p>
         * The list of groups that define which subnets and IP ranges the broker can use from different Availability
         * Zones.
         * </p>
         * 
         * @param subnetIds
         *        The list of groups that define which subnets and IP ranges the broker can use from different
         *        Availability Zones.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(Collection<String> subnetIds);

        /**
         * <p>
         * The list of groups that define which subnets and IP ranges the broker can use from different Availability
         * Zones.
         * </p>
         * 
         * @param subnetIds
         *        The list of groups that define which subnets and IP ranges the broker can use from different
         *        Availability Zones.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(String... subnetIds);

        /**
         * <p>
         * The list of all broker usernames for the specified broker. Doesn't apply to RabbitMQ brokers.
         * </p>
         * 
         * @param users
         *        The list of all broker usernames for the specified broker. Doesn't apply to RabbitMQ brokers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder users(Collection<AwsAmazonMqBrokerUsersDetails> users);

        /**
         * <p>
         * The list of all broker usernames for the specified broker. Doesn't apply to RabbitMQ brokers.
         * </p>
         * 
         * @param users
         *        The list of all broker usernames for the specified broker. Doesn't apply to RabbitMQ brokers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder users(AwsAmazonMqBrokerUsersDetails... users);

        /**
         * <p>
         * The list of all broker usernames for the specified broker. Doesn't apply to RabbitMQ brokers.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.securityhub.model.AwsAmazonMqBrokerUsersDetails.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.securityhub.model.AwsAmazonMqBrokerUsersDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.securityhub.model.AwsAmazonMqBrokerUsersDetails.Builder#build()} is
         * called immediately and its result is passed to {@link #users(List<AwsAmazonMqBrokerUsersDetails>)}.
         * 
         * @param users
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.securityhub.model.AwsAmazonMqBrokerUsersDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #users(java.util.Collection<AwsAmazonMqBrokerUsersDetails>)
         */
        Builder users(Consumer<AwsAmazonMqBrokerUsersDetails.Builder>... users);
    }

    static final class BuilderImpl implements Builder {
        private String authenticationStrategy;

        private Boolean autoMinorVersionUpgrade;

        private String brokerArn;

        private String brokerName;

        private String deploymentMode;

        private AwsAmazonMqBrokerEncryptionOptionsDetails encryptionOptions;

        private String engineType;

        private String engineVersion;

        private String hostInstanceType;

        private String brokerId;

        private AwsAmazonMqBrokerLdapServerMetadataDetails ldapServerMetadata;

        private AwsAmazonMqBrokerLogsDetails logs;

        private AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails maintenanceWindowStartTime;

        private Boolean publiclyAccessible;

        private List<String> securityGroups = DefaultSdkAutoConstructList.getInstance();

        private String storageType;

        private List<String> subnetIds = DefaultSdkAutoConstructList.getInstance();

        private List<AwsAmazonMqBrokerUsersDetails> users = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AwsAmazonMqBrokerDetails model) {
            authenticationStrategy(model.authenticationStrategy);
            autoMinorVersionUpgrade(model.autoMinorVersionUpgrade);
            brokerArn(model.brokerArn);
            brokerName(model.brokerName);
            deploymentMode(model.deploymentMode);
            encryptionOptions(model.encryptionOptions);
            engineType(model.engineType);
            engineVersion(model.engineVersion);
            hostInstanceType(model.hostInstanceType);
            brokerId(model.brokerId);
            ldapServerMetadata(model.ldapServerMetadata);
            logs(model.logs);
            maintenanceWindowStartTime(model.maintenanceWindowStartTime);
            publiclyAccessible(model.publiclyAccessible);
            securityGroups(model.securityGroups);
            storageType(model.storageType);
            subnetIds(model.subnetIds);
            users(model.users);
        }

        public final String getAuthenticationStrategy() {
            return authenticationStrategy;
        }

        public final void setAuthenticationStrategy(String authenticationStrategy) {
            this.authenticationStrategy = authenticationStrategy;
        }

        @Override
        public final Builder authenticationStrategy(String authenticationStrategy) {
            this.authenticationStrategy = authenticationStrategy;
            return this;
        }

        public final Boolean getAutoMinorVersionUpgrade() {
            return autoMinorVersionUpgrade;
        }

        public final void setAutoMinorVersionUpgrade(Boolean autoMinorVersionUpgrade) {
            this.autoMinorVersionUpgrade = autoMinorVersionUpgrade;
        }

        @Override
        public final Builder autoMinorVersionUpgrade(Boolean autoMinorVersionUpgrade) {
            this.autoMinorVersionUpgrade = autoMinorVersionUpgrade;
            return this;
        }

        public final String getBrokerArn() {
            return brokerArn;
        }

        public final void setBrokerArn(String brokerArn) {
            this.brokerArn = brokerArn;
        }

        @Override
        public final Builder brokerArn(String brokerArn) {
            this.brokerArn = brokerArn;
            return this;
        }

        public final String getBrokerName() {
            return brokerName;
        }

        public final void setBrokerName(String brokerName) {
            this.brokerName = brokerName;
        }

        @Override
        public final Builder brokerName(String brokerName) {
            this.brokerName = brokerName;
            return this;
        }

        public final String getDeploymentMode() {
            return deploymentMode;
        }

        public final void setDeploymentMode(String deploymentMode) {
            this.deploymentMode = deploymentMode;
        }

        @Override
        public final Builder deploymentMode(String deploymentMode) {
            this.deploymentMode = deploymentMode;
            return this;
        }

        public final AwsAmazonMqBrokerEncryptionOptionsDetails.Builder getEncryptionOptions() {
            return encryptionOptions != null ? encryptionOptions.toBuilder() : null;
        }

        public final void setEncryptionOptions(AwsAmazonMqBrokerEncryptionOptionsDetails.BuilderImpl encryptionOptions) {
            this.encryptionOptions = encryptionOptions != null ? encryptionOptions.build() : null;
        }

        @Override
        public final Builder encryptionOptions(AwsAmazonMqBrokerEncryptionOptionsDetails encryptionOptions) {
            this.encryptionOptions = encryptionOptions;
            return this;
        }

        public final String getEngineType() {
            return engineType;
        }

        public final void setEngineType(String engineType) {
            this.engineType = engineType;
        }

        @Override
        public final Builder engineType(String engineType) {
            this.engineType = engineType;
            return this;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final String getHostInstanceType() {
            return hostInstanceType;
        }

        public final void setHostInstanceType(String hostInstanceType) {
            this.hostInstanceType = hostInstanceType;
        }

        @Override
        public final Builder hostInstanceType(String hostInstanceType) {
            this.hostInstanceType = hostInstanceType;
            return this;
        }

        public final String getBrokerId() {
            return brokerId;
        }

        public final void setBrokerId(String brokerId) {
            this.brokerId = brokerId;
        }

        @Override
        public final Builder brokerId(String brokerId) {
            this.brokerId = brokerId;
            return this;
        }

        public final AwsAmazonMqBrokerLdapServerMetadataDetails.Builder getLdapServerMetadata() {
            return ldapServerMetadata != null ? ldapServerMetadata.toBuilder() : null;
        }

        public final void setLdapServerMetadata(AwsAmazonMqBrokerLdapServerMetadataDetails.BuilderImpl ldapServerMetadata) {
            this.ldapServerMetadata = ldapServerMetadata != null ? ldapServerMetadata.build() : null;
        }

        @Override
        public final Builder ldapServerMetadata(AwsAmazonMqBrokerLdapServerMetadataDetails ldapServerMetadata) {
            this.ldapServerMetadata = ldapServerMetadata;
            return this;
        }

        public final AwsAmazonMqBrokerLogsDetails.Builder getLogs() {
            return logs != null ? logs.toBuilder() : null;
        }

        public final void setLogs(AwsAmazonMqBrokerLogsDetails.BuilderImpl logs) {
            this.logs = logs != null ? logs.build() : null;
        }

        @Override
        public final Builder logs(AwsAmazonMqBrokerLogsDetails logs) {
            this.logs = logs;
            return this;
        }

        public final AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails.Builder getMaintenanceWindowStartTime() {
            return maintenanceWindowStartTime != null ? maintenanceWindowStartTime.toBuilder() : null;
        }

        public final void setMaintenanceWindowStartTime(
                AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails.BuilderImpl maintenanceWindowStartTime) {
            this.maintenanceWindowStartTime = maintenanceWindowStartTime != null ? maintenanceWindowStartTime.build() : null;
        }

        @Override
        public final Builder maintenanceWindowStartTime(
                AwsAmazonMqBrokerMaintenanceWindowStartTimeDetails maintenanceWindowStartTime) {
            this.maintenanceWindowStartTime = maintenanceWindowStartTime;
            return this;
        }

        public final Boolean getPubliclyAccessible() {
            return publiclyAccessible;
        }

        public final void setPubliclyAccessible(Boolean publiclyAccessible) {
            this.publiclyAccessible = publiclyAccessible;
        }

        @Override
        public final Builder publiclyAccessible(Boolean publiclyAccessible) {
            this.publiclyAccessible = publiclyAccessible;
            return this;
        }

        public final Collection<String> getSecurityGroups() {
            if (securityGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroups;
        }

        public final void setSecurityGroups(Collection<String> securityGroups) {
            this.securityGroups = StringListCopier.copy(securityGroups);
        }

        @Override
        public final Builder securityGroups(Collection<String> securityGroups) {
            this.securityGroups = StringListCopier.copy(securityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroups(String... securityGroups) {
            securityGroups(Arrays.asList(securityGroups));
            return this;
        }

        public final String getStorageType() {
            return storageType;
        }

        public final void setStorageType(String storageType) {
            this.storageType = storageType;
        }

        @Override
        public final Builder storageType(String storageType) {
            this.storageType = storageType;
            return this;
        }

        public final Collection<String> getSubnetIds() {
            if (subnetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return subnetIds;
        }

        public final void setSubnetIds(Collection<String> subnetIds) {
            this.subnetIds = StringListCopier.copy(subnetIds);
        }

        @Override
        public final Builder subnetIds(Collection<String> subnetIds) {
            this.subnetIds = StringListCopier.copy(subnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetIds(String... subnetIds) {
            subnetIds(Arrays.asList(subnetIds));
            return this;
        }

        public final List<AwsAmazonMqBrokerUsersDetails.Builder> getUsers() {
            List<AwsAmazonMqBrokerUsersDetails.Builder> result = AwsAmazonMqBrokerUsersListCopier.copyToBuilder(this.users);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUsers(Collection<AwsAmazonMqBrokerUsersDetails.BuilderImpl> users) {
            this.users = AwsAmazonMqBrokerUsersListCopier.copyFromBuilder(users);
        }

        @Override
        public final Builder users(Collection<AwsAmazonMqBrokerUsersDetails> users) {
            this.users = AwsAmazonMqBrokerUsersListCopier.copy(users);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder users(AwsAmazonMqBrokerUsersDetails... users) {
            users(Arrays.asList(users));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder users(Consumer<AwsAmazonMqBrokerUsersDetails.Builder>... users) {
            users(Stream.of(users).map(c -> AwsAmazonMqBrokerUsersDetails.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public AwsAmazonMqBrokerDetails build() {
            return new AwsAmazonMqBrokerDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
