/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.securityhub;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.securityhub.model.AcceptInvitationRequest;
import software.amazon.awssdk.services.securityhub.model.AcceptInvitationResponse;
import software.amazon.awssdk.services.securityhub.model.AccessDeniedException;
import software.amazon.awssdk.services.securityhub.model.BatchDisableStandardsRequest;
import software.amazon.awssdk.services.securityhub.model.BatchDisableStandardsResponse;
import software.amazon.awssdk.services.securityhub.model.BatchEnableStandardsRequest;
import software.amazon.awssdk.services.securityhub.model.BatchEnableStandardsResponse;
import software.amazon.awssdk.services.securityhub.model.BatchImportFindingsRequest;
import software.amazon.awssdk.services.securityhub.model.BatchImportFindingsResponse;
import software.amazon.awssdk.services.securityhub.model.CreateActionTargetRequest;
import software.amazon.awssdk.services.securityhub.model.CreateActionTargetResponse;
import software.amazon.awssdk.services.securityhub.model.CreateInsightRequest;
import software.amazon.awssdk.services.securityhub.model.CreateInsightResponse;
import software.amazon.awssdk.services.securityhub.model.CreateMembersRequest;
import software.amazon.awssdk.services.securityhub.model.CreateMembersResponse;
import software.amazon.awssdk.services.securityhub.model.DeclineInvitationsRequest;
import software.amazon.awssdk.services.securityhub.model.DeclineInvitationsResponse;
import software.amazon.awssdk.services.securityhub.model.DeleteActionTargetRequest;
import software.amazon.awssdk.services.securityhub.model.DeleteActionTargetResponse;
import software.amazon.awssdk.services.securityhub.model.DeleteInsightRequest;
import software.amazon.awssdk.services.securityhub.model.DeleteInsightResponse;
import software.amazon.awssdk.services.securityhub.model.DeleteInvitationsRequest;
import software.amazon.awssdk.services.securityhub.model.DeleteInvitationsResponse;
import software.amazon.awssdk.services.securityhub.model.DeleteMembersRequest;
import software.amazon.awssdk.services.securityhub.model.DeleteMembersResponse;
import software.amazon.awssdk.services.securityhub.model.DescribeActionTargetsRequest;
import software.amazon.awssdk.services.securityhub.model.DescribeActionTargetsResponse;
import software.amazon.awssdk.services.securityhub.model.DescribeHubRequest;
import software.amazon.awssdk.services.securityhub.model.DescribeHubResponse;
import software.amazon.awssdk.services.securityhub.model.DescribeProductsRequest;
import software.amazon.awssdk.services.securityhub.model.DescribeProductsResponse;
import software.amazon.awssdk.services.securityhub.model.DescribeStandardsControlsRequest;
import software.amazon.awssdk.services.securityhub.model.DescribeStandardsControlsResponse;
import software.amazon.awssdk.services.securityhub.model.DisableImportFindingsForProductRequest;
import software.amazon.awssdk.services.securityhub.model.DisableImportFindingsForProductResponse;
import software.amazon.awssdk.services.securityhub.model.DisableSecurityHubRequest;
import software.amazon.awssdk.services.securityhub.model.DisableSecurityHubResponse;
import software.amazon.awssdk.services.securityhub.model.DisassociateFromMasterAccountRequest;
import software.amazon.awssdk.services.securityhub.model.DisassociateFromMasterAccountResponse;
import software.amazon.awssdk.services.securityhub.model.DisassociateMembersRequest;
import software.amazon.awssdk.services.securityhub.model.DisassociateMembersResponse;
import software.amazon.awssdk.services.securityhub.model.EnableImportFindingsForProductRequest;
import software.amazon.awssdk.services.securityhub.model.EnableImportFindingsForProductResponse;
import software.amazon.awssdk.services.securityhub.model.EnableSecurityHubRequest;
import software.amazon.awssdk.services.securityhub.model.EnableSecurityHubResponse;
import software.amazon.awssdk.services.securityhub.model.GetEnabledStandardsRequest;
import software.amazon.awssdk.services.securityhub.model.GetEnabledStandardsResponse;
import software.amazon.awssdk.services.securityhub.model.GetFindingsRequest;
import software.amazon.awssdk.services.securityhub.model.GetFindingsResponse;
import software.amazon.awssdk.services.securityhub.model.GetInsightResultsRequest;
import software.amazon.awssdk.services.securityhub.model.GetInsightResultsResponse;
import software.amazon.awssdk.services.securityhub.model.GetInsightsRequest;
import software.amazon.awssdk.services.securityhub.model.GetInsightsResponse;
import software.amazon.awssdk.services.securityhub.model.GetInvitationsCountRequest;
import software.amazon.awssdk.services.securityhub.model.GetInvitationsCountResponse;
import software.amazon.awssdk.services.securityhub.model.GetMasterAccountRequest;
import software.amazon.awssdk.services.securityhub.model.GetMasterAccountResponse;
import software.amazon.awssdk.services.securityhub.model.GetMembersRequest;
import software.amazon.awssdk.services.securityhub.model.GetMembersResponse;
import software.amazon.awssdk.services.securityhub.model.InternalException;
import software.amazon.awssdk.services.securityhub.model.InvalidAccessException;
import software.amazon.awssdk.services.securityhub.model.InvalidInputException;
import software.amazon.awssdk.services.securityhub.model.InviteMembersRequest;
import software.amazon.awssdk.services.securityhub.model.InviteMembersResponse;
import software.amazon.awssdk.services.securityhub.model.LimitExceededException;
import software.amazon.awssdk.services.securityhub.model.ListEnabledProductsForImportRequest;
import software.amazon.awssdk.services.securityhub.model.ListEnabledProductsForImportResponse;
import software.amazon.awssdk.services.securityhub.model.ListInvitationsRequest;
import software.amazon.awssdk.services.securityhub.model.ListInvitationsResponse;
import software.amazon.awssdk.services.securityhub.model.ListMembersRequest;
import software.amazon.awssdk.services.securityhub.model.ListMembersResponse;
import software.amazon.awssdk.services.securityhub.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.securityhub.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.securityhub.model.ResourceConflictException;
import software.amazon.awssdk.services.securityhub.model.ResourceNotFoundException;
import software.amazon.awssdk.services.securityhub.model.SecurityHubException;
import software.amazon.awssdk.services.securityhub.model.SecurityHubRequest;
import software.amazon.awssdk.services.securityhub.model.TagResourceRequest;
import software.amazon.awssdk.services.securityhub.model.TagResourceResponse;
import software.amazon.awssdk.services.securityhub.model.UntagResourceRequest;
import software.amazon.awssdk.services.securityhub.model.UntagResourceResponse;
import software.amazon.awssdk.services.securityhub.model.UpdateActionTargetRequest;
import software.amazon.awssdk.services.securityhub.model.UpdateActionTargetResponse;
import software.amazon.awssdk.services.securityhub.model.UpdateFindingsRequest;
import software.amazon.awssdk.services.securityhub.model.UpdateFindingsResponse;
import software.amazon.awssdk.services.securityhub.model.UpdateInsightRequest;
import software.amazon.awssdk.services.securityhub.model.UpdateInsightResponse;
import software.amazon.awssdk.services.securityhub.model.UpdateStandardsControlRequest;
import software.amazon.awssdk.services.securityhub.model.UpdateStandardsControlResponse;
import software.amazon.awssdk.services.securityhub.paginators.DescribeActionTargetsIterable;
import software.amazon.awssdk.services.securityhub.paginators.DescribeProductsIterable;
import software.amazon.awssdk.services.securityhub.paginators.GetFindingsIterable;
import software.amazon.awssdk.services.securityhub.paginators.GetInsightsIterable;
import software.amazon.awssdk.services.securityhub.paginators.ListEnabledProductsForImportIterable;
import software.amazon.awssdk.services.securityhub.transform.AcceptInvitationRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.BatchDisableStandardsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.BatchEnableStandardsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.BatchImportFindingsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.CreateActionTargetRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.CreateInsightRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.CreateMembersRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DeclineInvitationsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DeleteActionTargetRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DeleteInsightRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DeleteInvitationsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DeleteMembersRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DescribeActionTargetsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DescribeHubRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DescribeProductsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DescribeStandardsControlsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DisableImportFindingsForProductRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DisableSecurityHubRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DisassociateFromMasterAccountRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.DisassociateMembersRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.EnableImportFindingsForProductRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.EnableSecurityHubRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetEnabledStandardsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetFindingsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetInsightResultsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetInsightsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetInvitationsCountRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetMasterAccountRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.GetMembersRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.InviteMembersRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.ListEnabledProductsForImportRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.ListInvitationsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.ListMembersRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.UpdateActionTargetRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.UpdateFindingsRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.UpdateInsightRequestMarshaller;
import software.amazon.awssdk.services.securityhub.transform.UpdateStandardsControlRequestMarshaller;

/**
 * Internal implementation of {@link SecurityHubClient}.
 *
 * @see SecurityHubClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSecurityHubClient implements SecurityHubClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSecurityHubClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Accepts the invitation to be a member account and be monitored by the Security Hub master account that the
     * invitation was sent from. When the member account accepts the invitation, permission is granted to the master
     * account to view findings generated in the member account.
     * </p>
     *
     * @param acceptInvitationRequest
     * @return Result of the AcceptInvitation operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.AcceptInvitation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/AcceptInvitation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AcceptInvitationResponse acceptInvitation(AcceptInvitationRequest acceptInvitationRequest) throws InternalException,
            InvalidInputException, LimitExceededException, ResourceNotFoundException, InvalidAccessException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AcceptInvitationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AcceptInvitationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AcceptInvitationRequest, AcceptInvitationResponse>()
                .withOperationName("AcceptInvitation").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(acceptInvitationRequest)
                .withMarshaller(new AcceptInvitationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Disables the standards specified by the provided <code>StandardsSubscriptionArns</code>. For more information,
     * see <a href="https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-standards.html">Standards
     * Supported in AWS Security Hub</a>.
     * </p>
     *
     * @param batchDisableStandardsRequest
     * @return Result of the BatchDisableStandards operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.BatchDisableStandards
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/BatchDisableStandards"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchDisableStandardsResponse batchDisableStandards(BatchDisableStandardsRequest batchDisableStandardsRequest)
            throws InternalException, InvalidInputException, InvalidAccessException, LimitExceededException, AwsServiceException,
            SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchDisableStandardsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchDisableStandardsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<BatchDisableStandardsRequest, BatchDisableStandardsResponse>()
                .withOperationName("BatchDisableStandards").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(batchDisableStandardsRequest)
                .withMarshaller(new BatchDisableStandardsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Enables the standards specified by the provided <code>standardsArn</code>. In this release, only CIS AWS
     * Foundations standards are supported. For more information, see <a
     * href="https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-standards.html">Standards Supported in
     * AWS Security Hub</a>.
     * </p>
     *
     * @param batchEnableStandardsRequest
     * @return Result of the BatchEnableStandards operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.BatchEnableStandards
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/BatchEnableStandards"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchEnableStandardsResponse batchEnableStandards(BatchEnableStandardsRequest batchEnableStandardsRequest)
            throws InternalException, InvalidInputException, InvalidAccessException, LimitExceededException, AwsServiceException,
            SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchEnableStandardsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchEnableStandardsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<BatchEnableStandardsRequest, BatchEnableStandardsResponse>()
                .withOperationName("BatchEnableStandards").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(batchEnableStandardsRequest)
                .withMarshaller(new BatchEnableStandardsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Imports security findings generated from an integrated third-party product into Security Hub. This action is
     * requested by the integrated product to import its findings into Security Hub. The maximum allowed size for a
     * finding is 240 Kb. An error is returned for any finding larger than 240 Kb.
     * </p>
     *
     * @param batchImportFindingsRequest
     * @return Result of the BatchImportFindings operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.BatchImportFindings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/BatchImportFindings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchImportFindingsResponse batchImportFindings(BatchImportFindingsRequest batchImportFindingsRequest)
            throws InternalException, InvalidInputException, LimitExceededException, InvalidAccessException, AwsServiceException,
            SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchImportFindingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchImportFindingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<BatchImportFindingsRequest, BatchImportFindingsResponse>()
                .withOperationName("BatchImportFindings").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(batchImportFindingsRequest)
                .withMarshaller(new BatchImportFindingsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a custom action target in Security Hub. You can use custom actions on findings and insights in Security
     * Hub to trigger target actions in Amazon CloudWatch Events.
     * </p>
     *
     * @param createActionTargetRequest
     * @return Result of the CreateActionTarget operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws ResourceConflictException
     *         The resource specified in the request conflicts with an existing resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.CreateActionTarget
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/CreateActionTarget" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateActionTargetResponse createActionTarget(CreateActionTargetRequest createActionTargetRequest)
            throws InternalException, InvalidInputException, InvalidAccessException, LimitExceededException,
            ResourceConflictException, AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateActionTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateActionTargetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateActionTargetRequest, CreateActionTargetResponse>()
                .withOperationName("CreateActionTarget").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createActionTargetRequest)
                .withMarshaller(new CreateActionTargetRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a custom insight in Security Hub. An insight is a consolidation of findings that relate to a security
     * issue that requires attention or remediation. Use the <code>GroupByAttribute</code> to group the related findings
     * in the insight.
     * </p>
     *
     * @param createInsightRequest
     * @return Result of the CreateInsight operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws ResourceConflictException
     *         The resource specified in the request conflicts with an existing resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.CreateInsight
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/CreateInsight" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateInsightResponse createInsight(CreateInsightRequest createInsightRequest) throws InternalException,
            InvalidInputException, LimitExceededException, InvalidAccessException, ResourceConflictException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateInsightResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateInsightResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateInsightRequest, CreateInsightResponse>()
                .withOperationName("CreateInsight").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createInsightRequest)
                .withMarshaller(new CreateInsightRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a member association in Security Hub between the specified accounts and the account used to make the
     * request, which is the master account. To successfully create a member, you must use this action from an account
     * that already has Security Hub enabled. You can use the <a>EnableSecurityHub</a> to enable Security Hub.
     * </p>
     * <p>
     * After you use <code>CreateMembers</code> to create member account associations in Security Hub, you need to use
     * the <a>InviteMembers</a> action, which invites the accounts to enable Security Hub and become member accounts in
     * Security Hub. If the invitation is accepted by the account owner, the account becomes a member account in
     * Security Hub, and a permission policy is added that permits the master account to view the findings generated in
     * the member account. When Security Hub is enabled in the invited account, findings start being sent to both the
     * member and master accounts.
     * </p>
     * <p>
     * You can remove the association between the master and member accounts by using the
     * <a>DisassociateFromMasterAccount</a> or <a>DisassociateMembers</a> operation.
     * </p>
     *
     * @param createMembersRequest
     * @return Result of the CreateMembers operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws ResourceConflictException
     *         The resource specified in the request conflicts with an existing resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.CreateMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/CreateMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateMembersResponse createMembers(CreateMembersRequest createMembersRequest) throws InternalException,
            InvalidInputException, LimitExceededException, InvalidAccessException, ResourceConflictException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateMembersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateMembersRequest, CreateMembersResponse>()
                .withOperationName("CreateMembers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createMembersRequest)
                .withMarshaller(new CreateMembersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Declines invitations to become a member account.
     * </p>
     *
     * @param declineInvitationsRequest
     * @return Result of the DeclineInvitations operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.DeclineInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DeclineInvitations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeclineInvitationsResponse declineInvitations(DeclineInvitationsRequest declineInvitationsRequest)
            throws InternalException, InvalidInputException, InvalidAccessException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeclineInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeclineInvitationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeclineInvitationsRequest, DeclineInvitationsResponse>()
                .withOperationName("DeclineInvitations").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(declineInvitationsRequest)
                .withMarshaller(new DeclineInvitationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a custom action target from Security Hub. Deleting a custom action target doesn't affect any findings or
     * insights that were already sent to Amazon CloudWatch Events using the custom action.
     * </p>
     *
     * @param deleteActionTargetRequest
     * @return Result of the DeleteActionTarget operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.DeleteActionTarget
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DeleteActionTarget" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteActionTargetResponse deleteActionTarget(DeleteActionTargetRequest deleteActionTargetRequest)
            throws InternalException, InvalidInputException, InvalidAccessException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteActionTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteActionTargetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteActionTargetRequest, DeleteActionTargetResponse>()
                .withOperationName("DeleteActionTarget").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteActionTargetRequest)
                .withMarshaller(new DeleteActionTargetRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the insight specified by the <code>InsightArn</code>.
     * </p>
     *
     * @param deleteInsightRequest
     * @return Result of the DeleteInsight operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.DeleteInsight
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DeleteInsight" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteInsightResponse deleteInsight(DeleteInsightRequest deleteInsightRequest) throws InternalException,
            InvalidInputException, InvalidAccessException, LimitExceededException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteInsightResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteInsightResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteInsightRequest, DeleteInsightResponse>()
                .withOperationName("DeleteInsight").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteInsightRequest)
                .withMarshaller(new DeleteInsightRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes invitations received by the AWS account to become a member account.
     * </p>
     *
     * @param deleteInvitationsRequest
     * @return Result of the DeleteInvitations operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.DeleteInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DeleteInvitations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteInvitationsResponse deleteInvitations(DeleteInvitationsRequest deleteInvitationsRequest)
            throws InternalException, InvalidInputException, LimitExceededException, ResourceNotFoundException,
            InvalidAccessException, AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteInvitationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteInvitationsRequest, DeleteInvitationsResponse>()
                .withOperationName("DeleteInvitations").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteInvitationsRequest)
                .withMarshaller(new DeleteInvitationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the specified member accounts from Security Hub.
     * </p>
     *
     * @param deleteMembersRequest
     * @return Result of the DeleteMembers operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.DeleteMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DeleteMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteMembersResponse deleteMembers(DeleteMembersRequest deleteMembersRequest) throws InternalException,
            InvalidInputException, InvalidAccessException, LimitExceededException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteMembersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteMembersRequest, DeleteMembersResponse>()
                .withOperationName("DeleteMembers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteMembersRequest)
                .withMarshaller(new DeleteMembersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of the custom action targets in Security Hub in your account.
     * </p>
     *
     * @param describeActionTargetsRequest
     * @return Result of the DescribeActionTargets operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.DescribeActionTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeActionTargets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeActionTargetsResponse describeActionTargets(DescribeActionTargetsRequest describeActionTargetsRequest)
            throws InternalException, InvalidInputException, InvalidAccessException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeActionTargetsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeActionTargetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeActionTargetsRequest, DescribeActionTargetsResponse>()
                .withOperationName("DescribeActionTargets").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeActionTargetsRequest)
                .withMarshaller(new DescribeActionTargetsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of the custom action targets in Security Hub in your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeActionTargets(software.amazon.awssdk.services.securityhub.model.DescribeActionTargetsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.DescribeActionTargetsIterable responses = client.describeActionTargetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.securityhub.paginators.DescribeActionTargetsIterable responses = client
     *             .describeActionTargetsPaginator(request);
     *     for (software.amazon.awssdk.services.securityhub.model.DescribeActionTargetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.DescribeActionTargetsIterable responses = client.describeActionTargetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeActionTargets(software.amazon.awssdk.services.securityhub.model.DescribeActionTargetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeActionTargetsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.DescribeActionTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeActionTargets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeActionTargetsIterable describeActionTargetsPaginator(DescribeActionTargetsRequest describeActionTargetsRequest)
            throws InternalException, InvalidInputException, InvalidAccessException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        return new DescribeActionTargetsIterable(this, applyPaginatorUserAgent(describeActionTargetsRequest));
    }

    /**
     * <p>
     * Returns details about the Hub resource in your account, including the <code>HubArn</code> and the time when you
     * enabled Security Hub.
     * </p>
     *
     * @param describeHubRequest
     * @return Result of the DescribeHub operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.DescribeHub
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeHub" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeHubResponse describeHub(DescribeHubRequest describeHubRequest) throws InternalException,
            LimitExceededException, InvalidAccessException, InvalidInputException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeHubResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeHubResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeHubRequest, DescribeHubResponse>()
                .withOperationName("DescribeHub").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeHubRequest)
                .withMarshaller(new DescribeHubRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns information about the products available that you can subscribe to and integrate with Security Hub to
     * consolidate findings.
     * </p>
     *
     * @param describeProductsRequest
     * @return Result of the DescribeProducts operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.DescribeProducts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeProducts" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeProductsResponse describeProducts(DescribeProductsRequest describeProductsRequest) throws InternalException,
            LimitExceededException, InvalidAccessException, InvalidInputException, AwsServiceException, SdkClientException,
            SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeProductsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeProductsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeProductsRequest, DescribeProductsResponse>()
                .withOperationName("DescribeProducts").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeProductsRequest)
                .withMarshaller(new DescribeProductsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns information about the products available that you can subscribe to and integrate with Security Hub to
     * consolidate findings.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeProducts(software.amazon.awssdk.services.securityhub.model.DescribeProductsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.DescribeProductsIterable responses = client.describeProductsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.securityhub.paginators.DescribeProductsIterable responses = client
     *             .describeProductsPaginator(request);
     *     for (software.amazon.awssdk.services.securityhub.model.DescribeProductsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.DescribeProductsIterable responses = client.describeProductsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeProducts(software.amazon.awssdk.services.securityhub.model.DescribeProductsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeProductsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalException
     *         Internal server error.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.DescribeProducts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeProducts" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeProductsIterable describeProductsPaginator(DescribeProductsRequest describeProductsRequest)
            throws InternalException, LimitExceededException, InvalidAccessException, InvalidInputException, AwsServiceException,
            SdkClientException, SecurityHubException {
        return new DescribeProductsIterable(this, applyPaginatorUserAgent(describeProductsRequest));
    }

    /**
     * <p>
     * Returns a list of compliance standards controls.
     * </p>
     * <p>
     * For each control, the results include information about whether it is currently enabled, the severity, and a link
     * to remediation information.
     * </p>
     *
     * @param describeStandardsControlsRequest
     * @return Result of the DescribeStandardsControls operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.DescribeStandardsControls
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DescribeStandardsControls"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeStandardsControlsResponse describeStandardsControls(
            DescribeStandardsControlsRequest describeStandardsControlsRequest) throws InternalException, InvalidInputException,
            InvalidAccessException, ResourceNotFoundException, AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeStandardsControlsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeStandardsControlsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeStandardsControlsRequest, DescribeStandardsControlsResponse>()
                        .withOperationName("DescribeStandardsControls").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeStandardsControlsRequest)
                        .withMarshaller(new DescribeStandardsControlsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Disables the integration of the specified product with Security Hub. Findings from that product are no longer
     * sent to Security Hub after the integration is disabled.
     * </p>
     *
     * @param disableImportFindingsForProductRequest
     * @return Result of the DisableImportFindingsForProduct operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.DisableImportFindingsForProduct
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DisableImportFindingsForProduct"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableImportFindingsForProductResponse disableImportFindingsForProduct(
            DisableImportFindingsForProductRequest disableImportFindingsForProductRequest) throws InternalException,
            InvalidInputException, ResourceNotFoundException, InvalidAccessException, LimitExceededException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableImportFindingsForProductResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisableImportFindingsForProductResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DisableImportFindingsForProductRequest, DisableImportFindingsForProductResponse>()
                        .withOperationName("DisableImportFindingsForProduct").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(disableImportFindingsForProductRequest)
                        .withMarshaller(new DisableImportFindingsForProductRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Disables Security Hub in your account only in the current Region. To disable Security Hub in all Regions, you
     * must submit one request per Region where you have enabled Security Hub. When you disable Security Hub for a
     * master account, it doesn't disable Security Hub for any associated member accounts.
     * </p>
     * <p>
     * When you disable Security Hub, your existing findings and insights and any Security Hub configuration settings
     * are deleted after 90 days and can't be recovered. Any standards that were enabled are disabled, and your master
     * and member account associations are removed. If you want to save your existing findings, you must export them
     * before you disable Security Hub.
     * </p>
     *
     * @param disableSecurityHubRequest
     * @return Result of the DisableSecurityHub operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.DisableSecurityHub
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DisableSecurityHub" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DisableSecurityHubResponse disableSecurityHub(DisableSecurityHubRequest disableSecurityHubRequest)
            throws InternalException, LimitExceededException, InvalidAccessException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableSecurityHubResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisableSecurityHubResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DisableSecurityHubRequest, DisableSecurityHubResponse>()
                .withOperationName("DisableSecurityHub").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(disableSecurityHubRequest)
                .withMarshaller(new DisableSecurityHubRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Disassociates the current Security Hub member account from the associated master account.
     * </p>
     *
     * @param disassociateFromMasterAccountRequest
     * @return Result of the DisassociateFromMasterAccount operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.DisassociateFromMasterAccount
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DisassociateFromMasterAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateFromMasterAccountResponse disassociateFromMasterAccount(
            DisassociateFromMasterAccountRequest disassociateFromMasterAccountRequest) throws InternalException,
            InvalidInputException, InvalidAccessException, LimitExceededException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateFromMasterAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateFromMasterAccountResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DisassociateFromMasterAccountRequest, DisassociateFromMasterAccountResponse>()
                        .withOperationName("DisassociateFromMasterAccount").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(disassociateFromMasterAccountRequest)
                        .withMarshaller(new DisassociateFromMasterAccountRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Disassociates the specified member accounts from the associated master account.
     * </p>
     *
     * @param disassociateMembersRequest
     * @return Result of the DisassociateMembers operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.DisassociateMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/DisassociateMembers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateMembersResponse disassociateMembers(DisassociateMembersRequest disassociateMembersRequest)
            throws InternalException, InvalidInputException, InvalidAccessException, LimitExceededException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateMembersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateMembersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DisassociateMembersRequest, DisassociateMembersResponse>()
                .withOperationName("DisassociateMembers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(disassociateMembersRequest)
                .withMarshaller(new DisassociateMembersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Enables the integration of a partner product with Security Hub. Integrated products send findings to Security
     * Hub. When you enable a product integration, a permission policy that grants permission for the product to send
     * findings to Security Hub is applied.
     * </p>
     *
     * @param enableImportFindingsForProductRequest
     * @return Result of the EnableImportFindingsForProduct operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws ResourceConflictException
     *         The resource specified in the request conflicts with an existing resource.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.EnableImportFindingsForProduct
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/EnableImportFindingsForProduct"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EnableImportFindingsForProductResponse enableImportFindingsForProduct(
            EnableImportFindingsForProductRequest enableImportFindingsForProductRequest) throws InternalException,
            InvalidInputException, InvalidAccessException, ResourceConflictException, LimitExceededException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableImportFindingsForProductResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, EnableImportFindingsForProductResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<EnableImportFindingsForProductRequest, EnableImportFindingsForProductResponse>()
                        .withOperationName("EnableImportFindingsForProduct").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(enableImportFindingsForProductRequest)
                        .withMarshaller(new EnableImportFindingsForProductRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Enables Security Hub for your account in the current Region or the Region you specify in the request. Enabling
     * Security Hub also enables the CIS AWS Foundations standard. When you enable Security Hub, you grant to Security
     * Hub the permissions necessary to gather findings from AWS Config, Amazon GuardDuty, Amazon Inspector, and Amazon
     * Macie. To learn more, see <a
     * href="https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-settingup.html">Setting Up AWS
     * Security Hub</a>.
     * </p>
     *
     * @param enableSecurityHubRequest
     * @return Result of the EnableSecurityHub operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws ResourceConflictException
     *         The resource specified in the request conflicts with an existing resource.
     * @throws AccessDeniedException
     *         You don't have permission to perform the action specified in the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.EnableSecurityHub
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/EnableSecurityHub" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public EnableSecurityHubResponse enableSecurityHub(EnableSecurityHubRequest enableSecurityHubRequest)
            throws InternalException, LimitExceededException, InvalidAccessException, ResourceConflictException,
            AccessDeniedException, AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableSecurityHubResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                EnableSecurityHubResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<EnableSecurityHubRequest, EnableSecurityHubResponse>()
                .withOperationName("EnableSecurityHub").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(enableSecurityHubRequest)
                .withMarshaller(new EnableSecurityHubRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of the standards that are currently enabled.
     * </p>
     *
     * @param getEnabledStandardsRequest
     * @return Result of the GetEnabledStandards operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.GetEnabledStandards
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetEnabledStandards"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetEnabledStandardsResponse getEnabledStandards(GetEnabledStandardsRequest getEnabledStandardsRequest)
            throws InternalException, InvalidInputException, InvalidAccessException, LimitExceededException, AwsServiceException,
            SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEnabledStandardsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetEnabledStandardsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetEnabledStandardsRequest, GetEnabledStandardsResponse>()
                .withOperationName("GetEnabledStandards").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getEnabledStandardsRequest)
                .withMarshaller(new GetEnabledStandardsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of findings that match the specified criteria.
     * </p>
     *
     * @param getFindingsRequest
     * @return Result of the GetFindings operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.GetFindings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetFindings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetFindingsResponse getFindings(GetFindingsRequest getFindingsRequest) throws InternalException,
            InvalidInputException, InvalidAccessException, LimitExceededException, AwsServiceException, SdkClientException,
            SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetFindingsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetFindingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetFindingsRequest, GetFindingsResponse>()
                .withOperationName("GetFindings").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getFindingsRequest)
                .withMarshaller(new GetFindingsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of findings that match the specified criteria.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getFindings(software.amazon.awssdk.services.securityhub.model.GetFindingsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.GetFindingsIterable responses = client.getFindingsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.securityhub.paginators.GetFindingsIterable responses = client.getFindingsPaginator(request);
     *     for (software.amazon.awssdk.services.securityhub.model.GetFindingsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.GetFindingsIterable responses = client.getFindingsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getFindings(software.amazon.awssdk.services.securityhub.model.GetFindingsRequest)} operation.</b>
     * </p>
     *
     * @param getFindingsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.GetFindings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetFindings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetFindingsIterable getFindingsPaginator(GetFindingsRequest getFindingsRequest) throws InternalException,
            InvalidInputException, InvalidAccessException, LimitExceededException, AwsServiceException, SdkClientException,
            SecurityHubException {
        return new GetFindingsIterable(this, applyPaginatorUserAgent(getFindingsRequest));
    }

    /**
     * <p>
     * Lists the results of the Security Hub insight that the insight ARN specifies.
     * </p>
     *
     * @param getInsightResultsRequest
     * @return Result of the GetInsightResults operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.GetInsightResults
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetInsightResults" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetInsightResultsResponse getInsightResults(GetInsightResultsRequest getInsightResultsRequest)
            throws InternalException, InvalidInputException, InvalidAccessException, LimitExceededException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInsightResultsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetInsightResultsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetInsightResultsRequest, GetInsightResultsResponse>()
                .withOperationName("GetInsightResults").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getInsightResultsRequest)
                .withMarshaller(new GetInsightResultsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists and describes insights that insight ARNs specify.
     * </p>
     *
     * @param getInsightsRequest
     * @return Result of the GetInsights operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.GetInsights
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetInsights" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetInsightsResponse getInsights(GetInsightsRequest getInsightsRequest) throws InternalException,
            InvalidInputException, InvalidAccessException, LimitExceededException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInsightsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetInsightsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetInsightsRequest, GetInsightsResponse>()
                .withOperationName("GetInsights").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getInsightsRequest)
                .withMarshaller(new GetInsightsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists and describes insights that insight ARNs specify.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getInsights(software.amazon.awssdk.services.securityhub.model.GetInsightsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.GetInsightsIterable responses = client.getInsightsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.securityhub.paginators.GetInsightsIterable responses = client.getInsightsPaginator(request);
     *     for (software.amazon.awssdk.services.securityhub.model.GetInsightsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.GetInsightsIterable responses = client.getInsightsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getInsights(software.amazon.awssdk.services.securityhub.model.GetInsightsRequest)} operation.</b>
     * </p>
     *
     * @param getInsightsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.GetInsights
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetInsights" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetInsightsIterable getInsightsPaginator(GetInsightsRequest getInsightsRequest) throws InternalException,
            InvalidInputException, InvalidAccessException, LimitExceededException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        return new GetInsightsIterable(this, applyPaginatorUserAgent(getInsightsRequest));
    }

    /**
     * <p>
     * Returns the count of all Security Hub membership invitations that were sent to the current member account, not
     * including the currently accepted invitation.
     * </p>
     *
     * @param getInvitationsCountRequest
     * @return Result of the GetInvitationsCount operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.GetInvitationsCount
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetInvitationsCount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetInvitationsCountResponse getInvitationsCount(GetInvitationsCountRequest getInvitationsCountRequest)
            throws InternalException, InvalidInputException, InvalidAccessException, LimitExceededException, AwsServiceException,
            SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInvitationsCountResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetInvitationsCountResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetInvitationsCountRequest, GetInvitationsCountResponse>()
                .withOperationName("GetInvitationsCount").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getInvitationsCountRequest)
                .withMarshaller(new GetInvitationsCountRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides the details for the Security Hub master account to the current member account.
     * </p>
     *
     * @param getMasterAccountRequest
     * @return Result of the GetMasterAccount operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.GetMasterAccount
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetMasterAccount" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetMasterAccountResponse getMasterAccount(GetMasterAccountRequest getMasterAccountRequest) throws InternalException,
            InvalidInputException, InvalidAccessException, LimitExceededException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMasterAccountResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMasterAccountResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetMasterAccountRequest, GetMasterAccountResponse>()
                .withOperationName("GetMasterAccount").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getMasterAccountRequest)
                .withMarshaller(new GetMasterAccountRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the details on the Security Hub member accounts that the account IDs specify.
     * </p>
     *
     * @param getMembersRequest
     * @return Result of the GetMembers operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.GetMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/GetMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMembersResponse getMembers(GetMembersRequest getMembersRequest) throws InternalException, InvalidInputException,
            InvalidAccessException, LimitExceededException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMembersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetMembersRequest, GetMembersResponse>()
                .withOperationName("GetMembers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getMembersRequest)
                .withMarshaller(new GetMembersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Invites other AWS accounts to become member accounts for the Security Hub master account that the invitation is
     * sent from. Before you can use this action to invite a member, you must first create the member account in
     * Security Hub by using the <a>CreateMembers</a> action. When the account owner accepts the invitation to become a
     * member account and enables Security Hub, the master account can view the findings generated from member account.
     * </p>
     *
     * @param inviteMembersRequest
     * @return Result of the InviteMembers operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.InviteMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/InviteMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public InviteMembersResponse inviteMembers(InviteMembersRequest inviteMembersRequest) throws InternalException,
            InvalidInputException, InvalidAccessException, LimitExceededException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InviteMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                InviteMembersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<InviteMembersRequest, InviteMembersResponse>()
                .withOperationName("InviteMembers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(inviteMembersRequest)
                .withMarshaller(new InviteMembersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists all findings-generating solutions (products) whose findings you have subscribed to receive in Security Hub.
     * </p>
     *
     * @param listEnabledProductsForImportRequest
     * @return Result of the ListEnabledProductsForImport operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.ListEnabledProductsForImport
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListEnabledProductsForImport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEnabledProductsForImportResponse listEnabledProductsForImport(
            ListEnabledProductsForImportRequest listEnabledProductsForImportRequest) throws InternalException,
            LimitExceededException, InvalidAccessException, AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEnabledProductsForImportResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListEnabledProductsForImportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListEnabledProductsForImportRequest, ListEnabledProductsForImportResponse>()
                        .withOperationName("ListEnabledProductsForImport").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listEnabledProductsForImportRequest)
                        .withMarshaller(new ListEnabledProductsForImportRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists all findings-generating solutions (products) whose findings you have subscribed to receive in Security Hub.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEnabledProductsForImport(software.amazon.awssdk.services.securityhub.model.ListEnabledProductsForImportRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.ListEnabledProductsForImportIterable responses = client.listEnabledProductsForImportPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.securityhub.paginators.ListEnabledProductsForImportIterable responses = client
     *             .listEnabledProductsForImportPaginator(request);
     *     for (software.amazon.awssdk.services.securityhub.model.ListEnabledProductsForImportResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.securityhub.paginators.ListEnabledProductsForImportIterable responses = client.listEnabledProductsForImportPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnabledProductsForImport(software.amazon.awssdk.services.securityhub.model.ListEnabledProductsForImportRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEnabledProductsForImportRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalException
     *         Internal server error.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.ListEnabledProductsForImport
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListEnabledProductsForImport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEnabledProductsForImportIterable listEnabledProductsForImportPaginator(
            ListEnabledProductsForImportRequest listEnabledProductsForImportRequest) throws InternalException,
            LimitExceededException, InvalidAccessException, AwsServiceException, SdkClientException, SecurityHubException {
        return new ListEnabledProductsForImportIterable(this, applyPaginatorUserAgent(listEnabledProductsForImportRequest));
    }

    /**
     * <p>
     * Lists all Security Hub membership invitations that were sent to the current AWS account.
     * </p>
     *
     * @param listInvitationsRequest
     * @return Result of the ListInvitations operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.ListInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListInvitations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListInvitationsResponse listInvitations(ListInvitationsRequest listInvitationsRequest) throws InternalException,
            InvalidInputException, InvalidAccessException, LimitExceededException, AwsServiceException, SdkClientException,
            SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInvitationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListInvitationsRequest, ListInvitationsResponse>()
                .withOperationName("ListInvitations").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listInvitationsRequest)
                .withMarshaller(new ListInvitationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists details about all member accounts for the current Security Hub master account.
     * </p>
     *
     * @param listMembersRequest
     * @return Result of the ListMembers operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.ListMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListMembersResponse listMembers(ListMembersRequest listMembersRequest) throws InternalException,
            InvalidInputException, InvalidAccessException, LimitExceededException, AwsServiceException, SdkClientException,
            SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListMembersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListMembersRequest, ListMembersResponse>()
                .withOperationName("ListMembers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listMembersRequest)
                .withMarshaller(new ListMembersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of tags associated with a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalException, InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds one or more tags to a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalException,
            InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                .withOperationName("TagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes one or more tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalException,
            InvalidInputException, ResourceNotFoundException, AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                .withOperationName("UntagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the name and description of a custom action target in Security Hub.
     * </p>
     *
     * @param updateActionTargetRequest
     * @return Result of the UpdateActionTarget operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.UpdateActionTarget
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/UpdateActionTarget" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateActionTargetResponse updateActionTarget(UpdateActionTargetRequest updateActionTargetRequest)
            throws InternalException, InvalidInputException, ResourceNotFoundException, InvalidAccessException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateActionTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateActionTargetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateActionTargetRequest, UpdateActionTargetResponse>()
                .withOperationName("UpdateActionTarget").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateActionTargetRequest)
                .withMarshaller(new UpdateActionTargetRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the <code>Note</code> and <code>RecordState</code> of the Security Hub-aggregated findings that the
     * filter attributes specify. Any member account that can view the finding also sees the update to the finding.
     * </p>
     *
     * @param updateFindingsRequest
     * @return Result of the UpdateFindings operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.UpdateFindings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/UpdateFindings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateFindingsResponse updateFindings(UpdateFindingsRequest updateFindingsRequest) throws InternalException,
            InvalidInputException, LimitExceededException, InvalidAccessException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFindingsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateFindingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateFindingsRequest, UpdateFindingsResponse>()
                .withOperationName("UpdateFindings").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateFindingsRequest)
                .withMarshaller(new UpdateFindingsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the Security Hub insight that the insight ARN specifies.
     * </p>
     *
     * @param updateInsightRequest
     * @return Result of the UpdateInsight operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws LimitExceededException
     *         The request was rejected because it attempted to create resources beyond the current AWS account limits.
     *         The error code describes the limit exceeded.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.UpdateInsight
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/UpdateInsight" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateInsightResponse updateInsight(UpdateInsightRequest updateInsightRequest) throws InternalException,
            InvalidInputException, InvalidAccessException, LimitExceededException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateInsightResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateInsightResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateInsightRequest, UpdateInsightResponse>()
                .withOperationName("UpdateInsight").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateInsightRequest)
                .withMarshaller(new UpdateInsightRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Used to control whether an individual compliance standard control is enabled or disabled.
     * </p>
     *
     * @param updateStandardsControlRequest
     * @return Result of the UpdateStandardsControl operation returned by the service.
     * @throws InternalException
     *         Internal server error.
     * @throws InvalidInputException
     *         The request was rejected because you supplied an invalid or out-of-range value for an input parameter.
     * @throws InvalidAccessException
     *         AWS Security Hub isn't enabled for the account used to make this request.
     * @throws ResourceNotFoundException
     *         The request was rejected because we can't find the specified resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SecurityHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SecurityHubClient.UpdateStandardsControl
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/UpdateStandardsControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateStandardsControlResponse updateStandardsControl(UpdateStandardsControlRequest updateStandardsControlRequest)
            throws InternalException, InvalidInputException, InvalidAccessException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, SecurityHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateStandardsControlResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateStandardsControlResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateStandardsControlRequest, UpdateStandardsControlResponse>()
                .withOperationName("UpdateStandardsControl").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateStandardsControlRequest)
                .withMarshaller(new UpdateStandardsControlRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SecurityHubException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidAccessException")
                                .exceptionBuilderSupplier(InvalidAccessException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceConflictException")
                                .exceptionBuilderSupplier(ResourceConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalException")
                                .exceptionBuilderSupplier(InternalException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends SecurityHubRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
