/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines how the algorithm is used for a training job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TrainingSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<TrainingSpecification.Builder, TrainingSpecification> {
    private static final SdkField<String> TRAINING_IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TrainingImage").getter(getter(TrainingSpecification::trainingImage))
            .setter(setter(Builder::trainingImage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingImage").build()).build();

    private static final SdkField<String> TRAINING_IMAGE_DIGEST_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TrainingImageDigest").getter(getter(TrainingSpecification::trainingImageDigest))
            .setter(setter(Builder::trainingImageDigest))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingImageDigest").build())
            .build();

    private static final SdkField<List<HyperParameterSpecification>> SUPPORTED_HYPER_PARAMETERS_FIELD = SdkField
            .<List<HyperParameterSpecification>> builder(MarshallingType.LIST)
            .memberName("SupportedHyperParameters")
            .getter(getter(TrainingSpecification::supportedHyperParameters))
            .setter(setter(Builder::supportedHyperParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedHyperParameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HyperParameterSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(HyperParameterSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SUPPORTED_TRAINING_INSTANCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SupportedTrainingInstanceTypes")
            .getter(getter(TrainingSpecification::supportedTrainingInstanceTypesAsStrings))
            .setter(setter(Builder::supportedTrainingInstanceTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedTrainingInstanceTypes")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> SUPPORTS_DISTRIBUTED_TRAINING_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SupportsDistributedTraining")
            .getter(getter(TrainingSpecification::supportsDistributedTraining))
            .setter(setter(Builder::supportsDistributedTraining))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsDistributedTraining")
                    .build()).build();

    private static final SdkField<List<MetricDefinition>> METRIC_DEFINITIONS_FIELD = SdkField
            .<List<MetricDefinition>> builder(MarshallingType.LIST)
            .memberName("MetricDefinitions")
            .getter(getter(TrainingSpecification::metricDefinitions))
            .setter(setter(Builder::metricDefinitions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricDefinitions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MetricDefinition> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricDefinition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ChannelSpecification>> TRAINING_CHANNELS_FIELD = SdkField
            .<List<ChannelSpecification>> builder(MarshallingType.LIST)
            .memberName("TrainingChannels")
            .getter(getter(TrainingSpecification::trainingChannels))
            .setter(setter(Builder::trainingChannels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingChannels").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ChannelSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(ChannelSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<HyperParameterTuningJobObjective>> SUPPORTED_TUNING_JOB_OBJECTIVE_METRICS_FIELD = SdkField
            .<List<HyperParameterTuningJobObjective>> builder(MarshallingType.LIST)
            .memberName("SupportedTuningJobObjectiveMetrics")
            .getter(getter(TrainingSpecification::supportedTuningJobObjectiveMetrics))
            .setter(setter(Builder::supportedTuningJobObjectiveMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedTuningJobObjectiveMetrics")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HyperParameterTuningJobObjective> builder(MarshallingType.SDK_POJO)
                                            .constructor(HyperParameterTuningJobObjective::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRAINING_IMAGE_FIELD,
            TRAINING_IMAGE_DIGEST_FIELD, SUPPORTED_HYPER_PARAMETERS_FIELD, SUPPORTED_TRAINING_INSTANCE_TYPES_FIELD,
            SUPPORTS_DISTRIBUTED_TRAINING_FIELD, METRIC_DEFINITIONS_FIELD, TRAINING_CHANNELS_FIELD,
            SUPPORTED_TUNING_JOB_OBJECTIVE_METRICS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String trainingImage;

    private final String trainingImageDigest;

    private final List<HyperParameterSpecification> supportedHyperParameters;

    private final List<String> supportedTrainingInstanceTypes;

    private final Boolean supportsDistributedTraining;

    private final List<MetricDefinition> metricDefinitions;

    private final List<ChannelSpecification> trainingChannels;

    private final List<HyperParameterTuningJobObjective> supportedTuningJobObjectiveMetrics;

    private TrainingSpecification(BuilderImpl builder) {
        this.trainingImage = builder.trainingImage;
        this.trainingImageDigest = builder.trainingImageDigest;
        this.supportedHyperParameters = builder.supportedHyperParameters;
        this.supportedTrainingInstanceTypes = builder.supportedTrainingInstanceTypes;
        this.supportsDistributedTraining = builder.supportsDistributedTraining;
        this.metricDefinitions = builder.metricDefinitions;
        this.trainingChannels = builder.trainingChannels;
        this.supportedTuningJobObjectiveMetrics = builder.supportedTuningJobObjectiveMetrics;
    }

    /**
     * <p>
     * The Amazon ECR registry path of the Docker image that contains the training algorithm.
     * </p>
     * 
     * @return The Amazon ECR registry path of the Docker image that contains the training algorithm.
     */
    public final String trainingImage() {
        return trainingImage;
    }

    /**
     * <p>
     * An MD5 hash of the training algorithm that identifies the Docker image used for training.
     * </p>
     * 
     * @return An MD5 hash of the training algorithm that identifies the Docker image used for training.
     */
    public final String trainingImageDigest() {
        return trainingImageDigest;
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedHyperParameters property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSupportedHyperParameters() {
        return supportedHyperParameters != null && !(supportedHyperParameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the <code>HyperParameterSpecification</code> objects, that define the supported hyperparameters. This
     * is required if the algorithm supports automatic model tuning.&gt;
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedHyperParameters} method.
     * </p>
     * 
     * @return A list of the <code>HyperParameterSpecification</code> objects, that define the supported
     *         hyperparameters. This is required if the algorithm supports automatic model tuning.&gt;
     */
    public final List<HyperParameterSpecification> supportedHyperParameters() {
        return supportedHyperParameters;
    }

    /**
     * <p>
     * A list of the instance types that this algorithm can use for training.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedTrainingInstanceTypes} method.
     * </p>
     * 
     * @return A list of the instance types that this algorithm can use for training.
     */
    public final List<TrainingInstanceType> supportedTrainingInstanceTypes() {
        return TrainingInstanceTypesCopier.copyStringToEnum(supportedTrainingInstanceTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedTrainingInstanceTypes property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSupportedTrainingInstanceTypes() {
        return supportedTrainingInstanceTypes != null && !(supportedTrainingInstanceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the instance types that this algorithm can use for training.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedTrainingInstanceTypes} method.
     * </p>
     * 
     * @return A list of the instance types that this algorithm can use for training.
     */
    public final List<String> supportedTrainingInstanceTypesAsStrings() {
        return supportedTrainingInstanceTypes;
    }

    /**
     * <p>
     * Indicates whether the algorithm supports distributed training. If set to false, buyers can't request more than
     * one instance during training.
     * </p>
     * 
     * @return Indicates whether the algorithm supports distributed training. If set to false, buyers can't request more
     *         than one instance during training.
     */
    public final Boolean supportsDistributedTraining() {
        return supportsDistributedTraining;
    }

    /**
     * For responses, this returns true if the service returned a value for the MetricDefinitions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMetricDefinitions() {
        return metricDefinitions != null && !(metricDefinitions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>MetricDefinition</code> objects, which are used for parsing metrics generated by the algorithm.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetricDefinitions} method.
     * </p>
     * 
     * @return A list of <code>MetricDefinition</code> objects, which are used for parsing metrics generated by the
     *         algorithm.
     */
    public final List<MetricDefinition> metricDefinitions() {
        return metricDefinitions;
    }

    /**
     * For responses, this returns true if the service returned a value for the TrainingChannels property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTrainingChannels() {
        return trainingChannels != null && !(trainingChannels instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>ChannelSpecification</code> objects, which specify the input sources to be used by the algorithm.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTrainingChannels} method.
     * </p>
     * 
     * @return A list of <code>ChannelSpecification</code> objects, which specify the input sources to be used by the
     *         algorithm.
     */
    public final List<ChannelSpecification> trainingChannels() {
        return trainingChannels;
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedTuningJobObjectiveMetrics
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasSupportedTuningJobObjectiveMetrics() {
        return supportedTuningJobObjectiveMetrics != null
                && !(supportedTuningJobObjectiveMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the metrics that the algorithm emits that can be used as the objective metric in a hyperparameter
     * tuning job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedTuningJobObjectiveMetrics}
     * method.
     * </p>
     * 
     * @return A list of the metrics that the algorithm emits that can be used as the objective metric in a
     *         hyperparameter tuning job.
     */
    public final List<HyperParameterTuningJobObjective> supportedTuningJobObjectiveMetrics() {
        return supportedTuningJobObjectiveMetrics;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(trainingImage());
        hashCode = 31 * hashCode + Objects.hashCode(trainingImageDigest());
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportedHyperParameters() ? supportedHyperParameters() : null);
        hashCode = 31 * hashCode
                + Objects.hashCode(hasSupportedTrainingInstanceTypes() ? supportedTrainingInstanceTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(supportsDistributedTraining());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricDefinitions() ? metricDefinitions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTrainingChannels() ? trainingChannels() : null);
        hashCode = 31 * hashCode
                + Objects.hashCode(hasSupportedTuningJobObjectiveMetrics() ? supportedTuningJobObjectiveMetrics() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TrainingSpecification)) {
            return false;
        }
        TrainingSpecification other = (TrainingSpecification) obj;
        return Objects.equals(trainingImage(), other.trainingImage())
                && Objects.equals(trainingImageDigest(), other.trainingImageDigest())
                && hasSupportedHyperParameters() == other.hasSupportedHyperParameters()
                && Objects.equals(supportedHyperParameters(), other.supportedHyperParameters())
                && hasSupportedTrainingInstanceTypes() == other.hasSupportedTrainingInstanceTypes()
                && Objects.equals(supportedTrainingInstanceTypesAsStrings(), other.supportedTrainingInstanceTypesAsStrings())
                && Objects.equals(supportsDistributedTraining(), other.supportsDistributedTraining())
                && hasMetricDefinitions() == other.hasMetricDefinitions()
                && Objects.equals(metricDefinitions(), other.metricDefinitions())
                && hasTrainingChannels() == other.hasTrainingChannels()
                && Objects.equals(trainingChannels(), other.trainingChannels())
                && hasSupportedTuningJobObjectiveMetrics() == other.hasSupportedTuningJobObjectiveMetrics()
                && Objects.equals(supportedTuningJobObjectiveMetrics(), other.supportedTuningJobObjectiveMetrics());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("TrainingSpecification")
                .add("TrainingImage", trainingImage())
                .add("TrainingImageDigest", trainingImageDigest())
                .add("SupportedHyperParameters", hasSupportedHyperParameters() ? supportedHyperParameters() : null)
                .add("SupportedTrainingInstanceTypes",
                        hasSupportedTrainingInstanceTypes() ? supportedTrainingInstanceTypesAsStrings() : null)
                .add("SupportsDistributedTraining", supportsDistributedTraining())
                .add("MetricDefinitions", hasMetricDefinitions() ? metricDefinitions() : null)
                .add("TrainingChannels", hasTrainingChannels() ? trainingChannels() : null)
                .add("SupportedTuningJobObjectiveMetrics",
                        hasSupportedTuningJobObjectiveMetrics() ? supportedTuningJobObjectiveMetrics() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TrainingImage":
            return Optional.ofNullable(clazz.cast(trainingImage()));
        case "TrainingImageDigest":
            return Optional.ofNullable(clazz.cast(trainingImageDigest()));
        case "SupportedHyperParameters":
            return Optional.ofNullable(clazz.cast(supportedHyperParameters()));
        case "SupportedTrainingInstanceTypes":
            return Optional.ofNullable(clazz.cast(supportedTrainingInstanceTypesAsStrings()));
        case "SupportsDistributedTraining":
            return Optional.ofNullable(clazz.cast(supportsDistributedTraining()));
        case "MetricDefinitions":
            return Optional.ofNullable(clazz.cast(metricDefinitions()));
        case "TrainingChannels":
            return Optional.ofNullable(clazz.cast(trainingChannels()));
        case "SupportedTuningJobObjectiveMetrics":
            return Optional.ofNullable(clazz.cast(supportedTuningJobObjectiveMetrics()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TrainingSpecification, T> g) {
        return obj -> g.apply((TrainingSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TrainingSpecification> {
        /**
         * <p>
         * The Amazon ECR registry path of the Docker image that contains the training algorithm.
         * </p>
         * 
         * @param trainingImage
         *        The Amazon ECR registry path of the Docker image that contains the training algorithm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingImage(String trainingImage);

        /**
         * <p>
         * An MD5 hash of the training algorithm that identifies the Docker image used for training.
         * </p>
         * 
         * @param trainingImageDigest
         *        An MD5 hash of the training algorithm that identifies the Docker image used for training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingImageDigest(String trainingImageDigest);

        /**
         * <p>
         * A list of the <code>HyperParameterSpecification</code> objects, that define the supported hyperparameters.
         * This is required if the algorithm supports automatic model tuning.&gt;
         * </p>
         * 
         * @param supportedHyperParameters
         *        A list of the <code>HyperParameterSpecification</code> objects, that define the supported
         *        hyperparameters. This is required if the algorithm supports automatic model tuning.&gt;
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedHyperParameters(Collection<HyperParameterSpecification> supportedHyperParameters);

        /**
         * <p>
         * A list of the <code>HyperParameterSpecification</code> objects, that define the supported hyperparameters.
         * This is required if the algorithm supports automatic model tuning.&gt;
         * </p>
         * 
         * @param supportedHyperParameters
         *        A list of the <code>HyperParameterSpecification</code> objects, that define the supported
         *        hyperparameters. This is required if the algorithm supports automatic model tuning.&gt;
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedHyperParameters(HyperParameterSpecification... supportedHyperParameters);

        /**
         * <p>
         * A list of the <code>HyperParameterSpecification</code> objects, that define the supported hyperparameters.
         * This is required if the algorithm supports automatic model tuning.&gt;
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.HyperParameterSpecification.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.sagemaker.model.HyperParameterSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.HyperParameterSpecification.Builder#build()} is called
         * immediately and its result is passed to {@link #supportedHyperParameters(List<HyperParameterSpecification>)}.
         * 
         * @param supportedHyperParameters
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.HyperParameterSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #supportedHyperParameters(java.util.Collection<HyperParameterSpecification>)
         */
        Builder supportedHyperParameters(Consumer<HyperParameterSpecification.Builder>... supportedHyperParameters);

        /**
         * <p>
         * A list of the instance types that this algorithm can use for training.
         * </p>
         * 
         * @param supportedTrainingInstanceTypes
         *        A list of the instance types that this algorithm can use for training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedTrainingInstanceTypesWithStrings(Collection<String> supportedTrainingInstanceTypes);

        /**
         * <p>
         * A list of the instance types that this algorithm can use for training.
         * </p>
         * 
         * @param supportedTrainingInstanceTypes
         *        A list of the instance types that this algorithm can use for training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedTrainingInstanceTypesWithStrings(String... supportedTrainingInstanceTypes);

        /**
         * <p>
         * A list of the instance types that this algorithm can use for training.
         * </p>
         * 
         * @param supportedTrainingInstanceTypes
         *        A list of the instance types that this algorithm can use for training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedTrainingInstanceTypes(Collection<TrainingInstanceType> supportedTrainingInstanceTypes);

        /**
         * <p>
         * A list of the instance types that this algorithm can use for training.
         * </p>
         * 
         * @param supportedTrainingInstanceTypes
         *        A list of the instance types that this algorithm can use for training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedTrainingInstanceTypes(TrainingInstanceType... supportedTrainingInstanceTypes);

        /**
         * <p>
         * Indicates whether the algorithm supports distributed training. If set to false, buyers can't request more
         * than one instance during training.
         * </p>
         * 
         * @param supportsDistributedTraining
         *        Indicates whether the algorithm supports distributed training. If set to false, buyers can't request
         *        more than one instance during training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsDistributedTraining(Boolean supportsDistributedTraining);

        /**
         * <p>
         * A list of <code>MetricDefinition</code> objects, which are used for parsing metrics generated by the
         * algorithm.
         * </p>
         * 
         * @param metricDefinitions
         *        A list of <code>MetricDefinition</code> objects, which are used for parsing metrics generated by the
         *        algorithm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricDefinitions(Collection<MetricDefinition> metricDefinitions);

        /**
         * <p>
         * A list of <code>MetricDefinition</code> objects, which are used for parsing metrics generated by the
         * algorithm.
         * </p>
         * 
         * @param metricDefinitions
         *        A list of <code>MetricDefinition</code> objects, which are used for parsing metrics generated by the
         *        algorithm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricDefinitions(MetricDefinition... metricDefinitions);

        /**
         * <p>
         * A list of <code>MetricDefinition</code> objects, which are used for parsing metrics generated by the
         * algorithm.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.MetricDefinition.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.sagemaker.model.MetricDefinition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.MetricDefinition.Builder#build()} is called
         * immediately and its result is passed to {@link #metricDefinitions(List<MetricDefinition>)}.
         * 
         * @param metricDefinitions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.MetricDefinition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metricDefinitions(java.util.Collection<MetricDefinition>)
         */
        Builder metricDefinitions(Consumer<MetricDefinition.Builder>... metricDefinitions);

        /**
         * <p>
         * A list of <code>ChannelSpecification</code> objects, which specify the input sources to be used by the
         * algorithm.
         * </p>
         * 
         * @param trainingChannels
         *        A list of <code>ChannelSpecification</code> objects, which specify the input sources to be used by the
         *        algorithm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingChannels(Collection<ChannelSpecification> trainingChannels);

        /**
         * <p>
         * A list of <code>ChannelSpecification</code> objects, which specify the input sources to be used by the
         * algorithm.
         * </p>
         * 
         * @param trainingChannels
         *        A list of <code>ChannelSpecification</code> objects, which specify the input sources to be used by the
         *        algorithm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingChannels(ChannelSpecification... trainingChannels);

        /**
         * <p>
         * A list of <code>ChannelSpecification</code> objects, which specify the input sources to be used by the
         * algorithm.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.ChannelSpecification.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.sagemaker.model.ChannelSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.ChannelSpecification.Builder#build()} is called
         * immediately and its result is passed to {@link #trainingChannels(List<ChannelSpecification>)}.
         * 
         * @param trainingChannels
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.ChannelSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #trainingChannels(java.util.Collection<ChannelSpecification>)
         */
        Builder trainingChannels(Consumer<ChannelSpecification.Builder>... trainingChannels);

        /**
         * <p>
         * A list of the metrics that the algorithm emits that can be used as the objective metric in a hyperparameter
         * tuning job.
         * </p>
         * 
         * @param supportedTuningJobObjectiveMetrics
         *        A list of the metrics that the algorithm emits that can be used as the objective metric in a
         *        hyperparameter tuning job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedTuningJobObjectiveMetrics(Collection<HyperParameterTuningJobObjective> supportedTuningJobObjectiveMetrics);

        /**
         * <p>
         * A list of the metrics that the algorithm emits that can be used as the objective metric in a hyperparameter
         * tuning job.
         * </p>
         * 
         * @param supportedTuningJobObjectiveMetrics
         *        A list of the metrics that the algorithm emits that can be used as the objective metric in a
         *        hyperparameter tuning job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedTuningJobObjectiveMetrics(HyperParameterTuningJobObjective... supportedTuningJobObjectiveMetrics);

        /**
         * <p>
         * A list of the metrics that the algorithm emits that can be used as the objective metric in a hyperparameter
         * tuning job.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.sagemaker.model.HyperParameterTuningJobObjective.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.sagemaker.model.HyperParameterTuningJobObjective#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.sagemaker.model.HyperParameterTuningJobObjective.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #supportedTuningJobObjectiveMetrics(List<HyperParameterTuningJobObjective>)}.
         * 
         * @param supportedTuningJobObjectiveMetrics
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.sagemaker.model.HyperParameterTuningJobObjective.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #supportedTuningJobObjectiveMetrics(java.util.Collection<HyperParameterTuningJobObjective>)
         */
        Builder supportedTuningJobObjectiveMetrics(
                Consumer<HyperParameterTuningJobObjective.Builder>... supportedTuningJobObjectiveMetrics);
    }

    static final class BuilderImpl implements Builder {
        private String trainingImage;

        private String trainingImageDigest;

        private List<HyperParameterSpecification> supportedHyperParameters = DefaultSdkAutoConstructList.getInstance();

        private List<String> supportedTrainingInstanceTypes = DefaultSdkAutoConstructList.getInstance();

        private Boolean supportsDistributedTraining;

        private List<MetricDefinition> metricDefinitions = DefaultSdkAutoConstructList.getInstance();

        private List<ChannelSpecification> trainingChannels = DefaultSdkAutoConstructList.getInstance();

        private List<HyperParameterTuningJobObjective> supportedTuningJobObjectiveMetrics = DefaultSdkAutoConstructList
                .getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TrainingSpecification model) {
            trainingImage(model.trainingImage);
            trainingImageDigest(model.trainingImageDigest);
            supportedHyperParameters(model.supportedHyperParameters);
            supportedTrainingInstanceTypesWithStrings(model.supportedTrainingInstanceTypes);
            supportsDistributedTraining(model.supportsDistributedTraining);
            metricDefinitions(model.metricDefinitions);
            trainingChannels(model.trainingChannels);
            supportedTuningJobObjectiveMetrics(model.supportedTuningJobObjectiveMetrics);
        }

        public final String getTrainingImage() {
            return trainingImage;
        }

        public final void setTrainingImage(String trainingImage) {
            this.trainingImage = trainingImage;
        }

        @Override
        public final Builder trainingImage(String trainingImage) {
            this.trainingImage = trainingImage;
            return this;
        }

        public final String getTrainingImageDigest() {
            return trainingImageDigest;
        }

        public final void setTrainingImageDigest(String trainingImageDigest) {
            this.trainingImageDigest = trainingImageDigest;
        }

        @Override
        public final Builder trainingImageDigest(String trainingImageDigest) {
            this.trainingImageDigest = trainingImageDigest;
            return this;
        }

        public final List<HyperParameterSpecification.Builder> getSupportedHyperParameters() {
            List<HyperParameterSpecification.Builder> result = HyperParameterSpecificationsCopier
                    .copyToBuilder(this.supportedHyperParameters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSupportedHyperParameters(Collection<HyperParameterSpecification.BuilderImpl> supportedHyperParameters) {
            this.supportedHyperParameters = HyperParameterSpecificationsCopier.copyFromBuilder(supportedHyperParameters);
        }

        @Override
        public final Builder supportedHyperParameters(Collection<HyperParameterSpecification> supportedHyperParameters) {
            this.supportedHyperParameters = HyperParameterSpecificationsCopier.copy(supportedHyperParameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedHyperParameters(HyperParameterSpecification... supportedHyperParameters) {
            supportedHyperParameters(Arrays.asList(supportedHyperParameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedHyperParameters(Consumer<HyperParameterSpecification.Builder>... supportedHyperParameters) {
            supportedHyperParameters(Stream.of(supportedHyperParameters)
                    .map(c -> HyperParameterSpecification.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getSupportedTrainingInstanceTypes() {
            if (supportedTrainingInstanceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return supportedTrainingInstanceTypes;
        }

        public final void setSupportedTrainingInstanceTypes(Collection<String> supportedTrainingInstanceTypes) {
            this.supportedTrainingInstanceTypes = TrainingInstanceTypesCopier.copy(supportedTrainingInstanceTypes);
        }

        @Override
        public final Builder supportedTrainingInstanceTypesWithStrings(Collection<String> supportedTrainingInstanceTypes) {
            this.supportedTrainingInstanceTypes = TrainingInstanceTypesCopier.copy(supportedTrainingInstanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedTrainingInstanceTypesWithStrings(String... supportedTrainingInstanceTypes) {
            supportedTrainingInstanceTypesWithStrings(Arrays.asList(supportedTrainingInstanceTypes));
            return this;
        }

        @Override
        public final Builder supportedTrainingInstanceTypes(Collection<TrainingInstanceType> supportedTrainingInstanceTypes) {
            this.supportedTrainingInstanceTypes = TrainingInstanceTypesCopier.copyEnumToString(supportedTrainingInstanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedTrainingInstanceTypes(TrainingInstanceType... supportedTrainingInstanceTypes) {
            supportedTrainingInstanceTypes(Arrays.asList(supportedTrainingInstanceTypes));
            return this;
        }

        public final Boolean getSupportsDistributedTraining() {
            return supportsDistributedTraining;
        }

        public final void setSupportsDistributedTraining(Boolean supportsDistributedTraining) {
            this.supportsDistributedTraining = supportsDistributedTraining;
        }

        @Override
        public final Builder supportsDistributedTraining(Boolean supportsDistributedTraining) {
            this.supportsDistributedTraining = supportsDistributedTraining;
            return this;
        }

        public final List<MetricDefinition.Builder> getMetricDefinitions() {
            List<MetricDefinition.Builder> result = MetricDefinitionListCopier.copyToBuilder(this.metricDefinitions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMetricDefinitions(Collection<MetricDefinition.BuilderImpl> metricDefinitions) {
            this.metricDefinitions = MetricDefinitionListCopier.copyFromBuilder(metricDefinitions);
        }

        @Override
        public final Builder metricDefinitions(Collection<MetricDefinition> metricDefinitions) {
            this.metricDefinitions = MetricDefinitionListCopier.copy(metricDefinitions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricDefinitions(MetricDefinition... metricDefinitions) {
            metricDefinitions(Arrays.asList(metricDefinitions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricDefinitions(Consumer<MetricDefinition.Builder>... metricDefinitions) {
            metricDefinitions(Stream.of(metricDefinitions).map(c -> MetricDefinition.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<ChannelSpecification.Builder> getTrainingChannels() {
            List<ChannelSpecification.Builder> result = ChannelSpecificationsCopier.copyToBuilder(this.trainingChannels);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTrainingChannels(Collection<ChannelSpecification.BuilderImpl> trainingChannels) {
            this.trainingChannels = ChannelSpecificationsCopier.copyFromBuilder(trainingChannels);
        }

        @Override
        public final Builder trainingChannels(Collection<ChannelSpecification> trainingChannels) {
            this.trainingChannels = ChannelSpecificationsCopier.copy(trainingChannels);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder trainingChannels(ChannelSpecification... trainingChannels) {
            trainingChannels(Arrays.asList(trainingChannels));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder trainingChannels(Consumer<ChannelSpecification.Builder>... trainingChannels) {
            trainingChannels(Stream.of(trainingChannels).map(c -> ChannelSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<HyperParameterTuningJobObjective.Builder> getSupportedTuningJobObjectiveMetrics() {
            List<HyperParameterTuningJobObjective.Builder> result = HyperParameterTuningJobObjectivesCopier
                    .copyToBuilder(this.supportedTuningJobObjectiveMetrics);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSupportedTuningJobObjectiveMetrics(
                Collection<HyperParameterTuningJobObjective.BuilderImpl> supportedTuningJobObjectiveMetrics) {
            this.supportedTuningJobObjectiveMetrics = HyperParameterTuningJobObjectivesCopier
                    .copyFromBuilder(supportedTuningJobObjectiveMetrics);
        }

        @Override
        public final Builder supportedTuningJobObjectiveMetrics(
                Collection<HyperParameterTuningJobObjective> supportedTuningJobObjectiveMetrics) {
            this.supportedTuningJobObjectiveMetrics = HyperParameterTuningJobObjectivesCopier
                    .copy(supportedTuningJobObjectiveMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedTuningJobObjectiveMetrics(
                HyperParameterTuningJobObjective... supportedTuningJobObjectiveMetrics) {
            supportedTuningJobObjectiveMetrics(Arrays.asList(supportedTuningJobObjectiveMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedTuningJobObjectiveMetrics(
                Consumer<HyperParameterTuningJobObjective.Builder>... supportedTuningJobObjectiveMetrics) {
            supportedTuningJobObjectiveMetrics(Stream.of(supportedTuningJobObjectiveMetrics)
                    .map(c -> HyperParameterTuningJobObjective.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public TrainingSpecification build() {
            return new TrainingSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
