/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the configuration parameters for a job-completion report.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobReport implements SdkPojo, Serializable, ToCopyableBuilder<JobReport.Builder, JobReport> {
    private static final SdkField<String> BUCKET_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Bucket")
            .getter(getter(JobReport::bucket))
            .setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Bucket")
                    .unmarshallLocationName("Bucket").build()).build();

    private static final SdkField<String> FORMAT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Format")
            .getter(getter(JobReport::formatAsString))
            .setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Format")
                    .unmarshallLocationName("Format").build()).build();

    private static final SdkField<Boolean> ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Enabled")
            .getter(getter(JobReport::enabled))
            .setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled")
                    .unmarshallLocationName("Enabled").build()).build();

    private static final SdkField<String> PREFIX_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Prefix")
            .getter(getter(JobReport::prefix))
            .setter(setter(Builder::prefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Prefix")
                    .unmarshallLocationName("Prefix").build()).build();

    private static final SdkField<String> REPORT_SCOPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ReportScope")
            .getter(getter(JobReport::reportScopeAsString))
            .setter(setter(Builder::reportScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReportScope")
                    .unmarshallLocationName("ReportScope").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_FIELD, FORMAT_FIELD,
            ENABLED_FIELD, PREFIX_FIELD, REPORT_SCOPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String bucket;

    private final String format;

    private final Boolean enabled;

    private final String prefix;

    private final String reportScope;

    private JobReport(BuilderImpl builder) {
        this.bucket = builder.bucket;
        this.format = builder.format;
        this.enabled = builder.enabled;
        this.prefix = builder.prefix;
        this.reportScope = builder.reportScope;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the bucket where specified job-completion report will be stored.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the bucket where specified job-completion report will be stored.
     */
    public final String bucket() {
        return bucket;
    }

    /**
     * <p>
     * The format of the specified job-completion report.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link JobReportFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the specified job-completion report.
     * @see JobReportFormat
     */
    public final JobReportFormat format() {
        return JobReportFormat.fromValue(format);
    }

    /**
     * <p>
     * The format of the specified job-completion report.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link JobReportFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the specified job-completion report.
     * @see JobReportFormat
     */
    public final String formatAsString() {
        return format;
    }

    /**
     * <p>
     * Indicates whether the specified job will generate a job-completion report.
     * </p>
     * 
     * @return Indicates whether the specified job will generate a job-completion report.
     */
    public final Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * An optional prefix to describe where in the specified bucket the job-completion report will be stored. Amazon S3
     * stores the job-completion report at <code>&lt;prefix&gt;/job-&lt;job-id&gt;/report.json</code>.
     * </p>
     * 
     * @return An optional prefix to describe where in the specified bucket the job-completion report will be stored.
     *         Amazon S3 stores the job-completion report at <code>&lt;prefix&gt;/job-&lt;job-id&gt;/report.json</code>.
     */
    public final String prefix() {
        return prefix;
    }

    /**
     * <p>
     * Indicates whether the job-completion report will include details of all tasks or only failed tasks.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #reportScope} will
     * return {@link JobReportScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #reportScopeAsString}.
     * </p>
     * 
     * @return Indicates whether the job-completion report will include details of all tasks or only failed tasks.
     * @see JobReportScope
     */
    public final JobReportScope reportScope() {
        return JobReportScope.fromValue(reportScope);
    }

    /**
     * <p>
     * Indicates whether the job-completion report will include details of all tasks or only failed tasks.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #reportScope} will
     * return {@link JobReportScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #reportScopeAsString}.
     * </p>
     * 
     * @return Indicates whether the job-completion report will include details of all tasks or only failed tasks.
     * @see JobReportScope
     */
    public final String reportScopeAsString() {
        return reportScope;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(prefix());
        hashCode = 31 * hashCode + Objects.hashCode(reportScopeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobReport)) {
            return false;
        }
        JobReport other = (JobReport) obj;
        return Objects.equals(bucket(), other.bucket()) && Objects.equals(formatAsString(), other.formatAsString())
                && Objects.equals(enabled(), other.enabled()) && Objects.equals(prefix(), other.prefix())
                && Objects.equals(reportScopeAsString(), other.reportScopeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JobReport").add("Bucket", bucket()).add("Format", formatAsString()).add("Enabled", enabled())
                .add("Prefix", prefix()).add("ReportScope", reportScopeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "Format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "Prefix":
            return Optional.ofNullable(clazz.cast(prefix()));
        case "ReportScope":
            return Optional.ofNullable(clazz.cast(reportScopeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<JobReport, T> g) {
        return obj -> g.apply((JobReport) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobReport> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) for the bucket where specified job-completion report will be stored.
         * </p>
         * 
         * @param bucket
         *        The Amazon Resource Name (ARN) for the bucket where specified job-completion report will be stored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * The format of the specified job-completion report.
         * </p>
         * 
         * @param format
         *        The format of the specified job-completion report.
         * @see JobReportFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobReportFormat
         */
        Builder format(String format);

        /**
         * <p>
         * The format of the specified job-completion report.
         * </p>
         * 
         * @param format
         *        The format of the specified job-completion report.
         * @see JobReportFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobReportFormat
         */
        Builder format(JobReportFormat format);

        /**
         * <p>
         * Indicates whether the specified job will generate a job-completion report.
         * </p>
         * 
         * @param enabled
         *        Indicates whether the specified job will generate a job-completion report.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * An optional prefix to describe where in the specified bucket the job-completion report will be stored. Amazon
         * S3 stores the job-completion report at <code>&lt;prefix&gt;/job-&lt;job-id&gt;/report.json</code>.
         * </p>
         * 
         * @param prefix
         *        An optional prefix to describe where in the specified bucket the job-completion report will be stored.
         *        Amazon S3 stores the job-completion report at
         *        <code>&lt;prefix&gt;/job-&lt;job-id&gt;/report.json</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder prefix(String prefix);

        /**
         * <p>
         * Indicates whether the job-completion report will include details of all tasks or only failed tasks.
         * </p>
         * 
         * @param reportScope
         *        Indicates whether the job-completion report will include details of all tasks or only failed tasks.
         * @see JobReportScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobReportScope
         */
        Builder reportScope(String reportScope);

        /**
         * <p>
         * Indicates whether the job-completion report will include details of all tasks or only failed tasks.
         * </p>
         * 
         * @param reportScope
         *        Indicates whether the job-completion report will include details of all tasks or only failed tasks.
         * @see JobReportScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobReportScope
         */
        Builder reportScope(JobReportScope reportScope);
    }

    static final class BuilderImpl implements Builder {
        private String bucket;

        private String format;

        private Boolean enabled;

        private String prefix;

        private String reportScope;

        private BuilderImpl() {
        }

        private BuilderImpl(JobReport model) {
            bucket(model.bucket);
            format(model.format);
            enabled(model.enabled);
            prefix(model.prefix);
            reportScope(model.reportScope);
        }

        public final String getBucket() {
            return bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        public final String getFormat() {
            return format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(JobReportFormat format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        public final String getPrefix() {
            return prefix;
        }

        @Override
        public final Builder prefix(String prefix) {
            this.prefix = prefix;
            return this;
        }

        public final void setPrefix(String prefix) {
            this.prefix = prefix;
        }

        public final String getReportScope() {
            return reportScope;
        }

        @Override
        public final Builder reportScope(String reportScope) {
            this.reportScope = reportScope;
            return this;
        }

        @Override
        public final Builder reportScope(JobReportScope reportScope) {
            this.reportScope(reportScope == null ? null : reportScope.toString());
            return this;
        }

        public final void setReportScope(String reportScope) {
            this.reportScope = reportScope;
        }

        @Override
        public JobReport build() {
            return new JobReport(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
