/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rum.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure that defines a key and values that you can use to filter the results. The only performance events that
 * are returned are those that have values matching the ones that you specify in one of your <code>QueryFilter</code>
 * structures.
 * </p>
 * <p>
 * For example, you could specify <code>Browser</code> as the <code>Name</code> and specify <code>Chrome,Firefox</code>
 * as the <code>Values</code> to return events generated only from those browsers.
 * </p>
 * <p>
 * Specifying <code>Invert</code> as the <code>Name</code> works as a "not equal to" filter. For example, specify
 * <code>Invert</code> as the <code>Name</code> and specify <code>Chrome</code> as the value to return all events except
 * events from user sessions with the Chrome browser.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class QueryFilter implements SdkPojo, Serializable, ToCopyableBuilder<QueryFilter.Builder, QueryFilter> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(QueryFilter::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<String>> VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Values")
            .getter(getter(QueryFilter::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, VALUES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<String> values;

    private QueryFilter(BuilderImpl builder) {
        this.name = builder.name;
        this.values = builder.values;
    }

    /**
     * <p>
     * The name of a key to search for. The filter returns only the events that match the <code>Name</code> and
     * <code>Values</code> that you specify.
     * </p>
     * <p>
     * Valid values for <code>Name</code> are <code>Browser</code> | <code>Device</code> | <code>Country</code> |
     * <code>Page</code> | <code>OS</code> | <code>EventType</code> | <code>Invert</code>
     * </p>
     * 
     * @return The name of a key to search for. The filter returns only the events that match the <code>Name</code> and
     *         <code>Values</code> that you specify. </p>
     *         <p>
     *         Valid values for <code>Name</code> are <code>Browser</code> | <code>Device</code> | <code>Country</code>
     *         | <code>Page</code> | <code>OS</code> | <code>EventType</code> | <code>Invert</code>
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Values property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The values of the <code>Name</code> that are to be be included in the returned results.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValues} method.
     * </p>
     * 
     * @return The values of the <code>Name</code> that are to be be included in the returned results.
     */
    public final List<String> values() {
        return values;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasValues() ? values() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof QueryFilter)) {
            return false;
        }
        QueryFilter other = (QueryFilter) obj;
        return Objects.equals(name(), other.name()) && hasValues() == other.hasValues()
                && Objects.equals(values(), other.values());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("QueryFilter").add("Name", name()).add("Values", hasValues() ? values() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Values":
            return Optional.ofNullable(clazz.cast(values()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Values", VALUES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<QueryFilter, T> g) {
        return obj -> g.apply((QueryFilter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, QueryFilter> {
        /**
         * <p>
         * The name of a key to search for. The filter returns only the events that match the <code>Name</code> and
         * <code>Values</code> that you specify.
         * </p>
         * <p>
         * Valid values for <code>Name</code> are <code>Browser</code> | <code>Device</code> | <code>Country</code> |
         * <code>Page</code> | <code>OS</code> | <code>EventType</code> | <code>Invert</code>
         * </p>
         * 
         * @param name
         *        The name of a key to search for. The filter returns only the events that match the <code>Name</code>
         *        and <code>Values</code> that you specify. </p>
         *        <p>
         *        Valid values for <code>Name</code> are <code>Browser</code> | <code>Device</code> |
         *        <code>Country</code> | <code>Page</code> | <code>OS</code> | <code>EventType</code> |
         *        <code>Invert</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The values of the <code>Name</code> that are to be be included in the returned results.
         * </p>
         * 
         * @param values
         *        The values of the <code>Name</code> that are to be be included in the returned results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * The values of the <code>Name</code> that are to be be included in the returned results.
         * </p>
         * 
         * @param values
         *        The values of the <code>Name</code> that are to be be included in the returned results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<String> values = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(QueryFilter model) {
            name(model.name);
            values(model.values);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Collection<String> getValues() {
            if (values instanceof SdkAutoConstructList) {
                return null;
            }
            return values;
        }

        public final void setValues(Collection<String> values) {
            this.values = QueryFilterValueListCopier.copy(values);
        }

        @Override
        public final Builder values(Collection<String> values) {
            this.values = QueryFilterValueListCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        @Override
        public QueryFilter build() {
            return new QueryFilter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
