/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rtbfabric;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.rtbfabric.internal.RtbFabricServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.rtbfabric.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.rtbfabric.model.AcceptLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.AcceptLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.AccessDeniedException;
import software.amazon.awssdk.services.rtbfabric.model.ConflictException;
import software.amazon.awssdk.services.rtbfabric.model.CreateInboundExternalLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.CreateInboundExternalLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.CreateLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.CreateLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.CreateOutboundExternalLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.CreateOutboundExternalLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.CreateRequesterGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.CreateRequesterGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.CreateResponderGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.CreateResponderGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.DeleteInboundExternalLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.DeleteInboundExternalLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.DeleteLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.DeleteLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.DeleteOutboundExternalLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.DeleteOutboundExternalLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.DeleteRequesterGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.DeleteRequesterGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.DeleteResponderGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.DeleteResponderGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.GetInboundExternalLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.GetInboundExternalLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.GetLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.GetLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.GetOutboundExternalLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.GetOutboundExternalLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.GetRequesterGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.GetRequesterGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.GetResponderGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.GetResponderGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.InternalServerException;
import software.amazon.awssdk.services.rtbfabric.model.ListLinksRequest;
import software.amazon.awssdk.services.rtbfabric.model.ListLinksResponse;
import software.amazon.awssdk.services.rtbfabric.model.ListRequesterGatewaysRequest;
import software.amazon.awssdk.services.rtbfabric.model.ListRequesterGatewaysResponse;
import software.amazon.awssdk.services.rtbfabric.model.ListResponderGatewaysRequest;
import software.amazon.awssdk.services.rtbfabric.model.ListResponderGatewaysResponse;
import software.amazon.awssdk.services.rtbfabric.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.rtbfabric.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.rtbfabric.model.RejectLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.RejectLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.ResourceNotFoundException;
import software.amazon.awssdk.services.rtbfabric.model.RtbFabricException;
import software.amazon.awssdk.services.rtbfabric.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.rtbfabric.model.TagResourceRequest;
import software.amazon.awssdk.services.rtbfabric.model.TagResourceResponse;
import software.amazon.awssdk.services.rtbfabric.model.ThrottlingException;
import software.amazon.awssdk.services.rtbfabric.model.UntagResourceRequest;
import software.amazon.awssdk.services.rtbfabric.model.UntagResourceResponse;
import software.amazon.awssdk.services.rtbfabric.model.UpdateLinkModuleFlowRequest;
import software.amazon.awssdk.services.rtbfabric.model.UpdateLinkModuleFlowResponse;
import software.amazon.awssdk.services.rtbfabric.model.UpdateLinkRequest;
import software.amazon.awssdk.services.rtbfabric.model.UpdateLinkResponse;
import software.amazon.awssdk.services.rtbfabric.model.UpdateRequesterGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.UpdateRequesterGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.UpdateResponderGatewayRequest;
import software.amazon.awssdk.services.rtbfabric.model.UpdateResponderGatewayResponse;
import software.amazon.awssdk.services.rtbfabric.model.ValidationException;
import software.amazon.awssdk.services.rtbfabric.transform.AcceptLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.CreateInboundExternalLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.CreateLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.CreateOutboundExternalLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.CreateRequesterGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.CreateResponderGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.DeleteInboundExternalLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.DeleteLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.DeleteOutboundExternalLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.DeleteRequesterGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.DeleteResponderGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.GetInboundExternalLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.GetLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.GetOutboundExternalLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.GetRequesterGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.GetResponderGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.ListLinksRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.ListRequesterGatewaysRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.ListResponderGatewaysRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.RejectLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.UpdateLinkModuleFlowRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.UpdateLinkRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.UpdateRequesterGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.transform.UpdateResponderGatewayRequestMarshaller;
import software.amazon.awssdk.services.rtbfabric.waiters.RtbFabricAsyncWaiter;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link RtbFabricAsyncClient}.
 *
 * @see RtbFabricAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRtbFabricAsyncClient implements RtbFabricAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRtbFabricAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ScheduledExecutorService executorService;

    protected DefaultRtbFabricAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "RTBFabric" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executorService = clientConfiguration.option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
    }

    /**
     * <p>
     * Accepts a link request between gateways.
     * </p>
     * <p>
     * When a requester gateway requests to link with a responder gateway, the responder can use this operation to
     * accept the link request and establish the connection.
     * </p>
     *
     * @param acceptLinkRequest
     * @return A Java Future containing the result of the AcceptLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>ConflictException The request could not be completed because of a conflict in the current state of
     *         the resource.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.AcceptLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/AcceptLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptLinkResponse> acceptLink(AcceptLinkRequest acceptLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(acceptLinkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AcceptLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<AcceptLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptLinkRequest, AcceptLinkResponse>().withOperationName("AcceptLink")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AcceptLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(acceptLinkRequest));
            CompletableFuture<AcceptLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an inbound external link.
     * </p>
     *
     * @param createInboundExternalLinkRequest
     * @return A Java Future containing the result of the CreateInboundExternalLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request could not be completed because you exceeded a service
     *         quota.</li>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>ConflictException The request could not be completed because of a conflict in the current state of
     *         the resource.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.CreateInboundExternalLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/CreateInboundExternalLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateInboundExternalLinkResponse> createInboundExternalLink(
            CreateInboundExternalLinkRequest createInboundExternalLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createInboundExternalLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createInboundExternalLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateInboundExternalLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateInboundExternalLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateInboundExternalLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateInboundExternalLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateInboundExternalLinkRequest, CreateInboundExternalLinkResponse>()
                            .withOperationName("CreateInboundExternalLink").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateInboundExternalLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createInboundExternalLinkRequest));
            CompletableFuture<CreateInboundExternalLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new link between gateways.
     * </p>
     * <p>
     * Establishes a connection that allows gateways to communicate and exchange bid requests and responses.
     * </p>
     *
     * @param createLinkRequest
     * @return A Java Future containing the result of the CreateLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request could not be completed because you exceeded a service
     *         quota.</li>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>ConflictException The request could not be completed because of a conflict in the current state of
     *         the resource.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.CreateLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/CreateLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLinkResponse> createLink(CreateLinkRequest createLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createLinkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLinkRequest, CreateLinkResponse>().withOperationName("CreateLink")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createLinkRequest));
            CompletableFuture<CreateLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an outbound external link.
     * </p>
     *
     * @param createOutboundExternalLinkRequest
     * @return A Java Future containing the result of the CreateOutboundExternalLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request could not be completed because you exceeded a service
     *         quota.</li>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.CreateOutboundExternalLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/CreateOutboundExternalLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateOutboundExternalLinkResponse> createOutboundExternalLink(
            CreateOutboundExternalLinkRequest createOutboundExternalLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createOutboundExternalLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createOutboundExternalLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateOutboundExternalLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateOutboundExternalLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateOutboundExternalLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateOutboundExternalLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateOutboundExternalLinkRequest, CreateOutboundExternalLinkResponse>()
                            .withOperationName("CreateOutboundExternalLink").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateOutboundExternalLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createOutboundExternalLinkRequest));
            CompletableFuture<CreateOutboundExternalLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a requester gateway.
     * </p>
     *
     * @param createRequesterGatewayRequest
     * @return A Java Future containing the result of the CreateRequesterGateway operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request could not be completed because you exceeded a service
     *         quota.</li>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.CreateRequesterGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/CreateRequesterGateway"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRequesterGatewayResponse> createRequesterGateway(
            CreateRequesterGatewayRequest createRequesterGatewayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRequesterGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRequesterGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRequesterGateway");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRequesterGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRequesterGatewayResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateRequesterGatewayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRequesterGatewayRequest, CreateRequesterGatewayResponse>()
                            .withOperationName("CreateRequesterGateway").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateRequesterGatewayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRequesterGatewayRequest));
            CompletableFuture<CreateRequesterGatewayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a responder gateway.
     * </p>
     * <important>
     * <p>
     * A domain name or managed endpoint is required.
     * </p>
     * </important>
     *
     * @param createResponderGatewayRequest
     * @return A Java Future containing the result of the CreateResponderGateway operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request could not be completed because you exceeded a service
     *         quota.</li>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.CreateResponderGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/CreateResponderGateway"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateResponderGatewayResponse> createResponderGateway(
            CreateResponderGatewayRequest createResponderGatewayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createResponderGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResponderGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResponderGateway");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateResponderGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateResponderGatewayResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateResponderGatewayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateResponderGatewayRequest, CreateResponderGatewayResponse>()
                            .withOperationName("CreateResponderGateway").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateResponderGatewayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createResponderGatewayRequest));
            CompletableFuture<CreateResponderGatewayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an inbound external link.
     * </p>
     *
     * @param deleteInboundExternalLinkRequest
     * @return A Java Future containing the result of the DeleteInboundExternalLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>ConflictException The request could not be completed because of a conflict in the current state of
     *         the resource.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.DeleteInboundExternalLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/DeleteInboundExternalLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteInboundExternalLinkResponse> deleteInboundExternalLink(
            DeleteInboundExternalLinkRequest deleteInboundExternalLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteInboundExternalLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInboundExternalLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInboundExternalLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteInboundExternalLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteInboundExternalLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteInboundExternalLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteInboundExternalLinkRequest, DeleteInboundExternalLinkResponse>()
                            .withOperationName("DeleteInboundExternalLink").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteInboundExternalLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteInboundExternalLinkRequest));
            CompletableFuture<DeleteInboundExternalLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a link between gateways.
     * </p>
     * <p>
     * Permanently removes the connection between gateways. This action cannot be undone.
     * </p>
     *
     * @param deleteLinkRequest
     * @return A Java Future containing the result of the DeleteLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>ConflictException The request could not be completed because of a conflict in the current state of
     *         the resource.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.DeleteLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/DeleteLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLinkResponse> deleteLink(DeleteLinkRequest deleteLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLinkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLinkRequest, DeleteLinkResponse>().withOperationName("DeleteLink")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteLinkRequest));
            CompletableFuture<DeleteLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an outbound external link.
     * </p>
     *
     * @param deleteOutboundExternalLinkRequest
     * @return A Java Future containing the result of the DeleteOutboundExternalLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>ConflictException The request could not be completed because of a conflict in the current state of
     *         the resource.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.DeleteOutboundExternalLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/DeleteOutboundExternalLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteOutboundExternalLinkResponse> deleteOutboundExternalLink(
            DeleteOutboundExternalLinkRequest deleteOutboundExternalLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteOutboundExternalLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteOutboundExternalLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteOutboundExternalLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteOutboundExternalLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteOutboundExternalLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteOutboundExternalLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteOutboundExternalLinkRequest, DeleteOutboundExternalLinkResponse>()
                            .withOperationName("DeleteOutboundExternalLink").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteOutboundExternalLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteOutboundExternalLinkRequest));
            CompletableFuture<DeleteOutboundExternalLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a requester gateway.
     * </p>
     *
     * @param deleteRequesterGatewayRequest
     * @return A Java Future containing the result of the DeleteRequesterGateway operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.DeleteRequesterGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/DeleteRequesterGateway"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRequesterGatewayResponse> deleteRequesterGateway(
            DeleteRequesterGatewayRequest deleteRequesterGatewayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRequesterGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRequesterGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRequesterGateway");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRequesterGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRequesterGatewayResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteRequesterGatewayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRequesterGatewayRequest, DeleteRequesterGatewayResponse>()
                            .withOperationName("DeleteRequesterGateway").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRequesterGatewayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRequesterGatewayRequest));
            CompletableFuture<DeleteRequesterGatewayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a responder gateway.
     * </p>
     *
     * @param deleteResponderGatewayRequest
     * @return A Java Future containing the result of the DeleteResponderGateway operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.DeleteResponderGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/DeleteResponderGateway"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResponderGatewayResponse> deleteResponderGateway(
            DeleteResponderGatewayRequest deleteResponderGatewayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResponderGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResponderGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResponderGateway");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResponderGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResponderGatewayResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteResponderGatewayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResponderGatewayRequest, DeleteResponderGatewayResponse>()
                            .withOperationName("DeleteResponderGateway").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteResponderGatewayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteResponderGatewayRequest));
            CompletableFuture<DeleteResponderGatewayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about an inbound external link.
     * </p>
     *
     * @param getInboundExternalLinkRequest
     * @return A Java Future containing the result of the GetInboundExternalLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.GetInboundExternalLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/GetInboundExternalLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetInboundExternalLinkResponse> getInboundExternalLink(
            GetInboundExternalLinkRequest getInboundExternalLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInboundExternalLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInboundExternalLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInboundExternalLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInboundExternalLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetInboundExternalLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetInboundExternalLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInboundExternalLinkRequest, GetInboundExternalLinkResponse>()
                            .withOperationName("GetInboundExternalLink").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetInboundExternalLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getInboundExternalLinkRequest));
            CompletableFuture<GetInboundExternalLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a link between gateways.
     * </p>
     * <p>
     * Returns detailed information about the link configuration, status, and associated gateways.
     * </p>
     *
     * @param getLinkRequest
     * @return A Java Future containing the result of the GetLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>ConflictException The request could not be completed because of a conflict in the current state of
     *         the resource.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.GetLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/GetLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLinkResponse> getLink(GetLinkRequest getLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLinkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLinkRequest, GetLinkResponse>().withOperationName("GetLink")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLinkRequest));
            CompletableFuture<GetLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about an outbound external link.
     * </p>
     *
     * @param getOutboundExternalLinkRequest
     * @return A Java Future containing the result of the GetOutboundExternalLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.GetOutboundExternalLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/GetOutboundExternalLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetOutboundExternalLinkResponse> getOutboundExternalLink(
            GetOutboundExternalLinkRequest getOutboundExternalLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getOutboundExternalLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOutboundExternalLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOutboundExternalLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOutboundExternalLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetOutboundExternalLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetOutboundExternalLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOutboundExternalLinkRequest, GetOutboundExternalLinkResponse>()
                            .withOperationName("GetOutboundExternalLink").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetOutboundExternalLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getOutboundExternalLinkRequest));
            CompletableFuture<GetOutboundExternalLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a requester gateway.
     * </p>
     *
     * @param getRequesterGatewayRequest
     * @return A Java Future containing the result of the GetRequesterGateway operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.GetRequesterGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/GetRequesterGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRequesterGatewayResponse> getRequesterGateway(
            GetRequesterGatewayRequest getRequesterGatewayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRequesterGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRequesterGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRequesterGateway");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRequesterGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRequesterGatewayResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetRequesterGatewayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRequesterGatewayRequest, GetRequesterGatewayResponse>()
                            .withOperationName("GetRequesterGateway").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRequesterGatewayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRequesterGatewayRequest));
            CompletableFuture<GetRequesterGatewayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a responder gateway.
     * </p>
     *
     * @param getResponderGatewayRequest
     * @return A Java Future containing the result of the GetResponderGateway operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.GetResponderGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/GetResponderGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResponderGatewayResponse> getResponderGateway(
            GetResponderGatewayRequest getResponderGatewayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResponderGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResponderGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResponderGateway");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResponderGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResponderGatewayResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetResponderGatewayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResponderGatewayRequest, GetResponderGatewayResponse>()
                            .withOperationName("GetResponderGateway").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResponderGatewayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResponderGatewayRequest));
            CompletableFuture<GetResponderGatewayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists links associated with gateways.
     * </p>
     * <p>
     * Returns a list of all links for the specified gateways, including their status and configuration details.
     * </p>
     *
     * @param listLinksRequest
     * @return A Java Future containing the result of the ListLinks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.ListLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/ListLinks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListLinksResponse> listLinks(ListLinksRequest listLinksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLinksRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLinksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLinks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLinksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListLinksResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListLinksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLinksRequest, ListLinksResponse>().withOperationName("ListLinks")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListLinksRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLinksRequest));
            CompletableFuture<ListLinksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists requester gateways.
     * </p>
     *
     * @param listRequesterGatewaysRequest
     * @return A Java Future containing the result of the ListRequesterGateways operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.ListRequesterGateways
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/ListRequesterGateways"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRequesterGatewaysResponse> listRequesterGateways(
            ListRequesterGatewaysRequest listRequesterGatewaysRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRequesterGatewaysRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRequesterGatewaysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRequesterGateways");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRequesterGatewaysResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRequesterGatewaysResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListRequesterGatewaysResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRequesterGatewaysRequest, ListRequesterGatewaysResponse>()
                            .withOperationName("ListRequesterGateways").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRequesterGatewaysRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRequesterGatewaysRequest));
            CompletableFuture<ListRequesterGatewaysResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists reponder gateways.
     * </p>
     *
     * @param listResponderGatewaysRequest
     * @return A Java Future containing the result of the ListResponderGateways operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.ListResponderGateways
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/ListResponderGateways"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResponderGatewaysResponse> listResponderGateways(
            ListResponderGatewaysRequest listResponderGatewaysRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResponderGatewaysRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResponderGatewaysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResponderGateways");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResponderGatewaysResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResponderGatewaysResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListResponderGatewaysResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResponderGatewaysRequest, ListResponderGatewaysResponse>()
                            .withOperationName("ListResponderGateways").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListResponderGatewaysRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listResponderGatewaysRequest));
            CompletableFuture<ListResponderGatewaysResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Rejects a link request between gateways.
     * </p>
     * <p>
     * When a requester gateway requests to link with a responder gateway, the responder can use this operation to
     * decline the link request.
     * </p>
     *
     * @param rejectLinkRequest
     * @return A Java Future containing the result of the RejectLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>ConflictException The request could not be completed because of a conflict in the current state of
     *         the resource.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.RejectLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/RejectLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RejectLinkResponse> rejectLink(RejectLinkRequest rejectLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rejectLinkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RejectLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RejectLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<RejectLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RejectLinkRequest, RejectLinkResponse>().withOperationName("RejectLink")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RejectLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(rejectLinkRequest));
            CompletableFuture<RejectLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag or tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration of a link between gateways.
     * </p>
     * <p>
     * Allows you to modify settings and parameters for an existing link.
     * </p>
     *
     * @param updateLinkRequest
     * @return A Java Future containing the result of the UpdateLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>ConflictException The request could not be completed because of a conflict in the current state of
     *         the resource.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.UpdateLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/UpdateLink" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLinkResponse> updateLink(UpdateLinkRequest updateLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateLinkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLinkRequest, UpdateLinkResponse>().withOperationName("UpdateLink")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateLinkRequest));
            CompletableFuture<UpdateLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a link module flow.
     * </p>
     *
     * @param updateLinkModuleFlowRequest
     * @return A Java Future containing the result of the UpdateLinkModuleFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The request could not be completed because you exceeded a service
     *         quota.</li>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>ConflictException The request could not be completed because of a conflict in the current state of
     *         the resource.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.UpdateLinkModuleFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/UpdateLinkModuleFlow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLinkModuleFlowResponse> updateLinkModuleFlow(
            UpdateLinkModuleFlowRequest updateLinkModuleFlowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateLinkModuleFlowRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLinkModuleFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLinkModuleFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLinkModuleFlowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLinkModuleFlowResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateLinkModuleFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLinkModuleFlowRequest, UpdateLinkModuleFlowResponse>()
                            .withOperationName("UpdateLinkModuleFlow").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateLinkModuleFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateLinkModuleFlowRequest));
            CompletableFuture<UpdateLinkModuleFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a requester gateway.
     * </p>
     *
     * @param updateRequesterGatewayRequest
     * @return A Java Future containing the result of the UpdateRequesterGateway operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>ConflictException The request could not be completed because of a conflict in the current state of
     *         the resource.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.UpdateRequesterGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/UpdateRequesterGateway"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRequesterGatewayResponse> updateRequesterGateway(
            UpdateRequesterGatewayRequest updateRequesterGatewayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRequesterGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRequesterGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRequesterGateway");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRequesterGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRequesterGatewayResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateRequesterGatewayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRequesterGatewayRequest, UpdateRequesterGatewayResponse>()
                            .withOperationName("UpdateRequesterGateway").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateRequesterGatewayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateRequesterGatewayRequest));
            CompletableFuture<UpdateRequesterGatewayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a responder gateway.
     * </p>
     *
     * @param updateResponderGatewayRequest
     * @return A Java Future containing the result of the UpdateResponderGateway operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request could not be completed because the resource does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have sufficient access to
     *         perform this action.</li>
     *         <li>ConflictException The request could not be completed because of a conflict in the current state of
     *         the resource.</li>
     *         <li>InternalServerException The request could not be completed because of an internal server error. Try
     *         your call again.</li>
     *         <li>ValidationException The request could not be completed because it fails satisfy the constraints
     *         specified by the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RtbFabricException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample RtbFabricAsyncClient.UpdateResponderGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rtbfabric-2023-05-15/UpdateResponderGateway"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateResponderGatewayResponse> updateResponderGateway(
            UpdateResponderGatewayRequest updateResponderGatewayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateResponderGatewayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResponderGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RTBFabric");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResponderGateway");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateResponderGatewayResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateResponderGatewayResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateResponderGatewayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateResponderGatewayRequest, UpdateResponderGatewayResponse>()
                            .withOperationName("UpdateResponderGateway").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateResponderGatewayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateResponderGatewayRequest));
            CompletableFuture<UpdateResponderGatewayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public RtbFabricAsyncWaiter waiter() {
        return RtbFabricAsyncWaiter.builder().client(this).scheduledExecutorService(executorService).build();
    }

    @Override
    public final RtbFabricServiceClientConfiguration serviceClientConfiguration() {
        return new RtbFabricServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(RtbFabricException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        RtbFabricServiceClientConfigurationBuilder serviceConfigBuilder = new RtbFabricServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
