/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53resolver.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * In an <a href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_UpdateResolverRule.html">
 * UpdateResolverRule</a> request, information about the changes that you want to make.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResolverRuleConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<ResolverRuleConfig.Builder, ResolverRuleConfig> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ResolverRuleConfig::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<TargetAddress>> TARGET_IPS_FIELD = SdkField
            .<List<TargetAddress>> builder(MarshallingType.LIST)
            .memberName("TargetIps")
            .getter(getter(ResolverRuleConfig::targetIps))
            .setter(setter(Builder::targetIps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetIps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TargetAddress> builder(MarshallingType.SDK_POJO)
                                            .constructor(TargetAddress::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RESOLVER_ENDPOINT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResolverEndpointId").getter(getter(ResolverRuleConfig::resolverEndpointId))
            .setter(setter(Builder::resolverEndpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResolverEndpointId").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, TARGET_IPS_FIELD,
            RESOLVER_ENDPOINT_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<TargetAddress> targetIps;

    private final String resolverEndpointId;

    private ResolverRuleConfig(BuilderImpl builder) {
        this.name = builder.name;
        this.targetIps = builder.targetIps;
        this.resolverEndpointId = builder.resolverEndpointId;
    }

    /**
     * <p>
     * The new name for the Resolver rule. The name that you specify appears in the Resolver dashboard in the Route 53
     * console.
     * </p>
     * 
     * @return The new name for the Resolver rule. The name that you specify appears in the Resolver dashboard in the
     *         Route 53 console.
     */
    public final String name() {
        return name;
    }

    /**
     * Returns true if the TargetIps property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTargetIps() {
        return targetIps != null && !(targetIps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * For DNS queries that originate in your VPC, the new IP addresses that you want to route outbound DNS queries to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTargetIps()} to see if a value was sent in this field.
     * </p>
     * 
     * @return For DNS queries that originate in your VPC, the new IP addresses that you want to route outbound DNS
     *         queries to.
     */
    public final List<TargetAddress> targetIps() {
        return targetIps;
    }

    /**
     * <p>
     * The ID of the new outbound Resolver endpoint that you want to use to route DNS queries to the IP addresses that
     * you specify in <code>TargetIps</code>.
     * </p>
     * 
     * @return The ID of the new outbound Resolver endpoint that you want to use to route DNS queries to the IP
     *         addresses that you specify in <code>TargetIps</code>.
     */
    public final String resolverEndpointId() {
        return resolverEndpointId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargetIps() ? targetIps() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resolverEndpointId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResolverRuleConfig)) {
            return false;
        }
        ResolverRuleConfig other = (ResolverRuleConfig) obj;
        return Objects.equals(name(), other.name()) && hasTargetIps() == other.hasTargetIps()
                && Objects.equals(targetIps(), other.targetIps())
                && Objects.equals(resolverEndpointId(), other.resolverEndpointId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResolverRuleConfig").add("Name", name()).add("TargetIps", hasTargetIps() ? targetIps() : null)
                .add("ResolverEndpointId", resolverEndpointId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "TargetIps":
            return Optional.ofNullable(clazz.cast(targetIps()));
        case "ResolverEndpointId":
            return Optional.ofNullable(clazz.cast(resolverEndpointId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResolverRuleConfig, T> g) {
        return obj -> g.apply((ResolverRuleConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResolverRuleConfig> {
        /**
         * <p>
         * The new name for the Resolver rule. The name that you specify appears in the Resolver dashboard in the Route
         * 53 console.
         * </p>
         * 
         * @param name
         *        The new name for the Resolver rule. The name that you specify appears in the Resolver dashboard in the
         *        Route 53 console.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * For DNS queries that originate in your VPC, the new IP addresses that you want to route outbound DNS queries
         * to.
         * </p>
         * 
         * @param targetIps
         *        For DNS queries that originate in your VPC, the new IP addresses that you want to route outbound DNS
         *        queries to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetIps(Collection<TargetAddress> targetIps);

        /**
         * <p>
         * For DNS queries that originate in your VPC, the new IP addresses that you want to route outbound DNS queries
         * to.
         * </p>
         * 
         * @param targetIps
         *        For DNS queries that originate in your VPC, the new IP addresses that you want to route outbound DNS
         *        queries to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetIps(TargetAddress... targetIps);

        /**
         * <p>
         * For DNS queries that originate in your VPC, the new IP addresses that you want to route outbound DNS queries
         * to.
         * </p>
         * This is a convenience that creates an instance of the {@link List<TargetAddress>.Builder} avoiding the need
         * to create one manually via {@link List<TargetAddress>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TargetAddress>.Builder#build()} is called immediately and
         * its result is passed to {@link #targetIps(List<TargetAddress>)}.
         * 
         * @param targetIps
         *        a consumer that will call methods on {@link List<TargetAddress>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetIps(List<TargetAddress>)
         */
        Builder targetIps(Consumer<TargetAddress.Builder>... targetIps);

        /**
         * <p>
         * The ID of the new outbound Resolver endpoint that you want to use to route DNS queries to the IP addresses
         * that you specify in <code>TargetIps</code>.
         * </p>
         * 
         * @param resolverEndpointId
         *        The ID of the new outbound Resolver endpoint that you want to use to route DNS queries to the IP
         *        addresses that you specify in <code>TargetIps</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resolverEndpointId(String resolverEndpointId);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<TargetAddress> targetIps = DefaultSdkAutoConstructList.getInstance();

        private String resolverEndpointId;

        private BuilderImpl() {
        }

        private BuilderImpl(ResolverRuleConfig model) {
            name(model.name);
            targetIps(model.targetIps);
            resolverEndpointId(model.resolverEndpointId);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Collection<TargetAddress.Builder> getTargetIps() {
            if (targetIps instanceof SdkAutoConstructList) {
                return null;
            }
            return targetIps != null ? targetIps.stream().map(TargetAddress::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder targetIps(Collection<TargetAddress> targetIps) {
            this.targetIps = TargetListCopier.copy(targetIps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetIps(TargetAddress... targetIps) {
            targetIps(Arrays.asList(targetIps));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetIps(Consumer<TargetAddress.Builder>... targetIps) {
            targetIps(Stream.of(targetIps).map(c -> TargetAddress.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setTargetIps(Collection<TargetAddress.BuilderImpl> targetIps) {
            this.targetIps = TargetListCopier.copyFromBuilder(targetIps);
        }

        public final String getResolverEndpointId() {
            return resolverEndpointId;
        }

        @Override
        public final Builder resolverEndpointId(String resolverEndpointId) {
            this.resolverEndpointId = resolverEndpointId;
            return this;
        }

        public final void setResolverEndpointId(String resolverEndpointId) {
            this.resolverEndpointId = resolverEndpointId;
        }

        @Override
        public ResolverRuleConfig build() {
            return new ResolverRuleConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
