/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53resolver;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.route53resolver.model.AccessDeniedException;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverEndpointIpAddressRequest;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverEndpointIpAddressResponse;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverQueryLogConfigRequest;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverQueryLogConfigResponse;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverQueryLogConfigRequest;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverQueryLogConfigResponse;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverQueryLogConfigRequest;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverQueryLogConfigResponse;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverEndpointIpAddressRequest;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverEndpointIpAddressResponse;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverQueryLogConfigRequest;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverQueryLogConfigResponse;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverDnssecConfigRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverDnssecConfigResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverQueryLogConfigAssociationRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverQueryLogConfigAssociationResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverQueryLogConfigPolicyRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverQueryLogConfigPolicyResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverQueryLogConfigRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverQueryLogConfigResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleAssociationRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleAssociationResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRulePolicyRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRulePolicyResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.InternalServiceErrorException;
import software.amazon.awssdk.services.route53resolver.model.InvalidNextTokenException;
import software.amazon.awssdk.services.route53resolver.model.InvalidParameterException;
import software.amazon.awssdk.services.route53resolver.model.InvalidPolicyDocumentException;
import software.amazon.awssdk.services.route53resolver.model.InvalidRequestException;
import software.amazon.awssdk.services.route53resolver.model.InvalidTagException;
import software.amazon.awssdk.services.route53resolver.model.LimitExceededException;
import software.amazon.awssdk.services.route53resolver.model.ListResolverDnssecConfigsRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverDnssecConfigsResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigAssociationsRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigAssociationsResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigsRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigsResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRulesRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRulesResponse;
import software.amazon.awssdk.services.route53resolver.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.route53resolver.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.route53resolver.model.PutResolverQueryLogConfigPolicyRequest;
import software.amazon.awssdk.services.route53resolver.model.PutResolverQueryLogConfigPolicyResponse;
import software.amazon.awssdk.services.route53resolver.model.PutResolverRulePolicyRequest;
import software.amazon.awssdk.services.route53resolver.model.PutResolverRulePolicyResponse;
import software.amazon.awssdk.services.route53resolver.model.ResourceExistsException;
import software.amazon.awssdk.services.route53resolver.model.ResourceInUseException;
import software.amazon.awssdk.services.route53resolver.model.ResourceNotFoundException;
import software.amazon.awssdk.services.route53resolver.model.ResourceUnavailableException;
import software.amazon.awssdk.services.route53resolver.model.Route53ResolverException;
import software.amazon.awssdk.services.route53resolver.model.Route53ResolverRequest;
import software.amazon.awssdk.services.route53resolver.model.TagResourceRequest;
import software.amazon.awssdk.services.route53resolver.model.TagResourceResponse;
import software.amazon.awssdk.services.route53resolver.model.ThrottlingException;
import software.amazon.awssdk.services.route53resolver.model.UnknownResourceException;
import software.amazon.awssdk.services.route53resolver.model.UntagResourceRequest;
import software.amazon.awssdk.services.route53resolver.model.UntagResourceResponse;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverDnssecConfigRequest;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverDnssecConfigResponse;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverDnssecConfigsIterable;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointIpAddressesIterable;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointsIterable;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverQueryLogConfigAssociationsIterable;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverQueryLogConfigsIterable;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverRuleAssociationsIterable;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverRulesIterable;
import software.amazon.awssdk.services.route53resolver.paginators.ListTagsForResourceIterable;
import software.amazon.awssdk.services.route53resolver.transform.AssociateResolverEndpointIpAddressRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.AssociateResolverQueryLogConfigRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.AssociateResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.CreateResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.CreateResolverQueryLogConfigRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.CreateResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DeleteResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DeleteResolverQueryLogConfigRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DeleteResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DisassociateResolverEndpointIpAddressRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DisassociateResolverQueryLogConfigRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DisassociateResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverDnssecConfigRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverQueryLogConfigAssociationRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverQueryLogConfigPolicyRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverQueryLogConfigRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverRuleAssociationRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverRulePolicyRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverDnssecConfigsRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverEndpointIpAddressesRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverEndpointsRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverQueryLogConfigAssociationsRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverQueryLogConfigsRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverRuleAssociationsRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverRulesRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.PutResolverQueryLogConfigPolicyRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.PutResolverRulePolicyRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.UpdateResolverDnssecConfigRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.UpdateResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.UpdateResolverRuleRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link Route53ResolverClient}.
 *
 * @see Route53ResolverClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRoute53ResolverClient implements Route53ResolverClient {
    private static final Logger log = Logger.loggerFor(DefaultRoute53ResolverClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRoute53ResolverClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds IP addresses to an inbound or an outbound Resolver endpoint. If you want to add more than one IP address,
     * submit one <code>AssociateResolverEndpointIpAddress</code> request for each IP address.
     * </p>
     * <p>
     * To remove an IP address from an endpoint, see <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverEndpointIpAddress.html"
     * >DisassociateResolverEndpointIpAddress</a>.
     * </p>
     *
     * @param associateResolverEndpointIpAddressRequest
     * @return Result of the AssociateResolverEndpointIpAddress operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws ResourceExistsException
     *         The resource that you tried to create already exists.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.AssociateResolverEndpointIpAddress
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/AssociateResolverEndpointIpAddress"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateResolverEndpointIpAddressResponse associateResolverEndpointIpAddress(
            AssociateResolverEndpointIpAddressRequest associateResolverEndpointIpAddressRequest)
            throws ResourceNotFoundException, InvalidParameterException, InvalidRequestException, ResourceExistsException,
            InternalServiceErrorException, LimitExceededException, ThrottlingException, AwsServiceException, SdkClientException,
            Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateResolverEndpointIpAddressResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateResolverEndpointIpAddressResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateResolverEndpointIpAddressRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResolverEndpointIpAddress");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateResolverEndpointIpAddressRequest, AssociateResolverEndpointIpAddressResponse>()
                            .withOperationName("AssociateResolverEndpointIpAddress").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(associateResolverEndpointIpAddressRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateResolverEndpointIpAddressRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates an Amazon VPC with a specified query logging configuration. Route 53 Resolver logs DNS queries that
     * originate in all of the Amazon VPCs that are associated with a specified query logging configuration. To
     * associate more than one VPC with a configuration, submit one <code>AssociateResolverQueryLogConfig</code> request
     * for each VPC.
     * </p>
     * <note>
     * <p>
     * The VPCs that you associate with a query logging configuration must be in the same Region as the configuration.
     * </p>
     * </note>
     * <p>
     * To remove a VPC from a query logging configuration, see <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverQueryLogConfig.html"
     * >DisassociateResolverQueryLogConfig</a>.
     * </p>
     *
     * @param associateResolverQueryLogConfigRequest
     * @return Result of the AssociateResolverQueryLogConfig operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws ResourceExistsException
     *         The resource that you tried to create already exists.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.AssociateResolverQueryLogConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/AssociateResolverQueryLogConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateResolverQueryLogConfigResponse associateResolverQueryLogConfig(
            AssociateResolverQueryLogConfigRequest associateResolverQueryLogConfigRequest) throws InvalidParameterException,
            ResourceNotFoundException, InvalidRequestException, ResourceExistsException, LimitExceededException,
            InternalServiceErrorException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateResolverQueryLogConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateResolverQueryLogConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateResolverQueryLogConfigRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResolverQueryLogConfig");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateResolverQueryLogConfigRequest, AssociateResolverQueryLogConfigResponse>()
                            .withOperationName("AssociateResolverQueryLogConfig").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(associateResolverQueryLogConfigRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateResolverQueryLogConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates a Resolver rule with a VPC. When you associate a rule with a VPC, Resolver forwards all DNS queries
     * for the domain name that is specified in the rule and that originate in the VPC. The queries are forwarded to the
     * IP addresses for the DNS resolvers that are specified in the rule. For more information about rules, see <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverRule.html"
     * >CreateResolverRule</a>.
     * </p>
     *
     * @param associateResolverRuleRequest
     * @return Result of the AssociateResolverRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ResourceUnavailableException
     *         The specified resource isn't available.
     * @throws ResourceExistsException
     *         The resource that you tried to create already exists.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.AssociateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/AssociateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateResolverRuleResponse associateResolverRule(AssociateResolverRuleRequest associateResolverRuleRequest)
            throws ResourceNotFoundException, InvalidRequestException, LimitExceededException, InvalidParameterException,
            ResourceUnavailableException, ResourceExistsException, InternalServiceErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateResolverRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateResolverRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResolverRule");

            return clientHandler.execute(new ClientExecutionParams<AssociateResolverRuleRequest, AssociateResolverRuleResponse>()
                    .withOperationName("AssociateResolverRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(associateResolverRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateResolverRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a Resolver endpoint. There are two types of Resolver endpoints, inbound and outbound:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An <i>inbound Resolver endpoint</i> forwards DNS queries to the DNS service for a VPC from your network.
     * </p>
     * </li>
     * <li>
     * <p>
     * An <i>outbound Resolver endpoint</i> forwards DNS queries from the DNS service for a VPC to your network.
     * </p>
     * </li>
     * </ul>
     *
     * @param createResolverEndpointRequest
     * @return Result of the CreateResolverEndpoint operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws ResourceExistsException
     *         The resource that you tried to create already exists.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.CreateResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/CreateResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateResolverEndpointResponse createResolverEndpoint(CreateResolverEndpointRequest createResolverEndpointRequest)
            throws InvalidParameterException, ResourceNotFoundException, InvalidRequestException, ResourceExistsException,
            LimitExceededException, InternalServiceErrorException, ThrottlingException, AwsServiceException, SdkClientException,
            Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateResolverEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResolverEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResolverEndpoint");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateResolverEndpointRequest, CreateResolverEndpointResponse>()
                            .withOperationName("CreateResolverEndpoint").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createResolverEndpointRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateResolverEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a Resolver query logging configuration, which defines where you want Resolver to save DNS query logs that
     * originate in your VPCs. Resolver can log queries only for VPCs that are in the same Region as the query logging
     * configuration.
     * </p>
     * <p>
     * To specify which VPCs you want to log queries for, you use <code>AssociateResolverQueryLogConfig</code>. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_AssociateResolverQueryLogConfig.html"
     * >AssociateResolverQueryLogConfig</a>.
     * </p>
     * <p>
     * You can optionally use AWS Resource Access Manager (AWS RAM) to share a query logging configuration with other
     * AWS accounts. The other accounts can then associate VPCs with the configuration. The query logs that Resolver
     * creates for a configuration include all DNS queries that originate in all VPCs that are associated with the
     * configuration.
     * </p>
     *
     * @param createResolverQueryLogConfigRequest
     * @return Result of the CreateResolverQueryLogConfig operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws ResourceExistsException
     *         The resource that you tried to create already exists.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.CreateResolverQueryLogConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/CreateResolverQueryLogConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateResolverQueryLogConfigResponse createResolverQueryLogConfig(
            CreateResolverQueryLogConfigRequest createResolverQueryLogConfigRequest) throws InvalidParameterException,
            ResourceNotFoundException, InvalidRequestException, ResourceExistsException, LimitExceededException,
            InternalServiceErrorException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateResolverQueryLogConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateResolverQueryLogConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResolverQueryLogConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResolverQueryLogConfig");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateResolverQueryLogConfigRequest, CreateResolverQueryLogConfigResponse>()
                            .withOperationName("CreateResolverQueryLogConfig").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createResolverQueryLogConfigRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateResolverQueryLogConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * For DNS queries that originate in your VPCs, specifies which Resolver endpoint the queries pass through, one
     * domain name that you want to forward to your network, and the IP addresses of the DNS resolvers in your network.
     * </p>
     *
     * @param createResolverRuleRequest
     * @return Result of the CreateResolverRule operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourceExistsException
     *         The resource that you tried to create already exists.
     * @throws ResourceUnavailableException
     *         The specified resource isn't available.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.CreateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/CreateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateResolverRuleResponse createResolverRule(CreateResolverRuleRequest createResolverRuleRequest)
            throws InvalidParameterException, InvalidRequestException, LimitExceededException, ResourceNotFoundException,
            ResourceExistsException, ResourceUnavailableException, InternalServiceErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateResolverRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResolverRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResolverRule");

            return clientHandler.execute(new ClientExecutionParams<CreateResolverRuleRequest, CreateResolverRuleResponse>()
                    .withOperationName("CreateResolverRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createResolverRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateResolverRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a Resolver endpoint. The effect of deleting a Resolver endpoint depends on whether it's an inbound or an
     * outbound Resolver endpoint:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Inbound</b>: DNS queries from your network are no longer routed to the DNS service for the specified VPC.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Outbound</b>: DNS queries from a VPC are no longer routed to your network.
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteResolverEndpointRequest
     * @return Result of the DeleteResolverEndpoint operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.DeleteResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DeleteResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteResolverEndpointResponse deleteResolverEndpoint(DeleteResolverEndpointRequest deleteResolverEndpointRequest)
            throws ResourceNotFoundException, InvalidParameterException, InvalidRequestException, InternalServiceErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResolverEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResolverEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResolverEndpoint");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteResolverEndpointRequest, DeleteResolverEndpointResponse>()
                            .withOperationName("DeleteResolverEndpoint").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteResolverEndpointRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteResolverEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a query logging configuration. When you delete a configuration, Resolver stops logging DNS queries for
     * all of the Amazon VPCs that are associated with the configuration. This also applies if the query logging
     * configuration is shared with other AWS accounts, and the other accounts have associated VPCs with the shared
     * configuration.
     * </p>
     * <p>
     * Before you can delete a query logging configuration, you must first disassociate all VPCs from the configuration.
     * See <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverQueryLogConfig.html"
     * >DisassociateResolverQueryLogConfig</a>.
     * </p>
     * <p>
     * If you used Resource Access Manager (RAM) to share a query logging configuration with other accounts, you must
     * stop sharing the configuration before you can delete a configuration. The accounts that you shared the
     * configuration with can first disassociate VPCs that they associated with the configuration, but that's not
     * necessary. If you stop sharing the configuration, those VPCs are automatically disassociated from the
     * configuration.
     * </p>
     *
     * @param deleteResolverQueryLogConfigRequest
     * @return Result of the DeleteResolverQueryLogConfig operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.DeleteResolverQueryLogConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DeleteResolverQueryLogConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteResolverQueryLogConfigResponse deleteResolverQueryLogConfig(
            DeleteResolverQueryLogConfigRequest deleteResolverQueryLogConfigRequest) throws ResourceNotFoundException,
            InvalidParameterException, InvalidRequestException, InternalServiceErrorException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResolverQueryLogConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResolverQueryLogConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResolverQueryLogConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResolverQueryLogConfig");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteResolverQueryLogConfigRequest, DeleteResolverQueryLogConfigResponse>()
                            .withOperationName("DeleteResolverQueryLogConfig").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteResolverQueryLogConfigRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteResolverQueryLogConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a Resolver rule. Before you can delete a Resolver rule, you must disassociate it from all the VPCs that
     * you associated the Resolver rule with. For more information, see <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverRule.html"
     * >DisassociateResolverRule</a>.
     * </p>
     *
     * @param deleteResolverRuleRequest
     * @return Result of the DeleteResolverRule operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourceInUseException
     *         The resource that you tried to update or delete is currently in use.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.DeleteResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DeleteResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteResolverRuleResponse deleteResolverRule(DeleteResolverRuleRequest deleteResolverRuleRequest)
            throws InvalidParameterException, ResourceNotFoundException, ResourceInUseException, InternalServiceErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResolverRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResolverRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResolverRule");

            return clientHandler.execute(new ClientExecutionParams<DeleteResolverRuleRequest, DeleteResolverRuleResponse>()
                    .withOperationName("DeleteResolverRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteResolverRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteResolverRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes IP addresses from an inbound or an outbound Resolver endpoint. If you want to remove more than one IP
     * address, submit one <code>DisassociateResolverEndpointIpAddress</code> request for each IP address.
     * </p>
     * <p>
     * To add an IP address to an endpoint, see <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_AssociateResolverEndpointIpAddress.html"
     * >AssociateResolverEndpointIpAddress</a>.
     * </p>
     *
     * @param disassociateResolverEndpointIpAddressRequest
     * @return Result of the DisassociateResolverEndpointIpAddress operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws ResourceExistsException
     *         The resource that you tried to create already exists.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.DisassociateResolverEndpointIpAddress
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DisassociateResolverEndpointIpAddress"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateResolverEndpointIpAddressResponse disassociateResolverEndpointIpAddress(
            DisassociateResolverEndpointIpAddressRequest disassociateResolverEndpointIpAddressRequest)
            throws ResourceNotFoundException, InvalidParameterException, InvalidRequestException, ResourceExistsException,
            InternalServiceErrorException, ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateResolverEndpointIpAddressResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DisassociateResolverEndpointIpAddressResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateResolverEndpointIpAddressRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResolverEndpointIpAddress");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateResolverEndpointIpAddressRequest, DisassociateResolverEndpointIpAddressResponse>()
                            .withOperationName("DisassociateResolverEndpointIpAddress").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(disassociateResolverEndpointIpAddressRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateResolverEndpointIpAddressRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates a VPC from a query logging configuration.
     * </p>
     * <note>
     * <p>
     * Before you can delete a query logging configuration, you must first disassociate all VPCs from the configuration.
     * If you used Resource Access Manager (RAM) to share a query logging configuration with other accounts, VPCs can be
     * disassociated from the configuration in the following ways:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The accounts that you shared the configuration with can disassociate VPCs from the configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can stop sharing the configuration.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param disassociateResolverQueryLogConfigRequest
     * @return Result of the DisassociateResolverQueryLogConfig operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.DisassociateResolverQueryLogConfig
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DisassociateResolverQueryLogConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateResolverQueryLogConfigResponse disassociateResolverQueryLogConfig(
            DisassociateResolverQueryLogConfigRequest disassociateResolverQueryLogConfigRequest)
            throws ResourceNotFoundException, InvalidParameterException, InvalidRequestException, InternalServiceErrorException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateResolverQueryLogConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateResolverQueryLogConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateResolverQueryLogConfigRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResolverQueryLogConfig");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateResolverQueryLogConfigRequest, DisassociateResolverQueryLogConfigResponse>()
                            .withOperationName("DisassociateResolverQueryLogConfig").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateResolverQueryLogConfigRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateResolverQueryLogConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the association between a specified Resolver rule and a specified VPC.
     * </p>
     * <important>
     * <p>
     * If you disassociate a Resolver rule from a VPC, Resolver stops forwarding DNS queries for the domain name that
     * you specified in the Resolver rule.
     * </p>
     * </important>
     *
     * @param disassociateResolverRuleRequest
     * @return Result of the DisassociateResolverRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.DisassociateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DisassociateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateResolverRuleResponse disassociateResolverRule(
            DisassociateResolverRuleRequest disassociateResolverRuleRequest) throws ResourceNotFoundException,
            InvalidParameterException, InternalServiceErrorException, ThrottlingException, AwsServiceException,
            SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateResolverRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateResolverRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResolverRule");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateResolverRuleRequest, DisassociateResolverRuleResponse>()
                            .withOperationName("DisassociateResolverRule").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateResolverRuleRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateResolverRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets DNSSEC validation information for a specified resource.
     * </p>
     *
     * @param getResolverDnssecConfigRequest
     * @return Result of the GetResolverDnssecConfig operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.GetResolverDnssecConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverDnssecConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResolverDnssecConfigResponse getResolverDnssecConfig(GetResolverDnssecConfigRequest getResolverDnssecConfigRequest)
            throws InvalidParameterException, ResourceNotFoundException, InvalidRequestException, InternalServiceErrorException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResolverDnssecConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResolverDnssecConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResolverDnssecConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverDnssecConfig");

            return clientHandler
                    .execute(new ClientExecutionParams<GetResolverDnssecConfigRequest, GetResolverDnssecConfigResponse>()
                            .withOperationName("GetResolverDnssecConfig").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getResolverDnssecConfigRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetResolverDnssecConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a specified Resolver endpoint, such as whether it's an inbound or an outbound Resolver
     * endpoint, and the current status of the endpoint.
     * </p>
     *
     * @param getResolverEndpointRequest
     * @return Result of the GetResolverEndpoint operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.GetResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResolverEndpointResponse getResolverEndpoint(GetResolverEndpointRequest getResolverEndpointRequest)
            throws ResourceNotFoundException, InvalidParameterException, InternalServiceErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResolverEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResolverEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverEndpoint");

            return clientHandler.execute(new ClientExecutionParams<GetResolverEndpointRequest, GetResolverEndpointResponse>()
                    .withOperationName("GetResolverEndpoint").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getResolverEndpointRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResolverEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a specified Resolver query logging configuration, such as the number of VPCs that the
     * configuration is logging queries for and the location that logs are sent to.
     * </p>
     *
     * @param getResolverQueryLogConfigRequest
     * @return Result of the GetResolverQueryLogConfig operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.GetResolverQueryLogConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverQueryLogConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResolverQueryLogConfigResponse getResolverQueryLogConfig(
            GetResolverQueryLogConfigRequest getResolverQueryLogConfigRequest) throws ResourceNotFoundException,
            InvalidRequestException, InvalidParameterException, InternalServiceErrorException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResolverQueryLogConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResolverQueryLogConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResolverQueryLogConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverQueryLogConfig");

            return clientHandler
                    .execute(new ClientExecutionParams<GetResolverQueryLogConfigRequest, GetResolverQueryLogConfigResponse>()
                            .withOperationName("GetResolverQueryLogConfig").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getResolverQueryLogConfigRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetResolverQueryLogConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a specified association between a Resolver query logging configuration and an Amazon VPC.
     * When you associate a VPC with a query logging configuration, Resolver logs DNS queries that originate in that
     * VPC.
     * </p>
     *
     * @param getResolverQueryLogConfigAssociationRequest
     * @return Result of the GetResolverQueryLogConfigAssociation operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.GetResolverQueryLogConfigAssociation
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverQueryLogConfigAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResolverQueryLogConfigAssociationResponse getResolverQueryLogConfigAssociation(
            GetResolverQueryLogConfigAssociationRequest getResolverQueryLogConfigAssociationRequest)
            throws ResourceNotFoundException, InvalidRequestException, InvalidParameterException, InternalServiceErrorException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResolverQueryLogConfigAssociationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetResolverQueryLogConfigAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getResolverQueryLogConfigAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverQueryLogConfigAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<GetResolverQueryLogConfigAssociationRequest, GetResolverQueryLogConfigAssociationResponse>()
                            .withOperationName("GetResolverQueryLogConfigAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(getResolverQueryLogConfigAssociationRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetResolverQueryLogConfigAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a query logging policy. A query logging policy specifies the Resolver query logging
     * operations and resources that you want to allow another AWS account to be able to use.
     * </p>
     *
     * @param getResolverQueryLogConfigPolicyRequest
     * @return Result of the GetResolverQueryLogConfigPolicy operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws UnknownResourceException
     *         The specified resource doesn't exist.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.GetResolverQueryLogConfigPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverQueryLogConfigPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResolverQueryLogConfigPolicyResponse getResolverQueryLogConfigPolicy(
            GetResolverQueryLogConfigPolicyRequest getResolverQueryLogConfigPolicyRequest) throws InvalidParameterException,
            InvalidRequestException, UnknownResourceException, InternalServiceErrorException, AccessDeniedException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResolverQueryLogConfigPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResolverQueryLogConfigPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getResolverQueryLogConfigPolicyRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverQueryLogConfigPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<GetResolverQueryLogConfigPolicyRequest, GetResolverQueryLogConfigPolicyResponse>()
                            .withOperationName("GetResolverQueryLogConfigPolicy").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getResolverQueryLogConfigPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetResolverQueryLogConfigPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a specified Resolver rule, such as the domain name that the rule forwards DNS queries for
     * and the ID of the outbound Resolver endpoint that the rule is associated with.
     * </p>
     *
     * @param getResolverRuleRequest
     * @return Result of the GetResolverRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.GetResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResolverRuleResponse getResolverRule(GetResolverRuleRequest getResolverRuleRequest)
            throws ResourceNotFoundException, InvalidParameterException, InternalServiceErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResolverRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResolverRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverRule");

            return clientHandler.execute(new ClientExecutionParams<GetResolverRuleRequest, GetResolverRuleResponse>()
                    .withOperationName("GetResolverRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getResolverRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResolverRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about an association between a specified Resolver rule and a VPC. You associate a Resolver rule
     * and a VPC using <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_AssociateResolverRule.html"
     * >AssociateResolverRule</a>.
     * </p>
     *
     * @param getResolverRuleAssociationRequest
     * @return Result of the GetResolverRuleAssociation operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.GetResolverRuleAssociation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverRuleAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResolverRuleAssociationResponse getResolverRuleAssociation(
            GetResolverRuleAssociationRequest getResolverRuleAssociationRequest) throws ResourceNotFoundException,
            InvalidParameterException, InternalServiceErrorException, ThrottlingException, AwsServiceException,
            SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResolverRuleAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResolverRuleAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResolverRuleAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverRuleAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<GetResolverRuleAssociationRequest, GetResolverRuleAssociationResponse>()
                            .withOperationName("GetResolverRuleAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getResolverRuleAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetResolverRuleAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about the Resolver rule policy for a specified rule. A Resolver rule policy includes the rule
     * that you want to share with another account, the account that you want to share the rule with, and the Resolver
     * operations that you want to allow the account to use.
     * </p>
     *
     * @param getResolverRulePolicyRequest
     * @return Result of the GetResolverRulePolicy operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws UnknownResourceException
     *         The specified resource doesn't exist.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.GetResolverRulePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverRulePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResolverRulePolicyResponse getResolverRulePolicy(GetResolverRulePolicyRequest getResolverRulePolicyRequest)
            throws InvalidParameterException, UnknownResourceException, InternalServiceErrorException, AwsServiceException,
            SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResolverRulePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResolverRulePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResolverRulePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverRulePolicy");

            return clientHandler.execute(new ClientExecutionParams<GetResolverRulePolicyRequest, GetResolverRulePolicyResponse>()
                    .withOperationName("GetResolverRulePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getResolverRulePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResolverRulePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the configurations for DNSSEC validation that are associated with the current AWS account.
     * </p>
     *
     * @param listResolverDnssecConfigsRequest
     * @return Result of the ListResolverDnssecConfigs operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverDnssecConfigs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverDnssecConfigs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverDnssecConfigsResponse listResolverDnssecConfigs(
            ListResolverDnssecConfigsRequest listResolverDnssecConfigsRequest) throws InvalidNextTokenException,
            InvalidParameterException, InvalidRequestException, InternalServiceErrorException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResolverDnssecConfigsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResolverDnssecConfigsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResolverDnssecConfigsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolverDnssecConfigs");

            return clientHandler
                    .execute(new ClientExecutionParams<ListResolverDnssecConfigsRequest, ListResolverDnssecConfigsResponse>()
                            .withOperationName("ListResolverDnssecConfigs").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listResolverDnssecConfigsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListResolverDnssecConfigsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the configurations for DNSSEC validation that are associated with the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverDnssecConfigs(software.amazon.awssdk.services.route53resolver.model.ListResolverDnssecConfigsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverDnssecConfigsIterable responses = client.listResolverDnssecConfigsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53resolver.paginators.ListResolverDnssecConfigsIterable responses = client
     *             .listResolverDnssecConfigsPaginator(request);
     *     for (software.amazon.awssdk.services.route53resolver.model.ListResolverDnssecConfigsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverDnssecConfigsIterable responses = client.listResolverDnssecConfigsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverDnssecConfigs(software.amazon.awssdk.services.route53resolver.model.ListResolverDnssecConfigsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverDnssecConfigsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverDnssecConfigs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverDnssecConfigs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverDnssecConfigsIterable listResolverDnssecConfigsPaginator(
            ListResolverDnssecConfigsRequest listResolverDnssecConfigsRequest) throws InvalidNextTokenException,
            InvalidParameterException, InvalidRequestException, InternalServiceErrorException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53ResolverException {
        return new ListResolverDnssecConfigsIterable(this, applyPaginatorUserAgent(listResolverDnssecConfigsRequest));
    }

    /**
     * <p>
     * Gets the IP addresses for a specified Resolver endpoint.
     * </p>
     *
     * @param listResolverEndpointIpAddressesRequest
     * @return Result of the ListResolverEndpointIpAddresses operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverEndpointIpAddresses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpointIpAddresses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverEndpointIpAddressesResponse listResolverEndpointIpAddresses(
            ListResolverEndpointIpAddressesRequest listResolverEndpointIpAddressesRequest) throws ResourceNotFoundException,
            InvalidParameterException, InternalServiceErrorException, InvalidNextTokenException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResolverEndpointIpAddressesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResolverEndpointIpAddressesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listResolverEndpointIpAddressesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolverEndpointIpAddresses");

            return clientHandler
                    .execute(new ClientExecutionParams<ListResolverEndpointIpAddressesRequest, ListResolverEndpointIpAddressesResponse>()
                            .withOperationName("ListResolverEndpointIpAddresses").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listResolverEndpointIpAddressesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListResolverEndpointIpAddressesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the IP addresses for a specified Resolver endpoint.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverEndpointIpAddresses(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointIpAddressesIterable responses = client.listResolverEndpointIpAddressesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointIpAddressesIterable responses = client
     *             .listResolverEndpointIpAddressesPaginator(request);
     *     for (software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointIpAddressesIterable responses = client.listResolverEndpointIpAddressesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverEndpointIpAddresses(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverEndpointIpAddressesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverEndpointIpAddresses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpointIpAddresses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverEndpointIpAddressesIterable listResolverEndpointIpAddressesPaginator(
            ListResolverEndpointIpAddressesRequest listResolverEndpointIpAddressesRequest) throws ResourceNotFoundException,
            InvalidParameterException, InternalServiceErrorException, InvalidNextTokenException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        return new ListResolverEndpointIpAddressesIterable(this, applyPaginatorUserAgent(listResolverEndpointIpAddressesRequest));
    }

    /**
     * <p>
     * Lists all the Resolver endpoints that were created using the current AWS account.
     * </p>
     *
     * @param listResolverEndpointsRequest
     * @return Result of the ListResolverEndpoints operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverEndpoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverEndpointsResponse listResolverEndpoints(ListResolverEndpointsRequest listResolverEndpointsRequest)
            throws InvalidNextTokenException, InvalidRequestException, InvalidParameterException, InternalServiceErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResolverEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResolverEndpointsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResolverEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolverEndpoints");

            return clientHandler.execute(new ClientExecutionParams<ListResolverEndpointsRequest, ListResolverEndpointsResponse>()
                    .withOperationName("ListResolverEndpoints").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listResolverEndpointsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResolverEndpointsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the Resolver endpoints that were created using the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverEndpoints(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointsIterable responses = client.listResolverEndpointsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointsIterable responses = client
     *             .listResolverEndpointsPaginator(request);
     *     for (software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointsIterable responses = client.listResolverEndpointsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverEndpoints(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverEndpointsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverEndpoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverEndpointsIterable listResolverEndpointsPaginator(ListResolverEndpointsRequest listResolverEndpointsRequest)
            throws InvalidNextTokenException, InvalidRequestException, InvalidParameterException, InternalServiceErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        return new ListResolverEndpointsIterable(this, applyPaginatorUserAgent(listResolverEndpointsRequest));
    }

    /**
     * <p>
     * Lists information about associations between Amazon VPCs and query logging configurations.
     * </p>
     *
     * @param listResolverQueryLogConfigAssociationsRequest
     * @return Result of the ListResolverQueryLogConfigAssociations operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverQueryLogConfigAssociations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverQueryLogConfigAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverQueryLogConfigAssociationsResponse listResolverQueryLogConfigAssociations(
            ListResolverQueryLogConfigAssociationsRequest listResolverQueryLogConfigAssociationsRequest)
            throws InvalidParameterException, InvalidRequestException, LimitExceededException, InternalServiceErrorException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResolverQueryLogConfigAssociationsResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, ListResolverQueryLogConfigAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listResolverQueryLogConfigAssociationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolverQueryLogConfigAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListResolverQueryLogConfigAssociationsRequest, ListResolverQueryLogConfigAssociationsResponse>()
                            .withOperationName("ListResolverQueryLogConfigAssociations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(listResolverQueryLogConfigAssociationsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListResolverQueryLogConfigAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists information about associations between Amazon VPCs and query logging configurations.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverQueryLogConfigAssociations(software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigAssociationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverQueryLogConfigAssociationsIterable responses = client.listResolverQueryLogConfigAssociationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53resolver.paginators.ListResolverQueryLogConfigAssociationsIterable responses = client
     *             .listResolverQueryLogConfigAssociationsPaginator(request);
     *     for (software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigAssociationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverQueryLogConfigAssociationsIterable responses = client.listResolverQueryLogConfigAssociationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverQueryLogConfigAssociations(software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigAssociationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverQueryLogConfigAssociationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverQueryLogConfigAssociations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverQueryLogConfigAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverQueryLogConfigAssociationsIterable listResolverQueryLogConfigAssociationsPaginator(
            ListResolverQueryLogConfigAssociationsRequest listResolverQueryLogConfigAssociationsRequest)
            throws InvalidParameterException, InvalidRequestException, LimitExceededException, InternalServiceErrorException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, Route53ResolverException {
        return new ListResolverQueryLogConfigAssociationsIterable(this,
                applyPaginatorUserAgent(listResolverQueryLogConfigAssociationsRequest));
    }

    /**
     * <p>
     * Lists information about the specified query logging configurations. Each configuration defines where you want
     * Resolver to save DNS query logs and specifies the VPCs that you want to log queries for.
     * </p>
     *
     * @param listResolverQueryLogConfigsRequest
     * @return Result of the ListResolverQueryLogConfigs operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverQueryLogConfigs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverQueryLogConfigs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverQueryLogConfigsResponse listResolverQueryLogConfigs(
            ListResolverQueryLogConfigsRequest listResolverQueryLogConfigsRequest) throws InvalidNextTokenException,
            InvalidRequestException, InvalidParameterException, InternalServiceErrorException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResolverQueryLogConfigsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResolverQueryLogConfigsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResolverQueryLogConfigsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolverQueryLogConfigs");

            return clientHandler
                    .execute(new ClientExecutionParams<ListResolverQueryLogConfigsRequest, ListResolverQueryLogConfigsResponse>()
                            .withOperationName("ListResolverQueryLogConfigs").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listResolverQueryLogConfigsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListResolverQueryLogConfigsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists information about the specified query logging configurations. Each configuration defines where you want
     * Resolver to save DNS query logs and specifies the VPCs that you want to log queries for.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverQueryLogConfigs(software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverQueryLogConfigsIterable responses = client.listResolverQueryLogConfigsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53resolver.paginators.ListResolverQueryLogConfigsIterable responses = client
     *             .listResolverQueryLogConfigsPaginator(request);
     *     for (software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverQueryLogConfigsIterable responses = client.listResolverQueryLogConfigsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverQueryLogConfigs(software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverQueryLogConfigsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverQueryLogConfigs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverQueryLogConfigs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverQueryLogConfigsIterable listResolverQueryLogConfigsPaginator(
            ListResolverQueryLogConfigsRequest listResolverQueryLogConfigsRequest) throws InvalidNextTokenException,
            InvalidRequestException, InvalidParameterException, InternalServiceErrorException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53ResolverException {
        return new ListResolverQueryLogConfigsIterable(this, applyPaginatorUserAgent(listResolverQueryLogConfigsRequest));
    }

    /**
     * <p>
     * Lists the associations that were created between Resolver rules and VPCs using the current AWS account.
     * </p>
     *
     * @param listResolverRuleAssociationsRequest
     * @return Result of the ListResolverRuleAssociations operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverRuleAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRuleAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverRuleAssociationsResponse listResolverRuleAssociations(
            ListResolverRuleAssociationsRequest listResolverRuleAssociationsRequest) throws InvalidNextTokenException,
            InvalidRequestException, InvalidParameterException, InternalServiceErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResolverRuleAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResolverRuleAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResolverRuleAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolverRuleAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListResolverRuleAssociationsRequest, ListResolverRuleAssociationsResponse>()
                            .withOperationName("ListResolverRuleAssociations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listResolverRuleAssociationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListResolverRuleAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the associations that were created between Resolver rules and VPCs using the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverRuleAssociations(software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRuleAssociationsIterable responses = client.listResolverRuleAssociationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53resolver.paginators.ListResolverRuleAssociationsIterable responses = client
     *             .listResolverRuleAssociationsPaginator(request);
     *     for (software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRuleAssociationsIterable responses = client.listResolverRuleAssociationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverRuleAssociations(software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverRuleAssociationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverRuleAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRuleAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverRuleAssociationsIterable listResolverRuleAssociationsPaginator(
            ListResolverRuleAssociationsRequest listResolverRuleAssociationsRequest) throws InvalidNextTokenException,
            InvalidRequestException, InvalidParameterException, InternalServiceErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        return new ListResolverRuleAssociationsIterable(this, applyPaginatorUserAgent(listResolverRuleAssociationsRequest));
    }

    /**
     * <p>
     * Lists the Resolver rules that were created using the current AWS account.
     * </p>
     *
     * @param listResolverRulesRequest
     * @return Result of the ListResolverRules operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverRulesResponse listResolverRules(ListResolverRulesRequest listResolverRulesRequest)
            throws InvalidNextTokenException, InvalidRequestException, InvalidParameterException, InternalServiceErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResolverRulesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListResolverRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResolverRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolverRules");

            return clientHandler.execute(new ClientExecutionParams<ListResolverRulesRequest, ListResolverRulesResponse>()
                    .withOperationName("ListResolverRules").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listResolverRulesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResolverRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the Resolver rules that were created using the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverRules(software.amazon.awssdk.services.route53resolver.model.ListResolverRulesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRulesIterable responses = client.listResolverRulesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53resolver.paginators.ListResolverRulesIterable responses = client
     *             .listResolverRulesPaginator(request);
     *     for (software.amazon.awssdk.services.route53resolver.model.ListResolverRulesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRulesIterable responses = client.listResolverRulesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverRules(software.amazon.awssdk.services.route53resolver.model.ListResolverRulesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverRulesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListResolverRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResolverRulesIterable listResolverRulesPaginator(ListResolverRulesRequest listResolverRulesRequest)
            throws InvalidNextTokenException, InvalidRequestException, InvalidParameterException, InternalServiceErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        return new ListResolverRulesIterable(this, applyPaginatorUserAgent(listResolverRulesRequest));
    }

    /**
     * <p>
     * Lists the tags that you associated with the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InvalidParameterException, InvalidNextTokenException, InvalidRequestException,
            InternalServiceErrorException, ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags that you associated with the specified resource.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.route53resolver.model.ListTagsForResourceRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53resolver.paginators.ListTagsForResourceIterable responses = client
     *             .listTagsForResourcePaginator(request);
     *     for (software.amazon.awssdk.services.route53resolver.model.ListTagsForResourceResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.route53resolver.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidNextTokenException
     *         The value that you specified for <code>NextToken</code> in a <code>List</code> request isn't valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceIterable listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InvalidParameterException, InvalidNextTokenException, InvalidRequestException,
            InternalServiceErrorException, ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        return new ListTagsForResourceIterable(this, applyPaginatorUserAgent(listTagsForResourceRequest));
    }

    /**
     * <p>
     * Specifies an AWS account that you want to share a query logging configuration with, the query logging
     * configuration that you want to share, and the operations that you want the account to be able to perform on the
     * configuration.
     * </p>
     *
     * @param putResolverQueryLogConfigPolicyRequest
     * @return Result of the PutResolverQueryLogConfigPolicy operation returned by the service.
     * @throws InvalidPolicyDocumentException
     *         The specified Resolver rule policy is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws UnknownResourceException
     *         The specified resource doesn't exist.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.PutResolverQueryLogConfigPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/PutResolverQueryLogConfigPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutResolverQueryLogConfigPolicyResponse putResolverQueryLogConfigPolicy(
            PutResolverQueryLogConfigPolicyRequest putResolverQueryLogConfigPolicyRequest) throws InvalidPolicyDocumentException,
            InvalidParameterException, InvalidRequestException, UnknownResourceException, InternalServiceErrorException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutResolverQueryLogConfigPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutResolverQueryLogConfigPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putResolverQueryLogConfigPolicyRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResolverQueryLogConfigPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<PutResolverQueryLogConfigPolicyRequest, PutResolverQueryLogConfigPolicyResponse>()
                            .withOperationName("PutResolverQueryLogConfigPolicy").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putResolverQueryLogConfigPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutResolverQueryLogConfigPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Specifies an AWS rule that you want to share with another account, the account that you want to share the rule
     * with, and the operations that you want the account to be able to perform on the rule.
     * </p>
     *
     * @param putResolverRulePolicyRequest
     * @return Result of the PutResolverRulePolicy operation returned by the service.
     * @throws InvalidPolicyDocumentException
     *         The specified Resolver rule policy is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws UnknownResourceException
     *         The specified resource doesn't exist.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.PutResolverRulePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/PutResolverRulePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutResolverRulePolicyResponse putResolverRulePolicy(PutResolverRulePolicyRequest putResolverRulePolicyRequest)
            throws InvalidPolicyDocumentException, InvalidParameterException, UnknownResourceException,
            InternalServiceErrorException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutResolverRulePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutResolverRulePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResolverRulePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResolverRulePolicy");

            return clientHandler.execute(new ClientExecutionParams<PutResolverRulePolicyRequest, PutResolverRulePolicyResponse>()
                    .withOperationName("PutResolverRulePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putResolverRulePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutResolverRulePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more tags to a specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidTagException
     *         The specified tag is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws LimitExceededException,
            ResourceNotFoundException, InvalidParameterException, InvalidRequestException, InvalidTagException,
            InternalServiceErrorException, ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from a specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InvalidParameterException, InvalidRequestException, InternalServiceErrorException, ThrottlingException,
            AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing DNSSEC validation configuration. If there is no existing DNSSEC validation configuration, one
     * is created.
     * </p>
     *
     * @param updateResolverDnssecConfigRequest
     * @return Result of the UpdateResolverDnssecConfig operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws AccessDeniedException
     *         The current account doesn't have the IAM permissions required to perform the specified Resolver
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.UpdateResolverDnssecConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/UpdateResolverDnssecConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateResolverDnssecConfigResponse updateResolverDnssecConfig(
            UpdateResolverDnssecConfigRequest updateResolverDnssecConfigRequest) throws InvalidParameterException,
            ResourceNotFoundException, InvalidRequestException, InternalServiceErrorException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateResolverDnssecConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateResolverDnssecConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResolverDnssecConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResolverDnssecConfig");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateResolverDnssecConfigRequest, UpdateResolverDnssecConfigResponse>()
                            .withOperationName("UpdateResolverDnssecConfig").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateResolverDnssecConfigRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateResolverDnssecConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the name of an inbound or an outbound Resolver endpoint.
     * </p>
     *
     * @param updateResolverEndpointRequest
     * @return Result of the UpdateResolverEndpoint operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.UpdateResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/UpdateResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateResolverEndpointResponse updateResolverEndpoint(UpdateResolverEndpointRequest updateResolverEndpointRequest)
            throws ResourceNotFoundException, InvalidParameterException, InvalidRequestException, InternalServiceErrorException,
            ThrottlingException, AwsServiceException, SdkClientException, Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateResolverEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResolverEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResolverEndpoint");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateResolverEndpointRequest, UpdateResolverEndpointResponse>()
                            .withOperationName("UpdateResolverEndpoint").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateResolverEndpointRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateResolverEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates settings for a specified Resolver rule. <code>ResolverRuleId</code> is required, and all other parameters
     * are optional. If you don't specify a parameter, it retains its current value.
     * </p>
     *
     * @param updateResolverRuleRequest
     * @return Result of the UpdateResolverRule operation returned by the service.
     * @throws InvalidRequestException
     *         The request is invalid.
     * @throws InvalidParameterException
     *         One or more parameters in this request are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ResourceUnavailableException
     *         The specified resource isn't available.
     * @throws LimitExceededException
     *         The request caused one or more limits to be exceeded.
     * @throws InternalServiceErrorException
     *         We encountered an unknown error. Try again in a few minutes.
     * @throws ThrottlingException
     *         The request was throttled. Try again in a few minutes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53ResolverException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53ResolverClient.UpdateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/UpdateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateResolverRuleResponse updateResolverRule(UpdateResolverRuleRequest updateResolverRuleRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ResourceUnavailableException,
            LimitExceededException, InternalServiceErrorException, ThrottlingException, AwsServiceException, SdkClientException,
            Route53ResolverException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateResolverRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResolverRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResolverRule");

            return clientHandler.execute(new ClientExecutionParams<UpdateResolverRuleRequest, UpdateResolverRuleResponse>()
                    .withOperationName("UpdateResolverRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateResolverRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateResolverRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Route53ResolverException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPolicyDocument")
                                .exceptionBuilderSupplier(InvalidPolicyDocumentException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagException")
                                .exceptionBuilderSupplier(InvalidTagException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnknownResourceException")
                                .exceptionBuilderSupplier(UnknownResourceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceUnavailableException")
                                .exceptionBuilderSupplier(ResourceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceExistsException")
                                .exceptionBuilderSupplier(ResourceExistsException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends Route53ResolverRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
