/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53resolver;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.route53resolver.model.AccessDeniedException;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverEndpointIpAddressRequest;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverEndpointIpAddressResponse;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverQueryLogConfigRequest;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverQueryLogConfigResponse;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.AssociateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverQueryLogConfigRequest;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverQueryLogConfigResponse;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.CreateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverQueryLogConfigRequest;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverQueryLogConfigResponse;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.DeleteResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverEndpointIpAddressRequest;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverEndpointIpAddressResponse;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverQueryLogConfigRequest;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverQueryLogConfigResponse;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.DisassociateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverDnssecConfigRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverDnssecConfigResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverQueryLogConfigAssociationRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverQueryLogConfigAssociationResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverQueryLogConfigPolicyRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverQueryLogConfigPolicyResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverQueryLogConfigRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverQueryLogConfigResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleAssociationRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleAssociationResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRulePolicyRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRulePolicyResponse;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.GetResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.model.InternalServiceErrorException;
import software.amazon.awssdk.services.route53resolver.model.InvalidNextTokenException;
import software.amazon.awssdk.services.route53resolver.model.InvalidParameterException;
import software.amazon.awssdk.services.route53resolver.model.InvalidPolicyDocumentException;
import software.amazon.awssdk.services.route53resolver.model.InvalidRequestException;
import software.amazon.awssdk.services.route53resolver.model.InvalidTagException;
import software.amazon.awssdk.services.route53resolver.model.LimitExceededException;
import software.amazon.awssdk.services.route53resolver.model.ListResolverDnssecConfigsRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverDnssecConfigsResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigAssociationsRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigAssociationsResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigsRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigsResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsResponse;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRulesRequest;
import software.amazon.awssdk.services.route53resolver.model.ListResolverRulesResponse;
import software.amazon.awssdk.services.route53resolver.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.route53resolver.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.route53resolver.model.PutResolverQueryLogConfigPolicyRequest;
import software.amazon.awssdk.services.route53resolver.model.PutResolverQueryLogConfigPolicyResponse;
import software.amazon.awssdk.services.route53resolver.model.PutResolverRulePolicyRequest;
import software.amazon.awssdk.services.route53resolver.model.PutResolverRulePolicyResponse;
import software.amazon.awssdk.services.route53resolver.model.ResourceExistsException;
import software.amazon.awssdk.services.route53resolver.model.ResourceInUseException;
import software.amazon.awssdk.services.route53resolver.model.ResourceNotFoundException;
import software.amazon.awssdk.services.route53resolver.model.ResourceUnavailableException;
import software.amazon.awssdk.services.route53resolver.model.Route53ResolverException;
import software.amazon.awssdk.services.route53resolver.model.Route53ResolverRequest;
import software.amazon.awssdk.services.route53resolver.model.TagResourceRequest;
import software.amazon.awssdk.services.route53resolver.model.TagResourceResponse;
import software.amazon.awssdk.services.route53resolver.model.ThrottlingException;
import software.amazon.awssdk.services.route53resolver.model.UnknownResourceException;
import software.amazon.awssdk.services.route53resolver.model.UntagResourceRequest;
import software.amazon.awssdk.services.route53resolver.model.UntagResourceResponse;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverDnssecConfigRequest;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverDnssecConfigResponse;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverEndpointRequest;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverEndpointResponse;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverRuleRequest;
import software.amazon.awssdk.services.route53resolver.model.UpdateResolverRuleResponse;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverDnssecConfigsPublisher;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointIpAddressesPublisher;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointsPublisher;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverQueryLogConfigAssociationsPublisher;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverQueryLogConfigsPublisher;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverRuleAssociationsPublisher;
import software.amazon.awssdk.services.route53resolver.paginators.ListResolverRulesPublisher;
import software.amazon.awssdk.services.route53resolver.paginators.ListTagsForResourcePublisher;
import software.amazon.awssdk.services.route53resolver.transform.AssociateResolverEndpointIpAddressRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.AssociateResolverQueryLogConfigRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.AssociateResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.CreateResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.CreateResolverQueryLogConfigRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.CreateResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DeleteResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DeleteResolverQueryLogConfigRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DeleteResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DisassociateResolverEndpointIpAddressRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DisassociateResolverQueryLogConfigRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.DisassociateResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverDnssecConfigRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverQueryLogConfigAssociationRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverQueryLogConfigPolicyRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverQueryLogConfigRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverRuleAssociationRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverRulePolicyRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.GetResolverRuleRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverDnssecConfigsRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverEndpointIpAddressesRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverEndpointsRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverQueryLogConfigAssociationsRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverQueryLogConfigsRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverRuleAssociationsRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListResolverRulesRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.PutResolverQueryLogConfigPolicyRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.PutResolverRulePolicyRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.UpdateResolverDnssecConfigRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.UpdateResolverEndpointRequestMarshaller;
import software.amazon.awssdk.services.route53resolver.transform.UpdateResolverRuleRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link Route53ResolverAsyncClient}.
 *
 * @see Route53ResolverAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRoute53ResolverAsyncClient implements Route53ResolverAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRoute53ResolverAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRoute53ResolverAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds IP addresses to an inbound or an outbound Resolver endpoint. If you want to add more than one IP address,
     * submit one <code>AssociateResolverEndpointIpAddress</code> request for each IP address.
     * </p>
     * <p>
     * To remove an IP address from an endpoint, see <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverEndpointIpAddress.html"
     * >DisassociateResolverEndpointIpAddress</a>.
     * </p>
     *
     * @param associateResolverEndpointIpAddressRequest
     * @return A Java Future containing the result of the AssociateResolverEndpointIpAddress operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>ResourceExistsException The resource that you tried to create already exists.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.AssociateResolverEndpointIpAddress
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/AssociateResolverEndpointIpAddress"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateResolverEndpointIpAddressResponse> associateResolverEndpointIpAddress(
            AssociateResolverEndpointIpAddressRequest associateResolverEndpointIpAddressRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateResolverEndpointIpAddressRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResolverEndpointIpAddress");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateResolverEndpointIpAddressResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AssociateResolverEndpointIpAddressResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateResolverEndpointIpAddressResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateResolverEndpointIpAddressRequest, AssociateResolverEndpointIpAddressResponse>()
                            .withOperationName("AssociateResolverEndpointIpAddress")
                            .withMarshaller(new AssociateResolverEndpointIpAddressRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateResolverEndpointIpAddressRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = associateResolverEndpointIpAddressRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<AssociateResolverEndpointIpAddressResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates an Amazon VPC with a specified query logging configuration. Route 53 Resolver logs DNS queries that
     * originate in all of the Amazon VPCs that are associated with a specified query logging configuration. To
     * associate more than one VPC with a configuration, submit one <code>AssociateResolverQueryLogConfig</code> request
     * for each VPC.
     * </p>
     * <note>
     * <p>
     * The VPCs that you associate with a query logging configuration must be in the same Region as the configuration.
     * </p>
     * </note>
     * <p>
     * To remove a VPC from a query logging configuration, see <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverQueryLogConfig.html"
     * >DisassociateResolverQueryLogConfig</a>.
     * </p>
     *
     * @param associateResolverQueryLogConfigRequest
     * @return A Java Future containing the result of the AssociateResolverQueryLogConfig operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>ResourceExistsException The resource that you tried to create already exists.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.AssociateResolverQueryLogConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/AssociateResolverQueryLogConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateResolverQueryLogConfigResponse> associateResolverQueryLogConfig(
            AssociateResolverQueryLogConfigRequest associateResolverQueryLogConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateResolverQueryLogConfigRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResolverQueryLogConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateResolverQueryLogConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateResolverQueryLogConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateResolverQueryLogConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateResolverQueryLogConfigRequest, AssociateResolverQueryLogConfigResponse>()
                            .withOperationName("AssociateResolverQueryLogConfig")
                            .withMarshaller(new AssociateResolverQueryLogConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateResolverQueryLogConfigRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = associateResolverQueryLogConfigRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<AssociateResolverQueryLogConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates a Resolver rule with a VPC. When you associate a rule with a VPC, Resolver forwards all DNS queries
     * for the domain name that is specified in the rule and that originate in the VPC. The queries are forwarded to the
     * IP addresses for the DNS resolvers that are specified in the rule. For more information about rules, see <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_CreateResolverRule.html"
     * >CreateResolverRule</a>.
     * </p>
     *
     * @param associateResolverRuleRequest
     * @return A Java Future containing the result of the AssociateResolverRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ResourceUnavailableException The specified resource isn't available.</li>
     *         <li>ResourceExistsException The resource that you tried to create already exists.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.AssociateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/AssociateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateResolverRuleResponse> associateResolverRule(
            AssociateResolverRuleRequest associateResolverRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateResolverRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateResolverRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateResolverRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateResolverRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateResolverRuleRequest, AssociateResolverRuleResponse>()
                            .withOperationName("AssociateResolverRule")
                            .withMarshaller(new AssociateResolverRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateResolverRuleRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = associateResolverRuleRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<AssociateResolverRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a Resolver endpoint. There are two types of Resolver endpoints, inbound and outbound:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An <i>inbound Resolver endpoint</i> forwards DNS queries to the DNS service for a VPC from your network.
     * </p>
     * </li>
     * <li>
     * <p>
     * An <i>outbound Resolver endpoint</i> forwards DNS queries from the DNS service for a VPC to your network.
     * </p>
     * </li>
     * </ul>
     *
     * @param createResolverEndpointRequest
     * @return A Java Future containing the result of the CreateResolverEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>ResourceExistsException The resource that you tried to create already exists.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.CreateResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/CreateResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateResolverEndpointResponse> createResolverEndpoint(
            CreateResolverEndpointRequest createResolverEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResolverEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResolverEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateResolverEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateResolverEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateResolverEndpointRequest, CreateResolverEndpointResponse>()
                            .withOperationName("CreateResolverEndpoint")
                            .withMarshaller(new CreateResolverEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createResolverEndpointRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createResolverEndpointRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreateResolverEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a Resolver query logging configuration, which defines where you want Resolver to save DNS query logs that
     * originate in your VPCs. Resolver can log queries only for VPCs that are in the same Region as the query logging
     * configuration.
     * </p>
     * <p>
     * To specify which VPCs you want to log queries for, you use <code>AssociateResolverQueryLogConfig</code>. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_AssociateResolverQueryLogConfig.html"
     * >AssociateResolverQueryLogConfig</a>.
     * </p>
     * <p>
     * You can optionally use AWS Resource Access Manager (AWS RAM) to share a query logging configuration with other
     * AWS accounts. The other accounts can then associate VPCs with the configuration. The query logs that Resolver
     * creates for a configuration include all DNS queries that originate in all VPCs that are associated with the
     * configuration.
     * </p>
     *
     * @param createResolverQueryLogConfigRequest
     * @return A Java Future containing the result of the CreateResolverQueryLogConfig operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>ResourceExistsException The resource that you tried to create already exists.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.CreateResolverQueryLogConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/CreateResolverQueryLogConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateResolverQueryLogConfigResponse> createResolverQueryLogConfig(
            CreateResolverQueryLogConfigRequest createResolverQueryLogConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResolverQueryLogConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResolverQueryLogConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateResolverQueryLogConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateResolverQueryLogConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateResolverQueryLogConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateResolverQueryLogConfigRequest, CreateResolverQueryLogConfigResponse>()
                            .withOperationName("CreateResolverQueryLogConfig")
                            .withMarshaller(new CreateResolverQueryLogConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createResolverQueryLogConfigRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createResolverQueryLogConfigRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateResolverQueryLogConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For DNS queries that originate in your VPCs, specifies which Resolver endpoint the queries pass through, one
     * domain name that you want to forward to your network, and the IP addresses of the DNS resolvers in your network.
     * </p>
     *
     * @param createResolverRuleRequest
     * @return A Java Future containing the result of the CreateResolverRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourceExistsException The resource that you tried to create already exists.</li>
     *         <li>ResourceUnavailableException The specified resource isn't available.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.CreateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/CreateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateResolverRuleResponse> createResolverRule(CreateResolverRuleRequest createResolverRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResolverRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResolverRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateResolverRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateResolverRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateResolverRuleRequest, CreateResolverRuleResponse>()
                            .withOperationName("CreateResolverRule")
                            .withMarshaller(new CreateResolverRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createResolverRuleRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createResolverRuleRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateResolverRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Resolver endpoint. The effect of deleting a Resolver endpoint depends on whether it's an inbound or an
     * outbound Resolver endpoint:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Inbound</b>: DNS queries from your network are no longer routed to the DNS service for the specified VPC.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Outbound</b>: DNS queries from a VPC are no longer routed to your network.
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteResolverEndpointRequest
     * @return A Java Future containing the result of the DeleteResolverEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.DeleteResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DeleteResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResolverEndpointResponse> deleteResolverEndpoint(
            DeleteResolverEndpointRequest deleteResolverEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResolverEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResolverEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResolverEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResolverEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResolverEndpointRequest, DeleteResolverEndpointResponse>()
                            .withOperationName("DeleteResolverEndpoint")
                            .withMarshaller(new DeleteResolverEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteResolverEndpointRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteResolverEndpointRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteResolverEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a query logging configuration. When you delete a configuration, Resolver stops logging DNS queries for
     * all of the Amazon VPCs that are associated with the configuration. This also applies if the query logging
     * configuration is shared with other AWS accounts, and the other accounts have associated VPCs with the shared
     * configuration.
     * </p>
     * <p>
     * Before you can delete a query logging configuration, you must first disassociate all VPCs from the configuration.
     * See <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverQueryLogConfig.html"
     * >DisassociateResolverQueryLogConfig</a>.
     * </p>
     * <p>
     * If you used Resource Access Manager (RAM) to share a query logging configuration with other accounts, you must
     * stop sharing the configuration before you can delete a configuration. The accounts that you shared the
     * configuration with can first disassociate VPCs that they associated with the configuration, but that's not
     * necessary. If you stop sharing the configuration, those VPCs are automatically disassociated from the
     * configuration.
     * </p>
     *
     * @param deleteResolverQueryLogConfigRequest
     * @return A Java Future containing the result of the DeleteResolverQueryLogConfig operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.DeleteResolverQueryLogConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DeleteResolverQueryLogConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResolverQueryLogConfigResponse> deleteResolverQueryLogConfig(
            DeleteResolverQueryLogConfigRequest deleteResolverQueryLogConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResolverQueryLogConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResolverQueryLogConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResolverQueryLogConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResolverQueryLogConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResolverQueryLogConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResolverQueryLogConfigRequest, DeleteResolverQueryLogConfigResponse>()
                            .withOperationName("DeleteResolverQueryLogConfig")
                            .withMarshaller(new DeleteResolverQueryLogConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteResolverQueryLogConfigRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteResolverQueryLogConfigRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteResolverQueryLogConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Resolver rule. Before you can delete a Resolver rule, you must disassociate it from all the VPCs that
     * you associated the Resolver rule with. For more information, see <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_DisassociateResolverRule.html"
     * >DisassociateResolverRule</a>.
     * </p>
     *
     * @param deleteResolverRuleRequest
     * @return A Java Future containing the result of the DeleteResolverRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourceInUseException The resource that you tried to update or delete is currently in use.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.DeleteResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DeleteResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResolverRuleResponse> deleteResolverRule(DeleteResolverRuleRequest deleteResolverRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResolverRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResolverRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResolverRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResolverRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResolverRuleRequest, DeleteResolverRuleResponse>()
                            .withOperationName("DeleteResolverRule")
                            .withMarshaller(new DeleteResolverRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteResolverRuleRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteResolverRuleRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteResolverRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes IP addresses from an inbound or an outbound Resolver endpoint. If you want to remove more than one IP
     * address, submit one <code>DisassociateResolverEndpointIpAddress</code> request for each IP address.
     * </p>
     * <p>
     * To add an IP address to an endpoint, see <a href=
     * "https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_AssociateResolverEndpointIpAddress.html"
     * >AssociateResolverEndpointIpAddress</a>.
     * </p>
     *
     * @param disassociateResolverEndpointIpAddressRequest
     * @return A Java Future containing the result of the DisassociateResolverEndpointIpAddress operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>ResourceExistsException The resource that you tried to create already exists.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.DisassociateResolverEndpointIpAddress
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DisassociateResolverEndpointIpAddress"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateResolverEndpointIpAddressResponse> disassociateResolverEndpointIpAddress(
            DisassociateResolverEndpointIpAddressRequest disassociateResolverEndpointIpAddressRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateResolverEndpointIpAddressRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResolverEndpointIpAddress");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateResolverEndpointIpAddressResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateResolverEndpointIpAddressResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateResolverEndpointIpAddressResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateResolverEndpointIpAddressRequest, DisassociateResolverEndpointIpAddressResponse>()
                            .withOperationName("DisassociateResolverEndpointIpAddress")
                            .withMarshaller(new DisassociateResolverEndpointIpAddressRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateResolverEndpointIpAddressRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = disassociateResolverEndpointIpAddressRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DisassociateResolverEndpointIpAddressResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates a VPC from a query logging configuration.
     * </p>
     * <note>
     * <p>
     * Before you can delete a query logging configuration, you must first disassociate all VPCs from the configuration.
     * If you used Resource Access Manager (RAM) to share a query logging configuration with other accounts, VPCs can be
     * disassociated from the configuration in the following ways:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The accounts that you shared the configuration with can disassociate VPCs from the configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can stop sharing the configuration.
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param disassociateResolverQueryLogConfigRequest
     * @return A Java Future containing the result of the DisassociateResolverQueryLogConfig operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.DisassociateResolverQueryLogConfig
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DisassociateResolverQueryLogConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateResolverQueryLogConfigResponse> disassociateResolverQueryLogConfig(
            DisassociateResolverQueryLogConfigRequest disassociateResolverQueryLogConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateResolverQueryLogConfigRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResolverQueryLogConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateResolverQueryLogConfigResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateResolverQueryLogConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateResolverQueryLogConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateResolverQueryLogConfigRequest, DisassociateResolverQueryLogConfigResponse>()
                            .withOperationName("DisassociateResolverQueryLogConfig")
                            .withMarshaller(new DisassociateResolverQueryLogConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateResolverQueryLogConfigRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = disassociateResolverQueryLogConfigRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DisassociateResolverQueryLogConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the association between a specified Resolver rule and a specified VPC.
     * </p>
     * <important>
     * <p>
     * If you disassociate a Resolver rule from a VPC, Resolver stops forwarding DNS queries for the domain name that
     * you specified in the Resolver rule.
     * </p>
     * </important>
     *
     * @param disassociateResolverRuleRequest
     * @return A Java Future containing the result of the DisassociateResolverRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.DisassociateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/DisassociateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateResolverRuleResponse> disassociateResolverRule(
            DisassociateResolverRuleRequest disassociateResolverRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateResolverRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateResolverRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateResolverRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateResolverRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateResolverRuleRequest, DisassociateResolverRuleResponse>()
                            .withOperationName("DisassociateResolverRule")
                            .withMarshaller(new DisassociateResolverRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateResolverRuleRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = disassociateResolverRuleRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DisassociateResolverRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets DNSSEC validation information for a specified resource.
     * </p>
     *
     * @param getResolverDnssecConfigRequest
     * @return A Java Future containing the result of the GetResolverDnssecConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.GetResolverDnssecConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverDnssecConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResolverDnssecConfigResponse> getResolverDnssecConfig(
            GetResolverDnssecConfigRequest getResolverDnssecConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResolverDnssecConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverDnssecConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResolverDnssecConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResolverDnssecConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResolverDnssecConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResolverDnssecConfigRequest, GetResolverDnssecConfigResponse>()
                            .withOperationName("GetResolverDnssecConfig")
                            .withMarshaller(new GetResolverDnssecConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResolverDnssecConfigRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getResolverDnssecConfigRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetResolverDnssecConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a specified Resolver endpoint, such as whether it's an inbound or an outbound Resolver
     * endpoint, and the current status of the endpoint.
     * </p>
     *
     * @param getResolverEndpointRequest
     * @return A Java Future containing the result of the GetResolverEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.GetResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResolverEndpointResponse> getResolverEndpoint(
            GetResolverEndpointRequest getResolverEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResolverEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResolverEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResolverEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResolverEndpointRequest, GetResolverEndpointResponse>()
                            .withOperationName("GetResolverEndpoint")
                            .withMarshaller(new GetResolverEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResolverEndpointRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getResolverEndpointRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetResolverEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a specified Resolver query logging configuration, such as the number of VPCs that the
     * configuration is logging queries for and the location that logs are sent to.
     * </p>
     *
     * @param getResolverQueryLogConfigRequest
     * @return A Java Future containing the result of the GetResolverQueryLogConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.GetResolverQueryLogConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverQueryLogConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResolverQueryLogConfigResponse> getResolverQueryLogConfig(
            GetResolverQueryLogConfigRequest getResolverQueryLogConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResolverQueryLogConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverQueryLogConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResolverQueryLogConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResolverQueryLogConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResolverQueryLogConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResolverQueryLogConfigRequest, GetResolverQueryLogConfigResponse>()
                            .withOperationName("GetResolverQueryLogConfig")
                            .withMarshaller(new GetResolverQueryLogConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResolverQueryLogConfigRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getResolverQueryLogConfigRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetResolverQueryLogConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a specified association between a Resolver query logging configuration and an Amazon VPC.
     * When you associate a VPC with a query logging configuration, Resolver logs DNS queries that originate in that
     * VPC.
     * </p>
     *
     * @param getResolverQueryLogConfigAssociationRequest
     * @return A Java Future containing the result of the GetResolverQueryLogConfigAssociation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.GetResolverQueryLogConfigAssociation
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverQueryLogConfigAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResolverQueryLogConfigAssociationResponse> getResolverQueryLogConfigAssociation(
            GetResolverQueryLogConfigAssociationRequest getResolverQueryLogConfigAssociationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getResolverQueryLogConfigAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverQueryLogConfigAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResolverQueryLogConfigAssociationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetResolverQueryLogConfigAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResolverQueryLogConfigAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResolverQueryLogConfigAssociationRequest, GetResolverQueryLogConfigAssociationResponse>()
                            .withOperationName("GetResolverQueryLogConfigAssociation")
                            .withMarshaller(new GetResolverQueryLogConfigAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResolverQueryLogConfigAssociationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getResolverQueryLogConfigAssociationRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetResolverQueryLogConfigAssociationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a query logging policy. A query logging policy specifies the Resolver query logging
     * operations and resources that you want to allow another AWS account to be able to use.
     * </p>
     *
     * @param getResolverQueryLogConfigPolicyRequest
     * @return A Java Future containing the result of the GetResolverQueryLogConfigPolicy operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>UnknownResourceException The specified resource doesn't exist.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.GetResolverQueryLogConfigPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverQueryLogConfigPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResolverQueryLogConfigPolicyResponse> getResolverQueryLogConfigPolicy(
            GetResolverQueryLogConfigPolicyRequest getResolverQueryLogConfigPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getResolverQueryLogConfigPolicyRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverQueryLogConfigPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResolverQueryLogConfigPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResolverQueryLogConfigPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResolverQueryLogConfigPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResolverQueryLogConfigPolicyRequest, GetResolverQueryLogConfigPolicyResponse>()
                            .withOperationName("GetResolverQueryLogConfigPolicy")
                            .withMarshaller(new GetResolverQueryLogConfigPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResolverQueryLogConfigPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getResolverQueryLogConfigPolicyRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetResolverQueryLogConfigPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a specified Resolver rule, such as the domain name that the rule forwards DNS queries for
     * and the ID of the outbound Resolver endpoint that the rule is associated with.
     * </p>
     *
     * @param getResolverRuleRequest
     * @return A Java Future containing the result of the GetResolverRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.GetResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResolverRuleResponse> getResolverRule(GetResolverRuleRequest getResolverRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResolverRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResolverRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResolverRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResolverRuleRequest, GetResolverRuleResponse>()
                            .withOperationName("GetResolverRule")
                            .withMarshaller(new GetResolverRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResolverRuleRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getResolverRuleRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetResolverRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about an association between a specified Resolver rule and a VPC. You associate a Resolver rule
     * and a VPC using <a
     * href="https://docs.aws.amazon.com/Route53/latest/APIReference/API_route53resolver_AssociateResolverRule.html"
     * >AssociateResolverRule</a>.
     * </p>
     *
     * @param getResolverRuleAssociationRequest
     * @return A Java Future containing the result of the GetResolverRuleAssociation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.GetResolverRuleAssociation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverRuleAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResolverRuleAssociationResponse> getResolverRuleAssociation(
            GetResolverRuleAssociationRequest getResolverRuleAssociationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResolverRuleAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverRuleAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResolverRuleAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResolverRuleAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResolverRuleAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResolverRuleAssociationRequest, GetResolverRuleAssociationResponse>()
                            .withOperationName("GetResolverRuleAssociation")
                            .withMarshaller(new GetResolverRuleAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResolverRuleAssociationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getResolverRuleAssociationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetResolverRuleAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the Resolver rule policy for a specified rule. A Resolver rule policy includes the rule
     * that you want to share with another account, the account that you want to share the rule with, and the Resolver
     * operations that you want to allow the account to use.
     * </p>
     *
     * @param getResolverRulePolicyRequest
     * @return A Java Future containing the result of the GetResolverRulePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>UnknownResourceException The specified resource doesn't exist.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.GetResolverRulePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/GetResolverRulePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResolverRulePolicyResponse> getResolverRulePolicy(
            GetResolverRulePolicyRequest getResolverRulePolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResolverRulePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResolverRulePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResolverRulePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResolverRulePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResolverRulePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResolverRulePolicyRequest, GetResolverRulePolicyResponse>()
                            .withOperationName("GetResolverRulePolicy")
                            .withMarshaller(new GetResolverRulePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResolverRulePolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getResolverRulePolicyRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetResolverRulePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the configurations for DNSSEC validation that are associated with the current AWS account.
     * </p>
     *
     * @param listResolverDnssecConfigsRequest
     * @return A Java Future containing the result of the ListResolverDnssecConfigs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverDnssecConfigs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverDnssecConfigs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResolverDnssecConfigsResponse> listResolverDnssecConfigs(
            ListResolverDnssecConfigsRequest listResolverDnssecConfigsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResolverDnssecConfigsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolverDnssecConfigs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResolverDnssecConfigsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResolverDnssecConfigsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResolverDnssecConfigsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResolverDnssecConfigsRequest, ListResolverDnssecConfigsResponse>()
                            .withOperationName("ListResolverDnssecConfigs")
                            .withMarshaller(new ListResolverDnssecConfigsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResolverDnssecConfigsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listResolverDnssecConfigsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListResolverDnssecConfigsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the configurations for DNSSEC validation that are associated with the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverDnssecConfigs(software.amazon.awssdk.services.route53resolver.model.ListResolverDnssecConfigsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverDnssecConfigsPublisher publisher = client.listResolverDnssecConfigsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverDnssecConfigsPublisher publisher = client.listResolverDnssecConfigsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.route53resolver.model.ListResolverDnssecConfigsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.route53resolver.model.ListResolverDnssecConfigsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverDnssecConfigs(software.amazon.awssdk.services.route53resolver.model.ListResolverDnssecConfigsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverDnssecConfigsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverDnssecConfigs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverDnssecConfigs"
     *      target="_top">AWS API Documentation</a>
     */
    public ListResolverDnssecConfigsPublisher listResolverDnssecConfigsPaginator(
            ListResolverDnssecConfigsRequest listResolverDnssecConfigsRequest) {
        return new ListResolverDnssecConfigsPublisher(this, applyPaginatorUserAgent(listResolverDnssecConfigsRequest));
    }

    /**
     * <p>
     * Gets the IP addresses for a specified Resolver endpoint.
     * </p>
     *
     * @param listResolverEndpointIpAddressesRequest
     * @return A Java Future containing the result of the ListResolverEndpointIpAddresses operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverEndpointIpAddresses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpointIpAddresses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResolverEndpointIpAddressesResponse> listResolverEndpointIpAddresses(
            ListResolverEndpointIpAddressesRequest listResolverEndpointIpAddressesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listResolverEndpointIpAddressesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolverEndpointIpAddresses");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResolverEndpointIpAddressesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResolverEndpointIpAddressesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResolverEndpointIpAddressesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResolverEndpointIpAddressesRequest, ListResolverEndpointIpAddressesResponse>()
                            .withOperationName("ListResolverEndpointIpAddresses")
                            .withMarshaller(new ListResolverEndpointIpAddressesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResolverEndpointIpAddressesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listResolverEndpointIpAddressesRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListResolverEndpointIpAddressesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the IP addresses for a specified Resolver endpoint.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverEndpointIpAddresses(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointIpAddressesPublisher publisher = client.listResolverEndpointIpAddressesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointIpAddressesPublisher publisher = client.listResolverEndpointIpAddressesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverEndpointIpAddresses(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointIpAddressesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverEndpointIpAddressesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverEndpointIpAddresses
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpointIpAddresses"
     *      target="_top">AWS API Documentation</a>
     */
    public ListResolverEndpointIpAddressesPublisher listResolverEndpointIpAddressesPaginator(
            ListResolverEndpointIpAddressesRequest listResolverEndpointIpAddressesRequest) {
        return new ListResolverEndpointIpAddressesPublisher(this, applyPaginatorUserAgent(listResolverEndpointIpAddressesRequest));
    }

    /**
     * <p>
     * Lists all the Resolver endpoints that were created using the current AWS account.
     * </p>
     *
     * @param listResolverEndpointsRequest
     * @return A Java Future containing the result of the ListResolverEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverEndpoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResolverEndpointsResponse> listResolverEndpoints(
            ListResolverEndpointsRequest listResolverEndpointsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResolverEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolverEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResolverEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResolverEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResolverEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResolverEndpointsRequest, ListResolverEndpointsResponse>()
                            .withOperationName("ListResolverEndpoints")
                            .withMarshaller(new ListResolverEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResolverEndpointsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listResolverEndpointsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListResolverEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the Resolver endpoints that were created using the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverEndpoints(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointsPublisher publisher = client.listResolverEndpointsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverEndpointsPublisher publisher = client.listResolverEndpointsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverEndpoints(software.amazon.awssdk.services.route53resolver.model.ListResolverEndpointsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverEndpointsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverEndpoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    public ListResolverEndpointsPublisher listResolverEndpointsPaginator(ListResolverEndpointsRequest listResolverEndpointsRequest) {
        return new ListResolverEndpointsPublisher(this, applyPaginatorUserAgent(listResolverEndpointsRequest));
    }

    /**
     * <p>
     * Lists information about associations between Amazon VPCs and query logging configurations.
     * </p>
     *
     * @param listResolverQueryLogConfigAssociationsRequest
     * @return A Java Future containing the result of the ListResolverQueryLogConfigAssociations operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverQueryLogConfigAssociations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverQueryLogConfigAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResolverQueryLogConfigAssociationsResponse> listResolverQueryLogConfigAssociations(
            ListResolverQueryLogConfigAssociationsRequest listResolverQueryLogConfigAssociationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listResolverQueryLogConfigAssociationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolverQueryLogConfigAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResolverQueryLogConfigAssociationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListResolverQueryLogConfigAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResolverQueryLogConfigAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResolverQueryLogConfigAssociationsRequest, ListResolverQueryLogConfigAssociationsResponse>()
                            .withOperationName("ListResolverQueryLogConfigAssociations")
                            .withMarshaller(new ListResolverQueryLogConfigAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResolverQueryLogConfigAssociationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listResolverQueryLogConfigAssociationsRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListResolverQueryLogConfigAssociationsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists information about associations between Amazon VPCs and query logging configurations.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverQueryLogConfigAssociations(software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigAssociationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverQueryLogConfigAssociationsPublisher publisher = client.listResolverQueryLogConfigAssociationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverQueryLogConfigAssociationsPublisher publisher = client.listResolverQueryLogConfigAssociationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigAssociationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigAssociationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverQueryLogConfigAssociations(software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigAssociationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverQueryLogConfigAssociationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverQueryLogConfigAssociations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverQueryLogConfigAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListResolverQueryLogConfigAssociationsPublisher listResolverQueryLogConfigAssociationsPaginator(
            ListResolverQueryLogConfigAssociationsRequest listResolverQueryLogConfigAssociationsRequest) {
        return new ListResolverQueryLogConfigAssociationsPublisher(this,
                applyPaginatorUserAgent(listResolverQueryLogConfigAssociationsRequest));
    }

    /**
     * <p>
     * Lists information about the specified query logging configurations. Each configuration defines where you want
     * Resolver to save DNS query logs and specifies the VPCs that you want to log queries for.
     * </p>
     *
     * @param listResolverQueryLogConfigsRequest
     * @return A Java Future containing the result of the ListResolverQueryLogConfigs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverQueryLogConfigs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverQueryLogConfigs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResolverQueryLogConfigsResponse> listResolverQueryLogConfigs(
            ListResolverQueryLogConfigsRequest listResolverQueryLogConfigsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResolverQueryLogConfigsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolverQueryLogConfigs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResolverQueryLogConfigsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResolverQueryLogConfigsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResolverQueryLogConfigsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResolverQueryLogConfigsRequest, ListResolverQueryLogConfigsResponse>()
                            .withOperationName("ListResolverQueryLogConfigs")
                            .withMarshaller(new ListResolverQueryLogConfigsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResolverQueryLogConfigsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listResolverQueryLogConfigsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListResolverQueryLogConfigsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists information about the specified query logging configurations. Each configuration defines where you want
     * Resolver to save DNS query logs and specifies the VPCs that you want to log queries for.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverQueryLogConfigs(software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverQueryLogConfigsPublisher publisher = client.listResolverQueryLogConfigsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverQueryLogConfigsPublisher publisher = client.listResolverQueryLogConfigsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverQueryLogConfigs(software.amazon.awssdk.services.route53resolver.model.ListResolverQueryLogConfigsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverQueryLogConfigsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverQueryLogConfigs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverQueryLogConfigs"
     *      target="_top">AWS API Documentation</a>
     */
    public ListResolverQueryLogConfigsPublisher listResolverQueryLogConfigsPaginator(
            ListResolverQueryLogConfigsRequest listResolverQueryLogConfigsRequest) {
        return new ListResolverQueryLogConfigsPublisher(this, applyPaginatorUserAgent(listResolverQueryLogConfigsRequest));
    }

    /**
     * <p>
     * Lists the associations that were created between Resolver rules and VPCs using the current AWS account.
     * </p>
     *
     * @param listResolverRuleAssociationsRequest
     * @return A Java Future containing the result of the ListResolverRuleAssociations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverRuleAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRuleAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResolverRuleAssociationsResponse> listResolverRuleAssociations(
            ListResolverRuleAssociationsRequest listResolverRuleAssociationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResolverRuleAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolverRuleAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResolverRuleAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResolverRuleAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResolverRuleAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResolverRuleAssociationsRequest, ListResolverRuleAssociationsResponse>()
                            .withOperationName("ListResolverRuleAssociations")
                            .withMarshaller(new ListResolverRuleAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResolverRuleAssociationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listResolverRuleAssociationsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListResolverRuleAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the associations that were created between Resolver rules and VPCs using the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverRuleAssociations(software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRuleAssociationsPublisher publisher = client.listResolverRuleAssociationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRuleAssociationsPublisher publisher = client.listResolverRuleAssociationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverRuleAssociations(software.amazon.awssdk.services.route53resolver.model.ListResolverRuleAssociationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverRuleAssociationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverRuleAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRuleAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListResolverRuleAssociationsPublisher listResolverRuleAssociationsPaginator(
            ListResolverRuleAssociationsRequest listResolverRuleAssociationsRequest) {
        return new ListResolverRuleAssociationsPublisher(this, applyPaginatorUserAgent(listResolverRuleAssociationsRequest));
    }

    /**
     * <p>
     * Lists the Resolver rules that were created using the current AWS account.
     * </p>
     *
     * @param listResolverRulesRequest
     * @return A Java Future containing the result of the ListResolverRules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResolverRulesResponse> listResolverRules(ListResolverRulesRequest listResolverRulesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResolverRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResolverRules");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResolverRulesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResolverRulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResolverRulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResolverRulesRequest, ListResolverRulesResponse>()
                            .withOperationName("ListResolverRules")
                            .withMarshaller(new ListResolverRulesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResolverRulesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listResolverRulesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListResolverRulesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Resolver rules that were created using the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResolverRules(software.amazon.awssdk.services.route53resolver.model.ListResolverRulesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRulesPublisher publisher = client.listResolverRulesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListResolverRulesPublisher publisher = client.listResolverRulesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.route53resolver.model.ListResolverRulesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.route53resolver.model.ListResolverRulesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResolverRules(software.amazon.awssdk.services.route53resolver.model.ListResolverRulesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResolverRulesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListResolverRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListResolverRules"
     *      target="_top">AWS API Documentation</a>
     */
    public ListResolverRulesPublisher listResolverRulesPaginator(ListResolverRulesRequest listResolverRulesRequest) {
        return new ListResolverRulesPublisher(this, applyPaginatorUserAgent(listResolverRulesRequest));
    }

    /**
     * <p>
     * Lists the tags that you associated with the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags that you associated with the specified resource.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.route53resolver.model.ListTagsForResourceRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53resolver.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.route53resolver.model.ListTagsForResourceResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.route53resolver.model.ListTagsForResourceResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.route53resolver.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidNextTokenException The value that you specified for <code>NextToken</code> in a
     *         <code>List</code> request isn't valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    public ListTagsForResourcePublisher listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest) {
        return new ListTagsForResourcePublisher(this, applyPaginatorUserAgent(listTagsForResourceRequest));
    }

    /**
     * <p>
     * Specifies an AWS account that you want to share a query logging configuration with, the query logging
     * configuration that you want to share, and the operations that you want the account to be able to perform on the
     * configuration.
     * </p>
     *
     * @param putResolverQueryLogConfigPolicyRequest
     * @return A Java Future containing the result of the PutResolverQueryLogConfigPolicy operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPolicyDocumentException The specified Resolver rule policy is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>UnknownResourceException The specified resource doesn't exist.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.PutResolverQueryLogConfigPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/PutResolverQueryLogConfigPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutResolverQueryLogConfigPolicyResponse> putResolverQueryLogConfigPolicy(
            PutResolverQueryLogConfigPolicyRequest putResolverQueryLogConfigPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putResolverQueryLogConfigPolicyRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResolverQueryLogConfigPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutResolverQueryLogConfigPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutResolverQueryLogConfigPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutResolverQueryLogConfigPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutResolverQueryLogConfigPolicyRequest, PutResolverQueryLogConfigPolicyResponse>()
                            .withOperationName("PutResolverQueryLogConfigPolicy")
                            .withMarshaller(new PutResolverQueryLogConfigPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putResolverQueryLogConfigPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putResolverQueryLogConfigPolicyRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<PutResolverQueryLogConfigPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Specifies an AWS rule that you want to share with another account, the account that you want to share the rule
     * with, and the operations that you want the account to be able to perform on the rule.
     * </p>
     *
     * @param putResolverRulePolicyRequest
     * @return A Java Future containing the result of the PutResolverRulePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPolicyDocumentException The specified Resolver rule policy is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>UnknownResourceException The specified resource doesn't exist.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.PutResolverRulePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/PutResolverRulePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutResolverRulePolicyResponse> putResolverRulePolicy(
            PutResolverRulePolicyRequest putResolverRulePolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResolverRulePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResolverRulePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutResolverRulePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutResolverRulePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutResolverRulePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutResolverRulePolicyRequest, PutResolverRulePolicyResponse>()
                            .withOperationName("PutResolverRulePolicy")
                            .withMarshaller(new PutResolverRulePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putResolverRulePolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putResolverRulePolicyRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<PutResolverRulePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to a specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidTagException The specified tag is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from a specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing DNSSEC validation configuration. If there is no existing DNSSEC validation configuration, one
     * is created.
     * </p>
     *
     * @param updateResolverDnssecConfigRequest
     * @return A Java Future containing the result of the UpdateResolverDnssecConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>AccessDeniedException The current account doesn't have the IAM permissions required to perform the
     *         specified Resolver operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.UpdateResolverDnssecConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/UpdateResolverDnssecConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateResolverDnssecConfigResponse> updateResolverDnssecConfig(
            UpdateResolverDnssecConfigRequest updateResolverDnssecConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResolverDnssecConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResolverDnssecConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateResolverDnssecConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateResolverDnssecConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateResolverDnssecConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateResolverDnssecConfigRequest, UpdateResolverDnssecConfigResponse>()
                            .withOperationName("UpdateResolverDnssecConfig")
                            .withMarshaller(new UpdateResolverDnssecConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateResolverDnssecConfigRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateResolverDnssecConfigRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateResolverDnssecConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the name of an inbound or an outbound Resolver endpoint.
     * </p>
     *
     * @param updateResolverEndpointRequest
     * @return A Java Future containing the result of the UpdateResolverEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.UpdateResolverEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/UpdateResolverEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateResolverEndpointResponse> updateResolverEndpoint(
            UpdateResolverEndpointRequest updateResolverEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResolverEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResolverEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateResolverEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateResolverEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateResolverEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateResolverEndpointRequest, UpdateResolverEndpointResponse>()
                            .withOperationName("UpdateResolverEndpoint")
                            .withMarshaller(new UpdateResolverEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateResolverEndpointRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateResolverEndpointRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdateResolverEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates settings for a specified Resolver rule. <code>ResolverRuleId</code> is required, and all other parameters
     * are optional. If you don't specify a parameter, it retains its current value.
     * </p>
     *
     * @param updateResolverRuleRequest
     * @return A Java Future containing the result of the UpdateResolverRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is invalid.</li>
     *         <li>InvalidParameterException One or more parameters in this request are not valid.</li>
     *         <li>ResourceNotFoundException The specified resource doesn't exist.</li>
     *         <li>ResourceUnavailableException The specified resource isn't available.</li>
     *         <li>LimitExceededException The request caused one or more limits to be exceeded.</li>
     *         <li>InternalServiceErrorException We encountered an unknown error. Try again in a few minutes.</li>
     *         <li>ThrottlingException The request was throttled. Try again in a few minutes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Route53ResolverException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample Route53ResolverAsyncClient.UpdateResolverRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/route53resolver-2018-04-01/UpdateResolverRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateResolverRuleResponse> updateResolverRule(UpdateResolverRuleRequest updateResolverRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResolverRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53Resolver");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResolverRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateResolverRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateResolverRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateResolverRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateResolverRuleRequest, UpdateResolverRuleResponse>()
                            .withOperationName("UpdateResolverRule")
                            .withMarshaller(new UpdateResolverRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateResolverRuleRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateResolverRuleRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateResolverRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Route53ResolverException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPolicyDocument")
                                .exceptionBuilderSupplier(InvalidPolicyDocumentException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagException")
                                .exceptionBuilderSupplier(InvalidTagException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnknownResourceException")
                                .exceptionBuilderSupplier(UnknownResourceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceUnavailableException")
                                .exceptionBuilderSupplier(ResourceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceExistsException")
                                .exceptionBuilderSupplier(ResourceExistsException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends Route53ResolverRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
