/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a connection endpoint.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Endpoint implements SdkPojo, Serializable, ToCopyableBuilder<Endpoint.Builder, Endpoint> {
    private static final SdkField<String> ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Address")
            .getter(getter(Endpoint::address)).setter(setter(Builder::address))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Address").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Port")
            .getter(getter(Endpoint::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<List<VpcEndpoint>> VPC_ENDPOINTS_FIELD = SdkField
            .<List<VpcEndpoint>> builder(MarshallingType.LIST)
            .memberName("VpcEndpoints")
            .getter(getter(Endpoint::vpcEndpoints))
            .setter(setter(Builder::vpcEndpoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcEndpoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("VpcEndpoint")
                            .memberFieldInfo(
                                    SdkField.<VpcEndpoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(VpcEndpoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("VpcEndpoint").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ADDRESS_FIELD, PORT_FIELD,
            VPC_ENDPOINTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String address;

    private final Integer port;

    private final List<VpcEndpoint> vpcEndpoints;

    private Endpoint(BuilderImpl builder) {
        this.address = builder.address;
        this.port = builder.port;
        this.vpcEndpoints = builder.vpcEndpoints;
    }

    /**
     * <p>
     * The DNS address of the Cluster.
     * </p>
     * 
     * @return The DNS address of the Cluster.
     */
    public final String address() {
        return address;
    }

    /**
     * <p>
     * The port that the database engine is listening on.
     * </p>
     * 
     * @return The port that the database engine is listening on.
     */
    public final Integer port() {
        return port;
    }

    /**
     * Returns true if the VpcEndpoints property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasVpcEndpoints() {
        return vpcEndpoints != null && !(vpcEndpoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Describes a connection endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasVpcEndpoints()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Describes a connection endpoint.
     */
    public final List<VpcEndpoint> vpcEndpoints() {
        return vpcEndpoints;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(address());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(hasVpcEndpoints() ? vpcEndpoints() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Endpoint)) {
            return false;
        }
        Endpoint other = (Endpoint) obj;
        return Objects.equals(address(), other.address()) && Objects.equals(port(), other.port())
                && hasVpcEndpoints() == other.hasVpcEndpoints() && Objects.equals(vpcEndpoints(), other.vpcEndpoints());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Endpoint").add("Address", address()).add("Port", port())
                .add("VpcEndpoints", hasVpcEndpoints() ? vpcEndpoints() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Address":
            return Optional.ofNullable(clazz.cast(address()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "VpcEndpoints":
            return Optional.ofNullable(clazz.cast(vpcEndpoints()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Endpoint, T> g) {
        return obj -> g.apply((Endpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Endpoint> {
        /**
         * <p>
         * The DNS address of the Cluster.
         * </p>
         * 
         * @param address
         *        The DNS address of the Cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder address(String address);

        /**
         * <p>
         * The port that the database engine is listening on.
         * </p>
         * 
         * @param port
         *        The port that the database engine is listening on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * Describes a connection endpoint.
         * </p>
         * 
         * @param vpcEndpoints
         *        Describes a connection endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcEndpoints(Collection<VpcEndpoint> vpcEndpoints);

        /**
         * <p>
         * Describes a connection endpoint.
         * </p>
         * 
         * @param vpcEndpoints
         *        Describes a connection endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcEndpoints(VpcEndpoint... vpcEndpoints);

        /**
         * <p>
         * Describes a connection endpoint.
         * </p>
         * This is a convenience that creates an instance of the {@link List<VpcEndpoint>.Builder} avoiding the need to
         * create one manually via {@link List<VpcEndpoint>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<VpcEndpoint>.Builder#build()} is called immediately and its
         * result is passed to {@link #vpcEndpoints(List<VpcEndpoint>)}.
         * 
         * @param vpcEndpoints
         *        a consumer that will call methods on {@link List<VpcEndpoint>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcEndpoints(List<VpcEndpoint>)
         */
        Builder vpcEndpoints(Consumer<VpcEndpoint.Builder>... vpcEndpoints);
    }

    static final class BuilderImpl implements Builder {
        private String address;

        private Integer port;

        private List<VpcEndpoint> vpcEndpoints = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Endpoint model) {
            address(model.address);
            port(model.port);
            vpcEndpoints(model.vpcEndpoints);
        }

        public final String getAddress() {
            return address;
        }

        @Override
        public final Builder address(String address) {
            this.address = address;
            return this;
        }

        public final void setAddress(String address) {
            this.address = address;
        }

        public final Integer getPort() {
            return port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        public final Collection<VpcEndpoint.Builder> getVpcEndpoints() {
            if (vpcEndpoints instanceof SdkAutoConstructList) {
                return null;
            }
            return vpcEndpoints != null ? vpcEndpoints.stream().map(VpcEndpoint::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder vpcEndpoints(Collection<VpcEndpoint> vpcEndpoints) {
            this.vpcEndpoints = VpcEndpointsListCopier.copy(vpcEndpoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vpcEndpoints(VpcEndpoint... vpcEndpoints) {
            vpcEndpoints(Arrays.asList(vpcEndpoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vpcEndpoints(Consumer<VpcEndpoint.Builder>... vpcEndpoints) {
            vpcEndpoints(Stream.of(vpcEndpoints).map(c -> VpcEndpoint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setVpcEndpoints(Collection<VpcEndpoint.BuilderImpl> vpcEndpoints) {
            this.vpcEndpoints = VpcEndpointsListCopier.copyFromBuilder(vpcEndpoints);
        }

        @Override
        public Endpoint build() {
            return new Endpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
