/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains a list of available options for a DB instance.
 * </p>
 * <p>
 * This data type is used as a response element in the <code>DescribeOrderableDBInstanceOptions</code> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OrderableDBInstanceOption implements SdkPojo, Serializable,
        ToCopyableBuilder<OrderableDBInstanceOption.Builder, OrderableDBInstanceOption> {
    private static final SdkField<String> ENGINE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(OrderableDBInstanceOption::engine)).setter(setter(Builder::engine))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Engine").build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(OrderableDBInstanceOption::engineVersion)).setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<String> DB_INSTANCE_CLASS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(OrderableDBInstanceOption::dbInstanceClass)).setter(setter(Builder::dbInstanceClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBInstanceClass").build()).build();

    private static final SdkField<String> LICENSE_MODEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(OrderableDBInstanceOption::licenseModel)).setter(setter(Builder::licenseModel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseModel").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(OrderableDBInstanceOption::availabilityZoneGroup)).setter(setter(Builder::availabilityZoneGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZoneGroup").build())
            .build();

    private static final SdkField<List<AvailabilityZone>> AVAILABILITY_ZONES_FIELD = SdkField
            .<List<AvailabilityZone>> builder(MarshallingType.LIST)
            .getter(getter(OrderableDBInstanceOption::availabilityZones))
            .setter(setter(Builder::availabilityZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("AvailabilityZone")
                            .memberFieldInfo(
                                    SdkField.<AvailabilityZone> builder(MarshallingType.SDK_POJO)
                                            .constructor(AvailabilityZone::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("AvailabilityZone").build()).build()).build()).build();

    private static final SdkField<Boolean> MULTI_AZ_CAPABLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(OrderableDBInstanceOption::multiAZCapable)).setter(setter(Builder::multiAZCapable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MultiAZCapable").build()).build();

    private static final SdkField<Boolean> READ_REPLICA_CAPABLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(OrderableDBInstanceOption::readReplicaCapable)).setter(setter(Builder::readReplicaCapable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReadReplicaCapable").build())
            .build();

    private static final SdkField<Boolean> VPC_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(OrderableDBInstanceOption::vpc)).setter(setter(Builder::vpc))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Vpc").build()).build();

    private static final SdkField<Boolean> SUPPORTS_STORAGE_ENCRYPTION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).getter(getter(OrderableDBInstanceOption::supportsStorageEncryption))
            .setter(setter(Builder::supportsStorageEncryption))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsStorageEncryption").build())
            .build();

    private static final SdkField<String> STORAGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(OrderableDBInstanceOption::storageType)).setter(setter(Builder::storageType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageType").build()).build();

    private static final SdkField<Boolean> SUPPORTS_IOPS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(OrderableDBInstanceOption::supportsIops)).setter(setter(Builder::supportsIops))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsIops").build()).build();

    private static final SdkField<Boolean> SUPPORTS_ENHANCED_MONITORING_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(OrderableDBInstanceOption::supportsEnhancedMonitoring))
            .setter(setter(Builder::supportsEnhancedMonitoring))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsEnhancedMonitoring").build())
            .build();

    private static final SdkField<Boolean> SUPPORTS_IAM_DATABASE_AUTHENTICATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(OrderableDBInstanceOption::supportsIAMDatabaseAuthentication))
            .setter(setter(Builder::supportsIAMDatabaseAuthentication))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsIAMDatabaseAuthentication")
                    .build()).build();

    private static final SdkField<Boolean> SUPPORTS_PERFORMANCE_INSIGHTS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(OrderableDBInstanceOption::supportsPerformanceInsights))
            .setter(setter(Builder::supportsPerformanceInsights))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsPerformanceInsights")
                    .build()).build();

    private static final SdkField<Integer> MIN_STORAGE_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(OrderableDBInstanceOption::minStorageSize)).setter(setter(Builder::minStorageSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinStorageSize").build()).build();

    private static final SdkField<Integer> MAX_STORAGE_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(OrderableDBInstanceOption::maxStorageSize)).setter(setter(Builder::maxStorageSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxStorageSize").build()).build();

    private static final SdkField<Integer> MIN_IOPS_PER_DB_INSTANCE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(OrderableDBInstanceOption::minIopsPerDbInstance)).setter(setter(Builder::minIopsPerDbInstance))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinIopsPerDbInstance").build())
            .build();

    private static final SdkField<Integer> MAX_IOPS_PER_DB_INSTANCE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(OrderableDBInstanceOption::maxIopsPerDbInstance)).setter(setter(Builder::maxIopsPerDbInstance))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxIopsPerDbInstance").build())
            .build();

    private static final SdkField<Double> MIN_IOPS_PER_GIB_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(OrderableDBInstanceOption::minIopsPerGib)).setter(setter(Builder::minIopsPerGib))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinIopsPerGib").build()).build();

    private static final SdkField<Double> MAX_IOPS_PER_GIB_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(OrderableDBInstanceOption::maxIopsPerGib)).setter(setter(Builder::maxIopsPerGib))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxIopsPerGib").build()).build();

    private static final SdkField<List<AvailableProcessorFeature>> AVAILABLE_PROCESSOR_FEATURES_FIELD = SdkField
            .<List<AvailableProcessorFeature>> builder(MarshallingType.LIST)
            .getter(getter(OrderableDBInstanceOption::availableProcessorFeatures))
            .setter(setter(Builder::availableProcessorFeatures))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailableProcessorFeatures").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("AvailableProcessorFeature")
                            .memberFieldInfo(
                                    SdkField.<AvailableProcessorFeature> builder(MarshallingType.SDK_POJO)
                                            .constructor(AvailableProcessorFeature::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("AvailableProcessorFeature").build()).build()).build()).build();

    private static final SdkField<List<String>> SUPPORTED_ENGINE_MODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(OrderableDBInstanceOption::supportedEngineModes))
            .setter(setter(Builder::supportedEngineModes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedEngineModes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> SUPPORTS_STORAGE_AUTOSCALING_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(OrderableDBInstanceOption::supportsStorageAutoscaling))
            .setter(setter(Builder::supportsStorageAutoscaling))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsStorageAutoscaling").build())
            .build();

    private static final SdkField<Boolean> SUPPORTS_KERBEROS_AUTHENTICATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(OrderableDBInstanceOption::supportsKerberosAuthentication))
            .setter(setter(Builder::supportsKerberosAuthentication))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsKerberosAuthentication")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENGINE_FIELD,
            ENGINE_VERSION_FIELD, DB_INSTANCE_CLASS_FIELD, LICENSE_MODEL_FIELD, AVAILABILITY_ZONE_GROUP_FIELD,
            AVAILABILITY_ZONES_FIELD, MULTI_AZ_CAPABLE_FIELD, READ_REPLICA_CAPABLE_FIELD, VPC_FIELD,
            SUPPORTS_STORAGE_ENCRYPTION_FIELD, STORAGE_TYPE_FIELD, SUPPORTS_IOPS_FIELD, SUPPORTS_ENHANCED_MONITORING_FIELD,
            SUPPORTS_IAM_DATABASE_AUTHENTICATION_FIELD, SUPPORTS_PERFORMANCE_INSIGHTS_FIELD, MIN_STORAGE_SIZE_FIELD,
            MAX_STORAGE_SIZE_FIELD, MIN_IOPS_PER_DB_INSTANCE_FIELD, MAX_IOPS_PER_DB_INSTANCE_FIELD, MIN_IOPS_PER_GIB_FIELD,
            MAX_IOPS_PER_GIB_FIELD, AVAILABLE_PROCESSOR_FEATURES_FIELD, SUPPORTED_ENGINE_MODES_FIELD,
            SUPPORTS_STORAGE_AUTOSCALING_FIELD, SUPPORTS_KERBEROS_AUTHENTICATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String engine;

    private final String engineVersion;

    private final String dbInstanceClass;

    private final String licenseModel;

    private final String availabilityZoneGroup;

    private final List<AvailabilityZone> availabilityZones;

    private final Boolean multiAZCapable;

    private final Boolean readReplicaCapable;

    private final Boolean vpc;

    private final Boolean supportsStorageEncryption;

    private final String storageType;

    private final Boolean supportsIops;

    private final Boolean supportsEnhancedMonitoring;

    private final Boolean supportsIAMDatabaseAuthentication;

    private final Boolean supportsPerformanceInsights;

    private final Integer minStorageSize;

    private final Integer maxStorageSize;

    private final Integer minIopsPerDbInstance;

    private final Integer maxIopsPerDbInstance;

    private final Double minIopsPerGib;

    private final Double maxIopsPerGib;

    private final List<AvailableProcessorFeature> availableProcessorFeatures;

    private final List<String> supportedEngineModes;

    private final Boolean supportsStorageAutoscaling;

    private final Boolean supportsKerberosAuthentication;

    private OrderableDBInstanceOption(BuilderImpl builder) {
        this.engine = builder.engine;
        this.engineVersion = builder.engineVersion;
        this.dbInstanceClass = builder.dbInstanceClass;
        this.licenseModel = builder.licenseModel;
        this.availabilityZoneGroup = builder.availabilityZoneGroup;
        this.availabilityZones = builder.availabilityZones;
        this.multiAZCapable = builder.multiAZCapable;
        this.readReplicaCapable = builder.readReplicaCapable;
        this.vpc = builder.vpc;
        this.supportsStorageEncryption = builder.supportsStorageEncryption;
        this.storageType = builder.storageType;
        this.supportsIops = builder.supportsIops;
        this.supportsEnhancedMonitoring = builder.supportsEnhancedMonitoring;
        this.supportsIAMDatabaseAuthentication = builder.supportsIAMDatabaseAuthentication;
        this.supportsPerformanceInsights = builder.supportsPerformanceInsights;
        this.minStorageSize = builder.minStorageSize;
        this.maxStorageSize = builder.maxStorageSize;
        this.minIopsPerDbInstance = builder.minIopsPerDbInstance;
        this.maxIopsPerDbInstance = builder.maxIopsPerDbInstance;
        this.minIopsPerGib = builder.minIopsPerGib;
        this.maxIopsPerGib = builder.maxIopsPerGib;
        this.availableProcessorFeatures = builder.availableProcessorFeatures;
        this.supportedEngineModes = builder.supportedEngineModes;
        this.supportsStorageAutoscaling = builder.supportsStorageAutoscaling;
        this.supportsKerberosAuthentication = builder.supportsKerberosAuthentication;
    }

    /**
     * <p>
     * The engine type of a DB instance.
     * </p>
     * 
     * @return The engine type of a DB instance.
     */
    public String engine() {
        return engine;
    }

    /**
     * <p>
     * The engine version of a DB instance.
     * </p>
     * 
     * @return The engine version of a DB instance.
     */
    public String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * The DB instance class for a DB instance.
     * </p>
     * 
     * @return The DB instance class for a DB instance.
     */
    public String dbInstanceClass() {
        return dbInstanceClass;
    }

    /**
     * <p>
     * The license model for a DB instance.
     * </p>
     * 
     * @return The license model for a DB instance.
     */
    public String licenseModel() {
        return licenseModel;
    }

    /**
     * <p>
     * The Availability Zone group for a DB instance.
     * </p>
     * 
     * @return The Availability Zone group for a DB instance.
     */
    public String availabilityZoneGroup() {
        return availabilityZoneGroup;
    }

    /**
     * Returns true if the AvailabilityZones property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasAvailabilityZones() {
        return availabilityZones != null && !(availabilityZones instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of Availability Zones for a DB instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAvailabilityZones()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of Availability Zones for a DB instance.
     */
    public List<AvailabilityZone> availabilityZones() {
        return availabilityZones;
    }

    /**
     * <p>
     * Indicates whether a DB instance is Multi-AZ capable.
     * </p>
     * 
     * @return Indicates whether a DB instance is Multi-AZ capable.
     */
    public Boolean multiAZCapable() {
        return multiAZCapable;
    }

    /**
     * <p>
     * Indicates whether a DB instance can have a read replica.
     * </p>
     * 
     * @return Indicates whether a DB instance can have a read replica.
     */
    public Boolean readReplicaCapable() {
        return readReplicaCapable;
    }

    /**
     * <p>
     * Indicates whether a DB instance is in a VPC.
     * </p>
     * 
     * @return Indicates whether a DB instance is in a VPC.
     */
    public Boolean vpc() {
        return vpc;
    }

    /**
     * <p>
     * Indicates whether a DB instance supports encrypted storage.
     * </p>
     * 
     * @return Indicates whether a DB instance supports encrypted storage.
     */
    public Boolean supportsStorageEncryption() {
        return supportsStorageEncryption;
    }

    /**
     * <p>
     * Indicates the storage type for a DB instance.
     * </p>
     * 
     * @return Indicates the storage type for a DB instance.
     */
    public String storageType() {
        return storageType;
    }

    /**
     * <p>
     * Indicates whether a DB instance supports provisioned IOPS.
     * </p>
     * 
     * @return Indicates whether a DB instance supports provisioned IOPS.
     */
    public Boolean supportsIops() {
        return supportsIops;
    }

    /**
     * <p>
     * Indicates whether a DB instance supports Enhanced Monitoring at intervals from 1 to 60 seconds.
     * </p>
     * 
     * @return Indicates whether a DB instance supports Enhanced Monitoring at intervals from 1 to 60 seconds.
     */
    public Boolean supportsEnhancedMonitoring() {
        return supportsEnhancedMonitoring;
    }

    /**
     * <p>
     * Indicates whether a DB instance supports IAM database authentication.
     * </p>
     * 
     * @return Indicates whether a DB instance supports IAM database authentication.
     */
    public Boolean supportsIAMDatabaseAuthentication() {
        return supportsIAMDatabaseAuthentication;
    }

    /**
     * <p>
     * True if a DB instance supports Performance Insights, otherwise false.
     * </p>
     * 
     * @return True if a DB instance supports Performance Insights, otherwise false.
     */
    public Boolean supportsPerformanceInsights() {
        return supportsPerformanceInsights;
    }

    /**
     * <p>
     * Minimum storage size for a DB instance.
     * </p>
     * 
     * @return Minimum storage size for a DB instance.
     */
    public Integer minStorageSize() {
        return minStorageSize;
    }

    /**
     * <p>
     * Maximum storage size for a DB instance.
     * </p>
     * 
     * @return Maximum storage size for a DB instance.
     */
    public Integer maxStorageSize() {
        return maxStorageSize;
    }

    /**
     * <p>
     * Minimum total provisioned IOPS for a DB instance.
     * </p>
     * 
     * @return Minimum total provisioned IOPS for a DB instance.
     */
    public Integer minIopsPerDbInstance() {
        return minIopsPerDbInstance;
    }

    /**
     * <p>
     * Maximum total provisioned IOPS for a DB instance.
     * </p>
     * 
     * @return Maximum total provisioned IOPS for a DB instance.
     */
    public Integer maxIopsPerDbInstance() {
        return maxIopsPerDbInstance;
    }

    /**
     * <p>
     * Minimum provisioned IOPS per GiB for a DB instance.
     * </p>
     * 
     * @return Minimum provisioned IOPS per GiB for a DB instance.
     */
    public Double minIopsPerGib() {
        return minIopsPerGib;
    }

    /**
     * <p>
     * Maximum provisioned IOPS per GiB for a DB instance.
     * </p>
     * 
     * @return Maximum provisioned IOPS per GiB for a DB instance.
     */
    public Double maxIopsPerGib() {
        return maxIopsPerGib;
    }

    /**
     * Returns true if the AvailableProcessorFeatures property was specified by the sender (it may be empty), or false
     * if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the
     * AWS service.
     */
    public boolean hasAvailableProcessorFeatures() {
        return availableProcessorFeatures != null && !(availableProcessorFeatures instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the available processor features for the DB instance class of a DB instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAvailableProcessorFeatures()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the available processor features for the DB instance class of a DB instance.
     */
    public List<AvailableProcessorFeature> availableProcessorFeatures() {
        return availableProcessorFeatures;
    }

    /**
     * Returns true if the SupportedEngineModes property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasSupportedEngineModes() {
        return supportedEngineModes != null && !(supportedEngineModes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the supported DB engine modes.
     * </p>
     * <note>
     * <p>
     * <code>global</code> engine mode only applies for global database clusters created with Aurora MySQL version
     * 5.6.10a. For higher Aurora MySQL versions, the clusters in a global database use <code>provisioned</code> engine
     * mode.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSupportedEngineModes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the supported DB engine modes.</p> <note>
     *         <p>
     *         <code>global</code> engine mode only applies for global database clusters created with Aurora MySQL
     *         version 5.6.10a. For higher Aurora MySQL versions, the clusters in a global database use
     *         <code>provisioned</code> engine mode.
     *         </p>
     */
    public List<String> supportedEngineModes() {
        return supportedEngineModes;
    }

    /**
     * <p>
     * Whether Amazon RDS can automatically scale storage for DB instances that use the specified DB instance class.
     * </p>
     * 
     * @return Whether Amazon RDS can automatically scale storage for DB instances that use the specified DB instance
     *         class.
     */
    public Boolean supportsStorageAutoscaling() {
        return supportsStorageAutoscaling;
    }

    /**
     * <p>
     * Whether a DB instance supports Kerberos Authentication.
     * </p>
     * 
     * @return Whether a DB instance supports Kerberos Authentication.
     */
    public Boolean supportsKerberosAuthentication() {
        return supportsKerberosAuthentication;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(engine());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(dbInstanceClass());
        hashCode = 31 * hashCode + Objects.hashCode(licenseModel());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZoneGroup());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZones());
        hashCode = 31 * hashCode + Objects.hashCode(multiAZCapable());
        hashCode = 31 * hashCode + Objects.hashCode(readReplicaCapable());
        hashCode = 31 * hashCode + Objects.hashCode(vpc());
        hashCode = 31 * hashCode + Objects.hashCode(supportsStorageEncryption());
        hashCode = 31 * hashCode + Objects.hashCode(storageType());
        hashCode = 31 * hashCode + Objects.hashCode(supportsIops());
        hashCode = 31 * hashCode + Objects.hashCode(supportsEnhancedMonitoring());
        hashCode = 31 * hashCode + Objects.hashCode(supportsIAMDatabaseAuthentication());
        hashCode = 31 * hashCode + Objects.hashCode(supportsPerformanceInsights());
        hashCode = 31 * hashCode + Objects.hashCode(minStorageSize());
        hashCode = 31 * hashCode + Objects.hashCode(maxStorageSize());
        hashCode = 31 * hashCode + Objects.hashCode(minIopsPerDbInstance());
        hashCode = 31 * hashCode + Objects.hashCode(maxIopsPerDbInstance());
        hashCode = 31 * hashCode + Objects.hashCode(minIopsPerGib());
        hashCode = 31 * hashCode + Objects.hashCode(maxIopsPerGib());
        hashCode = 31 * hashCode + Objects.hashCode(availableProcessorFeatures());
        hashCode = 31 * hashCode + Objects.hashCode(supportedEngineModes());
        hashCode = 31 * hashCode + Objects.hashCode(supportsStorageAutoscaling());
        hashCode = 31 * hashCode + Objects.hashCode(supportsKerberosAuthentication());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OrderableDBInstanceOption)) {
            return false;
        }
        OrderableDBInstanceOption other = (OrderableDBInstanceOption) obj;
        return Objects.equals(engine(), other.engine()) && Objects.equals(engineVersion(), other.engineVersion())
                && Objects.equals(dbInstanceClass(), other.dbInstanceClass())
                && Objects.equals(licenseModel(), other.licenseModel())
                && Objects.equals(availabilityZoneGroup(), other.availabilityZoneGroup())
                && Objects.equals(availabilityZones(), other.availabilityZones())
                && Objects.equals(multiAZCapable(), other.multiAZCapable())
                && Objects.equals(readReplicaCapable(), other.readReplicaCapable()) && Objects.equals(vpc(), other.vpc())
                && Objects.equals(supportsStorageEncryption(), other.supportsStorageEncryption())
                && Objects.equals(storageType(), other.storageType()) && Objects.equals(supportsIops(), other.supportsIops())
                && Objects.equals(supportsEnhancedMonitoring(), other.supportsEnhancedMonitoring())
                && Objects.equals(supportsIAMDatabaseAuthentication(), other.supportsIAMDatabaseAuthentication())
                && Objects.equals(supportsPerformanceInsights(), other.supportsPerformanceInsights())
                && Objects.equals(minStorageSize(), other.minStorageSize())
                && Objects.equals(maxStorageSize(), other.maxStorageSize())
                && Objects.equals(minIopsPerDbInstance(), other.minIopsPerDbInstance())
                && Objects.equals(maxIopsPerDbInstance(), other.maxIopsPerDbInstance())
                && Objects.equals(minIopsPerGib(), other.minIopsPerGib())
                && Objects.equals(maxIopsPerGib(), other.maxIopsPerGib())
                && Objects.equals(availableProcessorFeatures(), other.availableProcessorFeatures())
                && Objects.equals(supportedEngineModes(), other.supportedEngineModes())
                && Objects.equals(supportsStorageAutoscaling(), other.supportsStorageAutoscaling())
                && Objects.equals(supportsKerberosAuthentication(), other.supportsKerberosAuthentication());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("OrderableDBInstanceOption").add("Engine", engine()).add("EngineVersion", engineVersion())
                .add("DBInstanceClass", dbInstanceClass()).add("LicenseModel", licenseModel())
                .add("AvailabilityZoneGroup", availabilityZoneGroup()).add("AvailabilityZones", availabilityZones())
                .add("MultiAZCapable", multiAZCapable()).add("ReadReplicaCapable", readReplicaCapable()).add("Vpc", vpc())
                .add("SupportsStorageEncryption", supportsStorageEncryption()).add("StorageType", storageType())
                .add("SupportsIops", supportsIops()).add("SupportsEnhancedMonitoring", supportsEnhancedMonitoring())
                .add("SupportsIAMDatabaseAuthentication", supportsIAMDatabaseAuthentication())
                .add("SupportsPerformanceInsights", supportsPerformanceInsights()).add("MinStorageSize", minStorageSize())
                .add("MaxStorageSize", maxStorageSize()).add("MinIopsPerDbInstance", minIopsPerDbInstance())
                .add("MaxIopsPerDbInstance", maxIopsPerDbInstance()).add("MinIopsPerGib", minIopsPerGib())
                .add("MaxIopsPerGib", maxIopsPerGib()).add("AvailableProcessorFeatures", availableProcessorFeatures())
                .add("SupportedEngineModes", supportedEngineModes())
                .add("SupportsStorageAutoscaling", supportsStorageAutoscaling())
                .add("SupportsKerberosAuthentication", supportsKerberosAuthentication()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Engine":
            return Optional.ofNullable(clazz.cast(engine()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "DBInstanceClass":
            return Optional.ofNullable(clazz.cast(dbInstanceClass()));
        case "LicenseModel":
            return Optional.ofNullable(clazz.cast(licenseModel()));
        case "AvailabilityZoneGroup":
            return Optional.ofNullable(clazz.cast(availabilityZoneGroup()));
        case "AvailabilityZones":
            return Optional.ofNullable(clazz.cast(availabilityZones()));
        case "MultiAZCapable":
            return Optional.ofNullable(clazz.cast(multiAZCapable()));
        case "ReadReplicaCapable":
            return Optional.ofNullable(clazz.cast(readReplicaCapable()));
        case "Vpc":
            return Optional.ofNullable(clazz.cast(vpc()));
        case "SupportsStorageEncryption":
            return Optional.ofNullable(clazz.cast(supportsStorageEncryption()));
        case "StorageType":
            return Optional.ofNullable(clazz.cast(storageType()));
        case "SupportsIops":
            return Optional.ofNullable(clazz.cast(supportsIops()));
        case "SupportsEnhancedMonitoring":
            return Optional.ofNullable(clazz.cast(supportsEnhancedMonitoring()));
        case "SupportsIAMDatabaseAuthentication":
            return Optional.ofNullable(clazz.cast(supportsIAMDatabaseAuthentication()));
        case "SupportsPerformanceInsights":
            return Optional.ofNullable(clazz.cast(supportsPerformanceInsights()));
        case "MinStorageSize":
            return Optional.ofNullable(clazz.cast(minStorageSize()));
        case "MaxStorageSize":
            return Optional.ofNullable(clazz.cast(maxStorageSize()));
        case "MinIopsPerDbInstance":
            return Optional.ofNullable(clazz.cast(minIopsPerDbInstance()));
        case "MaxIopsPerDbInstance":
            return Optional.ofNullable(clazz.cast(maxIopsPerDbInstance()));
        case "MinIopsPerGib":
            return Optional.ofNullable(clazz.cast(minIopsPerGib()));
        case "MaxIopsPerGib":
            return Optional.ofNullable(clazz.cast(maxIopsPerGib()));
        case "AvailableProcessorFeatures":
            return Optional.ofNullable(clazz.cast(availableProcessorFeatures()));
        case "SupportedEngineModes":
            return Optional.ofNullable(clazz.cast(supportedEngineModes()));
        case "SupportsStorageAutoscaling":
            return Optional.ofNullable(clazz.cast(supportsStorageAutoscaling()));
        case "SupportsKerberosAuthentication":
            return Optional.ofNullable(clazz.cast(supportsKerberosAuthentication()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OrderableDBInstanceOption, T> g) {
        return obj -> g.apply((OrderableDBInstanceOption) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OrderableDBInstanceOption> {
        /**
         * <p>
         * The engine type of a DB instance.
         * </p>
         * 
         * @param engine
         *        The engine type of a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engine(String engine);

        /**
         * <p>
         * The engine version of a DB instance.
         * </p>
         * 
         * @param engineVersion
         *        The engine version of a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * The DB instance class for a DB instance.
         * </p>
         * 
         * @param dbInstanceClass
         *        The DB instance class for a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbInstanceClass(String dbInstanceClass);

        /**
         * <p>
         * The license model for a DB instance.
         * </p>
         * 
         * @param licenseModel
         *        The license model for a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseModel(String licenseModel);

        /**
         * <p>
         * The Availability Zone group for a DB instance.
         * </p>
         * 
         * @param availabilityZoneGroup
         *        The Availability Zone group for a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZoneGroup(String availabilityZoneGroup);

        /**
         * <p>
         * A list of Availability Zones for a DB instance.
         * </p>
         * 
         * @param availabilityZones
         *        A list of Availability Zones for a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(Collection<AvailabilityZone> availabilityZones);

        /**
         * <p>
         * A list of Availability Zones for a DB instance.
         * </p>
         * 
         * @param availabilityZones
         *        A list of Availability Zones for a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(AvailabilityZone... availabilityZones);

        /**
         * <p>
         * A list of Availability Zones for a DB instance.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AvailabilityZone>.Builder} avoiding the
         * need to create one manually via {@link List<AvailabilityZone>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AvailabilityZone>.Builder#build()} is called immediately and
         * its result is passed to {@link #availabilityZones(List<AvailabilityZone>)}.
         * 
         * @param availabilityZones
         *        a consumer that will call methods on {@link List<AvailabilityZone>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #availabilityZones(List<AvailabilityZone>)
         */
        Builder availabilityZones(Consumer<AvailabilityZone.Builder>... availabilityZones);

        /**
         * <p>
         * Indicates whether a DB instance is Multi-AZ capable.
         * </p>
         * 
         * @param multiAZCapable
         *        Indicates whether a DB instance is Multi-AZ capable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder multiAZCapable(Boolean multiAZCapable);

        /**
         * <p>
         * Indicates whether a DB instance can have a read replica.
         * </p>
         * 
         * @param readReplicaCapable
         *        Indicates whether a DB instance can have a read replica.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readReplicaCapable(Boolean readReplicaCapable);

        /**
         * <p>
         * Indicates whether a DB instance is in a VPC.
         * </p>
         * 
         * @param vpc
         *        Indicates whether a DB instance is in a VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpc(Boolean vpc);

        /**
         * <p>
         * Indicates whether a DB instance supports encrypted storage.
         * </p>
         * 
         * @param supportsStorageEncryption
         *        Indicates whether a DB instance supports encrypted storage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsStorageEncryption(Boolean supportsStorageEncryption);

        /**
         * <p>
         * Indicates the storage type for a DB instance.
         * </p>
         * 
         * @param storageType
         *        Indicates the storage type for a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageType(String storageType);

        /**
         * <p>
         * Indicates whether a DB instance supports provisioned IOPS.
         * </p>
         * 
         * @param supportsIops
         *        Indicates whether a DB instance supports provisioned IOPS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsIops(Boolean supportsIops);

        /**
         * <p>
         * Indicates whether a DB instance supports Enhanced Monitoring at intervals from 1 to 60 seconds.
         * </p>
         * 
         * @param supportsEnhancedMonitoring
         *        Indicates whether a DB instance supports Enhanced Monitoring at intervals from 1 to 60 seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsEnhancedMonitoring(Boolean supportsEnhancedMonitoring);

        /**
         * <p>
         * Indicates whether a DB instance supports IAM database authentication.
         * </p>
         * 
         * @param supportsIAMDatabaseAuthentication
         *        Indicates whether a DB instance supports IAM database authentication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsIAMDatabaseAuthentication(Boolean supportsIAMDatabaseAuthentication);

        /**
         * <p>
         * True if a DB instance supports Performance Insights, otherwise false.
         * </p>
         * 
         * @param supportsPerformanceInsights
         *        True if a DB instance supports Performance Insights, otherwise false.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsPerformanceInsights(Boolean supportsPerformanceInsights);

        /**
         * <p>
         * Minimum storage size for a DB instance.
         * </p>
         * 
         * @param minStorageSize
         *        Minimum storage size for a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minStorageSize(Integer minStorageSize);

        /**
         * <p>
         * Maximum storage size for a DB instance.
         * </p>
         * 
         * @param maxStorageSize
         *        Maximum storage size for a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxStorageSize(Integer maxStorageSize);

        /**
         * <p>
         * Minimum total provisioned IOPS for a DB instance.
         * </p>
         * 
         * @param minIopsPerDbInstance
         *        Minimum total provisioned IOPS for a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minIopsPerDbInstance(Integer minIopsPerDbInstance);

        /**
         * <p>
         * Maximum total provisioned IOPS for a DB instance.
         * </p>
         * 
         * @param maxIopsPerDbInstance
         *        Maximum total provisioned IOPS for a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxIopsPerDbInstance(Integer maxIopsPerDbInstance);

        /**
         * <p>
         * Minimum provisioned IOPS per GiB for a DB instance.
         * </p>
         * 
         * @param minIopsPerGib
         *        Minimum provisioned IOPS per GiB for a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minIopsPerGib(Double minIopsPerGib);

        /**
         * <p>
         * Maximum provisioned IOPS per GiB for a DB instance.
         * </p>
         * 
         * @param maxIopsPerGib
         *        Maximum provisioned IOPS per GiB for a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxIopsPerGib(Double maxIopsPerGib);

        /**
         * <p>
         * A list of the available processor features for the DB instance class of a DB instance.
         * </p>
         * 
         * @param availableProcessorFeatures
         *        A list of the available processor features for the DB instance class of a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availableProcessorFeatures(Collection<AvailableProcessorFeature> availableProcessorFeatures);

        /**
         * <p>
         * A list of the available processor features for the DB instance class of a DB instance.
         * </p>
         * 
         * @param availableProcessorFeatures
         *        A list of the available processor features for the DB instance class of a DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availableProcessorFeatures(AvailableProcessorFeature... availableProcessorFeatures);

        /**
         * <p>
         * A list of the available processor features for the DB instance class of a DB instance.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AvailableProcessorFeature>.Builder}
         * avoiding the need to create one manually via {@link List<AvailableProcessorFeature>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AvailableProcessorFeature>.Builder#build()} is called
         * immediately and its result is passed to {@link #availableProcessorFeatures(List<AvailableProcessorFeature>)}.
         * 
         * @param availableProcessorFeatures
         *        a consumer that will call methods on {@link List<AvailableProcessorFeature>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #availableProcessorFeatures(List<AvailableProcessorFeature>)
         */
        Builder availableProcessorFeatures(Consumer<AvailableProcessorFeature.Builder>... availableProcessorFeatures);

        /**
         * <p>
         * A list of the supported DB engine modes.
         * </p>
         * <note>
         * <p>
         * <code>global</code> engine mode only applies for global database clusters created with Aurora MySQL version
         * 5.6.10a. For higher Aurora MySQL versions, the clusters in a global database use <code>provisioned</code>
         * engine mode.
         * </p>
         * </note>
         * 
         * @param supportedEngineModes
         *        A list of the supported DB engine modes.</p> <note>
         *        <p>
         *        <code>global</code> engine mode only applies for global database clusters created with Aurora MySQL
         *        version 5.6.10a. For higher Aurora MySQL versions, the clusters in a global database use
         *        <code>provisioned</code> engine mode.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedEngineModes(Collection<String> supportedEngineModes);

        /**
         * <p>
         * A list of the supported DB engine modes.
         * </p>
         * <note>
         * <p>
         * <code>global</code> engine mode only applies for global database clusters created with Aurora MySQL version
         * 5.6.10a. For higher Aurora MySQL versions, the clusters in a global database use <code>provisioned</code>
         * engine mode.
         * </p>
         * </note>
         * 
         * @param supportedEngineModes
         *        A list of the supported DB engine modes.</p> <note>
         *        <p>
         *        <code>global</code> engine mode only applies for global database clusters created with Aurora MySQL
         *        version 5.6.10a. For higher Aurora MySQL versions, the clusters in a global database use
         *        <code>provisioned</code> engine mode.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedEngineModes(String... supportedEngineModes);

        /**
         * <p>
         * Whether Amazon RDS can automatically scale storage for DB instances that use the specified DB instance class.
         * </p>
         * 
         * @param supportsStorageAutoscaling
         *        Whether Amazon RDS can automatically scale storage for DB instances that use the specified DB instance
         *        class.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsStorageAutoscaling(Boolean supportsStorageAutoscaling);

        /**
         * <p>
         * Whether a DB instance supports Kerberos Authentication.
         * </p>
         * 
         * @param supportsKerberosAuthentication
         *        Whether a DB instance supports Kerberos Authentication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsKerberosAuthentication(Boolean supportsKerberosAuthentication);
    }

    static final class BuilderImpl implements Builder {
        private String engine;

        private String engineVersion;

        private String dbInstanceClass;

        private String licenseModel;

        private String availabilityZoneGroup;

        private List<AvailabilityZone> availabilityZones = DefaultSdkAutoConstructList.getInstance();

        private Boolean multiAZCapable;

        private Boolean readReplicaCapable;

        private Boolean vpc;

        private Boolean supportsStorageEncryption;

        private String storageType;

        private Boolean supportsIops;

        private Boolean supportsEnhancedMonitoring;

        private Boolean supportsIAMDatabaseAuthentication;

        private Boolean supportsPerformanceInsights;

        private Integer minStorageSize;

        private Integer maxStorageSize;

        private Integer minIopsPerDbInstance;

        private Integer maxIopsPerDbInstance;

        private Double minIopsPerGib;

        private Double maxIopsPerGib;

        private List<AvailableProcessorFeature> availableProcessorFeatures = DefaultSdkAutoConstructList.getInstance();

        private List<String> supportedEngineModes = DefaultSdkAutoConstructList.getInstance();

        private Boolean supportsStorageAutoscaling;

        private Boolean supportsKerberosAuthentication;

        private BuilderImpl() {
        }

        private BuilderImpl(OrderableDBInstanceOption model) {
            engine(model.engine);
            engineVersion(model.engineVersion);
            dbInstanceClass(model.dbInstanceClass);
            licenseModel(model.licenseModel);
            availabilityZoneGroup(model.availabilityZoneGroup);
            availabilityZones(model.availabilityZones);
            multiAZCapable(model.multiAZCapable);
            readReplicaCapable(model.readReplicaCapable);
            vpc(model.vpc);
            supportsStorageEncryption(model.supportsStorageEncryption);
            storageType(model.storageType);
            supportsIops(model.supportsIops);
            supportsEnhancedMonitoring(model.supportsEnhancedMonitoring);
            supportsIAMDatabaseAuthentication(model.supportsIAMDatabaseAuthentication);
            supportsPerformanceInsights(model.supportsPerformanceInsights);
            minStorageSize(model.minStorageSize);
            maxStorageSize(model.maxStorageSize);
            minIopsPerDbInstance(model.minIopsPerDbInstance);
            maxIopsPerDbInstance(model.maxIopsPerDbInstance);
            minIopsPerGib(model.minIopsPerGib);
            maxIopsPerGib(model.maxIopsPerGib);
            availableProcessorFeatures(model.availableProcessorFeatures);
            supportedEngineModes(model.supportedEngineModes);
            supportsStorageAutoscaling(model.supportsStorageAutoscaling);
            supportsKerberosAuthentication(model.supportsKerberosAuthentication);
        }

        public final String getEngine() {
            return engine;
        }

        @Override
        public final Builder engine(String engine) {
            this.engine = engine;
            return this;
        }

        public final void setEngine(String engine) {
            this.engine = engine;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        public final String getDbInstanceClass() {
            return dbInstanceClass;
        }

        @Override
        public final Builder dbInstanceClass(String dbInstanceClass) {
            this.dbInstanceClass = dbInstanceClass;
            return this;
        }

        public final void setDbInstanceClass(String dbInstanceClass) {
            this.dbInstanceClass = dbInstanceClass;
        }

        public final String getLicenseModel() {
            return licenseModel;
        }

        @Override
        public final Builder licenseModel(String licenseModel) {
            this.licenseModel = licenseModel;
            return this;
        }

        public final void setLicenseModel(String licenseModel) {
            this.licenseModel = licenseModel;
        }

        public final String getAvailabilityZoneGroup() {
            return availabilityZoneGroup;
        }

        @Override
        public final Builder availabilityZoneGroup(String availabilityZoneGroup) {
            this.availabilityZoneGroup = availabilityZoneGroup;
            return this;
        }

        public final void setAvailabilityZoneGroup(String availabilityZoneGroup) {
            this.availabilityZoneGroup = availabilityZoneGroup;
        }

        public final Collection<AvailabilityZone.Builder> getAvailabilityZones() {
            return availabilityZones != null ? availabilityZones.stream().map(AvailabilityZone::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder availabilityZones(Collection<AvailabilityZone> availabilityZones) {
            this.availabilityZones = AvailabilityZoneListCopier.copy(availabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availabilityZones(AvailabilityZone... availabilityZones) {
            availabilityZones(Arrays.asList(availabilityZones));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availabilityZones(Consumer<AvailabilityZone.Builder>... availabilityZones) {
            availabilityZones(Stream.of(availabilityZones).map(c -> AvailabilityZone.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setAvailabilityZones(Collection<AvailabilityZone.BuilderImpl> availabilityZones) {
            this.availabilityZones = AvailabilityZoneListCopier.copyFromBuilder(availabilityZones);
        }

        public final Boolean getMultiAZCapable() {
            return multiAZCapable;
        }

        @Override
        public final Builder multiAZCapable(Boolean multiAZCapable) {
            this.multiAZCapable = multiAZCapable;
            return this;
        }

        public final void setMultiAZCapable(Boolean multiAZCapable) {
            this.multiAZCapable = multiAZCapable;
        }

        public final Boolean getReadReplicaCapable() {
            return readReplicaCapable;
        }

        @Override
        public final Builder readReplicaCapable(Boolean readReplicaCapable) {
            this.readReplicaCapable = readReplicaCapable;
            return this;
        }

        public final void setReadReplicaCapable(Boolean readReplicaCapable) {
            this.readReplicaCapable = readReplicaCapable;
        }

        public final Boolean getVpc() {
            return vpc;
        }

        @Override
        public final Builder vpc(Boolean vpc) {
            this.vpc = vpc;
            return this;
        }

        public final void setVpc(Boolean vpc) {
            this.vpc = vpc;
        }

        public final Boolean getSupportsStorageEncryption() {
            return supportsStorageEncryption;
        }

        @Override
        public final Builder supportsStorageEncryption(Boolean supportsStorageEncryption) {
            this.supportsStorageEncryption = supportsStorageEncryption;
            return this;
        }

        public final void setSupportsStorageEncryption(Boolean supportsStorageEncryption) {
            this.supportsStorageEncryption = supportsStorageEncryption;
        }

        public final String getStorageType() {
            return storageType;
        }

        @Override
        public final Builder storageType(String storageType) {
            this.storageType = storageType;
            return this;
        }

        public final void setStorageType(String storageType) {
            this.storageType = storageType;
        }

        public final Boolean getSupportsIops() {
            return supportsIops;
        }

        @Override
        public final Builder supportsIops(Boolean supportsIops) {
            this.supportsIops = supportsIops;
            return this;
        }

        public final void setSupportsIops(Boolean supportsIops) {
            this.supportsIops = supportsIops;
        }

        public final Boolean getSupportsEnhancedMonitoring() {
            return supportsEnhancedMonitoring;
        }

        @Override
        public final Builder supportsEnhancedMonitoring(Boolean supportsEnhancedMonitoring) {
            this.supportsEnhancedMonitoring = supportsEnhancedMonitoring;
            return this;
        }

        public final void setSupportsEnhancedMonitoring(Boolean supportsEnhancedMonitoring) {
            this.supportsEnhancedMonitoring = supportsEnhancedMonitoring;
        }

        public final Boolean getSupportsIAMDatabaseAuthentication() {
            return supportsIAMDatabaseAuthentication;
        }

        @Override
        public final Builder supportsIAMDatabaseAuthentication(Boolean supportsIAMDatabaseAuthentication) {
            this.supportsIAMDatabaseAuthentication = supportsIAMDatabaseAuthentication;
            return this;
        }

        public final void setSupportsIAMDatabaseAuthentication(Boolean supportsIAMDatabaseAuthentication) {
            this.supportsIAMDatabaseAuthentication = supportsIAMDatabaseAuthentication;
        }

        public final Boolean getSupportsPerformanceInsights() {
            return supportsPerformanceInsights;
        }

        @Override
        public final Builder supportsPerformanceInsights(Boolean supportsPerformanceInsights) {
            this.supportsPerformanceInsights = supportsPerformanceInsights;
            return this;
        }

        public final void setSupportsPerformanceInsights(Boolean supportsPerformanceInsights) {
            this.supportsPerformanceInsights = supportsPerformanceInsights;
        }

        public final Integer getMinStorageSize() {
            return minStorageSize;
        }

        @Override
        public final Builder minStorageSize(Integer minStorageSize) {
            this.minStorageSize = minStorageSize;
            return this;
        }

        public final void setMinStorageSize(Integer minStorageSize) {
            this.minStorageSize = minStorageSize;
        }

        public final Integer getMaxStorageSize() {
            return maxStorageSize;
        }

        @Override
        public final Builder maxStorageSize(Integer maxStorageSize) {
            this.maxStorageSize = maxStorageSize;
            return this;
        }

        public final void setMaxStorageSize(Integer maxStorageSize) {
            this.maxStorageSize = maxStorageSize;
        }

        public final Integer getMinIopsPerDbInstance() {
            return minIopsPerDbInstance;
        }

        @Override
        public final Builder minIopsPerDbInstance(Integer minIopsPerDbInstance) {
            this.minIopsPerDbInstance = minIopsPerDbInstance;
            return this;
        }

        public final void setMinIopsPerDbInstance(Integer minIopsPerDbInstance) {
            this.minIopsPerDbInstance = minIopsPerDbInstance;
        }

        public final Integer getMaxIopsPerDbInstance() {
            return maxIopsPerDbInstance;
        }

        @Override
        public final Builder maxIopsPerDbInstance(Integer maxIopsPerDbInstance) {
            this.maxIopsPerDbInstance = maxIopsPerDbInstance;
            return this;
        }

        public final void setMaxIopsPerDbInstance(Integer maxIopsPerDbInstance) {
            this.maxIopsPerDbInstance = maxIopsPerDbInstance;
        }

        public final Double getMinIopsPerGib() {
            return minIopsPerGib;
        }

        @Override
        public final Builder minIopsPerGib(Double minIopsPerGib) {
            this.minIopsPerGib = minIopsPerGib;
            return this;
        }

        public final void setMinIopsPerGib(Double minIopsPerGib) {
            this.minIopsPerGib = minIopsPerGib;
        }

        public final Double getMaxIopsPerGib() {
            return maxIopsPerGib;
        }

        @Override
        public final Builder maxIopsPerGib(Double maxIopsPerGib) {
            this.maxIopsPerGib = maxIopsPerGib;
            return this;
        }

        public final void setMaxIopsPerGib(Double maxIopsPerGib) {
            this.maxIopsPerGib = maxIopsPerGib;
        }

        public final Collection<AvailableProcessorFeature.Builder> getAvailableProcessorFeatures() {
            return availableProcessorFeatures != null ? availableProcessorFeatures.stream()
                    .map(AvailableProcessorFeature::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder availableProcessorFeatures(Collection<AvailableProcessorFeature> availableProcessorFeatures) {
            this.availableProcessorFeatures = AvailableProcessorFeatureListCopier.copy(availableProcessorFeatures);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availableProcessorFeatures(AvailableProcessorFeature... availableProcessorFeatures) {
            availableProcessorFeatures(Arrays.asList(availableProcessorFeatures));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availableProcessorFeatures(Consumer<AvailableProcessorFeature.Builder>... availableProcessorFeatures) {
            availableProcessorFeatures(Stream.of(availableProcessorFeatures)
                    .map(c -> AvailableProcessorFeature.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setAvailableProcessorFeatures(
                Collection<AvailableProcessorFeature.BuilderImpl> availableProcessorFeatures) {
            this.availableProcessorFeatures = AvailableProcessorFeatureListCopier.copyFromBuilder(availableProcessorFeatures);
        }

        public final Collection<String> getSupportedEngineModes() {
            return supportedEngineModes;
        }

        @Override
        public final Builder supportedEngineModes(Collection<String> supportedEngineModes) {
            this.supportedEngineModes = EngineModeListCopier.copy(supportedEngineModes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedEngineModes(String... supportedEngineModes) {
            supportedEngineModes(Arrays.asList(supportedEngineModes));
            return this;
        }

        public final void setSupportedEngineModes(Collection<String> supportedEngineModes) {
            this.supportedEngineModes = EngineModeListCopier.copy(supportedEngineModes);
        }

        public final Boolean getSupportsStorageAutoscaling() {
            return supportsStorageAutoscaling;
        }

        @Override
        public final Builder supportsStorageAutoscaling(Boolean supportsStorageAutoscaling) {
            this.supportsStorageAutoscaling = supportsStorageAutoscaling;
            return this;
        }

        public final void setSupportsStorageAutoscaling(Boolean supportsStorageAutoscaling) {
            this.supportsStorageAutoscaling = supportsStorageAutoscaling;
        }

        public final Boolean getSupportsKerberosAuthentication() {
            return supportsKerberosAuthentication;
        }

        @Override
        public final Builder supportsKerberosAuthentication(Boolean supportsKerberosAuthentication) {
            this.supportsKerberosAuthentication = supportsKerberosAuthentication;
            return this;
        }

        public final void setSupportsKerberosAuthentication(Boolean supportsKerberosAuthentication) {
            this.supportsKerberosAuthentication = supportsKerberosAuthentication;
        }

        @Override
        public OrderableDBInstanceOption build() {
            return new OrderableDBInstanceOption(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
