/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the details of an Amazon RDS DB snapshot.
 * </p>
 * <p>
 * This data type is used as a response element in the <code>DescribeDBSnapshots</code> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DBSnapshot implements SdkPojo, Serializable, ToCopyableBuilder<DBSnapshot.Builder, DBSnapshot> {
    private static final SdkField<String> DB_SNAPSHOT_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::dbSnapshotIdentifier)).setter(setter(Builder::dbSnapshotIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBSnapshotIdentifier").build())
            .build();

    private static final SdkField<String> DB_INSTANCE_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::dbInstanceIdentifier)).setter(setter(Builder::dbInstanceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBInstanceIdentifier").build())
            .build();

    private static final SdkField<Instant> SNAPSHOT_CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(DBSnapshot::snapshotCreateTime)).setter(setter(Builder::snapshotCreateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotCreateTime").build())
            .build();

    private static final SdkField<String> ENGINE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::engine)).setter(setter(Builder::engine))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Engine").build()).build();

    private static final SdkField<Integer> ALLOCATED_STORAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(DBSnapshot::allocatedStorage)).setter(setter(Builder::allocatedStorage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocatedStorage").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(DBSnapshot::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::availabilityZone)).setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZone").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<Instant> INSTANCE_CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(DBSnapshot::instanceCreateTime)).setter(setter(Builder::instanceCreateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceCreateTime").build())
            .build();

    private static final SdkField<String> MASTER_USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::masterUsername)).setter(setter(Builder::masterUsername))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MasterUsername").build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::engineVersion)).setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<String> LICENSE_MODEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::licenseModel)).setter(setter(Builder::licenseModel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicenseModel").build()).build();

    private static final SdkField<String> SNAPSHOT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::snapshotType)).setter(setter(Builder::snapshotType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotType").build()).build();

    private static final SdkField<Integer> IOPS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(DBSnapshot::iops)).setter(setter(Builder::iops))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Iops").build()).build();

    private static final SdkField<String> OPTION_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::optionGroupName)).setter(setter(Builder::optionGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OptionGroupName").build()).build();

    private static final SdkField<Integer> PERCENT_PROGRESS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(DBSnapshot::percentProgress)).setter(setter(Builder::percentProgress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PercentProgress").build()).build();

    private static final SdkField<String> SOURCE_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::sourceRegion)).setter(setter(Builder::sourceRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceRegion").build()).build();

    private static final SdkField<String> SOURCE_DB_SNAPSHOT_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::sourceDBSnapshotIdentifier))
            .setter(setter(Builder::sourceDBSnapshotIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceDBSnapshotIdentifier").build())
            .build();

    private static final SdkField<String> STORAGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::storageType)).setter(setter(Builder::storageType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageType").build()).build();

    private static final SdkField<String> TDE_CREDENTIAL_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::tdeCredentialArn)).setter(setter(Builder::tdeCredentialArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TdeCredentialArn").build()).build();

    private static final SdkField<Boolean> ENCRYPTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(DBSnapshot::encrypted)).setter(setter(Builder::encrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Encrypted").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<String> DB_SNAPSHOT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::dbSnapshotArn)).setter(setter(Builder::dbSnapshotArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBSnapshotArn").build()).build();

    private static final SdkField<String> TIMEZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::timezone)).setter(setter(Builder::timezone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timezone").build()).build();

    private static final SdkField<Boolean> IAM_DATABASE_AUTHENTICATION_ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(DBSnapshot::iamDatabaseAuthenticationEnabled))
            .setter(setter(Builder::iamDatabaseAuthenticationEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IAMDatabaseAuthenticationEnabled")
                    .build()).build();

    private static final SdkField<List<ProcessorFeature>> PROCESSOR_FEATURES_FIELD = SdkField
            .<List<ProcessorFeature>> builder(MarshallingType.LIST)
            .getter(getter(DBSnapshot::processorFeatures))
            .setter(setter(Builder::processorFeatures))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProcessorFeatures").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("ProcessorFeature")
                            .memberFieldInfo(
                                    SdkField.<ProcessorFeature> builder(MarshallingType.SDK_POJO)
                                            .constructor(ProcessorFeature::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("ProcessorFeature").build()).build()).build()).build();

    private static final SdkField<String> DBI_RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBSnapshot::dbiResourceId)).setter(setter(Builder::dbiResourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DbiResourceId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DB_SNAPSHOT_IDENTIFIER_FIELD,
            DB_INSTANCE_IDENTIFIER_FIELD, SNAPSHOT_CREATE_TIME_FIELD, ENGINE_FIELD, ALLOCATED_STORAGE_FIELD, STATUS_FIELD,
            PORT_FIELD, AVAILABILITY_ZONE_FIELD, VPC_ID_FIELD, INSTANCE_CREATE_TIME_FIELD, MASTER_USERNAME_FIELD,
            ENGINE_VERSION_FIELD, LICENSE_MODEL_FIELD, SNAPSHOT_TYPE_FIELD, IOPS_FIELD, OPTION_GROUP_NAME_FIELD,
            PERCENT_PROGRESS_FIELD, SOURCE_REGION_FIELD, SOURCE_DB_SNAPSHOT_IDENTIFIER_FIELD, STORAGE_TYPE_FIELD,
            TDE_CREDENTIAL_ARN_FIELD, ENCRYPTED_FIELD, KMS_KEY_ID_FIELD, DB_SNAPSHOT_ARN_FIELD, TIMEZONE_FIELD,
            IAM_DATABASE_AUTHENTICATION_ENABLED_FIELD, PROCESSOR_FEATURES_FIELD, DBI_RESOURCE_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String dbSnapshotIdentifier;

    private final String dbInstanceIdentifier;

    private final Instant snapshotCreateTime;

    private final String engine;

    private final Integer allocatedStorage;

    private final String status;

    private final Integer port;

    private final String availabilityZone;

    private final String vpcId;

    private final Instant instanceCreateTime;

    private final String masterUsername;

    private final String engineVersion;

    private final String licenseModel;

    private final String snapshotType;

    private final Integer iops;

    private final String optionGroupName;

    private final Integer percentProgress;

    private final String sourceRegion;

    private final String sourceDBSnapshotIdentifier;

    private final String storageType;

    private final String tdeCredentialArn;

    private final Boolean encrypted;

    private final String kmsKeyId;

    private final String dbSnapshotArn;

    private final String timezone;

    private final Boolean iamDatabaseAuthenticationEnabled;

    private final List<ProcessorFeature> processorFeatures;

    private final String dbiResourceId;

    private DBSnapshot(BuilderImpl builder) {
        this.dbSnapshotIdentifier = builder.dbSnapshotIdentifier;
        this.dbInstanceIdentifier = builder.dbInstanceIdentifier;
        this.snapshotCreateTime = builder.snapshotCreateTime;
        this.engine = builder.engine;
        this.allocatedStorage = builder.allocatedStorage;
        this.status = builder.status;
        this.port = builder.port;
        this.availabilityZone = builder.availabilityZone;
        this.vpcId = builder.vpcId;
        this.instanceCreateTime = builder.instanceCreateTime;
        this.masterUsername = builder.masterUsername;
        this.engineVersion = builder.engineVersion;
        this.licenseModel = builder.licenseModel;
        this.snapshotType = builder.snapshotType;
        this.iops = builder.iops;
        this.optionGroupName = builder.optionGroupName;
        this.percentProgress = builder.percentProgress;
        this.sourceRegion = builder.sourceRegion;
        this.sourceDBSnapshotIdentifier = builder.sourceDBSnapshotIdentifier;
        this.storageType = builder.storageType;
        this.tdeCredentialArn = builder.tdeCredentialArn;
        this.encrypted = builder.encrypted;
        this.kmsKeyId = builder.kmsKeyId;
        this.dbSnapshotArn = builder.dbSnapshotArn;
        this.timezone = builder.timezone;
        this.iamDatabaseAuthenticationEnabled = builder.iamDatabaseAuthenticationEnabled;
        this.processorFeatures = builder.processorFeatures;
        this.dbiResourceId = builder.dbiResourceId;
    }

    /**
     * <p>
     * Specifies the identifier for the DB snapshot.
     * </p>
     * 
     * @return Specifies the identifier for the DB snapshot.
     */
    public String dbSnapshotIdentifier() {
        return dbSnapshotIdentifier;
    }

    /**
     * <p>
     * Specifies the DB instance identifier of the DB instance this DB snapshot was created from.
     * </p>
     * 
     * @return Specifies the DB instance identifier of the DB instance this DB snapshot was created from.
     */
    public String dbInstanceIdentifier() {
        return dbInstanceIdentifier;
    }

    /**
     * <p>
     * Provides the time when the snapshot was taken, in Universal Coordinated Time (UTC).
     * </p>
     * 
     * @return Provides the time when the snapshot was taken, in Universal Coordinated Time (UTC).
     */
    public Instant snapshotCreateTime() {
        return snapshotCreateTime;
    }

    /**
     * <p>
     * Specifies the name of the database engine.
     * </p>
     * 
     * @return Specifies the name of the database engine.
     */
    public String engine() {
        return engine;
    }

    /**
     * <p>
     * Specifies the allocated storage size in gibibytes (GiB).
     * </p>
     * 
     * @return Specifies the allocated storage size in gibibytes (GiB).
     */
    public Integer allocatedStorage() {
        return allocatedStorage;
    }

    /**
     * <p>
     * Specifies the status of this DB snapshot.
     * </p>
     * 
     * @return Specifies the status of this DB snapshot.
     */
    public String status() {
        return status;
    }

    /**
     * <p>
     * Specifies the port that the database engine was listening on at the time of the snapshot.
     * </p>
     * 
     * @return Specifies the port that the database engine was listening on at the time of the snapshot.
     */
    public Integer port() {
        return port;
    }

    /**
     * <p>
     * Specifies the name of the Availability Zone the DB instance was located in at the time of the DB snapshot.
     * </p>
     * 
     * @return Specifies the name of the Availability Zone the DB instance was located in at the time of the DB
     *         snapshot.
     */
    public String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * Provides the VPC ID associated with the DB snapshot.
     * </p>
     * 
     * @return Provides the VPC ID associated with the DB snapshot.
     */
    public String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * Specifies the time when the snapshot was taken, in Universal Coordinated Time (UTC).
     * </p>
     * 
     * @return Specifies the time when the snapshot was taken, in Universal Coordinated Time (UTC).
     */
    public Instant instanceCreateTime() {
        return instanceCreateTime;
    }

    /**
     * <p>
     * Provides the master username for the DB snapshot.
     * </p>
     * 
     * @return Provides the master username for the DB snapshot.
     */
    public String masterUsername() {
        return masterUsername;
    }

    /**
     * <p>
     * Specifies the version of the database engine.
     * </p>
     * 
     * @return Specifies the version of the database engine.
     */
    public String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * License model information for the restored DB instance.
     * </p>
     * 
     * @return License model information for the restored DB instance.
     */
    public String licenseModel() {
        return licenseModel;
    }

    /**
     * <p>
     * Provides the type of the DB snapshot.
     * </p>
     * 
     * @return Provides the type of the DB snapshot.
     */
    public String snapshotType() {
        return snapshotType;
    }

    /**
     * <p>
     * Specifies the Provisioned IOPS (I/O operations per second) value of the DB instance at the time of the snapshot.
     * </p>
     * 
     * @return Specifies the Provisioned IOPS (I/O operations per second) value of the DB instance at the time of the
     *         snapshot.
     */
    public Integer iops() {
        return iops;
    }

    /**
     * <p>
     * Provides the option group name for the DB snapshot.
     * </p>
     * 
     * @return Provides the option group name for the DB snapshot.
     */
    public String optionGroupName() {
        return optionGroupName;
    }

    /**
     * <p>
     * The percentage of the estimated data that has been transferred.
     * </p>
     * 
     * @return The percentage of the estimated data that has been transferred.
     */
    public Integer percentProgress() {
        return percentProgress;
    }

    /**
     * <p>
     * The AWS Region that the DB snapshot was created in or copied from.
     * </p>
     * 
     * @return The AWS Region that the DB snapshot was created in or copied from.
     */
    public String sourceRegion() {
        return sourceRegion;
    }

    /**
     * <p>
     * The DB snapshot Amazon Resource Name (ARN) that the DB snapshot was copied from. It only has value in case of
     * cross-customer or cross-region copy.
     * </p>
     * 
     * @return The DB snapshot Amazon Resource Name (ARN) that the DB snapshot was copied from. It only has value in
     *         case of cross-customer or cross-region copy.
     */
    public String sourceDBSnapshotIdentifier() {
        return sourceDBSnapshotIdentifier;
    }

    /**
     * <p>
     * Specifies the storage type associated with DB snapshot.
     * </p>
     * 
     * @return Specifies the storage type associated with DB snapshot.
     */
    public String storageType() {
        return storageType;
    }

    /**
     * <p>
     * The ARN from the key store with which to associate the instance for TDE encryption.
     * </p>
     * 
     * @return The ARN from the key store with which to associate the instance for TDE encryption.
     */
    public String tdeCredentialArn() {
        return tdeCredentialArn;
    }

    /**
     * <p>
     * Specifies whether the DB snapshot is encrypted.
     * </p>
     * 
     * @return Specifies whether the DB snapshot is encrypted.
     */
    public Boolean encrypted() {
        return encrypted;
    }

    /**
     * <p>
     * If <code>Encrypted</code> is true, the AWS KMS key identifier for the encrypted DB snapshot.
     * </p>
     * 
     * @return If <code>Encrypted</code> is true, the AWS KMS key identifier for the encrypted DB snapshot.
     */
    public String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the DB snapshot.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the DB snapshot.
     */
    public String dbSnapshotArn() {
        return dbSnapshotArn;
    }

    /**
     * <p>
     * The time zone of the DB snapshot. In most cases, the <code>Timezone</code> element is empty.
     * <code>Timezone</code> content appears only for snapshots taken from Microsoft SQL Server DB instances that were
     * created with a time zone specified.
     * </p>
     * 
     * @return The time zone of the DB snapshot. In most cases, the <code>Timezone</code> element is empty.
     *         <code>Timezone</code> content appears only for snapshots taken from Microsoft SQL Server DB instances
     *         that were created with a time zone specified.
     */
    public String timezone() {
        return timezone;
    }

    /**
     * <p>
     * True if mapping of AWS Identity and Access Management (IAM) accounts to database accounts is enabled, and
     * otherwise false.
     * </p>
     * 
     * @return True if mapping of AWS Identity and Access Management (IAM) accounts to database accounts is enabled, and
     *         otherwise false.
     */
    public Boolean iamDatabaseAuthenticationEnabled() {
        return iamDatabaseAuthenticationEnabled;
    }

    /**
     * Returns true if the ProcessorFeatures property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasProcessorFeatures() {
        return processorFeatures != null && !(processorFeatures instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The number of CPU cores and the number of threads per core for the DB instance class of the DB instance when the
     * DB snapshot was created.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasProcessorFeatures()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The number of CPU cores and the number of threads per core for the DB instance class of the DB instance
     *         when the DB snapshot was created.
     */
    public List<ProcessorFeature> processorFeatures() {
        return processorFeatures;
    }

    /**
     * <p>
     * The identifier for the source DB instance, which can't be changed and which is unique to an AWS Region.
     * </p>
     * 
     * @return The identifier for the source DB instance, which can't be changed and which is unique to an AWS Region.
     */
    public String dbiResourceId() {
        return dbiResourceId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dbSnapshotIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(dbInstanceIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotCreateTime());
        hashCode = 31 * hashCode + Objects.hashCode(engine());
        hashCode = 31 * hashCode + Objects.hashCode(allocatedStorage());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceCreateTime());
        hashCode = 31 * hashCode + Objects.hashCode(masterUsername());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(licenseModel());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotType());
        hashCode = 31 * hashCode + Objects.hashCode(iops());
        hashCode = 31 * hashCode + Objects.hashCode(optionGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(percentProgress());
        hashCode = 31 * hashCode + Objects.hashCode(sourceRegion());
        hashCode = 31 * hashCode + Objects.hashCode(sourceDBSnapshotIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(storageType());
        hashCode = 31 * hashCode + Objects.hashCode(tdeCredentialArn());
        hashCode = 31 * hashCode + Objects.hashCode(encrypted());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(dbSnapshotArn());
        hashCode = 31 * hashCode + Objects.hashCode(timezone());
        hashCode = 31 * hashCode + Objects.hashCode(iamDatabaseAuthenticationEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(processorFeatures());
        hashCode = 31 * hashCode + Objects.hashCode(dbiResourceId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DBSnapshot)) {
            return false;
        }
        DBSnapshot other = (DBSnapshot) obj;
        return Objects.equals(dbSnapshotIdentifier(), other.dbSnapshotIdentifier())
                && Objects.equals(dbInstanceIdentifier(), other.dbInstanceIdentifier())
                && Objects.equals(snapshotCreateTime(), other.snapshotCreateTime()) && Objects.equals(engine(), other.engine())
                && Objects.equals(allocatedStorage(), other.allocatedStorage()) && Objects.equals(status(), other.status())
                && Objects.equals(port(), other.port()) && Objects.equals(availabilityZone(), other.availabilityZone())
                && Objects.equals(vpcId(), other.vpcId()) && Objects.equals(instanceCreateTime(), other.instanceCreateTime())
                && Objects.equals(masterUsername(), other.masterUsername())
                && Objects.equals(engineVersion(), other.engineVersion()) && Objects.equals(licenseModel(), other.licenseModel())
                && Objects.equals(snapshotType(), other.snapshotType()) && Objects.equals(iops(), other.iops())
                && Objects.equals(optionGroupName(), other.optionGroupName())
                && Objects.equals(percentProgress(), other.percentProgress())
                && Objects.equals(sourceRegion(), other.sourceRegion())
                && Objects.equals(sourceDBSnapshotIdentifier(), other.sourceDBSnapshotIdentifier())
                && Objects.equals(storageType(), other.storageType())
                && Objects.equals(tdeCredentialArn(), other.tdeCredentialArn()) && Objects.equals(encrypted(), other.encrypted())
                && Objects.equals(kmsKeyId(), other.kmsKeyId()) && Objects.equals(dbSnapshotArn(), other.dbSnapshotArn())
                && Objects.equals(timezone(), other.timezone())
                && Objects.equals(iamDatabaseAuthenticationEnabled(), other.iamDatabaseAuthenticationEnabled())
                && Objects.equals(processorFeatures(), other.processorFeatures())
                && Objects.equals(dbiResourceId(), other.dbiResourceId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DBSnapshot").add("DBSnapshotIdentifier", dbSnapshotIdentifier())
                .add("DBInstanceIdentifier", dbInstanceIdentifier()).add("SnapshotCreateTime", snapshotCreateTime())
                .add("Engine", engine()).add("AllocatedStorage", allocatedStorage()).add("Status", status()).add("Port", port())
                .add("AvailabilityZone", availabilityZone()).add("VpcId", vpcId())
                .add("InstanceCreateTime", instanceCreateTime()).add("MasterUsername", masterUsername())
                .add("EngineVersion", engineVersion()).add("LicenseModel", licenseModel()).add("SnapshotType", snapshotType())
                .add("Iops", iops()).add("OptionGroupName", optionGroupName()).add("PercentProgress", percentProgress())
                .add("SourceRegion", sourceRegion()).add("SourceDBSnapshotIdentifier", sourceDBSnapshotIdentifier())
                .add("StorageType", storageType()).add("TdeCredentialArn", tdeCredentialArn()).add("Encrypted", encrypted())
                .add("KmsKeyId", kmsKeyId()).add("DBSnapshotArn", dbSnapshotArn()).add("Timezone", timezone())
                .add("IAMDatabaseAuthenticationEnabled", iamDatabaseAuthenticationEnabled())
                .add("ProcessorFeatures", processorFeatures()).add("DbiResourceId", dbiResourceId()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DBSnapshotIdentifier":
            return Optional.ofNullable(clazz.cast(dbSnapshotIdentifier()));
        case "DBInstanceIdentifier":
            return Optional.ofNullable(clazz.cast(dbInstanceIdentifier()));
        case "SnapshotCreateTime":
            return Optional.ofNullable(clazz.cast(snapshotCreateTime()));
        case "Engine":
            return Optional.ofNullable(clazz.cast(engine()));
        case "AllocatedStorage":
            return Optional.ofNullable(clazz.cast(allocatedStorage()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "InstanceCreateTime":
            return Optional.ofNullable(clazz.cast(instanceCreateTime()));
        case "MasterUsername":
            return Optional.ofNullable(clazz.cast(masterUsername()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "LicenseModel":
            return Optional.ofNullable(clazz.cast(licenseModel()));
        case "SnapshotType":
            return Optional.ofNullable(clazz.cast(snapshotType()));
        case "Iops":
            return Optional.ofNullable(clazz.cast(iops()));
        case "OptionGroupName":
            return Optional.ofNullable(clazz.cast(optionGroupName()));
        case "PercentProgress":
            return Optional.ofNullable(clazz.cast(percentProgress()));
        case "SourceRegion":
            return Optional.ofNullable(clazz.cast(sourceRegion()));
        case "SourceDBSnapshotIdentifier":
            return Optional.ofNullable(clazz.cast(sourceDBSnapshotIdentifier()));
        case "StorageType":
            return Optional.ofNullable(clazz.cast(storageType()));
        case "TdeCredentialArn":
            return Optional.ofNullable(clazz.cast(tdeCredentialArn()));
        case "Encrypted":
            return Optional.ofNullable(clazz.cast(encrypted()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "DBSnapshotArn":
            return Optional.ofNullable(clazz.cast(dbSnapshotArn()));
        case "Timezone":
            return Optional.ofNullable(clazz.cast(timezone()));
        case "IAMDatabaseAuthenticationEnabled":
            return Optional.ofNullable(clazz.cast(iamDatabaseAuthenticationEnabled()));
        case "ProcessorFeatures":
            return Optional.ofNullable(clazz.cast(processorFeatures()));
        case "DbiResourceId":
            return Optional.ofNullable(clazz.cast(dbiResourceId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DBSnapshot, T> g) {
        return obj -> g.apply((DBSnapshot) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DBSnapshot> {
        /**
         * <p>
         * Specifies the identifier for the DB snapshot.
         * </p>
         * 
         * @param dbSnapshotIdentifier
         *        Specifies the identifier for the DB snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbSnapshotIdentifier(String dbSnapshotIdentifier);

        /**
         * <p>
         * Specifies the DB instance identifier of the DB instance this DB snapshot was created from.
         * </p>
         * 
         * @param dbInstanceIdentifier
         *        Specifies the DB instance identifier of the DB instance this DB snapshot was created from.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbInstanceIdentifier(String dbInstanceIdentifier);

        /**
         * <p>
         * Provides the time when the snapshot was taken, in Universal Coordinated Time (UTC).
         * </p>
         * 
         * @param snapshotCreateTime
         *        Provides the time when the snapshot was taken, in Universal Coordinated Time (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotCreateTime(Instant snapshotCreateTime);

        /**
         * <p>
         * Specifies the name of the database engine.
         * </p>
         * 
         * @param engine
         *        Specifies the name of the database engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engine(String engine);

        /**
         * <p>
         * Specifies the allocated storage size in gibibytes (GiB).
         * </p>
         * 
         * @param allocatedStorage
         *        Specifies the allocated storage size in gibibytes (GiB).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocatedStorage(Integer allocatedStorage);

        /**
         * <p>
         * Specifies the status of this DB snapshot.
         * </p>
         * 
         * @param status
         *        Specifies the status of this DB snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * Specifies the port that the database engine was listening on at the time of the snapshot.
         * </p>
         * 
         * @param port
         *        Specifies the port that the database engine was listening on at the time of the snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * Specifies the name of the Availability Zone the DB instance was located in at the time of the DB snapshot.
         * </p>
         * 
         * @param availabilityZone
         *        Specifies the name of the Availability Zone the DB instance was located in at the time of the DB
         *        snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * Provides the VPC ID associated with the DB snapshot.
         * </p>
         * 
         * @param vpcId
         *        Provides the VPC ID associated with the DB snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * Specifies the time when the snapshot was taken, in Universal Coordinated Time (UTC).
         * </p>
         * 
         * @param instanceCreateTime
         *        Specifies the time when the snapshot was taken, in Universal Coordinated Time (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceCreateTime(Instant instanceCreateTime);

        /**
         * <p>
         * Provides the master username for the DB snapshot.
         * </p>
         * 
         * @param masterUsername
         *        Provides the master username for the DB snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterUsername(String masterUsername);

        /**
         * <p>
         * Specifies the version of the database engine.
         * </p>
         * 
         * @param engineVersion
         *        Specifies the version of the database engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * License model information for the restored DB instance.
         * </p>
         * 
         * @param licenseModel
         *        License model information for the restored DB instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseModel(String licenseModel);

        /**
         * <p>
         * Provides the type of the DB snapshot.
         * </p>
         * 
         * @param snapshotType
         *        Provides the type of the DB snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotType(String snapshotType);

        /**
         * <p>
         * Specifies the Provisioned IOPS (I/O operations per second) value of the DB instance at the time of the
         * snapshot.
         * </p>
         * 
         * @param iops
         *        Specifies the Provisioned IOPS (I/O operations per second) value of the DB instance at the time of the
         *        snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iops(Integer iops);

        /**
         * <p>
         * Provides the option group name for the DB snapshot.
         * </p>
         * 
         * @param optionGroupName
         *        Provides the option group name for the DB snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder optionGroupName(String optionGroupName);

        /**
         * <p>
         * The percentage of the estimated data that has been transferred.
         * </p>
         * 
         * @param percentProgress
         *        The percentage of the estimated data that has been transferred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentProgress(Integer percentProgress);

        /**
         * <p>
         * The AWS Region that the DB snapshot was created in or copied from.
         * </p>
         * 
         * @param sourceRegion
         *        The AWS Region that the DB snapshot was created in or copied from.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceRegion(String sourceRegion);

        /**
         * <p>
         * The DB snapshot Amazon Resource Name (ARN) that the DB snapshot was copied from. It only has value in case of
         * cross-customer or cross-region copy.
         * </p>
         * 
         * @param sourceDBSnapshotIdentifier
         *        The DB snapshot Amazon Resource Name (ARN) that the DB snapshot was copied from. It only has value in
         *        case of cross-customer or cross-region copy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceDBSnapshotIdentifier(String sourceDBSnapshotIdentifier);

        /**
         * <p>
         * Specifies the storage type associated with DB snapshot.
         * </p>
         * 
         * @param storageType
         *        Specifies the storage type associated with DB snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageType(String storageType);

        /**
         * <p>
         * The ARN from the key store with which to associate the instance for TDE encryption.
         * </p>
         * 
         * @param tdeCredentialArn
         *        The ARN from the key store with which to associate the instance for TDE encryption.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tdeCredentialArn(String tdeCredentialArn);

        /**
         * <p>
         * Specifies whether the DB snapshot is encrypted.
         * </p>
         * 
         * @param encrypted
         *        Specifies whether the DB snapshot is encrypted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encrypted(Boolean encrypted);

        /**
         * <p>
         * If <code>Encrypted</code> is true, the AWS KMS key identifier for the encrypted DB snapshot.
         * </p>
         * 
         * @param kmsKeyId
         *        If <code>Encrypted</code> is true, the AWS KMS key identifier for the encrypted DB snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the DB snapshot.
         * </p>
         * 
         * @param dbSnapshotArn
         *        The Amazon Resource Name (ARN) for the DB snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbSnapshotArn(String dbSnapshotArn);

        /**
         * <p>
         * The time zone of the DB snapshot. In most cases, the <code>Timezone</code> element is empty.
         * <code>Timezone</code> content appears only for snapshots taken from Microsoft SQL Server DB instances that
         * were created with a time zone specified.
         * </p>
         * 
         * @param timezone
         *        The time zone of the DB snapshot. In most cases, the <code>Timezone</code> element is empty.
         *        <code>Timezone</code> content appears only for snapshots taken from Microsoft SQL Server DB instances
         *        that were created with a time zone specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timezone(String timezone);

        /**
         * <p>
         * True if mapping of AWS Identity and Access Management (IAM) accounts to database accounts is enabled, and
         * otherwise false.
         * </p>
         * 
         * @param iamDatabaseAuthenticationEnabled
         *        True if mapping of AWS Identity and Access Management (IAM) accounts to database accounts is enabled,
         *        and otherwise false.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamDatabaseAuthenticationEnabled(Boolean iamDatabaseAuthenticationEnabled);

        /**
         * <p>
         * The number of CPU cores and the number of threads per core for the DB instance class of the DB instance when
         * the DB snapshot was created.
         * </p>
         * 
         * @param processorFeatures
         *        The number of CPU cores and the number of threads per core for the DB instance class of the DB
         *        instance when the DB snapshot was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processorFeatures(Collection<ProcessorFeature> processorFeatures);

        /**
         * <p>
         * The number of CPU cores and the number of threads per core for the DB instance class of the DB instance when
         * the DB snapshot was created.
         * </p>
         * 
         * @param processorFeatures
         *        The number of CPU cores and the number of threads per core for the DB instance class of the DB
         *        instance when the DB snapshot was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processorFeatures(ProcessorFeature... processorFeatures);

        /**
         * <p>
         * The number of CPU cores and the number of threads per core for the DB instance class of the DB instance when
         * the DB snapshot was created.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ProcessorFeature>.Builder} avoiding the
         * need to create one manually via {@link List<ProcessorFeature>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ProcessorFeature>.Builder#build()} is called immediately and
         * its result is passed to {@link #processorFeatures(List<ProcessorFeature>)}.
         * 
         * @param processorFeatures
         *        a consumer that will call methods on {@link List<ProcessorFeature>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #processorFeatures(List<ProcessorFeature>)
         */
        Builder processorFeatures(Consumer<ProcessorFeature.Builder>... processorFeatures);

        /**
         * <p>
         * The identifier for the source DB instance, which can't be changed and which is unique to an AWS Region.
         * </p>
         * 
         * @param dbiResourceId
         *        The identifier for the source DB instance, which can't be changed and which is unique to an AWS
         *        Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbiResourceId(String dbiResourceId);
    }

    static final class BuilderImpl implements Builder {
        private String dbSnapshotIdentifier;

        private String dbInstanceIdentifier;

        private Instant snapshotCreateTime;

        private String engine;

        private Integer allocatedStorage;

        private String status;

        private Integer port;

        private String availabilityZone;

        private String vpcId;

        private Instant instanceCreateTime;

        private String masterUsername;

        private String engineVersion;

        private String licenseModel;

        private String snapshotType;

        private Integer iops;

        private String optionGroupName;

        private Integer percentProgress;

        private String sourceRegion;

        private String sourceDBSnapshotIdentifier;

        private String storageType;

        private String tdeCredentialArn;

        private Boolean encrypted;

        private String kmsKeyId;

        private String dbSnapshotArn;

        private String timezone;

        private Boolean iamDatabaseAuthenticationEnabled;

        private List<ProcessorFeature> processorFeatures = DefaultSdkAutoConstructList.getInstance();

        private String dbiResourceId;

        private BuilderImpl() {
        }

        private BuilderImpl(DBSnapshot model) {
            dbSnapshotIdentifier(model.dbSnapshotIdentifier);
            dbInstanceIdentifier(model.dbInstanceIdentifier);
            snapshotCreateTime(model.snapshotCreateTime);
            engine(model.engine);
            allocatedStorage(model.allocatedStorage);
            status(model.status);
            port(model.port);
            availabilityZone(model.availabilityZone);
            vpcId(model.vpcId);
            instanceCreateTime(model.instanceCreateTime);
            masterUsername(model.masterUsername);
            engineVersion(model.engineVersion);
            licenseModel(model.licenseModel);
            snapshotType(model.snapshotType);
            iops(model.iops);
            optionGroupName(model.optionGroupName);
            percentProgress(model.percentProgress);
            sourceRegion(model.sourceRegion);
            sourceDBSnapshotIdentifier(model.sourceDBSnapshotIdentifier);
            storageType(model.storageType);
            tdeCredentialArn(model.tdeCredentialArn);
            encrypted(model.encrypted);
            kmsKeyId(model.kmsKeyId);
            dbSnapshotArn(model.dbSnapshotArn);
            timezone(model.timezone);
            iamDatabaseAuthenticationEnabled(model.iamDatabaseAuthenticationEnabled);
            processorFeatures(model.processorFeatures);
            dbiResourceId(model.dbiResourceId);
        }

        public final String getDbSnapshotIdentifier() {
            return dbSnapshotIdentifier;
        }

        @Override
        public final Builder dbSnapshotIdentifier(String dbSnapshotIdentifier) {
            this.dbSnapshotIdentifier = dbSnapshotIdentifier;
            return this;
        }

        public final void setDbSnapshotIdentifier(String dbSnapshotIdentifier) {
            this.dbSnapshotIdentifier = dbSnapshotIdentifier;
        }

        public final String getDbInstanceIdentifier() {
            return dbInstanceIdentifier;
        }

        @Override
        public final Builder dbInstanceIdentifier(String dbInstanceIdentifier) {
            this.dbInstanceIdentifier = dbInstanceIdentifier;
            return this;
        }

        public final void setDbInstanceIdentifier(String dbInstanceIdentifier) {
            this.dbInstanceIdentifier = dbInstanceIdentifier;
        }

        public final Instant getSnapshotCreateTime() {
            return snapshotCreateTime;
        }

        @Override
        public final Builder snapshotCreateTime(Instant snapshotCreateTime) {
            this.snapshotCreateTime = snapshotCreateTime;
            return this;
        }

        public final void setSnapshotCreateTime(Instant snapshotCreateTime) {
            this.snapshotCreateTime = snapshotCreateTime;
        }

        public final String getEngine() {
            return engine;
        }

        @Override
        public final Builder engine(String engine) {
            this.engine = engine;
            return this;
        }

        public final void setEngine(String engine) {
            this.engine = engine;
        }

        public final Integer getAllocatedStorage() {
            return allocatedStorage;
        }

        @Override
        public final Builder allocatedStorage(Integer allocatedStorage) {
            this.allocatedStorage = allocatedStorage;
            return this;
        }

        public final void setAllocatedStorage(Integer allocatedStorage) {
            this.allocatedStorage = allocatedStorage;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Integer getPort() {
            return port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        public final String getVpcId() {
            return vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        public final Instant getInstanceCreateTime() {
            return instanceCreateTime;
        }

        @Override
        public final Builder instanceCreateTime(Instant instanceCreateTime) {
            this.instanceCreateTime = instanceCreateTime;
            return this;
        }

        public final void setInstanceCreateTime(Instant instanceCreateTime) {
            this.instanceCreateTime = instanceCreateTime;
        }

        public final String getMasterUsername() {
            return masterUsername;
        }

        @Override
        public final Builder masterUsername(String masterUsername) {
            this.masterUsername = masterUsername;
            return this;
        }

        public final void setMasterUsername(String masterUsername) {
            this.masterUsername = masterUsername;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        public final String getLicenseModel() {
            return licenseModel;
        }

        @Override
        public final Builder licenseModel(String licenseModel) {
            this.licenseModel = licenseModel;
            return this;
        }

        public final void setLicenseModel(String licenseModel) {
            this.licenseModel = licenseModel;
        }

        public final String getSnapshotType() {
            return snapshotType;
        }

        @Override
        public final Builder snapshotType(String snapshotType) {
            this.snapshotType = snapshotType;
            return this;
        }

        public final void setSnapshotType(String snapshotType) {
            this.snapshotType = snapshotType;
        }

        public final Integer getIops() {
            return iops;
        }

        @Override
        public final Builder iops(Integer iops) {
            this.iops = iops;
            return this;
        }

        public final void setIops(Integer iops) {
            this.iops = iops;
        }

        public final String getOptionGroupName() {
            return optionGroupName;
        }

        @Override
        public final Builder optionGroupName(String optionGroupName) {
            this.optionGroupName = optionGroupName;
            return this;
        }

        public final void setOptionGroupName(String optionGroupName) {
            this.optionGroupName = optionGroupName;
        }

        public final Integer getPercentProgress() {
            return percentProgress;
        }

        @Override
        public final Builder percentProgress(Integer percentProgress) {
            this.percentProgress = percentProgress;
            return this;
        }

        public final void setPercentProgress(Integer percentProgress) {
            this.percentProgress = percentProgress;
        }

        public final String getSourceRegion() {
            return sourceRegion;
        }

        @Override
        public final Builder sourceRegion(String sourceRegion) {
            this.sourceRegion = sourceRegion;
            return this;
        }

        public final void setSourceRegion(String sourceRegion) {
            this.sourceRegion = sourceRegion;
        }

        public final String getSourceDBSnapshotIdentifier() {
            return sourceDBSnapshotIdentifier;
        }

        @Override
        public final Builder sourceDBSnapshotIdentifier(String sourceDBSnapshotIdentifier) {
            this.sourceDBSnapshotIdentifier = sourceDBSnapshotIdentifier;
            return this;
        }

        public final void setSourceDBSnapshotIdentifier(String sourceDBSnapshotIdentifier) {
            this.sourceDBSnapshotIdentifier = sourceDBSnapshotIdentifier;
        }

        public final String getStorageType() {
            return storageType;
        }

        @Override
        public final Builder storageType(String storageType) {
            this.storageType = storageType;
            return this;
        }

        public final void setStorageType(String storageType) {
            this.storageType = storageType;
        }

        public final String getTdeCredentialArn() {
            return tdeCredentialArn;
        }

        @Override
        public final Builder tdeCredentialArn(String tdeCredentialArn) {
            this.tdeCredentialArn = tdeCredentialArn;
            return this;
        }

        public final void setTdeCredentialArn(String tdeCredentialArn) {
            this.tdeCredentialArn = tdeCredentialArn;
        }

        public final Boolean getEncrypted() {
            return encrypted;
        }

        @Override
        public final Builder encrypted(Boolean encrypted) {
            this.encrypted = encrypted;
            return this;
        }

        public final void setEncrypted(Boolean encrypted) {
            this.encrypted = encrypted;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        public final String getDbSnapshotArn() {
            return dbSnapshotArn;
        }

        @Override
        public final Builder dbSnapshotArn(String dbSnapshotArn) {
            this.dbSnapshotArn = dbSnapshotArn;
            return this;
        }

        public final void setDbSnapshotArn(String dbSnapshotArn) {
            this.dbSnapshotArn = dbSnapshotArn;
        }

        public final String getTimezone() {
            return timezone;
        }

        @Override
        public final Builder timezone(String timezone) {
            this.timezone = timezone;
            return this;
        }

        public final void setTimezone(String timezone) {
            this.timezone = timezone;
        }

        public final Boolean getIamDatabaseAuthenticationEnabled() {
            return iamDatabaseAuthenticationEnabled;
        }

        @Override
        public final Builder iamDatabaseAuthenticationEnabled(Boolean iamDatabaseAuthenticationEnabled) {
            this.iamDatabaseAuthenticationEnabled = iamDatabaseAuthenticationEnabled;
            return this;
        }

        public final void setIamDatabaseAuthenticationEnabled(Boolean iamDatabaseAuthenticationEnabled) {
            this.iamDatabaseAuthenticationEnabled = iamDatabaseAuthenticationEnabled;
        }

        public final Collection<ProcessorFeature.Builder> getProcessorFeatures() {
            return processorFeatures != null ? processorFeatures.stream().map(ProcessorFeature::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder processorFeatures(Collection<ProcessorFeature> processorFeatures) {
            this.processorFeatures = ProcessorFeatureListCopier.copy(processorFeatures);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder processorFeatures(ProcessorFeature... processorFeatures) {
            processorFeatures(Arrays.asList(processorFeatures));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder processorFeatures(Consumer<ProcessorFeature.Builder>... processorFeatures) {
            processorFeatures(Stream.of(processorFeatures).map(c -> ProcessorFeature.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setProcessorFeatures(Collection<ProcessorFeature.BuilderImpl> processorFeatures) {
            this.processorFeatures = ProcessorFeatureListCopier.copyFromBuilder(processorFeatures);
        }

        public final String getDbiResourceId() {
            return dbiResourceId;
        }

        @Override
        public final Builder dbiResourceId(String dbiResourceId) {
            this.dbiResourceId = dbiResourceId;
            return this;
        }

        public final void setDbiResourceId(String dbiResourceId) {
            this.dbiResourceId = dbiResourceId;
        }

        @Override
        public DBSnapshot build() {
            return new DBSnapshot(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
