/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is prerelease documentation for the RDS Database Proxy feature in preview release. It is subject to change.
 * </p>
 * </note>
 * <p>
 * The data structure representing a proxy managed by the RDS Proxy.
 * </p>
 * <p>
 * This data type is used as a response element in the <code>DescribeDBProxies</code> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DBProxy implements SdkPojo, Serializable, ToCopyableBuilder<DBProxy.Builder, DBProxy> {
    private static final SdkField<String> DB_PROXY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBProxy::dbProxyName)).setter(setter(Builder::dbProxyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBProxyName").build()).build();

    private static final SdkField<String> DB_PROXY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBProxy::dbProxyArn)).setter(setter(Builder::dbProxyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBProxyArn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBProxy::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> ENGINE_FAMILY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBProxy::engineFamily)).setter(setter(Builder::engineFamily))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineFamily").build()).build();

    private static final SdkField<List<String>> VPC_SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DBProxy::vpcSecurityGroupIds))
            .setter(setter(Builder::vpcSecurityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcSecurityGroupIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> VPC_SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DBProxy::vpcSubnetIds))
            .setter(setter(Builder::vpcSubnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcSubnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<UserAuthConfigInfo>> AUTH_FIELD = SdkField
            .<List<UserAuthConfigInfo>> builder(MarshallingType.LIST)
            .getter(getter(DBProxy::auth))
            .setter(setter(Builder::auth))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Auth").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<UserAuthConfigInfo> builder(MarshallingType.SDK_POJO)
                                            .constructor(UserAuthConfigInfo::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBProxy::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<String> ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DBProxy::endpoint)).setter(setter(Builder::endpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoint").build()).build();

    private static final SdkField<Boolean> REQUIRE_TLS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(DBProxy::requireTLS)).setter(setter(Builder::requireTLS))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequireTLS").build()).build();

    private static final SdkField<Integer> IDLE_CLIENT_TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(DBProxy::idleClientTimeout)).setter(setter(Builder::idleClientTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdleClientTimeout").build()).build();

    private static final SdkField<Boolean> DEBUG_LOGGING_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(DBProxy::debugLogging)).setter(setter(Builder::debugLogging))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DebugLogging").build()).build();

    private static final SdkField<Instant> CREATED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(DBProxy::createdDate)).setter(setter(Builder::createdDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedDate").build()).build();

    private static final SdkField<Instant> UPDATED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(DBProxy::updatedDate)).setter(setter(Builder::updatedDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdatedDate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DB_PROXY_NAME_FIELD,
            DB_PROXY_ARN_FIELD, STATUS_FIELD, ENGINE_FAMILY_FIELD, VPC_SECURITY_GROUP_IDS_FIELD, VPC_SUBNET_IDS_FIELD,
            AUTH_FIELD, ROLE_ARN_FIELD, ENDPOINT_FIELD, REQUIRE_TLS_FIELD, IDLE_CLIENT_TIMEOUT_FIELD, DEBUG_LOGGING_FIELD,
            CREATED_DATE_FIELD, UPDATED_DATE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String dbProxyName;

    private final String dbProxyArn;

    private final String status;

    private final String engineFamily;

    private final List<String> vpcSecurityGroupIds;

    private final List<String> vpcSubnetIds;

    private final List<UserAuthConfigInfo> auth;

    private final String roleArn;

    private final String endpoint;

    private final Boolean requireTLS;

    private final Integer idleClientTimeout;

    private final Boolean debugLogging;

    private final Instant createdDate;

    private final Instant updatedDate;

    private DBProxy(BuilderImpl builder) {
        this.dbProxyName = builder.dbProxyName;
        this.dbProxyArn = builder.dbProxyArn;
        this.status = builder.status;
        this.engineFamily = builder.engineFamily;
        this.vpcSecurityGroupIds = builder.vpcSecurityGroupIds;
        this.vpcSubnetIds = builder.vpcSubnetIds;
        this.auth = builder.auth;
        this.roleArn = builder.roleArn;
        this.endpoint = builder.endpoint;
        this.requireTLS = builder.requireTLS;
        this.idleClientTimeout = builder.idleClientTimeout;
        this.debugLogging = builder.debugLogging;
        this.createdDate = builder.createdDate;
        this.updatedDate = builder.updatedDate;
    }

    /**
     * <p>
     * The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in the specified
     * AWS Region.
     * </p>
     * 
     * @return The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in the
     *         specified AWS Region.
     */
    public String dbProxyName() {
        return dbProxyName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the proxy.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the proxy.
     */
    public String dbProxyArn() {
        return dbProxyArn;
    }

    /**
     * <p>
     * The current status of this proxy. A status of <code>available</code> means the proxy is ready to handle requests.
     * Other values indicate that you must wait for the proxy to be ready, or take some action to resolve an issue.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DBProxyStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of this proxy. A status of <code>available</code> means the proxy is ready to handle
     *         requests. Other values indicate that you must wait for the proxy to be ready, or take some action to
     *         resolve an issue.
     * @see DBProxyStatus
     */
    public DBProxyStatus status() {
        return DBProxyStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of this proxy. A status of <code>available</code> means the proxy is ready to handle requests.
     * Other values indicate that you must wait for the proxy to be ready, or take some action to resolve an issue.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DBProxyStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of this proxy. A status of <code>available</code> means the proxy is ready to handle
     *         requests. Other values indicate that you must wait for the proxy to be ready, or take some action to
     *         resolve an issue.
     * @see DBProxyStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The engine family applies to MySQL and PostgreSQL for both RDS and Aurora.
     * </p>
     * 
     * @return The engine family applies to MySQL and PostgreSQL for both RDS and Aurora.
     */
    public String engineFamily() {
        return engineFamily;
    }

    /**
     * Returns true if the VpcSecurityGroupIds property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasVpcSecurityGroupIds() {
        return vpcSecurityGroupIds != null && !(vpcSecurityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Provides a list of VPC security groups that the proxy belongs to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasVpcSecurityGroupIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Provides a list of VPC security groups that the proxy belongs to.
     */
    public List<String> vpcSecurityGroupIds() {
        return vpcSecurityGroupIds;
    }

    /**
     * Returns true if the VpcSubnetIds property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasVpcSubnetIds() {
        return vpcSubnetIds != null && !(vpcSubnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The EC2 subnet IDs for the proxy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasVpcSubnetIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The EC2 subnet IDs for the proxy.
     */
    public List<String> vpcSubnetIds() {
        return vpcSubnetIds;
    }

    /**
     * Returns true if the Auth property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasAuth() {
        return auth != null && !(auth instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more data structures specifying the authorization mechanism to connect to the associated RDS DB instance
     * or Aurora DB cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAuth()} to see if a value was sent in this field.
     * </p>
     * 
     * @return One or more data structures specifying the authorization mechanism to connect to the associated RDS DB
     *         instance or Aurora DB cluster.
     */
    public List<UserAuthConfigInfo> auth() {
        return auth;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the IAM role that the proxy uses to access Amazon Secrets Manager.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the IAM role that the proxy uses to access Amazon Secrets Manager.
     */
    public String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The endpoint that you can use to connect to the proxy. You include the endpoint value in the connection string
     * for a database client application.
     * </p>
     * 
     * @return The endpoint that you can use to connect to the proxy. You include the endpoint value in the connection
     *         string for a database client application.
     */
    public String endpoint() {
        return endpoint;
    }

    /**
     * <p>
     * Indicates whether Transport Layer Security (TLS) encryption is required for connections to the proxy.
     * </p>
     * 
     * @return Indicates whether Transport Layer Security (TLS) encryption is required for connections to the proxy.
     */
    public Boolean requireTLS() {
        return requireTLS;
    }

    /**
     * <p>
     * The number of seconds a connection to the proxy can have no activity before the proxy drops the client
     * connection. The proxy keeps the underlying database connection open and puts it back into the connection pool for
     * reuse by later connection requests.
     * </p>
     * <p>
     * Default: 1800 (30 minutes)
     * </p>
     * <p>
     * Constraints: 1 to 28,800
     * </p>
     * 
     * @return The number of seconds a connection to the proxy can have no activity before the proxy drops the client
     *         connection. The proxy keeps the underlying database connection open and puts it back into the connection
     *         pool for reuse by later connection requests.</p>
     *         <p>
     *         Default: 1800 (30 minutes)
     *         </p>
     *         <p>
     *         Constraints: 1 to 28,800
     */
    public Integer idleClientTimeout() {
        return idleClientTimeout;
    }

    /**
     * <p>
     * Whether the proxy includes detailed information about SQL statements in its logs. This information helps you to
     * debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug
     * information includes the text of SQL statements that you submit through the proxy. Thus, only enable this setting
     * when needed for debugging, and only when you have security measures in place to safeguard any sensitive
     * information that appears in the logs.
     * </p>
     * 
     * @return Whether the proxy includes detailed information about SQL statements in its logs. This information helps
     *         you to debug issues involving SQL behavior or the performance and scalability of the proxy connections.
     *         The debug information includes the text of SQL statements that you submit through the proxy. Thus, only
     *         enable this setting when needed for debugging, and only when you have security measures in place to
     *         safeguard any sensitive information that appears in the logs.
     */
    public Boolean debugLogging() {
        return debugLogging;
    }

    /**
     * <p>
     * The date and time when the proxy was first created.
     * </p>
     * 
     * @return The date and time when the proxy was first created.
     */
    public Instant createdDate() {
        return createdDate;
    }

    /**
     * <p>
     * The date and time when the proxy was last updated.
     * </p>
     * 
     * @return The date and time when the proxy was last updated.
     */
    public Instant updatedDate() {
        return updatedDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dbProxyName());
        hashCode = 31 * hashCode + Objects.hashCode(dbProxyArn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(engineFamily());
        hashCode = 31 * hashCode + Objects.hashCode(vpcSecurityGroupIds());
        hashCode = 31 * hashCode + Objects.hashCode(vpcSubnetIds());
        hashCode = 31 * hashCode + Objects.hashCode(auth());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(endpoint());
        hashCode = 31 * hashCode + Objects.hashCode(requireTLS());
        hashCode = 31 * hashCode + Objects.hashCode(idleClientTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(debugLogging());
        hashCode = 31 * hashCode + Objects.hashCode(createdDate());
        hashCode = 31 * hashCode + Objects.hashCode(updatedDate());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DBProxy)) {
            return false;
        }
        DBProxy other = (DBProxy) obj;
        return Objects.equals(dbProxyName(), other.dbProxyName()) && Objects.equals(dbProxyArn(), other.dbProxyArn())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(engineFamily(), other.engineFamily())
                && Objects.equals(vpcSecurityGroupIds(), other.vpcSecurityGroupIds())
                && Objects.equals(vpcSubnetIds(), other.vpcSubnetIds()) && Objects.equals(auth(), other.auth())
                && Objects.equals(roleArn(), other.roleArn()) && Objects.equals(endpoint(), other.endpoint())
                && Objects.equals(requireTLS(), other.requireTLS())
                && Objects.equals(idleClientTimeout(), other.idleClientTimeout())
                && Objects.equals(debugLogging(), other.debugLogging()) && Objects.equals(createdDate(), other.createdDate())
                && Objects.equals(updatedDate(), other.updatedDate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DBProxy").add("DBProxyName", dbProxyName()).add("DBProxyArn", dbProxyArn())
                .add("Status", statusAsString()).add("EngineFamily", engineFamily())
                .add("VpcSecurityGroupIds", vpcSecurityGroupIds()).add("VpcSubnetIds", vpcSubnetIds()).add("Auth", auth())
                .add("RoleArn", roleArn()).add("Endpoint", endpoint()).add("RequireTLS", requireTLS())
                .add("IdleClientTimeout", idleClientTimeout()).add("DebugLogging", debugLogging())
                .add("CreatedDate", createdDate()).add("UpdatedDate", updatedDate()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DBProxyName":
            return Optional.ofNullable(clazz.cast(dbProxyName()));
        case "DBProxyArn":
            return Optional.ofNullable(clazz.cast(dbProxyArn()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "EngineFamily":
            return Optional.ofNullable(clazz.cast(engineFamily()));
        case "VpcSecurityGroupIds":
            return Optional.ofNullable(clazz.cast(vpcSecurityGroupIds()));
        case "VpcSubnetIds":
            return Optional.ofNullable(clazz.cast(vpcSubnetIds()));
        case "Auth":
            return Optional.ofNullable(clazz.cast(auth()));
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "Endpoint":
            return Optional.ofNullable(clazz.cast(endpoint()));
        case "RequireTLS":
            return Optional.ofNullable(clazz.cast(requireTLS()));
        case "IdleClientTimeout":
            return Optional.ofNullable(clazz.cast(idleClientTimeout()));
        case "DebugLogging":
            return Optional.ofNullable(clazz.cast(debugLogging()));
        case "CreatedDate":
            return Optional.ofNullable(clazz.cast(createdDate()));
        case "UpdatedDate":
            return Optional.ofNullable(clazz.cast(updatedDate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DBProxy, T> g) {
        return obj -> g.apply((DBProxy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DBProxy> {
        /**
         * <p>
         * The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in the
         * specified AWS Region.
         * </p>
         * 
         * @param dbProxyName
         *        The identifier for the proxy. This name must be unique for all proxies owned by your AWS account in
         *        the specified AWS Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbProxyName(String dbProxyName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the proxy.
         * </p>
         * 
         * @param dbProxyArn
         *        The Amazon Resource Name (ARN) for the proxy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbProxyArn(String dbProxyArn);

        /**
         * <p>
         * The current status of this proxy. A status of <code>available</code> means the proxy is ready to handle
         * requests. Other values indicate that you must wait for the proxy to be ready, or take some action to resolve
         * an issue.
         * </p>
         * 
         * @param status
         *        The current status of this proxy. A status of <code>available</code> means the proxy is ready to
         *        handle requests. Other values indicate that you must wait for the proxy to be ready, or take some
         *        action to resolve an issue.
         * @see DBProxyStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DBProxyStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of this proxy. A status of <code>available</code> means the proxy is ready to handle
         * requests. Other values indicate that you must wait for the proxy to be ready, or take some action to resolve
         * an issue.
         * </p>
         * 
         * @param status
         *        The current status of this proxy. A status of <code>available</code> means the proxy is ready to
         *        handle requests. Other values indicate that you must wait for the proxy to be ready, or take some
         *        action to resolve an issue.
         * @see DBProxyStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DBProxyStatus
         */
        Builder status(DBProxyStatus status);

        /**
         * <p>
         * The engine family applies to MySQL and PostgreSQL for both RDS and Aurora.
         * </p>
         * 
         * @param engineFamily
         *        The engine family applies to MySQL and PostgreSQL for both RDS and Aurora.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineFamily(String engineFamily);

        /**
         * <p>
         * Provides a list of VPC security groups that the proxy belongs to.
         * </p>
         * 
         * @param vpcSecurityGroupIds
         *        Provides a list of VPC security groups that the proxy belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSecurityGroupIds(Collection<String> vpcSecurityGroupIds);

        /**
         * <p>
         * Provides a list of VPC security groups that the proxy belongs to.
         * </p>
         * 
         * @param vpcSecurityGroupIds
         *        Provides a list of VPC security groups that the proxy belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSecurityGroupIds(String... vpcSecurityGroupIds);

        /**
         * <p>
         * The EC2 subnet IDs for the proxy.
         * </p>
         * 
         * @param vpcSubnetIds
         *        The EC2 subnet IDs for the proxy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSubnetIds(Collection<String> vpcSubnetIds);

        /**
         * <p>
         * The EC2 subnet IDs for the proxy.
         * </p>
         * 
         * @param vpcSubnetIds
         *        The EC2 subnet IDs for the proxy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSubnetIds(String... vpcSubnetIds);

        /**
         * <p>
         * One or more data structures specifying the authorization mechanism to connect to the associated RDS DB
         * instance or Aurora DB cluster.
         * </p>
         * 
         * @param auth
         *        One or more data structures specifying the authorization mechanism to connect to the associated RDS DB
         *        instance or Aurora DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder auth(Collection<UserAuthConfigInfo> auth);

        /**
         * <p>
         * One or more data structures specifying the authorization mechanism to connect to the associated RDS DB
         * instance or Aurora DB cluster.
         * </p>
         * 
         * @param auth
         *        One or more data structures specifying the authorization mechanism to connect to the associated RDS DB
         *        instance or Aurora DB cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder auth(UserAuthConfigInfo... auth);

        /**
         * <p>
         * One or more data structures specifying the authorization mechanism to connect to the associated RDS DB
         * instance or Aurora DB cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link List<UserAuthConfigInfo>.Builder} avoiding the
         * need to create one manually via {@link List<UserAuthConfigInfo>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<UserAuthConfigInfo>.Builder#build()} is called immediately
         * and its result is passed to {@link #auth(List<UserAuthConfigInfo>)}.
         * 
         * @param auth
         *        a consumer that will call methods on {@link List<UserAuthConfigInfo>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #auth(List<UserAuthConfigInfo>)
         */
        Builder auth(Consumer<UserAuthConfigInfo.Builder>... auth);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the IAM role that the proxy uses to access Amazon Secrets Manager.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) for the IAM role that the proxy uses to access Amazon Secrets Manager.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The endpoint that you can use to connect to the proxy. You include the endpoint value in the connection
         * string for a database client application.
         * </p>
         * 
         * @param endpoint
         *        The endpoint that you can use to connect to the proxy. You include the endpoint value in the
         *        connection string for a database client application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoint(String endpoint);

        /**
         * <p>
         * Indicates whether Transport Layer Security (TLS) encryption is required for connections to the proxy.
         * </p>
         * 
         * @param requireTLS
         *        Indicates whether Transport Layer Security (TLS) encryption is required for connections to the proxy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requireTLS(Boolean requireTLS);

        /**
         * <p>
         * The number of seconds a connection to the proxy can have no activity before the proxy drops the client
         * connection. The proxy keeps the underlying database connection open and puts it back into the connection pool
         * for reuse by later connection requests.
         * </p>
         * <p>
         * Default: 1800 (30 minutes)
         * </p>
         * <p>
         * Constraints: 1 to 28,800
         * </p>
         * 
         * @param idleClientTimeout
         *        The number of seconds a connection to the proxy can have no activity before the proxy drops the client
         *        connection. The proxy keeps the underlying database connection open and puts it back into the
         *        connection pool for reuse by later connection requests.</p>
         *        <p>
         *        Default: 1800 (30 minutes)
         *        </p>
         *        <p>
         *        Constraints: 1 to 28,800
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idleClientTimeout(Integer idleClientTimeout);

        /**
         * <p>
         * Whether the proxy includes detailed information about SQL statements in its logs. This information helps you
         * to debug issues involving SQL behavior or the performance and scalability of the proxy connections. The debug
         * information includes the text of SQL statements that you submit through the proxy. Thus, only enable this
         * setting when needed for debugging, and only when you have security measures in place to safeguard any
         * sensitive information that appears in the logs.
         * </p>
         * 
         * @param debugLogging
         *        Whether the proxy includes detailed information about SQL statements in its logs. This information
         *        helps you to debug issues involving SQL behavior or the performance and scalability of the proxy
         *        connections. The debug information includes the text of SQL statements that you submit through the
         *        proxy. Thus, only enable this setting when needed for debugging, and only when you have security
         *        measures in place to safeguard any sensitive information that appears in the logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder debugLogging(Boolean debugLogging);

        /**
         * <p>
         * The date and time when the proxy was first created.
         * </p>
         * 
         * @param createdDate
         *        The date and time when the proxy was first created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdDate(Instant createdDate);

        /**
         * <p>
         * The date and time when the proxy was last updated.
         * </p>
         * 
         * @param updatedDate
         *        The date and time when the proxy was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedDate(Instant updatedDate);
    }

    static final class BuilderImpl implements Builder {
        private String dbProxyName;

        private String dbProxyArn;

        private String status;

        private String engineFamily;

        private List<String> vpcSecurityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> vpcSubnetIds = DefaultSdkAutoConstructList.getInstance();

        private List<UserAuthConfigInfo> auth = DefaultSdkAutoConstructList.getInstance();

        private String roleArn;

        private String endpoint;

        private Boolean requireTLS;

        private Integer idleClientTimeout;

        private Boolean debugLogging;

        private Instant createdDate;

        private Instant updatedDate;

        private BuilderImpl() {
        }

        private BuilderImpl(DBProxy model) {
            dbProxyName(model.dbProxyName);
            dbProxyArn(model.dbProxyArn);
            status(model.status);
            engineFamily(model.engineFamily);
            vpcSecurityGroupIds(model.vpcSecurityGroupIds);
            vpcSubnetIds(model.vpcSubnetIds);
            auth(model.auth);
            roleArn(model.roleArn);
            endpoint(model.endpoint);
            requireTLS(model.requireTLS);
            idleClientTimeout(model.idleClientTimeout);
            debugLogging(model.debugLogging);
            createdDate(model.createdDate);
            updatedDate(model.updatedDate);
        }

        public final String getDbProxyName() {
            return dbProxyName;
        }

        @Override
        public final Builder dbProxyName(String dbProxyName) {
            this.dbProxyName = dbProxyName;
            return this;
        }

        public final void setDbProxyName(String dbProxyName) {
            this.dbProxyName = dbProxyName;
        }

        public final String getDbProxyArn() {
            return dbProxyArn;
        }

        @Override
        public final Builder dbProxyArn(String dbProxyArn) {
            this.dbProxyArn = dbProxyArn;
            return this;
        }

        public final void setDbProxyArn(String dbProxyArn) {
            this.dbProxyArn = dbProxyArn;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(DBProxyStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getEngineFamily() {
            return engineFamily;
        }

        @Override
        public final Builder engineFamily(String engineFamily) {
            this.engineFamily = engineFamily;
            return this;
        }

        public final void setEngineFamily(String engineFamily) {
            this.engineFamily = engineFamily;
        }

        public final Collection<String> getVpcSecurityGroupIds() {
            return vpcSecurityGroupIds;
        }

        @Override
        public final Builder vpcSecurityGroupIds(Collection<String> vpcSecurityGroupIds) {
            this.vpcSecurityGroupIds = StringListCopier.copy(vpcSecurityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vpcSecurityGroupIds(String... vpcSecurityGroupIds) {
            vpcSecurityGroupIds(Arrays.asList(vpcSecurityGroupIds));
            return this;
        }

        public final void setVpcSecurityGroupIds(Collection<String> vpcSecurityGroupIds) {
            this.vpcSecurityGroupIds = StringListCopier.copy(vpcSecurityGroupIds);
        }

        public final Collection<String> getVpcSubnetIds() {
            return vpcSubnetIds;
        }

        @Override
        public final Builder vpcSubnetIds(Collection<String> vpcSubnetIds) {
            this.vpcSubnetIds = StringListCopier.copy(vpcSubnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vpcSubnetIds(String... vpcSubnetIds) {
            vpcSubnetIds(Arrays.asList(vpcSubnetIds));
            return this;
        }

        public final void setVpcSubnetIds(Collection<String> vpcSubnetIds) {
            this.vpcSubnetIds = StringListCopier.copy(vpcSubnetIds);
        }

        public final Collection<UserAuthConfigInfo.Builder> getAuth() {
            return auth != null ? auth.stream().map(UserAuthConfigInfo::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder auth(Collection<UserAuthConfigInfo> auth) {
            this.auth = UserAuthConfigInfoListCopier.copy(auth);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder auth(UserAuthConfigInfo... auth) {
            auth(Arrays.asList(auth));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder auth(Consumer<UserAuthConfigInfo.Builder>... auth) {
            auth(Stream.of(auth).map(c -> UserAuthConfigInfo.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setAuth(Collection<UserAuthConfigInfo.BuilderImpl> auth) {
            this.auth = UserAuthConfigInfoListCopier.copyFromBuilder(auth);
        }

        public final String getRoleArn() {
            return roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        public final String getEndpoint() {
            return endpoint;
        }

        @Override
        public final Builder endpoint(String endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        public final void setEndpoint(String endpoint) {
            this.endpoint = endpoint;
        }

        public final Boolean getRequireTLS() {
            return requireTLS;
        }

        @Override
        public final Builder requireTLS(Boolean requireTLS) {
            this.requireTLS = requireTLS;
            return this;
        }

        public final void setRequireTLS(Boolean requireTLS) {
            this.requireTLS = requireTLS;
        }

        public final Integer getIdleClientTimeout() {
            return idleClientTimeout;
        }

        @Override
        public final Builder idleClientTimeout(Integer idleClientTimeout) {
            this.idleClientTimeout = idleClientTimeout;
            return this;
        }

        public final void setIdleClientTimeout(Integer idleClientTimeout) {
            this.idleClientTimeout = idleClientTimeout;
        }

        public final Boolean getDebugLogging() {
            return debugLogging;
        }

        @Override
        public final Builder debugLogging(Boolean debugLogging) {
            this.debugLogging = debugLogging;
            return this;
        }

        public final void setDebugLogging(Boolean debugLogging) {
            this.debugLogging = debugLogging;
        }

        public final Instant getCreatedDate() {
            return createdDate;
        }

        @Override
        public final Builder createdDate(Instant createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public final void setCreatedDate(Instant createdDate) {
            this.createdDate = createdDate;
        }

        public final Instant getUpdatedDate() {
            return updatedDate;
        }

        @Override
        public final Builder updatedDate(Instant updatedDate) {
            this.updatedDate = updatedDate;
            return this;
        }

        public final void setUpdatedDate(Instant updatedDate) {
            this.updatedDate = updatedDate;
        }

        @Override
        public DBProxy build() {
            return new DBProxy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
