/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <note>
 * <p>
 * This is prerelease documentation for the RDS Database Proxy feature in preview release. It is subject to change.
 * </p>
 * </note>
 * <p>
 * Displays the settings that control the size and behavior of the connection pool associated with a
 * <code>DBProxyTarget</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConnectionPoolConfigurationInfo implements SdkPojo, Serializable,
        ToCopyableBuilder<ConnectionPoolConfigurationInfo.Builder, ConnectionPoolConfigurationInfo> {
    private static final SdkField<Integer> MAX_CONNECTIONS_PERCENT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ConnectionPoolConfigurationInfo::maxConnectionsPercent))
            .setter(setter(Builder::maxConnectionsPercent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxConnectionsPercent").build())
            .build();

    private static final SdkField<Integer> MAX_IDLE_CONNECTIONS_PERCENT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ConnectionPoolConfigurationInfo::maxIdleConnectionsPercent))
            .setter(setter(Builder::maxIdleConnectionsPercent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxIdleConnectionsPercent").build())
            .build();

    private static final SdkField<Integer> CONNECTION_BORROW_TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ConnectionPoolConfigurationInfo::connectionBorrowTimeout))
            .setter(setter(Builder::connectionBorrowTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionBorrowTimeout").build())
            .build();

    private static final SdkField<List<String>> SESSION_PINNING_FILTERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(ConnectionPoolConfigurationInfo::sessionPinningFilters))
            .setter(setter(Builder::sessionPinningFilters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SessionPinningFilters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> INIT_QUERY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ConnectionPoolConfigurationInfo::initQuery)).setter(setter(Builder::initQuery))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitQuery").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections
            .unmodifiableList(Arrays.asList(MAX_CONNECTIONS_PERCENT_FIELD, MAX_IDLE_CONNECTIONS_PERCENT_FIELD,
                    CONNECTION_BORROW_TIMEOUT_FIELD, SESSION_PINNING_FILTERS_FIELD, INIT_QUERY_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer maxConnectionsPercent;

    private final Integer maxIdleConnectionsPercent;

    private final Integer connectionBorrowTimeout;

    private final List<String> sessionPinningFilters;

    private final String initQuery;

    private ConnectionPoolConfigurationInfo(BuilderImpl builder) {
        this.maxConnectionsPercent = builder.maxConnectionsPercent;
        this.maxIdleConnectionsPercent = builder.maxIdleConnectionsPercent;
        this.connectionBorrowTimeout = builder.connectionBorrowTimeout;
        this.sessionPinningFilters = builder.sessionPinningFilters;
        this.initQuery = builder.initQuery;
    }

    /**
     * <p>
     * The maximum size of the connection pool for each target in a target group. For Aurora MySQL, it is expressed as a
     * percentage of the <code>max_connections</code> setting for the RDS DB instance or Aurora DB cluster used by the
     * target group.
     * </p>
     * 
     * @return The maximum size of the connection pool for each target in a target group. For Aurora MySQL, it is
     *         expressed as a percentage of the <code>max_connections</code> setting for the RDS DB instance or Aurora
     *         DB cluster used by the target group.
     */
    public Integer maxConnectionsPercent() {
        return maxConnectionsPercent;
    }

    /**
     * <p>
     * Controls how actively the proxy closes idle database connections in the connection pool. A high value enables the
     * proxy to leave a high percentage of idle connections open. A low value causes the proxy to close idle client
     * connections and return the underlying database connections to the connection pool. For Aurora MySQL, it is
     * expressed as a percentage of the <code>max_connections</code> setting for the RDS DB instance or Aurora DB
     * cluster used by the target group.
     * </p>
     * 
     * @return Controls how actively the proxy closes idle database connections in the connection pool. A high value
     *         enables the proxy to leave a high percentage of idle connections open. A low value causes the proxy to
     *         close idle client connections and return the underlying database connections to the connection pool. For
     *         Aurora MySQL, it is expressed as a percentage of the <code>max_connections</code> setting for the RDS DB
     *         instance or Aurora DB cluster used by the target group.
     */
    public Integer maxIdleConnectionsPercent() {
        return maxIdleConnectionsPercent;
    }

    /**
     * <p>
     * The number of seconds for a proxy to wait for a connection to become available in the connection pool. Only
     * applies when the proxy has opened its maximum number of connections and all connections are busy with client
     * sessions.
     * </p>
     * 
     * @return The number of seconds for a proxy to wait for a connection to become available in the connection pool.
     *         Only applies when the proxy has opened its maximum number of connections and all connections are busy
     *         with client sessions.
     */
    public Integer connectionBorrowTimeout() {
        return connectionBorrowTimeout;
    }

    /**
     * Returns true if the SessionPinningFilters property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasSessionPinningFilters() {
        return sessionPinningFilters != null && !(sessionPinningFilters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Each item in the list represents a class of SQL operations that normally cause all later statements in a session
     * using a proxy to be pinned to the same underlying database connection. Including an item in the list exempts that
     * class of SQL operations from the pinning behavior. Currently, the only allowed value is
     * <code>EXCLUDE_VARIABLE_SETS</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSessionPinningFilters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Each item in the list represents a class of SQL operations that normally cause all later statements in a
     *         session using a proxy to be pinned to the same underlying database connection. Including an item in the
     *         list exempts that class of SQL operations from the pinning behavior. Currently, the only allowed value is
     *         <code>EXCLUDE_VARIABLE_SETS</code>.
     */
    public List<String> sessionPinningFilters() {
        return sessionPinningFilters;
    }

    /**
     * <p>
     * One or more SQL statements for the proxy to run when opening each new database connection. Typically used with
     * <code>SET</code> statements to make sure that each connection has identical settings such as time zone and
     * character set. This setting is empty by default. For multiple statements, use semicolons as the separator. You
     * can also include multiple variables in a single <code>SET</code> statement, such as <code>SET x=1, y=2</code>.
     * </p>
     * <p>
     * <code>InitQuery</code> is not currently supported for PostgreSQL.
     * </p>
     * 
     * @return One or more SQL statements for the proxy to run when opening each new database connection. Typically used
     *         with <code>SET</code> statements to make sure that each connection has identical settings such as time
     *         zone and character set. This setting is empty by default. For multiple statements, use semicolons as the
     *         separator. You can also include multiple variables in a single <code>SET</code> statement, such as
     *         <code>SET x=1, y=2</code>. </p>
     *         <p>
     *         <code>InitQuery</code> is not currently supported for PostgreSQL.
     */
    public String initQuery() {
        return initQuery;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(maxConnectionsPercent());
        hashCode = 31 * hashCode + Objects.hashCode(maxIdleConnectionsPercent());
        hashCode = 31 * hashCode + Objects.hashCode(connectionBorrowTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(sessionPinningFilters());
        hashCode = 31 * hashCode + Objects.hashCode(initQuery());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConnectionPoolConfigurationInfo)) {
            return false;
        }
        ConnectionPoolConfigurationInfo other = (ConnectionPoolConfigurationInfo) obj;
        return Objects.equals(maxConnectionsPercent(), other.maxConnectionsPercent())
                && Objects.equals(maxIdleConnectionsPercent(), other.maxIdleConnectionsPercent())
                && Objects.equals(connectionBorrowTimeout(), other.connectionBorrowTimeout())
                && Objects.equals(sessionPinningFilters(), other.sessionPinningFilters())
                && Objects.equals(initQuery(), other.initQuery());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ConnectionPoolConfigurationInfo").add("MaxConnectionsPercent", maxConnectionsPercent())
                .add("MaxIdleConnectionsPercent", maxIdleConnectionsPercent())
                .add("ConnectionBorrowTimeout", connectionBorrowTimeout()).add("SessionPinningFilters", sessionPinningFilters())
                .add("InitQuery", initQuery()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MaxConnectionsPercent":
            return Optional.ofNullable(clazz.cast(maxConnectionsPercent()));
        case "MaxIdleConnectionsPercent":
            return Optional.ofNullable(clazz.cast(maxIdleConnectionsPercent()));
        case "ConnectionBorrowTimeout":
            return Optional.ofNullable(clazz.cast(connectionBorrowTimeout()));
        case "SessionPinningFilters":
            return Optional.ofNullable(clazz.cast(sessionPinningFilters()));
        case "InitQuery":
            return Optional.ofNullable(clazz.cast(initQuery()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ConnectionPoolConfigurationInfo, T> g) {
        return obj -> g.apply((ConnectionPoolConfigurationInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ConnectionPoolConfigurationInfo> {
        /**
         * <p>
         * The maximum size of the connection pool for each target in a target group. For Aurora MySQL, it is expressed
         * as a percentage of the <code>max_connections</code> setting for the RDS DB instance or Aurora DB cluster used
         * by the target group.
         * </p>
         * 
         * @param maxConnectionsPercent
         *        The maximum size of the connection pool for each target in a target group. For Aurora MySQL, it is
         *        expressed as a percentage of the <code>max_connections</code> setting for the RDS DB instance or
         *        Aurora DB cluster used by the target group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConnectionsPercent(Integer maxConnectionsPercent);

        /**
         * <p>
         * Controls how actively the proxy closes idle database connections in the connection pool. A high value enables
         * the proxy to leave a high percentage of idle connections open. A low value causes the proxy to close idle
         * client connections and return the underlying database connections to the connection pool. For Aurora MySQL,
         * it is expressed as a percentage of the <code>max_connections</code> setting for the RDS DB instance or Aurora
         * DB cluster used by the target group.
         * </p>
         * 
         * @param maxIdleConnectionsPercent
         *        Controls how actively the proxy closes idle database connections in the connection pool. A high value
         *        enables the proxy to leave a high percentage of idle connections open. A low value causes the proxy to
         *        close idle client connections and return the underlying database connections to the connection pool.
         *        For Aurora MySQL, it is expressed as a percentage of the <code>max_connections</code> setting for the
         *        RDS DB instance or Aurora DB cluster used by the target group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxIdleConnectionsPercent(Integer maxIdleConnectionsPercent);

        /**
         * <p>
         * The number of seconds for a proxy to wait for a connection to become available in the connection pool. Only
         * applies when the proxy has opened its maximum number of connections and all connections are busy with client
         * sessions.
         * </p>
         * 
         * @param connectionBorrowTimeout
         *        The number of seconds for a proxy to wait for a connection to become available in the connection pool.
         *        Only applies when the proxy has opened its maximum number of connections and all connections are busy
         *        with client sessions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionBorrowTimeout(Integer connectionBorrowTimeout);

        /**
         * <p>
         * Each item in the list represents a class of SQL operations that normally cause all later statements in a
         * session using a proxy to be pinned to the same underlying database connection. Including an item in the list
         * exempts that class of SQL operations from the pinning behavior. Currently, the only allowed value is
         * <code>EXCLUDE_VARIABLE_SETS</code>.
         * </p>
         * 
         * @param sessionPinningFilters
         *        Each item in the list represents a class of SQL operations that normally cause all later statements in
         *        a session using a proxy to be pinned to the same underlying database connection. Including an item in
         *        the list exempts that class of SQL operations from the pinning behavior. Currently, the only allowed
         *        value is <code>EXCLUDE_VARIABLE_SETS</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionPinningFilters(Collection<String> sessionPinningFilters);

        /**
         * <p>
         * Each item in the list represents a class of SQL operations that normally cause all later statements in a
         * session using a proxy to be pinned to the same underlying database connection. Including an item in the list
         * exempts that class of SQL operations from the pinning behavior. Currently, the only allowed value is
         * <code>EXCLUDE_VARIABLE_SETS</code>.
         * </p>
         * 
         * @param sessionPinningFilters
         *        Each item in the list represents a class of SQL operations that normally cause all later statements in
         *        a session using a proxy to be pinned to the same underlying database connection. Including an item in
         *        the list exempts that class of SQL operations from the pinning behavior. Currently, the only allowed
         *        value is <code>EXCLUDE_VARIABLE_SETS</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionPinningFilters(String... sessionPinningFilters);

        /**
         * <p>
         * One or more SQL statements for the proxy to run when opening each new database connection. Typically used
         * with <code>SET</code> statements to make sure that each connection has identical settings such as time zone
         * and character set. This setting is empty by default. For multiple statements, use semicolons as the
         * separator. You can also include multiple variables in a single <code>SET</code> statement, such as
         * <code>SET x=1, y=2</code>.
         * </p>
         * <p>
         * <code>InitQuery</code> is not currently supported for PostgreSQL.
         * </p>
         * 
         * @param initQuery
         *        One or more SQL statements for the proxy to run when opening each new database connection. Typically
         *        used with <code>SET</code> statements to make sure that each connection has identical settings such as
         *        time zone and character set. This setting is empty by default. For multiple statements, use semicolons
         *        as the separator. You can also include multiple variables in a single <code>SET</code> statement, such
         *        as <code>SET x=1, y=2</code>. </p>
         *        <p>
         *        <code>InitQuery</code> is not currently supported for PostgreSQL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initQuery(String initQuery);
    }

    static final class BuilderImpl implements Builder {
        private Integer maxConnectionsPercent;

        private Integer maxIdleConnectionsPercent;

        private Integer connectionBorrowTimeout;

        private List<String> sessionPinningFilters = DefaultSdkAutoConstructList.getInstance();

        private String initQuery;

        private BuilderImpl() {
        }

        private BuilderImpl(ConnectionPoolConfigurationInfo model) {
            maxConnectionsPercent(model.maxConnectionsPercent);
            maxIdleConnectionsPercent(model.maxIdleConnectionsPercent);
            connectionBorrowTimeout(model.connectionBorrowTimeout);
            sessionPinningFilters(model.sessionPinningFilters);
            initQuery(model.initQuery);
        }

        public final Integer getMaxConnectionsPercent() {
            return maxConnectionsPercent;
        }

        @Override
        public final Builder maxConnectionsPercent(Integer maxConnectionsPercent) {
            this.maxConnectionsPercent = maxConnectionsPercent;
            return this;
        }

        public final void setMaxConnectionsPercent(Integer maxConnectionsPercent) {
            this.maxConnectionsPercent = maxConnectionsPercent;
        }

        public final Integer getMaxIdleConnectionsPercent() {
            return maxIdleConnectionsPercent;
        }

        @Override
        public final Builder maxIdleConnectionsPercent(Integer maxIdleConnectionsPercent) {
            this.maxIdleConnectionsPercent = maxIdleConnectionsPercent;
            return this;
        }

        public final void setMaxIdleConnectionsPercent(Integer maxIdleConnectionsPercent) {
            this.maxIdleConnectionsPercent = maxIdleConnectionsPercent;
        }

        public final Integer getConnectionBorrowTimeout() {
            return connectionBorrowTimeout;
        }

        @Override
        public final Builder connectionBorrowTimeout(Integer connectionBorrowTimeout) {
            this.connectionBorrowTimeout = connectionBorrowTimeout;
            return this;
        }

        public final void setConnectionBorrowTimeout(Integer connectionBorrowTimeout) {
            this.connectionBorrowTimeout = connectionBorrowTimeout;
        }

        public final Collection<String> getSessionPinningFilters() {
            return sessionPinningFilters;
        }

        @Override
        public final Builder sessionPinningFilters(Collection<String> sessionPinningFilters) {
            this.sessionPinningFilters = StringListCopier.copy(sessionPinningFilters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sessionPinningFilters(String... sessionPinningFilters) {
            sessionPinningFilters(Arrays.asList(sessionPinningFilters));
            return this;
        }

        public final void setSessionPinningFilters(Collection<String> sessionPinningFilters) {
            this.sessionPinningFilters = StringListCopier.copy(sessionPinningFilters);
        }

        public final String getInitQuery() {
            return initQuery;
        }

        @Override
        public final Builder initQuery(String initQuery) {
            this.initQuery = initQuery;
            return this;
        }

        public final void setInitQuery(String initQuery) {
            this.initQuery = initQuery;
        }

        @Override
        public ConnectionPoolConfigurationInfo build() {
            return new ConnectionPoolConfigurationInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
