/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.protocoljsonrpc;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.protocoljsonrpc.model.AllTypesRequest;
import software.amazon.awssdk.services.protocoljsonrpc.model.AllTypesResponse;
import software.amazon.awssdk.services.protocoljsonrpc.model.EmptyModeledException;
import software.amazon.awssdk.services.protocoljsonrpc.model.FurtherNestedContainersRequest;
import software.amazon.awssdk.services.protocoljsonrpc.model.FurtherNestedContainersResponse;
import software.amazon.awssdk.services.protocoljsonrpc.model.IdempotentOperationRequest;
import software.amazon.awssdk.services.protocoljsonrpc.model.IdempotentOperationResponse;
import software.amazon.awssdk.services.protocoljsonrpc.model.ImplicitPayloadException;
import software.amazon.awssdk.services.protocoljsonrpc.model.NestedContainersRequest;
import software.amazon.awssdk.services.protocoljsonrpc.model.NestedContainersResponse;
import software.amazon.awssdk.services.protocoljsonrpc.model.OperationWithNoInputOrOutputRequest;
import software.amazon.awssdk.services.protocoljsonrpc.model.OperationWithNoInputOrOutputResponse;
import software.amazon.awssdk.services.protocoljsonrpc.model.ProtocolJsonRpcException;
import software.amazon.awssdk.services.protocoljsonrpc.transform.AllTypesRequestMarshaller;
import software.amazon.awssdk.services.protocoljsonrpc.transform.FurtherNestedContainersRequestMarshaller;
import software.amazon.awssdk.services.protocoljsonrpc.transform.IdempotentOperationRequestMarshaller;
import software.amazon.awssdk.services.protocoljsonrpc.transform.NestedContainersRequestMarshaller;
import software.amazon.awssdk.services.protocoljsonrpc.transform.OperationWithNoInputOrOutputRequestMarshaller;

/**
 * Internal implementation of {@link ProtocolJsonRpcClient}.
 *
 * @see ProtocolJsonRpcClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultProtocolJsonRpcClient implements ProtocolJsonRpcClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultProtocolJsonRpcClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Invokes the AllTypes operation.
     *
     * @param allTypesRequest
     * @return Result of the AllTypes operation returned by the service.
     * @throws EmptyModeledException
     * @throws ImplicitPayloadException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ProtocolJsonRpcException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ProtocolJsonRpcClient.AllTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/jsonrpc-2016-03-11/AllTypes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AllTypesResponse allTypes(AllTypesRequest allTypesRequest) throws EmptyModeledException, ImplicitPayloadException,
            AwsServiceException, SdkClientException, ProtocolJsonRpcException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AllTypesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AllTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AllTypesRequest, AllTypesResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(allTypesRequest)
                .withMarshaller(new AllTypesRequestMarshaller(protocolFactory)));
    }

    /**
     * Invokes the FurtherNestedContainers operation.
     *
     * @param furtherNestedContainersRequest
     * @return Result of the FurtherNestedContainers operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ProtocolJsonRpcException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ProtocolJsonRpcClient.FurtherNestedContainers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/jsonrpc-2016-03-11/FurtherNestedContainers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public FurtherNestedContainersResponse furtherNestedContainers(FurtherNestedContainersRequest furtherNestedContainersRequest)
            throws AwsServiceException, SdkClientException, ProtocolJsonRpcException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<FurtherNestedContainersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, FurtherNestedContainersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<FurtherNestedContainersRequest, FurtherNestedContainersResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(furtherNestedContainersRequest)
                .withMarshaller(new FurtherNestedContainersRequestMarshaller(protocolFactory)));
    }

    /**
     * Invokes the IdempotentOperation operation.
     *
     * @param idempotentOperationRequest
     * @return Result of the IdempotentOperation operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ProtocolJsonRpcException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ProtocolJsonRpcClient.IdempotentOperation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/jsonrpc-2016-03-11/IdempotentOperation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public IdempotentOperationResponse idempotentOperation(IdempotentOperationRequest idempotentOperationRequest)
            throws AwsServiceException, SdkClientException, ProtocolJsonRpcException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<IdempotentOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, IdempotentOperationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<IdempotentOperationRequest, IdempotentOperationResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(idempotentOperationRequest).withMarshaller(new IdempotentOperationRequestMarshaller(protocolFactory)));
    }

    /**
     * Invokes the NestedContainers operation.
     *
     * @param nestedContainersRequest
     * @return Result of the NestedContainers operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ProtocolJsonRpcException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ProtocolJsonRpcClient.NestedContainers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/jsonrpc-2016-03-11/NestedContainers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public NestedContainersResponse nestedContainers(NestedContainersRequest nestedContainersRequest) throws AwsServiceException,
            SdkClientException, ProtocolJsonRpcException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<NestedContainersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                NestedContainersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<NestedContainersRequest, NestedContainersResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(nestedContainersRequest).withMarshaller(new NestedContainersRequestMarshaller(protocolFactory)));
    }

    /**
     * Invokes the OperationWithNoInputOrOutput operation.
     *
     * @param operationWithNoInputOrOutputRequest
     * @return Result of the OperationWithNoInputOrOutput operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ProtocolJsonRpcException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ProtocolJsonRpcClient.OperationWithNoInputOrOutput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/jsonrpc-2016-03-11/OperationWithNoInputOrOutput"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public OperationWithNoInputOrOutputResponse operationWithNoInputOrOutput(
            OperationWithNoInputOrOutputRequest operationWithNoInputOrOutputRequest) throws AwsServiceException,
            SdkClientException, ProtocolJsonRpcException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<OperationWithNoInputOrOutputResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, OperationWithNoInputOrOutputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<OperationWithNoInputOrOutputRequest, OperationWithNoInputOrOutputResponse>()
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(operationWithNoInputOrOutputRequest)
                        .withMarshaller(new OperationWithNoInputOrOutputRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ProtocolJsonRpcException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImplicitPayloadException")
                                .exceptionBuilderSupplier(ImplicitPayloadException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EmptyModeledException")
                                .exceptionBuilderSupplier(EmptyModeledException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
