/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.personalize.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <important>
 * <p>
 * By default, all new solutions use automatic training. With automatic training, you incur training costs while your
 * solution is active. To avoid unnecessary costs, when you are finished you can <a
 * href="https://docs.aws.amazon.com/personalize/latest/dg/API_UpdateSolution.html">update the solution</a> to turn off
 * automatic training. For information about training costs, see <a
 * href="https://aws.amazon.com/personalize/pricing/">Amazon Personalize pricing</a>.
 * </p>
 * </important>
 * <p>
 * An object that provides information about a solution. A solution includes the custom recipe, customized parameters,
 * and trained models (Solution Versions) that Amazon Personalize uses to generate recommendations.
 * </p>
 * <p>
 * After you create a solution, you can’t change its configuration. If you need to make changes, you can <a
 * href="https://docs.aws.amazon.com/personalize/latest/dg/cloning-solution.html">clone the solution</a> with the Amazon
 * Personalize console or create a new one.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Solution implements SdkPojo, Serializable, ToCopyableBuilder<Solution.Builder, Solution> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Solution::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> SOLUTION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("solutionArn").getter(getter(Solution::solutionArn)).setter(setter(Builder::solutionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("solutionArn").build()).build();

    private static final SdkField<Boolean> PERFORM_HPO_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("performHPO").getter(getter(Solution::performHPO)).setter(setter(Builder::performHPO))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("performHPO").build()).build();

    private static final SdkField<Boolean> PERFORM_AUTO_ML_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("performAutoML").getter(getter(Solution::performAutoML)).setter(setter(Builder::performAutoML))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("performAutoML").build()).build();

    private static final SdkField<Boolean> PERFORM_AUTO_TRAINING_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("performAutoTraining").getter(getter(Solution::performAutoTraining))
            .setter(setter(Builder::performAutoTraining))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("performAutoTraining").build())
            .build();

    private static final SdkField<Boolean> PERFORM_INCREMENTAL_UPDATE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("performIncrementalUpdate").getter(getter(Solution::performIncrementalUpdate))
            .setter(setter(Builder::performIncrementalUpdate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("performIncrementalUpdate").build())
            .build();

    private static final SdkField<String> RECIPE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("recipeArn").getter(getter(Solution::recipeArn)).setter(setter(Builder::recipeArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recipeArn").build()).build();

    private static final SdkField<String> DATASET_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("datasetGroupArn").getter(getter(Solution::datasetGroupArn)).setter(setter(Builder::datasetGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("datasetGroupArn").build()).build();

    private static final SdkField<String> EVENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eventType").getter(getter(Solution::eventType)).setter(setter(Builder::eventType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventType").build()).build();

    private static final SdkField<SolutionConfig> SOLUTION_CONFIG_FIELD = SdkField
            .<SolutionConfig> builder(MarshallingType.SDK_POJO).memberName("solutionConfig")
            .getter(getter(Solution::solutionConfig)).setter(setter(Builder::solutionConfig))
            .constructor(SolutionConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("solutionConfig").build()).build();

    private static final SdkField<AutoMLResult> AUTO_ML_RESULT_FIELD = SdkField.<AutoMLResult> builder(MarshallingType.SDK_POJO)
            .memberName("autoMLResult").getter(getter(Solution::autoMLResult)).setter(setter(Builder::autoMLResult))
            .constructor(AutoMLResult::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoMLResult").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(Solution::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> CREATION_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("creationDateTime").getter(getter(Solution::creationDateTime)).setter(setter(Builder::creationDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationDateTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdatedDateTime").getter(getter(Solution::lastUpdatedDateTime))
            .setter(setter(Builder::lastUpdatedDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedDateTime").build())
            .build();

    private static final SdkField<SolutionVersionSummary> LATEST_SOLUTION_VERSION_FIELD = SdkField
            .<SolutionVersionSummary> builder(MarshallingType.SDK_POJO).memberName("latestSolutionVersion")
            .getter(getter(Solution::latestSolutionVersion)).setter(setter(Builder::latestSolutionVersion))
            .constructor(SolutionVersionSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("latestSolutionVersion").build())
            .build();

    private static final SdkField<SolutionUpdateSummary> LATEST_SOLUTION_UPDATE_FIELD = SdkField
            .<SolutionUpdateSummary> builder(MarshallingType.SDK_POJO).memberName("latestSolutionUpdate")
            .getter(getter(Solution::latestSolutionUpdate)).setter(setter(Builder::latestSolutionUpdate))
            .constructor(SolutionUpdateSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("latestSolutionUpdate").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            SOLUTION_ARN_FIELD, PERFORM_HPO_FIELD, PERFORM_AUTO_ML_FIELD, PERFORM_AUTO_TRAINING_FIELD,
            PERFORM_INCREMENTAL_UPDATE_FIELD, RECIPE_ARN_FIELD, DATASET_GROUP_ARN_FIELD, EVENT_TYPE_FIELD, SOLUTION_CONFIG_FIELD,
            AUTO_ML_RESULT_FIELD, STATUS_FIELD, CREATION_DATE_TIME_FIELD, LAST_UPDATED_DATE_TIME_FIELD,
            LATEST_SOLUTION_VERSION_FIELD, LATEST_SOLUTION_UPDATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String solutionArn;

    private final Boolean performHPO;

    private final Boolean performAutoML;

    private final Boolean performAutoTraining;

    private final Boolean performIncrementalUpdate;

    private final String recipeArn;

    private final String datasetGroupArn;

    private final String eventType;

    private final SolutionConfig solutionConfig;

    private final AutoMLResult autoMLResult;

    private final String status;

    private final Instant creationDateTime;

    private final Instant lastUpdatedDateTime;

    private final SolutionVersionSummary latestSolutionVersion;

    private final SolutionUpdateSummary latestSolutionUpdate;

    private Solution(BuilderImpl builder) {
        this.name = builder.name;
        this.solutionArn = builder.solutionArn;
        this.performHPO = builder.performHPO;
        this.performAutoML = builder.performAutoML;
        this.performAutoTraining = builder.performAutoTraining;
        this.performIncrementalUpdate = builder.performIncrementalUpdate;
        this.recipeArn = builder.recipeArn;
        this.datasetGroupArn = builder.datasetGroupArn;
        this.eventType = builder.eventType;
        this.solutionConfig = builder.solutionConfig;
        this.autoMLResult = builder.autoMLResult;
        this.status = builder.status;
        this.creationDateTime = builder.creationDateTime;
        this.lastUpdatedDateTime = builder.lastUpdatedDateTime;
        this.latestSolutionVersion = builder.latestSolutionVersion;
        this.latestSolutionUpdate = builder.latestSolutionUpdate;
    }

    /**
     * <p>
     * The name of the solution.
     * </p>
     * 
     * @return The name of the solution.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The ARN of the solution.
     * </p>
     * 
     * @return The ARN of the solution.
     */
    public final String solutionArn() {
        return solutionArn;
    }

    /**
     * <p>
     * Whether to perform hyperparameter optimization (HPO) on the chosen recipe. The default is <code>false</code>.
     * </p>
     * 
     * @return Whether to perform hyperparameter optimization (HPO) on the chosen recipe. The default is
     *         <code>false</code>.
     */
    public final Boolean performHPO() {
        return performHPO;
    }

    /**
     * <important>
     * <p>
     * We don't recommend enabling automated machine learning. Instead, match your use case to the available Amazon
     * Personalize recipes. For more information, see <a
     * href="https://docs.aws.amazon.com/personalize/latest/dg/determining-use-case.html">Determining your use case.</a>
     * </p>
     * </important>
     * <p>
     * When true, Amazon Personalize performs a search for the best USER_PERSONALIZATION recipe from the list specified
     * in the solution configuration (<code>recipeArn</code> must not be specified). When false (the default), Amazon
     * Personalize uses <code>recipeArn</code> for training.
     * </p>
     * 
     * @return <p>
     *         We don't recommend enabling automated machine learning. Instead, match your use case to the available
     *         Amazon Personalize recipes. For more information, see <a
     *         href="https://docs.aws.amazon.com/personalize/latest/dg/determining-use-case.html">Determining your use
     *         case.</a>
     *         </p>
     *         </important>
     *         <p>
     *         When true, Amazon Personalize performs a search for the best USER_PERSONALIZATION recipe from the list
     *         specified in the solution configuration (<code>recipeArn</code> must not be specified). When false (the
     *         default), Amazon Personalize uses <code>recipeArn</code> for training.
     */
    public final Boolean performAutoML() {
        return performAutoML;
    }

    /**
     * <p>
     * Specifies whether the solution automatically creates solution versions. The default is <code>True</code> and the
     * solution automatically creates new solution versions every 7 days.
     * </p>
     * <p>
     * For more information about auto training, see <a
     * href="https://docs.aws.amazon.com/personalize/latest/dg/customizing-solution-config.html">Creating and
     * configuring a solution</a>.
     * </p>
     * 
     * @return Specifies whether the solution automatically creates solution versions. The default is <code>True</code>
     *         and the solution automatically creates new solution versions every 7 days.</p>
     *         <p>
     *         For more information about auto training, see <a
     *         href="https://docs.aws.amazon.com/personalize/latest/dg/customizing-solution-config.html">Creating and
     *         configuring a solution</a>.
     */
    public final Boolean performAutoTraining() {
        return performAutoTraining;
    }

    /**
     * <p>
     * A Boolean value that indicates whether incremental training updates are performed on the model. When enabled,
     * this allows the model to learn from new data more frequently without requiring full retraining, which enables
     * near real-time personalization. This parameter is supported only for solutions that use the semantic-similarity
     * recipe
     * </p>
     * 
     * @return A Boolean value that indicates whether incremental training updates are performed on the model. When
     *         enabled, this allows the model to learn from new data more frequently without requiring full retraining,
     *         which enables near real-time personalization. This parameter is supported only for solutions that use the
     *         semantic-similarity recipe
     */
    public final Boolean performIncrementalUpdate() {
        return performIncrementalUpdate;
    }

    /**
     * <p>
     * The ARN of the recipe used to create the solution. This is required when <code>performAutoML</code> is false.
     * </p>
     * 
     * @return The ARN of the recipe used to create the solution. This is required when <code>performAutoML</code> is
     *         false.
     */
    public final String recipeArn() {
        return recipeArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the dataset group that provides the training data.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the dataset group that provides the training data.
     */
    public final String datasetGroupArn() {
        return datasetGroupArn;
    }

    /**
     * <p>
     * The event type (for example, 'click' or 'like') that is used for training the model. If no <code>eventType</code>
     * is provided, Amazon Personalize uses all interactions for training with equal weight regardless of type.
     * </p>
     * 
     * @return The event type (for example, 'click' or 'like') that is used for training the model. If no
     *         <code>eventType</code> is provided, Amazon Personalize uses all interactions for training with equal
     *         weight regardless of type.
     */
    public final String eventType() {
        return eventType;
    }

    /**
     * <p>
     * Describes the configuration properties for the solution.
     * </p>
     * 
     * @return Describes the configuration properties for the solution.
     */
    public final SolutionConfig solutionConfig() {
        return solutionConfig;
    }

    /**
     * <p>
     * When <code>performAutoML</code> is true, specifies the best recipe found.
     * </p>
     * 
     * @return When <code>performAutoML</code> is true, specifies the best recipe found.
     */
    public final AutoMLResult autoMLResult() {
        return autoMLResult;
    }

    /**
     * <p>
     * The status of the solution.
     * </p>
     * <p>
     * A solution can be in one of the following states:
     * </p>
     * <ul>
     * <li>
     * <p>
     * CREATE PENDING &gt; CREATE IN_PROGRESS &gt; ACTIVE -or- CREATE FAILED
     * </p>
     * </li>
     * <li>
     * <p>
     * DELETE PENDING &gt; DELETE IN_PROGRESS
     * </p>
     * </li>
     * </ul>
     * 
     * @return The status of the solution.</p>
     *         <p>
     *         A solution can be in one of the following states:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         CREATE PENDING &gt; CREATE IN_PROGRESS &gt; ACTIVE -or- CREATE FAILED
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DELETE PENDING &gt; DELETE IN_PROGRESS
     *         </p>
     *         </li>
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * The creation date and time (in Unix time) of the solution.
     * </p>
     * 
     * @return The creation date and time (in Unix time) of the solution.
     */
    public final Instant creationDateTime() {
        return creationDateTime;
    }

    /**
     * <p>
     * The date and time (in Unix time) that the solution was last updated.
     * </p>
     * 
     * @return The date and time (in Unix time) that the solution was last updated.
     */
    public final Instant lastUpdatedDateTime() {
        return lastUpdatedDateTime;
    }

    /**
     * <p>
     * Describes the latest version of the solution, including the status and the ARN.
     * </p>
     * 
     * @return Describes the latest version of the solution, including the status and the ARN.
     */
    public final SolutionVersionSummary latestSolutionVersion() {
        return latestSolutionVersion;
    }

    /**
     * <p>
     * Provides a summary of the latest updates to the solution.
     * </p>
     * 
     * @return Provides a summary of the latest updates to the solution.
     */
    public final SolutionUpdateSummary latestSolutionUpdate() {
        return latestSolutionUpdate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(solutionArn());
        hashCode = 31 * hashCode + Objects.hashCode(performHPO());
        hashCode = 31 * hashCode + Objects.hashCode(performAutoML());
        hashCode = 31 * hashCode + Objects.hashCode(performAutoTraining());
        hashCode = 31 * hashCode + Objects.hashCode(performIncrementalUpdate());
        hashCode = 31 * hashCode + Objects.hashCode(recipeArn());
        hashCode = 31 * hashCode + Objects.hashCode(datasetGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(eventType());
        hashCode = 31 * hashCode + Objects.hashCode(solutionConfig());
        hashCode = 31 * hashCode + Objects.hashCode(autoMLResult());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(creationDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(latestSolutionVersion());
        hashCode = 31 * hashCode + Objects.hashCode(latestSolutionUpdate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Solution)) {
            return false;
        }
        Solution other = (Solution) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(solutionArn(), other.solutionArn())
                && Objects.equals(performHPO(), other.performHPO()) && Objects.equals(performAutoML(), other.performAutoML())
                && Objects.equals(performAutoTraining(), other.performAutoTraining())
                && Objects.equals(performIncrementalUpdate(), other.performIncrementalUpdate())
                && Objects.equals(recipeArn(), other.recipeArn()) && Objects.equals(datasetGroupArn(), other.datasetGroupArn())
                && Objects.equals(eventType(), other.eventType()) && Objects.equals(solutionConfig(), other.solutionConfig())
                && Objects.equals(autoMLResult(), other.autoMLResult()) && Objects.equals(status(), other.status())
                && Objects.equals(creationDateTime(), other.creationDateTime())
                && Objects.equals(lastUpdatedDateTime(), other.lastUpdatedDateTime())
                && Objects.equals(latestSolutionVersion(), other.latestSolutionVersion())
                && Objects.equals(latestSolutionUpdate(), other.latestSolutionUpdate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Solution").add("Name", name()).add("SolutionArn", solutionArn()).add("PerformHPO", performHPO())
                .add("PerformAutoML", performAutoML()).add("PerformAutoTraining", performAutoTraining())
                .add("PerformIncrementalUpdate", performIncrementalUpdate()).add("RecipeArn", recipeArn())
                .add("DatasetGroupArn", datasetGroupArn()).add("EventType", eventType()).add("SolutionConfig", solutionConfig())
                .add("AutoMLResult", autoMLResult()).add("Status", status()).add("CreationDateTime", creationDateTime())
                .add("LastUpdatedDateTime", lastUpdatedDateTime()).add("LatestSolutionVersion", latestSolutionVersion())
                .add("LatestSolutionUpdate", latestSolutionUpdate()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "solutionArn":
            return Optional.ofNullable(clazz.cast(solutionArn()));
        case "performHPO":
            return Optional.ofNullable(clazz.cast(performHPO()));
        case "performAutoML":
            return Optional.ofNullable(clazz.cast(performAutoML()));
        case "performAutoTraining":
            return Optional.ofNullable(clazz.cast(performAutoTraining()));
        case "performIncrementalUpdate":
            return Optional.ofNullable(clazz.cast(performIncrementalUpdate()));
        case "recipeArn":
            return Optional.ofNullable(clazz.cast(recipeArn()));
        case "datasetGroupArn":
            return Optional.ofNullable(clazz.cast(datasetGroupArn()));
        case "eventType":
            return Optional.ofNullable(clazz.cast(eventType()));
        case "solutionConfig":
            return Optional.ofNullable(clazz.cast(solutionConfig()));
        case "autoMLResult":
            return Optional.ofNullable(clazz.cast(autoMLResult()));
        case "status":
            return Optional.ofNullable(clazz.cast(status()));
        case "creationDateTime":
            return Optional.ofNullable(clazz.cast(creationDateTime()));
        case "lastUpdatedDateTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedDateTime()));
        case "latestSolutionVersion":
            return Optional.ofNullable(clazz.cast(latestSolutionVersion()));
        case "latestSolutionUpdate":
            return Optional.ofNullable(clazz.cast(latestSolutionUpdate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("solutionArn", SOLUTION_ARN_FIELD);
        map.put("performHPO", PERFORM_HPO_FIELD);
        map.put("performAutoML", PERFORM_AUTO_ML_FIELD);
        map.put("performAutoTraining", PERFORM_AUTO_TRAINING_FIELD);
        map.put("performIncrementalUpdate", PERFORM_INCREMENTAL_UPDATE_FIELD);
        map.put("recipeArn", RECIPE_ARN_FIELD);
        map.put("datasetGroupArn", DATASET_GROUP_ARN_FIELD);
        map.put("eventType", EVENT_TYPE_FIELD);
        map.put("solutionConfig", SOLUTION_CONFIG_FIELD);
        map.put("autoMLResult", AUTO_ML_RESULT_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("creationDateTime", CREATION_DATE_TIME_FIELD);
        map.put("lastUpdatedDateTime", LAST_UPDATED_DATE_TIME_FIELD);
        map.put("latestSolutionVersion", LATEST_SOLUTION_VERSION_FIELD);
        map.put("latestSolutionUpdate", LATEST_SOLUTION_UPDATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Solution, T> g) {
        return obj -> g.apply((Solution) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Solution> {
        /**
         * <p>
         * The name of the solution.
         * </p>
         * 
         * @param name
         *        The name of the solution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ARN of the solution.
         * </p>
         * 
         * @param solutionArn
         *        The ARN of the solution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder solutionArn(String solutionArn);

        /**
         * <p>
         * Whether to perform hyperparameter optimization (HPO) on the chosen recipe. The default is <code>false</code>.
         * </p>
         * 
         * @param performHPO
         *        Whether to perform hyperparameter optimization (HPO) on the chosen recipe. The default is
         *        <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performHPO(Boolean performHPO);

        /**
         * <important>
         * <p>
         * We don't recommend enabling automated machine learning. Instead, match your use case to the available Amazon
         * Personalize recipes. For more information, see <a
         * href="https://docs.aws.amazon.com/personalize/latest/dg/determining-use-case.html">Determining your use
         * case.</a>
         * </p>
         * </important>
         * <p>
         * When true, Amazon Personalize performs a search for the best USER_PERSONALIZATION recipe from the list
         * specified in the solution configuration (<code>recipeArn</code> must not be specified). When false (the
         * default), Amazon Personalize uses <code>recipeArn</code> for training.
         * </p>
         * 
         * @param performAutoML
         *        <p>
         *        We don't recommend enabling automated machine learning. Instead, match your use case to the available
         *        Amazon Personalize recipes. For more information, see <a
         *        href="https://docs.aws.amazon.com/personalize/latest/dg/determining-use-case.html">Determining your
         *        use case.</a>
         *        </p>
         *        </important>
         *        <p>
         *        When true, Amazon Personalize performs a search for the best USER_PERSONALIZATION recipe from the list
         *        specified in the solution configuration (<code>recipeArn</code> must not be specified). When false
         *        (the default), Amazon Personalize uses <code>recipeArn</code> for training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performAutoML(Boolean performAutoML);

        /**
         * <p>
         * Specifies whether the solution automatically creates solution versions. The default is <code>True</code> and
         * the solution automatically creates new solution versions every 7 days.
         * </p>
         * <p>
         * For more information about auto training, see <a
         * href="https://docs.aws.amazon.com/personalize/latest/dg/customizing-solution-config.html">Creating and
         * configuring a solution</a>.
         * </p>
         * 
         * @param performAutoTraining
         *        Specifies whether the solution automatically creates solution versions. The default is
         *        <code>True</code> and the solution automatically creates new solution versions every 7 days.</p>
         *        <p>
         *        For more information about auto training, see <a
         *        href="https://docs.aws.amazon.com/personalize/latest/dg/customizing-solution-config.html">Creating and
         *        configuring a solution</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performAutoTraining(Boolean performAutoTraining);

        /**
         * <p>
         * A Boolean value that indicates whether incremental training updates are performed on the model. When enabled,
         * this allows the model to learn from new data more frequently without requiring full retraining, which enables
         * near real-time personalization. This parameter is supported only for solutions that use the
         * semantic-similarity recipe
         * </p>
         * 
         * @param performIncrementalUpdate
         *        A Boolean value that indicates whether incremental training updates are performed on the model. When
         *        enabled, this allows the model to learn from new data more frequently without requiring full
         *        retraining, which enables near real-time personalization. This parameter is supported only for
         *        solutions that use the semantic-similarity recipe
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performIncrementalUpdate(Boolean performIncrementalUpdate);

        /**
         * <p>
         * The ARN of the recipe used to create the solution. This is required when <code>performAutoML</code> is false.
         * </p>
         * 
         * @param recipeArn
         *        The ARN of the recipe used to create the solution. This is required when <code>performAutoML</code> is
         *        false.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recipeArn(String recipeArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the dataset group that provides the training data.
         * </p>
         * 
         * @param datasetGroupArn
         *        The Amazon Resource Name (ARN) of the dataset group that provides the training data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datasetGroupArn(String datasetGroupArn);

        /**
         * <p>
         * The event type (for example, 'click' or 'like') that is used for training the model. If no
         * <code>eventType</code> is provided, Amazon Personalize uses all interactions for training with equal weight
         * regardless of type.
         * </p>
         * 
         * @param eventType
         *        The event type (for example, 'click' or 'like') that is used for training the model. If no
         *        <code>eventType</code> is provided, Amazon Personalize uses all interactions for training with equal
         *        weight regardless of type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventType(String eventType);

        /**
         * <p>
         * Describes the configuration properties for the solution.
         * </p>
         * 
         * @param solutionConfig
         *        Describes the configuration properties for the solution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder solutionConfig(SolutionConfig solutionConfig);

        /**
         * <p>
         * Describes the configuration properties for the solution.
         * </p>
         * This is a convenience method that creates an instance of the {@link SolutionConfig.Builder} avoiding the need
         * to create one manually via {@link SolutionConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SolutionConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #solutionConfig(SolutionConfig)}.
         * 
         * @param solutionConfig
         *        a consumer that will call methods on {@link SolutionConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #solutionConfig(SolutionConfig)
         */
        default Builder solutionConfig(Consumer<SolutionConfig.Builder> solutionConfig) {
            return solutionConfig(SolutionConfig.builder().applyMutation(solutionConfig).build());
        }

        /**
         * <p>
         * When <code>performAutoML</code> is true, specifies the best recipe found.
         * </p>
         * 
         * @param autoMLResult
         *        When <code>performAutoML</code> is true, specifies the best recipe found.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoMLResult(AutoMLResult autoMLResult);

        /**
         * <p>
         * When <code>performAutoML</code> is true, specifies the best recipe found.
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoMLResult.Builder} avoiding the need
         * to create one manually via {@link AutoMLResult#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoMLResult.Builder#build()} is called immediately and its
         * result is passed to {@link #autoMLResult(AutoMLResult)}.
         * 
         * @param autoMLResult
         *        a consumer that will call methods on {@link AutoMLResult.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoMLResult(AutoMLResult)
         */
        default Builder autoMLResult(Consumer<AutoMLResult.Builder> autoMLResult) {
            return autoMLResult(AutoMLResult.builder().applyMutation(autoMLResult).build());
        }

        /**
         * <p>
         * The status of the solution.
         * </p>
         * <p>
         * A solution can be in one of the following states:
         * </p>
         * <ul>
         * <li>
         * <p>
         * CREATE PENDING &gt; CREATE IN_PROGRESS &gt; ACTIVE -or- CREATE FAILED
         * </p>
         * </li>
         * <li>
         * <p>
         * DELETE PENDING &gt; DELETE IN_PROGRESS
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the solution.</p>
         *        <p>
         *        A solution can be in one of the following states:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        CREATE PENDING &gt; CREATE IN_PROGRESS &gt; ACTIVE -or- CREATE FAILED
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DELETE PENDING &gt; DELETE IN_PROGRESS
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * The creation date and time (in Unix time) of the solution.
         * </p>
         * 
         * @param creationDateTime
         *        The creation date and time (in Unix time) of the solution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDateTime(Instant creationDateTime);

        /**
         * <p>
         * The date and time (in Unix time) that the solution was last updated.
         * </p>
         * 
         * @param lastUpdatedDateTime
         *        The date and time (in Unix time) that the solution was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedDateTime(Instant lastUpdatedDateTime);

        /**
         * <p>
         * Describes the latest version of the solution, including the status and the ARN.
         * </p>
         * 
         * @param latestSolutionVersion
         *        Describes the latest version of the solution, including the status and the ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latestSolutionVersion(SolutionVersionSummary latestSolutionVersion);

        /**
         * <p>
         * Describes the latest version of the solution, including the status and the ARN.
         * </p>
         * This is a convenience method that creates an instance of the {@link SolutionVersionSummary.Builder} avoiding
         * the need to create one manually via {@link SolutionVersionSummary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SolutionVersionSummary.Builder#build()} is called immediately and
         * its result is passed to {@link #latestSolutionVersion(SolutionVersionSummary)}.
         * 
         * @param latestSolutionVersion
         *        a consumer that will call methods on {@link SolutionVersionSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #latestSolutionVersion(SolutionVersionSummary)
         */
        default Builder latestSolutionVersion(Consumer<SolutionVersionSummary.Builder> latestSolutionVersion) {
            return latestSolutionVersion(SolutionVersionSummary.builder().applyMutation(latestSolutionVersion).build());
        }

        /**
         * <p>
         * Provides a summary of the latest updates to the solution.
         * </p>
         * 
         * @param latestSolutionUpdate
         *        Provides a summary of the latest updates to the solution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latestSolutionUpdate(SolutionUpdateSummary latestSolutionUpdate);

        /**
         * <p>
         * Provides a summary of the latest updates to the solution.
         * </p>
         * This is a convenience method that creates an instance of the {@link SolutionUpdateSummary.Builder} avoiding
         * the need to create one manually via {@link SolutionUpdateSummary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SolutionUpdateSummary.Builder#build()} is called immediately and
         * its result is passed to {@link #latestSolutionUpdate(SolutionUpdateSummary)}.
         * 
         * @param latestSolutionUpdate
         *        a consumer that will call methods on {@link SolutionUpdateSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #latestSolutionUpdate(SolutionUpdateSummary)
         */
        default Builder latestSolutionUpdate(Consumer<SolutionUpdateSummary.Builder> latestSolutionUpdate) {
            return latestSolutionUpdate(SolutionUpdateSummary.builder().applyMutation(latestSolutionUpdate).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String solutionArn;

        private Boolean performHPO;

        private Boolean performAutoML;

        private Boolean performAutoTraining;

        private Boolean performIncrementalUpdate;

        private String recipeArn;

        private String datasetGroupArn;

        private String eventType;

        private SolutionConfig solutionConfig;

        private AutoMLResult autoMLResult;

        private String status;

        private Instant creationDateTime;

        private Instant lastUpdatedDateTime;

        private SolutionVersionSummary latestSolutionVersion;

        private SolutionUpdateSummary latestSolutionUpdate;

        private BuilderImpl() {
        }

        private BuilderImpl(Solution model) {
            name(model.name);
            solutionArn(model.solutionArn);
            performHPO(model.performHPO);
            performAutoML(model.performAutoML);
            performAutoTraining(model.performAutoTraining);
            performIncrementalUpdate(model.performIncrementalUpdate);
            recipeArn(model.recipeArn);
            datasetGroupArn(model.datasetGroupArn);
            eventType(model.eventType);
            solutionConfig(model.solutionConfig);
            autoMLResult(model.autoMLResult);
            status(model.status);
            creationDateTime(model.creationDateTime);
            lastUpdatedDateTime(model.lastUpdatedDateTime);
            latestSolutionVersion(model.latestSolutionVersion);
            latestSolutionUpdate(model.latestSolutionUpdate);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getSolutionArn() {
            return solutionArn;
        }

        public final void setSolutionArn(String solutionArn) {
            this.solutionArn = solutionArn;
        }

        @Override
        public final Builder solutionArn(String solutionArn) {
            this.solutionArn = solutionArn;
            return this;
        }

        public final Boolean getPerformHPO() {
            return performHPO;
        }

        public final void setPerformHPO(Boolean performHPO) {
            this.performHPO = performHPO;
        }

        @Override
        public final Builder performHPO(Boolean performHPO) {
            this.performHPO = performHPO;
            return this;
        }

        public final Boolean getPerformAutoML() {
            return performAutoML;
        }

        public final void setPerformAutoML(Boolean performAutoML) {
            this.performAutoML = performAutoML;
        }

        @Override
        public final Builder performAutoML(Boolean performAutoML) {
            this.performAutoML = performAutoML;
            return this;
        }

        public final Boolean getPerformAutoTraining() {
            return performAutoTraining;
        }

        public final void setPerformAutoTraining(Boolean performAutoTraining) {
            this.performAutoTraining = performAutoTraining;
        }

        @Override
        public final Builder performAutoTraining(Boolean performAutoTraining) {
            this.performAutoTraining = performAutoTraining;
            return this;
        }

        public final Boolean getPerformIncrementalUpdate() {
            return performIncrementalUpdate;
        }

        public final void setPerformIncrementalUpdate(Boolean performIncrementalUpdate) {
            this.performIncrementalUpdate = performIncrementalUpdate;
        }

        @Override
        public final Builder performIncrementalUpdate(Boolean performIncrementalUpdate) {
            this.performIncrementalUpdate = performIncrementalUpdate;
            return this;
        }

        public final String getRecipeArn() {
            return recipeArn;
        }

        public final void setRecipeArn(String recipeArn) {
            this.recipeArn = recipeArn;
        }

        @Override
        public final Builder recipeArn(String recipeArn) {
            this.recipeArn = recipeArn;
            return this;
        }

        public final String getDatasetGroupArn() {
            return datasetGroupArn;
        }

        public final void setDatasetGroupArn(String datasetGroupArn) {
            this.datasetGroupArn = datasetGroupArn;
        }

        @Override
        public final Builder datasetGroupArn(String datasetGroupArn) {
            this.datasetGroupArn = datasetGroupArn;
            return this;
        }

        public final String getEventType() {
            return eventType;
        }

        public final void setEventType(String eventType) {
            this.eventType = eventType;
        }

        @Override
        public final Builder eventType(String eventType) {
            this.eventType = eventType;
            return this;
        }

        public final SolutionConfig.Builder getSolutionConfig() {
            return solutionConfig != null ? solutionConfig.toBuilder() : null;
        }

        public final void setSolutionConfig(SolutionConfig.BuilderImpl solutionConfig) {
            this.solutionConfig = solutionConfig != null ? solutionConfig.build() : null;
        }

        @Override
        public final Builder solutionConfig(SolutionConfig solutionConfig) {
            this.solutionConfig = solutionConfig;
            return this;
        }

        public final AutoMLResult.Builder getAutoMLResult() {
            return autoMLResult != null ? autoMLResult.toBuilder() : null;
        }

        public final void setAutoMLResult(AutoMLResult.BuilderImpl autoMLResult) {
            this.autoMLResult = autoMLResult != null ? autoMLResult.build() : null;
        }

        @Override
        public final Builder autoMLResult(AutoMLResult autoMLResult) {
            this.autoMLResult = autoMLResult;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final Instant getCreationDateTime() {
            return creationDateTime;
        }

        public final void setCreationDateTime(Instant creationDateTime) {
            this.creationDateTime = creationDateTime;
        }

        @Override
        public final Builder creationDateTime(Instant creationDateTime) {
            this.creationDateTime = creationDateTime;
            return this;
        }

        public final Instant getLastUpdatedDateTime() {
            return lastUpdatedDateTime;
        }

        public final void setLastUpdatedDateTime(Instant lastUpdatedDateTime) {
            this.lastUpdatedDateTime = lastUpdatedDateTime;
        }

        @Override
        public final Builder lastUpdatedDateTime(Instant lastUpdatedDateTime) {
            this.lastUpdatedDateTime = lastUpdatedDateTime;
            return this;
        }

        public final SolutionVersionSummary.Builder getLatestSolutionVersion() {
            return latestSolutionVersion != null ? latestSolutionVersion.toBuilder() : null;
        }

        public final void setLatestSolutionVersion(SolutionVersionSummary.BuilderImpl latestSolutionVersion) {
            this.latestSolutionVersion = latestSolutionVersion != null ? latestSolutionVersion.build() : null;
        }

        @Override
        public final Builder latestSolutionVersion(SolutionVersionSummary latestSolutionVersion) {
            this.latestSolutionVersion = latestSolutionVersion;
            return this;
        }

        public final SolutionUpdateSummary.Builder getLatestSolutionUpdate() {
            return latestSolutionUpdate != null ? latestSolutionUpdate.toBuilder() : null;
        }

        public final void setLatestSolutionUpdate(SolutionUpdateSummary.BuilderImpl latestSolutionUpdate) {
            this.latestSolutionUpdate = latestSolutionUpdate != null ? latestSolutionUpdate.build() : null;
        }

        @Override
        public final Builder latestSolutionUpdate(SolutionUpdateSummary latestSolutionUpdate) {
            this.latestSolutionUpdate = latestSolutionUpdate;
            return this;
        }

        @Override
        public Solution build() {
            return new Solution(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
