/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.personalize.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a custom algorithm.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Algorithm implements SdkPojo, Serializable, ToCopyableBuilder<Algorithm.Builder, Algorithm> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Algorithm::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ALGORITHM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Algorithm::algorithmArn)).setter(setter(Builder::algorithmArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("algorithmArn").build()).build();

    private static final SdkField<AlgorithmImage> ALGORITHM_IMAGE_FIELD = SdkField
            .<AlgorithmImage> builder(MarshallingType.SDK_POJO).getter(getter(Algorithm::algorithmImage))
            .setter(setter(Builder::algorithmImage)).constructor(AlgorithmImage::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("algorithmImage").build()).build();

    private static final SdkField<Map<String, String>> DEFAULT_HYPER_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(Algorithm::defaultHyperParameters))
            .setter(setter(Builder::defaultHyperParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("defaultHyperParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<DefaultHyperParameterRanges> DEFAULT_HYPER_PARAMETER_RANGES_FIELD = SdkField
            .<DefaultHyperParameterRanges> builder(MarshallingType.SDK_POJO)
            .getter(getter(Algorithm::defaultHyperParameterRanges))
            .setter(setter(Builder::defaultHyperParameterRanges))
            .constructor(DefaultHyperParameterRanges::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("defaultHyperParameterRanges")
                    .build()).build();

    private static final SdkField<Map<String, String>> DEFAULT_RESOURCE_CONFIG_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(Algorithm::defaultResourceConfig))
            .setter(setter(Builder::defaultResourceConfig))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("defaultResourceConfig").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> TRAINING_INPUT_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Algorithm::trainingInputMode)).setter(setter(Builder::trainingInputMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("trainingInputMode").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Algorithm::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<Instant> CREATION_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Algorithm::creationDateTime)).setter(setter(Builder::creationDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationDateTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Algorithm::lastUpdatedDateTime)).setter(setter(Builder::lastUpdatedDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedDateTime").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            ALGORITHM_ARN_FIELD, ALGORITHM_IMAGE_FIELD, DEFAULT_HYPER_PARAMETERS_FIELD, DEFAULT_HYPER_PARAMETER_RANGES_FIELD,
            DEFAULT_RESOURCE_CONFIG_FIELD, TRAINING_INPUT_MODE_FIELD, ROLE_ARN_FIELD, CREATION_DATE_TIME_FIELD,
            LAST_UPDATED_DATE_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String algorithmArn;

    private final AlgorithmImage algorithmImage;

    private final Map<String, String> defaultHyperParameters;

    private final DefaultHyperParameterRanges defaultHyperParameterRanges;

    private final Map<String, String> defaultResourceConfig;

    private final String trainingInputMode;

    private final String roleArn;

    private final Instant creationDateTime;

    private final Instant lastUpdatedDateTime;

    private Algorithm(BuilderImpl builder) {
        this.name = builder.name;
        this.algorithmArn = builder.algorithmArn;
        this.algorithmImage = builder.algorithmImage;
        this.defaultHyperParameters = builder.defaultHyperParameters;
        this.defaultHyperParameterRanges = builder.defaultHyperParameterRanges;
        this.defaultResourceConfig = builder.defaultResourceConfig;
        this.trainingInputMode = builder.trainingInputMode;
        this.roleArn = builder.roleArn;
        this.creationDateTime = builder.creationDateTime;
        this.lastUpdatedDateTime = builder.lastUpdatedDateTime;
    }

    /**
     * <p>
     * The name of the algorithm.
     * </p>
     * 
     * @return The name of the algorithm.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the algorithm.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the algorithm.
     */
    public String algorithmArn() {
        return algorithmArn;
    }

    /**
     * <p>
     * The URI of the Docker container for the algorithm image.
     * </p>
     * 
     * @return The URI of the Docker container for the algorithm image.
     */
    public AlgorithmImage algorithmImage() {
        return algorithmImage;
    }

    /**
     * Returns true if the DefaultHyperParameters property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasDefaultHyperParameters() {
        return defaultHyperParameters != null && !(defaultHyperParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the default hyperparameters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDefaultHyperParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the default hyperparameters.
     */
    public Map<String, String> defaultHyperParameters() {
        return defaultHyperParameters;
    }

    /**
     * <p>
     * Specifies the default hyperparameters, their ranges, and whether they are tunable. A tunable hyperparameter can
     * have its value determined during hyperparameter optimization (HPO).
     * </p>
     * 
     * @return Specifies the default hyperparameters, their ranges, and whether they are tunable. A tunable
     *         hyperparameter can have its value determined during hyperparameter optimization (HPO).
     */
    public DefaultHyperParameterRanges defaultHyperParameterRanges() {
        return defaultHyperParameterRanges;
    }

    /**
     * Returns true if the DefaultResourceConfig property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasDefaultResourceConfig() {
        return defaultResourceConfig != null && !(defaultResourceConfig instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the default maximum number of training jobs and parallel training jobs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDefaultResourceConfig()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the default maximum number of training jobs and parallel training jobs.
     */
    public Map<String, String> defaultResourceConfig() {
        return defaultResourceConfig;
    }

    /**
     * <p>
     * The training input mode.
     * </p>
     * 
     * @return The training input mode.
     */
    public String trainingInputMode() {
        return trainingInputMode;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the role.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the role.
     */
    public String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The date and time (in Unix time) that the algorithm was created.
     * </p>
     * 
     * @return The date and time (in Unix time) that the algorithm was created.
     */
    public Instant creationDateTime() {
        return creationDateTime;
    }

    /**
     * <p>
     * The date and time (in Unix time) that the algorithm was last updated.
     * </p>
     * 
     * @return The date and time (in Unix time) that the algorithm was last updated.
     */
    public Instant lastUpdatedDateTime() {
        return lastUpdatedDateTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(algorithmArn());
        hashCode = 31 * hashCode + Objects.hashCode(algorithmImage());
        hashCode = 31 * hashCode + Objects.hashCode(defaultHyperParameters());
        hashCode = 31 * hashCode + Objects.hashCode(defaultHyperParameterRanges());
        hashCode = 31 * hashCode + Objects.hashCode(defaultResourceConfig());
        hashCode = 31 * hashCode + Objects.hashCode(trainingInputMode());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(creationDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedDateTime());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Algorithm)) {
            return false;
        }
        Algorithm other = (Algorithm) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(algorithmArn(), other.algorithmArn())
                && Objects.equals(algorithmImage(), other.algorithmImage())
                && Objects.equals(defaultHyperParameters(), other.defaultHyperParameters())
                && Objects.equals(defaultHyperParameterRanges(), other.defaultHyperParameterRanges())
                && Objects.equals(defaultResourceConfig(), other.defaultResourceConfig())
                && Objects.equals(trainingInputMode(), other.trainingInputMode()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(creationDateTime(), other.creationDateTime())
                && Objects.equals(lastUpdatedDateTime(), other.lastUpdatedDateTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Algorithm").add("Name", name()).add("AlgorithmArn", algorithmArn())
                .add("AlgorithmImage", algorithmImage()).add("DefaultHyperParameters", defaultHyperParameters())
                .add("DefaultHyperParameterRanges", defaultHyperParameterRanges())
                .add("DefaultResourceConfig", defaultResourceConfig()).add("TrainingInputMode", trainingInputMode())
                .add("RoleArn", roleArn()).add("CreationDateTime", creationDateTime())
                .add("LastUpdatedDateTime", lastUpdatedDateTime()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "algorithmArn":
            return Optional.ofNullable(clazz.cast(algorithmArn()));
        case "algorithmImage":
            return Optional.ofNullable(clazz.cast(algorithmImage()));
        case "defaultHyperParameters":
            return Optional.ofNullable(clazz.cast(defaultHyperParameters()));
        case "defaultHyperParameterRanges":
            return Optional.ofNullable(clazz.cast(defaultHyperParameterRanges()));
        case "defaultResourceConfig":
            return Optional.ofNullable(clazz.cast(defaultResourceConfig()));
        case "trainingInputMode":
            return Optional.ofNullable(clazz.cast(trainingInputMode()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "creationDateTime":
            return Optional.ofNullable(clazz.cast(creationDateTime()));
        case "lastUpdatedDateTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedDateTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Algorithm, T> g) {
        return obj -> g.apply((Algorithm) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Algorithm> {
        /**
         * <p>
         * The name of the algorithm.
         * </p>
         * 
         * @param name
         *        The name of the algorithm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the algorithm.
         * </p>
         * 
         * @param algorithmArn
         *        The Amazon Resource Name (ARN) of the algorithm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder algorithmArn(String algorithmArn);

        /**
         * <p>
         * The URI of the Docker container for the algorithm image.
         * </p>
         * 
         * @param algorithmImage
         *        The URI of the Docker container for the algorithm image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder algorithmImage(AlgorithmImage algorithmImage);

        /**
         * <p>
         * The URI of the Docker container for the algorithm image.
         * </p>
         * This is a convenience that creates an instance of the {@link AlgorithmImage.Builder} avoiding the need to
         * create one manually via {@link AlgorithmImage#builder()}.
         *
         * When the {@link Consumer} completes, {@link AlgorithmImage.Builder#build()} is called immediately and its
         * result is passed to {@link #algorithmImage(AlgorithmImage)}.
         * 
         * @param algorithmImage
         *        a consumer that will call methods on {@link AlgorithmImage.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #algorithmImage(AlgorithmImage)
         */
        default Builder algorithmImage(Consumer<AlgorithmImage.Builder> algorithmImage) {
            return algorithmImage(AlgorithmImage.builder().applyMutation(algorithmImage).build());
        }

        /**
         * <p>
         * Specifies the default hyperparameters.
         * </p>
         * 
         * @param defaultHyperParameters
         *        Specifies the default hyperparameters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultHyperParameters(Map<String, String> defaultHyperParameters);

        /**
         * <p>
         * Specifies the default hyperparameters, their ranges, and whether they are tunable. A tunable hyperparameter
         * can have its value determined during hyperparameter optimization (HPO).
         * </p>
         * 
         * @param defaultHyperParameterRanges
         *        Specifies the default hyperparameters, their ranges, and whether they are tunable. A tunable
         *        hyperparameter can have its value determined during hyperparameter optimization (HPO).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultHyperParameterRanges(DefaultHyperParameterRanges defaultHyperParameterRanges);

        /**
         * <p>
         * Specifies the default hyperparameters, their ranges, and whether they are tunable. A tunable hyperparameter
         * can have its value determined during hyperparameter optimization (HPO).
         * </p>
         * This is a convenience that creates an instance of the {@link DefaultHyperParameterRanges.Builder} avoiding
         * the need to create one manually via {@link DefaultHyperParameterRanges#builder()}.
         *
         * When the {@link Consumer} completes, {@link DefaultHyperParameterRanges.Builder#build()} is called
         * immediately and its result is passed to {@link #defaultHyperParameterRanges(DefaultHyperParameterRanges)}.
         * 
         * @param defaultHyperParameterRanges
         *        a consumer that will call methods on {@link DefaultHyperParameterRanges.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #defaultHyperParameterRanges(DefaultHyperParameterRanges)
         */
        default Builder defaultHyperParameterRanges(Consumer<DefaultHyperParameterRanges.Builder> defaultHyperParameterRanges) {
            return defaultHyperParameterRanges(DefaultHyperParameterRanges.builder().applyMutation(defaultHyperParameterRanges)
                    .build());
        }

        /**
         * <p>
         * Specifies the default maximum number of training jobs and parallel training jobs.
         * </p>
         * 
         * @param defaultResourceConfig
         *        Specifies the default maximum number of training jobs and parallel training jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultResourceConfig(Map<String, String> defaultResourceConfig);

        /**
         * <p>
         * The training input mode.
         * </p>
         * 
         * @param trainingInputMode
         *        The training input mode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingInputMode(String trainingInputMode);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the role.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) of the role.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The date and time (in Unix time) that the algorithm was created.
         * </p>
         * 
         * @param creationDateTime
         *        The date and time (in Unix time) that the algorithm was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDateTime(Instant creationDateTime);

        /**
         * <p>
         * The date and time (in Unix time) that the algorithm was last updated.
         * </p>
         * 
         * @param lastUpdatedDateTime
         *        The date and time (in Unix time) that the algorithm was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedDateTime(Instant lastUpdatedDateTime);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String algorithmArn;

        private AlgorithmImage algorithmImage;

        private Map<String, String> defaultHyperParameters = DefaultSdkAutoConstructMap.getInstance();

        private DefaultHyperParameterRanges defaultHyperParameterRanges;

        private Map<String, String> defaultResourceConfig = DefaultSdkAutoConstructMap.getInstance();

        private String trainingInputMode;

        private String roleArn;

        private Instant creationDateTime;

        private Instant lastUpdatedDateTime;

        private BuilderImpl() {
        }

        private BuilderImpl(Algorithm model) {
            name(model.name);
            algorithmArn(model.algorithmArn);
            algorithmImage(model.algorithmImage);
            defaultHyperParameters(model.defaultHyperParameters);
            defaultHyperParameterRanges(model.defaultHyperParameterRanges);
            defaultResourceConfig(model.defaultResourceConfig);
            trainingInputMode(model.trainingInputMode);
            roleArn(model.roleArn);
            creationDateTime(model.creationDateTime);
            lastUpdatedDateTime(model.lastUpdatedDateTime);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getAlgorithmArn() {
            return algorithmArn;
        }

        @Override
        public final Builder algorithmArn(String algorithmArn) {
            this.algorithmArn = algorithmArn;
            return this;
        }

        public final void setAlgorithmArn(String algorithmArn) {
            this.algorithmArn = algorithmArn;
        }

        public final AlgorithmImage.Builder getAlgorithmImage() {
            return algorithmImage != null ? algorithmImage.toBuilder() : null;
        }

        @Override
        public final Builder algorithmImage(AlgorithmImage algorithmImage) {
            this.algorithmImage = algorithmImage;
            return this;
        }

        public final void setAlgorithmImage(AlgorithmImage.BuilderImpl algorithmImage) {
            this.algorithmImage = algorithmImage != null ? algorithmImage.build() : null;
        }

        public final Map<String, String> getDefaultHyperParameters() {
            return defaultHyperParameters;
        }

        @Override
        public final Builder defaultHyperParameters(Map<String, String> defaultHyperParameters) {
            this.defaultHyperParameters = HyperParametersCopier.copy(defaultHyperParameters);
            return this;
        }

        public final void setDefaultHyperParameters(Map<String, String> defaultHyperParameters) {
            this.defaultHyperParameters = HyperParametersCopier.copy(defaultHyperParameters);
        }

        public final DefaultHyperParameterRanges.Builder getDefaultHyperParameterRanges() {
            return defaultHyperParameterRanges != null ? defaultHyperParameterRanges.toBuilder() : null;
        }

        @Override
        public final Builder defaultHyperParameterRanges(DefaultHyperParameterRanges defaultHyperParameterRanges) {
            this.defaultHyperParameterRanges = defaultHyperParameterRanges;
            return this;
        }

        public final void setDefaultHyperParameterRanges(DefaultHyperParameterRanges.BuilderImpl defaultHyperParameterRanges) {
            this.defaultHyperParameterRanges = defaultHyperParameterRanges != null ? defaultHyperParameterRanges.build() : null;
        }

        public final Map<String, String> getDefaultResourceConfig() {
            return defaultResourceConfig;
        }

        @Override
        public final Builder defaultResourceConfig(Map<String, String> defaultResourceConfig) {
            this.defaultResourceConfig = ResourceConfigCopier.copy(defaultResourceConfig);
            return this;
        }

        public final void setDefaultResourceConfig(Map<String, String> defaultResourceConfig) {
            this.defaultResourceConfig = ResourceConfigCopier.copy(defaultResourceConfig);
        }

        public final String getTrainingInputMode() {
            return trainingInputMode;
        }

        @Override
        public final Builder trainingInputMode(String trainingInputMode) {
            this.trainingInputMode = trainingInputMode;
            return this;
        }

        public final void setTrainingInputMode(String trainingInputMode) {
            this.trainingInputMode = trainingInputMode;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        public final Instant getCreationDateTime() {
            return creationDateTime;
        }

        @Override
        public final Builder creationDateTime(Instant creationDateTime) {
            this.creationDateTime = creationDateTime;
            return this;
        }

        public final void setCreationDateTime(Instant creationDateTime) {
            this.creationDateTime = creationDateTime;
        }

        public final Instant getLastUpdatedDateTime() {
            return lastUpdatedDateTime;
        }

        @Override
        public final Builder lastUpdatedDateTime(Instant lastUpdatedDateTime) {
            this.lastUpdatedDateTime = lastUpdatedDateTime;
            return this;
        }

        public final void setLastUpdatedDateTime(Instant lastUpdatedDateTime) {
            this.lastUpdatedDateTime = lastUpdatedDateTime;
        }

        @Override
        public Algorithm build() {
            return new Algorithm(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
