/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptography.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Metadata about an Amazon Web Services Payment Cryptography key.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Key implements SdkPojo, Serializable, ToCopyableBuilder<Key.Builder, Key> {
    private static final SdkField<String> KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("KeyArn")
            .getter(getter(Key::keyArn)).setter(setter(Builder::keyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyArn").build()).build();

    private static final SdkField<KeyAttributes> KEY_ATTRIBUTES_FIELD = SdkField
            .<KeyAttributes> builder(MarshallingType.SDK_POJO).memberName("KeyAttributes").getter(getter(Key::keyAttributes))
            .setter(setter(Builder::keyAttributes)).constructor(KeyAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyAttributes").build()).build();

    private static final SdkField<String> KEY_CHECK_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyCheckValue").getter(getter(Key::keyCheckValue)).setter(setter(Builder::keyCheckValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyCheckValue").build()).build();

    private static final SdkField<String> KEY_CHECK_VALUE_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyCheckValueAlgorithm").getter(getter(Key::keyCheckValueAlgorithmAsString))
            .setter(setter(Builder::keyCheckValueAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyCheckValueAlgorithm").build())
            .build();

    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Enabled").getter(getter(Key::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled").build()).build();

    private static final SdkField<Boolean> EXPORTABLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Exportable").getter(getter(Key::exportable)).setter(setter(Builder::exportable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Exportable").build()).build();

    private static final SdkField<String> KEY_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyState").getter(getter(Key::keyStateAsString)).setter(setter(Builder::keyState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyState").build()).build();

    private static final SdkField<String> KEY_ORIGIN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyOrigin").getter(getter(Key::keyOriginAsString)).setter(setter(Builder::keyOrigin))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyOrigin").build()).build();

    private static final SdkField<Instant> CREATE_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateTimestamp").getter(getter(Key::createTimestamp)).setter(setter(Builder::createTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTimestamp").build()).build();

    private static final SdkField<Instant> USAGE_START_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("UsageStartTimestamp").getter(getter(Key::usageStartTimestamp))
            .setter(setter(Builder::usageStartTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UsageStartTimestamp").build())
            .build();

    private static final SdkField<Instant> USAGE_STOP_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("UsageStopTimestamp").getter(getter(Key::usageStopTimestamp)).setter(setter(Builder::usageStopTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UsageStopTimestamp").build())
            .build();

    private static final SdkField<Instant> DELETE_PENDING_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("DeletePendingTimestamp").getter(getter(Key::deletePendingTimestamp))
            .setter(setter(Builder::deletePendingTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeletePendingTimestamp").build())
            .build();

    private static final SdkField<Instant> DELETE_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("DeleteTimestamp").getter(getter(Key::deleteTimestamp)).setter(setter(Builder::deleteTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeleteTimestamp").build()).build();

    private static final SdkField<String> DERIVE_KEY_USAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeriveKeyUsage").getter(getter(Key::deriveKeyUsageAsString)).setter(setter(Builder::deriveKeyUsage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeriveKeyUsage").build()).build();

    private static final SdkField<String> MULTI_REGION_KEY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MultiRegionKeyType").getter(getter(Key::multiRegionKeyTypeAsString))
            .setter(setter(Builder::multiRegionKeyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MultiRegionKeyType").build())
            .build();

    private static final SdkField<String> PRIMARY_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PrimaryRegion").getter(getter(Key::primaryRegion)).setter(setter(Builder::primaryRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrimaryRegion").build()).build();

    private static final SdkField<Map<String, ReplicationStatusType>> REPLICATION_STATUS_FIELD = SdkField
            .<Map<String, ReplicationStatusType>> builder(MarshallingType.MAP)
            .memberName("ReplicationStatus")
            .getter(getter(Key::replicationStatus))
            .setter(setter(Builder::replicationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationStatus").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<ReplicationStatusType> builder(MarshallingType.SDK_POJO)
                                            .constructor(ReplicationStatusType::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Boolean> USING_DEFAULT_REPLICATION_REGIONS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("UsingDefaultReplicationRegions")
            .getter(getter(Key::usingDefaultReplicationRegions))
            .setter(setter(Builder::usingDefaultReplicationRegions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UsingDefaultReplicationRegions")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_ARN_FIELD,
            KEY_ATTRIBUTES_FIELD, KEY_CHECK_VALUE_FIELD, KEY_CHECK_VALUE_ALGORITHM_FIELD, ENABLED_FIELD, EXPORTABLE_FIELD,
            KEY_STATE_FIELD, KEY_ORIGIN_FIELD, CREATE_TIMESTAMP_FIELD, USAGE_START_TIMESTAMP_FIELD, USAGE_STOP_TIMESTAMP_FIELD,
            DELETE_PENDING_TIMESTAMP_FIELD, DELETE_TIMESTAMP_FIELD, DERIVE_KEY_USAGE_FIELD, MULTI_REGION_KEY_TYPE_FIELD,
            PRIMARY_REGION_FIELD, REPLICATION_STATUS_FIELD, USING_DEFAULT_REPLICATION_REGIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String keyArn;

    private final KeyAttributes keyAttributes;

    private final String keyCheckValue;

    private final String keyCheckValueAlgorithm;

    private final Boolean enabled;

    private final Boolean exportable;

    private final String keyState;

    private final String keyOrigin;

    private final Instant createTimestamp;

    private final Instant usageStartTimestamp;

    private final Instant usageStopTimestamp;

    private final Instant deletePendingTimestamp;

    private final Instant deleteTimestamp;

    private final String deriveKeyUsage;

    private final String multiRegionKeyType;

    private final String primaryRegion;

    private final Map<String, ReplicationStatusType> replicationStatus;

    private final Boolean usingDefaultReplicationRegions;

    private Key(BuilderImpl builder) {
        this.keyArn = builder.keyArn;
        this.keyAttributes = builder.keyAttributes;
        this.keyCheckValue = builder.keyCheckValue;
        this.keyCheckValueAlgorithm = builder.keyCheckValueAlgorithm;
        this.enabled = builder.enabled;
        this.exportable = builder.exportable;
        this.keyState = builder.keyState;
        this.keyOrigin = builder.keyOrigin;
        this.createTimestamp = builder.createTimestamp;
        this.usageStartTimestamp = builder.usageStartTimestamp;
        this.usageStopTimestamp = builder.usageStopTimestamp;
        this.deletePendingTimestamp = builder.deletePendingTimestamp;
        this.deleteTimestamp = builder.deleteTimestamp;
        this.deriveKeyUsage = builder.deriveKeyUsage;
        this.multiRegionKeyType = builder.multiRegionKeyType;
        this.primaryRegion = builder.primaryRegion;
        this.replicationStatus = builder.replicationStatus;
        this.usingDefaultReplicationRegions = builder.usingDefaultReplicationRegions;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the key.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the key.
     */
    public final String keyArn() {
        return keyArn;
    }

    /**
     * <p>
     * The role of the key, the algorithm it supports, and the cryptographic operations allowed with the key. This data
     * is immutable after the key is created.
     * </p>
     * 
     * @return The role of the key, the algorithm it supports, and the cryptographic operations allowed with the key.
     *         This data is immutable after the key is created.
     */
    public final KeyAttributes keyAttributes() {
        return keyAttributes;
    }

    /**
     * <p>
     * The key check value (KCV) is used to check if all parties holding a given key have the same key or to detect that
     * a key has changed.
     * </p>
     * 
     * @return The key check value (KCV) is used to check if all parties holding a given key have the same key or to
     *         detect that a key has changed.
     */
    public final String keyCheckValue() {
        return keyCheckValue;
    }

    /**
     * <p>
     * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It is
     * used to validate the key integrity.
     * </p>
     * <p>
     * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked and
     * retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a CMAC
     * algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the encrypted
     * result.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #keyCheckValueAlgorithm} will return {@link KeyCheckValueAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #keyCheckValueAlgorithmAsString}.
     * </p>
     * 
     * @return The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV).
     *         It is used to validate the key integrity.</p>
     *         <p>
     *         For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
     *         checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
     *         computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
     *         order bytes of the encrypted result.
     * @see KeyCheckValueAlgorithm
     */
    public final KeyCheckValueAlgorithm keyCheckValueAlgorithm() {
        return KeyCheckValueAlgorithm.fromValue(keyCheckValueAlgorithm);
    }

    /**
     * <p>
     * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It is
     * used to validate the key integrity.
     * </p>
     * <p>
     * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked and
     * retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a CMAC
     * algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the encrypted
     * result.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #keyCheckValueAlgorithm} will return {@link KeyCheckValueAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #keyCheckValueAlgorithmAsString}.
     * </p>
     * 
     * @return The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV).
     *         It is used to validate the key integrity.</p>
     *         <p>
     *         For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
     *         checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
     *         computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
     *         order bytes of the encrypted result.
     * @see KeyCheckValueAlgorithm
     */
    public final String keyCheckValueAlgorithmAsString() {
        return keyCheckValueAlgorithm;
    }

    /**
     * <p>
     * Specifies whether the key is enabled.
     * </p>
     * 
     * @return Specifies whether the key is enabled.
     */
    public final Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * Specifies whether the key is exportable. This data is immutable after the key is created.
     * </p>
     * 
     * @return Specifies whether the key is exportable. This data is immutable after the key is created.
     */
    public final Boolean exportable() {
        return exportable;
    }

    /**
     * <p>
     * The state of key that is being created or deleted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyState} will
     * return {@link KeyState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyStateAsString}.
     * </p>
     * 
     * @return The state of key that is being created or deleted.
     * @see KeyState
     */
    public final KeyState keyState() {
        return KeyState.fromValue(keyState);
    }

    /**
     * <p>
     * The state of key that is being created or deleted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyState} will
     * return {@link KeyState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyStateAsString}.
     * </p>
     * 
     * @return The state of key that is being created or deleted.
     * @see KeyState
     */
    public final String keyStateAsString() {
        return keyState;
    }

    /**
     * <p>
     * The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the value is
     * <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment Cryptography, the value
     * is <code>EXTERNAL</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyOrigin} will
     * return {@link KeyOrigin#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyOriginAsString}.
     * </p>
     * 
     * @return The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the
     *         value is <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment
     *         Cryptography, the value is <code>EXTERNAL</code>.
     * @see KeyOrigin
     */
    public final KeyOrigin keyOrigin() {
        return KeyOrigin.fromValue(keyOrigin);
    }

    /**
     * <p>
     * The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the value is
     * <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment Cryptography, the value
     * is <code>EXTERNAL</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyOrigin} will
     * return {@link KeyOrigin#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyOriginAsString}.
     * </p>
     * 
     * @return The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the
     *         value is <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment
     *         Cryptography, the value is <code>EXTERNAL</code>.
     * @see KeyOrigin
     */
    public final String keyOriginAsString() {
        return keyOrigin;
    }

    /**
     * <p>
     * The date and time when the key was created.
     * </p>
     * 
     * @return The date and time when the key was created.
     */
    public final Instant createTimestamp() {
        return createTimestamp;
    }

    /**
     * <p>
     * The date and time after which Amazon Web Services Payment Cryptography will start using the key material for
     * cryptographic operations.
     * </p>
     * 
     * @return The date and time after which Amazon Web Services Payment Cryptography will start using the key material
     *         for cryptographic operations.
     */
    public final Instant usageStartTimestamp() {
        return usageStartTimestamp;
    }

    /**
     * <p>
     * The date and time after which Amazon Web Services Payment Cryptography will stop using the key material for
     * cryptographic operations.
     * </p>
     * 
     * @return The date and time after which Amazon Web Services Payment Cryptography will stop using the key material
     *         for cryptographic operations.
     */
    public final Instant usageStopTimestamp() {
        return usageStopTimestamp;
    }

    /**
     * <p>
     * The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value is present
     * only when <code>KeyState</code> is <code>DELETE_PENDING</code> and the key is scheduled for deletion.
     * </p>
     * 
     * @return The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value is
     *         present only when <code>KeyState</code> is <code>DELETE_PENDING</code> and the key is scheduled for
     *         deletion.
     */
    public final Instant deletePendingTimestamp() {
        return deletePendingTimestamp;
    }

    /**
     * <p>
     * The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value is present
     * only when when the <code>KeyState</code> is <code>DELETE_COMPLETE</code> and the Amazon Web Services Payment
     * Cryptography key is deleted.
     * </p>
     * 
     * @return The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value is
     *         present only when when the <code>KeyState</code> is <code>DELETE_COMPLETE</code> and the Amazon Web
     *         Services Payment Cryptography key is deleted.
     */
    public final Instant deleteTimestamp() {
        return deleteTimestamp;
    }

    /**
     * <p>
     * The cryptographic usage of an ECDH derived key as deﬁned in section A.5.2 of the TR-31 spec.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deriveKeyUsage}
     * will return {@link DeriveKeyUsage#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #deriveKeyUsageAsString}.
     * </p>
     * 
     * @return The cryptographic usage of an ECDH derived key as deﬁned in section A.5.2 of the TR-31 spec.
     * @see DeriveKeyUsage
     */
    public final DeriveKeyUsage deriveKeyUsage() {
        return DeriveKeyUsage.fromValue(deriveKeyUsage);
    }

    /**
     * <p>
     * The cryptographic usage of an ECDH derived key as deﬁned in section A.5.2 of the TR-31 spec.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deriveKeyUsage}
     * will return {@link DeriveKeyUsage#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #deriveKeyUsageAsString}.
     * </p>
     * 
     * @return The cryptographic usage of an ECDH derived key as deﬁned in section A.5.2 of the TR-31 spec.
     * @see DeriveKeyUsage
     */
    public final String deriveKeyUsageAsString() {
        return deriveKeyUsage;
    }

    /**
     * <p>
     * Indicates whether this key is a Multi-Region key and its role in the Multi-Region key hierarchy.
     * </p>
     * <p>
     * Multi-Region replication keys allow the same key material to be used across multiple Amazon Web Services Regions.
     * This field specifies whether the key is a Primary Region key (PRK) (which can be replicated to other Amazon Web
     * Services Regions) or a Replica Region key (RRK) (which is a copy of a PRK in another Region). For more
     * information, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
     * >Multi-Region key replication</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #multiRegionKeyType} will return {@link MultiRegionKeyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #multiRegionKeyTypeAsString}.
     * </p>
     * 
     * @return Indicates whether this key is a Multi-Region key and its role in the Multi-Region key hierarchy.</p>
     *         <p>
     *         Multi-Region replication keys allow the same key material to be used across multiple Amazon Web Services
     *         Regions. This field specifies whether the key is a Primary Region key (PRK) (which can be replicated to
     *         other Amazon Web Services Regions) or a Replica Region key (RRK) (which is a copy of a PRK in another
     *         Region). For more information, see <a href=
     *         "https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
     *         >Multi-Region key replication</a>.
     * @see MultiRegionKeyType
     */
    public final MultiRegionKeyType multiRegionKeyType() {
        return MultiRegionKeyType.fromValue(multiRegionKeyType);
    }

    /**
     * <p>
     * Indicates whether this key is a Multi-Region key and its role in the Multi-Region key hierarchy.
     * </p>
     * <p>
     * Multi-Region replication keys allow the same key material to be used across multiple Amazon Web Services Regions.
     * This field specifies whether the key is a Primary Region key (PRK) (which can be replicated to other Amazon Web
     * Services Regions) or a Replica Region key (RRK) (which is a copy of a PRK in another Region). For more
     * information, see <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
     * >Multi-Region key replication</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #multiRegionKeyType} will return {@link MultiRegionKeyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #multiRegionKeyTypeAsString}.
     * </p>
     * 
     * @return Indicates whether this key is a Multi-Region key and its role in the Multi-Region key hierarchy.</p>
     *         <p>
     *         Multi-Region replication keys allow the same key material to be used across multiple Amazon Web Services
     *         Regions. This field specifies whether the key is a Primary Region key (PRK) (which can be replicated to
     *         other Amazon Web Services Regions) or a Replica Region key (RRK) (which is a copy of a PRK in another
     *         Region). For more information, see <a href=
     *         "https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
     *         >Multi-Region key replication</a>.
     * @see MultiRegionKeyType
     */
    public final String multiRegionKeyTypeAsString() {
        return multiRegionKeyType;
    }

    /**
     * Returns the value of the PrimaryRegion property for this object.
     * 
     * @return The value of the PrimaryRegion property for this object.
     */
    public final String primaryRegion() {
        return primaryRegion;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReplicationStatus property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasReplicationStatus() {
        return replicationStatus != null && !(replicationStatus instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Information about the replication status of the key across different Amazon Web Services Regions.
     * </p>
     * <p>
     * This field provides details about the current state of key replication, including any status messages or
     * operational information. It helps track the progress and health of key replication operations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReplicationStatus} method.
     * </p>
     * 
     * @return Information about the replication status of the key across different Amazon Web Services Regions.</p>
     *         <p>
     *         This field provides details about the current state of key replication, including any status messages or
     *         operational information. It helps track the progress and health of key replication operations.
     */
    public final Map<String, ReplicationStatusType> replicationStatus() {
        return replicationStatus;
    }

    /**
     * <p>
     * Indicates whether this key is using the account's default replication regions configuration for <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
     * >Multi-Region key replication</a>.
     * </p>
     * <p>
     * When set to <code>true</code>, the key automatically replicates to the regions specified in the account's default
     * replication settings. When set to <code>false</code>, the key has a custom replication configuration that
     * overrides the account defaults.
     * </p>
     * 
     * @return Indicates whether this key is using the account's default replication regions configuration for <a
     *         href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
     *         >Multi-Region key replication</a>.</p>
     *         <p>
     *         When set to <code>true</code>, the key automatically replicates to the regions specified in the account's
     *         default replication settings. When set to <code>false</code>, the key has a custom replication
     *         configuration that overrides the account defaults.
     */
    public final Boolean usingDefaultReplicationRegions() {
        return usingDefaultReplicationRegions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(keyArn());
        hashCode = 31 * hashCode + Objects.hashCode(keyAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(keyCheckValue());
        hashCode = 31 * hashCode + Objects.hashCode(keyCheckValueAlgorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(exportable());
        hashCode = 31 * hashCode + Objects.hashCode(keyStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(keyOriginAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(usageStartTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(usageStopTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(deletePendingTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(deleteTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(deriveKeyUsageAsString());
        hashCode = 31 * hashCode + Objects.hashCode(multiRegionKeyTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(primaryRegion());
        hashCode = 31 * hashCode + Objects.hashCode(hasReplicationStatus() ? replicationStatus() : null);
        hashCode = 31 * hashCode + Objects.hashCode(usingDefaultReplicationRegions());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Key)) {
            return false;
        }
        Key other = (Key) obj;
        return Objects.equals(keyArn(), other.keyArn()) && Objects.equals(keyAttributes(), other.keyAttributes())
                && Objects.equals(keyCheckValue(), other.keyCheckValue())
                && Objects.equals(keyCheckValueAlgorithmAsString(), other.keyCheckValueAlgorithmAsString())
                && Objects.equals(enabled(), other.enabled()) && Objects.equals(exportable(), other.exportable())
                && Objects.equals(keyStateAsString(), other.keyStateAsString())
                && Objects.equals(keyOriginAsString(), other.keyOriginAsString())
                && Objects.equals(createTimestamp(), other.createTimestamp())
                && Objects.equals(usageStartTimestamp(), other.usageStartTimestamp())
                && Objects.equals(usageStopTimestamp(), other.usageStopTimestamp())
                && Objects.equals(deletePendingTimestamp(), other.deletePendingTimestamp())
                && Objects.equals(deleteTimestamp(), other.deleteTimestamp())
                && Objects.equals(deriveKeyUsageAsString(), other.deriveKeyUsageAsString())
                && Objects.equals(multiRegionKeyTypeAsString(), other.multiRegionKeyTypeAsString())
                && Objects.equals(primaryRegion(), other.primaryRegion())
                && hasReplicationStatus() == other.hasReplicationStatus()
                && Objects.equals(replicationStatus(), other.replicationStatus())
                && Objects.equals(usingDefaultReplicationRegions(), other.usingDefaultReplicationRegions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Key").add("KeyArn", keyArn()).add("KeyAttributes", keyAttributes())
                .add("KeyCheckValue", keyCheckValue()).add("KeyCheckValueAlgorithm", keyCheckValueAlgorithmAsString())
                .add("Enabled", enabled()).add("Exportable", exportable()).add("KeyState", keyStateAsString())
                .add("KeyOrigin", keyOriginAsString()).add("CreateTimestamp", createTimestamp())
                .add("UsageStartTimestamp", usageStartTimestamp()).add("UsageStopTimestamp", usageStopTimestamp())
                .add("DeletePendingTimestamp", deletePendingTimestamp()).add("DeleteTimestamp", deleteTimestamp())
                .add("DeriveKeyUsage", deriveKeyUsageAsString()).add("MultiRegionKeyType", multiRegionKeyTypeAsString())
                .add("PrimaryRegion", primaryRegion())
                .add("ReplicationStatus", hasReplicationStatus() ? replicationStatus() : null)
                .add("UsingDefaultReplicationRegions", usingDefaultReplicationRegions()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "KeyArn":
            return Optional.ofNullable(clazz.cast(keyArn()));
        case "KeyAttributes":
            return Optional.ofNullable(clazz.cast(keyAttributes()));
        case "KeyCheckValue":
            return Optional.ofNullable(clazz.cast(keyCheckValue()));
        case "KeyCheckValueAlgorithm":
            return Optional.ofNullable(clazz.cast(keyCheckValueAlgorithmAsString()));
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "Exportable":
            return Optional.ofNullable(clazz.cast(exportable()));
        case "KeyState":
            return Optional.ofNullable(clazz.cast(keyStateAsString()));
        case "KeyOrigin":
            return Optional.ofNullable(clazz.cast(keyOriginAsString()));
        case "CreateTimestamp":
            return Optional.ofNullable(clazz.cast(createTimestamp()));
        case "UsageStartTimestamp":
            return Optional.ofNullable(clazz.cast(usageStartTimestamp()));
        case "UsageStopTimestamp":
            return Optional.ofNullable(clazz.cast(usageStopTimestamp()));
        case "DeletePendingTimestamp":
            return Optional.ofNullable(clazz.cast(deletePendingTimestamp()));
        case "DeleteTimestamp":
            return Optional.ofNullable(clazz.cast(deleteTimestamp()));
        case "DeriveKeyUsage":
            return Optional.ofNullable(clazz.cast(deriveKeyUsageAsString()));
        case "MultiRegionKeyType":
            return Optional.ofNullable(clazz.cast(multiRegionKeyTypeAsString()));
        case "PrimaryRegion":
            return Optional.ofNullable(clazz.cast(primaryRegion()));
        case "ReplicationStatus":
            return Optional.ofNullable(clazz.cast(replicationStatus()));
        case "UsingDefaultReplicationRegions":
            return Optional.ofNullable(clazz.cast(usingDefaultReplicationRegions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("KeyArn", KEY_ARN_FIELD);
        map.put("KeyAttributes", KEY_ATTRIBUTES_FIELD);
        map.put("KeyCheckValue", KEY_CHECK_VALUE_FIELD);
        map.put("KeyCheckValueAlgorithm", KEY_CHECK_VALUE_ALGORITHM_FIELD);
        map.put("Enabled", ENABLED_FIELD);
        map.put("Exportable", EXPORTABLE_FIELD);
        map.put("KeyState", KEY_STATE_FIELD);
        map.put("KeyOrigin", KEY_ORIGIN_FIELD);
        map.put("CreateTimestamp", CREATE_TIMESTAMP_FIELD);
        map.put("UsageStartTimestamp", USAGE_START_TIMESTAMP_FIELD);
        map.put("UsageStopTimestamp", USAGE_STOP_TIMESTAMP_FIELD);
        map.put("DeletePendingTimestamp", DELETE_PENDING_TIMESTAMP_FIELD);
        map.put("DeleteTimestamp", DELETE_TIMESTAMP_FIELD);
        map.put("DeriveKeyUsage", DERIVE_KEY_USAGE_FIELD);
        map.put("MultiRegionKeyType", MULTI_REGION_KEY_TYPE_FIELD);
        map.put("PrimaryRegion", PRIMARY_REGION_FIELD);
        map.put("ReplicationStatus", REPLICATION_STATUS_FIELD);
        map.put("UsingDefaultReplicationRegions", USING_DEFAULT_REPLICATION_REGIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Key, T> g) {
        return obj -> g.apply((Key) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Key> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the key.
         * </p>
         * 
         * @param keyArn
         *        The Amazon Resource Name (ARN) of the key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyArn(String keyArn);

        /**
         * <p>
         * The role of the key, the algorithm it supports, and the cryptographic operations allowed with the key. This
         * data is immutable after the key is created.
         * </p>
         * 
         * @param keyAttributes
         *        The role of the key, the algorithm it supports, and the cryptographic operations allowed with the key.
         *        This data is immutable after the key is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyAttributes(KeyAttributes keyAttributes);

        /**
         * <p>
         * The role of the key, the algorithm it supports, and the cryptographic operations allowed with the key. This
         * data is immutable after the key is created.
         * </p>
         * This is a convenience method that creates an instance of the {@link KeyAttributes.Builder} avoiding the need
         * to create one manually via {@link KeyAttributes#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KeyAttributes.Builder#build()} is called immediately and its
         * result is passed to {@link #keyAttributes(KeyAttributes)}.
         * 
         * @param keyAttributes
         *        a consumer that will call methods on {@link KeyAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #keyAttributes(KeyAttributes)
         */
        default Builder keyAttributes(Consumer<KeyAttributes.Builder> keyAttributes) {
            return keyAttributes(KeyAttributes.builder().applyMutation(keyAttributes).build());
        }

        /**
         * <p>
         * The key check value (KCV) is used to check if all parties holding a given key have the same key or to detect
         * that a key has changed.
         * </p>
         * 
         * @param keyCheckValue
         *        The key check value (KCV) is used to check if all parties holding a given key have the same key or to
         *        detect that a key has changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyCheckValue(String keyCheckValue);

        /**
         * <p>
         * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It
         * is used to validate the key integrity.
         * </p>
         * <p>
         * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked
         * and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a
         * CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the
         * encrypted result.
         * </p>
         * 
         * @param keyCheckValueAlgorithm
         *        The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value
         *        (KCV). It is used to validate the key integrity.</p>
         *        <p>
         *        For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
         *        checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
         *        computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
         *        order bytes of the encrypted result.
         * @see KeyCheckValueAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyCheckValueAlgorithm
         */
        Builder keyCheckValueAlgorithm(String keyCheckValueAlgorithm);

        /**
         * <p>
         * The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value (KCV). It
         * is used to validate the key integrity.
         * </p>
         * <p>
         * For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be checked
         * and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is computed using a
         * CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest order bytes of the
         * encrypted result.
         * </p>
         * 
         * @param keyCheckValueAlgorithm
         *        The algorithm that Amazon Web Services Payment Cryptography uses to calculate the key check value
         *        (KCV). It is used to validate the key integrity.</p>
         *        <p>
         *        For TDES keys, the KCV is computed by encrypting 8 bytes, each with value of zero, with the key to be
         *        checked and retaining the 3 highest order bytes of the encrypted result. For AES keys, the KCV is
         *        computed using a CMAC algorithm where the input data is 16 bytes of zero and retaining the 3 highest
         *        order bytes of the encrypted result.
         * @see KeyCheckValueAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyCheckValueAlgorithm
         */
        Builder keyCheckValueAlgorithm(KeyCheckValueAlgorithm keyCheckValueAlgorithm);

        /**
         * <p>
         * Specifies whether the key is enabled.
         * </p>
         * 
         * @param enabled
         *        Specifies whether the key is enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * Specifies whether the key is exportable. This data is immutable after the key is created.
         * </p>
         * 
         * @param exportable
         *        Specifies whether the key is exportable. This data is immutable after the key is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportable(Boolean exportable);

        /**
         * <p>
         * The state of key that is being created or deleted.
         * </p>
         * 
         * @param keyState
         *        The state of key that is being created or deleted.
         * @see KeyState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyState
         */
        Builder keyState(String keyState);

        /**
         * <p>
         * The state of key that is being created or deleted.
         * </p>
         * 
         * @param keyState
         *        The state of key that is being created or deleted.
         * @see KeyState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyState
         */
        Builder keyState(KeyState keyState);

        /**
         * <p>
         * The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the value
         * is <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment Cryptography,
         * the value is <code>EXTERNAL</code>.
         * </p>
         * 
         * @param keyOrigin
         *        The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the
         *        value is <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment
         *        Cryptography, the value is <code>EXTERNAL</code>.
         * @see KeyOrigin
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyOrigin
         */
        Builder keyOrigin(String keyOrigin);

        /**
         * <p>
         * The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the value
         * is <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment Cryptography,
         * the value is <code>EXTERNAL</code>.
         * </p>
         * 
         * @param keyOrigin
         *        The source of the key material. For keys created within Amazon Web Services Payment Cryptography, the
         *        value is <code>AWS_PAYMENT_CRYPTOGRAPHY</code>. For keys imported into Amazon Web Services Payment
         *        Cryptography, the value is <code>EXTERNAL</code>.
         * @see KeyOrigin
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyOrigin
         */
        Builder keyOrigin(KeyOrigin keyOrigin);

        /**
         * <p>
         * The date and time when the key was created.
         * </p>
         * 
         * @param createTimestamp
         *        The date and time when the key was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTimestamp(Instant createTimestamp);

        /**
         * <p>
         * The date and time after which Amazon Web Services Payment Cryptography will start using the key material for
         * cryptographic operations.
         * </p>
         * 
         * @param usageStartTimestamp
         *        The date and time after which Amazon Web Services Payment Cryptography will start using the key
         *        material for cryptographic operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usageStartTimestamp(Instant usageStartTimestamp);

        /**
         * <p>
         * The date and time after which Amazon Web Services Payment Cryptography will stop using the key material for
         * cryptographic operations.
         * </p>
         * 
         * @param usageStopTimestamp
         *        The date and time after which Amazon Web Services Payment Cryptography will stop using the key
         *        material for cryptographic operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usageStopTimestamp(Instant usageStopTimestamp);

        /**
         * <p>
         * The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value is
         * present only when <code>KeyState</code> is <code>DELETE_PENDING</code> and the key is scheduled for deletion.
         * </p>
         * 
         * @param deletePendingTimestamp
         *        The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value
         *        is present only when <code>KeyState</code> is <code>DELETE_PENDING</code> and the key is scheduled for
         *        deletion.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletePendingTimestamp(Instant deletePendingTimestamp);

        /**
         * <p>
         * The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value is
         * present only when when the <code>KeyState</code> is <code>DELETE_COMPLETE</code> and the Amazon Web Services
         * Payment Cryptography key is deleted.
         * </p>
         * 
         * @param deleteTimestamp
         *        The date and time after which Amazon Web Services Payment Cryptography will delete the key. This value
         *        is present only when when the <code>KeyState</code> is <code>DELETE_COMPLETE</code> and the Amazon Web
         *        Services Payment Cryptography key is deleted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteTimestamp(Instant deleteTimestamp);

        /**
         * <p>
         * The cryptographic usage of an ECDH derived key as deﬁned in section A.5.2 of the TR-31 spec.
         * </p>
         * 
         * @param deriveKeyUsage
         *        The cryptographic usage of an ECDH derived key as deﬁned in section A.5.2 of the TR-31 spec.
         * @see DeriveKeyUsage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeriveKeyUsage
         */
        Builder deriveKeyUsage(String deriveKeyUsage);

        /**
         * <p>
         * The cryptographic usage of an ECDH derived key as deﬁned in section A.5.2 of the TR-31 spec.
         * </p>
         * 
         * @param deriveKeyUsage
         *        The cryptographic usage of an ECDH derived key as deﬁned in section A.5.2 of the TR-31 spec.
         * @see DeriveKeyUsage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeriveKeyUsage
         */
        Builder deriveKeyUsage(DeriveKeyUsage deriveKeyUsage);

        /**
         * <p>
         * Indicates whether this key is a Multi-Region key and its role in the Multi-Region key hierarchy.
         * </p>
         * <p>
         * Multi-Region replication keys allow the same key material to be used across multiple Amazon Web Services
         * Regions. This field specifies whether the key is a Primary Region key (PRK) (which can be replicated to other
         * Amazon Web Services Regions) or a Replica Region key (RRK) (which is a copy of a PRK in another Region). For
         * more information, see <a
         * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
         * >Multi-Region key replication</a>.
         * </p>
         * 
         * @param multiRegionKeyType
         *        Indicates whether this key is a Multi-Region key and its role in the Multi-Region key hierarchy.</p>
         *        <p>
         *        Multi-Region replication keys allow the same key material to be used across multiple Amazon Web
         *        Services Regions. This field specifies whether the key is a Primary Region key (PRK) (which can be
         *        replicated to other Amazon Web Services Regions) or a Replica Region key (RRK) (which is a copy of a
         *        PRK in another Region). For more information, see <a href=
         *        "https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
         *        >Multi-Region key replication</a>.
         * @see MultiRegionKeyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MultiRegionKeyType
         */
        Builder multiRegionKeyType(String multiRegionKeyType);

        /**
         * <p>
         * Indicates whether this key is a Multi-Region key and its role in the Multi-Region key hierarchy.
         * </p>
         * <p>
         * Multi-Region replication keys allow the same key material to be used across multiple Amazon Web Services
         * Regions. This field specifies whether the key is a Primary Region key (PRK) (which can be replicated to other
         * Amazon Web Services Regions) or a Replica Region key (RRK) (which is a copy of a PRK in another Region). For
         * more information, see <a
         * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
         * >Multi-Region key replication</a>.
         * </p>
         * 
         * @param multiRegionKeyType
         *        Indicates whether this key is a Multi-Region key and its role in the Multi-Region key hierarchy.</p>
         *        <p>
         *        Multi-Region replication keys allow the same key material to be used across multiple Amazon Web
         *        Services Regions. This field specifies whether the key is a Primary Region key (PRK) (which can be
         *        replicated to other Amazon Web Services Regions) or a Replica Region key (RRK) (which is a copy of a
         *        PRK in another Region). For more information, see <a href=
         *        "https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
         *        >Multi-Region key replication</a>.
         * @see MultiRegionKeyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MultiRegionKeyType
         */
        Builder multiRegionKeyType(MultiRegionKeyType multiRegionKeyType);

        /**
         * Sets the value of the PrimaryRegion property for this object.
         *
         * @param primaryRegion
         *        The new value for the PrimaryRegion property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder primaryRegion(String primaryRegion);

        /**
         * <p>
         * Information about the replication status of the key across different Amazon Web Services Regions.
         * </p>
         * <p>
         * This field provides details about the current state of key replication, including any status messages or
         * operational information. It helps track the progress and health of key replication operations.
         * </p>
         * 
         * @param replicationStatus
         *        Information about the replication status of the key across different Amazon Web Services Regions.</p>
         *        <p>
         *        This field provides details about the current state of key replication, including any status messages
         *        or operational information. It helps track the progress and health of key replication operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationStatus(Map<String, ReplicationStatusType> replicationStatus);

        /**
         * <p>
         * Indicates whether this key is using the account's default replication regions configuration for <a
         * href="https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
         * >Multi-Region key replication</a>.
         * </p>
         * <p>
         * When set to <code>true</code>, the key automatically replicates to the regions specified in the account's
         * default replication settings. When set to <code>false</code>, the key has a custom replication configuration
         * that overrides the account defaults.
         * </p>
         * 
         * @param usingDefaultReplicationRegions
         *        Indicates whether this key is using the account's default replication regions configuration for <a
         *        href=
         *        "https://docs.aws.amazon.com/payment-cryptography/latest/userguide/keys-multi-region-replication.html"
         *        >Multi-Region key replication</a>.</p>
         *        <p>
         *        When set to <code>true</code>, the key automatically replicates to the regions specified in the
         *        account's default replication settings. When set to <code>false</code>, the key has a custom
         *        replication configuration that overrides the account defaults.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usingDefaultReplicationRegions(Boolean usingDefaultReplicationRegions);
    }

    static final class BuilderImpl implements Builder {
        private String keyArn;

        private KeyAttributes keyAttributes;

        private String keyCheckValue;

        private String keyCheckValueAlgorithm;

        private Boolean enabled;

        private Boolean exportable;

        private String keyState;

        private String keyOrigin;

        private Instant createTimestamp;

        private Instant usageStartTimestamp;

        private Instant usageStopTimestamp;

        private Instant deletePendingTimestamp;

        private Instant deleteTimestamp;

        private String deriveKeyUsage;

        private String multiRegionKeyType;

        private String primaryRegion;

        private Map<String, ReplicationStatusType> replicationStatus = DefaultSdkAutoConstructMap.getInstance();

        private Boolean usingDefaultReplicationRegions;

        private BuilderImpl() {
        }

        private BuilderImpl(Key model) {
            keyArn(model.keyArn);
            keyAttributes(model.keyAttributes);
            keyCheckValue(model.keyCheckValue);
            keyCheckValueAlgorithm(model.keyCheckValueAlgorithm);
            enabled(model.enabled);
            exportable(model.exportable);
            keyState(model.keyState);
            keyOrigin(model.keyOrigin);
            createTimestamp(model.createTimestamp);
            usageStartTimestamp(model.usageStartTimestamp);
            usageStopTimestamp(model.usageStopTimestamp);
            deletePendingTimestamp(model.deletePendingTimestamp);
            deleteTimestamp(model.deleteTimestamp);
            deriveKeyUsage(model.deriveKeyUsage);
            multiRegionKeyType(model.multiRegionKeyType);
            primaryRegion(model.primaryRegion);
            replicationStatus(model.replicationStatus);
            usingDefaultReplicationRegions(model.usingDefaultReplicationRegions);
        }

        public final String getKeyArn() {
            return keyArn;
        }

        public final void setKeyArn(String keyArn) {
            this.keyArn = keyArn;
        }

        @Override
        public final Builder keyArn(String keyArn) {
            this.keyArn = keyArn;
            return this;
        }

        public final KeyAttributes.Builder getKeyAttributes() {
            return keyAttributes != null ? keyAttributes.toBuilder() : null;
        }

        public final void setKeyAttributes(KeyAttributes.BuilderImpl keyAttributes) {
            this.keyAttributes = keyAttributes != null ? keyAttributes.build() : null;
        }

        @Override
        public final Builder keyAttributes(KeyAttributes keyAttributes) {
            this.keyAttributes = keyAttributes;
            return this;
        }

        public final String getKeyCheckValue() {
            return keyCheckValue;
        }

        public final void setKeyCheckValue(String keyCheckValue) {
            this.keyCheckValue = keyCheckValue;
        }

        @Override
        public final Builder keyCheckValue(String keyCheckValue) {
            this.keyCheckValue = keyCheckValue;
            return this;
        }

        public final String getKeyCheckValueAlgorithm() {
            return keyCheckValueAlgorithm;
        }

        public final void setKeyCheckValueAlgorithm(String keyCheckValueAlgorithm) {
            this.keyCheckValueAlgorithm = keyCheckValueAlgorithm;
        }

        @Override
        public final Builder keyCheckValueAlgorithm(String keyCheckValueAlgorithm) {
            this.keyCheckValueAlgorithm = keyCheckValueAlgorithm;
            return this;
        }

        @Override
        public final Builder keyCheckValueAlgorithm(KeyCheckValueAlgorithm keyCheckValueAlgorithm) {
            this.keyCheckValueAlgorithm(keyCheckValueAlgorithm == null ? null : keyCheckValueAlgorithm.toString());
            return this;
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final Boolean getExportable() {
            return exportable;
        }

        public final void setExportable(Boolean exportable) {
            this.exportable = exportable;
        }

        @Override
        public final Builder exportable(Boolean exportable) {
            this.exportable = exportable;
            return this;
        }

        public final String getKeyState() {
            return keyState;
        }

        public final void setKeyState(String keyState) {
            this.keyState = keyState;
        }

        @Override
        public final Builder keyState(String keyState) {
            this.keyState = keyState;
            return this;
        }

        @Override
        public final Builder keyState(KeyState keyState) {
            this.keyState(keyState == null ? null : keyState.toString());
            return this;
        }

        public final String getKeyOrigin() {
            return keyOrigin;
        }

        public final void setKeyOrigin(String keyOrigin) {
            this.keyOrigin = keyOrigin;
        }

        @Override
        public final Builder keyOrigin(String keyOrigin) {
            this.keyOrigin = keyOrigin;
            return this;
        }

        @Override
        public final Builder keyOrigin(KeyOrigin keyOrigin) {
            this.keyOrigin(keyOrigin == null ? null : keyOrigin.toString());
            return this;
        }

        public final Instant getCreateTimestamp() {
            return createTimestamp;
        }

        public final void setCreateTimestamp(Instant createTimestamp) {
            this.createTimestamp = createTimestamp;
        }

        @Override
        public final Builder createTimestamp(Instant createTimestamp) {
            this.createTimestamp = createTimestamp;
            return this;
        }

        public final Instant getUsageStartTimestamp() {
            return usageStartTimestamp;
        }

        public final void setUsageStartTimestamp(Instant usageStartTimestamp) {
            this.usageStartTimestamp = usageStartTimestamp;
        }

        @Override
        public final Builder usageStartTimestamp(Instant usageStartTimestamp) {
            this.usageStartTimestamp = usageStartTimestamp;
            return this;
        }

        public final Instant getUsageStopTimestamp() {
            return usageStopTimestamp;
        }

        public final void setUsageStopTimestamp(Instant usageStopTimestamp) {
            this.usageStopTimestamp = usageStopTimestamp;
        }

        @Override
        public final Builder usageStopTimestamp(Instant usageStopTimestamp) {
            this.usageStopTimestamp = usageStopTimestamp;
            return this;
        }

        public final Instant getDeletePendingTimestamp() {
            return deletePendingTimestamp;
        }

        public final void setDeletePendingTimestamp(Instant deletePendingTimestamp) {
            this.deletePendingTimestamp = deletePendingTimestamp;
        }

        @Override
        public final Builder deletePendingTimestamp(Instant deletePendingTimestamp) {
            this.deletePendingTimestamp = deletePendingTimestamp;
            return this;
        }

        public final Instant getDeleteTimestamp() {
            return deleteTimestamp;
        }

        public final void setDeleteTimestamp(Instant deleteTimestamp) {
            this.deleteTimestamp = deleteTimestamp;
        }

        @Override
        public final Builder deleteTimestamp(Instant deleteTimestamp) {
            this.deleteTimestamp = deleteTimestamp;
            return this;
        }

        public final String getDeriveKeyUsage() {
            return deriveKeyUsage;
        }

        public final void setDeriveKeyUsage(String deriveKeyUsage) {
            this.deriveKeyUsage = deriveKeyUsage;
        }

        @Override
        public final Builder deriveKeyUsage(String deriveKeyUsage) {
            this.deriveKeyUsage = deriveKeyUsage;
            return this;
        }

        @Override
        public final Builder deriveKeyUsage(DeriveKeyUsage deriveKeyUsage) {
            this.deriveKeyUsage(deriveKeyUsage == null ? null : deriveKeyUsage.toString());
            return this;
        }

        public final String getMultiRegionKeyType() {
            return multiRegionKeyType;
        }

        public final void setMultiRegionKeyType(String multiRegionKeyType) {
            this.multiRegionKeyType = multiRegionKeyType;
        }

        @Override
        public final Builder multiRegionKeyType(String multiRegionKeyType) {
            this.multiRegionKeyType = multiRegionKeyType;
            return this;
        }

        @Override
        public final Builder multiRegionKeyType(MultiRegionKeyType multiRegionKeyType) {
            this.multiRegionKeyType(multiRegionKeyType == null ? null : multiRegionKeyType.toString());
            return this;
        }

        public final String getPrimaryRegion() {
            return primaryRegion;
        }

        public final void setPrimaryRegion(String primaryRegion) {
            this.primaryRegion = primaryRegion;
        }

        @Override
        public final Builder primaryRegion(String primaryRegion) {
            this.primaryRegion = primaryRegion;
            return this;
        }

        public final Map<String, ReplicationStatusType.Builder> getReplicationStatus() {
            Map<String, ReplicationStatusType.Builder> result = ReplicationStatusCopier.copyToBuilder(this.replicationStatus);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setReplicationStatus(Map<String, ReplicationStatusType.BuilderImpl> replicationStatus) {
            this.replicationStatus = ReplicationStatusCopier.copyFromBuilder(replicationStatus);
        }

        @Override
        public final Builder replicationStatus(Map<String, ReplicationStatusType> replicationStatus) {
            this.replicationStatus = ReplicationStatusCopier.copy(replicationStatus);
            return this;
        }

        public final Boolean getUsingDefaultReplicationRegions() {
            return usingDefaultReplicationRegions;
        }

        public final void setUsingDefaultReplicationRegions(Boolean usingDefaultReplicationRegions) {
            this.usingDefaultReplicationRegions = usingDefaultReplicationRegions;
        }

        @Override
        public final Builder usingDefaultReplicationRegions(Boolean usingDefaultReplicationRegions) {
            this.usingDefaultReplicationRegions = usingDefaultReplicationRegions;
            return this;
        }

        @Override
        public Key build() {
            return new Key(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
