/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.paymentcryptography.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Parameter information for key material export using the asymmetric TR-34 key exchange method.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ExportTr34KeyBlock implements SdkPojo, Serializable,
        ToCopyableBuilder<ExportTr34KeyBlock.Builder, ExportTr34KeyBlock> {
    private static final SdkField<String> CERTIFICATE_AUTHORITY_PUBLIC_KEY_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CertificateAuthorityPublicKeyIdentifier")
            .getter(getter(ExportTr34KeyBlock::certificateAuthorityPublicKeyIdentifier))
            .setter(setter(Builder::certificateAuthorityPublicKeyIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("CertificateAuthorityPublicKeyIdentifier").build()).build();

    private static final SdkField<String> WRAPPING_KEY_CERTIFICATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WrappingKeyCertificate").getter(getter(ExportTr34KeyBlock::wrappingKeyCertificate))
            .setter(setter(Builder::wrappingKeyCertificate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WrappingKeyCertificate").build())
            .build();

    private static final SdkField<String> EXPORT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExportToken").getter(getter(ExportTr34KeyBlock::exportToken)).setter(setter(Builder::exportToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExportToken").build()).build();

    private static final SdkField<String> SIGNING_KEY_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SigningKeyIdentifier").getter(getter(ExportTr34KeyBlock::signingKeyIdentifier))
            .setter(setter(Builder::signingKeyIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SigningKeyIdentifier").build())
            .build();

    private static final SdkField<String> SIGNING_KEY_CERTIFICATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SigningKeyCertificate").getter(getter(ExportTr34KeyBlock::signingKeyCertificate))
            .setter(setter(Builder::signingKeyCertificate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SigningKeyCertificate").build())
            .build();

    private static final SdkField<String> KEY_BLOCK_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyBlockFormat").getter(getter(ExportTr34KeyBlock::keyBlockFormatAsString))
            .setter(setter(Builder::keyBlockFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyBlockFormat").build()).build();

    private static final SdkField<String> RANDOM_NONCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RandomNonce").getter(getter(ExportTr34KeyBlock::randomNonce)).setter(setter(Builder::randomNonce))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RandomNonce").build()).build();

    private static final SdkField<KeyBlockHeaders> KEY_BLOCK_HEADERS_FIELD = SdkField
            .<KeyBlockHeaders> builder(MarshallingType.SDK_POJO).memberName("KeyBlockHeaders")
            .getter(getter(ExportTr34KeyBlock::keyBlockHeaders)).setter(setter(Builder::keyBlockHeaders))
            .constructor(KeyBlockHeaders::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyBlockHeaders").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CERTIFICATE_AUTHORITY_PUBLIC_KEY_IDENTIFIER_FIELD, WRAPPING_KEY_CERTIFICATE_FIELD, EXPORT_TOKEN_FIELD,
            SIGNING_KEY_IDENTIFIER_FIELD, SIGNING_KEY_CERTIFICATE_FIELD, KEY_BLOCK_FORMAT_FIELD, RANDOM_NONCE_FIELD,
            KEY_BLOCK_HEADERS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String certificateAuthorityPublicKeyIdentifier;

    private final String wrappingKeyCertificate;

    private final String exportToken;

    private final String signingKeyIdentifier;

    private final String signingKeyCertificate;

    private final String keyBlockFormat;

    private final String randomNonce;

    private final KeyBlockHeaders keyBlockHeaders;

    private ExportTr34KeyBlock(BuilderImpl builder) {
        this.certificateAuthorityPublicKeyIdentifier = builder.certificateAuthorityPublicKeyIdentifier;
        this.wrappingKeyCertificate = builder.wrappingKeyCertificate;
        this.exportToken = builder.exportToken;
        this.signingKeyIdentifier = builder.signingKeyIdentifier;
        this.signingKeyCertificate = builder.signingKeyCertificate;
        this.keyBlockFormat = builder.keyBlockFormat;
        this.randomNonce = builder.randomNonce;
        this.keyBlockHeaders = builder.keyBlockHeaders;
    }

    /**
     * <p>
     * The <code>KeyARN</code> of the certificate chain that signs the wrapping key certificate during TR-34 key export.
     * </p>
     * 
     * @return The <code>KeyARN</code> of the certificate chain that signs the wrapping key certificate during TR-34 key
     *         export.
     */
    public final String certificateAuthorityPublicKeyIdentifier() {
        return certificateAuthorityPublicKeyIdentifier;
    }

    /**
     * <p>
     * The <code>KeyARN</code> of the wrapping key certificate. Amazon Web Services Payment Cryptography uses this
     * certificate to wrap the key under export.
     * </p>
     * 
     * @return The <code>KeyARN</code> of the wrapping key certificate. Amazon Web Services Payment Cryptography uses
     *         this certificate to wrap the key under export.
     */
    public final String wrappingKeyCertificate() {
        return wrappingKeyCertificate;
    }

    /**
     * <p>
     * The export token to initiate key export from Amazon Web Services Payment Cryptography. It also contains the
     * signing key certificate that will sign the wrapped key during TR-34 key block generation. Call <a
     * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetParametersForExport.html"
     * >GetParametersForExport</a> to receive an export token. It expires after 30 days. You can use the same export
     * token to export multiple keys from the same service account.
     * </p>
     * 
     * @return The export token to initiate key export from Amazon Web Services Payment Cryptography. It also contains
     *         the signing key certificate that will sign the wrapped key during TR-34 key block generation. Call <a
     *         href
     *         ="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetParametersForExport.html"
     *         >GetParametersForExport</a> to receive an export token. It expires after 30 days. You can use the same
     *         export token to export multiple keys from the same service account.
     */
    public final String exportToken() {
        return exportToken;
    }

    /**
     * <p>
     * Key Identifier used for signing the export key
     * </p>
     * 
     * @return Key Identifier used for signing the export key
     */
    public final String signingKeyIdentifier() {
        return signingKeyIdentifier;
    }

    /**
     * <p>
     * The certificate used to sign the TR-34 key block.
     * </p>
     * 
     * @return The certificate used to sign the TR-34 key block.
     */
    public final String signingKeyCertificate() {
        return signingKeyCertificate;
    }

    /**
     * <p>
     * The format of key block that Amazon Web Services Payment Cryptography will use during key export.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyBlockFormat}
     * will return {@link Tr34KeyBlockFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #keyBlockFormatAsString}.
     * </p>
     * 
     * @return The format of key block that Amazon Web Services Payment Cryptography will use during key export.
     * @see Tr34KeyBlockFormat
     */
    public final Tr34KeyBlockFormat keyBlockFormat() {
        return Tr34KeyBlockFormat.fromValue(keyBlockFormat);
    }

    /**
     * <p>
     * The format of key block that Amazon Web Services Payment Cryptography will use during key export.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyBlockFormat}
     * will return {@link Tr34KeyBlockFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #keyBlockFormatAsString}.
     * </p>
     * 
     * @return The format of key block that Amazon Web Services Payment Cryptography will use during key export.
     * @see Tr34KeyBlockFormat
     */
    public final String keyBlockFormatAsString() {
        return keyBlockFormat;
    }

    /**
     * <p>
     * A random number value that is unique to the TR-34 key block generated using 2 pass. The operation will fail, if a
     * random nonce value is not provided for a TR-34 key block generated using 2 pass.
     * </p>
     * 
     * @return A random number value that is unique to the TR-34 key block generated using 2 pass. The operation will
     *         fail, if a random nonce value is not provided for a TR-34 key block generated using 2 pass.
     */
    public final String randomNonce() {
        return randomNonce;
    }

    /**
     * <p>
     * Optional metadata for export associated with the key material. This data is signed but transmitted in clear text.
     * </p>
     * 
     * @return Optional metadata for export associated with the key material. This data is signed but transmitted in
     *         clear text.
     */
    public final KeyBlockHeaders keyBlockHeaders() {
        return keyBlockHeaders;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(certificateAuthorityPublicKeyIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(wrappingKeyCertificate());
        hashCode = 31 * hashCode + Objects.hashCode(exportToken());
        hashCode = 31 * hashCode + Objects.hashCode(signingKeyIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(signingKeyCertificate());
        hashCode = 31 * hashCode + Objects.hashCode(keyBlockFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(randomNonce());
        hashCode = 31 * hashCode + Objects.hashCode(keyBlockHeaders());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ExportTr34KeyBlock)) {
            return false;
        }
        ExportTr34KeyBlock other = (ExportTr34KeyBlock) obj;
        return Objects.equals(certificateAuthorityPublicKeyIdentifier(), other.certificateAuthorityPublicKeyIdentifier())
                && Objects.equals(wrappingKeyCertificate(), other.wrappingKeyCertificate())
                && Objects.equals(exportToken(), other.exportToken())
                && Objects.equals(signingKeyIdentifier(), other.signingKeyIdentifier())
                && Objects.equals(signingKeyCertificate(), other.signingKeyCertificate())
                && Objects.equals(keyBlockFormatAsString(), other.keyBlockFormatAsString())
                && Objects.equals(randomNonce(), other.randomNonce())
                && Objects.equals(keyBlockHeaders(), other.keyBlockHeaders());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ExportTr34KeyBlock")
                .add("CertificateAuthorityPublicKeyIdentifier", certificateAuthorityPublicKeyIdentifier())
                .add("WrappingKeyCertificate", wrappingKeyCertificate()).add("ExportToken", exportToken())
                .add("SigningKeyIdentifier", signingKeyIdentifier()).add("SigningKeyCertificate", signingKeyCertificate())
                .add("KeyBlockFormat", keyBlockFormatAsString()).add("RandomNonce", randomNonce())
                .add("KeyBlockHeaders", keyBlockHeaders()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CertificateAuthorityPublicKeyIdentifier":
            return Optional.ofNullable(clazz.cast(certificateAuthorityPublicKeyIdentifier()));
        case "WrappingKeyCertificate":
            return Optional.ofNullable(clazz.cast(wrappingKeyCertificate()));
        case "ExportToken":
            return Optional.ofNullable(clazz.cast(exportToken()));
        case "SigningKeyIdentifier":
            return Optional.ofNullable(clazz.cast(signingKeyIdentifier()));
        case "SigningKeyCertificate":
            return Optional.ofNullable(clazz.cast(signingKeyCertificate()));
        case "KeyBlockFormat":
            return Optional.ofNullable(clazz.cast(keyBlockFormatAsString()));
        case "RandomNonce":
            return Optional.ofNullable(clazz.cast(randomNonce()));
        case "KeyBlockHeaders":
            return Optional.ofNullable(clazz.cast(keyBlockHeaders()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CertificateAuthorityPublicKeyIdentifier", CERTIFICATE_AUTHORITY_PUBLIC_KEY_IDENTIFIER_FIELD);
        map.put("WrappingKeyCertificate", WRAPPING_KEY_CERTIFICATE_FIELD);
        map.put("ExportToken", EXPORT_TOKEN_FIELD);
        map.put("SigningKeyIdentifier", SIGNING_KEY_IDENTIFIER_FIELD);
        map.put("SigningKeyCertificate", SIGNING_KEY_CERTIFICATE_FIELD);
        map.put("KeyBlockFormat", KEY_BLOCK_FORMAT_FIELD);
        map.put("RandomNonce", RANDOM_NONCE_FIELD);
        map.put("KeyBlockHeaders", KEY_BLOCK_HEADERS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ExportTr34KeyBlock, T> g) {
        return obj -> g.apply((ExportTr34KeyBlock) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ExportTr34KeyBlock> {
        /**
         * <p>
         * The <code>KeyARN</code> of the certificate chain that signs the wrapping key certificate during TR-34 key
         * export.
         * </p>
         * 
         * @param certificateAuthorityPublicKeyIdentifier
         *        The <code>KeyARN</code> of the certificate chain that signs the wrapping key certificate during TR-34
         *        key export.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateAuthorityPublicKeyIdentifier(String certificateAuthorityPublicKeyIdentifier);

        /**
         * <p>
         * The <code>KeyARN</code> of the wrapping key certificate. Amazon Web Services Payment Cryptography uses this
         * certificate to wrap the key under export.
         * </p>
         * 
         * @param wrappingKeyCertificate
         *        The <code>KeyARN</code> of the wrapping key certificate. Amazon Web Services Payment Cryptography uses
         *        this certificate to wrap the key under export.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder wrappingKeyCertificate(String wrappingKeyCertificate);

        /**
         * <p>
         * The export token to initiate key export from Amazon Web Services Payment Cryptography. It also contains the
         * signing key certificate that will sign the wrapped key during TR-34 key block generation. Call <a
         * href="https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetParametersForExport.html"
         * >GetParametersForExport</a> to receive an export token. It expires after 30 days. You can use the same export
         * token to export multiple keys from the same service account.
         * </p>
         * 
         * @param exportToken
         *        The export token to initiate key export from Amazon Web Services Payment Cryptography. It also
         *        contains the signing key certificate that will sign the wrapped key during TR-34 key block generation.
         *        Call <a href=
         *        "https://docs.aws.amazon.com/payment-cryptography/latest/APIReference/API_GetParametersForExport.html"
         *        >GetParametersForExport</a> to receive an export token. It expires after 30 days. You can use the same
         *        export token to export multiple keys from the same service account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportToken(String exportToken);

        /**
         * <p>
         * Key Identifier used for signing the export key
         * </p>
         * 
         * @param signingKeyIdentifier
         *        Key Identifier used for signing the export key
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signingKeyIdentifier(String signingKeyIdentifier);

        /**
         * <p>
         * The certificate used to sign the TR-34 key block.
         * </p>
         * 
         * @param signingKeyCertificate
         *        The certificate used to sign the TR-34 key block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signingKeyCertificate(String signingKeyCertificate);

        /**
         * <p>
         * The format of key block that Amazon Web Services Payment Cryptography will use during key export.
         * </p>
         * 
         * @param keyBlockFormat
         *        The format of key block that Amazon Web Services Payment Cryptography will use during key export.
         * @see Tr34KeyBlockFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Tr34KeyBlockFormat
         */
        Builder keyBlockFormat(String keyBlockFormat);

        /**
         * <p>
         * The format of key block that Amazon Web Services Payment Cryptography will use during key export.
         * </p>
         * 
         * @param keyBlockFormat
         *        The format of key block that Amazon Web Services Payment Cryptography will use during key export.
         * @see Tr34KeyBlockFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Tr34KeyBlockFormat
         */
        Builder keyBlockFormat(Tr34KeyBlockFormat keyBlockFormat);

        /**
         * <p>
         * A random number value that is unique to the TR-34 key block generated using 2 pass. The operation will fail,
         * if a random nonce value is not provided for a TR-34 key block generated using 2 pass.
         * </p>
         * 
         * @param randomNonce
         *        A random number value that is unique to the TR-34 key block generated using 2 pass. The operation will
         *        fail, if a random nonce value is not provided for a TR-34 key block generated using 2 pass.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder randomNonce(String randomNonce);

        /**
         * <p>
         * Optional metadata for export associated with the key material. This data is signed but transmitted in clear
         * text.
         * </p>
         * 
         * @param keyBlockHeaders
         *        Optional metadata for export associated with the key material. This data is signed but transmitted in
         *        clear text.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyBlockHeaders(KeyBlockHeaders keyBlockHeaders);

        /**
         * <p>
         * Optional metadata for export associated with the key material. This data is signed but transmitted in clear
         * text.
         * </p>
         * This is a convenience method that creates an instance of the {@link KeyBlockHeaders.Builder} avoiding the
         * need to create one manually via {@link KeyBlockHeaders#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KeyBlockHeaders.Builder#build()} is called immediately and its
         * result is passed to {@link #keyBlockHeaders(KeyBlockHeaders)}.
         * 
         * @param keyBlockHeaders
         *        a consumer that will call methods on {@link KeyBlockHeaders.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #keyBlockHeaders(KeyBlockHeaders)
         */
        default Builder keyBlockHeaders(Consumer<KeyBlockHeaders.Builder> keyBlockHeaders) {
            return keyBlockHeaders(KeyBlockHeaders.builder().applyMutation(keyBlockHeaders).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String certificateAuthorityPublicKeyIdentifier;

        private String wrappingKeyCertificate;

        private String exportToken;

        private String signingKeyIdentifier;

        private String signingKeyCertificate;

        private String keyBlockFormat;

        private String randomNonce;

        private KeyBlockHeaders keyBlockHeaders;

        private BuilderImpl() {
        }

        private BuilderImpl(ExportTr34KeyBlock model) {
            certificateAuthorityPublicKeyIdentifier(model.certificateAuthorityPublicKeyIdentifier);
            wrappingKeyCertificate(model.wrappingKeyCertificate);
            exportToken(model.exportToken);
            signingKeyIdentifier(model.signingKeyIdentifier);
            signingKeyCertificate(model.signingKeyCertificate);
            keyBlockFormat(model.keyBlockFormat);
            randomNonce(model.randomNonce);
            keyBlockHeaders(model.keyBlockHeaders);
        }

        public final String getCertificateAuthorityPublicKeyIdentifier() {
            return certificateAuthorityPublicKeyIdentifier;
        }

        public final void setCertificateAuthorityPublicKeyIdentifier(String certificateAuthorityPublicKeyIdentifier) {
            this.certificateAuthorityPublicKeyIdentifier = certificateAuthorityPublicKeyIdentifier;
        }

        @Override
        public final Builder certificateAuthorityPublicKeyIdentifier(String certificateAuthorityPublicKeyIdentifier) {
            this.certificateAuthorityPublicKeyIdentifier = certificateAuthorityPublicKeyIdentifier;
            return this;
        }

        public final String getWrappingKeyCertificate() {
            return wrappingKeyCertificate;
        }

        public final void setWrappingKeyCertificate(String wrappingKeyCertificate) {
            this.wrappingKeyCertificate = wrappingKeyCertificate;
        }

        @Override
        public final Builder wrappingKeyCertificate(String wrappingKeyCertificate) {
            this.wrappingKeyCertificate = wrappingKeyCertificate;
            return this;
        }

        public final String getExportToken() {
            return exportToken;
        }

        public final void setExportToken(String exportToken) {
            this.exportToken = exportToken;
        }

        @Override
        public final Builder exportToken(String exportToken) {
            this.exportToken = exportToken;
            return this;
        }

        public final String getSigningKeyIdentifier() {
            return signingKeyIdentifier;
        }

        public final void setSigningKeyIdentifier(String signingKeyIdentifier) {
            this.signingKeyIdentifier = signingKeyIdentifier;
        }

        @Override
        public final Builder signingKeyIdentifier(String signingKeyIdentifier) {
            this.signingKeyIdentifier = signingKeyIdentifier;
            return this;
        }

        public final String getSigningKeyCertificate() {
            return signingKeyCertificate;
        }

        public final void setSigningKeyCertificate(String signingKeyCertificate) {
            this.signingKeyCertificate = signingKeyCertificate;
        }

        @Override
        public final Builder signingKeyCertificate(String signingKeyCertificate) {
            this.signingKeyCertificate = signingKeyCertificate;
            return this;
        }

        public final String getKeyBlockFormat() {
            return keyBlockFormat;
        }

        public final void setKeyBlockFormat(String keyBlockFormat) {
            this.keyBlockFormat = keyBlockFormat;
        }

        @Override
        public final Builder keyBlockFormat(String keyBlockFormat) {
            this.keyBlockFormat = keyBlockFormat;
            return this;
        }

        @Override
        public final Builder keyBlockFormat(Tr34KeyBlockFormat keyBlockFormat) {
            this.keyBlockFormat(keyBlockFormat == null ? null : keyBlockFormat.toString());
            return this;
        }

        public final String getRandomNonce() {
            return randomNonce;
        }

        public final void setRandomNonce(String randomNonce) {
            this.randomNonce = randomNonce;
        }

        @Override
        public final Builder randomNonce(String randomNonce) {
            this.randomNonce = randomNonce;
            return this;
        }

        public final KeyBlockHeaders.Builder getKeyBlockHeaders() {
            return keyBlockHeaders != null ? keyBlockHeaders.toBuilder() : null;
        }

        public final void setKeyBlockHeaders(KeyBlockHeaders.BuilderImpl keyBlockHeaders) {
            this.keyBlockHeaders = keyBlockHeaders != null ? keyBlockHeaders.build() : null;
        }

        @Override
        public final Builder keyBlockHeaders(KeyBlockHeaders keyBlockHeaders) {
            this.keyBlockHeaders = keyBlockHeaders;
            return this;
        }

        @Override
        public ExportTr34KeyBlock build() {
            return new ExportTr34KeyBlock(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
