/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.odb.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a database server.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DbServerSummary implements SdkPojo, Serializable, ToCopyableBuilder<DbServerSummary.Builder, DbServerSummary> {
    private static final SdkField<String> DB_SERVER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dbServerId").getter(getter(DbServerSummary::dbServerId)).setter(setter(Builder::dbServerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dbServerId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(DbServerSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusReason").getter(getter(DbServerSummary::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusReason").build()).build();

    private static final SdkField<Integer> CPU_CORE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("cpuCoreCount").getter(getter(DbServerSummary::cpuCoreCount)).setter(setter(Builder::cpuCoreCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cpuCoreCount").build()).build();

    private static final SdkField<Integer> DB_NODE_STORAGE_SIZE_IN_G_BS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("dbNodeStorageSizeInGBs")
            .getter(getter(DbServerSummary::dbNodeStorageSizeInGBs)).setter(setter(Builder::dbNodeStorageSizeInGBs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dbNodeStorageSizeInGBs").build())
            .build();

    private static final SdkField<DbServerPatchingDetails> DB_SERVER_PATCHING_DETAILS_FIELD = SdkField
            .<DbServerPatchingDetails> builder(MarshallingType.SDK_POJO).memberName("dbServerPatchingDetails")
            .getter(getter(DbServerSummary::dbServerPatchingDetails)).setter(setter(Builder::dbServerPatchingDetails))
            .constructor(DbServerPatchingDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dbServerPatchingDetails").build())
            .build();

    private static final SdkField<String> DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("displayName").getter(getter(DbServerSummary::displayName)).setter(setter(Builder::displayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("displayName").build()).build();

    private static final SdkField<String> EXADATA_INFRASTRUCTURE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("exadataInfrastructureId").getter(getter(DbServerSummary::exadataInfrastructureId))
            .setter(setter(Builder::exadataInfrastructureId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("exadataInfrastructureId").build())
            .build();

    private static final SdkField<String> OCID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ocid")
            .getter(getter(DbServerSummary::ocid)).setter(setter(Builder::ocid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ocid").build()).build();

    private static final SdkField<String> OCI_RESOURCE_ANCHOR_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ociResourceAnchorName").getter(getter(DbServerSummary::ociResourceAnchorName))
            .setter(setter(Builder::ociResourceAnchorName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ociResourceAnchorName").build())
            .build();

    private static final SdkField<Integer> MAX_CPU_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxCpuCount").getter(getter(DbServerSummary::maxCpuCount)).setter(setter(Builder::maxCpuCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxCpuCount").build()).build();

    private static final SdkField<Integer> MAX_DB_NODE_STORAGE_IN_G_BS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("maxDbNodeStorageInGBs")
            .getter(getter(DbServerSummary::maxDbNodeStorageInGBs)).setter(setter(Builder::maxDbNodeStorageInGBs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxDbNodeStorageInGBs").build())
            .build();

    private static final SdkField<Integer> MAX_MEMORY_IN_G_BS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxMemoryInGBs").getter(getter(DbServerSummary::maxMemoryInGBs)).setter(setter(Builder::maxMemoryInGBs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxMemoryInGBs").build()).build();

    private static final SdkField<Integer> MEMORY_SIZE_IN_G_BS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("memorySizeInGBs").getter(getter(DbServerSummary::memorySizeInGBs))
            .setter(setter(Builder::memorySizeInGBs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memorySizeInGBs").build()).build();

    private static final SdkField<String> SHAPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("shape")
            .getter(getter(DbServerSummary::shape)).setter(setter(Builder::shape))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("shape").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(DbServerSummary::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<List<String>> VM_CLUSTER_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("vmClusterIds")
            .getter(getter(DbServerSummary::vmClusterIds))
            .setter(setter(Builder::vmClusterIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vmClusterIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> COMPUTE_MODEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("computeModel").getter(getter(DbServerSummary::computeModelAsString))
            .setter(setter(Builder::computeModel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("computeModel").build()).build();

    private static final SdkField<List<String>> AUTONOMOUS_VM_CLUSTER_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("autonomousVmClusterIds")
            .getter(getter(DbServerSummary::autonomousVmClusterIds))
            .setter(setter(Builder::autonomousVmClusterIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autonomousVmClusterIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> AUTONOMOUS_VIRTUAL_MACHINE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("autonomousVirtualMachineIds")
            .getter(getter(DbServerSummary::autonomousVirtualMachineIds))
            .setter(setter(Builder::autonomousVirtualMachineIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autonomousVirtualMachineIds")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DB_SERVER_ID_FIELD,
            STATUS_FIELD, STATUS_REASON_FIELD, CPU_CORE_COUNT_FIELD, DB_NODE_STORAGE_SIZE_IN_G_BS_FIELD,
            DB_SERVER_PATCHING_DETAILS_FIELD, DISPLAY_NAME_FIELD, EXADATA_INFRASTRUCTURE_ID_FIELD, OCID_FIELD,
            OCI_RESOURCE_ANCHOR_NAME_FIELD, MAX_CPU_COUNT_FIELD, MAX_DB_NODE_STORAGE_IN_G_BS_FIELD, MAX_MEMORY_IN_G_BS_FIELD,
            MEMORY_SIZE_IN_G_BS_FIELD, SHAPE_FIELD, CREATED_AT_FIELD, VM_CLUSTER_IDS_FIELD, COMPUTE_MODEL_FIELD,
            AUTONOMOUS_VM_CLUSTER_IDS_FIELD, AUTONOMOUS_VIRTUAL_MACHINE_IDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String dbServerId;

    private final String status;

    private final String statusReason;

    private final Integer cpuCoreCount;

    private final Integer dbNodeStorageSizeInGBs;

    private final DbServerPatchingDetails dbServerPatchingDetails;

    private final String displayName;

    private final String exadataInfrastructureId;

    private final String ocid;

    private final String ociResourceAnchorName;

    private final Integer maxCpuCount;

    private final Integer maxDbNodeStorageInGBs;

    private final Integer maxMemoryInGBs;

    private final Integer memorySizeInGBs;

    private final String shape;

    private final Instant createdAt;

    private final List<String> vmClusterIds;

    private final String computeModel;

    private final List<String> autonomousVmClusterIds;

    private final List<String> autonomousVirtualMachineIds;

    private DbServerSummary(BuilderImpl builder) {
        this.dbServerId = builder.dbServerId;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
        this.cpuCoreCount = builder.cpuCoreCount;
        this.dbNodeStorageSizeInGBs = builder.dbNodeStorageSizeInGBs;
        this.dbServerPatchingDetails = builder.dbServerPatchingDetails;
        this.displayName = builder.displayName;
        this.exadataInfrastructureId = builder.exadataInfrastructureId;
        this.ocid = builder.ocid;
        this.ociResourceAnchorName = builder.ociResourceAnchorName;
        this.maxCpuCount = builder.maxCpuCount;
        this.maxDbNodeStorageInGBs = builder.maxDbNodeStorageInGBs;
        this.maxMemoryInGBs = builder.maxMemoryInGBs;
        this.memorySizeInGBs = builder.memorySizeInGBs;
        this.shape = builder.shape;
        this.createdAt = builder.createdAt;
        this.vmClusterIds = builder.vmClusterIds;
        this.computeModel = builder.computeModel;
        this.autonomousVmClusterIds = builder.autonomousVmClusterIds;
        this.autonomousVirtualMachineIds = builder.autonomousVirtualMachineIds;
    }

    /**
     * <p>
     * The unique identifier of the database server.
     * </p>
     * 
     * @return The unique identifier of the database server.
     */
    public final String dbServerId() {
        return dbServerId;
    }

    /**
     * <p>
     * The current status of the database server.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the database server.
     * @see ResourceStatus
     */
    public final ResourceStatus status() {
        return ResourceStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the database server.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the database server.
     * @see ResourceStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Additional information about the status of the database server.
     * </p>
     * 
     * @return Additional information about the status of the database server.
     */
    public final String statusReason() {
        return statusReason;
    }

    /**
     * <p>
     * The number of CPU cores enabled on the database server.
     * </p>
     * 
     * @return The number of CPU cores enabled on the database server.
     */
    public final Integer cpuCoreCount() {
        return cpuCoreCount;
    }

    /**
     * <p>
     * The amount of local node storage, in gigabytes (GB), that's allocated on the database server.
     * </p>
     * 
     * @return The amount of local node storage, in gigabytes (GB), that's allocated on the database server.
     */
    public final Integer dbNodeStorageSizeInGBs() {
        return dbNodeStorageSizeInGBs;
    }

    /**
     * Returns the value of the DbServerPatchingDetails property for this object.
     * 
     * @return The value of the DbServerPatchingDetails property for this object.
     */
    public final DbServerPatchingDetails dbServerPatchingDetails() {
        return dbServerPatchingDetails;
    }

    /**
     * <p>
     * The user-friendly name of the database server. The name doesn't need to be unique.
     * </p>
     * 
     * @return The user-friendly name of the database server. The name doesn't need to be unique.
     */
    public final String displayName() {
        return displayName;
    }

    /**
     * <p>
     * The ID of the Exadata infrastructure that hosts the database server.
     * </p>
     * 
     * @return The ID of the Exadata infrastructure that hosts the database server.
     */
    public final String exadataInfrastructureId() {
        return exadataInfrastructureId;
    }

    /**
     * <p>
     * The OCID of the database server.
     * </p>
     * 
     * @return The OCID of the database server.
     */
    public final String ocid() {
        return ocid;
    }

    /**
     * <p>
     * The name of the OCI resource anchor for the database server.
     * </p>
     * 
     * @return The name of the OCI resource anchor for the database server.
     */
    public final String ociResourceAnchorName() {
        return ociResourceAnchorName;
    }

    /**
     * <p>
     * The total number of CPU cores available on the database server.
     * </p>
     * 
     * @return The total number of CPU cores available on the database server.
     */
    public final Integer maxCpuCount() {
        return maxCpuCount;
    }

    /**
     * <p>
     * The total amount of local node storage, in gigabytes (GB), that's available on the database server.
     * </p>
     * 
     * @return The total amount of local node storage, in gigabytes (GB), that's available on the database server.
     */
    public final Integer maxDbNodeStorageInGBs() {
        return maxDbNodeStorageInGBs;
    }

    /**
     * <p>
     * The total amount of memory, in gigabytes (GB), that's available on the database server.
     * </p>
     * 
     * @return The total amount of memory, in gigabytes (GB), that's available on the database server.
     */
    public final Integer maxMemoryInGBs() {
        return maxMemoryInGBs;
    }

    /**
     * <p>
     * The amount of memory, in gigabytes (GB), that's allocated on the database server.
     * </p>
     * 
     * @return The amount of memory, in gigabytes (GB), that's allocated on the database server.
     */
    public final Integer memorySizeInGBs() {
        return memorySizeInGBs;
    }

    /**
     * <p>
     * The hardware system model of the Exadata infrastructure that the database server is hosted on. The shape
     * determines the amount of CPU, storage, and memory resources available.
     * </p>
     * 
     * @return The hardware system model of the Exadata infrastructure that the database server is hosted on. The shape
     *         determines the amount of CPU, storage, and memory resources available.
     */
    public final String shape() {
        return shape;
    }

    /**
     * <p>
     * The date and time when the database server was created.
     * </p>
     * 
     * @return The date and time when the database server was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the VmClusterIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVmClusterIds() {
        return vmClusterIds != null && !(vmClusterIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the VM clusters that are associated with the database server.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVmClusterIds} method.
     * </p>
     * 
     * @return The IDs of the VM clusters that are associated with the database server.
     */
    public final List<String> vmClusterIds() {
        return vmClusterIds;
    }

    /**
     * <p>
     * The OCI model compute model used when you create or clone an instance: ECPU or OCPU. An ECPU is an abstracted
     * measure of compute resources. ECPUs are based on the number of cores elastically allocated from a pool of compute
     * and storage servers. An OCPU is a legacy physical measure of compute resources. OCPUs are based on the physical
     * core of a processor with hyper-threading enabled.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #computeModel} will
     * return {@link ComputeModel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #computeModelAsString}.
     * </p>
     * 
     * @return The OCI model compute model used when you create or clone an instance: ECPU or OCPU. An ECPU is an
     *         abstracted measure of compute resources. ECPUs are based on the number of cores elastically allocated
     *         from a pool of compute and storage servers. An OCPU is a legacy physical measure of compute resources.
     *         OCPUs are based on the physical core of a processor with hyper-threading enabled.
     * @see ComputeModel
     */
    public final ComputeModel computeModel() {
        return ComputeModel.fromValue(computeModel);
    }

    /**
     * <p>
     * The OCI model compute model used when you create or clone an instance: ECPU or OCPU. An ECPU is an abstracted
     * measure of compute resources. ECPUs are based on the number of cores elastically allocated from a pool of compute
     * and storage servers. An OCPU is a legacy physical measure of compute resources. OCPUs are based on the physical
     * core of a processor with hyper-threading enabled.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #computeModel} will
     * return {@link ComputeModel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #computeModelAsString}.
     * </p>
     * 
     * @return The OCI model compute model used when you create or clone an instance: ECPU or OCPU. An ECPU is an
     *         abstracted measure of compute resources. ECPUs are based on the number of cores elastically allocated
     *         from a pool of compute and storage servers. An OCPU is a legacy physical measure of compute resources.
     *         OCPUs are based on the physical core of a processor with hyper-threading enabled.
     * @see ComputeModel
     */
    public final String computeModelAsString() {
        return computeModel;
    }

    /**
     * For responses, this returns true if the service returned a value for the AutonomousVmClusterIds property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAutonomousVmClusterIds() {
        return autonomousVmClusterIds != null && !(autonomousVmClusterIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of identifiers for the Autonomous VM clusters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAutonomousVmClusterIds} method.
     * </p>
     * 
     * @return A list of identifiers for the Autonomous VM clusters.
     */
    public final List<String> autonomousVmClusterIds() {
        return autonomousVmClusterIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the AutonomousVirtualMachineIds property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAutonomousVirtualMachineIds() {
        return autonomousVirtualMachineIds != null && !(autonomousVirtualMachineIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of unique identifiers for the Autonomous VMs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAutonomousVirtualMachineIds} method.
     * </p>
     * 
     * @return A list of unique identifiers for the Autonomous VMs.
     */
    public final List<String> autonomousVirtualMachineIds() {
        return autonomousVirtualMachineIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dbServerId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(cpuCoreCount());
        hashCode = 31 * hashCode + Objects.hashCode(dbNodeStorageSizeInGBs());
        hashCode = 31 * hashCode + Objects.hashCode(dbServerPatchingDetails());
        hashCode = 31 * hashCode + Objects.hashCode(displayName());
        hashCode = 31 * hashCode + Objects.hashCode(exadataInfrastructureId());
        hashCode = 31 * hashCode + Objects.hashCode(ocid());
        hashCode = 31 * hashCode + Objects.hashCode(ociResourceAnchorName());
        hashCode = 31 * hashCode + Objects.hashCode(maxCpuCount());
        hashCode = 31 * hashCode + Objects.hashCode(maxDbNodeStorageInGBs());
        hashCode = 31 * hashCode + Objects.hashCode(maxMemoryInGBs());
        hashCode = 31 * hashCode + Objects.hashCode(memorySizeInGBs());
        hashCode = 31 * hashCode + Objects.hashCode(shape());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasVmClusterIds() ? vmClusterIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(computeModelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAutonomousVmClusterIds() ? autonomousVmClusterIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAutonomousVirtualMachineIds() ? autonomousVirtualMachineIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DbServerSummary)) {
            return false;
        }
        DbServerSummary other = (DbServerSummary) obj;
        return Objects.equals(dbServerId(), other.dbServerId()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason()) && Objects.equals(cpuCoreCount(), other.cpuCoreCount())
                && Objects.equals(dbNodeStorageSizeInGBs(), other.dbNodeStorageSizeInGBs())
                && Objects.equals(dbServerPatchingDetails(), other.dbServerPatchingDetails())
                && Objects.equals(displayName(), other.displayName())
                && Objects.equals(exadataInfrastructureId(), other.exadataInfrastructureId())
                && Objects.equals(ocid(), other.ocid()) && Objects.equals(ociResourceAnchorName(), other.ociResourceAnchorName())
                && Objects.equals(maxCpuCount(), other.maxCpuCount())
                && Objects.equals(maxDbNodeStorageInGBs(), other.maxDbNodeStorageInGBs())
                && Objects.equals(maxMemoryInGBs(), other.maxMemoryInGBs())
                && Objects.equals(memorySizeInGBs(), other.memorySizeInGBs()) && Objects.equals(shape(), other.shape())
                && Objects.equals(createdAt(), other.createdAt()) && hasVmClusterIds() == other.hasVmClusterIds()
                && Objects.equals(vmClusterIds(), other.vmClusterIds())
                && Objects.equals(computeModelAsString(), other.computeModelAsString())
                && hasAutonomousVmClusterIds() == other.hasAutonomousVmClusterIds()
                && Objects.equals(autonomousVmClusterIds(), other.autonomousVmClusterIds())
                && hasAutonomousVirtualMachineIds() == other.hasAutonomousVirtualMachineIds()
                && Objects.equals(autonomousVirtualMachineIds(), other.autonomousVirtualMachineIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DbServerSummary").add("DbServerId", dbServerId()).add("Status", statusAsString())
                .add("StatusReason", statusReason()).add("CpuCoreCount", cpuCoreCount())
                .add("DbNodeStorageSizeInGBs", dbNodeStorageSizeInGBs())
                .add("DbServerPatchingDetails", dbServerPatchingDetails()).add("DisplayName", displayName())
                .add("ExadataInfrastructureId", exadataInfrastructureId()).add("Ocid", ocid())
                .add("OciResourceAnchorName", ociResourceAnchorName()).add("MaxCpuCount", maxCpuCount())
                .add("MaxDbNodeStorageInGBs", maxDbNodeStorageInGBs()).add("MaxMemoryInGBs", maxMemoryInGBs())
                .add("MemorySizeInGBs", memorySizeInGBs()).add("Shape", shape()).add("CreatedAt", createdAt())
                .add("VmClusterIds", hasVmClusterIds() ? vmClusterIds() : null).add("ComputeModel", computeModelAsString())
                .add("AutonomousVmClusterIds", hasAutonomousVmClusterIds() ? autonomousVmClusterIds() : null)
                .add("AutonomousVirtualMachineIds", hasAutonomousVirtualMachineIds() ? autonomousVirtualMachineIds() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "dbServerId":
            return Optional.ofNullable(clazz.cast(dbServerId()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "cpuCoreCount":
            return Optional.ofNullable(clazz.cast(cpuCoreCount()));
        case "dbNodeStorageSizeInGBs":
            return Optional.ofNullable(clazz.cast(dbNodeStorageSizeInGBs()));
        case "dbServerPatchingDetails":
            return Optional.ofNullable(clazz.cast(dbServerPatchingDetails()));
        case "displayName":
            return Optional.ofNullable(clazz.cast(displayName()));
        case "exadataInfrastructureId":
            return Optional.ofNullable(clazz.cast(exadataInfrastructureId()));
        case "ocid":
            return Optional.ofNullable(clazz.cast(ocid()));
        case "ociResourceAnchorName":
            return Optional.ofNullable(clazz.cast(ociResourceAnchorName()));
        case "maxCpuCount":
            return Optional.ofNullable(clazz.cast(maxCpuCount()));
        case "maxDbNodeStorageInGBs":
            return Optional.ofNullable(clazz.cast(maxDbNodeStorageInGBs()));
        case "maxMemoryInGBs":
            return Optional.ofNullable(clazz.cast(maxMemoryInGBs()));
        case "memorySizeInGBs":
            return Optional.ofNullable(clazz.cast(memorySizeInGBs()));
        case "shape":
            return Optional.ofNullable(clazz.cast(shape()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "vmClusterIds":
            return Optional.ofNullable(clazz.cast(vmClusterIds()));
        case "computeModel":
            return Optional.ofNullable(clazz.cast(computeModelAsString()));
        case "autonomousVmClusterIds":
            return Optional.ofNullable(clazz.cast(autonomousVmClusterIds()));
        case "autonomousVirtualMachineIds":
            return Optional.ofNullable(clazz.cast(autonomousVirtualMachineIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("dbServerId", DB_SERVER_ID_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("statusReason", STATUS_REASON_FIELD);
        map.put("cpuCoreCount", CPU_CORE_COUNT_FIELD);
        map.put("dbNodeStorageSizeInGBs", DB_NODE_STORAGE_SIZE_IN_G_BS_FIELD);
        map.put("dbServerPatchingDetails", DB_SERVER_PATCHING_DETAILS_FIELD);
        map.put("displayName", DISPLAY_NAME_FIELD);
        map.put("exadataInfrastructureId", EXADATA_INFRASTRUCTURE_ID_FIELD);
        map.put("ocid", OCID_FIELD);
        map.put("ociResourceAnchorName", OCI_RESOURCE_ANCHOR_NAME_FIELD);
        map.put("maxCpuCount", MAX_CPU_COUNT_FIELD);
        map.put("maxDbNodeStorageInGBs", MAX_DB_NODE_STORAGE_IN_G_BS_FIELD);
        map.put("maxMemoryInGBs", MAX_MEMORY_IN_G_BS_FIELD);
        map.put("memorySizeInGBs", MEMORY_SIZE_IN_G_BS_FIELD);
        map.put("shape", SHAPE_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("vmClusterIds", VM_CLUSTER_IDS_FIELD);
        map.put("computeModel", COMPUTE_MODEL_FIELD);
        map.put("autonomousVmClusterIds", AUTONOMOUS_VM_CLUSTER_IDS_FIELD);
        map.put("autonomousVirtualMachineIds", AUTONOMOUS_VIRTUAL_MACHINE_IDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DbServerSummary, T> g) {
        return obj -> g.apply((DbServerSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DbServerSummary> {
        /**
         * <p>
         * The unique identifier of the database server.
         * </p>
         * 
         * @param dbServerId
         *        The unique identifier of the database server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbServerId(String dbServerId);

        /**
         * <p>
         * The current status of the database server.
         * </p>
         * 
         * @param status
         *        The current status of the database server.
         * @see ResourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the database server.
         * </p>
         * 
         * @param status
         *        The current status of the database server.
         * @see ResourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceStatus
         */
        Builder status(ResourceStatus status);

        /**
         * <p>
         * Additional information about the status of the database server.
         * </p>
         * 
         * @param statusReason
         *        Additional information about the status of the database server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * The number of CPU cores enabled on the database server.
         * </p>
         * 
         * @param cpuCoreCount
         *        The number of CPU cores enabled on the database server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuCoreCount(Integer cpuCoreCount);

        /**
         * <p>
         * The amount of local node storage, in gigabytes (GB), that's allocated on the database server.
         * </p>
         * 
         * @param dbNodeStorageSizeInGBs
         *        The amount of local node storage, in gigabytes (GB), that's allocated on the database server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbNodeStorageSizeInGBs(Integer dbNodeStorageSizeInGBs);

        /**
         * Sets the value of the DbServerPatchingDetails property for this object.
         *
         * @param dbServerPatchingDetails
         *        The new value for the DbServerPatchingDetails property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbServerPatchingDetails(DbServerPatchingDetails dbServerPatchingDetails);

        /**
         * Sets the value of the DbServerPatchingDetails property for this object.
         *
         * This is a convenience method that creates an instance of the {@link DbServerPatchingDetails.Builder} avoiding
         * the need to create one manually via {@link DbServerPatchingDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DbServerPatchingDetails.Builder#build()} is called immediately
         * and its result is passed to {@link #dbServerPatchingDetails(DbServerPatchingDetails)}.
         * 
         * @param dbServerPatchingDetails
         *        a consumer that will call methods on {@link DbServerPatchingDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dbServerPatchingDetails(DbServerPatchingDetails)
         */
        default Builder dbServerPatchingDetails(Consumer<DbServerPatchingDetails.Builder> dbServerPatchingDetails) {
            return dbServerPatchingDetails(DbServerPatchingDetails.builder().applyMutation(dbServerPatchingDetails).build());
        }

        /**
         * <p>
         * The user-friendly name of the database server. The name doesn't need to be unique.
         * </p>
         * 
         * @param displayName
         *        The user-friendly name of the database server. The name doesn't need to be unique.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder displayName(String displayName);

        /**
         * <p>
         * The ID of the Exadata infrastructure that hosts the database server.
         * </p>
         * 
         * @param exadataInfrastructureId
         *        The ID of the Exadata infrastructure that hosts the database server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exadataInfrastructureId(String exadataInfrastructureId);

        /**
         * <p>
         * The OCID of the database server.
         * </p>
         * 
         * @param ocid
         *        The OCID of the database server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ocid(String ocid);

        /**
         * <p>
         * The name of the OCI resource anchor for the database server.
         * </p>
         * 
         * @param ociResourceAnchorName
         *        The name of the OCI resource anchor for the database server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ociResourceAnchorName(String ociResourceAnchorName);

        /**
         * <p>
         * The total number of CPU cores available on the database server.
         * </p>
         * 
         * @param maxCpuCount
         *        The total number of CPU cores available on the database server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxCpuCount(Integer maxCpuCount);

        /**
         * <p>
         * The total amount of local node storage, in gigabytes (GB), that's available on the database server.
         * </p>
         * 
         * @param maxDbNodeStorageInGBs
         *        The total amount of local node storage, in gigabytes (GB), that's available on the database server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxDbNodeStorageInGBs(Integer maxDbNodeStorageInGBs);

        /**
         * <p>
         * The total amount of memory, in gigabytes (GB), that's available on the database server.
         * </p>
         * 
         * @param maxMemoryInGBs
         *        The total amount of memory, in gigabytes (GB), that's available on the database server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxMemoryInGBs(Integer maxMemoryInGBs);

        /**
         * <p>
         * The amount of memory, in gigabytes (GB), that's allocated on the database server.
         * </p>
         * 
         * @param memorySizeInGBs
         *        The amount of memory, in gigabytes (GB), that's allocated on the database server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memorySizeInGBs(Integer memorySizeInGBs);

        /**
         * <p>
         * The hardware system model of the Exadata infrastructure that the database server is hosted on. The shape
         * determines the amount of CPU, storage, and memory resources available.
         * </p>
         * 
         * @param shape
         *        The hardware system model of the Exadata infrastructure that the database server is hosted on. The
         *        shape determines the amount of CPU, storage, and memory resources available.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shape(String shape);

        /**
         * <p>
         * The date and time when the database server was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time when the database server was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The IDs of the VM clusters that are associated with the database server.
         * </p>
         * 
         * @param vmClusterIds
         *        The IDs of the VM clusters that are associated with the database server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vmClusterIds(Collection<String> vmClusterIds);

        /**
         * <p>
         * The IDs of the VM clusters that are associated with the database server.
         * </p>
         * 
         * @param vmClusterIds
         *        The IDs of the VM clusters that are associated with the database server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vmClusterIds(String... vmClusterIds);

        /**
         * <p>
         * The OCI model compute model used when you create or clone an instance: ECPU or OCPU. An ECPU is an abstracted
         * measure of compute resources. ECPUs are based on the number of cores elastically allocated from a pool of
         * compute and storage servers. An OCPU is a legacy physical measure of compute resources. OCPUs are based on
         * the physical core of a processor with hyper-threading enabled.
         * </p>
         * 
         * @param computeModel
         *        The OCI model compute model used when you create or clone an instance: ECPU or OCPU. An ECPU is an
         *        abstracted measure of compute resources. ECPUs are based on the number of cores elastically allocated
         *        from a pool of compute and storage servers. An OCPU is a legacy physical measure of compute resources.
         *        OCPUs are based on the physical core of a processor with hyper-threading enabled.
         * @see ComputeModel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComputeModel
         */
        Builder computeModel(String computeModel);

        /**
         * <p>
         * The OCI model compute model used when you create or clone an instance: ECPU or OCPU. An ECPU is an abstracted
         * measure of compute resources. ECPUs are based on the number of cores elastically allocated from a pool of
         * compute and storage servers. An OCPU is a legacy physical measure of compute resources. OCPUs are based on
         * the physical core of a processor with hyper-threading enabled.
         * </p>
         * 
         * @param computeModel
         *        The OCI model compute model used when you create or clone an instance: ECPU or OCPU. An ECPU is an
         *        abstracted measure of compute resources. ECPUs are based on the number of cores elastically allocated
         *        from a pool of compute and storage servers. An OCPU is a legacy physical measure of compute resources.
         *        OCPUs are based on the physical core of a processor with hyper-threading enabled.
         * @see ComputeModel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComputeModel
         */
        Builder computeModel(ComputeModel computeModel);

        /**
         * <p>
         * A list of identifiers for the Autonomous VM clusters.
         * </p>
         * 
         * @param autonomousVmClusterIds
         *        A list of identifiers for the Autonomous VM clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autonomousVmClusterIds(Collection<String> autonomousVmClusterIds);

        /**
         * <p>
         * A list of identifiers for the Autonomous VM clusters.
         * </p>
         * 
         * @param autonomousVmClusterIds
         *        A list of identifiers for the Autonomous VM clusters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autonomousVmClusterIds(String... autonomousVmClusterIds);

        /**
         * <p>
         * A list of unique identifiers for the Autonomous VMs.
         * </p>
         * 
         * @param autonomousVirtualMachineIds
         *        A list of unique identifiers for the Autonomous VMs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autonomousVirtualMachineIds(Collection<String> autonomousVirtualMachineIds);

        /**
         * <p>
         * A list of unique identifiers for the Autonomous VMs.
         * </p>
         * 
         * @param autonomousVirtualMachineIds
         *        A list of unique identifiers for the Autonomous VMs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autonomousVirtualMachineIds(String... autonomousVirtualMachineIds);
    }

    static final class BuilderImpl implements Builder {
        private String dbServerId;

        private String status;

        private String statusReason;

        private Integer cpuCoreCount;

        private Integer dbNodeStorageSizeInGBs;

        private DbServerPatchingDetails dbServerPatchingDetails;

        private String displayName;

        private String exadataInfrastructureId;

        private String ocid;

        private String ociResourceAnchorName;

        private Integer maxCpuCount;

        private Integer maxDbNodeStorageInGBs;

        private Integer maxMemoryInGBs;

        private Integer memorySizeInGBs;

        private String shape;

        private Instant createdAt;

        private List<String> vmClusterIds = DefaultSdkAutoConstructList.getInstance();

        private String computeModel;

        private List<String> autonomousVmClusterIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> autonomousVirtualMachineIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DbServerSummary model) {
            dbServerId(model.dbServerId);
            status(model.status);
            statusReason(model.statusReason);
            cpuCoreCount(model.cpuCoreCount);
            dbNodeStorageSizeInGBs(model.dbNodeStorageSizeInGBs);
            dbServerPatchingDetails(model.dbServerPatchingDetails);
            displayName(model.displayName);
            exadataInfrastructureId(model.exadataInfrastructureId);
            ocid(model.ocid);
            ociResourceAnchorName(model.ociResourceAnchorName);
            maxCpuCount(model.maxCpuCount);
            maxDbNodeStorageInGBs(model.maxDbNodeStorageInGBs);
            maxMemoryInGBs(model.maxMemoryInGBs);
            memorySizeInGBs(model.memorySizeInGBs);
            shape(model.shape);
            createdAt(model.createdAt);
            vmClusterIds(model.vmClusterIds);
            computeModel(model.computeModel);
            autonomousVmClusterIds(model.autonomousVmClusterIds);
            autonomousVirtualMachineIds(model.autonomousVirtualMachineIds);
        }

        public final String getDbServerId() {
            return dbServerId;
        }

        public final void setDbServerId(String dbServerId) {
            this.dbServerId = dbServerId;
        }

        @Override
        public final Builder dbServerId(String dbServerId) {
            this.dbServerId = dbServerId;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ResourceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final Integer getCpuCoreCount() {
            return cpuCoreCount;
        }

        public final void setCpuCoreCount(Integer cpuCoreCount) {
            this.cpuCoreCount = cpuCoreCount;
        }

        @Override
        public final Builder cpuCoreCount(Integer cpuCoreCount) {
            this.cpuCoreCount = cpuCoreCount;
            return this;
        }

        public final Integer getDbNodeStorageSizeInGBs() {
            return dbNodeStorageSizeInGBs;
        }

        public final void setDbNodeStorageSizeInGBs(Integer dbNodeStorageSizeInGBs) {
            this.dbNodeStorageSizeInGBs = dbNodeStorageSizeInGBs;
        }

        @Override
        public final Builder dbNodeStorageSizeInGBs(Integer dbNodeStorageSizeInGBs) {
            this.dbNodeStorageSizeInGBs = dbNodeStorageSizeInGBs;
            return this;
        }

        public final DbServerPatchingDetails.Builder getDbServerPatchingDetails() {
            return dbServerPatchingDetails != null ? dbServerPatchingDetails.toBuilder() : null;
        }

        public final void setDbServerPatchingDetails(DbServerPatchingDetails.BuilderImpl dbServerPatchingDetails) {
            this.dbServerPatchingDetails = dbServerPatchingDetails != null ? dbServerPatchingDetails.build() : null;
        }

        @Override
        public final Builder dbServerPatchingDetails(DbServerPatchingDetails dbServerPatchingDetails) {
            this.dbServerPatchingDetails = dbServerPatchingDetails;
            return this;
        }

        public final String getDisplayName() {
            return displayName;
        }

        public final void setDisplayName(String displayName) {
            this.displayName = displayName;
        }

        @Override
        public final Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public final String getExadataInfrastructureId() {
            return exadataInfrastructureId;
        }

        public final void setExadataInfrastructureId(String exadataInfrastructureId) {
            this.exadataInfrastructureId = exadataInfrastructureId;
        }

        @Override
        public final Builder exadataInfrastructureId(String exadataInfrastructureId) {
            this.exadataInfrastructureId = exadataInfrastructureId;
            return this;
        }

        public final String getOcid() {
            return ocid;
        }

        public final void setOcid(String ocid) {
            this.ocid = ocid;
        }

        @Override
        public final Builder ocid(String ocid) {
            this.ocid = ocid;
            return this;
        }

        public final String getOciResourceAnchorName() {
            return ociResourceAnchorName;
        }

        public final void setOciResourceAnchorName(String ociResourceAnchorName) {
            this.ociResourceAnchorName = ociResourceAnchorName;
        }

        @Override
        public final Builder ociResourceAnchorName(String ociResourceAnchorName) {
            this.ociResourceAnchorName = ociResourceAnchorName;
            return this;
        }

        public final Integer getMaxCpuCount() {
            return maxCpuCount;
        }

        public final void setMaxCpuCount(Integer maxCpuCount) {
            this.maxCpuCount = maxCpuCount;
        }

        @Override
        public final Builder maxCpuCount(Integer maxCpuCount) {
            this.maxCpuCount = maxCpuCount;
            return this;
        }

        public final Integer getMaxDbNodeStorageInGBs() {
            return maxDbNodeStorageInGBs;
        }

        public final void setMaxDbNodeStorageInGBs(Integer maxDbNodeStorageInGBs) {
            this.maxDbNodeStorageInGBs = maxDbNodeStorageInGBs;
        }

        @Override
        public final Builder maxDbNodeStorageInGBs(Integer maxDbNodeStorageInGBs) {
            this.maxDbNodeStorageInGBs = maxDbNodeStorageInGBs;
            return this;
        }

        public final Integer getMaxMemoryInGBs() {
            return maxMemoryInGBs;
        }

        public final void setMaxMemoryInGBs(Integer maxMemoryInGBs) {
            this.maxMemoryInGBs = maxMemoryInGBs;
        }

        @Override
        public final Builder maxMemoryInGBs(Integer maxMemoryInGBs) {
            this.maxMemoryInGBs = maxMemoryInGBs;
            return this;
        }

        public final Integer getMemorySizeInGBs() {
            return memorySizeInGBs;
        }

        public final void setMemorySizeInGBs(Integer memorySizeInGBs) {
            this.memorySizeInGBs = memorySizeInGBs;
        }

        @Override
        public final Builder memorySizeInGBs(Integer memorySizeInGBs) {
            this.memorySizeInGBs = memorySizeInGBs;
            return this;
        }

        public final String getShape() {
            return shape;
        }

        public final void setShape(String shape) {
            this.shape = shape;
        }

        @Override
        public final Builder shape(String shape) {
            this.shape = shape;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Collection<String> getVmClusterIds() {
            if (vmClusterIds instanceof SdkAutoConstructList) {
                return null;
            }
            return vmClusterIds;
        }

        public final void setVmClusterIds(Collection<String> vmClusterIds) {
            this.vmClusterIds = StringListCopier.copy(vmClusterIds);
        }

        @Override
        public final Builder vmClusterIds(Collection<String> vmClusterIds) {
            this.vmClusterIds = StringListCopier.copy(vmClusterIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vmClusterIds(String... vmClusterIds) {
            vmClusterIds(Arrays.asList(vmClusterIds));
            return this;
        }

        public final String getComputeModel() {
            return computeModel;
        }

        public final void setComputeModel(String computeModel) {
            this.computeModel = computeModel;
        }

        @Override
        public final Builder computeModel(String computeModel) {
            this.computeModel = computeModel;
            return this;
        }

        @Override
        public final Builder computeModel(ComputeModel computeModel) {
            this.computeModel(computeModel == null ? null : computeModel.toString());
            return this;
        }

        public final Collection<String> getAutonomousVmClusterIds() {
            if (autonomousVmClusterIds instanceof SdkAutoConstructList) {
                return null;
            }
            return autonomousVmClusterIds;
        }

        public final void setAutonomousVmClusterIds(Collection<String> autonomousVmClusterIds) {
            this.autonomousVmClusterIds = StringListCopier.copy(autonomousVmClusterIds);
        }

        @Override
        public final Builder autonomousVmClusterIds(Collection<String> autonomousVmClusterIds) {
            this.autonomousVmClusterIds = StringListCopier.copy(autonomousVmClusterIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder autonomousVmClusterIds(String... autonomousVmClusterIds) {
            autonomousVmClusterIds(Arrays.asList(autonomousVmClusterIds));
            return this;
        }

        public final Collection<String> getAutonomousVirtualMachineIds() {
            if (autonomousVirtualMachineIds instanceof SdkAutoConstructList) {
                return null;
            }
            return autonomousVirtualMachineIds;
        }

        public final void setAutonomousVirtualMachineIds(Collection<String> autonomousVirtualMachineIds) {
            this.autonomousVirtualMachineIds = StringListCopier.copy(autonomousVirtualMachineIds);
        }

        @Override
        public final Builder autonomousVirtualMachineIds(Collection<String> autonomousVirtualMachineIds) {
            this.autonomousVirtualMachineIds = StringListCopier.copy(autonomousVirtualMachineIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder autonomousVirtualMachineIds(String... autonomousVirtualMachineIds) {
            autonomousVirtualMachineIds(Arrays.asList(autonomousVirtualMachineIds));
            return this;
        }

        @Override
        public DbServerSummary build() {
            return new DbServerSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
