/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkfirewall.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Detailed information about the current status of a <a>Firewall</a>. You can retrieve this for a firewall by calling
 * <a>DescribeFirewall</a> and providing the firewall name and ARN.
 * </p>
 * <p>
 * The firewall status indicates a combined status. It indicates whether all subnets are up-to-date with the latest
 * firewall configurations, which is based on the sync states config values, and also whether all subnets have their
 * endpoints fully enabled, based on their sync states attachment values.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FirewallStatus implements SdkPojo, Serializable, ToCopyableBuilder<FirewallStatus.Builder, FirewallStatus> {
    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(FirewallStatus::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> CONFIGURATION_SYNC_STATE_SUMMARY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ConfigurationSyncStateSummary")
            .getter(getter(FirewallStatus::configurationSyncStateSummaryAsString))
            .setter(setter(Builder::configurationSyncStateSummary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationSyncStateSummary")
                    .build()).build();

    private static final SdkField<Map<String, SyncState>> SYNC_STATES_FIELD = SdkField
            .<Map<String, SyncState>> builder(MarshallingType.MAP)
            .memberName("SyncStates")
            .getter(getter(FirewallStatus::syncStates))
            .setter(setter(Builder::syncStates))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SyncStates").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<SyncState> builder(MarshallingType.SDK_POJO)
                                            .constructor(SyncState::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<CapacityUsageSummary> CAPACITY_USAGE_SUMMARY_FIELD = SdkField
            .<CapacityUsageSummary> builder(MarshallingType.SDK_POJO).memberName("CapacityUsageSummary")
            .getter(getter(FirewallStatus::capacityUsageSummary)).setter(setter(Builder::capacityUsageSummary))
            .constructor(CapacityUsageSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityUsageSummary").build())
            .build();

    private static final SdkField<TransitGatewayAttachmentSyncState> TRANSIT_GATEWAY_ATTACHMENT_SYNC_STATE_FIELD = SdkField
            .<TransitGatewayAttachmentSyncState> builder(MarshallingType.SDK_POJO)
            .memberName("TransitGatewayAttachmentSyncState")
            .getter(getter(FirewallStatus::transitGatewayAttachmentSyncState))
            .setter(setter(Builder::transitGatewayAttachmentSyncState))
            .constructor(TransitGatewayAttachmentSyncState::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransitGatewayAttachmentSyncState")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD,
            CONFIGURATION_SYNC_STATE_SUMMARY_FIELD, SYNC_STATES_FIELD, CAPACITY_USAGE_SUMMARY_FIELD,
            TRANSIT_GATEWAY_ATTACHMENT_SYNC_STATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String status;

    private final String configurationSyncStateSummary;

    private final Map<String, SyncState> syncStates;

    private final CapacityUsageSummary capacityUsageSummary;

    private final TransitGatewayAttachmentSyncState transitGatewayAttachmentSyncState;

    private FirewallStatus(BuilderImpl builder) {
        this.status = builder.status;
        this.configurationSyncStateSummary = builder.configurationSyncStateSummary;
        this.syncStates = builder.syncStates;
        this.capacityUsageSummary = builder.capacityUsageSummary;
        this.transitGatewayAttachmentSyncState = builder.transitGatewayAttachmentSyncState;
    }

    /**
     * <p>
     * The readiness of the configured firewall to handle network traffic across all of the Availability Zones where you
     * have it configured. This setting is <code>READY</code> only when the <code>ConfigurationSyncStateSummary</code>
     * value is <code>IN_SYNC</code> and the <code>Attachment</code> <code>Status</code> values for all of the
     * configured subnets are <code>READY</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link FirewallStatusValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The readiness of the configured firewall to handle network traffic across all of the Availability Zones
     *         where you have it configured. This setting is <code>READY</code> only when the
     *         <code>ConfigurationSyncStateSummary</code> value is <code>IN_SYNC</code> and the <code>Attachment</code>
     *         <code>Status</code> values for all of the configured subnets are <code>READY</code>.
     * @see FirewallStatusValue
     */
    public final FirewallStatusValue status() {
        return FirewallStatusValue.fromValue(status);
    }

    /**
     * <p>
     * The readiness of the configured firewall to handle network traffic across all of the Availability Zones where you
     * have it configured. This setting is <code>READY</code> only when the <code>ConfigurationSyncStateSummary</code>
     * value is <code>IN_SYNC</code> and the <code>Attachment</code> <code>Status</code> values for all of the
     * configured subnets are <code>READY</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link FirewallStatusValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The readiness of the configured firewall to handle network traffic across all of the Availability Zones
     *         where you have it configured. This setting is <code>READY</code> only when the
     *         <code>ConfigurationSyncStateSummary</code> value is <code>IN_SYNC</code> and the <code>Attachment</code>
     *         <code>Status</code> values for all of the configured subnets are <code>READY</code>.
     * @see FirewallStatusValue
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The configuration sync state for the firewall. This summarizes the <code>Config</code> settings in the
     * <code>SyncStates</code> for this firewall status object.
     * </p>
     * <p>
     * When you create a firewall or update its configuration, for example by adding a rule group to its firewall
     * policy, Network Firewall distributes the configuration changes to all Availability Zones that have subnets
     * defined for the firewall. This summary indicates whether the configuration changes have been applied everywhere.
     * </p>
     * <p>
     * This status must be <code>IN_SYNC</code> for the firewall to be ready for use, but it doesn't indicate that the
     * firewall is ready. The <code>Status</code> setting indicates firewall readiness. It's based on this setting and
     * the readiness of the firewall endpoints to take traffic.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #configurationSyncStateSummary} will return {@link ConfigurationSyncState#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #configurationSyncStateSummaryAsString}.
     * </p>
     * 
     * @return The configuration sync state for the firewall. This summarizes the <code>Config</code> settings in the
     *         <code>SyncStates</code> for this firewall status object. </p>
     *         <p>
     *         When you create a firewall or update its configuration, for example by adding a rule group to its
     *         firewall policy, Network Firewall distributes the configuration changes to all Availability Zones that
     *         have subnets defined for the firewall. This summary indicates whether the configuration changes have been
     *         applied everywhere.
     *         </p>
     *         <p>
     *         This status must be <code>IN_SYNC</code> for the firewall to be ready for use, but it doesn't indicate
     *         that the firewall is ready. The <code>Status</code> setting indicates firewall readiness. It's based on
     *         this setting and the readiness of the firewall endpoints to take traffic.
     * @see ConfigurationSyncState
     */
    public final ConfigurationSyncState configurationSyncStateSummary() {
        return ConfigurationSyncState.fromValue(configurationSyncStateSummary);
    }

    /**
     * <p>
     * The configuration sync state for the firewall. This summarizes the <code>Config</code> settings in the
     * <code>SyncStates</code> for this firewall status object.
     * </p>
     * <p>
     * When you create a firewall or update its configuration, for example by adding a rule group to its firewall
     * policy, Network Firewall distributes the configuration changes to all Availability Zones that have subnets
     * defined for the firewall. This summary indicates whether the configuration changes have been applied everywhere.
     * </p>
     * <p>
     * This status must be <code>IN_SYNC</code> for the firewall to be ready for use, but it doesn't indicate that the
     * firewall is ready. The <code>Status</code> setting indicates firewall readiness. It's based on this setting and
     * the readiness of the firewall endpoints to take traffic.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #configurationSyncStateSummary} will return {@link ConfigurationSyncState#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #configurationSyncStateSummaryAsString}.
     * </p>
     * 
     * @return The configuration sync state for the firewall. This summarizes the <code>Config</code> settings in the
     *         <code>SyncStates</code> for this firewall status object. </p>
     *         <p>
     *         When you create a firewall or update its configuration, for example by adding a rule group to its
     *         firewall policy, Network Firewall distributes the configuration changes to all Availability Zones that
     *         have subnets defined for the firewall. This summary indicates whether the configuration changes have been
     *         applied everywhere.
     *         </p>
     *         <p>
     *         This status must be <code>IN_SYNC</code> for the firewall to be ready for use, but it doesn't indicate
     *         that the firewall is ready. The <code>Status</code> setting indicates firewall readiness. It's based on
     *         this setting and the readiness of the firewall endpoints to take traffic.
     * @see ConfigurationSyncState
     */
    public final String configurationSyncStateSummaryAsString() {
        return configurationSyncStateSummary;
    }

    /**
     * For responses, this returns true if the service returned a value for the SyncStates property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSyncStates() {
        return syncStates != null && !(syncStates instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Status for the subnets that you've configured in the firewall. This contains one array element per Availability
     * Zone where you've configured a subnet in the firewall.
     * </p>
     * <p>
     * These objects provide detailed information for the settings <code>ConfigurationSyncStateSummary</code> and
     * <code>Status</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSyncStates} method.
     * </p>
     * 
     * @return Status for the subnets that you've configured in the firewall. This contains one array element per
     *         Availability Zone where you've configured a subnet in the firewall. </p>
     *         <p>
     *         These objects provide detailed information for the settings <code>ConfigurationSyncStateSummary</code>
     *         and <code>Status</code>.
     */
    public final Map<String, SyncState> syncStates() {
        return syncStates;
    }

    /**
     * <p>
     * Describes the capacity usage of the resources contained in a firewall's reference sets. Network Firewall
     * calculates the capacity usage by taking an aggregated count of all of the resources used by all of the reference
     * sets in a firewall.
     * </p>
     * 
     * @return Describes the capacity usage of the resources contained in a firewall's reference sets. Network Firewall
     *         calculates the capacity usage by taking an aggregated count of all of the resources used by all of the
     *         reference sets in a firewall.
     */
    public final CapacityUsageSummary capacityUsageSummary() {
        return capacityUsageSummary;
    }

    /**
     * <p>
     * The synchronization state of the transit gateway attachment. This indicates whether the firewall's transit
     * gateway configuration is properly synchronized and operational. Use this to verify that your transit gateway
     * configuration changes have been applied.
     * </p>
     * 
     * @return The synchronization state of the transit gateway attachment. This indicates whether the firewall's
     *         transit gateway configuration is properly synchronized and operational. Use this to verify that your
     *         transit gateway configuration changes have been applied.
     */
    public final TransitGatewayAttachmentSyncState transitGatewayAttachmentSyncState() {
        return transitGatewayAttachmentSyncState;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(configurationSyncStateSummaryAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasSyncStates() ? syncStates() : null);
        hashCode = 31 * hashCode + Objects.hashCode(capacityUsageSummary());
        hashCode = 31 * hashCode + Objects.hashCode(transitGatewayAttachmentSyncState());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FirewallStatus)) {
            return false;
        }
        FirewallStatus other = (FirewallStatus) obj;
        return Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(configurationSyncStateSummaryAsString(), other.configurationSyncStateSummaryAsString())
                && hasSyncStates() == other.hasSyncStates() && Objects.equals(syncStates(), other.syncStates())
                && Objects.equals(capacityUsageSummary(), other.capacityUsageSummary())
                && Objects.equals(transitGatewayAttachmentSyncState(), other.transitGatewayAttachmentSyncState());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FirewallStatus").add("Status", statusAsString())
                .add("ConfigurationSyncStateSummary", configurationSyncStateSummaryAsString())
                .add("SyncStates", hasSyncStates() ? syncStates() : null).add("CapacityUsageSummary", capacityUsageSummary())
                .add("TransitGatewayAttachmentSyncState", transitGatewayAttachmentSyncState()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "ConfigurationSyncStateSummary":
            return Optional.ofNullable(clazz.cast(configurationSyncStateSummaryAsString()));
        case "SyncStates":
            return Optional.ofNullable(clazz.cast(syncStates()));
        case "CapacityUsageSummary":
            return Optional.ofNullable(clazz.cast(capacityUsageSummary()));
        case "TransitGatewayAttachmentSyncState":
            return Optional.ofNullable(clazz.cast(transitGatewayAttachmentSyncState()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Status", STATUS_FIELD);
        map.put("ConfigurationSyncStateSummary", CONFIGURATION_SYNC_STATE_SUMMARY_FIELD);
        map.put("SyncStates", SYNC_STATES_FIELD);
        map.put("CapacityUsageSummary", CAPACITY_USAGE_SUMMARY_FIELD);
        map.put("TransitGatewayAttachmentSyncState", TRANSIT_GATEWAY_ATTACHMENT_SYNC_STATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<FirewallStatus, T> g) {
        return obj -> g.apply((FirewallStatus) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FirewallStatus> {
        /**
         * <p>
         * The readiness of the configured firewall to handle network traffic across all of the Availability Zones where
         * you have it configured. This setting is <code>READY</code> only when the
         * <code>ConfigurationSyncStateSummary</code> value is <code>IN_SYNC</code> and the <code>Attachment</code>
         * <code>Status</code> values for all of the configured subnets are <code>READY</code>.
         * </p>
         * 
         * @param status
         *        The readiness of the configured firewall to handle network traffic across all of the Availability
         *        Zones where you have it configured. This setting is <code>READY</code> only when the
         *        <code>ConfigurationSyncStateSummary</code> value is <code>IN_SYNC</code> and the
         *        <code>Attachment</code> <code>Status</code> values for all of the configured subnets are
         *        <code>READY</code>.
         * @see FirewallStatusValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FirewallStatusValue
         */
        Builder status(String status);

        /**
         * <p>
         * The readiness of the configured firewall to handle network traffic across all of the Availability Zones where
         * you have it configured. This setting is <code>READY</code> only when the
         * <code>ConfigurationSyncStateSummary</code> value is <code>IN_SYNC</code> and the <code>Attachment</code>
         * <code>Status</code> values for all of the configured subnets are <code>READY</code>.
         * </p>
         * 
         * @param status
         *        The readiness of the configured firewall to handle network traffic across all of the Availability
         *        Zones where you have it configured. This setting is <code>READY</code> only when the
         *        <code>ConfigurationSyncStateSummary</code> value is <code>IN_SYNC</code> and the
         *        <code>Attachment</code> <code>Status</code> values for all of the configured subnets are
         *        <code>READY</code>.
         * @see FirewallStatusValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FirewallStatusValue
         */
        Builder status(FirewallStatusValue status);

        /**
         * <p>
         * The configuration sync state for the firewall. This summarizes the <code>Config</code> settings in the
         * <code>SyncStates</code> for this firewall status object.
         * </p>
         * <p>
         * When you create a firewall or update its configuration, for example by adding a rule group to its firewall
         * policy, Network Firewall distributes the configuration changes to all Availability Zones that have subnets
         * defined for the firewall. This summary indicates whether the configuration changes have been applied
         * everywhere.
         * </p>
         * <p>
         * This status must be <code>IN_SYNC</code> for the firewall to be ready for use, but it doesn't indicate that
         * the firewall is ready. The <code>Status</code> setting indicates firewall readiness. It's based on this
         * setting and the readiness of the firewall endpoints to take traffic.
         * </p>
         * 
         * @param configurationSyncStateSummary
         *        The configuration sync state for the firewall. This summarizes the <code>Config</code> settings in the
         *        <code>SyncStates</code> for this firewall status object. </p>
         *        <p>
         *        When you create a firewall or update its configuration, for example by adding a rule group to its
         *        firewall policy, Network Firewall distributes the configuration changes to all Availability Zones that
         *        have subnets defined for the firewall. This summary indicates whether the configuration changes have
         *        been applied everywhere.
         *        </p>
         *        <p>
         *        This status must be <code>IN_SYNC</code> for the firewall to be ready for use, but it doesn't indicate
         *        that the firewall is ready. The <code>Status</code> setting indicates firewall readiness. It's based
         *        on this setting and the readiness of the firewall endpoints to take traffic.
         * @see ConfigurationSyncState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfigurationSyncState
         */
        Builder configurationSyncStateSummary(String configurationSyncStateSummary);

        /**
         * <p>
         * The configuration sync state for the firewall. This summarizes the <code>Config</code> settings in the
         * <code>SyncStates</code> for this firewall status object.
         * </p>
         * <p>
         * When you create a firewall or update its configuration, for example by adding a rule group to its firewall
         * policy, Network Firewall distributes the configuration changes to all Availability Zones that have subnets
         * defined for the firewall. This summary indicates whether the configuration changes have been applied
         * everywhere.
         * </p>
         * <p>
         * This status must be <code>IN_SYNC</code> for the firewall to be ready for use, but it doesn't indicate that
         * the firewall is ready. The <code>Status</code> setting indicates firewall readiness. It's based on this
         * setting and the readiness of the firewall endpoints to take traffic.
         * </p>
         * 
         * @param configurationSyncStateSummary
         *        The configuration sync state for the firewall. This summarizes the <code>Config</code> settings in the
         *        <code>SyncStates</code> for this firewall status object. </p>
         *        <p>
         *        When you create a firewall or update its configuration, for example by adding a rule group to its
         *        firewall policy, Network Firewall distributes the configuration changes to all Availability Zones that
         *        have subnets defined for the firewall. This summary indicates whether the configuration changes have
         *        been applied everywhere.
         *        </p>
         *        <p>
         *        This status must be <code>IN_SYNC</code> for the firewall to be ready for use, but it doesn't indicate
         *        that the firewall is ready. The <code>Status</code> setting indicates firewall readiness. It's based
         *        on this setting and the readiness of the firewall endpoints to take traffic.
         * @see ConfigurationSyncState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfigurationSyncState
         */
        Builder configurationSyncStateSummary(ConfigurationSyncState configurationSyncStateSummary);

        /**
         * <p>
         * Status for the subnets that you've configured in the firewall. This contains one array element per
         * Availability Zone where you've configured a subnet in the firewall.
         * </p>
         * <p>
         * These objects provide detailed information for the settings <code>ConfigurationSyncStateSummary</code> and
         * <code>Status</code>.
         * </p>
         * 
         * @param syncStates
         *        Status for the subnets that you've configured in the firewall. This contains one array element per
         *        Availability Zone where you've configured a subnet in the firewall. </p>
         *        <p>
         *        These objects provide detailed information for the settings <code>ConfigurationSyncStateSummary</code>
         *        and <code>Status</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder syncStates(Map<String, SyncState> syncStates);

        /**
         * <p>
         * Describes the capacity usage of the resources contained in a firewall's reference sets. Network Firewall
         * calculates the capacity usage by taking an aggregated count of all of the resources used by all of the
         * reference sets in a firewall.
         * </p>
         * 
         * @param capacityUsageSummary
         *        Describes the capacity usage of the resources contained in a firewall's reference sets. Network
         *        Firewall calculates the capacity usage by taking an aggregated count of all of the resources used by
         *        all of the reference sets in a firewall.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityUsageSummary(CapacityUsageSummary capacityUsageSummary);

        /**
         * <p>
         * Describes the capacity usage of the resources contained in a firewall's reference sets. Network Firewall
         * calculates the capacity usage by taking an aggregated count of all of the resources used by all of the
         * reference sets in a firewall.
         * </p>
         * This is a convenience method that creates an instance of the {@link CapacityUsageSummary.Builder} avoiding
         * the need to create one manually via {@link CapacityUsageSummary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CapacityUsageSummary.Builder#build()} is called immediately and
         * its result is passed to {@link #capacityUsageSummary(CapacityUsageSummary)}.
         * 
         * @param capacityUsageSummary
         *        a consumer that will call methods on {@link CapacityUsageSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #capacityUsageSummary(CapacityUsageSummary)
         */
        default Builder capacityUsageSummary(Consumer<CapacityUsageSummary.Builder> capacityUsageSummary) {
            return capacityUsageSummary(CapacityUsageSummary.builder().applyMutation(capacityUsageSummary).build());
        }

        /**
         * <p>
         * The synchronization state of the transit gateway attachment. This indicates whether the firewall's transit
         * gateway configuration is properly synchronized and operational. Use this to verify that your transit gateway
         * configuration changes have been applied.
         * </p>
         * 
         * @param transitGatewayAttachmentSyncState
         *        The synchronization state of the transit gateway attachment. This indicates whether the firewall's
         *        transit gateway configuration is properly synchronized and operational. Use this to verify that your
         *        transit gateway configuration changes have been applied.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transitGatewayAttachmentSyncState(TransitGatewayAttachmentSyncState transitGatewayAttachmentSyncState);

        /**
         * <p>
         * The synchronization state of the transit gateway attachment. This indicates whether the firewall's transit
         * gateway configuration is properly synchronized and operational. Use this to verify that your transit gateway
         * configuration changes have been applied.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link TransitGatewayAttachmentSyncState.Builder} avoiding the need to create one manually via
         * {@link TransitGatewayAttachmentSyncState#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TransitGatewayAttachmentSyncState.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #transitGatewayAttachmentSyncState(TransitGatewayAttachmentSyncState)}.
         * 
         * @param transitGatewayAttachmentSyncState
         *        a consumer that will call methods on {@link TransitGatewayAttachmentSyncState.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transitGatewayAttachmentSyncState(TransitGatewayAttachmentSyncState)
         */
        default Builder transitGatewayAttachmentSyncState(
                Consumer<TransitGatewayAttachmentSyncState.Builder> transitGatewayAttachmentSyncState) {
            return transitGatewayAttachmentSyncState(TransitGatewayAttachmentSyncState.builder()
                    .applyMutation(transitGatewayAttachmentSyncState).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String status;

        private String configurationSyncStateSummary;

        private Map<String, SyncState> syncStates = DefaultSdkAutoConstructMap.getInstance();

        private CapacityUsageSummary capacityUsageSummary;

        private TransitGatewayAttachmentSyncState transitGatewayAttachmentSyncState;

        private BuilderImpl() {
        }

        private BuilderImpl(FirewallStatus model) {
            status(model.status);
            configurationSyncStateSummary(model.configurationSyncStateSummary);
            syncStates(model.syncStates);
            capacityUsageSummary(model.capacityUsageSummary);
            transitGatewayAttachmentSyncState(model.transitGatewayAttachmentSyncState);
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(FirewallStatusValue status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getConfigurationSyncStateSummary() {
            return configurationSyncStateSummary;
        }

        public final void setConfigurationSyncStateSummary(String configurationSyncStateSummary) {
            this.configurationSyncStateSummary = configurationSyncStateSummary;
        }

        @Override
        public final Builder configurationSyncStateSummary(String configurationSyncStateSummary) {
            this.configurationSyncStateSummary = configurationSyncStateSummary;
            return this;
        }

        @Override
        public final Builder configurationSyncStateSummary(ConfigurationSyncState configurationSyncStateSummary) {
            this.configurationSyncStateSummary(configurationSyncStateSummary == null ? null : configurationSyncStateSummary
                    .toString());
            return this;
        }

        public final Map<String, SyncState.Builder> getSyncStates() {
            Map<String, SyncState.Builder> result = SyncStatesCopier.copyToBuilder(this.syncStates);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setSyncStates(Map<String, SyncState.BuilderImpl> syncStates) {
            this.syncStates = SyncStatesCopier.copyFromBuilder(syncStates);
        }

        @Override
        public final Builder syncStates(Map<String, SyncState> syncStates) {
            this.syncStates = SyncStatesCopier.copy(syncStates);
            return this;
        }

        public final CapacityUsageSummary.Builder getCapacityUsageSummary() {
            return capacityUsageSummary != null ? capacityUsageSummary.toBuilder() : null;
        }

        public final void setCapacityUsageSummary(CapacityUsageSummary.BuilderImpl capacityUsageSummary) {
            this.capacityUsageSummary = capacityUsageSummary != null ? capacityUsageSummary.build() : null;
        }

        @Override
        public final Builder capacityUsageSummary(CapacityUsageSummary capacityUsageSummary) {
            this.capacityUsageSummary = capacityUsageSummary;
            return this;
        }

        public final TransitGatewayAttachmentSyncState.Builder getTransitGatewayAttachmentSyncState() {
            return transitGatewayAttachmentSyncState != null ? transitGatewayAttachmentSyncState.toBuilder() : null;
        }

        public final void setTransitGatewayAttachmentSyncState(
                TransitGatewayAttachmentSyncState.BuilderImpl transitGatewayAttachmentSyncState) {
            this.transitGatewayAttachmentSyncState = transitGatewayAttachmentSyncState != null ? transitGatewayAttachmentSyncState
                    .build() : null;
        }

        @Override
        public final Builder transitGatewayAttachmentSyncState(TransitGatewayAttachmentSyncState transitGatewayAttachmentSyncState) {
            this.transitGatewayAttachmentSyncState = transitGatewayAttachmentSyncState;
            return this;
        }

        @Override
        public FirewallStatus build() {
            return new FirewallStatus(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
