/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkfirewall.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The definition and status of the firewall endpoint for a single subnet. In each configured subnet, Network Firewall
 * instantiates a firewall endpoint to handle network traffic.
 * </p>
 * <p>
 * This data type is used for any firewall endpoint type:
 * </p>
 * <ul>
 * <li>
 * <p>
 * For <code>Firewall.SubnetMappings</code>, this <code>Attachment</code> is part of the <code>FirewallStatus</code>
 * sync states information. You define firewall subnets using <code>CreateFirewall</code> and
 * <code>AssociateSubnets</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * For <code>VpcEndpointAssociation</code>, this <code>Attachment</code> is part of the
 * <code>VpcEndpointAssociationStatus</code> sync states information. You define these subnets using
 * <code>CreateVpcEndpointAssociation</code>.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Attachment implements SdkPojo, Serializable, ToCopyableBuilder<Attachment.Builder, Attachment> {
    private static final SdkField<String> SUBNET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubnetId").getter(getter(Attachment::subnetId)).setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId").build()).build();

    private static final SdkField<String> ENDPOINT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointId").getter(getter(Attachment::endpointId)).setter(setter(Builder::endpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Attachment::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusMessage").getter(getter(Attachment::statusMessage)).setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusMessage").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SUBNET_ID_FIELD,
            ENDPOINT_ID_FIELD, STATUS_FIELD, STATUS_MESSAGE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String subnetId;

    private final String endpointId;

    private final String status;

    private final String statusMessage;

    private Attachment(BuilderImpl builder) {
        this.subnetId = builder.subnetId;
        this.endpointId = builder.endpointId;
        this.status = builder.status;
        this.statusMessage = builder.statusMessage;
    }

    /**
     * <p>
     * The unique identifier of the subnet that you've specified to be used for a firewall endpoint.
     * </p>
     * 
     * @return The unique identifier of the subnet that you've specified to be used for a firewall endpoint.
     */
    public final String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * The identifier of the firewall endpoint that Network Firewall has instantiated in the subnet. You use this to
     * identify the firewall endpoint in the VPC route tables, when you redirect the VPC traffic through the endpoint.
     * </p>
     * 
     * @return The identifier of the firewall endpoint that Network Firewall has instantiated in the subnet. You use
     *         this to identify the firewall endpoint in the VPC route tables, when you redirect the VPC traffic through
     *         the endpoint.
     */
    public final String endpointId() {
        return endpointId;
    }

    /**
     * <p>
     * The current status of the firewall endpoint instantiation in the subnet.
     * </p>
     * <p>
     * When this value is <code>READY</code>, the endpoint is available to handle network traffic. Otherwise, this value
     * reflects its state, for example <code>CREATING</code> or <code>DELETING</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AttachmentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the firewall endpoint instantiation in the subnet. </p>
     *         <p>
     *         When this value is <code>READY</code>, the endpoint is available to handle network traffic. Otherwise,
     *         this value reflects its state, for example <code>CREATING</code> or <code>DELETING</code>.
     * @see AttachmentStatus
     */
    public final AttachmentStatus status() {
        return AttachmentStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the firewall endpoint instantiation in the subnet.
     * </p>
     * <p>
     * When this value is <code>READY</code>, the endpoint is available to handle network traffic. Otherwise, this value
     * reflects its state, for example <code>CREATING</code> or <code>DELETING</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AttachmentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the firewall endpoint instantiation in the subnet. </p>
     *         <p>
     *         When this value is <code>READY</code>, the endpoint is available to handle network traffic. Otherwise,
     *         this value reflects its state, for example <code>CREATING</code> or <code>DELETING</code>.
     * @see AttachmentStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * If Network Firewall fails to create or delete the firewall endpoint in the subnet, it populates this with the
     * reason for the error or failure and how to resolve it. A <code>FAILED</code> status indicates a non-recoverable
     * state, and a <code>ERROR</code> status indicates an issue that you can fix. Depending on the error, it can take
     * as many as 15 minutes to populate this field. For more information about the causes for failiure or errors and
     * solutions available for this field, see <a href=
     * "https://docs.aws.amazon.com/network-firewall/latest/developerguide/firewall-troubleshooting-endpoint-failures.html"
     * >Troubleshooting firewall endpoint failures</a> in the <i>Network Firewall Developer Guide</i>.
     * </p>
     * 
     * @return If Network Firewall fails to create or delete the firewall endpoint in the subnet, it populates this with
     *         the reason for the error or failure and how to resolve it. A <code>FAILED</code> status indicates a
     *         non-recoverable state, and a <code>ERROR</code> status indicates an issue that you can fix. Depending on
     *         the error, it can take as many as 15 minutes to populate this field. For more information about the
     *         causes for failiure or errors and solutions available for this field, see <a href=
     *         "https://docs.aws.amazon.com/network-firewall/latest/developerguide/firewall-troubleshooting-endpoint-failures.html"
     *         >Troubleshooting firewall endpoint failures</a> in the <i>Network Firewall Developer Guide</i>.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(endpointId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Attachment)) {
            return false;
        }
        Attachment other = (Attachment) obj;
        return Objects.equals(subnetId(), other.subnetId()) && Objects.equals(endpointId(), other.endpointId())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusMessage(), other.statusMessage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Attachment").add("SubnetId", subnetId()).add("EndpointId", endpointId())
                .add("Status", statusAsString()).add("StatusMessage", statusMessage()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "EndpointId":
            return Optional.ofNullable(clazz.cast(endpointId()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SubnetId", SUBNET_ID_FIELD);
        map.put("EndpointId", ENDPOINT_ID_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("StatusMessage", STATUS_MESSAGE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Attachment, T> g) {
        return obj -> g.apply((Attachment) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Attachment> {
        /**
         * <p>
         * The unique identifier of the subnet that you've specified to be used for a firewall endpoint.
         * </p>
         * 
         * @param subnetId
         *        The unique identifier of the subnet that you've specified to be used for a firewall endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * The identifier of the firewall endpoint that Network Firewall has instantiated in the subnet. You use this to
         * identify the firewall endpoint in the VPC route tables, when you redirect the VPC traffic through the
         * endpoint.
         * </p>
         * 
         * @param endpointId
         *        The identifier of the firewall endpoint that Network Firewall has instantiated in the subnet. You use
         *        this to identify the firewall endpoint in the VPC route tables, when you redirect the VPC traffic
         *        through the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointId(String endpointId);

        /**
         * <p>
         * The current status of the firewall endpoint instantiation in the subnet.
         * </p>
         * <p>
         * When this value is <code>READY</code>, the endpoint is available to handle network traffic. Otherwise, this
         * value reflects its state, for example <code>CREATING</code> or <code>DELETING</code>.
         * </p>
         * 
         * @param status
         *        The current status of the firewall endpoint instantiation in the subnet. </p>
         *        <p>
         *        When this value is <code>READY</code>, the endpoint is available to handle network traffic. Otherwise,
         *        this value reflects its state, for example <code>CREATING</code> or <code>DELETING</code>.
         * @see AttachmentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AttachmentStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the firewall endpoint instantiation in the subnet.
         * </p>
         * <p>
         * When this value is <code>READY</code>, the endpoint is available to handle network traffic. Otherwise, this
         * value reflects its state, for example <code>CREATING</code> or <code>DELETING</code>.
         * </p>
         * 
         * @param status
         *        The current status of the firewall endpoint instantiation in the subnet. </p>
         *        <p>
         *        When this value is <code>READY</code>, the endpoint is available to handle network traffic. Otherwise,
         *        this value reflects its state, for example <code>CREATING</code> or <code>DELETING</code>.
         * @see AttachmentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AttachmentStatus
         */
        Builder status(AttachmentStatus status);

        /**
         * <p>
         * If Network Firewall fails to create or delete the firewall endpoint in the subnet, it populates this with the
         * reason for the error or failure and how to resolve it. A <code>FAILED</code> status indicates a
         * non-recoverable state, and a <code>ERROR</code> status indicates an issue that you can fix. Depending on the
         * error, it can take as many as 15 minutes to populate this field. For more information about the causes for
         * failiure or errors and solutions available for this field, see <a href=
         * "https://docs.aws.amazon.com/network-firewall/latest/developerguide/firewall-troubleshooting-endpoint-failures.html"
         * >Troubleshooting firewall endpoint failures</a> in the <i>Network Firewall Developer Guide</i>.
         * </p>
         * 
         * @param statusMessage
         *        If Network Firewall fails to create or delete the firewall endpoint in the subnet, it populates this
         *        with the reason for the error or failure and how to resolve it. A <code>FAILED</code> status indicates
         *        a non-recoverable state, and a <code>ERROR</code> status indicates an issue that you can fix.
         *        Depending on the error, it can take as many as 15 minutes to populate this field. For more information
         *        about the causes for failiure or errors and solutions available for this field, see <a href=
         *        "https://docs.aws.amazon.com/network-firewall/latest/developerguide/firewall-troubleshooting-endpoint-failures.html"
         *        >Troubleshooting firewall endpoint failures</a> in the <i>Network Firewall Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);
    }

    static final class BuilderImpl implements Builder {
        private String subnetId;

        private String endpointId;

        private String status;

        private String statusMessage;

        private BuilderImpl() {
        }

        private BuilderImpl(Attachment model) {
            subnetId(model.subnetId);
            endpointId(model.endpointId);
            status(model.status);
            statusMessage(model.statusMessage);
        }

        public final String getSubnetId() {
            return subnetId;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final String getEndpointId() {
            return endpointId;
        }

        public final void setEndpointId(String endpointId) {
            this.endpointId = endpointId;
        }

        @Override
        public final Builder endpointId(String endpointId) {
            this.endpointId = endpointId;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AttachmentStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        @Override
        public Attachment build() {
            return new Attachment(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
