/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.networkfirewall.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Criteria for Network Firewall to use to inspect an individual packet in stateless rule inspection. Each match
 * attributes set can include one or more items such as IP address, CIDR range, port number, protocol, and TCP flags.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MatchAttributes implements SdkPojo, Serializable, ToCopyableBuilder<MatchAttributes.Builder, MatchAttributes> {
    private static final SdkField<List<Address>> SOURCES_FIELD = SdkField
            .<List<Address>> builder(MarshallingType.LIST)
            .memberName("Sources")
            .getter(getter(MatchAttributes::sources))
            .setter(setter(Builder::sources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Sources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Address> builder(MarshallingType.SDK_POJO)
                                            .constructor(Address::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Address>> DESTINATIONS_FIELD = SdkField
            .<List<Address>> builder(MarshallingType.LIST)
            .memberName("Destinations")
            .getter(getter(MatchAttributes::destinations))
            .setter(setter(Builder::destinations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Destinations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Address> builder(MarshallingType.SDK_POJO)
                                            .constructor(Address::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<PortRange>> SOURCE_PORTS_FIELD = SdkField
            .<List<PortRange>> builder(MarshallingType.LIST)
            .memberName("SourcePorts")
            .getter(getter(MatchAttributes::sourcePorts))
            .setter(setter(Builder::sourcePorts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourcePorts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PortRange> builder(MarshallingType.SDK_POJO)
                                            .constructor(PortRange::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<PortRange>> DESTINATION_PORTS_FIELD = SdkField
            .<List<PortRange>> builder(MarshallingType.LIST)
            .memberName("DestinationPorts")
            .getter(getter(MatchAttributes::destinationPorts))
            .setter(setter(Builder::destinationPorts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationPorts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PortRange> builder(MarshallingType.SDK_POJO)
                                            .constructor(PortRange::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Integer>> PROTOCOLS_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .memberName("Protocols")
            .getter(getter(MatchAttributes::protocols))
            .setter(setter(Builder::protocols))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Protocols").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<TCPFlagField>> TCP_FLAGS_FIELD = SdkField
            .<List<TCPFlagField>> builder(MarshallingType.LIST)
            .memberName("TCPFlags")
            .getter(getter(MatchAttributes::tcpFlags))
            .setter(setter(Builder::tcpFlags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TCPFlags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TCPFlagField> builder(MarshallingType.SDK_POJO)
                                            .constructor(TCPFlagField::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCES_FIELD,
            DESTINATIONS_FIELD, SOURCE_PORTS_FIELD, DESTINATION_PORTS_FIELD, PROTOCOLS_FIELD, TCP_FLAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<Address> sources;

    private final List<Address> destinations;

    private final List<PortRange> sourcePorts;

    private final List<PortRange> destinationPorts;

    private final List<Integer> protocols;

    private final List<TCPFlagField> tcpFlags;

    private MatchAttributes(BuilderImpl builder) {
        this.sources = builder.sources;
        this.destinations = builder.destinations;
        this.sourcePorts = builder.sourcePorts;
        this.destinationPorts = builder.destinationPorts;
        this.protocols = builder.protocols;
        this.tcpFlags = builder.tcpFlags;
    }

    /**
     * Returns true if the Sources property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasSources() {
        return sources != null && !(sources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The source IP addresses and address ranges to inspect for, in CIDR notation. If not specified, this matches with
     * any source address.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSources()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The source IP addresses and address ranges to inspect for, in CIDR notation. If not specified, this
     *         matches with any source address.
     */
    public final List<Address> sources() {
        return sources;
    }

    /**
     * Returns true if the Destinations property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasDestinations() {
        return destinations != null && !(destinations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The destination IP addresses and address ranges to inspect for, in CIDR notation. If not specified, this matches
     * with any destination address.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDestinations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The destination IP addresses and address ranges to inspect for, in CIDR notation. If not specified, this
     *         matches with any destination address.
     */
    public final List<Address> destinations() {
        return destinations;
    }

    /**
     * Returns true if the SourcePorts property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasSourcePorts() {
        return sourcePorts != null && !(sourcePorts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The source ports to inspect for. If not specified, this matches with any source port. This setting is only used
     * for protocols 6 (TCP) and 17 (UDP).
     * </p>
     * <p>
     * You can specify individual ports, for example <code>1994</code> and you can specify port ranges, for example
     * <code>1990-1994</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSourcePorts()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The source ports to inspect for. If not specified, this matches with any source port. This setting is
     *         only used for protocols 6 (TCP) and 17 (UDP). </p>
     *         <p>
     *         You can specify individual ports, for example <code>1994</code> and you can specify port ranges, for
     *         example <code>1990-1994</code>.
     */
    public final List<PortRange> sourcePorts() {
        return sourcePorts;
    }

    /**
     * Returns true if the DestinationPorts property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasDestinationPorts() {
        return destinationPorts != null && !(destinationPorts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The destination ports to inspect for. If not specified, this matches with any destination port. This setting is
     * only used for protocols 6 (TCP) and 17 (UDP).
     * </p>
     * <p>
     * You can specify individual ports, for example <code>1994</code> and you can specify port ranges, for example
     * <code>1990-1994</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDestinationPorts()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The destination ports to inspect for. If not specified, this matches with any destination port. This
     *         setting is only used for protocols 6 (TCP) and 17 (UDP). </p>
     *         <p>
     *         You can specify individual ports, for example <code>1994</code> and you can specify port ranges, for
     *         example <code>1990-1994</code>.
     */
    public final List<PortRange> destinationPorts() {
        return destinationPorts;
    }

    /**
     * Returns true if the Protocols property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasProtocols() {
        return protocols != null && !(protocols instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The protocols to inspect for, specified using each protocol's assigned internet protocol number (IANA). If not
     * specified, this matches with any protocol.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasProtocols()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The protocols to inspect for, specified using each protocol's assigned internet protocol number (IANA).
     *         If not specified, this matches with any protocol.
     */
    public final List<Integer> protocols() {
        return protocols;
    }

    /**
     * Returns true if the TCPFlags property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTcpFlags() {
        return tcpFlags != null && !(tcpFlags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The TCP flags and masks to inspect for. If not specified, this matches with any settings. This setting is only
     * used for protocol 6 (TCP).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTcpFlags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The TCP flags and masks to inspect for. If not specified, this matches with any settings. This setting is
     *         only used for protocol 6 (TCP).
     */
    public final List<TCPFlagField> tcpFlags() {
        return tcpFlags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasSources() ? sources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDestinations() ? destinations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSourcePorts() ? sourcePorts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDestinationPorts() ? destinationPorts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasProtocols() ? protocols() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTcpFlags() ? tcpFlags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MatchAttributes)) {
            return false;
        }
        MatchAttributes other = (MatchAttributes) obj;
        return hasSources() == other.hasSources() && Objects.equals(sources(), other.sources())
                && hasDestinations() == other.hasDestinations() && Objects.equals(destinations(), other.destinations())
                && hasSourcePorts() == other.hasSourcePorts() && Objects.equals(sourcePorts(), other.sourcePorts())
                && hasDestinationPorts() == other.hasDestinationPorts()
                && Objects.equals(destinationPorts(), other.destinationPorts()) && hasProtocols() == other.hasProtocols()
                && Objects.equals(protocols(), other.protocols()) && hasTcpFlags() == other.hasTcpFlags()
                && Objects.equals(tcpFlags(), other.tcpFlags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MatchAttributes").add("Sources", hasSources() ? sources() : null)
                .add("Destinations", hasDestinations() ? destinations() : null)
                .add("SourcePorts", hasSourcePorts() ? sourcePorts() : null)
                .add("DestinationPorts", hasDestinationPorts() ? destinationPorts() : null)
                .add("Protocols", hasProtocols() ? protocols() : null).add("TCPFlags", hasTcpFlags() ? tcpFlags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Sources":
            return Optional.ofNullable(clazz.cast(sources()));
        case "Destinations":
            return Optional.ofNullable(clazz.cast(destinations()));
        case "SourcePorts":
            return Optional.ofNullable(clazz.cast(sourcePorts()));
        case "DestinationPorts":
            return Optional.ofNullable(clazz.cast(destinationPorts()));
        case "Protocols":
            return Optional.ofNullable(clazz.cast(protocols()));
        case "TCPFlags":
            return Optional.ofNullable(clazz.cast(tcpFlags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MatchAttributes, T> g) {
        return obj -> g.apply((MatchAttributes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MatchAttributes> {
        /**
         * <p>
         * The source IP addresses and address ranges to inspect for, in CIDR notation. If not specified, this matches
         * with any source address.
         * </p>
         * 
         * @param sources
         *        The source IP addresses and address ranges to inspect for, in CIDR notation. If not specified, this
         *        matches with any source address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sources(Collection<Address> sources);

        /**
         * <p>
         * The source IP addresses and address ranges to inspect for, in CIDR notation. If not specified, this matches
         * with any source address.
         * </p>
         * 
         * @param sources
         *        The source IP addresses and address ranges to inspect for, in CIDR notation. If not specified, this
         *        matches with any source address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sources(Address... sources);

        /**
         * <p>
         * The source IP addresses and address ranges to inspect for, in CIDR notation. If not specified, this matches
         * with any source address.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Address>.Builder} avoiding the need to
         * create one manually via {@link List<Address>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Address>.Builder#build()} is called immediately and its
         * result is passed to {@link #sources(List<Address>)}.
         * 
         * @param sources
         *        a consumer that will call methods on {@link List<Address>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sources(List<Address>)
         */
        Builder sources(Consumer<Address.Builder>... sources);

        /**
         * <p>
         * The destination IP addresses and address ranges to inspect for, in CIDR notation. If not specified, this
         * matches with any destination address.
         * </p>
         * 
         * @param destinations
         *        The destination IP addresses and address ranges to inspect for, in CIDR notation. If not specified,
         *        this matches with any destination address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(Collection<Address> destinations);

        /**
         * <p>
         * The destination IP addresses and address ranges to inspect for, in CIDR notation. If not specified, this
         * matches with any destination address.
         * </p>
         * 
         * @param destinations
         *        The destination IP addresses and address ranges to inspect for, in CIDR notation. If not specified,
         *        this matches with any destination address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(Address... destinations);

        /**
         * <p>
         * The destination IP addresses and address ranges to inspect for, in CIDR notation. If not specified, this
         * matches with any destination address.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Address>.Builder} avoiding the need to
         * create one manually via {@link List<Address>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Address>.Builder#build()} is called immediately and its
         * result is passed to {@link #destinations(List<Address>)}.
         * 
         * @param destinations
         *        a consumer that will call methods on {@link List<Address>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinations(List<Address>)
         */
        Builder destinations(Consumer<Address.Builder>... destinations);

        /**
         * <p>
         * The source ports to inspect for. If not specified, this matches with any source port. This setting is only
         * used for protocols 6 (TCP) and 17 (UDP).
         * </p>
         * <p>
         * You can specify individual ports, for example <code>1994</code> and you can specify port ranges, for example
         * <code>1990-1994</code>.
         * </p>
         * 
         * @param sourcePorts
         *        The source ports to inspect for. If not specified, this matches with any source port. This setting is
         *        only used for protocols 6 (TCP) and 17 (UDP). </p>
         *        <p>
         *        You can specify individual ports, for example <code>1994</code> and you can specify port ranges, for
         *        example <code>1990-1994</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourcePorts(Collection<PortRange> sourcePorts);

        /**
         * <p>
         * The source ports to inspect for. If not specified, this matches with any source port. This setting is only
         * used for protocols 6 (TCP) and 17 (UDP).
         * </p>
         * <p>
         * You can specify individual ports, for example <code>1994</code> and you can specify port ranges, for example
         * <code>1990-1994</code>.
         * </p>
         * 
         * @param sourcePorts
         *        The source ports to inspect for. If not specified, this matches with any source port. This setting is
         *        only used for protocols 6 (TCP) and 17 (UDP). </p>
         *        <p>
         *        You can specify individual ports, for example <code>1994</code> and you can specify port ranges, for
         *        example <code>1990-1994</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourcePorts(PortRange... sourcePorts);

        /**
         * <p>
         * The source ports to inspect for. If not specified, this matches with any source port. This setting is only
         * used for protocols 6 (TCP) and 17 (UDP).
         * </p>
         * <p>
         * You can specify individual ports, for example <code>1994</code> and you can specify port ranges, for example
         * <code>1990-1994</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<PortRange>.Builder} avoiding the need to
         * create one manually via {@link List<PortRange>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PortRange>.Builder#build()} is called immediately and its
         * result is passed to {@link #sourcePorts(List<PortRange>)}.
         * 
         * @param sourcePorts
         *        a consumer that will call methods on {@link List<PortRange>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourcePorts(List<PortRange>)
         */
        Builder sourcePorts(Consumer<PortRange.Builder>... sourcePorts);

        /**
         * <p>
         * The destination ports to inspect for. If not specified, this matches with any destination port. This setting
         * is only used for protocols 6 (TCP) and 17 (UDP).
         * </p>
         * <p>
         * You can specify individual ports, for example <code>1994</code> and you can specify port ranges, for example
         * <code>1990-1994</code>.
         * </p>
         * 
         * @param destinationPorts
         *        The destination ports to inspect for. If not specified, this matches with any destination port. This
         *        setting is only used for protocols 6 (TCP) and 17 (UDP). </p>
         *        <p>
         *        You can specify individual ports, for example <code>1994</code> and you can specify port ranges, for
         *        example <code>1990-1994</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationPorts(Collection<PortRange> destinationPorts);

        /**
         * <p>
         * The destination ports to inspect for. If not specified, this matches with any destination port. This setting
         * is only used for protocols 6 (TCP) and 17 (UDP).
         * </p>
         * <p>
         * You can specify individual ports, for example <code>1994</code> and you can specify port ranges, for example
         * <code>1990-1994</code>.
         * </p>
         * 
         * @param destinationPorts
         *        The destination ports to inspect for. If not specified, this matches with any destination port. This
         *        setting is only used for protocols 6 (TCP) and 17 (UDP). </p>
         *        <p>
         *        You can specify individual ports, for example <code>1994</code> and you can specify port ranges, for
         *        example <code>1990-1994</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationPorts(PortRange... destinationPorts);

        /**
         * <p>
         * The destination ports to inspect for. If not specified, this matches with any destination port. This setting
         * is only used for protocols 6 (TCP) and 17 (UDP).
         * </p>
         * <p>
         * You can specify individual ports, for example <code>1994</code> and you can specify port ranges, for example
         * <code>1990-1994</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<PortRange>.Builder} avoiding the need to
         * create one manually via {@link List<PortRange>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PortRange>.Builder#build()} is called immediately and its
         * result is passed to {@link #destinationPorts(List<PortRange>)}.
         * 
         * @param destinationPorts
         *        a consumer that will call methods on {@link List<PortRange>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinationPorts(List<PortRange>)
         */
        Builder destinationPorts(Consumer<PortRange.Builder>... destinationPorts);

        /**
         * <p>
         * The protocols to inspect for, specified using each protocol's assigned internet protocol number (IANA). If
         * not specified, this matches with any protocol.
         * </p>
         * 
         * @param protocols
         *        The protocols to inspect for, specified using each protocol's assigned internet protocol number
         *        (IANA). If not specified, this matches with any protocol.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(Collection<Integer> protocols);

        /**
         * <p>
         * The protocols to inspect for, specified using each protocol's assigned internet protocol number (IANA). If
         * not specified, this matches with any protocol.
         * </p>
         * 
         * @param protocols
         *        The protocols to inspect for, specified using each protocol's assigned internet protocol number
         *        (IANA). If not specified, this matches with any protocol.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(Integer... protocols);

        /**
         * <p>
         * The TCP flags and masks to inspect for. If not specified, this matches with any settings. This setting is
         * only used for protocol 6 (TCP).
         * </p>
         * 
         * @param tcpFlags
         *        The TCP flags and masks to inspect for. If not specified, this matches with any settings. This setting
         *        is only used for protocol 6 (TCP).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tcpFlags(Collection<TCPFlagField> tcpFlags);

        /**
         * <p>
         * The TCP flags and masks to inspect for. If not specified, this matches with any settings. This setting is
         * only used for protocol 6 (TCP).
         * </p>
         * 
         * @param tcpFlags
         *        The TCP flags and masks to inspect for. If not specified, this matches with any settings. This setting
         *        is only used for protocol 6 (TCP).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tcpFlags(TCPFlagField... tcpFlags);

        /**
         * <p>
         * The TCP flags and masks to inspect for. If not specified, this matches with any settings. This setting is
         * only used for protocol 6 (TCP).
         * </p>
         * This is a convenience that creates an instance of the {@link List<TCPFlagField>.Builder} avoiding the need to
         * create one manually via {@link List<TCPFlagField>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TCPFlagField>.Builder#build()} is called immediately and its
         * result is passed to {@link #tcpFlags(List<TCPFlagField>)}.
         * 
         * @param tcpFlags
         *        a consumer that will call methods on {@link List<TCPFlagField>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tcpFlags(List<TCPFlagField>)
         */
        Builder tcpFlags(Consumer<TCPFlagField.Builder>... tcpFlags);
    }

    static final class BuilderImpl implements Builder {
        private List<Address> sources = DefaultSdkAutoConstructList.getInstance();

        private List<Address> destinations = DefaultSdkAutoConstructList.getInstance();

        private List<PortRange> sourcePorts = DefaultSdkAutoConstructList.getInstance();

        private List<PortRange> destinationPorts = DefaultSdkAutoConstructList.getInstance();

        private List<Integer> protocols = DefaultSdkAutoConstructList.getInstance();

        private List<TCPFlagField> tcpFlags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MatchAttributes model) {
            sources(model.sources);
            destinations(model.destinations);
            sourcePorts(model.sourcePorts);
            destinationPorts(model.destinationPorts);
            protocols(model.protocols);
            tcpFlags(model.tcpFlags);
        }

        public final Collection<Address.Builder> getSources() {
            if (sources instanceof SdkAutoConstructList) {
                return null;
            }
            return sources != null ? sources.stream().map(Address::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder sources(Collection<Address> sources) {
            this.sources = AddressesCopier.copy(sources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sources(Address... sources) {
            sources(Arrays.asList(sources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sources(Consumer<Address.Builder>... sources) {
            sources(Stream.of(sources).map(c -> Address.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setSources(Collection<Address.BuilderImpl> sources) {
            this.sources = AddressesCopier.copyFromBuilder(sources);
        }

        public final Collection<Address.Builder> getDestinations() {
            if (destinations instanceof SdkAutoConstructList) {
                return null;
            }
            return destinations != null ? destinations.stream().map(Address::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder destinations(Collection<Address> destinations) {
            this.destinations = AddressesCopier.copy(destinations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(Address... destinations) {
            destinations(Arrays.asList(destinations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(Consumer<Address.Builder>... destinations) {
            destinations(Stream.of(destinations).map(c -> Address.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDestinations(Collection<Address.BuilderImpl> destinations) {
            this.destinations = AddressesCopier.copyFromBuilder(destinations);
        }

        public final Collection<PortRange.Builder> getSourcePorts() {
            if (sourcePorts instanceof SdkAutoConstructList) {
                return null;
            }
            return sourcePorts != null ? sourcePorts.stream().map(PortRange::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder sourcePorts(Collection<PortRange> sourcePorts) {
            this.sourcePorts = PortRangesCopier.copy(sourcePorts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourcePorts(PortRange... sourcePorts) {
            sourcePorts(Arrays.asList(sourcePorts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourcePorts(Consumer<PortRange.Builder>... sourcePorts) {
            sourcePorts(Stream.of(sourcePorts).map(c -> PortRange.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setSourcePorts(Collection<PortRange.BuilderImpl> sourcePorts) {
            this.sourcePorts = PortRangesCopier.copyFromBuilder(sourcePorts);
        }

        public final Collection<PortRange.Builder> getDestinationPorts() {
            if (destinationPorts instanceof SdkAutoConstructList) {
                return null;
            }
            return destinationPorts != null ? destinationPorts.stream().map(PortRange::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder destinationPorts(Collection<PortRange> destinationPorts) {
            this.destinationPorts = PortRangesCopier.copy(destinationPorts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinationPorts(PortRange... destinationPorts) {
            destinationPorts(Arrays.asList(destinationPorts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinationPorts(Consumer<PortRange.Builder>... destinationPorts) {
            destinationPorts(Stream.of(destinationPorts).map(c -> PortRange.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDestinationPorts(Collection<PortRange.BuilderImpl> destinationPorts) {
            this.destinationPorts = PortRangesCopier.copyFromBuilder(destinationPorts);
        }

        public final Collection<Integer> getProtocols() {
            if (protocols instanceof SdkAutoConstructList) {
                return null;
            }
            return protocols;
        }

        @Override
        public final Builder protocols(Collection<Integer> protocols) {
            this.protocols = ProtocolNumbersCopier.copy(protocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder protocols(Integer... protocols) {
            protocols(Arrays.asList(protocols));
            return this;
        }

        public final void setProtocols(Collection<Integer> protocols) {
            this.protocols = ProtocolNumbersCopier.copy(protocols);
        }

        public final Collection<TCPFlagField.Builder> getTcpFlags() {
            if (tcpFlags instanceof SdkAutoConstructList) {
                return null;
            }
            return tcpFlags != null ? tcpFlags.stream().map(TCPFlagField::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tcpFlags(Collection<TCPFlagField> tcpFlags) {
            this.tcpFlags = TCPFlagsCopier.copy(tcpFlags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tcpFlags(TCPFlagField... tcpFlags) {
            tcpFlags(Arrays.asList(tcpFlags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tcpFlags(Consumer<TCPFlagField.Builder>... tcpFlags) {
            tcpFlags(Stream.of(tcpFlags).map(c -> TCPFlagField.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTcpFlags(Collection<TCPFlagField.BuilderImpl> tcpFlags) {
            this.tcpFlags = TCPFlagsCopier.copyFromBuilder(tcpFlags);
        }

        @Override
        public MatchAttributes build() {
            return new MatchAttributes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
