/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mwaa;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mwaa.model.AccessDeniedException;
import software.amazon.awssdk.services.mwaa.model.CreateCliTokenRequest;
import software.amazon.awssdk.services.mwaa.model.CreateCliTokenResponse;
import software.amazon.awssdk.services.mwaa.model.CreateEnvironmentRequest;
import software.amazon.awssdk.services.mwaa.model.CreateEnvironmentResponse;
import software.amazon.awssdk.services.mwaa.model.CreateWebLoginTokenRequest;
import software.amazon.awssdk.services.mwaa.model.CreateWebLoginTokenResponse;
import software.amazon.awssdk.services.mwaa.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.mwaa.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.mwaa.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.mwaa.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.mwaa.model.InternalServerException;
import software.amazon.awssdk.services.mwaa.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.mwaa.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.mwaa.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mwaa.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mwaa.model.MwaaException;
import software.amazon.awssdk.services.mwaa.model.MwaaRequest;
import software.amazon.awssdk.services.mwaa.model.PublishMetricsRequest;
import software.amazon.awssdk.services.mwaa.model.PublishMetricsResponse;
import software.amazon.awssdk.services.mwaa.model.ResourceNotFoundException;
import software.amazon.awssdk.services.mwaa.model.TagResourceRequest;
import software.amazon.awssdk.services.mwaa.model.TagResourceResponse;
import software.amazon.awssdk.services.mwaa.model.UntagResourceRequest;
import software.amazon.awssdk.services.mwaa.model.UntagResourceResponse;
import software.amazon.awssdk.services.mwaa.model.UpdateEnvironmentRequest;
import software.amazon.awssdk.services.mwaa.model.UpdateEnvironmentResponse;
import software.amazon.awssdk.services.mwaa.model.ValidationException;
import software.amazon.awssdk.services.mwaa.paginators.ListEnvironmentsIterable;
import software.amazon.awssdk.services.mwaa.transform.CreateCliTokenRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.CreateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.CreateWebLoginTokenRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.DeleteEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.GetEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.ListEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.PublishMetricsRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mwaa.transform.UpdateEnvironmentRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MwaaClient}.
 *
 * @see MwaaClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMwaaClient implements MwaaClient {
    private static final Logger log = Logger.loggerFor(DefaultMwaaClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMwaaClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Create a CLI token to use Airflow CLI.
     * </p>
     *
     * @param createCliTokenRequest
     * @return Result of the CreateCliToken operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.CreateCliToken
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/CreateCliToken" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateCliTokenResponse createCliToken(CreateCliTokenRequest createCliTokenRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCliTokenResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateCliTokenResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCliTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCliToken");
            String hostPrefix = "env.";
            String resolvedHostExpression = "env.";

            return clientHandler.execute(new ClientExecutionParams<CreateCliTokenRequest, CreateCliTokenResponse>()
                    .withOperationName("CreateCliToken").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(createCliTokenRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateCliTokenRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * JSON blob that describes the environment to create.
     * </p>
     *
     * @param createEnvironmentRequest
     *        This section contains the Amazon Managed Workflows for Apache Airflow (MWAA) API reference documentation
     *        to create an environment. For more information, see <a
     *        href="https://docs.aws.amazon.com/mwaa/latest/userguide/get-started.html">Get started with Amazon Managed
     *        Workflows for Apache Airflow</a>.
     * @return Result of the CreateEnvironment operation returned by the service.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.CreateEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/CreateEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateEnvironmentResponse createEnvironment(CreateEnvironmentRequest createEnvironmentRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEnvironment");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<CreateEnvironmentRequest, CreateEnvironmentResponse>()
                    .withOperationName("CreateEnvironment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(createEnvironmentRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create a JWT token to be used to login to Airflow Web UI with claims based Authentication.
     * </p>
     *
     * @param createWebLoginTokenRequest
     * @return Result of the CreateWebLoginToken operation returned by the service.
     * @throws AccessDeniedException
     *         Access to the Airflow Web UI or CLI has been Denied. Please follow the MWAA user guide to setup
     *         permissions to access the Web UI and CLI functionality.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.CreateWebLoginToken
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/CreateWebLoginToken" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateWebLoginTokenResponse createWebLoginToken(CreateWebLoginTokenRequest createWebLoginTokenRequest)
            throws AccessDeniedException, ResourceNotFoundException, ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateWebLoginTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateWebLoginTokenResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWebLoginTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWebLoginToken");
            String hostPrefix = "env.";
            String resolvedHostExpression = "env.";

            return clientHandler.execute(new ClientExecutionParams<CreateWebLoginTokenRequest, CreateWebLoginTokenResponse>()
                    .withOperationName("CreateWebLoginToken").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(createWebLoginTokenRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateWebLoginTokenRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete an existing environment.
     * </p>
     *
     * @param deleteEnvironmentRequest
     * @return Result of the DeleteEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.DeleteEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/DeleteEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteEnvironmentResponse deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEnvironment");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<DeleteEnvironmentRequest, DeleteEnvironmentResponse>()
                    .withOperationName("DeleteEnvironment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(deleteEnvironmentRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get details of an existing environment.
     * </p>
     *
     * @param getEnvironmentRequest
     * @return Result of the GetEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.GetEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/GetEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetEnvironmentResponse getEnvironment(GetEnvironmentRequest getEnvironmentRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnvironment");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<GetEnvironmentRequest, GetEnvironmentResponse>()
                    .withOperationName("GetEnvironment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(getEnvironmentRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List Amazon MWAA Environments.
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return Result of the ListEnvironments operation returned by the service.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.ListEnvironments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListEnvironmentsResponse listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEnvironmentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnvironments");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<ListEnvironmentsRequest, ListEnvironmentsResponse>()
                    .withOperationName("ListEnvironments").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(listEnvironmentsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEnvironmentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List Amazon MWAA Environments.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEnvironments(software.amazon.awssdk.services.mwaa.model.ListEnvironmentsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mwaa.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mwaa.paginators.ListEnvironmentsIterable responses = client
     *             .listEnvironmentsPaginator(request);
     *     for (software.amazon.awssdk.services.mwaa.model.ListEnvironmentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mwaa.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironments(software.amazon.awssdk.services.mwaa.model.ListEnvironmentsRequest)} operation.</b>
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.ListEnvironments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListEnvironmentsIterable listEnvironmentsPaginator(ListEnvironmentsRequest listEnvironmentsRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        return new ListEnvironmentsIterable(this, applyPaginatorUserAgent(listEnvironmentsRequest));
    }

    /**
     * <p>
     * List the tags for MWAA environments.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(listTagsForResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * An operation for publishing metrics from the customers to the Ops plane.
     * </p>
     *
     * @param publishMetricsRequest
     * @return Result of the PublishMetrics operation returned by the service.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.PublishMetrics
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/PublishMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PublishMetricsResponse publishMetrics(PublishMetricsRequest publishMetricsRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PublishMetricsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PublishMetricsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, publishMetricsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PublishMetrics");
            String hostPrefix = "ops.";
            String resolvedHostExpression = "ops.";

            return clientHandler.execute(new ClientExecutionParams<PublishMetricsRequest, PublishMetricsResponse>()
                    .withOperationName("PublishMetrics").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(publishMetricsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PublishMetricsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Add tag to the MWAA environments.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Remove a tag from the MWAA environments.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(untagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update an MWAA environment.
     * </p>
     *
     * @param updateEnvironmentRequest
     * @return Result of the UpdateEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         ResourceNotFoundException: The resource is not available.
     * @throws ValidationException
     *         ValidationException: The provided input is not valid.
     * @throws InternalServerException
     *         InternalServerException: An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MwaaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MwaaClient.UpdateEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mwaa-2020-07-01/UpdateEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateEnvironmentResponse updateEnvironment(UpdateEnvironmentRequest updateEnvironmentRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, MwaaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MWAA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnvironment");
            String hostPrefix = "api.";
            String resolvedHostExpression = "api.";

            return clientHandler.execute(new ClientExecutionParams<UpdateEnvironmentRequest, UpdateEnvironmentResponse>()
                    .withOperationName("UpdateEnvironment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).hostPrefixExpression(resolvedHostExpression)
                    .withInput(updateEnvironmentRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MwaaException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends MwaaRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
