/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconnect;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowOutputs420Exception;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowOutputsRequest;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowOutputsResponse;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowSourcesRequest;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowSourcesResponse;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowVpcInterfacesRequest;
import software.amazon.awssdk.services.mediaconnect.model.AddFlowVpcInterfacesResponse;
import software.amazon.awssdk.services.mediaconnect.model.BadRequestException;
import software.amazon.awssdk.services.mediaconnect.model.CreateFlow420Exception;
import software.amazon.awssdk.services.mediaconnect.model.CreateFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.CreateFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.DeleteFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.DeleteFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.DescribeFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.DescribeFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.DescribeOfferingRequest;
import software.amazon.awssdk.services.mediaconnect.model.DescribeOfferingResponse;
import software.amazon.awssdk.services.mediaconnect.model.DescribeReservationRequest;
import software.amazon.awssdk.services.mediaconnect.model.DescribeReservationResponse;
import software.amazon.awssdk.services.mediaconnect.model.ForbiddenException;
import software.amazon.awssdk.services.mediaconnect.model.GrantFlowEntitlements420Exception;
import software.amazon.awssdk.services.mediaconnect.model.GrantFlowEntitlementsRequest;
import software.amazon.awssdk.services.mediaconnect.model.GrantFlowEntitlementsResponse;
import software.amazon.awssdk.services.mediaconnect.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediaconnect.model.ListEntitlementsRequest;
import software.amazon.awssdk.services.mediaconnect.model.ListEntitlementsResponse;
import software.amazon.awssdk.services.mediaconnect.model.ListFlowsRequest;
import software.amazon.awssdk.services.mediaconnect.model.ListFlowsResponse;
import software.amazon.awssdk.services.mediaconnect.model.ListOfferingsRequest;
import software.amazon.awssdk.services.mediaconnect.model.ListOfferingsResponse;
import software.amazon.awssdk.services.mediaconnect.model.ListReservationsRequest;
import software.amazon.awssdk.services.mediaconnect.model.ListReservationsResponse;
import software.amazon.awssdk.services.mediaconnect.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediaconnect.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediaconnect.model.MediaConnectException;
import software.amazon.awssdk.services.mediaconnect.model.MediaConnectRequest;
import software.amazon.awssdk.services.mediaconnect.model.NotFoundException;
import software.amazon.awssdk.services.mediaconnect.model.PurchaseOfferingRequest;
import software.amazon.awssdk.services.mediaconnect.model.PurchaseOfferingResponse;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowOutputRequest;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowOutputResponse;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowSourceRequest;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowSourceResponse;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowVpcInterfaceRequest;
import software.amazon.awssdk.services.mediaconnect.model.RemoveFlowVpcInterfaceResponse;
import software.amazon.awssdk.services.mediaconnect.model.RevokeFlowEntitlementRequest;
import software.amazon.awssdk.services.mediaconnect.model.RevokeFlowEntitlementResponse;
import software.amazon.awssdk.services.mediaconnect.model.ServiceUnavailableException;
import software.amazon.awssdk.services.mediaconnect.model.StartFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.StartFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.StopFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.StopFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.TagResourceRequest;
import software.amazon.awssdk.services.mediaconnect.model.TagResourceResponse;
import software.amazon.awssdk.services.mediaconnect.model.TooManyRequestsException;
import software.amazon.awssdk.services.mediaconnect.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediaconnect.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowEntitlementRequest;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowEntitlementResponse;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowOutputRequest;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowOutputResponse;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowResponse;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowSourceRequest;
import software.amazon.awssdk.services.mediaconnect.model.UpdateFlowSourceResponse;
import software.amazon.awssdk.services.mediaconnect.paginators.ListEntitlementsIterable;
import software.amazon.awssdk.services.mediaconnect.paginators.ListFlowsIterable;
import software.amazon.awssdk.services.mediaconnect.paginators.ListOfferingsIterable;
import software.amazon.awssdk.services.mediaconnect.paginators.ListReservationsIterable;
import software.amazon.awssdk.services.mediaconnect.transform.AddFlowOutputsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.AddFlowSourcesRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.AddFlowVpcInterfacesRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.CreateFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.DeleteFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.DescribeFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.DescribeOfferingRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.DescribeReservationRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.GrantFlowEntitlementsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.ListEntitlementsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.ListFlowsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.ListOfferingsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.ListReservationsRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.PurchaseOfferingRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.RemoveFlowOutputRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.RemoveFlowSourceRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.RemoveFlowVpcInterfaceRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.RevokeFlowEntitlementRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.StartFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.StopFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UpdateFlowEntitlementRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UpdateFlowOutputRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UpdateFlowRequestMarshaller;
import software.amazon.awssdk.services.mediaconnect.transform.UpdateFlowSourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MediaConnectClient}.
 *
 * @see MediaConnectClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaConnectClient implements MediaConnectClient {
    private static final Logger log = Logger.loggerFor(DefaultMediaConnectClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMediaConnectClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Adds outputs to an existing flow. You can create up to 50 outputs per flow.
     *
     * @param addFlowOutputsRequest
     *        A request to add outputs to the specified flow.
     * @return Result of the AddFlowOutputs operation returned by the service.
     * @throws AddFlowOutputs420Exception
     *         AWS Elemental MediaConnect can't complete this request because this flow already has the maximum number
     *         of allowed outputs (50). For more information, contact AWS Customer Support.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.AddFlowOutputs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/AddFlowOutputs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AddFlowOutputsResponse addFlowOutputs(AddFlowOutputsRequest addFlowOutputsRequest) throws AddFlowOutputs420Exception,
            BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddFlowOutputsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddFlowOutputsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addFlowOutputsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddFlowOutputs");

            return clientHandler.execute(new ClientExecutionParams<AddFlowOutputsRequest, AddFlowOutputsResponse>()
                    .withOperationName("AddFlowOutputs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(addFlowOutputsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AddFlowOutputsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Adds Sources to flow
     *
     * @param addFlowSourcesRequest
     *        A request to add sources to the flow.
     * @return Result of the AddFlowSources operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.AddFlowSources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/AddFlowSources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AddFlowSourcesResponse addFlowSources(AddFlowSourcesRequest addFlowSourcesRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddFlowSourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddFlowSourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addFlowSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddFlowSources");

            return clientHandler.execute(new ClientExecutionParams<AddFlowSourcesRequest, AddFlowSourcesResponse>()
                    .withOperationName("AddFlowSources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(addFlowSourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AddFlowSourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Adds VPC interfaces to flow
     *
     * @param addFlowVpcInterfacesRequest
     *        A request to add VPC interfaces to the flow.
     * @return Result of the AddFlowVpcInterfaces operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.AddFlowVpcInterfaces
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/AddFlowVpcInterfaces"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AddFlowVpcInterfacesResponse addFlowVpcInterfaces(AddFlowVpcInterfacesRequest addFlowVpcInterfacesRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddFlowVpcInterfacesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AddFlowVpcInterfacesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addFlowVpcInterfacesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddFlowVpcInterfaces");

            return clientHandler.execute(new ClientExecutionParams<AddFlowVpcInterfacesRequest, AddFlowVpcInterfacesResponse>()
                    .withOperationName("AddFlowVpcInterfaces").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(addFlowVpcInterfacesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AddFlowVpcInterfacesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Creates a new flow. The request must include one source. The request optionally can include outputs (up to 50)
     * and entitlements (up to 50).
     *
     * @param createFlowRequest
     *        Creates a new flow. The request must include one source. The request optionally can include outputs (up to
     *        50) and entitlements (up to 50).
     * @return Result of the CreateFlow operation returned by the service.
     * @throws CreateFlow420Exception
     *         Your account already contains the maximum number of 20 flows per account, per Region. For more
     *         information, contact AWS Customer Support.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.CreateFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/CreateFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateFlowResponse createFlow(CreateFlowRequest createFlowRequest) throws CreateFlow420Exception, BadRequestException,
            InternalServerErrorException, ForbiddenException, ServiceUnavailableException, TooManyRequestsException,
            AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFlow");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateFlowRequest, CreateFlowResponse>().withOperationName("CreateFlow")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createFlowRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateFlowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Deletes a flow. Before you can delete a flow, you must stop the flow.
     *
     * @param deleteFlowRequest
     * @return Result of the DeleteFlow operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.DeleteFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/DeleteFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteFlowResponse deleteFlow(DeleteFlowRequest deleteFlowRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFlow");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteFlowRequest, DeleteFlowResponse>().withOperationName("DeleteFlow")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteFlowRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteFlowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Displays the details of a flow. The response includes the flow ARN, name, and Availability Zone, as well as
     * details about the source, outputs, and entitlements.
     *
     * @param describeFlowRequest
     * @return Result of the DescribeFlow operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.DescribeFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/DescribeFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeFlowResponse describeFlow(DescribeFlowRequest describeFlowRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFlow");

            return clientHandler.execute(new ClientExecutionParams<DescribeFlowRequest, DescribeFlowResponse>()
                    .withOperationName("DescribeFlow").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeFlowRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFlowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Displays the details of an offering. The response includes the offering description, duration, outbound
     * bandwidth, price, and Amazon Resource Name (ARN).
     *
     * @param describeOfferingRequest
     * @return Result of the DescribeOffering operation returned by the service.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.DescribeOffering
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/DescribeOffering" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeOfferingResponse describeOffering(DescribeOfferingRequest describeOfferingRequest) throws NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, BadRequestException, InternalServerErrorException,
            AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeOfferingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeOfferingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeOfferingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeOffering");

            return clientHandler.execute(new ClientExecutionParams<DescribeOfferingRequest, DescribeOfferingResponse>()
                    .withOperationName("DescribeOffering").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeOfferingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeOfferingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Displays the details of a reservation. The response includes the reservation name, state, start date and time,
     * and the details of the offering that make up the rest of the reservation (such as price, duration, and outbound
     * bandwidth).
     *
     * @param describeReservationRequest
     * @return Result of the DescribeReservation operation returned by the service.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.DescribeReservation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/DescribeReservation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeReservationResponse describeReservation(DescribeReservationRequest describeReservationRequest)
            throws NotFoundException, ServiceUnavailableException, TooManyRequestsException, BadRequestException,
            InternalServerErrorException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeReservationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeReservationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeReservationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeReservation");

            return clientHandler.execute(new ClientExecutionParams<DescribeReservationRequest, DescribeReservationResponse>()
                    .withOperationName("DescribeReservation").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeReservationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeReservationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Grants entitlements to an existing flow.
     *
     * @param grantFlowEntitlementsRequest
     *        A request to grant entitlements on a flow.
     * @return Result of the GrantFlowEntitlements operation returned by the service.
     * @throws GrantFlowEntitlements420Exception
     *         AWS Elemental MediaConnect can't complete this request because this flow already has the maximum number
     *         of allowed entitlements (50). For more information, contact AWS Customer Support.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.GrantFlowEntitlements
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/GrantFlowEntitlements"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GrantFlowEntitlementsResponse grantFlowEntitlements(GrantFlowEntitlementsRequest grantFlowEntitlementsRequest)
            throws GrantFlowEntitlements420Exception, BadRequestException, InternalServerErrorException, ForbiddenException,
            NotFoundException, ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException,
            MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GrantFlowEntitlementsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GrantFlowEntitlementsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, grantFlowEntitlementsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GrantFlowEntitlements");

            return clientHandler.execute(new ClientExecutionParams<GrantFlowEntitlementsRequest, GrantFlowEntitlementsResponse>()
                    .withOperationName("GrantFlowEntitlements").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(grantFlowEntitlementsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GrantFlowEntitlementsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Displays a list of all entitlements that have been granted to this account. This request returns 20 results per
     * page.
     *
     * @param listEntitlementsRequest
     * @return Result of the ListEntitlements operation returned by the service.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.ListEntitlements
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListEntitlements" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListEntitlementsResponse listEntitlements(ListEntitlementsRequest listEntitlementsRequest)
            throws ServiceUnavailableException, TooManyRequestsException, BadRequestException, InternalServerErrorException,
            AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEntitlementsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEntitlementsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEntitlementsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEntitlements");

            return clientHandler.execute(new ClientExecutionParams<ListEntitlementsRequest, ListEntitlementsResponse>()
                    .withOperationName("ListEntitlements").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listEntitlementsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEntitlementsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Displays a list of all entitlements that have been granted to this account. This request returns 20 results per
     * page.<br/>
     * <p>
     * This is a variant of
     * {@link #listEntitlements(software.amazon.awssdk.services.mediaconnect.model.ListEntitlementsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListEntitlementsIterable responses = client.listEntitlementsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediaconnect.paginators.ListEntitlementsIterable responses = client
     *             .listEntitlementsPaginator(request);
     *     for (software.amazon.awssdk.services.mediaconnect.model.ListEntitlementsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListEntitlementsIterable responses = client.listEntitlementsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEntitlements(software.amazon.awssdk.services.mediaconnect.model.ListEntitlementsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEntitlementsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.ListEntitlements
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListEntitlements" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListEntitlementsIterable listEntitlementsPaginator(ListEntitlementsRequest listEntitlementsRequest)
            throws ServiceUnavailableException, TooManyRequestsException, BadRequestException, InternalServerErrorException,
            AwsServiceException, SdkClientException, MediaConnectException {
        return new ListEntitlementsIterable(this, applyPaginatorUserAgent(listEntitlementsRequest));
    }

    /**
     * Displays a list of flows that are associated with this account. This request returns a paginated result.
     *
     * @param listFlowsRequest
     * @return Result of the ListFlows operation returned by the service.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.ListFlows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListFlows" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListFlowsResponse listFlows(ListFlowsRequest listFlowsRequest) throws ServiceUnavailableException,
            TooManyRequestsException, BadRequestException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFlowsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListFlowsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFlowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFlows");

            return clientHandler.execute(new ClientExecutionParams<ListFlowsRequest, ListFlowsResponse>()
                    .withOperationName("ListFlows").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listFlowsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListFlowsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Displays a list of flows that are associated with this account. This request returns a paginated result.<br/>
     * <p>
     * This is a variant of {@link #listFlows(software.amazon.awssdk.services.mediaconnect.model.ListFlowsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListFlowsIterable responses = client.listFlowsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediaconnect.paginators.ListFlowsIterable responses = client.listFlowsPaginator(request);
     *     for (software.amazon.awssdk.services.mediaconnect.model.ListFlowsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListFlowsIterable responses = client.listFlowsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFlows(software.amazon.awssdk.services.mediaconnect.model.ListFlowsRequest)} operation.</b>
     * </p>
     *
     * @param listFlowsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.ListFlows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListFlows" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListFlowsIterable listFlowsPaginator(ListFlowsRequest listFlowsRequest) throws ServiceUnavailableException,
            TooManyRequestsException, BadRequestException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaConnectException {
        return new ListFlowsIterable(this, applyPaginatorUserAgent(listFlowsRequest));
    }

    /**
     * Displays a list of all offerings that are available to this account in the current AWS Region. If you have an
     * active reservation (which means you've purchased an offering that has already started and hasn't expired yet),
     * your account isn't eligible for other offerings.
     *
     * @param listOfferingsRequest
     * @return Result of the ListOfferings operation returned by the service.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.ListOfferings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListOfferings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListOfferingsResponse listOfferings(ListOfferingsRequest listOfferingsRequest) throws ServiceUnavailableException,
            TooManyRequestsException, BadRequestException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListOfferingsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListOfferingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOfferingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOfferings");

            return clientHandler.execute(new ClientExecutionParams<ListOfferingsRequest, ListOfferingsResponse>()
                    .withOperationName("ListOfferings").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listOfferingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListOfferingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Displays a list of all offerings that are available to this account in the current AWS Region. If you have an
     * active reservation (which means you've purchased an offering that has already started and hasn't expired yet),
     * your account isn't eligible for other offerings.<br/>
     * <p>
     * This is a variant of
     * {@link #listOfferings(software.amazon.awssdk.services.mediaconnect.model.ListOfferingsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListOfferingsIterable responses = client.listOfferingsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediaconnect.paginators.ListOfferingsIterable responses = client
     *             .listOfferingsPaginator(request);
     *     for (software.amazon.awssdk.services.mediaconnect.model.ListOfferingsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListOfferingsIterable responses = client.listOfferingsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOfferings(software.amazon.awssdk.services.mediaconnect.model.ListOfferingsRequest)} operation.</b>
     * </p>
     *
     * @param listOfferingsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.ListOfferings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListOfferings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListOfferingsIterable listOfferingsPaginator(ListOfferingsRequest listOfferingsRequest)
            throws ServiceUnavailableException, TooManyRequestsException, BadRequestException, InternalServerErrorException,
            AwsServiceException, SdkClientException, MediaConnectException {
        return new ListOfferingsIterable(this, applyPaginatorUserAgent(listOfferingsRequest));
    }

    /**
     * Displays a list of all reservations that have been purchased by this account in the current AWS Region. This list
     * includes all reservations in all states (such as active and expired).
     *
     * @param listReservationsRequest
     * @return Result of the ListReservations operation returned by the service.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.ListReservations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListReservations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListReservationsResponse listReservations(ListReservationsRequest listReservationsRequest)
            throws ServiceUnavailableException, TooManyRequestsException, BadRequestException, InternalServerErrorException,
            AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListReservationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListReservationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReservationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReservations");

            return clientHandler.execute(new ClientExecutionParams<ListReservationsRequest, ListReservationsResponse>()
                    .withOperationName("ListReservations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listReservationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListReservationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Displays a list of all reservations that have been purchased by this account in the current AWS Region. This list
     * includes all reservations in all states (such as active and expired).<br/>
     * <p>
     * This is a variant of
     * {@link #listReservations(software.amazon.awssdk.services.mediaconnect.model.ListReservationsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListReservationsIterable responses = client.listReservationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.mediaconnect.paginators.ListReservationsIterable responses = client
     *             .listReservationsPaginator(request);
     *     for (software.amazon.awssdk.services.mediaconnect.model.ListReservationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.mediaconnect.paginators.ListReservationsIterable responses = client.listReservationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listReservations(software.amazon.awssdk.services.mediaconnect.model.ListReservationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listReservationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.ListReservations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListReservations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListReservationsIterable listReservationsPaginator(ListReservationsRequest listReservationsRequest)
            throws ServiceUnavailableException, TooManyRequestsException, BadRequestException, InternalServerErrorException,
            AwsServiceException, SdkClientException, MediaConnectException {
        return new ListReservationsIterable(this, applyPaginatorUserAgent(listReservationsRequest));
    }

    /**
     * List all tags on an AWS Elemental MediaConnect resource
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws NotFoundException
     *         The requested resource was not found
     * @throws BadRequestException
     *         The client performed an invalid request
     * @throws InternalServerErrorException
     *         Internal service error
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws NotFoundException, BadRequestException, InternalServerErrorException, AwsServiceException, SdkClientException,
            MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Submits a request to purchase an offering. If you already have an active reservation, you can't purchase another
     * offering.
     *
     * @param purchaseOfferingRequest
     *        A request to purchase a offering.
     * @return Result of the PurchaseOffering operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.PurchaseOffering
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/PurchaseOffering" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PurchaseOfferingResponse purchaseOffering(PurchaseOfferingRequest purchaseOfferingRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PurchaseOfferingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PurchaseOfferingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, purchaseOfferingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PurchaseOffering");

            return clientHandler.execute(new ClientExecutionParams<PurchaseOfferingRequest, PurchaseOfferingResponse>()
                    .withOperationName("PurchaseOffering").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(purchaseOfferingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PurchaseOfferingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Removes an output from an existing flow. This request can be made only on an output that does not have an
     * entitlement associated with it. If the output has an entitlement, you must revoke the entitlement instead. When
     * an entitlement is revoked from a flow, the service automatically removes the associated output.
     *
     * @param removeFlowOutputRequest
     * @return Result of the RemoveFlowOutput operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.RemoveFlowOutput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/RemoveFlowOutput" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RemoveFlowOutputResponse removeFlowOutput(RemoveFlowOutputRequest removeFlowOutputRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveFlowOutputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RemoveFlowOutputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeFlowOutputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveFlowOutput");

            return clientHandler.execute(new ClientExecutionParams<RemoveFlowOutputRequest, RemoveFlowOutputResponse>()
                    .withOperationName("RemoveFlowOutput").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(removeFlowOutputRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RemoveFlowOutputRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Removes a source from an existing flow. This request can be made only if there is more than one source on the
     * flow.
     *
     * @param removeFlowSourceRequest
     * @return Result of the RemoveFlowSource operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.RemoveFlowSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/RemoveFlowSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RemoveFlowSourceResponse removeFlowSource(RemoveFlowSourceRequest removeFlowSourceRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveFlowSourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RemoveFlowSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeFlowSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveFlowSource");

            return clientHandler.execute(new ClientExecutionParams<RemoveFlowSourceRequest, RemoveFlowSourceResponse>()
                    .withOperationName("RemoveFlowSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(removeFlowSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RemoveFlowSourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Removes a VPC Interface from an existing flow. This request can be made only on a VPC interface that does not
     * have a Source or Output associated with it. If the VPC interface is referenced by a Source or Output, you must
     * first delete or update the Source or Output to no longer reference the VPC interface.
     *
     * @param removeFlowVpcInterfaceRequest
     * @return Result of the RemoveFlowVpcInterface operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.RemoveFlowVpcInterface
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/RemoveFlowVpcInterface"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemoveFlowVpcInterfaceResponse removeFlowVpcInterface(RemoveFlowVpcInterfaceRequest removeFlowVpcInterfaceRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveFlowVpcInterfaceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveFlowVpcInterfaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeFlowVpcInterfaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveFlowVpcInterface");

            return clientHandler
                    .execute(new ClientExecutionParams<RemoveFlowVpcInterfaceRequest, RemoveFlowVpcInterfaceResponse>()
                            .withOperationName("RemoveFlowVpcInterface").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(removeFlowVpcInterfaceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RemoveFlowVpcInterfaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Revokes an entitlement from a flow. Once an entitlement is revoked, the content becomes unavailable to the
     * subscriber and the associated output is removed.
     *
     * @param revokeFlowEntitlementRequest
     * @return Result of the RevokeFlowEntitlement operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.RevokeFlowEntitlement
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/RevokeFlowEntitlement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RevokeFlowEntitlementResponse revokeFlowEntitlement(RevokeFlowEntitlementRequest revokeFlowEntitlementRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RevokeFlowEntitlementResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RevokeFlowEntitlementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, revokeFlowEntitlementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RevokeFlowEntitlement");

            return clientHandler.execute(new ClientExecutionParams<RevokeFlowEntitlementRequest, RevokeFlowEntitlementResponse>()
                    .withOperationName("RevokeFlowEntitlement").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(revokeFlowEntitlementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RevokeFlowEntitlementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Starts a flow.
     *
     * @param startFlowRequest
     * @return Result of the StartFlow operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.StartFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/StartFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartFlowResponse startFlow(StartFlowRequest startFlowRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartFlow");

            return clientHandler.execute(new ClientExecutionParams<StartFlowRequest, StartFlowResponse>()
                    .withOperationName("StartFlow").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startFlowRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new StartFlowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Stops a flow.
     *
     * @param stopFlowRequest
     * @return Result of the StopFlow operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.StopFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/StopFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopFlowResponse stopFlow(StopFlowRequest stopFlowRequest) throws BadRequestException, InternalServerErrorException,
            ForbiddenException, NotFoundException, ServiceUnavailableException, TooManyRequestsException, AwsServiceException,
            SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopFlow");

            return clientHandler.execute(new ClientExecutionParams<StopFlowRequest, StopFlowResponse>()
                    .withOperationName("StopFlow").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopFlowRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new StopFlowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Associates the specified tags to a resource with the specified resourceArn. If existing tags on a resource are
     * not specified in the request parameters, they are not changed. When a resource is deleted, the tags associated
     * with that resource are deleted as well.
     *
     * @param tagResourceRequest
     *        The tags to add to the resource. A tag is an array of key-value pairs. Tag keys can have a maximum
     *        character length of 128 characters, and tag values can have a maximum length of 256 characters.
     * @return Result of the TagResource operation returned by the service.
     * @throws NotFoundException
     *         The requested resource was not found
     * @throws BadRequestException
     *         The client performed an invalid request
     * @throws InternalServerErrorException
     *         Internal service error
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws NotFoundException, BadRequestException,
            InternalServerErrorException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Deletes specified tags from a resource.
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws NotFoundException
     *         The requested resource was not found
     * @throws BadRequestException
     *         The client performed an invalid request
     * @throws InternalServerErrorException
     *         Internal service error
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws NotFoundException,
            BadRequestException, InternalServerErrorException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Updates flow
     *
     * @param updateFlowRequest
     *        A request to update flow.
     * @return Result of the UpdateFlow operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.UpdateFlow
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UpdateFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateFlowResponse updateFlow(UpdateFlowRequest updateFlowRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFlow");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateFlowRequest, UpdateFlowResponse>().withOperationName("UpdateFlow")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateFlowRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateFlowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * You can change an entitlement's description, subscribers, and encryption. If you change the subscribers, the
     * service will remove the outputs that are are used by the subscribers that are removed.
     *
     * @param updateFlowEntitlementRequest
     *        The entitlement fields that you want to update.
     * @return Result of the UpdateFlowEntitlement operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.UpdateFlowEntitlement
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UpdateFlowEntitlement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateFlowEntitlementResponse updateFlowEntitlement(UpdateFlowEntitlementRequest updateFlowEntitlementRequest)
            throws BadRequestException, InternalServerErrorException, ForbiddenException, NotFoundException,
            ServiceUnavailableException, TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFlowEntitlementResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateFlowEntitlementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFlowEntitlementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFlowEntitlement");

            return clientHandler.execute(new ClientExecutionParams<UpdateFlowEntitlementRequest, UpdateFlowEntitlementResponse>()
                    .withOperationName("UpdateFlowEntitlement").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateFlowEntitlementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateFlowEntitlementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Updates an existing flow output.
     *
     * @param updateFlowOutputRequest
     *        The fields that you want to update in the output.
     * @return Result of the UpdateFlowOutput operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.UpdateFlowOutput
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UpdateFlowOutput" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateFlowOutputResponse updateFlowOutput(UpdateFlowOutputRequest updateFlowOutputRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFlowOutputResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateFlowOutputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFlowOutputRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFlowOutput");

            return clientHandler.execute(new ClientExecutionParams<UpdateFlowOutputRequest, UpdateFlowOutputResponse>()
                    .withOperationName("UpdateFlowOutput").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateFlowOutputRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateFlowOutputRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Updates the source of a flow.
     *
     * @param updateFlowSourceRequest
     *        A request to update the source of a flow.
     * @return Result of the UpdateFlowSource operation returned by the service.
     * @throws BadRequestException
     *         The request that you submitted is not valid.
     * @throws InternalServerErrorException
     *         AWS Elemental MediaConnect can't fulfill your request because it encountered an unexpected condition.
     * @throws ForbiddenException
     *         You don't have the required permissions to perform this operation.
     * @throws NotFoundException
     *         AWS Elemental MediaConnect did not find the resource that you specified in the request.
     * @throws ServiceUnavailableException
     *         AWS Elemental MediaConnect is currently unavailable. Try again later.
     * @throws TooManyRequestsException
     *         You have exceeded the service request rate limit for your AWS Elemental MediaConnect account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MediaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MediaConnectClient.UpdateFlowSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/mediaconnect-2018-11-14/UpdateFlowSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateFlowSourceResponse updateFlowSource(UpdateFlowSourceRequest updateFlowSourceRequest) throws BadRequestException,
            InternalServerErrorException, ForbiddenException, NotFoundException, ServiceUnavailableException,
            TooManyRequestsException, AwsServiceException, SdkClientException, MediaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFlowSourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateFlowSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFlowSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFlowSource");

            return clientHandler.execute(new ClientExecutionParams<UpdateFlowSourceRequest, UpdateFlowSourceResponse>()
                    .withOperationName("UpdateFlowSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateFlowSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateFlowSourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaConnectException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CreateFlow420Exception")
                                .exceptionBuilderSupplier(CreateFlow420Exception::builder).httpStatusCode(420).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GrantFlowEntitlements420Exception")
                                .exceptionBuilderSupplier(GrantFlowEntitlements420Exception::builder).httpStatusCode(420).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AddFlowOutputs420Exception")
                                .exceptionBuilderSupplier(AddFlowOutputs420Exception::builder).httpStatusCode(420).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends MediaConnectRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
