/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.managedblockchain.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A policy type that defines the voting rules for the network. The rules decide if a proposal is approved. Approval may
 * be based on criteria such as the percentage of <code>YES</code> votes and the duration of the proposal. The policy
 * applies to all proposals and is specified when the network is created.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ApprovalThresholdPolicy implements SdkPojo, Serializable,
        ToCopyableBuilder<ApprovalThresholdPolicy.Builder, ApprovalThresholdPolicy> {
    private static final SdkField<Integer> THRESHOLD_PERCENTAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ApprovalThresholdPolicy::thresholdPercentage)).setter(setter(Builder::thresholdPercentage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ThresholdPercentage").build())
            .build();

    private static final SdkField<Integer> PROPOSAL_DURATION_IN_HOURS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ApprovalThresholdPolicy::proposalDurationInHours)).setter(setter(Builder::proposalDurationInHours))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProposalDurationInHours").build())
            .build();

    private static final SdkField<String> THRESHOLD_COMPARATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ApprovalThresholdPolicy::thresholdComparatorAsString)).setter(setter(Builder::thresholdComparator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ThresholdComparator").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(THRESHOLD_PERCENTAGE_FIELD,
            PROPOSAL_DURATION_IN_HOURS_FIELD, THRESHOLD_COMPARATOR_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer thresholdPercentage;

    private final Integer proposalDurationInHours;

    private final String thresholdComparator;

    private ApprovalThresholdPolicy(BuilderImpl builder) {
        this.thresholdPercentage = builder.thresholdPercentage;
        this.proposalDurationInHours = builder.proposalDurationInHours;
        this.thresholdComparator = builder.thresholdComparator;
    }

    /**
     * <p>
     * The percentage of votes among all members that must be <code>YES</code> for a proposal to be approved. For
     * example, a <code>ThresholdPercentage</code> value of <code>50</code> indicates 50%. The
     * <code>ThresholdComparator</code> determines the precise comparison. If a <code>ThresholdPercentage</code> value
     * of <code>50</code> is specified on a network with 10 members, along with a <code>ThresholdComparator</code> value
     * of <code>GREATER_THAN</code>, this indicates that 6 <code>YES</code> votes are required for the proposal to be
     * approved.
     * </p>
     * 
     * @return The percentage of votes among all members that must be <code>YES</code> for a proposal to be approved.
     *         For example, a <code>ThresholdPercentage</code> value of <code>50</code> indicates 50%. The
     *         <code>ThresholdComparator</code> determines the precise comparison. If a <code>ThresholdPercentage</code>
     *         value of <code>50</code> is specified on a network with 10 members, along with a
     *         <code>ThresholdComparator</code> value of <code>GREATER_THAN</code>, this indicates that 6
     *         <code>YES</code> votes are required for the proposal to be approved.
     */
    public Integer thresholdPercentage() {
        return thresholdPercentage;
    }

    /**
     * <p>
     * The duration from the time that a proposal is created until it expires. If members cast neither the required
     * number of <code>YES</code> votes to approve the proposal nor the number of <code>NO</code> votes required to
     * reject it before the duration expires, the proposal is <code>EXPIRED</code> and <code>ProposalActions</code> are
     * not carried out.
     * </p>
     * 
     * @return The duration from the time that a proposal is created until it expires. If members cast neither the
     *         required number of <code>YES</code> votes to approve the proposal nor the number of <code>NO</code> votes
     *         required to reject it before the duration expires, the proposal is <code>EXPIRED</code> and
     *         <code>ProposalActions</code> are not carried out.
     */
    public Integer proposalDurationInHours() {
        return proposalDurationInHours;
    }

    /**
     * <p>
     * Determines whether the vote percentage must be greater than the <code>ThresholdPercentage</code> or must be
     * greater than or equal to the <code>ThreholdPercentage</code> to be approved.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #thresholdComparator} will return {@link ThresholdComparator#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #thresholdComparatorAsString}.
     * </p>
     * 
     * @return Determines whether the vote percentage must be greater than the <code>ThresholdPercentage</code> or must
     *         be greater than or equal to the <code>ThreholdPercentage</code> to be approved.
     * @see ThresholdComparator
     */
    public ThresholdComparator thresholdComparator() {
        return ThresholdComparator.fromValue(thresholdComparator);
    }

    /**
     * <p>
     * Determines whether the vote percentage must be greater than the <code>ThresholdPercentage</code> or must be
     * greater than or equal to the <code>ThreholdPercentage</code> to be approved.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #thresholdComparator} will return {@link ThresholdComparator#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #thresholdComparatorAsString}.
     * </p>
     * 
     * @return Determines whether the vote percentage must be greater than the <code>ThresholdPercentage</code> or must
     *         be greater than or equal to the <code>ThreholdPercentage</code> to be approved.
     * @see ThresholdComparator
     */
    public String thresholdComparatorAsString() {
        return thresholdComparator;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(thresholdPercentage());
        hashCode = 31 * hashCode + Objects.hashCode(proposalDurationInHours());
        hashCode = 31 * hashCode + Objects.hashCode(thresholdComparatorAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ApprovalThresholdPolicy)) {
            return false;
        }
        ApprovalThresholdPolicy other = (ApprovalThresholdPolicy) obj;
        return Objects.equals(thresholdPercentage(), other.thresholdPercentage())
                && Objects.equals(proposalDurationInHours(), other.proposalDurationInHours())
                && Objects.equals(thresholdComparatorAsString(), other.thresholdComparatorAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ApprovalThresholdPolicy").add("ThresholdPercentage", thresholdPercentage())
                .add("ProposalDurationInHours", proposalDurationInHours())
                .add("ThresholdComparator", thresholdComparatorAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ThresholdPercentage":
            return Optional.ofNullable(clazz.cast(thresholdPercentage()));
        case "ProposalDurationInHours":
            return Optional.ofNullable(clazz.cast(proposalDurationInHours()));
        case "ThresholdComparator":
            return Optional.ofNullable(clazz.cast(thresholdComparatorAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ApprovalThresholdPolicy, T> g) {
        return obj -> g.apply((ApprovalThresholdPolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ApprovalThresholdPolicy> {
        /**
         * <p>
         * The percentage of votes among all members that must be <code>YES</code> for a proposal to be approved. For
         * example, a <code>ThresholdPercentage</code> value of <code>50</code> indicates 50%. The
         * <code>ThresholdComparator</code> determines the precise comparison. If a <code>ThresholdPercentage</code>
         * value of <code>50</code> is specified on a network with 10 members, along with a
         * <code>ThresholdComparator</code> value of <code>GREATER_THAN</code>, this indicates that 6 <code>YES</code>
         * votes are required for the proposal to be approved.
         * </p>
         * 
         * @param thresholdPercentage
         *        The percentage of votes among all members that must be <code>YES</code> for a proposal to be approved.
         *        For example, a <code>ThresholdPercentage</code> value of <code>50</code> indicates 50%. The
         *        <code>ThresholdComparator</code> determines the precise comparison. If a
         *        <code>ThresholdPercentage</code> value of <code>50</code> is specified on a network with 10 members,
         *        along with a <code>ThresholdComparator</code> value of <code>GREATER_THAN</code>, this indicates that
         *        6 <code>YES</code> votes are required for the proposal to be approved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder thresholdPercentage(Integer thresholdPercentage);

        /**
         * <p>
         * The duration from the time that a proposal is created until it expires. If members cast neither the required
         * number of <code>YES</code> votes to approve the proposal nor the number of <code>NO</code> votes required to
         * reject it before the duration expires, the proposal is <code>EXPIRED</code> and <code>ProposalActions</code>
         * are not carried out.
         * </p>
         * 
         * @param proposalDurationInHours
         *        The duration from the time that a proposal is created until it expires. If members cast neither the
         *        required number of <code>YES</code> votes to approve the proposal nor the number of <code>NO</code>
         *        votes required to reject it before the duration expires, the proposal is <code>EXPIRED</code> and
         *        <code>ProposalActions</code> are not carried out.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder proposalDurationInHours(Integer proposalDurationInHours);

        /**
         * <p>
         * Determines whether the vote percentage must be greater than the <code>ThresholdPercentage</code> or must be
         * greater than or equal to the <code>ThreholdPercentage</code> to be approved.
         * </p>
         * 
         * @param thresholdComparator
         *        Determines whether the vote percentage must be greater than the <code>ThresholdPercentage</code> or
         *        must be greater than or equal to the <code>ThreholdPercentage</code> to be approved.
         * @see ThresholdComparator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ThresholdComparator
         */
        Builder thresholdComparator(String thresholdComparator);

        /**
         * <p>
         * Determines whether the vote percentage must be greater than the <code>ThresholdPercentage</code> or must be
         * greater than or equal to the <code>ThreholdPercentage</code> to be approved.
         * </p>
         * 
         * @param thresholdComparator
         *        Determines whether the vote percentage must be greater than the <code>ThresholdPercentage</code> or
         *        must be greater than or equal to the <code>ThreholdPercentage</code> to be approved.
         * @see ThresholdComparator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ThresholdComparator
         */
        Builder thresholdComparator(ThresholdComparator thresholdComparator);
    }

    static final class BuilderImpl implements Builder {
        private Integer thresholdPercentage;

        private Integer proposalDurationInHours;

        private String thresholdComparator;

        private BuilderImpl() {
        }

        private BuilderImpl(ApprovalThresholdPolicy model) {
            thresholdPercentage(model.thresholdPercentage);
            proposalDurationInHours(model.proposalDurationInHours);
            thresholdComparator(model.thresholdComparator);
        }

        public final Integer getThresholdPercentage() {
            return thresholdPercentage;
        }

        @Override
        public final Builder thresholdPercentage(Integer thresholdPercentage) {
            this.thresholdPercentage = thresholdPercentage;
            return this;
        }

        public final void setThresholdPercentage(Integer thresholdPercentage) {
            this.thresholdPercentage = thresholdPercentage;
        }

        public final Integer getProposalDurationInHours() {
            return proposalDurationInHours;
        }

        @Override
        public final Builder proposalDurationInHours(Integer proposalDurationInHours) {
            this.proposalDurationInHours = proposalDurationInHours;
            return this;
        }

        public final void setProposalDurationInHours(Integer proposalDurationInHours) {
            this.proposalDurationInHours = proposalDurationInHours;
        }

        public final String getThresholdComparator() {
            return thresholdComparator;
        }

        @Override
        public final Builder thresholdComparator(String thresholdComparator) {
            this.thresholdComparator = thresholdComparator;
            return this;
        }

        @Override
        public final Builder thresholdComparator(ThresholdComparator thresholdComparator) {
            this.thresholdComparator(thresholdComparator == null ? null : thresholdComparator.toString());
            return this;
        }

        public final void setThresholdComparator(String thresholdComparator) {
            this.thresholdComparator = thresholdComparator;
        }

        @Override
        public ApprovalThresholdPolicy build() {
            return new ApprovalThresholdPolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
