/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the full details of an Amazon Lightsail SSL/TLS certificate.
 * </p>
 * <note>
 * <p>
 * To get a summary of a certificate, use the <code>GetCertificates</code> action and ommit
 * <code>includeCertificateDetails</code> from your request. The response will include only the certificate Amazon
 * Resource Name (ARN), certificate name, domain name, and tags.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Certificate implements SdkPojo, Serializable, ToCopyableBuilder<Certificate.Builder, Certificate> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Certificate::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Certificate::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("domainName").getter(getter(Certificate::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("domainName").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(Certificate::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> SERIAL_NUMBER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serialNumber").getter(getter(Certificate::serialNumber)).setter(setter(Builder::serialNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serialNumber").build()).build();

    private static final SdkField<List<String>> SUBJECT_ALTERNATIVE_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("subjectAlternativeNames")
            .getter(getter(Certificate::subjectAlternativeNames))
            .setter(setter(Builder::subjectAlternativeNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subjectAlternativeNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DomainValidationRecord>> DOMAIN_VALIDATION_RECORDS_FIELD = SdkField
            .<List<DomainValidationRecord>> builder(MarshallingType.LIST)
            .memberName("domainValidationRecords")
            .getter(getter(Certificate::domainValidationRecords))
            .setter(setter(Builder::domainValidationRecords))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("domainValidationRecords").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DomainValidationRecord> builder(MarshallingType.SDK_POJO)
                                            .constructor(DomainValidationRecord::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> REQUEST_FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("requestFailureReason").getter(getter(Certificate::requestFailureReason))
            .setter(setter(Builder::requestFailureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("requestFailureReason").build())
            .build();

    private static final SdkField<Integer> IN_USE_RESOURCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("inUseResourceCount").getter(getter(Certificate::inUseResourceCount))
            .setter(setter(Builder::inUseResourceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inUseResourceCount").build())
            .build();

    private static final SdkField<String> KEY_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("keyAlgorithm").getter(getter(Certificate::keyAlgorithm)).setter(setter(Builder::keyAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("keyAlgorithm").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(Certificate::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<Instant> ISSUED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("issuedAt").getter(getter(Certificate::issuedAt)).setter(setter(Builder::issuedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("issuedAt").build()).build();

    private static final SdkField<String> ISSUER_CA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("issuerCA").getter(getter(Certificate::issuerCA)).setter(setter(Builder::issuerCA))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("issuerCA").build()).build();

    private static final SdkField<Instant> NOT_BEFORE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("notBefore").getter(getter(Certificate::notBefore)).setter(setter(Builder::notBefore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("notBefore").build()).build();

    private static final SdkField<Instant> NOT_AFTER_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("notAfter").getter(getter(Certificate::notAfter)).setter(setter(Builder::notAfter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("notAfter").build()).build();

    private static final SdkField<String> ELIGIBLE_TO_RENEW_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eligibleToRenew").getter(getter(Certificate::eligibleToRenew)).setter(setter(Builder::eligibleToRenew))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eligibleToRenew").build()).build();

    private static final SdkField<RenewalSummary> RENEWAL_SUMMARY_FIELD = SdkField
            .<RenewalSummary> builder(MarshallingType.SDK_POJO).memberName("renewalSummary")
            .getter(getter(Certificate::renewalSummary)).setter(setter(Builder::renewalSummary))
            .constructor(RenewalSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("renewalSummary").build()).build();

    private static final SdkField<Instant> REVOKED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("revokedAt").getter(getter(Certificate::revokedAt)).setter(setter(Builder::revokedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("revokedAt").build()).build();

    private static final SdkField<String> REVOCATION_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("revocationReason").getter(getter(Certificate::revocationReason))
            .setter(setter(Builder::revocationReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("revocationReason").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(Certificate::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SUPPORT_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("supportCode").getter(getter(Certificate::supportCode)).setter(setter(Builder::supportCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportCode").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, NAME_FIELD,
            DOMAIN_NAME_FIELD, STATUS_FIELD, SERIAL_NUMBER_FIELD, SUBJECT_ALTERNATIVE_NAMES_FIELD,
            DOMAIN_VALIDATION_RECORDS_FIELD, REQUEST_FAILURE_REASON_FIELD, IN_USE_RESOURCE_COUNT_FIELD, KEY_ALGORITHM_FIELD,
            CREATED_AT_FIELD, ISSUED_AT_FIELD, ISSUER_CA_FIELD, NOT_BEFORE_FIELD, NOT_AFTER_FIELD, ELIGIBLE_TO_RENEW_FIELD,
            RENEWAL_SUMMARY_FIELD, REVOKED_AT_FIELD, REVOCATION_REASON_FIELD, TAGS_FIELD, SUPPORT_CODE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String name;

    private final String domainName;

    private final String status;

    private final String serialNumber;

    private final List<String> subjectAlternativeNames;

    private final List<DomainValidationRecord> domainValidationRecords;

    private final String requestFailureReason;

    private final Integer inUseResourceCount;

    private final String keyAlgorithm;

    private final Instant createdAt;

    private final Instant issuedAt;

    private final String issuerCA;

    private final Instant notBefore;

    private final Instant notAfter;

    private final String eligibleToRenew;

    private final RenewalSummary renewalSummary;

    private final Instant revokedAt;

    private final String revocationReason;

    private final List<Tag> tags;

    private final String supportCode;

    private Certificate(BuilderImpl builder) {
        this.arn = builder.arn;
        this.name = builder.name;
        this.domainName = builder.domainName;
        this.status = builder.status;
        this.serialNumber = builder.serialNumber;
        this.subjectAlternativeNames = builder.subjectAlternativeNames;
        this.domainValidationRecords = builder.domainValidationRecords;
        this.requestFailureReason = builder.requestFailureReason;
        this.inUseResourceCount = builder.inUseResourceCount;
        this.keyAlgorithm = builder.keyAlgorithm;
        this.createdAt = builder.createdAt;
        this.issuedAt = builder.issuedAt;
        this.issuerCA = builder.issuerCA;
        this.notBefore = builder.notBefore;
        this.notAfter = builder.notAfter;
        this.eligibleToRenew = builder.eligibleToRenew;
        this.renewalSummary = builder.renewalSummary;
        this.revokedAt = builder.revokedAt;
        this.revocationReason = builder.revocationReason;
        this.tags = builder.tags;
        this.supportCode = builder.supportCode;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the certificate.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the certificate.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The name of the certificate (e.g., <code>my-certificate</code>).
     * </p>
     * 
     * @return The name of the certificate (e.g., <code>my-certificate</code>).
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The domain name of the certificate.
     * </p>
     * 
     * @return The domain name of the certificate.
     */
    public final String domainName() {
        return domainName;
    }

    /**
     * <p>
     * The validation status of the certificate.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CertificateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The validation status of the certificate.
     * @see CertificateStatus
     */
    public final CertificateStatus status() {
        return CertificateStatus.fromValue(status);
    }

    /**
     * <p>
     * The validation status of the certificate.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CertificateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The validation status of the certificate.
     * @see CertificateStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The serial number of the certificate.
     * </p>
     * 
     * @return The serial number of the certificate.
     */
    public final String serialNumber() {
        return serialNumber;
    }

    /**
     * Returns true if the SubjectAlternativeNames property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasSubjectAlternativeNames() {
        return subjectAlternativeNames != null && !(subjectAlternativeNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of strings that specify the alternate domains (e.g., <code>example2.com</code>) and subdomains (e.g.,
     * <code>blog.example.com</code>) of the certificate.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSubjectAlternativeNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of strings that specify the alternate domains (e.g., <code>example2.com</code>) and subdomains
     *         (e.g., <code>blog.example.com</code>) of the certificate.
     */
    public final List<String> subjectAlternativeNames() {
        return subjectAlternativeNames;
    }

    /**
     * Returns true if the DomainValidationRecords property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasDomainValidationRecords() {
        return domainValidationRecords != null && !(domainValidationRecords instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the domain validation records of the certificate.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDomainValidationRecords()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of objects that describe the domain validation records of the certificate.
     */
    public final List<DomainValidationRecord> domainValidationRecords() {
        return domainValidationRecords;
    }

    /**
     * <p>
     * The validation failure reason, if any, of the certificate.
     * </p>
     * <p>
     * The following failure reasons are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>NO_AVAILABLE_CONTACTS</code> </b> - This failure applies to email validation, which is not available
     * for Lightsail certificates.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>ADDITIONAL_VERIFICATION_REQUIRED</code> </b> - Lightsail requires additional information to process
     * this certificate request. This can happen as a fraud-protection measure, such as when the domain ranks within the
     * Alexa top 1000 websites. To provide the required information, use the <a
     * href="https://console.aws.amazon.com/support/home">AWS Support Center</a> to contact AWS Support.
     * </p>
     * <note>
     * <p>
     * You cannot request a certificate for Amazon-owned domain names such as those ending in amazonaws.com,
     * cloudfront.net, or elasticbeanstalk.com.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <b> <code>DOMAIN_NOT_ALLOWED</code> </b> - One or more of the domain names in the certificate request was
     * reported as an unsafe domain by <a href="https://www.virustotal.com/gui/home/url">VirusTotal</a>. To correct the
     * problem, search for your domain name on the <a href="https://www.virustotal.com/gui/home/url">VirusTotal</a>
     * website. If your domain is reported as suspicious, see <a
     * href="https://developers.google.com/web/fundamentals/security/hacked">Google Help for Hacked Websites</a> to
     * learn what you can do.
     * </p>
     * <p>
     * If you believe that the result is a false positive, notify the organization that is reporting the domain.
     * VirusTotal is an aggregate of several antivirus and URL scanners and cannot remove your domain from a block list
     * itself. After you correct the problem and the VirusTotal registry has been updated, request a new certificate.
     * </p>
     * <p>
     * If you see this error and your domain is not included in the VirusTotal list, visit the <a
     * href="https://console.aws.amazon.com/support/home">AWS Support Center</a> and create a case.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>INVALID_PUBLIC_DOMAIN</code> </b> - One or more of the domain names in the certificate request is not
     * valid. Typically, this is because a domain name in the request is not a valid top-level domain. Try to request a
     * certificate again, correcting any spelling errors or typos that were in the failed request, and ensure that all
     * domain names in the request are for valid top-level domains. For example, you cannot request a certificate for
     * <code>example.invalidpublicdomain</code> because <code>invalidpublicdomain</code> is not a valid top-level
     * domain.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>OTHER</code> </b> - Typically, this failure occurs when there is a typographical error in one or more
     * of the domain names in the certificate request. Try to request a certificate again, correcting any spelling
     * errors or typos that were in the failed request.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The validation failure reason, if any, of the certificate.</p>
     *         <p>
     *         The following failure reasons are possible:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>NO_AVAILABLE_CONTACTS</code> </b> - This failure applies to email validation, which is not
     *         available for Lightsail certificates.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>ADDITIONAL_VERIFICATION_REQUIRED</code> </b> - Lightsail requires additional information to
     *         process this certificate request. This can happen as a fraud-protection measure, such as when the domain
     *         ranks within the Alexa top 1000 websites. To provide the required information, use the <a
     *         href="https://console.aws.amazon.com/support/home">AWS Support Center</a> to contact AWS Support.
     *         </p>
     *         <note>
     *         <p>
     *         You cannot request a certificate for Amazon-owned domain names such as those ending in amazonaws.com,
     *         cloudfront.net, or elasticbeanstalk.com.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <b> <code>DOMAIN_NOT_ALLOWED</code> </b> - One or more of the domain names in the certificate request was
     *         reported as an unsafe domain by <a href="https://www.virustotal.com/gui/home/url">VirusTotal</a>. To
     *         correct the problem, search for your domain name on the <a
     *         href="https://www.virustotal.com/gui/home/url">VirusTotal</a> website. If your domain is reported as
     *         suspicious, see <a href="https://developers.google.com/web/fundamentals/security/hacked">Google Help for
     *         Hacked Websites</a> to learn what you can do.
     *         </p>
     *         <p>
     *         If you believe that the result is a false positive, notify the organization that is reporting the domain.
     *         VirusTotal is an aggregate of several antivirus and URL scanners and cannot remove your domain from a
     *         block list itself. After you correct the problem and the VirusTotal registry has been updated, request a
     *         new certificate.
     *         </p>
     *         <p>
     *         If you see this error and your domain is not included in the VirusTotal list, visit the <a
     *         href="https://console.aws.amazon.com/support/home">AWS Support Center</a> and create a case.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>INVALID_PUBLIC_DOMAIN</code> </b> - One or more of the domain names in the certificate request
     *         is not valid. Typically, this is because a domain name in the request is not a valid top-level domain.
     *         Try to request a certificate again, correcting any spelling errors or typos that were in the failed
     *         request, and ensure that all domain names in the request are for valid top-level domains. For example,
     *         you cannot request a certificate for <code>example.invalidpublicdomain</code> because
     *         <code>invalidpublicdomain</code> is not a valid top-level domain.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>OTHER</code> </b> - Typically, this failure occurs when there is a typographical error in one
     *         or more of the domain names in the certificate request. Try to request a certificate again, correcting
     *         any spelling errors or typos that were in the failed request.
     *         </p>
     *         </li>
     */
    public final String requestFailureReason() {
        return requestFailureReason;
    }

    /**
     * <p>
     * The number of Lightsail resources that the certificate is attached to.
     * </p>
     * 
     * @return The number of Lightsail resources that the certificate is attached to.
     */
    public final Integer inUseResourceCount() {
        return inUseResourceCount;
    }

    /**
     * <p>
     * The algorithm used to generate the key pair (the public and private key) of the certificate.
     * </p>
     * 
     * @return The algorithm used to generate the key pair (the public and private key) of the certificate.
     */
    public final String keyAlgorithm() {
        return keyAlgorithm;
    }

    /**
     * <p>
     * The timestamp when the certificate was created.
     * </p>
     * 
     * @return The timestamp when the certificate was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The timestamp when the certificate was issued.
     * </p>
     * 
     * @return The timestamp when the certificate was issued.
     */
    public final Instant issuedAt() {
        return issuedAt;
    }

    /**
     * <p>
     * The certificate authority that issued the certificate.
     * </p>
     * 
     * @return The certificate authority that issued the certificate.
     */
    public final String issuerCA() {
        return issuerCA;
    }

    /**
     * <p>
     * The timestamp when the certificate is first valid.
     * </p>
     * 
     * @return The timestamp when the certificate is first valid.
     */
    public final Instant notBefore() {
        return notBefore;
    }

    /**
     * <p>
     * The timestamp when the certificate expires.
     * </p>
     * 
     * @return The timestamp when the certificate expires.
     */
    public final Instant notAfter() {
        return notAfter;
    }

    /**
     * <p>
     * The renewal eligibility of the certificate.
     * </p>
     * 
     * @return The renewal eligibility of the certificate.
     */
    public final String eligibleToRenew() {
        return eligibleToRenew;
    }

    /**
     * <p>
     * An object that describes the status of the certificate renewal managed by Lightsail.
     * </p>
     * 
     * @return An object that describes the status of the certificate renewal managed by Lightsail.
     */
    public final RenewalSummary renewalSummary() {
        return renewalSummary;
    }

    /**
     * <p>
     * The timestamp when the certificate was revoked. This value is present only when the certificate status is
     * <code>REVOKED</code>.
     * </p>
     * 
     * @return The timestamp when the certificate was revoked. This value is present only when the certificate status is
     *         <code>REVOKED</code>.
     */
    public final Instant revokedAt() {
        return revokedAt;
    }

    /**
     * <p>
     * The reason the certificate was revoked. This value is present only when the certificate status is
     * <code>REVOKED</code>.
     * </p>
     * 
     * @return The reason the certificate was revoked. This value is present only when the certificate status is
     *         <code>REVOKED</code>.
     */
    public final String revocationReason() {
        return revocationReason;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The tag keys and optional values for the resource. For more information about tags in Lightsail, see the
     *         <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
     *         Guide</a>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The support code. Include this code in your email to support when you have questions about your Lightsail
     * certificate. This code enables our support team to look up your Lightsail information more easily.
     * </p>
     * 
     * @return The support code. Include this code in your email to support when you have questions about your Lightsail
     *         certificate. This code enables our support team to look up your Lightsail information more easily.
     */
    public final String supportCode() {
        return supportCode;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(serialNumber());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubjectAlternativeNames() ? subjectAlternativeNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDomainValidationRecords() ? domainValidationRecords() : null);
        hashCode = 31 * hashCode + Objects.hashCode(requestFailureReason());
        hashCode = 31 * hashCode + Objects.hashCode(inUseResourceCount());
        hashCode = 31 * hashCode + Objects.hashCode(keyAlgorithm());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(issuedAt());
        hashCode = 31 * hashCode + Objects.hashCode(issuerCA());
        hashCode = 31 * hashCode + Objects.hashCode(notBefore());
        hashCode = 31 * hashCode + Objects.hashCode(notAfter());
        hashCode = 31 * hashCode + Objects.hashCode(eligibleToRenew());
        hashCode = 31 * hashCode + Objects.hashCode(renewalSummary());
        hashCode = 31 * hashCode + Objects.hashCode(revokedAt());
        hashCode = 31 * hashCode + Objects.hashCode(revocationReason());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(supportCode());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Certificate)) {
            return false;
        }
        Certificate other = (Certificate) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(name(), other.name())
                && Objects.equals(domainName(), other.domainName()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(serialNumber(), other.serialNumber())
                && hasSubjectAlternativeNames() == other.hasSubjectAlternativeNames()
                && Objects.equals(subjectAlternativeNames(), other.subjectAlternativeNames())
                && hasDomainValidationRecords() == other.hasDomainValidationRecords()
                && Objects.equals(domainValidationRecords(), other.domainValidationRecords())
                && Objects.equals(requestFailureReason(), other.requestFailureReason())
                && Objects.equals(inUseResourceCount(), other.inUseResourceCount())
                && Objects.equals(keyAlgorithm(), other.keyAlgorithm()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(issuedAt(), other.issuedAt()) && Objects.equals(issuerCA(), other.issuerCA())
                && Objects.equals(notBefore(), other.notBefore()) && Objects.equals(notAfter(), other.notAfter())
                && Objects.equals(eligibleToRenew(), other.eligibleToRenew())
                && Objects.equals(renewalSummary(), other.renewalSummary()) && Objects.equals(revokedAt(), other.revokedAt())
                && Objects.equals(revocationReason(), other.revocationReason()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(supportCode(), other.supportCode());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Certificate").add("Arn", arn()).add("Name", name()).add("DomainName", domainName())
                .add("Status", statusAsString()).add("SerialNumber", serialNumber())
                .add("SubjectAlternativeNames", hasSubjectAlternativeNames() ? subjectAlternativeNames() : null)
                .add("DomainValidationRecords", hasDomainValidationRecords() ? domainValidationRecords() : null)
                .add("RequestFailureReason", requestFailureReason()).add("InUseResourceCount", inUseResourceCount())
                .add("KeyAlgorithm", keyAlgorithm()).add("CreatedAt", createdAt()).add("IssuedAt", issuedAt())
                .add("IssuerCA", issuerCA()).add("NotBefore", notBefore()).add("NotAfter", notAfter())
                .add("EligibleToRenew", eligibleToRenew()).add("RenewalSummary", renewalSummary()).add("RevokedAt", revokedAt())
                .add("RevocationReason", revocationReason()).add("Tags", hasTags() ? tags() : null)
                .add("SupportCode", supportCode()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "domainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "serialNumber":
            return Optional.ofNullable(clazz.cast(serialNumber()));
        case "subjectAlternativeNames":
            return Optional.ofNullable(clazz.cast(subjectAlternativeNames()));
        case "domainValidationRecords":
            return Optional.ofNullable(clazz.cast(domainValidationRecords()));
        case "requestFailureReason":
            return Optional.ofNullable(clazz.cast(requestFailureReason()));
        case "inUseResourceCount":
            return Optional.ofNullable(clazz.cast(inUseResourceCount()));
        case "keyAlgorithm":
            return Optional.ofNullable(clazz.cast(keyAlgorithm()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "issuedAt":
            return Optional.ofNullable(clazz.cast(issuedAt()));
        case "issuerCA":
            return Optional.ofNullable(clazz.cast(issuerCA()));
        case "notBefore":
            return Optional.ofNullable(clazz.cast(notBefore()));
        case "notAfter":
            return Optional.ofNullable(clazz.cast(notAfter()));
        case "eligibleToRenew":
            return Optional.ofNullable(clazz.cast(eligibleToRenew()));
        case "renewalSummary":
            return Optional.ofNullable(clazz.cast(renewalSummary()));
        case "revokedAt":
            return Optional.ofNullable(clazz.cast(revokedAt()));
        case "revocationReason":
            return Optional.ofNullable(clazz.cast(revocationReason()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "supportCode":
            return Optional.ofNullable(clazz.cast(supportCode()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Certificate, T> g) {
        return obj -> g.apply((Certificate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Certificate> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the certificate.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The name of the certificate (e.g., <code>my-certificate</code>).
         * </p>
         * 
         * @param name
         *        The name of the certificate (e.g., <code>my-certificate</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The domain name of the certificate.
         * </p>
         * 
         * @param domainName
         *        The domain name of the certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * The validation status of the certificate.
         * </p>
         * 
         * @param status
         *        The validation status of the certificate.
         * @see CertificateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CertificateStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The validation status of the certificate.
         * </p>
         * 
         * @param status
         *        The validation status of the certificate.
         * @see CertificateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CertificateStatus
         */
        Builder status(CertificateStatus status);

        /**
         * <p>
         * The serial number of the certificate.
         * </p>
         * 
         * @param serialNumber
         *        The serial number of the certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serialNumber(String serialNumber);

        /**
         * <p>
         * An array of strings that specify the alternate domains (e.g., <code>example2.com</code>) and subdomains
         * (e.g., <code>blog.example.com</code>) of the certificate.
         * </p>
         * 
         * @param subjectAlternativeNames
         *        An array of strings that specify the alternate domains (e.g., <code>example2.com</code>) and
         *        subdomains (e.g., <code>blog.example.com</code>) of the certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subjectAlternativeNames(Collection<String> subjectAlternativeNames);

        /**
         * <p>
         * An array of strings that specify the alternate domains (e.g., <code>example2.com</code>) and subdomains
         * (e.g., <code>blog.example.com</code>) of the certificate.
         * </p>
         * 
         * @param subjectAlternativeNames
         *        An array of strings that specify the alternate domains (e.g., <code>example2.com</code>) and
         *        subdomains (e.g., <code>blog.example.com</code>) of the certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subjectAlternativeNames(String... subjectAlternativeNames);

        /**
         * <p>
         * An array of objects that describe the domain validation records of the certificate.
         * </p>
         * 
         * @param domainValidationRecords
         *        An array of objects that describe the domain validation records of the certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainValidationRecords(Collection<DomainValidationRecord> domainValidationRecords);

        /**
         * <p>
         * An array of objects that describe the domain validation records of the certificate.
         * </p>
         * 
         * @param domainValidationRecords
         *        An array of objects that describe the domain validation records of the certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainValidationRecords(DomainValidationRecord... domainValidationRecords);

        /**
         * <p>
         * An array of objects that describe the domain validation records of the certificate.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DomainValidationRecord>.Builder} avoiding
         * the need to create one manually via {@link List<DomainValidationRecord>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DomainValidationRecord>.Builder#build()} is called
         * immediately and its result is passed to {@link #domainValidationRecords(List<DomainValidationRecord>)}.
         * 
         * @param domainValidationRecords
         *        a consumer that will call methods on {@link List<DomainValidationRecord>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #domainValidationRecords(List<DomainValidationRecord>)
         */
        Builder domainValidationRecords(Consumer<DomainValidationRecord.Builder>... domainValidationRecords);

        /**
         * <p>
         * The validation failure reason, if any, of the certificate.
         * </p>
         * <p>
         * The following failure reasons are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>NO_AVAILABLE_CONTACTS</code> </b> - This failure applies to email validation, which is not
         * available for Lightsail certificates.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>ADDITIONAL_VERIFICATION_REQUIRED</code> </b> - Lightsail requires additional information to process
         * this certificate request. This can happen as a fraud-protection measure, such as when the domain ranks within
         * the Alexa top 1000 websites. To provide the required information, use the <a
         * href="https://console.aws.amazon.com/support/home">AWS Support Center</a> to contact AWS Support.
         * </p>
         * <note>
         * <p>
         * You cannot request a certificate for Amazon-owned domain names such as those ending in amazonaws.com,
         * cloudfront.net, or elasticbeanstalk.com.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <b> <code>DOMAIN_NOT_ALLOWED</code> </b> - One or more of the domain names in the certificate request was
         * reported as an unsafe domain by <a href="https://www.virustotal.com/gui/home/url">VirusTotal</a>. To correct
         * the problem, search for your domain name on the <a
         * href="https://www.virustotal.com/gui/home/url">VirusTotal</a> website. If your domain is reported as
         * suspicious, see <a href="https://developers.google.com/web/fundamentals/security/hacked">Google Help for
         * Hacked Websites</a> to learn what you can do.
         * </p>
         * <p>
         * If you believe that the result is a false positive, notify the organization that is reporting the domain.
         * VirusTotal is an aggregate of several antivirus and URL scanners and cannot remove your domain from a block
         * list itself. After you correct the problem and the VirusTotal registry has been updated, request a new
         * certificate.
         * </p>
         * <p>
         * If you see this error and your domain is not included in the VirusTotal list, visit the <a
         * href="https://console.aws.amazon.com/support/home">AWS Support Center</a> and create a case.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>INVALID_PUBLIC_DOMAIN</code> </b> - One or more of the domain names in the certificate request is
         * not valid. Typically, this is because a domain name in the request is not a valid top-level domain. Try to
         * request a certificate again, correcting any spelling errors or typos that were in the failed request, and
         * ensure that all domain names in the request are for valid top-level domains. For example, you cannot request
         * a certificate for <code>example.invalidpublicdomain</code> because <code>invalidpublicdomain</code> is not a
         * valid top-level domain.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>OTHER</code> </b> - Typically, this failure occurs when there is a typographical error in one or
         * more of the domain names in the certificate request. Try to request a certificate again, correcting any
         * spelling errors or typos that were in the failed request.
         * </p>
         * </li>
         * </ul>
         * 
         * @param requestFailureReason
         *        The validation failure reason, if any, of the certificate.</p>
         *        <p>
         *        The following failure reasons are possible:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>NO_AVAILABLE_CONTACTS</code> </b> - This failure applies to email validation, which is not
         *        available for Lightsail certificates.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>ADDITIONAL_VERIFICATION_REQUIRED</code> </b> - Lightsail requires additional information to
         *        process this certificate request. This can happen as a fraud-protection measure, such as when the
         *        domain ranks within the Alexa top 1000 websites. To provide the required information, use the <a
         *        href="https://console.aws.amazon.com/support/home">AWS Support Center</a> to contact AWS Support.
         *        </p>
         *        <note>
         *        <p>
         *        You cannot request a certificate for Amazon-owned domain names such as those ending in amazonaws.com,
         *        cloudfront.net, or elasticbeanstalk.com.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <b> <code>DOMAIN_NOT_ALLOWED</code> </b> - One or more of the domain names in the certificate request
         *        was reported as an unsafe domain by <a href="https://www.virustotal.com/gui/home/url">VirusTotal</a>.
         *        To correct the problem, search for your domain name on the <a
         *        href="https://www.virustotal.com/gui/home/url">VirusTotal</a> website. If your domain is reported as
         *        suspicious, see <a href="https://developers.google.com/web/fundamentals/security/hacked">Google Help
         *        for Hacked Websites</a> to learn what you can do.
         *        </p>
         *        <p>
         *        If you believe that the result is a false positive, notify the organization that is reporting the
         *        domain. VirusTotal is an aggregate of several antivirus and URL scanners and cannot remove your domain
         *        from a block list itself. After you correct the problem and the VirusTotal registry has been updated,
         *        request a new certificate.
         *        </p>
         *        <p>
         *        If you see this error and your domain is not included in the VirusTotal list, visit the <a
         *        href="https://console.aws.amazon.com/support/home">AWS Support Center</a> and create a case.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>INVALID_PUBLIC_DOMAIN</code> </b> - One or more of the domain names in the certificate
         *        request is not valid. Typically, this is because a domain name in the request is not a valid top-level
         *        domain. Try to request a certificate again, correcting any spelling errors or typos that were in the
         *        failed request, and ensure that all domain names in the request are for valid top-level domains. For
         *        example, you cannot request a certificate for <code>example.invalidpublicdomain</code> because
         *        <code>invalidpublicdomain</code> is not a valid top-level domain.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>OTHER</code> </b> - Typically, this failure occurs when there is a typographical error in
         *        one or more of the domain names in the certificate request. Try to request a certificate again,
         *        correcting any spelling errors or typos that were in the failed request.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestFailureReason(String requestFailureReason);

        /**
         * <p>
         * The number of Lightsail resources that the certificate is attached to.
         * </p>
         * 
         * @param inUseResourceCount
         *        The number of Lightsail resources that the certificate is attached to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inUseResourceCount(Integer inUseResourceCount);

        /**
         * <p>
         * The algorithm used to generate the key pair (the public and private key) of the certificate.
         * </p>
         * 
         * @param keyAlgorithm
         *        The algorithm used to generate the key pair (the public and private key) of the certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyAlgorithm(String keyAlgorithm);

        /**
         * <p>
         * The timestamp when the certificate was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the certificate was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The timestamp when the certificate was issued.
         * </p>
         * 
         * @param issuedAt
         *        The timestamp when the certificate was issued.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issuedAt(Instant issuedAt);

        /**
         * <p>
         * The certificate authority that issued the certificate.
         * </p>
         * 
         * @param issuerCA
         *        The certificate authority that issued the certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issuerCA(String issuerCA);

        /**
         * <p>
         * The timestamp when the certificate is first valid.
         * </p>
         * 
         * @param notBefore
         *        The timestamp when the certificate is first valid.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notBefore(Instant notBefore);

        /**
         * <p>
         * The timestamp when the certificate expires.
         * </p>
         * 
         * @param notAfter
         *        The timestamp when the certificate expires.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notAfter(Instant notAfter);

        /**
         * <p>
         * The renewal eligibility of the certificate.
         * </p>
         * 
         * @param eligibleToRenew
         *        The renewal eligibility of the certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eligibleToRenew(String eligibleToRenew);

        /**
         * <p>
         * An object that describes the status of the certificate renewal managed by Lightsail.
         * </p>
         * 
         * @param renewalSummary
         *        An object that describes the status of the certificate renewal managed by Lightsail.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder renewalSummary(RenewalSummary renewalSummary);

        /**
         * <p>
         * An object that describes the status of the certificate renewal managed by Lightsail.
         * </p>
         * This is a convenience that creates an instance of the {@link RenewalSummary.Builder} avoiding the need to
         * create one manually via {@link RenewalSummary#builder()}.
         *
         * When the {@link Consumer} completes, {@link RenewalSummary.Builder#build()} is called immediately and its
         * result is passed to {@link #renewalSummary(RenewalSummary)}.
         * 
         * @param renewalSummary
         *        a consumer that will call methods on {@link RenewalSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #renewalSummary(RenewalSummary)
         */
        default Builder renewalSummary(Consumer<RenewalSummary.Builder> renewalSummary) {
            return renewalSummary(RenewalSummary.builder().applyMutation(renewalSummary).build());
        }

        /**
         * <p>
         * The timestamp when the certificate was revoked. This value is present only when the certificate status is
         * <code>REVOKED</code>.
         * </p>
         * 
         * @param revokedAt
         *        The timestamp when the certificate was revoked. This value is present only when the certificate status
         *        is <code>REVOKED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder revokedAt(Instant revokedAt);

        /**
         * <p>
         * The reason the certificate was revoked. This value is present only when the certificate status is
         * <code>REVOKED</code>.
         * </p>
         * 
         * @param revocationReason
         *        The reason the certificate was revoked. This value is present only when the certificate status is
         *        <code>REVOKED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder revocationReason(String revocationReason);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The support code. Include this code in your email to support when you have questions about your Lightsail
         * certificate. This code enables our support team to look up your Lightsail information more easily.
         * </p>
         * 
         * @param supportCode
         *        The support code. Include this code in your email to support when you have questions about your
         *        Lightsail certificate. This code enables our support team to look up your Lightsail information more
         *        easily.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportCode(String supportCode);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String name;

        private String domainName;

        private String status;

        private String serialNumber;

        private List<String> subjectAlternativeNames = DefaultSdkAutoConstructList.getInstance();

        private List<DomainValidationRecord> domainValidationRecords = DefaultSdkAutoConstructList.getInstance();

        private String requestFailureReason;

        private Integer inUseResourceCount;

        private String keyAlgorithm;

        private Instant createdAt;

        private Instant issuedAt;

        private String issuerCA;

        private Instant notBefore;

        private Instant notAfter;

        private String eligibleToRenew;

        private RenewalSummary renewalSummary;

        private Instant revokedAt;

        private String revocationReason;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String supportCode;

        private BuilderImpl() {
        }

        private BuilderImpl(Certificate model) {
            arn(model.arn);
            name(model.name);
            domainName(model.domainName);
            status(model.status);
            serialNumber(model.serialNumber);
            subjectAlternativeNames(model.subjectAlternativeNames);
            domainValidationRecords(model.domainValidationRecords);
            requestFailureReason(model.requestFailureReason);
            inUseResourceCount(model.inUseResourceCount);
            keyAlgorithm(model.keyAlgorithm);
            createdAt(model.createdAt);
            issuedAt(model.issuedAt);
            issuerCA(model.issuerCA);
            notBefore(model.notBefore);
            notAfter(model.notAfter);
            eligibleToRenew(model.eligibleToRenew);
            renewalSummary(model.renewalSummary);
            revokedAt(model.revokedAt);
            revocationReason(model.revocationReason);
            tags(model.tags);
            supportCode(model.supportCode);
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDomainName() {
            return domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(CertificateStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getSerialNumber() {
            return serialNumber;
        }

        @Override
        public final Builder serialNumber(String serialNumber) {
            this.serialNumber = serialNumber;
            return this;
        }

        public final void setSerialNumber(String serialNumber) {
            this.serialNumber = serialNumber;
        }

        public final Collection<String> getSubjectAlternativeNames() {
            if (subjectAlternativeNames instanceof SdkAutoConstructList) {
                return null;
            }
            return subjectAlternativeNames;
        }

        @Override
        public final Builder subjectAlternativeNames(Collection<String> subjectAlternativeNames) {
            this.subjectAlternativeNames = SubjectAlternativeNameListCopier.copy(subjectAlternativeNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subjectAlternativeNames(String... subjectAlternativeNames) {
            subjectAlternativeNames(Arrays.asList(subjectAlternativeNames));
            return this;
        }

        public final void setSubjectAlternativeNames(Collection<String> subjectAlternativeNames) {
            this.subjectAlternativeNames = SubjectAlternativeNameListCopier.copy(subjectAlternativeNames);
        }

        public final Collection<DomainValidationRecord.Builder> getDomainValidationRecords() {
            if (domainValidationRecords instanceof SdkAutoConstructList) {
                return null;
            }
            return domainValidationRecords != null ? domainValidationRecords.stream().map(DomainValidationRecord::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder domainValidationRecords(Collection<DomainValidationRecord> domainValidationRecords) {
            this.domainValidationRecords = DomainValidationRecordListCopier.copy(domainValidationRecords);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder domainValidationRecords(DomainValidationRecord... domainValidationRecords) {
            domainValidationRecords(Arrays.asList(domainValidationRecords));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder domainValidationRecords(Consumer<DomainValidationRecord.Builder>... domainValidationRecords) {
            domainValidationRecords(Stream.of(domainValidationRecords)
                    .map(c -> DomainValidationRecord.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setDomainValidationRecords(Collection<DomainValidationRecord.BuilderImpl> domainValidationRecords) {
            this.domainValidationRecords = DomainValidationRecordListCopier.copyFromBuilder(domainValidationRecords);
        }

        public final String getRequestFailureReason() {
            return requestFailureReason;
        }

        @Override
        public final Builder requestFailureReason(String requestFailureReason) {
            this.requestFailureReason = requestFailureReason;
            return this;
        }

        public final void setRequestFailureReason(String requestFailureReason) {
            this.requestFailureReason = requestFailureReason;
        }

        public final Integer getInUseResourceCount() {
            return inUseResourceCount;
        }

        @Override
        public final Builder inUseResourceCount(Integer inUseResourceCount) {
            this.inUseResourceCount = inUseResourceCount;
            return this;
        }

        public final void setInUseResourceCount(Integer inUseResourceCount) {
            this.inUseResourceCount = inUseResourceCount;
        }

        public final String getKeyAlgorithm() {
            return keyAlgorithm;
        }

        @Override
        public final Builder keyAlgorithm(String keyAlgorithm) {
            this.keyAlgorithm = keyAlgorithm;
            return this;
        }

        public final void setKeyAlgorithm(String keyAlgorithm) {
            this.keyAlgorithm = keyAlgorithm;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final Instant getIssuedAt() {
            return issuedAt;
        }

        @Override
        public final Builder issuedAt(Instant issuedAt) {
            this.issuedAt = issuedAt;
            return this;
        }

        public final void setIssuedAt(Instant issuedAt) {
            this.issuedAt = issuedAt;
        }

        public final String getIssuerCA() {
            return issuerCA;
        }

        @Override
        public final Builder issuerCA(String issuerCA) {
            this.issuerCA = issuerCA;
            return this;
        }

        public final void setIssuerCA(String issuerCA) {
            this.issuerCA = issuerCA;
        }

        public final Instant getNotBefore() {
            return notBefore;
        }

        @Override
        public final Builder notBefore(Instant notBefore) {
            this.notBefore = notBefore;
            return this;
        }

        public final void setNotBefore(Instant notBefore) {
            this.notBefore = notBefore;
        }

        public final Instant getNotAfter() {
            return notAfter;
        }

        @Override
        public final Builder notAfter(Instant notAfter) {
            this.notAfter = notAfter;
            return this;
        }

        public final void setNotAfter(Instant notAfter) {
            this.notAfter = notAfter;
        }

        public final String getEligibleToRenew() {
            return eligibleToRenew;
        }

        @Override
        public final Builder eligibleToRenew(String eligibleToRenew) {
            this.eligibleToRenew = eligibleToRenew;
            return this;
        }

        public final void setEligibleToRenew(String eligibleToRenew) {
            this.eligibleToRenew = eligibleToRenew;
        }

        public final RenewalSummary.Builder getRenewalSummary() {
            return renewalSummary != null ? renewalSummary.toBuilder() : null;
        }

        @Override
        public final Builder renewalSummary(RenewalSummary renewalSummary) {
            this.renewalSummary = renewalSummary;
            return this;
        }

        public final void setRenewalSummary(RenewalSummary.BuilderImpl renewalSummary) {
            this.renewalSummary = renewalSummary != null ? renewalSummary.build() : null;
        }

        public final Instant getRevokedAt() {
            return revokedAt;
        }

        @Override
        public final Builder revokedAt(Instant revokedAt) {
            this.revokedAt = revokedAt;
            return this;
        }

        public final void setRevokedAt(Instant revokedAt) {
            this.revokedAt = revokedAt;
        }

        public final String getRevocationReason() {
            return revocationReason;
        }

        @Override
        public final Builder revocationReason(String revocationReason) {
            this.revocationReason = revocationReason;
            return this;
        }

        public final void setRevocationReason(String revocationReason) {
            this.revocationReason = revocationReason;
        }

        public final Collection<Tag.Builder> getTags() {
            if (tags instanceof SdkAutoConstructList) {
                return null;
            }
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final String getSupportCode() {
            return supportCode;
        }

        @Override
        public final Builder supportCode(String supportCode) {
            this.supportCode = supportCode;
            return this;
        }

        public final void setSupportCode(String supportCode) {
            this.supportCode = supportCode;
        }

        @Override
        public Certificate build() {
            return new Certificate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
