/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.licensemanagerlinuxsubscriptions;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.internal.LicenseManagerLinuxSubscriptionsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.DeregisterSubscriptionProviderRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.DeregisterSubscriptionProviderResponse;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.GetRegisteredSubscriptionProviderRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.GetRegisteredSubscriptionProviderResponse;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.GetServiceSettingsRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.GetServiceSettingsResponse;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.InternalServerException;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.LicenseManagerLinuxSubscriptionsException;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionInstancesRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionInstancesResponse;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionsRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListLinuxSubscriptionsResponse;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListRegisteredSubscriptionProvidersRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListRegisteredSubscriptionProvidersResponse;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.RegisterSubscriptionProviderRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.RegisterSubscriptionProviderResponse;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ResourceNotFoundException;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.TagResourceRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.TagResourceResponse;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ThrottlingException;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.UntagResourceRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.UntagResourceResponse;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.UpdateServiceSettingsRequest;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.UpdateServiceSettingsResponse;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.model.ValidationException;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.transform.DeregisterSubscriptionProviderRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.transform.GetRegisteredSubscriptionProviderRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.transform.GetServiceSettingsRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.transform.ListLinuxSubscriptionInstancesRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.transform.ListLinuxSubscriptionsRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.transform.ListRegisteredSubscriptionProvidersRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.transform.RegisterSubscriptionProviderRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanagerlinuxsubscriptions.transform.UpdateServiceSettingsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link LicenseManagerLinuxSubscriptionsAsyncClient}.
 *
 * @see LicenseManagerLinuxSubscriptionsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLicenseManagerLinuxSubscriptionsAsyncClient implements LicenseManagerLinuxSubscriptionsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultLicenseManagerLinuxSubscriptionsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLicenseManagerLinuxSubscriptionsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "License_Manager_Linux_Subscriptions" + "#" + ServiceVersionInfo.VERSION)
                .build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Remove a third-party subscription provider from the Bring Your Own License (BYOL) subscriptions registered to
     * your account.
     * </p>
     *
     * @param deregisterSubscriptionProviderRequest
     * @return A Java Future containing the result of the DeregisterSubscriptionProvider operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException Unable to find the requested Amazon Web Services resource.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.DeregisterSubscriptionProvider
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/DeregisterSubscriptionProvider"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterSubscriptionProviderResponse> deregisterSubscriptionProvider(
            DeregisterSubscriptionProviderRequest deregisterSubscriptionProviderRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deregisterSubscriptionProviderRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deregisterSubscriptionProviderRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager Linux Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterSubscriptionProvider");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterSubscriptionProviderResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterSubscriptionProviderResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException")
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException")
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeregisterSubscriptionProviderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterSubscriptionProviderRequest, DeregisterSubscriptionProviderResponse>()
                            .withOperationName("DeregisterSubscriptionProvider").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeregisterSubscriptionProviderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deregisterSubscriptionProviderRequest));
            CompletableFuture<DeregisterSubscriptionProviderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get details for a Bring Your Own License (BYOL) subscription that's registered to your account.
     * </p>
     *
     * @param getRegisteredSubscriptionProviderRequest
     * @return A Java Future containing the result of the GetRegisteredSubscriptionProvider operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException Unable to find the requested Amazon Web Services resource.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.GetRegisteredSubscriptionProvider
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/GetRegisteredSubscriptionProvider"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRegisteredSubscriptionProviderResponse> getRegisteredSubscriptionProvider(
            GetRegisteredSubscriptionProviderRequest getRegisteredSubscriptionProviderRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRegisteredSubscriptionProviderRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getRegisteredSubscriptionProviderRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager Linux Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRegisteredSubscriptionProvider");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRegisteredSubscriptionProviderResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetRegisteredSubscriptionProviderResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException")
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException")
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetRegisteredSubscriptionProviderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRegisteredSubscriptionProviderRequest, GetRegisteredSubscriptionProviderResponse>()
                            .withOperationName("GetRegisteredSubscriptionProvider").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRegisteredSubscriptionProviderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRegisteredSubscriptionProviderRequest));
            CompletableFuture<GetRegisteredSubscriptionProviderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Linux subscriptions service settings for your account.
     * </p>
     *
     * @param getServiceSettingsRequest
     * @return A Java Future containing the result of the GetServiceSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.GetServiceSettings
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/GetServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceSettingsResponse> getServiceSettings(GetServiceSettingsRequest getServiceSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager Linux Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServiceSettingsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException")
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException")
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetServiceSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceSettingsRequest, GetServiceSettingsResponse>()
                            .withOperationName("GetServiceSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetServiceSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getServiceSettingsRequest));
            CompletableFuture<GetServiceSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the running Amazon EC2 instances that were discovered with commercial Linux subscriptions.
     * </p>
     *
     * @param listLinuxSubscriptionInstancesRequest
     *        NextToken length limit is half of ddb accepted limit. Increase this limit if parameters in request
     *        increases.
     * @return A Java Future containing the result of the ListLinuxSubscriptionInstances operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.ListLinuxSubscriptionInstances
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/ListLinuxSubscriptionInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLinuxSubscriptionInstancesResponse> listLinuxSubscriptionInstances(
            ListLinuxSubscriptionInstancesRequest listLinuxSubscriptionInstancesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLinuxSubscriptionInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listLinuxSubscriptionInstancesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager Linux Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLinuxSubscriptionInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLinuxSubscriptionInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLinuxSubscriptionInstancesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException")
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException")
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListLinuxSubscriptionInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLinuxSubscriptionInstancesRequest, ListLinuxSubscriptionInstancesResponse>()
                            .withOperationName("ListLinuxSubscriptionInstances").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListLinuxSubscriptionInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLinuxSubscriptionInstancesRequest));
            CompletableFuture<ListLinuxSubscriptionInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Linux subscriptions that have been discovered. If you have linked your organization, the returned
     * results will include data aggregated across your accounts in Organizations.
     * </p>
     *
     * @param listLinuxSubscriptionsRequest
     *        NextToken length limit is half of ddb accepted limit. Increase this limit if parameters in request
     *        increases.
     * @return A Java Future containing the result of the ListLinuxSubscriptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.ListLinuxSubscriptions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/ListLinuxSubscriptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLinuxSubscriptionsResponse> listLinuxSubscriptions(
            ListLinuxSubscriptionsRequest listLinuxSubscriptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLinuxSubscriptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLinuxSubscriptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager Linux Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLinuxSubscriptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLinuxSubscriptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLinuxSubscriptionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException")
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException")
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListLinuxSubscriptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLinuxSubscriptionsRequest, ListLinuxSubscriptionsResponse>()
                            .withOperationName("ListLinuxSubscriptions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListLinuxSubscriptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLinuxSubscriptionsRequest));
            CompletableFuture<ListLinuxSubscriptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List Bring Your Own License (BYOL) subscription registration resources for your account.
     * </p>
     *
     * @param listRegisteredSubscriptionProvidersRequest
     * @return A Java Future containing the result of the ListRegisteredSubscriptionProviders operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.ListRegisteredSubscriptionProviders
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/ListRegisteredSubscriptionProviders"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRegisteredSubscriptionProvidersResponse> listRegisteredSubscriptionProviders(
            ListRegisteredSubscriptionProvidersRequest listRegisteredSubscriptionProvidersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRegisteredSubscriptionProvidersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listRegisteredSubscriptionProvidersRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager Linux Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRegisteredSubscriptionProviders");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRegisteredSubscriptionProvidersResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListRegisteredSubscriptionProvidersResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException")
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException")
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListRegisteredSubscriptionProvidersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRegisteredSubscriptionProvidersRequest, ListRegisteredSubscriptionProvidersResponse>()
                            .withOperationName("ListRegisteredSubscriptionProviders").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRegisteredSubscriptionProvidersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRegisteredSubscriptionProvidersRequest));
            CompletableFuture<ListRegisteredSubscriptionProvidersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the metadata tags that are assigned to the specified Amazon Web Services resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException Unable to find the requested Amazon Web Services resource.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.ListTagsForResource
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager Linux Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException")
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException")
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Register the supported third-party subscription provider for your Bring Your Own License (BYOL) subscription.
     * </p>
     *
     * @param registerSubscriptionProviderRequest
     * @return A Java Future containing the result of the RegisterSubscriptionProvider operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.RegisterSubscriptionProvider
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/RegisterSubscriptionProvider"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterSubscriptionProviderResponse> registerSubscriptionProvider(
            RegisterSubscriptionProviderRequest registerSubscriptionProviderRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerSubscriptionProviderRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerSubscriptionProviderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager Linux Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterSubscriptionProvider");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterSubscriptionProviderResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterSubscriptionProviderResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException")
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException")
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<RegisterSubscriptionProviderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterSubscriptionProviderRequest, RegisterSubscriptionProviderResponse>()
                            .withOperationName("RegisterSubscriptionProvider").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RegisterSubscriptionProviderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(registerSubscriptionProviderRequest));
            CompletableFuture<RegisterSubscriptionProviderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add metadata tags to the specified Amazon Web Services resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException Unable to find the requested Amazon Web Services resource.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager Linux Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException")
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException")
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove one or more metadata tag from the specified Amazon Web Services resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ResourceNotFoundException Unable to find the requested Amazon Web Services resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.UntagResource
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager Linux Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException")
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException")
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the service settings for Linux subscriptions.
     * </p>
     *
     * @param updateServiceSettingsRequest
     * @return A Java Future containing the result of the UpdateServiceSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException An exception occurred with the service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerLinuxSubscriptionsException Base class for all service exceptions. Unknown exceptions
     *         will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerLinuxSubscriptionsAsyncClient.UpdateServiceSettings
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-linux-subscriptions-2018-05-10/UpdateServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateServiceSettingsResponse> updateServiceSettings(
            UpdateServiceSettingsRequest updateServiceSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateServiceSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager Linux Subscriptions");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServiceSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateServiceSettingsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException")
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException")
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateServiceSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceSettingsRequest, UpdateServiceSettingsResponse>()
                            .withOperationName("UpdateServiceSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateServiceSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateServiceSettingsRequest));
            CompletableFuture<UpdateServiceSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final LicenseManagerLinuxSubscriptionsServiceClientConfiguration serviceClientConfiguration() {
        return new LicenseManagerLinuxSubscriptionsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder())
                .build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LicenseManagerLinuxSubscriptionsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        LicenseManagerLinuxSubscriptionsServiceClientConfigurationBuilder serviceConfigBuilder = new LicenseManagerLinuxSubscriptionsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
