/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelsv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configures the Intent Disambiguation feature that helps resolve ambiguous user inputs when multiple intents could
 * match. When enabled, the system presents clarifying questions to users, helping them specify their exact intent for
 * improved conversation accuracy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IntentDisambiguationSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<IntentDisambiguationSettings.Builder, IntentDisambiguationSettings> {
    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("enabled").getter(getter(IntentDisambiguationSettings::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enabled").build()).build();

    private static final SdkField<Integer> MAX_DISAMBIGUATION_INTENTS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxDisambiguationIntents").getter(getter(IntentDisambiguationSettings::maxDisambiguationIntents))
            .setter(setter(Builder::maxDisambiguationIntents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxDisambiguationIntents").build())
            .build();

    private static final SdkField<String> CUSTOM_DISAMBIGUATION_MESSAGE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("customDisambiguationMessage")
            .getter(getter(IntentDisambiguationSettings::customDisambiguationMessage))
            .setter(setter(Builder::customDisambiguationMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("customDisambiguationMessage")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENABLED_FIELD,
            MAX_DISAMBIGUATION_INTENTS_FIELD, CUSTOM_DISAMBIGUATION_MESSAGE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Boolean enabled;

    private final Integer maxDisambiguationIntents;

    private final String customDisambiguationMessage;

    private IntentDisambiguationSettings(BuilderImpl builder) {
        this.enabled = builder.enabled;
        this.maxDisambiguationIntents = builder.maxDisambiguationIntents;
        this.customDisambiguationMessage = builder.customDisambiguationMessage;
    }

    /**
     * <p>
     * Determines whether the Intent Disambiguation feature is enabled. When set to <code>true</code>, Amazon Lex will
     * present disambiguation options to users when multiple intents could match their input, with the default being
     * <code>false</code>.
     * </p>
     * 
     * @return Determines whether the Intent Disambiguation feature is enabled. When set to <code>true</code>, Amazon
     *         Lex will present disambiguation options to users when multiple intents could match their input, with the
     *         default being <code>false</code>.
     */
    public final Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * Specifies the maximum number of intent options (2-5) to present to users when disambiguation is needed. This
     * setting determines how many intent options will be shown to users when the system detects ambiguous input. The
     * default value is 3.
     * </p>
     * 
     * @return Specifies the maximum number of intent options (2-5) to present to users when disambiguation is needed.
     *         This setting determines how many intent options will be shown to users when the system detects ambiguous
     *         input. The default value is 3.
     */
    public final Integer maxDisambiguationIntents() {
        return maxDisambiguationIntents;
    }

    /**
     * <p>
     * Provides a custom message that will be displayed before presenting the disambiguation options to users. This
     * message helps set the context for users and can be customized to match your bot's tone and brand. If not
     * specified, a default message will be used.
     * </p>
     * 
     * @return Provides a custom message that will be displayed before presenting the disambiguation options to users.
     *         This message helps set the context for users and can be customized to match your bot's tone and brand. If
     *         not specified, a default message will be used.
     */
    public final String customDisambiguationMessage() {
        return customDisambiguationMessage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(maxDisambiguationIntents());
        hashCode = 31 * hashCode + Objects.hashCode(customDisambiguationMessage());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IntentDisambiguationSettings)) {
            return false;
        }
        IntentDisambiguationSettings other = (IntentDisambiguationSettings) obj;
        return Objects.equals(enabled(), other.enabled())
                && Objects.equals(maxDisambiguationIntents(), other.maxDisambiguationIntents())
                && Objects.equals(customDisambiguationMessage(), other.customDisambiguationMessage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IntentDisambiguationSettings").add("Enabled", enabled())
                .add("MaxDisambiguationIntents", maxDisambiguationIntents())
                .add("CustomDisambiguationMessage", customDisambiguationMessage()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "maxDisambiguationIntents":
            return Optional.ofNullable(clazz.cast(maxDisambiguationIntents()));
        case "customDisambiguationMessage":
            return Optional.ofNullable(clazz.cast(customDisambiguationMessage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("enabled", ENABLED_FIELD);
        map.put("maxDisambiguationIntents", MAX_DISAMBIGUATION_INTENTS_FIELD);
        map.put("customDisambiguationMessage", CUSTOM_DISAMBIGUATION_MESSAGE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<IntentDisambiguationSettings, T> g) {
        return obj -> g.apply((IntentDisambiguationSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IntentDisambiguationSettings> {
        /**
         * <p>
         * Determines whether the Intent Disambiguation feature is enabled. When set to <code>true</code>, Amazon Lex
         * will present disambiguation options to users when multiple intents could match their input, with the default
         * being <code>false</code>.
         * </p>
         * 
         * @param enabled
         *        Determines whether the Intent Disambiguation feature is enabled. When set to <code>true</code>, Amazon
         *        Lex will present disambiguation options to users when multiple intents could match their input, with
         *        the default being <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * Specifies the maximum number of intent options (2-5) to present to users when disambiguation is needed. This
         * setting determines how many intent options will be shown to users when the system detects ambiguous input.
         * The default value is 3.
         * </p>
         * 
         * @param maxDisambiguationIntents
         *        Specifies the maximum number of intent options (2-5) to present to users when disambiguation is
         *        needed. This setting determines how many intent options will be shown to users when the system detects
         *        ambiguous input. The default value is 3.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxDisambiguationIntents(Integer maxDisambiguationIntents);

        /**
         * <p>
         * Provides a custom message that will be displayed before presenting the disambiguation options to users. This
         * message helps set the context for users and can be customized to match your bot's tone and brand. If not
         * specified, a default message will be used.
         * </p>
         * 
         * @param customDisambiguationMessage
         *        Provides a custom message that will be displayed before presenting the disambiguation options to
         *        users. This message helps set the context for users and can be customized to match your bot's tone and
         *        brand. If not specified, a default message will be used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customDisambiguationMessage(String customDisambiguationMessage);
    }

    static final class BuilderImpl implements Builder {
        private Boolean enabled;

        private Integer maxDisambiguationIntents;

        private String customDisambiguationMessage;

        private BuilderImpl() {
        }

        private BuilderImpl(IntentDisambiguationSettings model) {
            enabled(model.enabled);
            maxDisambiguationIntents(model.maxDisambiguationIntents);
            customDisambiguationMessage(model.customDisambiguationMessage);
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final Integer getMaxDisambiguationIntents() {
            return maxDisambiguationIntents;
        }

        public final void setMaxDisambiguationIntents(Integer maxDisambiguationIntents) {
            this.maxDisambiguationIntents = maxDisambiguationIntents;
        }

        @Override
        public final Builder maxDisambiguationIntents(Integer maxDisambiguationIntents) {
            this.maxDisambiguationIntents = maxDisambiguationIntents;
            return this;
        }

        public final String getCustomDisambiguationMessage() {
            return customDisambiguationMessage;
        }

        public final void setCustomDisambiguationMessage(String customDisambiguationMessage) {
            this.customDisambiguationMessage = customDisambiguationMessage;
        }

        @Override
        public final Builder customDisambiguationMessage(String customDisambiguationMessage) {
            this.customDisambiguationMessage = customDisambiguationMessage;
            return this;
        }

        @Override
        public IntentDisambiguationSettings build() {
            return new IntentDisambiguationSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
