/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lambda.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A ZIP archive that contains the contents of an <a
 * href="https://docs.aws.amazon.com/lambda/latest/dg/configuration-layers.html">AWS Lambda layer</a>. You can specify
 * either an Amazon S3 location, or upload a layer archive directly.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LayerVersionContentInput implements SdkPojo, Serializable,
        ToCopyableBuilder<LayerVersionContentInput.Builder, LayerVersionContentInput> {
    private static final SdkField<String> S3_BUCKET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LayerVersionContentInput::s3Bucket)).setter(setter(Builder::s3Bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Bucket").build()).build();

    private static final SdkField<String> S3_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LayerVersionContentInput::s3Key)).setter(setter(Builder::s3Key))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Key").build()).build();

    private static final SdkField<String> S3_OBJECT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LayerVersionContentInput::s3ObjectVersion)).setter(setter(Builder::s3ObjectVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3ObjectVersion").build()).build();

    private static final SdkField<SdkBytes> ZIP_FILE_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .getter(getter(LayerVersionContentInput::zipFile)).setter(setter(Builder::zipFile))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ZipFile").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_BUCKET_FIELD, S3_KEY_FIELD,
            S3_OBJECT_VERSION_FIELD, ZIP_FILE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String s3Bucket;

    private final String s3Key;

    private final String s3ObjectVersion;

    private final SdkBytes zipFile;

    private LayerVersionContentInput(BuilderImpl builder) {
        this.s3Bucket = builder.s3Bucket;
        this.s3Key = builder.s3Key;
        this.s3ObjectVersion = builder.s3ObjectVersion;
        this.zipFile = builder.zipFile;
    }

    /**
     * <p>
     * The Amazon S3 bucket of the layer archive.
     * </p>
     * 
     * @return The Amazon S3 bucket of the layer archive.
     */
    public String s3Bucket() {
        return s3Bucket;
    }

    /**
     * <p>
     * The Amazon S3 key of the layer archive.
     * </p>
     * 
     * @return The Amazon S3 key of the layer archive.
     */
    public String s3Key() {
        return s3Key;
    }

    /**
     * <p>
     * For versioned objects, the version of the layer archive object to use.
     * </p>
     * 
     * @return For versioned objects, the version of the layer archive object to use.
     */
    public String s3ObjectVersion() {
        return s3ObjectVersion;
    }

    /**
     * <p>
     * The base64-encoded contents of the layer archive. AWS SDK and AWS CLI clients handle the encoding for you.
     * </p>
     * 
     * @return The base64-encoded contents of the layer archive. AWS SDK and AWS CLI clients handle the encoding for
     *         you.
     */
    public SdkBytes zipFile() {
        return zipFile;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3Bucket());
        hashCode = 31 * hashCode + Objects.hashCode(s3Key());
        hashCode = 31 * hashCode + Objects.hashCode(s3ObjectVersion());
        hashCode = 31 * hashCode + Objects.hashCode(zipFile());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LayerVersionContentInput)) {
            return false;
        }
        LayerVersionContentInput other = (LayerVersionContentInput) obj;
        return Objects.equals(s3Bucket(), other.s3Bucket()) && Objects.equals(s3Key(), other.s3Key())
                && Objects.equals(s3ObjectVersion(), other.s3ObjectVersion()) && Objects.equals(zipFile(), other.zipFile());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("LayerVersionContentInput").add("S3Bucket", s3Bucket()).add("S3Key", s3Key())
                .add("S3ObjectVersion", s3ObjectVersion())
                .add("ZipFile", zipFile() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3Bucket":
            return Optional.ofNullable(clazz.cast(s3Bucket()));
        case "S3Key":
            return Optional.ofNullable(clazz.cast(s3Key()));
        case "S3ObjectVersion":
            return Optional.ofNullable(clazz.cast(s3ObjectVersion()));
        case "ZipFile":
            return Optional.ofNullable(clazz.cast(zipFile()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LayerVersionContentInput, T> g) {
        return obj -> g.apply((LayerVersionContentInput) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LayerVersionContentInput> {
        /**
         * <p>
         * The Amazon S3 bucket of the layer archive.
         * </p>
         * 
         * @param s3Bucket
         *        The Amazon S3 bucket of the layer archive.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Bucket(String s3Bucket);

        /**
         * <p>
         * The Amazon S3 key of the layer archive.
         * </p>
         * 
         * @param s3Key
         *        The Amazon S3 key of the layer archive.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Key(String s3Key);

        /**
         * <p>
         * For versioned objects, the version of the layer archive object to use.
         * </p>
         * 
         * @param s3ObjectVersion
         *        For versioned objects, the version of the layer archive object to use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3ObjectVersion(String s3ObjectVersion);

        /**
         * <p>
         * The base64-encoded contents of the layer archive. AWS SDK and AWS CLI clients handle the encoding for you.
         * </p>
         * 
         * @param zipFile
         *        The base64-encoded contents of the layer archive. AWS SDK and AWS CLI clients handle the encoding for
         *        you.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder zipFile(SdkBytes zipFile);
    }

    static final class BuilderImpl implements Builder {
        private String s3Bucket;

        private String s3Key;

        private String s3ObjectVersion;

        private SdkBytes zipFile;

        private BuilderImpl() {
        }

        private BuilderImpl(LayerVersionContentInput model) {
            s3Bucket(model.s3Bucket);
            s3Key(model.s3Key);
            s3ObjectVersion(model.s3ObjectVersion);
            zipFile(model.zipFile);
        }

        public final String getS3Bucket() {
            return s3Bucket;
        }

        @Override
        public final Builder s3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
            return this;
        }

        public final void setS3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
        }

        public final String getS3Key() {
            return s3Key;
        }

        @Override
        public final Builder s3Key(String s3Key) {
            this.s3Key = s3Key;
            return this;
        }

        public final void setS3Key(String s3Key) {
            this.s3Key = s3Key;
        }

        public final String getS3ObjectVersion() {
            return s3ObjectVersion;
        }

        @Override
        public final Builder s3ObjectVersion(String s3ObjectVersion) {
            this.s3ObjectVersion = s3ObjectVersion;
            return this;
        }

        public final void setS3ObjectVersion(String s3ObjectVersion) {
            this.s3ObjectVersion = s3ObjectVersion;
        }

        public final ByteBuffer getZipFile() {
            return zipFile == null ? null : zipFile.asByteBuffer();
        }

        @Override
        public final Builder zipFile(SdkBytes zipFile) {
            this.zipFile = StandardMemberCopier.copy(zipFile);
            return this;
        }

        public final void setZipFile(ByteBuffer zipFile) {
            zipFile(zipFile == null ? null : SdkBytes.fromByteBuffer(zipFile));
        }

        @Override
        public LayerVersionContentInput build() {
            return new LayerVersionContentInput(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
