/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.keyspacesstreams.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a change data capture record for a row in an Amazon Keyspaces table, containing both the new and old
 * states of the row.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Record implements SdkPojo, Serializable, ToCopyableBuilder<Record.Builder, Record> {
    private static final SdkField<String> EVENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eventVersion").getter(getter(Record::eventVersion)).setter(setter(Builder::eventVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventVersion").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(Record::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<String> ORIGIN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("origin")
            .getter(getter(Record::originAsString)).setter(setter(Builder::origin))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("origin").build()).build();

    private static final SdkField<Map<String, KeyspacesCellValue>> PARTITION_KEYS_FIELD = SdkField
            .<Map<String, KeyspacesCellValue>> builder(MarshallingType.MAP)
            .memberName("partitionKeys")
            .getter(getter(Record::partitionKeys))
            .setter(setter(Builder::partitionKeys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("partitionKeys").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<KeyspacesCellValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyspacesCellValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, KeyspacesCellValue>> CLUSTERING_KEYS_FIELD = SdkField
            .<Map<String, KeyspacesCellValue>> builder(MarshallingType.MAP)
            .memberName("clusteringKeys")
            .getter(getter(Record::clusteringKeys))
            .setter(setter(Builder::clusteringKeys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clusteringKeys").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<KeyspacesCellValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyspacesCellValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<KeyspacesRow> NEW_IMAGE_FIELD = SdkField.<KeyspacesRow> builder(MarshallingType.SDK_POJO)
            .memberName("newImage").getter(getter(Record::newImage)).setter(setter(Builder::newImage))
            .constructor(KeyspacesRow::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("newImage").build()).build();

    private static final SdkField<KeyspacesRow> OLD_IMAGE_FIELD = SdkField.<KeyspacesRow> builder(MarshallingType.SDK_POJO)
            .memberName("oldImage").getter(getter(Record::oldImage)).setter(setter(Builder::oldImage))
            .constructor(KeyspacesRow::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("oldImage").build()).build();

    private static final SdkField<String> SEQUENCE_NUMBER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sequenceNumber").getter(getter(Record::sequenceNumber)).setter(setter(Builder::sequenceNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sequenceNumber").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_VERSION_FIELD,
            CREATED_AT_FIELD, ORIGIN_FIELD, PARTITION_KEYS_FIELD, CLUSTERING_KEYS_FIELD, NEW_IMAGE_FIELD, OLD_IMAGE_FIELD,
            SEQUENCE_NUMBER_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String eventVersion;

    private final Instant createdAt;

    private final String origin;

    private final Map<String, KeyspacesCellValue> partitionKeys;

    private final Map<String, KeyspacesCellValue> clusteringKeys;

    private final KeyspacesRow newImage;

    private final KeyspacesRow oldImage;

    private final String sequenceNumber;

    private Record(BuilderImpl builder) {
        this.eventVersion = builder.eventVersion;
        this.createdAt = builder.createdAt;
        this.origin = builder.origin;
        this.partitionKeys = builder.partitionKeys;
        this.clusteringKeys = builder.clusteringKeys;
        this.newImage = builder.newImage;
        this.oldImage = builder.oldImage;
        this.sequenceNumber = builder.sequenceNumber;
    }

    /**
     * <p>
     * The version of the record format, used to track the evolution of the record structure over time.
     * </p>
     * 
     * @return The version of the record format, used to track the evolution of the record structure over time.
     */
    public final String eventVersion() {
        return eventVersion;
    }

    /**
     * <p>
     * The timestamp indicating when this change data capture record was created.
     * </p>
     * 
     * @return The timestamp indicating when this change data capture record was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The origin or source of this change data capture record.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #origin} will
     * return {@link OriginType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #originAsString}.
     * </p>
     * 
     * @return The origin or source of this change data capture record.
     * @see OriginType
     */
    public final OriginType origin() {
        return OriginType.fromValue(origin);
    }

    /**
     * <p>
     * The origin or source of this change data capture record.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #origin} will
     * return {@link OriginType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #originAsString}.
     * </p>
     * 
     * @return The origin or source of this change data capture record.
     * @see OriginType
     */
    public final String originAsString() {
        return origin;
    }

    /**
     * For responses, this returns true if the service returned a value for the PartitionKeys property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPartitionKeys() {
        return partitionKeys != null && !(partitionKeys instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The partition key columns and their values for the affected row.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPartitionKeys} method.
     * </p>
     * 
     * @return The partition key columns and their values for the affected row.
     */
    public final Map<String, KeyspacesCellValue> partitionKeys() {
        return partitionKeys;
    }

    /**
     * For responses, this returns true if the service returned a value for the ClusteringKeys property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasClusteringKeys() {
        return clusteringKeys != null && !(clusteringKeys instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The clustering key columns and their values for the affected row, which determine the order of rows within a
     * partition.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasClusteringKeys} method.
     * </p>
     * 
     * @return The clustering key columns and their values for the affected row, which determine the order of rows
     *         within a partition.
     */
    public final Map<String, KeyspacesCellValue> clusteringKeys() {
        return clusteringKeys;
    }

    /**
     * <p>
     * The state of the row after the change operation that generated this record.
     * </p>
     * 
     * @return The state of the row after the change operation that generated this record.
     */
    public final KeyspacesRow newImage() {
        return newImage;
    }

    /**
     * <p>
     * The state of the row before the change operation that generated this record.
     * </p>
     * 
     * @return The state of the row before the change operation that generated this record.
     */
    public final KeyspacesRow oldImage() {
        return oldImage;
    }

    /**
     * <p>
     * A unique identifier assigned to this record within the shard, used for ordering and tracking purposes.
     * </p>
     * 
     * @return A unique identifier assigned to this record within the shard, used for ordering and tracking purposes.
     */
    public final String sequenceNumber() {
        return sequenceNumber;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eventVersion());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(originAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasPartitionKeys() ? partitionKeys() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasClusteringKeys() ? clusteringKeys() : null);
        hashCode = 31 * hashCode + Objects.hashCode(newImage());
        hashCode = 31 * hashCode + Objects.hashCode(oldImage());
        hashCode = 31 * hashCode + Objects.hashCode(sequenceNumber());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Record)) {
            return false;
        }
        Record other = (Record) obj;
        return Objects.equals(eventVersion(), other.eventVersion()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(originAsString(), other.originAsString()) && hasPartitionKeys() == other.hasPartitionKeys()
                && Objects.equals(partitionKeys(), other.partitionKeys()) && hasClusteringKeys() == other.hasClusteringKeys()
                && Objects.equals(clusteringKeys(), other.clusteringKeys()) && Objects.equals(newImage(), other.newImage())
                && Objects.equals(oldImage(), other.oldImage()) && Objects.equals(sequenceNumber(), other.sequenceNumber());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Record").add("EventVersion", eventVersion()).add("CreatedAt", createdAt())
                .add("Origin", originAsString()).add("PartitionKeys", hasPartitionKeys() ? partitionKeys() : null)
                .add("ClusteringKeys", hasClusteringKeys() ? clusteringKeys() : null).add("NewImage", newImage())
                .add("OldImage", oldImage()).add("SequenceNumber", sequenceNumber()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "eventVersion":
            return Optional.ofNullable(clazz.cast(eventVersion()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "origin":
            return Optional.ofNullable(clazz.cast(originAsString()));
        case "partitionKeys":
            return Optional.ofNullable(clazz.cast(partitionKeys()));
        case "clusteringKeys":
            return Optional.ofNullable(clazz.cast(clusteringKeys()));
        case "newImage":
            return Optional.ofNullable(clazz.cast(newImage()));
        case "oldImage":
            return Optional.ofNullable(clazz.cast(oldImage()));
        case "sequenceNumber":
            return Optional.ofNullable(clazz.cast(sequenceNumber()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("eventVersion", EVENT_VERSION_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("origin", ORIGIN_FIELD);
        map.put("partitionKeys", PARTITION_KEYS_FIELD);
        map.put("clusteringKeys", CLUSTERING_KEYS_FIELD);
        map.put("newImage", NEW_IMAGE_FIELD);
        map.put("oldImage", OLD_IMAGE_FIELD);
        map.put("sequenceNumber", SEQUENCE_NUMBER_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Record, T> g) {
        return obj -> g.apply((Record) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Record> {
        /**
         * <p>
         * The version of the record format, used to track the evolution of the record structure over time.
         * </p>
         * 
         * @param eventVersion
         *        The version of the record format, used to track the evolution of the record structure over time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventVersion(String eventVersion);

        /**
         * <p>
         * The timestamp indicating when this change data capture record was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp indicating when this change data capture record was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The origin or source of this change data capture record.
         * </p>
         * 
         * @param origin
         *        The origin or source of this change data capture record.
         * @see OriginType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OriginType
         */
        Builder origin(String origin);

        /**
         * <p>
         * The origin or source of this change data capture record.
         * </p>
         * 
         * @param origin
         *        The origin or source of this change data capture record.
         * @see OriginType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OriginType
         */
        Builder origin(OriginType origin);

        /**
         * <p>
         * The partition key columns and their values for the affected row.
         * </p>
         * 
         * @param partitionKeys
         *        The partition key columns and their values for the affected row.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionKeys(Map<String, KeyspacesCellValue> partitionKeys);

        /**
         * <p>
         * The clustering key columns and their values for the affected row, which determine the order of rows within a
         * partition.
         * </p>
         * 
         * @param clusteringKeys
         *        The clustering key columns and their values for the affected row, which determine the order of rows
         *        within a partition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusteringKeys(Map<String, KeyspacesCellValue> clusteringKeys);

        /**
         * <p>
         * The state of the row after the change operation that generated this record.
         * </p>
         * 
         * @param newImage
         *        The state of the row after the change operation that generated this record.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder newImage(KeyspacesRow newImage);

        /**
         * <p>
         * The state of the row after the change operation that generated this record.
         * </p>
         * This is a convenience method that creates an instance of the {@link KeyspacesRow.Builder} avoiding the need
         * to create one manually via {@link KeyspacesRow#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KeyspacesRow.Builder#build()} is called immediately and its
         * result is passed to {@link #newImage(KeyspacesRow)}.
         * 
         * @param newImage
         *        a consumer that will call methods on {@link KeyspacesRow.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #newImage(KeyspacesRow)
         */
        default Builder newImage(Consumer<KeyspacesRow.Builder> newImage) {
            return newImage(KeyspacesRow.builder().applyMutation(newImage).build());
        }

        /**
         * <p>
         * The state of the row before the change operation that generated this record.
         * </p>
         * 
         * @param oldImage
         *        The state of the row before the change operation that generated this record.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder oldImage(KeyspacesRow oldImage);

        /**
         * <p>
         * The state of the row before the change operation that generated this record.
         * </p>
         * This is a convenience method that creates an instance of the {@link KeyspacesRow.Builder} avoiding the need
         * to create one manually via {@link KeyspacesRow#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KeyspacesRow.Builder#build()} is called immediately and its
         * result is passed to {@link #oldImage(KeyspacesRow)}.
         * 
         * @param oldImage
         *        a consumer that will call methods on {@link KeyspacesRow.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #oldImage(KeyspacesRow)
         */
        default Builder oldImage(Consumer<KeyspacesRow.Builder> oldImage) {
            return oldImage(KeyspacesRow.builder().applyMutation(oldImage).build());
        }

        /**
         * <p>
         * A unique identifier assigned to this record within the shard, used for ordering and tracking purposes.
         * </p>
         * 
         * @param sequenceNumber
         *        A unique identifier assigned to this record within the shard, used for ordering and tracking purposes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sequenceNumber(String sequenceNumber);
    }

    static final class BuilderImpl implements Builder {
        private String eventVersion;

        private Instant createdAt;

        private String origin;

        private Map<String, KeyspacesCellValue> partitionKeys = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, KeyspacesCellValue> clusteringKeys = DefaultSdkAutoConstructMap.getInstance();

        private KeyspacesRow newImage;

        private KeyspacesRow oldImage;

        private String sequenceNumber;

        private BuilderImpl() {
        }

        private BuilderImpl(Record model) {
            eventVersion(model.eventVersion);
            createdAt(model.createdAt);
            origin(model.origin);
            partitionKeys(model.partitionKeys);
            clusteringKeys(model.clusteringKeys);
            newImage(model.newImage);
            oldImage(model.oldImage);
            sequenceNumber(model.sequenceNumber);
        }

        public final String getEventVersion() {
            return eventVersion;
        }

        public final void setEventVersion(String eventVersion) {
            this.eventVersion = eventVersion;
        }

        @Override
        public final Builder eventVersion(String eventVersion) {
            this.eventVersion = eventVersion;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getOrigin() {
            return origin;
        }

        public final void setOrigin(String origin) {
            this.origin = origin;
        }

        @Override
        public final Builder origin(String origin) {
            this.origin = origin;
            return this;
        }

        @Override
        public final Builder origin(OriginType origin) {
            this.origin(origin == null ? null : origin.toString());
            return this;
        }

        public final Map<String, KeyspacesCellValue.Builder> getPartitionKeys() {
            Map<String, KeyspacesCellValue.Builder> result = KeyspacesKeysMapCopier.copyToBuilder(this.partitionKeys);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setPartitionKeys(Map<String, KeyspacesCellValue.BuilderImpl> partitionKeys) {
            this.partitionKeys = KeyspacesKeysMapCopier.copyFromBuilder(partitionKeys);
        }

        @Override
        public final Builder partitionKeys(Map<String, KeyspacesCellValue> partitionKeys) {
            this.partitionKeys = KeyspacesKeysMapCopier.copy(partitionKeys);
            return this;
        }

        public final Map<String, KeyspacesCellValue.Builder> getClusteringKeys() {
            Map<String, KeyspacesCellValue.Builder> result = KeyspacesKeysMapCopier.copyToBuilder(this.clusteringKeys);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setClusteringKeys(Map<String, KeyspacesCellValue.BuilderImpl> clusteringKeys) {
            this.clusteringKeys = KeyspacesKeysMapCopier.copyFromBuilder(clusteringKeys);
        }

        @Override
        public final Builder clusteringKeys(Map<String, KeyspacesCellValue> clusteringKeys) {
            this.clusteringKeys = KeyspacesKeysMapCopier.copy(clusteringKeys);
            return this;
        }

        public final KeyspacesRow.Builder getNewImage() {
            return newImage != null ? newImage.toBuilder() : null;
        }

        public final void setNewImage(KeyspacesRow.BuilderImpl newImage) {
            this.newImage = newImage != null ? newImage.build() : null;
        }

        @Override
        public final Builder newImage(KeyspacesRow newImage) {
            this.newImage = newImage;
            return this;
        }

        public final KeyspacesRow.Builder getOldImage() {
            return oldImage != null ? oldImage.toBuilder() : null;
        }

        public final void setOldImage(KeyspacesRow.BuilderImpl oldImage) {
            this.oldImage = oldImage != null ? oldImage.build() : null;
        }

        @Override
        public final Builder oldImage(KeyspacesRow oldImage) {
            this.oldImage = oldImage;
            return this;
        }

        public final String getSequenceNumber() {
            return sequenceNumber;
        }

        public final void setSequenceNumber(String sequenceNumber) {
            this.sequenceNumber = sequenceNumber;
        }

        @Override
        public final Builder sequenceNumber(String sequenceNumber) {
            this.sequenceNumber = sequenceNumber;
            return this;
        }

        @Override
        public Record build() {
            return new Record(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
