/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.internetmonitor.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a health event created in a monitor in Amazon CloudWatch Internet Monitor.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HealthEvent implements SdkPojo, Serializable, ToCopyableBuilder<HealthEvent.Builder, HealthEvent> {
    private static final SdkField<String> EVENT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventArn").getter(getter(HealthEvent::eventArn)).setter(setter(Builder::eventArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventArn").build()).build();

    private static final SdkField<String> EVENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventId").getter(getter(HealthEvent::eventId)).setter(setter(Builder::eventId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventId").build()).build();

    private static final SdkField<Instant> STARTED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartedAt")
            .getter(getter(HealthEvent::startedAt))
            .setter(setter(Builder::startedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> ENDED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndedAt")
            .getter(getter(HealthEvent::endedAt))
            .setter(setter(Builder::endedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt")
            .getter(getter(HealthEvent::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> LAST_UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedAt")
            .getter(getter(HealthEvent::lastUpdatedAt))
            .setter(setter(Builder::lastUpdatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<List<ImpactedLocation>> IMPACTED_LOCATIONS_FIELD = SdkField
            .<List<ImpactedLocation>> builder(MarshallingType.LIST)
            .memberName("ImpactedLocations")
            .getter(getter(HealthEvent::impactedLocations))
            .setter(setter(Builder::impactedLocations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImpactedLocations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ImpactedLocation> builder(MarshallingType.SDK_POJO)
                                            .constructor(ImpactedLocation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(HealthEvent::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Double> PERCENT_OF_TOTAL_TRAFFIC_IMPACTED_FIELD = SdkField
            .<Double> builder(MarshallingType.DOUBLE)
            .memberName("PercentOfTotalTrafficImpacted")
            .getter(getter(HealthEvent::percentOfTotalTrafficImpacted))
            .setter(setter(Builder::percentOfTotalTrafficImpacted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PercentOfTotalTrafficImpacted")
                    .build()).build();

    private static final SdkField<String> IMPACT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImpactType").getter(getter(HealthEvent::impactTypeAsString)).setter(setter(Builder::impactType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImpactType").build()).build();

    private static final SdkField<Double> HEALTH_SCORE_THRESHOLD_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("HealthScoreThreshold").getter(getter(HealthEvent::healthScoreThreshold))
            .setter(setter(Builder::healthScoreThreshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthScoreThreshold").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_ARN_FIELD,
            EVENT_ID_FIELD, STARTED_AT_FIELD, ENDED_AT_FIELD, CREATED_AT_FIELD, LAST_UPDATED_AT_FIELD, IMPACTED_LOCATIONS_FIELD,
            STATUS_FIELD, PERCENT_OF_TOTAL_TRAFFIC_IMPACTED_FIELD, IMPACT_TYPE_FIELD, HEALTH_SCORE_THRESHOLD_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String eventArn;

    private final String eventId;

    private final Instant startedAt;

    private final Instant endedAt;

    private final Instant createdAt;

    private final Instant lastUpdatedAt;

    private final List<ImpactedLocation> impactedLocations;

    private final String status;

    private final Double percentOfTotalTrafficImpacted;

    private final String impactType;

    private final Double healthScoreThreshold;

    private HealthEvent(BuilderImpl builder) {
        this.eventArn = builder.eventArn;
        this.eventId = builder.eventId;
        this.startedAt = builder.startedAt;
        this.endedAt = builder.endedAt;
        this.createdAt = builder.createdAt;
        this.lastUpdatedAt = builder.lastUpdatedAt;
        this.impactedLocations = builder.impactedLocations;
        this.status = builder.status;
        this.percentOfTotalTrafficImpacted = builder.percentOfTotalTrafficImpacted;
        this.impactType = builder.impactType;
        this.healthScoreThreshold = builder.healthScoreThreshold;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the event.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the event.
     */
    public final String eventArn() {
        return eventArn;
    }

    /**
     * <p>
     * The internally-generated identifier of a specific network traffic impairment health event.
     * </p>
     * 
     * @return The internally-generated identifier of a specific network traffic impairment health event.
     */
    public final String eventId() {
        return eventId;
    }

    /**
     * <p>
     * When a health event started.
     * </p>
     * 
     * @return When a health event started.
     */
    public final Instant startedAt() {
        return startedAt;
    }

    /**
     * <p>
     * The time when a health event ended. If the health event is still active, then the end time is not set.
     * </p>
     * 
     * @return The time when a health event ended. If the health event is still active, then the end time is not set.
     */
    public final Instant endedAt() {
        return endedAt;
    }

    /**
     * <p>
     * When the health event was created.
     * </p>
     * 
     * @return When the health event was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * When the health event was last updated.
     * </p>
     * 
     * @return When the health event was last updated.
     */
    public final Instant lastUpdatedAt() {
        return lastUpdatedAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the ImpactedLocations property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasImpactedLocations() {
        return impactedLocations != null && !(impactedLocations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The locations impacted by the health event.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasImpactedLocations} method.
     * </p>
     * 
     * @return The locations impacted by the health event.
     */
    public final List<ImpactedLocation> impactedLocations() {
        return impactedLocations;
    }

    /**
     * <p>
     * The status of a health event.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link HealthEventStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a health event.
     * @see HealthEventStatus
     */
    public final HealthEventStatus status() {
        return HealthEventStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of a health event.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link HealthEventStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a health event.
     * @see HealthEventStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The impact on total traffic that a health event has, in increased latency or reduced availability. This is the
     * percentage of how much latency has increased or availability has decreased during the event, compared to what is
     * typical for traffic from this client location to the Amazon Web Services location using this client network.
     * </p>
     * 
     * @return The impact on total traffic that a health event has, in increased latency or reduced availability. This
     *         is the percentage of how much latency has increased or availability has decreased during the event,
     *         compared to what is typical for traffic from this client location to the Amazon Web Services location
     *         using this client network.
     */
    public final Double percentOfTotalTrafficImpacted() {
        return percentOfTotalTrafficImpacted;
    }

    /**
     * <p>
     * The type of impairment for a health event.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #impactType} will
     * return {@link HealthEventImpactType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #impactTypeAsString}.
     * </p>
     * 
     * @return The type of impairment for a health event.
     * @see HealthEventImpactType
     */
    public final HealthEventImpactType impactType() {
        return HealthEventImpactType.fromValue(impactType);
    }

    /**
     * <p>
     * The type of impairment for a health event.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #impactType} will
     * return {@link HealthEventImpactType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #impactTypeAsString}.
     * </p>
     * 
     * @return The type of impairment for a health event.
     * @see HealthEventImpactType
     */
    public final String impactTypeAsString() {
        return impactType;
    }

    /**
     * <p>
     * The value of the threshold percentage for performance or availability that was configured when Amazon CloudWatch
     * Internet Monitor created the health event.
     * </p>
     * 
     * @return The value of the threshold percentage for performance or availability that was configured when Amazon
     *         CloudWatch Internet Monitor created the health event.
     */
    public final Double healthScoreThreshold() {
        return healthScoreThreshold;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eventArn());
        hashCode = 31 * hashCode + Objects.hashCode(eventId());
        hashCode = 31 * hashCode + Objects.hashCode(startedAt());
        hashCode = 31 * hashCode + Objects.hashCode(endedAt());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasImpactedLocations() ? impactedLocations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(percentOfTotalTrafficImpacted());
        hashCode = 31 * hashCode + Objects.hashCode(impactTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(healthScoreThreshold());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HealthEvent)) {
            return false;
        }
        HealthEvent other = (HealthEvent) obj;
        return Objects.equals(eventArn(), other.eventArn()) && Objects.equals(eventId(), other.eventId())
                && Objects.equals(startedAt(), other.startedAt()) && Objects.equals(endedAt(), other.endedAt())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(lastUpdatedAt(), other.lastUpdatedAt())
                && hasImpactedLocations() == other.hasImpactedLocations()
                && Objects.equals(impactedLocations(), other.impactedLocations())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(percentOfTotalTrafficImpacted(), other.percentOfTotalTrafficImpacted())
                && Objects.equals(impactTypeAsString(), other.impactTypeAsString())
                && Objects.equals(healthScoreThreshold(), other.healthScoreThreshold());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HealthEvent").add("EventArn", eventArn()).add("EventId", eventId())
                .add("StartedAt", startedAt()).add("EndedAt", endedAt()).add("CreatedAt", createdAt())
                .add("LastUpdatedAt", lastUpdatedAt())
                .add("ImpactedLocations", hasImpactedLocations() ? impactedLocations() : null).add("Status", statusAsString())
                .add("PercentOfTotalTrafficImpacted", percentOfTotalTrafficImpacted()).add("ImpactType", impactTypeAsString())
                .add("HealthScoreThreshold", healthScoreThreshold()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EventArn":
            return Optional.ofNullable(clazz.cast(eventArn()));
        case "EventId":
            return Optional.ofNullable(clazz.cast(eventId()));
        case "StartedAt":
            return Optional.ofNullable(clazz.cast(startedAt()));
        case "EndedAt":
            return Optional.ofNullable(clazz.cast(endedAt()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "LastUpdatedAt":
            return Optional.ofNullable(clazz.cast(lastUpdatedAt()));
        case "ImpactedLocations":
            return Optional.ofNullable(clazz.cast(impactedLocations()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "PercentOfTotalTrafficImpacted":
            return Optional.ofNullable(clazz.cast(percentOfTotalTrafficImpacted()));
        case "ImpactType":
            return Optional.ofNullable(clazz.cast(impactTypeAsString()));
        case "HealthScoreThreshold":
            return Optional.ofNullable(clazz.cast(healthScoreThreshold()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("EventArn", EVENT_ARN_FIELD);
        map.put("EventId", EVENT_ID_FIELD);
        map.put("StartedAt", STARTED_AT_FIELD);
        map.put("EndedAt", ENDED_AT_FIELD);
        map.put("CreatedAt", CREATED_AT_FIELD);
        map.put("LastUpdatedAt", LAST_UPDATED_AT_FIELD);
        map.put("ImpactedLocations", IMPACTED_LOCATIONS_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("PercentOfTotalTrafficImpacted", PERCENT_OF_TOTAL_TRAFFIC_IMPACTED_FIELD);
        map.put("ImpactType", IMPACT_TYPE_FIELD);
        map.put("HealthScoreThreshold", HEALTH_SCORE_THRESHOLD_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<HealthEvent, T> g) {
        return obj -> g.apply((HealthEvent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HealthEvent> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the event.
         * </p>
         * 
         * @param eventArn
         *        The Amazon Resource Name (ARN) of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventArn(String eventArn);

        /**
         * <p>
         * The internally-generated identifier of a specific network traffic impairment health event.
         * </p>
         * 
         * @param eventId
         *        The internally-generated identifier of a specific network traffic impairment health event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventId(String eventId);

        /**
         * <p>
         * When a health event started.
         * </p>
         * 
         * @param startedAt
         *        When a health event started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedAt(Instant startedAt);

        /**
         * <p>
         * The time when a health event ended. If the health event is still active, then the end time is not set.
         * </p>
         * 
         * @param endedAt
         *        The time when a health event ended. If the health event is still active, then the end time is not set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endedAt(Instant endedAt);

        /**
         * <p>
         * When the health event was created.
         * </p>
         * 
         * @param createdAt
         *        When the health event was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * When the health event was last updated.
         * </p>
         * 
         * @param lastUpdatedAt
         *        When the health event was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedAt(Instant lastUpdatedAt);

        /**
         * <p>
         * The locations impacted by the health event.
         * </p>
         * 
         * @param impactedLocations
         *        The locations impacted by the health event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder impactedLocations(Collection<ImpactedLocation> impactedLocations);

        /**
         * <p>
         * The locations impacted by the health event.
         * </p>
         * 
         * @param impactedLocations
         *        The locations impacted by the health event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder impactedLocations(ImpactedLocation... impactedLocations);

        /**
         * <p>
         * The locations impacted by the health event.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.internetmonitor.model.ImpactedLocation.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.internetmonitor.model.ImpactedLocation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.internetmonitor.model.ImpactedLocation.Builder#build()} is called
         * immediately and its result is passed to {@link #impactedLocations(List<ImpactedLocation>)}.
         * 
         * @param impactedLocations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.internetmonitor.model.ImpactedLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #impactedLocations(java.util.Collection<ImpactedLocation>)
         */
        Builder impactedLocations(Consumer<ImpactedLocation.Builder>... impactedLocations);

        /**
         * <p>
         * The status of a health event.
         * </p>
         * 
         * @param status
         *        The status of a health event.
         * @see HealthEventStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HealthEventStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of a health event.
         * </p>
         * 
         * @param status
         *        The status of a health event.
         * @see HealthEventStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HealthEventStatus
         */
        Builder status(HealthEventStatus status);

        /**
         * <p>
         * The impact on total traffic that a health event has, in increased latency or reduced availability. This is
         * the percentage of how much latency has increased or availability has decreased during the event, compared to
         * what is typical for traffic from this client location to the Amazon Web Services location using this client
         * network.
         * </p>
         * 
         * @param percentOfTotalTrafficImpacted
         *        The impact on total traffic that a health event has, in increased latency or reduced availability.
         *        This is the percentage of how much latency has increased or availability has decreased during the
         *        event, compared to what is typical for traffic from this client location to the Amazon Web Services
         *        location using this client network.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentOfTotalTrafficImpacted(Double percentOfTotalTrafficImpacted);

        /**
         * <p>
         * The type of impairment for a health event.
         * </p>
         * 
         * @param impactType
         *        The type of impairment for a health event.
         * @see HealthEventImpactType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HealthEventImpactType
         */
        Builder impactType(String impactType);

        /**
         * <p>
         * The type of impairment for a health event.
         * </p>
         * 
         * @param impactType
         *        The type of impairment for a health event.
         * @see HealthEventImpactType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HealthEventImpactType
         */
        Builder impactType(HealthEventImpactType impactType);

        /**
         * <p>
         * The value of the threshold percentage for performance or availability that was configured when Amazon
         * CloudWatch Internet Monitor created the health event.
         * </p>
         * 
         * @param healthScoreThreshold
         *        The value of the threshold percentage for performance or availability that was configured when Amazon
         *        CloudWatch Internet Monitor created the health event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthScoreThreshold(Double healthScoreThreshold);
    }

    static final class BuilderImpl implements Builder {
        private String eventArn;

        private String eventId;

        private Instant startedAt;

        private Instant endedAt;

        private Instant createdAt;

        private Instant lastUpdatedAt;

        private List<ImpactedLocation> impactedLocations = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private Double percentOfTotalTrafficImpacted;

        private String impactType;

        private Double healthScoreThreshold;

        private BuilderImpl() {
        }

        private BuilderImpl(HealthEvent model) {
            eventArn(model.eventArn);
            eventId(model.eventId);
            startedAt(model.startedAt);
            endedAt(model.endedAt);
            createdAt(model.createdAt);
            lastUpdatedAt(model.lastUpdatedAt);
            impactedLocations(model.impactedLocations);
            status(model.status);
            percentOfTotalTrafficImpacted(model.percentOfTotalTrafficImpacted);
            impactType(model.impactType);
            healthScoreThreshold(model.healthScoreThreshold);
        }

        public final String getEventArn() {
            return eventArn;
        }

        public final void setEventArn(String eventArn) {
            this.eventArn = eventArn;
        }

        @Override
        public final Builder eventArn(String eventArn) {
            this.eventArn = eventArn;
            return this;
        }

        public final String getEventId() {
            return eventId;
        }

        public final void setEventId(String eventId) {
            this.eventId = eventId;
        }

        @Override
        public final Builder eventId(String eventId) {
            this.eventId = eventId;
            return this;
        }

        public final Instant getStartedAt() {
            return startedAt;
        }

        public final void setStartedAt(Instant startedAt) {
            this.startedAt = startedAt;
        }

        @Override
        public final Builder startedAt(Instant startedAt) {
            this.startedAt = startedAt;
            return this;
        }

        public final Instant getEndedAt() {
            return endedAt;
        }

        public final void setEndedAt(Instant endedAt) {
            this.endedAt = endedAt;
        }

        @Override
        public final Builder endedAt(Instant endedAt) {
            this.endedAt = endedAt;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getLastUpdatedAt() {
            return lastUpdatedAt;
        }

        public final void setLastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
        }

        @Override
        public final Builder lastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
            return this;
        }

        public final List<ImpactedLocation.Builder> getImpactedLocations() {
            List<ImpactedLocation.Builder> result = ImpactedLocationsListCopier.copyToBuilder(this.impactedLocations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setImpactedLocations(Collection<ImpactedLocation.BuilderImpl> impactedLocations) {
            this.impactedLocations = ImpactedLocationsListCopier.copyFromBuilder(impactedLocations);
        }

        @Override
        public final Builder impactedLocations(Collection<ImpactedLocation> impactedLocations) {
            this.impactedLocations = ImpactedLocationsListCopier.copy(impactedLocations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder impactedLocations(ImpactedLocation... impactedLocations) {
            impactedLocations(Arrays.asList(impactedLocations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder impactedLocations(Consumer<ImpactedLocation.Builder>... impactedLocations) {
            impactedLocations(Stream.of(impactedLocations).map(c -> ImpactedLocation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(HealthEventStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Double getPercentOfTotalTrafficImpacted() {
            return percentOfTotalTrafficImpacted;
        }

        public final void setPercentOfTotalTrafficImpacted(Double percentOfTotalTrafficImpacted) {
            this.percentOfTotalTrafficImpacted = percentOfTotalTrafficImpacted;
        }

        @Override
        public final Builder percentOfTotalTrafficImpacted(Double percentOfTotalTrafficImpacted) {
            this.percentOfTotalTrafficImpacted = percentOfTotalTrafficImpacted;
            return this;
        }

        public final String getImpactType() {
            return impactType;
        }

        public final void setImpactType(String impactType) {
            this.impactType = impactType;
        }

        @Override
        public final Builder impactType(String impactType) {
            this.impactType = impactType;
            return this;
        }

        @Override
        public final Builder impactType(HealthEventImpactType impactType) {
            this.impactType(impactType == null ? null : impactType.toString());
            return this;
        }

        public final Double getHealthScoreThreshold() {
            return healthScoreThreshold;
        }

        public final void setHealthScoreThreshold(Double healthScoreThreshold) {
            this.healthScoreThreshold = healthScoreThreshold;
        }

        @Override
        public final Builder healthScoreThreshold(Double healthScoreThreshold) {
            this.healthScoreThreshold = healthScoreThreshold;
            return this;
        }

        @Override
        public HealthEvent build() {
            return new HealthEvent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
