/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iam.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the result of the simulation of a single API operation call on a single resource.
 * </p>
 * <p>
 * This data type is used by a member of the <a>EvaluationResult</a> data type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceSpecificResult implements SdkPojo, Serializable,
        ToCopyableBuilder<ResourceSpecificResult.Builder, ResourceSpecificResult> {
    private static final SdkField<String> EVAL_RESOURCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EvalResourceName").getter(getter(ResourceSpecificResult::evalResourceName))
            .setter(setter(Builder::evalResourceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvalResourceName").build()).build();

    private static final SdkField<String> EVAL_RESOURCE_DECISION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EvalResourceDecision").getter(getter(ResourceSpecificResult::evalResourceDecisionAsString))
            .setter(setter(Builder::evalResourceDecision))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvalResourceDecision").build())
            .build();

    private static final SdkField<List<Statement>> MATCHED_STATEMENTS_FIELD = SdkField
            .<List<Statement>> builder(MarshallingType.LIST)
            .memberName("MatchedStatements")
            .getter(getter(ResourceSpecificResult::matchedStatements))
            .setter(setter(Builder::matchedStatements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MatchedStatements").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Statement> builder(MarshallingType.SDK_POJO)
                                            .constructor(Statement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> MISSING_CONTEXT_VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("MissingContextValues")
            .getter(getter(ResourceSpecificResult::missingContextValues))
            .setter(setter(Builder::missingContextValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MissingContextValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> EVAL_DECISION_DETAILS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("EvalDecisionDetails")
            .getter(getter(ResourceSpecificResult::evalDecisionDetailsAsStrings))
            .setter(setter(Builder::evalDecisionDetailsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvalDecisionDetails").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<PermissionsBoundaryDecisionDetail> PERMISSIONS_BOUNDARY_DECISION_DETAIL_FIELD = SdkField
            .<PermissionsBoundaryDecisionDetail> builder(MarshallingType.SDK_POJO)
            .memberName("PermissionsBoundaryDecisionDetail")
            .getter(getter(ResourceSpecificResult::permissionsBoundaryDecisionDetail))
            .setter(setter(Builder::permissionsBoundaryDecisionDetail))
            .constructor(PermissionsBoundaryDecisionDetail::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PermissionsBoundaryDecisionDetail")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVAL_RESOURCE_NAME_FIELD,
            EVAL_RESOURCE_DECISION_FIELD, MATCHED_STATEMENTS_FIELD, MISSING_CONTEXT_VALUES_FIELD, EVAL_DECISION_DETAILS_FIELD,
            PERMISSIONS_BOUNDARY_DECISION_DETAIL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String evalResourceName;

    private final String evalResourceDecision;

    private final List<Statement> matchedStatements;

    private final List<String> missingContextValues;

    private final Map<String, String> evalDecisionDetails;

    private final PermissionsBoundaryDecisionDetail permissionsBoundaryDecisionDetail;

    private ResourceSpecificResult(BuilderImpl builder) {
        this.evalResourceName = builder.evalResourceName;
        this.evalResourceDecision = builder.evalResourceDecision;
        this.matchedStatements = builder.matchedStatements;
        this.missingContextValues = builder.missingContextValues;
        this.evalDecisionDetails = builder.evalDecisionDetails;
        this.permissionsBoundaryDecisionDetail = builder.permissionsBoundaryDecisionDetail;
    }

    /**
     * <p>
     * The name of the simulated resource, in Amazon Resource Name (ARN) format.
     * </p>
     * 
     * @return The name of the simulated resource, in Amazon Resource Name (ARN) format.
     */
    public final String evalResourceName() {
        return evalResourceName;
    }

    /**
     * <p>
     * The result of the simulation of the simulated API operation on the resource specified in
     * <code>EvalResourceName</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #evalResourceDecision} will return {@link PolicyEvaluationDecisionType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #evalResourceDecisionAsString}.
     * </p>
     * 
     * @return The result of the simulation of the simulated API operation on the resource specified in
     *         <code>EvalResourceName</code>.
     * @see PolicyEvaluationDecisionType
     */
    public final PolicyEvaluationDecisionType evalResourceDecision() {
        return PolicyEvaluationDecisionType.fromValue(evalResourceDecision);
    }

    /**
     * <p>
     * The result of the simulation of the simulated API operation on the resource specified in
     * <code>EvalResourceName</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #evalResourceDecision} will return {@link PolicyEvaluationDecisionType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #evalResourceDecisionAsString}.
     * </p>
     * 
     * @return The result of the simulation of the simulated API operation on the resource specified in
     *         <code>EvalResourceName</code>.
     * @see PolicyEvaluationDecisionType
     */
    public final String evalResourceDecisionAsString() {
        return evalResourceDecision;
    }

    /**
     * Returns true if the MatchedStatements property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasMatchedStatements() {
        return matchedStatements != null && !(matchedStatements instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the statements in the input policies that determine the result for this part of the simulation.
     * Remember that even if multiple statements allow the operation on the resource, if <i>any</i> statement denies
     * that operation, then the explicit deny overrides any allow. In addition, the deny statement is the only entry
     * included in the result.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMatchedStatements()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the statements in the input policies that determine the result for this part of the simulation.
     *         Remember that even if multiple statements allow the operation on the resource, if <i>any</i> statement
     *         denies that operation, then the explicit deny overrides any allow. In addition, the deny statement is the
     *         only entry included in the result.
     */
    public final List<Statement> matchedStatements() {
        return matchedStatements;
    }

    /**
     * Returns true if the MissingContextValues property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasMissingContextValues() {
        return missingContextValues != null && !(missingContextValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of context keys that are required by the included input policies but that were not provided by one of the
     * input parameters. This list is used when a list of ARNs is included in the <code>ResourceArns</code> parameter
     * instead of "*". If you do not specify individual resources, by setting <code>ResourceArns</code> to "*" or by not
     * including the <code>ResourceArns</code> parameter, then any missing context values are instead included under the
     * <code>EvaluationResults</code> section. To discover the context keys used by a set of policies, you can call
     * <a>GetContextKeysForCustomPolicy</a> or <a>GetContextKeysForPrincipalPolicy</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMissingContextValues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of context keys that are required by the included input policies but that were not provided by one
     *         of the input parameters. This list is used when a list of ARNs is included in the
     *         <code>ResourceArns</code> parameter instead of "*". If you do not specify individual resources, by
     *         setting <code>ResourceArns</code> to "*" or by not including the <code>ResourceArns</code> parameter,
     *         then any missing context values are instead included under the <code>EvaluationResults</code> section. To
     *         discover the context keys used by a set of policies, you can call <a>GetContextKeysForCustomPolicy</a> or
     *         <a>GetContextKeysForPrincipalPolicy</a>.
     */
    public final List<String> missingContextValues() {
        return missingContextValues;
    }

    /**
     * <p>
     * Additional details about the results of the evaluation decision on a single resource. This parameter is returned
     * only for cross-account simulations. This parameter explains how each policy type contributes to the
     * resource-specific evaluation decision.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEvalDecisionDetails()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Additional details about the results of the evaluation decision on a single resource. This parameter is
     *         returned only for cross-account simulations. This parameter explains how each policy type contributes to
     *         the resource-specific evaluation decision.
     */
    public final Map<String, PolicyEvaluationDecisionType> evalDecisionDetails() {
        return EvalDecisionDetailsTypeCopier.copyStringToEnum(evalDecisionDetails);
    }

    /**
     * Returns true if the EvalDecisionDetails property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasEvalDecisionDetails() {
        return evalDecisionDetails != null && !(evalDecisionDetails instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Additional details about the results of the evaluation decision on a single resource. This parameter is returned
     * only for cross-account simulations. This parameter explains how each policy type contributes to the
     * resource-specific evaluation decision.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEvalDecisionDetails()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Additional details about the results of the evaluation decision on a single resource. This parameter is
     *         returned only for cross-account simulations. This parameter explains how each policy type contributes to
     *         the resource-specific evaluation decision.
     */
    public final Map<String, String> evalDecisionDetailsAsStrings() {
        return evalDecisionDetails;
    }

    /**
     * <p>
     * Contains information about the effect that a permissions boundary has on a policy simulation when that boundary
     * is applied to an IAM entity.
     * </p>
     * 
     * @return Contains information about the effect that a permissions boundary has on a policy simulation when that
     *         boundary is applied to an IAM entity.
     */
    public final PermissionsBoundaryDecisionDetail permissionsBoundaryDecisionDetail() {
        return permissionsBoundaryDecisionDetail;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(evalResourceName());
        hashCode = 31 * hashCode + Objects.hashCode(evalResourceDecisionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasMatchedStatements() ? matchedStatements() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasMissingContextValues() ? missingContextValues() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEvalDecisionDetails() ? evalDecisionDetailsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(permissionsBoundaryDecisionDetail());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceSpecificResult)) {
            return false;
        }
        ResourceSpecificResult other = (ResourceSpecificResult) obj;
        return Objects.equals(evalResourceName(), other.evalResourceName())
                && Objects.equals(evalResourceDecisionAsString(), other.evalResourceDecisionAsString())
                && hasMatchedStatements() == other.hasMatchedStatements()
                && Objects.equals(matchedStatements(), other.matchedStatements())
                && hasMissingContextValues() == other.hasMissingContextValues()
                && Objects.equals(missingContextValues(), other.missingContextValues())
                && hasEvalDecisionDetails() == other.hasEvalDecisionDetails()
                && Objects.equals(evalDecisionDetailsAsStrings(), other.evalDecisionDetailsAsStrings())
                && Objects.equals(permissionsBoundaryDecisionDetail(), other.permissionsBoundaryDecisionDetail());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceSpecificResult").add("EvalResourceName", evalResourceName())
                .add("EvalResourceDecision", evalResourceDecisionAsString())
                .add("MatchedStatements", hasMatchedStatements() ? matchedStatements() : null)
                .add("MissingContextValues", hasMissingContextValues() ? missingContextValues() : null)
                .add("EvalDecisionDetails", hasEvalDecisionDetails() ? evalDecisionDetailsAsStrings() : null)
                .add("PermissionsBoundaryDecisionDetail", permissionsBoundaryDecisionDetail()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EvalResourceName":
            return Optional.ofNullable(clazz.cast(evalResourceName()));
        case "EvalResourceDecision":
            return Optional.ofNullable(clazz.cast(evalResourceDecisionAsString()));
        case "MatchedStatements":
            return Optional.ofNullable(clazz.cast(matchedStatements()));
        case "MissingContextValues":
            return Optional.ofNullable(clazz.cast(missingContextValues()));
        case "EvalDecisionDetails":
            return Optional.ofNullable(clazz.cast(evalDecisionDetailsAsStrings()));
        case "PermissionsBoundaryDecisionDetail":
            return Optional.ofNullable(clazz.cast(permissionsBoundaryDecisionDetail()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResourceSpecificResult, T> g) {
        return obj -> g.apply((ResourceSpecificResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceSpecificResult> {
        /**
         * <p>
         * The name of the simulated resource, in Amazon Resource Name (ARN) format.
         * </p>
         * 
         * @param evalResourceName
         *        The name of the simulated resource, in Amazon Resource Name (ARN) format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evalResourceName(String evalResourceName);

        /**
         * <p>
         * The result of the simulation of the simulated API operation on the resource specified in
         * <code>EvalResourceName</code>.
         * </p>
         * 
         * @param evalResourceDecision
         *        The result of the simulation of the simulated API operation on the resource specified in
         *        <code>EvalResourceName</code>.
         * @see PolicyEvaluationDecisionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyEvaluationDecisionType
         */
        Builder evalResourceDecision(String evalResourceDecision);

        /**
         * <p>
         * The result of the simulation of the simulated API operation on the resource specified in
         * <code>EvalResourceName</code>.
         * </p>
         * 
         * @param evalResourceDecision
         *        The result of the simulation of the simulated API operation on the resource specified in
         *        <code>EvalResourceName</code>.
         * @see PolicyEvaluationDecisionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyEvaluationDecisionType
         */
        Builder evalResourceDecision(PolicyEvaluationDecisionType evalResourceDecision);

        /**
         * <p>
         * A list of the statements in the input policies that determine the result for this part of the simulation.
         * Remember that even if multiple statements allow the operation on the resource, if <i>any</i> statement denies
         * that operation, then the explicit deny overrides any allow. In addition, the deny statement is the only entry
         * included in the result.
         * </p>
         * 
         * @param matchedStatements
         *        A list of the statements in the input policies that determine the result for this part of the
         *        simulation. Remember that even if multiple statements allow the operation on the resource, if
         *        <i>any</i> statement denies that operation, then the explicit deny overrides any allow. In addition,
         *        the deny statement is the only entry included in the result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchedStatements(Collection<Statement> matchedStatements);

        /**
         * <p>
         * A list of the statements in the input policies that determine the result for this part of the simulation.
         * Remember that even if multiple statements allow the operation on the resource, if <i>any</i> statement denies
         * that operation, then the explicit deny overrides any allow. In addition, the deny statement is the only entry
         * included in the result.
         * </p>
         * 
         * @param matchedStatements
         *        A list of the statements in the input policies that determine the result for this part of the
         *        simulation. Remember that even if multiple statements allow the operation on the resource, if
         *        <i>any</i> statement denies that operation, then the explicit deny overrides any allow. In addition,
         *        the deny statement is the only entry included in the result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchedStatements(Statement... matchedStatements);

        /**
         * <p>
         * A list of the statements in the input policies that determine the result for this part of the simulation.
         * Remember that even if multiple statements allow the operation on the resource, if <i>any</i> statement denies
         * that operation, then the explicit deny overrides any allow. In addition, the deny statement is the only entry
         * included in the result.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Statement>.Builder} avoiding the need to
         * create one manually via {@link List<Statement>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Statement>.Builder#build()} is called immediately and its
         * result is passed to {@link #matchedStatements(List<Statement>)}.
         * 
         * @param matchedStatements
         *        a consumer that will call methods on {@link List<Statement>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #matchedStatements(List<Statement>)
         */
        Builder matchedStatements(Consumer<Statement.Builder>... matchedStatements);

        /**
         * <p>
         * A list of context keys that are required by the included input policies but that were not provided by one of
         * the input parameters. This list is used when a list of ARNs is included in the <code>ResourceArns</code>
         * parameter instead of "*". If you do not specify individual resources, by setting <code>ResourceArns</code> to
         * "*" or by not including the <code>ResourceArns</code> parameter, then any missing context values are instead
         * included under the <code>EvaluationResults</code> section. To discover the context keys used by a set of
         * policies, you can call <a>GetContextKeysForCustomPolicy</a> or <a>GetContextKeysForPrincipalPolicy</a>.
         * </p>
         * 
         * @param missingContextValues
         *        A list of context keys that are required by the included input policies but that were not provided by
         *        one of the input parameters. This list is used when a list of ARNs is included in the
         *        <code>ResourceArns</code> parameter instead of "*". If you do not specify individual resources, by
         *        setting <code>ResourceArns</code> to "*" or by not including the <code>ResourceArns</code> parameter,
         *        then any missing context values are instead included under the <code>EvaluationResults</code> section.
         *        To discover the context keys used by a set of policies, you can call
         *        <a>GetContextKeysForCustomPolicy</a> or <a>GetContextKeysForPrincipalPolicy</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder missingContextValues(Collection<String> missingContextValues);

        /**
         * <p>
         * A list of context keys that are required by the included input policies but that were not provided by one of
         * the input parameters. This list is used when a list of ARNs is included in the <code>ResourceArns</code>
         * parameter instead of "*". If you do not specify individual resources, by setting <code>ResourceArns</code> to
         * "*" or by not including the <code>ResourceArns</code> parameter, then any missing context values are instead
         * included under the <code>EvaluationResults</code> section. To discover the context keys used by a set of
         * policies, you can call <a>GetContextKeysForCustomPolicy</a> or <a>GetContextKeysForPrincipalPolicy</a>.
         * </p>
         * 
         * @param missingContextValues
         *        A list of context keys that are required by the included input policies but that were not provided by
         *        one of the input parameters. This list is used when a list of ARNs is included in the
         *        <code>ResourceArns</code> parameter instead of "*". If you do not specify individual resources, by
         *        setting <code>ResourceArns</code> to "*" or by not including the <code>ResourceArns</code> parameter,
         *        then any missing context values are instead included under the <code>EvaluationResults</code> section.
         *        To discover the context keys used by a set of policies, you can call
         *        <a>GetContextKeysForCustomPolicy</a> or <a>GetContextKeysForPrincipalPolicy</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder missingContextValues(String... missingContextValues);

        /**
         * <p>
         * Additional details about the results of the evaluation decision on a single resource. This parameter is
         * returned only for cross-account simulations. This parameter explains how each policy type contributes to the
         * resource-specific evaluation decision.
         * </p>
         * 
         * @param evalDecisionDetails
         *        Additional details about the results of the evaluation decision on a single resource. This parameter
         *        is returned only for cross-account simulations. This parameter explains how each policy type
         *        contributes to the resource-specific evaluation decision.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evalDecisionDetailsWithStrings(Map<String, String> evalDecisionDetails);

        /**
         * <p>
         * Additional details about the results of the evaluation decision on a single resource. This parameter is
         * returned only for cross-account simulations. This parameter explains how each policy type contributes to the
         * resource-specific evaluation decision.
         * </p>
         * 
         * @param evalDecisionDetails
         *        Additional details about the results of the evaluation decision on a single resource. This parameter
         *        is returned only for cross-account simulations. This parameter explains how each policy type
         *        contributes to the resource-specific evaluation decision.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evalDecisionDetails(Map<String, PolicyEvaluationDecisionType> evalDecisionDetails);

        /**
         * <p>
         * Contains information about the effect that a permissions boundary has on a policy simulation when that
         * boundary is applied to an IAM entity.
         * </p>
         * 
         * @param permissionsBoundaryDecisionDetail
         *        Contains information about the effect that a permissions boundary has on a policy simulation when that
         *        boundary is applied to an IAM entity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissionsBoundaryDecisionDetail(PermissionsBoundaryDecisionDetail permissionsBoundaryDecisionDetail);

        /**
         * <p>
         * Contains information about the effect that a permissions boundary has on a policy simulation when that
         * boundary is applied to an IAM entity.
         * </p>
         * This is a convenience that creates an instance of the {@link PermissionsBoundaryDecisionDetail.Builder}
         * avoiding the need to create one manually via {@link PermissionsBoundaryDecisionDetail#builder()}.
         *
         * When the {@link Consumer} completes, {@link PermissionsBoundaryDecisionDetail.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #permissionsBoundaryDecisionDetail(PermissionsBoundaryDecisionDetail)}.
         * 
         * @param permissionsBoundaryDecisionDetail
         *        a consumer that will call methods on {@link PermissionsBoundaryDecisionDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #permissionsBoundaryDecisionDetail(PermissionsBoundaryDecisionDetail)
         */
        default Builder permissionsBoundaryDecisionDetail(
                Consumer<PermissionsBoundaryDecisionDetail.Builder> permissionsBoundaryDecisionDetail) {
            return permissionsBoundaryDecisionDetail(PermissionsBoundaryDecisionDetail.builder()
                    .applyMutation(permissionsBoundaryDecisionDetail).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String evalResourceName;

        private String evalResourceDecision;

        private List<Statement> matchedStatements = DefaultSdkAutoConstructList.getInstance();

        private List<String> missingContextValues = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> evalDecisionDetails = DefaultSdkAutoConstructMap.getInstance();

        private PermissionsBoundaryDecisionDetail permissionsBoundaryDecisionDetail;

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceSpecificResult model) {
            evalResourceName(model.evalResourceName);
            evalResourceDecision(model.evalResourceDecision);
            matchedStatements(model.matchedStatements);
            missingContextValues(model.missingContextValues);
            evalDecisionDetailsWithStrings(model.evalDecisionDetails);
            permissionsBoundaryDecisionDetail(model.permissionsBoundaryDecisionDetail);
        }

        public final String getEvalResourceName() {
            return evalResourceName;
        }

        @Override
        public final Builder evalResourceName(String evalResourceName) {
            this.evalResourceName = evalResourceName;
            return this;
        }

        public final void setEvalResourceName(String evalResourceName) {
            this.evalResourceName = evalResourceName;
        }

        public final String getEvalResourceDecision() {
            return evalResourceDecision;
        }

        @Override
        public final Builder evalResourceDecision(String evalResourceDecision) {
            this.evalResourceDecision = evalResourceDecision;
            return this;
        }

        @Override
        public final Builder evalResourceDecision(PolicyEvaluationDecisionType evalResourceDecision) {
            this.evalResourceDecision(evalResourceDecision == null ? null : evalResourceDecision.toString());
            return this;
        }

        public final void setEvalResourceDecision(String evalResourceDecision) {
            this.evalResourceDecision = evalResourceDecision;
        }

        public final Collection<Statement.Builder> getMatchedStatements() {
            if (matchedStatements instanceof SdkAutoConstructList) {
                return null;
            }
            return matchedStatements != null ? matchedStatements.stream().map(Statement::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder matchedStatements(Collection<Statement> matchedStatements) {
            this.matchedStatements = StatementListTypeCopier.copy(matchedStatements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchedStatements(Statement... matchedStatements) {
            matchedStatements(Arrays.asList(matchedStatements));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchedStatements(Consumer<Statement.Builder>... matchedStatements) {
            matchedStatements(Stream.of(matchedStatements).map(c -> Statement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setMatchedStatements(Collection<Statement.BuilderImpl> matchedStatements) {
            this.matchedStatements = StatementListTypeCopier.copyFromBuilder(matchedStatements);
        }

        public final Collection<String> getMissingContextValues() {
            if (missingContextValues instanceof SdkAutoConstructList) {
                return null;
            }
            return missingContextValues;
        }

        @Override
        public final Builder missingContextValues(Collection<String> missingContextValues) {
            this.missingContextValues = ContextKeyNamesResultListTypeCopier.copy(missingContextValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder missingContextValues(String... missingContextValues) {
            missingContextValues(Arrays.asList(missingContextValues));
            return this;
        }

        public final void setMissingContextValues(Collection<String> missingContextValues) {
            this.missingContextValues = ContextKeyNamesResultListTypeCopier.copy(missingContextValues);
        }

        public final Map<String, String> getEvalDecisionDetails() {
            if (evalDecisionDetails instanceof SdkAutoConstructMap) {
                return null;
            }
            return evalDecisionDetails;
        }

        @Override
        public final Builder evalDecisionDetailsWithStrings(Map<String, String> evalDecisionDetails) {
            this.evalDecisionDetails = EvalDecisionDetailsTypeCopier.copy(evalDecisionDetails);
            return this;
        }

        @Override
        public final Builder evalDecisionDetails(Map<String, PolicyEvaluationDecisionType> evalDecisionDetails) {
            this.evalDecisionDetails = EvalDecisionDetailsTypeCopier.copyEnumToString(evalDecisionDetails);
            return this;
        }

        public final void setEvalDecisionDetails(Map<String, String> evalDecisionDetails) {
            this.evalDecisionDetails = EvalDecisionDetailsTypeCopier.copy(evalDecisionDetails);
        }

        public final PermissionsBoundaryDecisionDetail.Builder getPermissionsBoundaryDecisionDetail() {
            return permissionsBoundaryDecisionDetail != null ? permissionsBoundaryDecisionDetail.toBuilder() : null;
        }

        @Override
        public final Builder permissionsBoundaryDecisionDetail(PermissionsBoundaryDecisionDetail permissionsBoundaryDecisionDetail) {
            this.permissionsBoundaryDecisionDetail = permissionsBoundaryDecisionDetail;
            return this;
        }

        public final void setPermissionsBoundaryDecisionDetail(
                PermissionsBoundaryDecisionDetail.BuilderImpl permissionsBoundaryDecisionDetail) {
            this.permissionsBoundaryDecisionDetail = permissionsBoundaryDecisionDetail != null ? permissionsBoundaryDecisionDetail
                    .build() : null;
        }

        @Override
        public ResourceSpecificResult build() {
            return new ResourceSpecificResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
