/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.globalaccelerator;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.globalaccelerator.model.AcceleratorNotDisabledException;
import software.amazon.awssdk.services.globalaccelerator.model.AcceleratorNotFoundException;
import software.amazon.awssdk.services.globalaccelerator.model.AccessDeniedException;
import software.amazon.awssdk.services.globalaccelerator.model.AdvertiseByoipCidrRequest;
import software.amazon.awssdk.services.globalaccelerator.model.AdvertiseByoipCidrResponse;
import software.amazon.awssdk.services.globalaccelerator.model.AssociatedEndpointGroupFoundException;
import software.amazon.awssdk.services.globalaccelerator.model.AssociatedListenerFoundException;
import software.amazon.awssdk.services.globalaccelerator.model.ByoipCidrNotFoundException;
import software.amazon.awssdk.services.globalaccelerator.model.CreateAcceleratorRequest;
import software.amazon.awssdk.services.globalaccelerator.model.CreateAcceleratorResponse;
import software.amazon.awssdk.services.globalaccelerator.model.CreateEndpointGroupRequest;
import software.amazon.awssdk.services.globalaccelerator.model.CreateEndpointGroupResponse;
import software.amazon.awssdk.services.globalaccelerator.model.CreateListenerRequest;
import software.amazon.awssdk.services.globalaccelerator.model.CreateListenerResponse;
import software.amazon.awssdk.services.globalaccelerator.model.DeleteAcceleratorRequest;
import software.amazon.awssdk.services.globalaccelerator.model.DeleteAcceleratorResponse;
import software.amazon.awssdk.services.globalaccelerator.model.DeleteEndpointGroupRequest;
import software.amazon.awssdk.services.globalaccelerator.model.DeleteEndpointGroupResponse;
import software.amazon.awssdk.services.globalaccelerator.model.DeleteListenerRequest;
import software.amazon.awssdk.services.globalaccelerator.model.DeleteListenerResponse;
import software.amazon.awssdk.services.globalaccelerator.model.DeprovisionByoipCidrRequest;
import software.amazon.awssdk.services.globalaccelerator.model.DeprovisionByoipCidrResponse;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeAcceleratorAttributesRequest;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeAcceleratorAttributesResponse;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeAcceleratorRequest;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeAcceleratorResponse;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeEndpointGroupRequest;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeEndpointGroupResponse;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeListenerRequest;
import software.amazon.awssdk.services.globalaccelerator.model.DescribeListenerResponse;
import software.amazon.awssdk.services.globalaccelerator.model.EndpointGroupAlreadyExistsException;
import software.amazon.awssdk.services.globalaccelerator.model.EndpointGroupNotFoundException;
import software.amazon.awssdk.services.globalaccelerator.model.GlobalAcceleratorException;
import software.amazon.awssdk.services.globalaccelerator.model.IncorrectCidrStateException;
import software.amazon.awssdk.services.globalaccelerator.model.InternalServiceErrorException;
import software.amazon.awssdk.services.globalaccelerator.model.InvalidArgumentException;
import software.amazon.awssdk.services.globalaccelerator.model.InvalidNextTokenException;
import software.amazon.awssdk.services.globalaccelerator.model.InvalidPortRangeException;
import software.amazon.awssdk.services.globalaccelerator.model.LimitExceededException;
import software.amazon.awssdk.services.globalaccelerator.model.ListAcceleratorsRequest;
import software.amazon.awssdk.services.globalaccelerator.model.ListAcceleratorsResponse;
import software.amazon.awssdk.services.globalaccelerator.model.ListByoipCidrsRequest;
import software.amazon.awssdk.services.globalaccelerator.model.ListByoipCidrsResponse;
import software.amazon.awssdk.services.globalaccelerator.model.ListEndpointGroupsRequest;
import software.amazon.awssdk.services.globalaccelerator.model.ListEndpointGroupsResponse;
import software.amazon.awssdk.services.globalaccelerator.model.ListListenersRequest;
import software.amazon.awssdk.services.globalaccelerator.model.ListListenersResponse;
import software.amazon.awssdk.services.globalaccelerator.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.globalaccelerator.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.globalaccelerator.model.ListenerNotFoundException;
import software.amazon.awssdk.services.globalaccelerator.model.ProvisionByoipCidrRequest;
import software.amazon.awssdk.services.globalaccelerator.model.ProvisionByoipCidrResponse;
import software.amazon.awssdk.services.globalaccelerator.model.TagResourceRequest;
import software.amazon.awssdk.services.globalaccelerator.model.TagResourceResponse;
import software.amazon.awssdk.services.globalaccelerator.model.UntagResourceRequest;
import software.amazon.awssdk.services.globalaccelerator.model.UntagResourceResponse;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateAcceleratorAttributesRequest;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateAcceleratorAttributesResponse;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateAcceleratorRequest;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateAcceleratorResponse;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateEndpointGroupRequest;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateEndpointGroupResponse;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateListenerRequest;
import software.amazon.awssdk.services.globalaccelerator.model.UpdateListenerResponse;
import software.amazon.awssdk.services.globalaccelerator.model.WithdrawByoipCidrRequest;
import software.amazon.awssdk.services.globalaccelerator.model.WithdrawByoipCidrResponse;
import software.amazon.awssdk.services.globalaccelerator.transform.AdvertiseByoipCidrRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.CreateAcceleratorRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.CreateEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.CreateListenerRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.DeleteAcceleratorRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.DeleteEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.DeleteListenerRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.DeprovisionByoipCidrRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.DescribeAcceleratorAttributesRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.DescribeAcceleratorRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.DescribeEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.DescribeListenerRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.ListAcceleratorsRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.ListByoipCidrsRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.ListEndpointGroupsRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.ListListenersRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.ProvisionByoipCidrRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.UpdateAcceleratorAttributesRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.UpdateAcceleratorRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.UpdateEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.UpdateListenerRequestMarshaller;
import software.amazon.awssdk.services.globalaccelerator.transform.WithdrawByoipCidrRequestMarshaller;

/**
 * Internal implementation of {@link GlobalAcceleratorClient}.
 *
 * @see GlobalAcceleratorClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGlobalAcceleratorClient implements GlobalAcceleratorClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultGlobalAcceleratorClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Advertises an IPv4 address range that is provisioned for use with your AWS resources through bring your own IP
     * addresses (BYOIP). It can take a few minutes before traffic to the specified addresses starts routing to AWS
     * because of propagation delays. To see an AWS CLI example of advertising an address range, scroll down to
     * <b>Example</b>.
     * </p>
     * <p>
     * To stop advertising the BYOIP address range, use <a
     * href="https://docs.aws.amazon.com/global-accelerator/latest/api/WithdrawByoipCidr.html"> WithdrawByoipCidr</a>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/global-accelerator/latest/dg/using-byoip.html">Bring Your Own IP Addresses
     * (BYOIP)</a> in the <i>AWS Global Accelerator Developer Guide</i>.
     * </p>
     *
     * @param advertiseByoipCidrRequest
     * @return Result of the AdvertiseByoipCidr operation returned by the service.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws AccessDeniedException
     *         You don't have access permission.
     * @throws ByoipCidrNotFoundException
     *         The CIDR that you specified was not found or is incorrect.
     * @throws IncorrectCidrStateException
     *         The CIDR that you specified is not valid for this action. For example, the state of the CIDR might be
     *         incorrect for this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.AdvertiseByoipCidr
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/AdvertiseByoipCidr"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AdvertiseByoipCidrResponse advertiseByoipCidr(AdvertiseByoipCidrRequest advertiseByoipCidrRequest)
            throws InternalServiceErrorException, InvalidArgumentException, AccessDeniedException, ByoipCidrNotFoundException,
            IncorrectCidrStateException, AwsServiceException, SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AdvertiseByoipCidrResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AdvertiseByoipCidrResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AdvertiseByoipCidrRequest, AdvertiseByoipCidrResponse>()
                .withOperationName("AdvertiseByoipCidr").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(advertiseByoipCidrRequest)
                .withMarshaller(new AdvertiseByoipCidrRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Create an accelerator. An accelerator includes one or more listeners that process inbound connections and direct
     * traffic to one or more endpoint groups, each of which includes endpoints, such as Network Load Balancers. To see
     * an AWS CLI example of creating an accelerator, scroll down to <b>Example</b>.
     * </p>
     * <p>
     * If you bring your own IP address ranges to AWS Global Accelerator (BYOIP), you can assign IP addresses from your
     * own pool to your accelerator as the static IP address entry points. Only one IP address from each of your IP
     * address ranges can be used for each accelerator.
     * </p>
     * <important>
     * <p>
     * You must specify the US West (Oregon) Region to create or update accelerators.
     * </p>
     * </important>
     *
     * @param createAcceleratorRequest
     * @return Result of the CreateAccelerator operation returned by the service.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws LimitExceededException
     *         Processing your request would cause you to exceed an AWS Global Accelerator limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.CreateAccelerator
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/CreateAccelerator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAcceleratorResponse createAccelerator(CreateAcceleratorRequest createAcceleratorRequest)
            throws InternalServiceErrorException, InvalidArgumentException, LimitExceededException, AwsServiceException,
            SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAcceleratorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAcceleratorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateAcceleratorRequest, CreateAcceleratorResponse>()
                .withOperationName("CreateAccelerator").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createAcceleratorRequest)
                .withMarshaller(new CreateAcceleratorRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Create an endpoint group for the specified listener. An endpoint group is a collection of endpoints in one AWS
     * Region. To see an AWS CLI example of creating an endpoint group, scroll down to <b>Example</b>.
     * </p>
     *
     * @param createEndpointGroupRequest
     * @return Result of the CreateEndpointGroup operation returned by the service.
     * @throws AcceleratorNotFoundException
     *         The accelerator that you specified doesn't exist.
     * @throws EndpointGroupAlreadyExistsException
     *         The endpoint group that you specified already exists.
     * @throws ListenerNotFoundException
     *         The listener that you specified doesn't exist.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws LimitExceededException
     *         Processing your request would cause you to exceed an AWS Global Accelerator limit.
     * @throws AccessDeniedException
     *         You don't have access permission.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.CreateEndpointGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/CreateEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateEndpointGroupResponse createEndpointGroup(CreateEndpointGroupRequest createEndpointGroupRequest)
            throws AcceleratorNotFoundException, EndpointGroupAlreadyExistsException, ListenerNotFoundException,
            InternalServiceErrorException, InvalidArgumentException, LimitExceededException, AccessDeniedException,
            AwsServiceException, SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateEndpointGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateEndpointGroupRequest, CreateEndpointGroupResponse>()
                .withOperationName("CreateEndpointGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createEndpointGroupRequest)
                .withMarshaller(new CreateEndpointGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Create a listener to process inbound connections from clients to an accelerator. Connections arrive to assigned
     * static IP addresses on a port, port range, or list of port ranges that you specify. To see an AWS CLI example of
     * creating a listener, scroll down to <b>Example</b>.
     * </p>
     *
     * @param createListenerRequest
     * @return Result of the CreateListener operation returned by the service.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws AcceleratorNotFoundException
     *         The accelerator that you specified doesn't exist.
     * @throws InvalidPortRangeException
     *         The port numbers that you specified are not valid numbers or are not unique for this accelerator.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws LimitExceededException
     *         Processing your request would cause you to exceed an AWS Global Accelerator limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.CreateListener
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/CreateListener"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateListenerResponse createListener(CreateListenerRequest createListenerRequest) throws InvalidArgumentException,
            AcceleratorNotFoundException, InvalidPortRangeException, InternalServiceErrorException, LimitExceededException,
            AwsServiceException, SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateListenerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateListenerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateListenerRequest, CreateListenerResponse>()
                .withOperationName("CreateListener").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createListenerRequest)
                .withMarshaller(new CreateListenerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Delete an accelerator. Before you can delete an accelerator, you must disable it and remove all dependent
     * resources (listeners and endpoint groups). To disable the accelerator, update the accelerator to set
     * <code>Enabled</code> to false.
     * </p>
     * <important>
     * <p>
     * When you create an accelerator, by default, Global Accelerator provides you with a set of two static IP
     * addresses. Alternatively, you can bring your own IP address ranges to Global Accelerator and assign IP addresses
     * from those ranges.
     * </p>
     * <p>
     * The IP addresses are assigned to your accelerator for as long as it exists, even if you disable the accelerator
     * and it no longer accepts or routes traffic. However, when you <i>delete</i> an accelerator, you lose the static
     * IP addresses that are assigned to the accelerator, so you can no longer route traffic by using them. As a best
     * practice, ensure that you have permissions in place to avoid inadvertently deleting accelerators. You can use IAM
     * policies with Global Accelerator to limit the users who have permissions to delete an accelerator. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/global-accelerator/latest/dg/auth-and-access-control.html">Authentication and
     * Access Control</a> in the <i>AWS Global Accelerator Developer Guide</i>.
     * </p>
     * </important>
     *
     * @param deleteAcceleratorRequest
     * @return Result of the DeleteAccelerator operation returned by the service.
     * @throws AcceleratorNotFoundException
     *         The accelerator that you specified doesn't exist.
     * @throws AcceleratorNotDisabledException
     *         The accelerator that you specified could not be disabled.
     * @throws AssociatedListenerFoundException
     *         The accelerator that you specified has a listener associated with it. You must remove all dependent
     *         resources from an accelerator before you can delete it.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.DeleteAccelerator
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/DeleteAccelerator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAcceleratorResponse deleteAccelerator(DeleteAcceleratorRequest deleteAcceleratorRequest)
            throws AcceleratorNotFoundException, AcceleratorNotDisabledException, AssociatedListenerFoundException,
            InternalServiceErrorException, InvalidArgumentException, AwsServiceException, SdkClientException,
            GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAcceleratorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAcceleratorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteAcceleratorRequest, DeleteAcceleratorResponse>()
                .withOperationName("DeleteAccelerator").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteAcceleratorRequest)
                .withMarshaller(new DeleteAcceleratorRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Delete an endpoint group from a listener.
     * </p>
     *
     * @param deleteEndpointGroupRequest
     * @return Result of the DeleteEndpointGroup operation returned by the service.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws EndpointGroupNotFoundException
     *         The endpoint group that you specified doesn't exist.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.DeleteEndpointGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/DeleteEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteEndpointGroupResponse deleteEndpointGroup(DeleteEndpointGroupRequest deleteEndpointGroupRequest)
            throws InvalidArgumentException, EndpointGroupNotFoundException, InternalServiceErrorException, AwsServiceException,
            SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteEndpointGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteEndpointGroupRequest, DeleteEndpointGroupResponse>()
                .withOperationName("DeleteEndpointGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteEndpointGroupRequest)
                .withMarshaller(new DeleteEndpointGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Delete a listener from an accelerator.
     * </p>
     *
     * @param deleteListenerRequest
     * @return Result of the DeleteListener operation returned by the service.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws ListenerNotFoundException
     *         The listener that you specified doesn't exist.
     * @throws AssociatedEndpointGroupFoundException
     *         The listener that you specified has an endpoint group associated with it. You must remove all dependent
     *         resources from a listener before you can delete it.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.DeleteListener
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/DeleteListener"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteListenerResponse deleteListener(DeleteListenerRequest deleteListenerRequest) throws InvalidArgumentException,
            ListenerNotFoundException, AssociatedEndpointGroupFoundException, InternalServiceErrorException, AwsServiceException,
            SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteListenerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteListenerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteListenerRequest, DeleteListenerResponse>()
                .withOperationName("DeleteListener").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteListenerRequest)
                .withMarshaller(new DeleteListenerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Releases the specified address range that you provisioned to use with your AWS resources through bring your own
     * IP addresses (BYOIP) and deletes the corresponding address pool. To see an AWS CLI example of deprovisioning an
     * address range, scroll down to <b>Example</b>.
     * </p>
     * <p>
     * Before you can release an address range, you must stop advertising it by using <a
     * href="https://docs.aws.amazon.com/global-accelerator/latest/api/WithdrawByoipCidr.html">WithdrawByoipCidr</a> and
     * you must not have any accelerators that are using static IP addresses allocated from its address range.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/global-accelerator/latest/dg/using-byoip.html">Bring Your Own IP Addresses
     * (BYOIP)</a> in the <i>AWS Global Accelerator Developer Guide</i>.
     * </p>
     *
     * @param deprovisionByoipCidrRequest
     * @return Result of the DeprovisionByoipCidr operation returned by the service.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws AccessDeniedException
     *         You don't have access permission.
     * @throws ByoipCidrNotFoundException
     *         The CIDR that you specified was not found or is incorrect.
     * @throws IncorrectCidrStateException
     *         The CIDR that you specified is not valid for this action. For example, the state of the CIDR might be
     *         incorrect for this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.DeprovisionByoipCidr
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/DeprovisionByoipCidr"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeprovisionByoipCidrResponse deprovisionByoipCidr(DeprovisionByoipCidrRequest deprovisionByoipCidrRequest)
            throws InternalServiceErrorException, InvalidArgumentException, AccessDeniedException, ByoipCidrNotFoundException,
            IncorrectCidrStateException, AwsServiceException, SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeprovisionByoipCidrResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeprovisionByoipCidrResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeprovisionByoipCidrRequest, DeprovisionByoipCidrResponse>()
                .withOperationName("DeprovisionByoipCidr").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deprovisionByoipCidrRequest)
                .withMarshaller(new DeprovisionByoipCidrRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describe an accelerator. To see an AWS CLI example of describing an accelerator, scroll down to <b>Example</b>.
     * </p>
     *
     * @param describeAcceleratorRequest
     * @return Result of the DescribeAccelerator operation returned by the service.
     * @throws AcceleratorNotFoundException
     *         The accelerator that you specified doesn't exist.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.DescribeAccelerator
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/DescribeAccelerator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAcceleratorResponse describeAccelerator(DescribeAcceleratorRequest describeAcceleratorRequest)
            throws AcceleratorNotFoundException, InternalServiceErrorException, InvalidArgumentException, AwsServiceException,
            SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAcceleratorResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAcceleratorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeAcceleratorRequest, DescribeAcceleratorResponse>()
                .withOperationName("DescribeAccelerator").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeAcceleratorRequest)
                .withMarshaller(new DescribeAcceleratorRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describe the attributes of an accelerator. To see an AWS CLI example of describing the attributes of an
     * accelerator, scroll down to <b>Example</b>.
     * </p>
     *
     * @param describeAcceleratorAttributesRequest
     * @return Result of the DescribeAcceleratorAttributes operation returned by the service.
     * @throws AcceleratorNotFoundException
     *         The accelerator that you specified doesn't exist.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.DescribeAcceleratorAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/DescribeAcceleratorAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAcceleratorAttributesResponse describeAcceleratorAttributes(
            DescribeAcceleratorAttributesRequest describeAcceleratorAttributesRequest) throws AcceleratorNotFoundException,
            InternalServiceErrorException, InvalidArgumentException, AwsServiceException, SdkClientException,
            GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAcceleratorAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAcceleratorAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeAcceleratorAttributesRequest, DescribeAcceleratorAttributesResponse>()
                        .withOperationName("DescribeAcceleratorAttributes").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeAcceleratorAttributesRequest)
                        .withMarshaller(new DescribeAcceleratorAttributesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describe an endpoint group. To see an AWS CLI example of describing an endpoint group, scroll down to
     * <b>Example</b>.
     * </p>
     *
     * @param describeEndpointGroupRequest
     * @return Result of the DescribeEndpointGroup operation returned by the service.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws EndpointGroupNotFoundException
     *         The endpoint group that you specified doesn't exist.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.DescribeEndpointGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/DescribeEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEndpointGroupResponse describeEndpointGroup(DescribeEndpointGroupRequest describeEndpointGroupRequest)
            throws InvalidArgumentException, EndpointGroupNotFoundException, InternalServiceErrorException, AwsServiceException,
            SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEndpointGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeEndpointGroupRequest, DescribeEndpointGroupResponse>()
                .withOperationName("DescribeEndpointGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeEndpointGroupRequest)
                .withMarshaller(new DescribeEndpointGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describe a listener. To see an AWS CLI example of describing a listener, scroll down to <b>Example</b>.
     * </p>
     *
     * @param describeListenerRequest
     * @return Result of the DescribeListener operation returned by the service.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws ListenerNotFoundException
     *         The listener that you specified doesn't exist.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.DescribeListener
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/DescribeListener"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeListenerResponse describeListener(DescribeListenerRequest describeListenerRequest)
            throws InvalidArgumentException, ListenerNotFoundException, InternalServiceErrorException, AwsServiceException,
            SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeListenerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeListenerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeListenerRequest, DescribeListenerResponse>()
                .withOperationName("DescribeListener").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeListenerRequest)
                .withMarshaller(new DescribeListenerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List the accelerators for an AWS account. To see an AWS CLI example of listing the accelerators for an AWS
     * account, scroll down to <b>Example</b>.
     * </p>
     *
     * @param listAcceleratorsRequest
     * @return Result of the ListAccelerators operation returned by the service.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws InvalidNextTokenException
     *         There isn't another item to return.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.ListAccelerators
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/ListAccelerators"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAcceleratorsResponse listAccelerators(ListAcceleratorsRequest listAcceleratorsRequest)
            throws InvalidArgumentException, InvalidNextTokenException, InternalServiceErrorException, AwsServiceException,
            SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAcceleratorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAcceleratorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListAcceleratorsRequest, ListAcceleratorsResponse>()
                .withOperationName("ListAccelerators").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listAcceleratorsRequest)
                .withMarshaller(new ListAcceleratorsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the IP address ranges that were specified in calls to <a
     * href="https://docs.aws.amazon.com/global-accelerator/latest/api/ProvisionByoipCidr.html">ProvisionByoipCidr</a>,
     * including the current state and a history of state changes.
     * </p>
     * <p>
     * To see an AWS CLI example of listing BYOIP CIDR addresses, scroll down to <b>Example</b>.
     * </p>
     *
     * @param listByoipCidrsRequest
     * @return Result of the ListByoipCidrs operation returned by the service.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws AccessDeniedException
     *         You don't have access permission.
     * @throws InvalidNextTokenException
     *         There isn't another item to return.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.ListByoipCidrs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/ListByoipCidrs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListByoipCidrsResponse listByoipCidrs(ListByoipCidrsRequest listByoipCidrsRequest)
            throws InternalServiceErrorException, InvalidArgumentException, AccessDeniedException, InvalidNextTokenException,
            AwsServiceException, SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListByoipCidrsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListByoipCidrsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListByoipCidrsRequest, ListByoipCidrsResponse>()
                .withOperationName("ListByoipCidrs").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listByoipCidrsRequest)
                .withMarshaller(new ListByoipCidrsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List the endpoint groups that are associated with a listener. To see an AWS CLI example of listing the endpoint
     * groups for listener, scroll down to <b>Example</b>.
     * </p>
     *
     * @param listEndpointGroupsRequest
     * @return Result of the ListEndpointGroups operation returned by the service.
     * @throws ListenerNotFoundException
     *         The listener that you specified doesn't exist.
     * @throws InvalidNextTokenException
     *         There isn't another item to return.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.ListEndpointGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/ListEndpointGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEndpointGroupsResponse listEndpointGroups(ListEndpointGroupsRequest listEndpointGroupsRequest)
            throws ListenerNotFoundException, InvalidNextTokenException, InvalidArgumentException, InternalServiceErrorException,
            AwsServiceException, SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEndpointGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListEndpointGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListEndpointGroupsRequest, ListEndpointGroupsResponse>()
                .withOperationName("ListEndpointGroups").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listEndpointGroupsRequest)
                .withMarshaller(new ListEndpointGroupsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List the listeners for an accelerator. To see an AWS CLI example of listing the listeners for an accelerator,
     * scroll down to <b>Example</b>.
     * </p>
     *
     * @param listListenersRequest
     * @return Result of the ListListeners operation returned by the service.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws AcceleratorNotFoundException
     *         The accelerator that you specified doesn't exist.
     * @throws InvalidNextTokenException
     *         There isn't another item to return.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.ListListeners
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/ListListeners"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListListenersResponse listListeners(ListListenersRequest listListenersRequest) throws InvalidArgumentException,
            AcceleratorNotFoundException, InvalidNextTokenException, InternalServiceErrorException, AwsServiceException,
            SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListListenersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListListenersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListListenersRequest, ListListenersResponse>()
                .withOperationName("ListListeners").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listListenersRequest)
                .withMarshaller(new ListListenersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List all tags for an accelerator. To see an AWS CLI example of listing tags for an accelerator, scroll down to
     * <b>Example</b>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/global-accelerator/latest/dg/tagging-in-global-accelerator.html">Tagging in AWS
     * Global Accelerator</a> in the <i>AWS Global Accelerator Developer Guide</i>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws AcceleratorNotFoundException
     *         The accelerator that you specified doesn't exist.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws AcceleratorNotFoundException, InternalServiceErrorException, InvalidArgumentException, AwsServiceException,
            SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provisions an IP address range to use with your AWS resources through bring your own IP addresses (BYOIP) and
     * creates a corresponding address pool. After the address range is provisioned, it is ready to be advertised using
     * <a href="https://docs.aws.amazon.com/global-accelerator/latest/api/AdvertiseByoipCidr.html">
     * AdvertiseByoipCidr</a>.
     * </p>
     * <p>
     * To see an AWS CLI example of provisioning an address range for BYOIP, scroll down to <b>Example</b>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/global-accelerator/latest/dg/using-byoip.html">Bring Your Own IP Addresses
     * (BYOIP)</a> in the <i>AWS Global Accelerator Developer Guide</i>.
     * </p>
     *
     * @param provisionByoipCidrRequest
     * @return Result of the ProvisionByoipCidr operation returned by the service.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws LimitExceededException
     *         Processing your request would cause you to exceed an AWS Global Accelerator limit.
     * @throws AccessDeniedException
     *         You don't have access permission.
     * @throws IncorrectCidrStateException
     *         The CIDR that you specified is not valid for this action. For example, the state of the CIDR might be
     *         incorrect for this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.ProvisionByoipCidr
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/ProvisionByoipCidr"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ProvisionByoipCidrResponse provisionByoipCidr(ProvisionByoipCidrRequest provisionByoipCidrRequest)
            throws InternalServiceErrorException, InvalidArgumentException, LimitExceededException, AccessDeniedException,
            IncorrectCidrStateException, AwsServiceException, SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ProvisionByoipCidrResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ProvisionByoipCidrResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ProvisionByoipCidrRequest, ProvisionByoipCidrResponse>()
                .withOperationName("ProvisionByoipCidr").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(provisionByoipCidrRequest)
                .withMarshaller(new ProvisionByoipCidrRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Add tags to an accelerator resource. To see an AWS CLI example of adding tags to an accelerator, scroll down to
     * <b>Example</b>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/global-accelerator/latest/dg/tagging-in-global-accelerator.html">Tagging in AWS
     * Global Accelerator</a> in the <i>AWS Global Accelerator Developer Guide</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws AcceleratorNotFoundException
     *         The accelerator that you specified doesn't exist.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws AcceleratorNotFoundException,
            InternalServiceErrorException, InvalidArgumentException, AwsServiceException, SdkClientException,
            GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                .withOperationName("TagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Remove tags from a Global Accelerator resource. When you specify a tag key, the action removes both that key and
     * its associated value. To see an AWS CLI example of removing tags from an accelerator, scroll down to
     * <b>Example</b>. The operation succeeds even if you attempt to remove tags from an accelerator that was already
     * removed.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/global-accelerator/latest/dg/tagging-in-global-accelerator.html">Tagging in AWS
     * Global Accelerator</a> in the <i>AWS Global Accelerator Developer Guide</i>.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws AcceleratorNotFoundException
     *         The accelerator that you specified doesn't exist.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws AcceleratorNotFoundException,
            InternalServiceErrorException, InvalidArgumentException, AwsServiceException, SdkClientException,
            GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                .withOperationName("UntagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Update an accelerator. To see an AWS CLI example of updating an accelerator, scroll down to <b>Example</b>.
     * </p>
     * <important>
     * <p>
     * You must specify the US West (Oregon) Region to create or update accelerators.
     * </p>
     * </important>
     *
     * @param updateAcceleratorRequest
     * @return Result of the UpdateAccelerator operation returned by the service.
     * @throws AcceleratorNotFoundException
     *         The accelerator that you specified doesn't exist.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.UpdateAccelerator
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/UpdateAccelerator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateAcceleratorResponse updateAccelerator(UpdateAcceleratorRequest updateAcceleratorRequest)
            throws AcceleratorNotFoundException, InternalServiceErrorException, InvalidArgumentException, AwsServiceException,
            SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAcceleratorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateAcceleratorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateAcceleratorRequest, UpdateAcceleratorResponse>()
                .withOperationName("UpdateAccelerator").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateAcceleratorRequest)
                .withMarshaller(new UpdateAcceleratorRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Update the attributes for an accelerator. To see an AWS CLI example of updating an accelerator to enable flow
     * logs, scroll down to <b>Example</b>.
     * </p>
     *
     * @param updateAcceleratorAttributesRequest
     * @return Result of the UpdateAcceleratorAttributes operation returned by the service.
     * @throws AcceleratorNotFoundException
     *         The accelerator that you specified doesn't exist.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws AccessDeniedException
     *         You don't have access permission.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.UpdateAcceleratorAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/UpdateAcceleratorAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateAcceleratorAttributesResponse updateAcceleratorAttributes(
            UpdateAcceleratorAttributesRequest updateAcceleratorAttributesRequest) throws AcceleratorNotFoundException,
            InternalServiceErrorException, InvalidArgumentException, AccessDeniedException, AwsServiceException,
            SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAcceleratorAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateAcceleratorAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpdateAcceleratorAttributesRequest, UpdateAcceleratorAttributesResponse>()
                        .withOperationName("UpdateAcceleratorAttributes").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(updateAcceleratorAttributesRequest)
                        .withMarshaller(new UpdateAcceleratorAttributesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Update an endpoint group. To see an AWS CLI example of updating an endpoint group, scroll down to <b>Example</b>.
     * </p>
     *
     * @param updateEndpointGroupRequest
     * @return Result of the UpdateEndpointGroup operation returned by the service.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws EndpointGroupNotFoundException
     *         The endpoint group that you specified doesn't exist.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws LimitExceededException
     *         Processing your request would cause you to exceed an AWS Global Accelerator limit.
     * @throws AccessDeniedException
     *         You don't have access permission.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.UpdateEndpointGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/UpdateEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateEndpointGroupResponse updateEndpointGroup(UpdateEndpointGroupRequest updateEndpointGroupRequest)
            throws InvalidArgumentException, EndpointGroupNotFoundException, InternalServiceErrorException,
            LimitExceededException, AccessDeniedException, AwsServiceException, SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateEndpointGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateEndpointGroupRequest, UpdateEndpointGroupResponse>()
                .withOperationName("UpdateEndpointGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateEndpointGroupRequest)
                .withMarshaller(new UpdateEndpointGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Update a listener. To see an AWS CLI example of updating listener, scroll down to <b>Example</b>.
     * </p>
     *
     * @param updateListenerRequest
     * @return Result of the UpdateListener operation returned by the service.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws InvalidPortRangeException
     *         The port numbers that you specified are not valid numbers or are not unique for this accelerator.
     * @throws ListenerNotFoundException
     *         The listener that you specified doesn't exist.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws LimitExceededException
     *         Processing your request would cause you to exceed an AWS Global Accelerator limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.UpdateListener
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/UpdateListener"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateListenerResponse updateListener(UpdateListenerRequest updateListenerRequest) throws InvalidArgumentException,
            InvalidPortRangeException, ListenerNotFoundException, InternalServiceErrorException, LimitExceededException,
            AwsServiceException, SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateListenerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateListenerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateListenerRequest, UpdateListenerResponse>()
                .withOperationName("UpdateListener").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateListenerRequest)
                .withMarshaller(new UpdateListenerRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Stops advertising an address range that is provisioned as an address pool. You can perform this operation at most
     * once every 10 seconds, even if you specify different address ranges each time. To see an AWS CLI example of
     * withdrawing an address range for BYOIP so it will no longer be advertised by AWS, scroll down to <b>Example</b>.
     * </p>
     * <p>
     * It can take a few minutes before traffic to the specified addresses stops routing to AWS because of propagation
     * delays.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/global-accelerator/latest/dg/using-byoip.html">Bring Your Own IP Addresses
     * (BYOIP)</a> in the <i>AWS Global Accelerator Developer Guide</i>.
     * </p>
     *
     * @param withdrawByoipCidrRequest
     * @return Result of the WithdrawByoipCidr operation returned by the service.
     * @throws InternalServiceErrorException
     *         There was an internal error for AWS Global Accelerator.
     * @throws InvalidArgumentException
     *         An argument that you specified is invalid.
     * @throws AccessDeniedException
     *         You don't have access permission.
     * @throws ByoipCidrNotFoundException
     *         The CIDR that you specified was not found or is incorrect.
     * @throws IncorrectCidrStateException
     *         The CIDR that you specified is not valid for this action. For example, the state of the CIDR might be
     *         incorrect for this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GlobalAcceleratorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GlobalAcceleratorClient.WithdrawByoipCidr
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/WithdrawByoipCidr"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public WithdrawByoipCidrResponse withdrawByoipCidr(WithdrawByoipCidrRequest withdrawByoipCidrRequest)
            throws InternalServiceErrorException, InvalidArgumentException, AccessDeniedException, ByoipCidrNotFoundException,
            IncorrectCidrStateException, AwsServiceException, SdkClientException, GlobalAcceleratorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<WithdrawByoipCidrResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                WithdrawByoipCidrResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<WithdrawByoipCidrRequest, WithdrawByoipCidrResponse>()
                .withOperationName("WithdrawByoipCidr").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(withdrawByoipCidrRequest)
                .withMarshaller(new WithdrawByoipCidrRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(GlobalAcceleratorException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ListenerNotFoundException")
                                .exceptionBuilderSupplier(ListenerNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AcceleratorNotFoundException")
                                .exceptionBuilderSupplier(AcceleratorNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IncorrectCidrStateException")
                                .exceptionBuilderSupplier(IncorrectCidrStateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AssociatedListenerFoundException")
                                .exceptionBuilderSupplier(AssociatedListenerFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ByoipCidrNotFoundException")
                                .exceptionBuilderSupplier(ByoipCidrNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AssociatedEndpointGroupFoundException")
                                .exceptionBuilderSupplier(AssociatedEndpointGroupFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EndpointGroupNotFoundException")
                                .exceptionBuilderSupplier(EndpointGroupNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AcceleratorNotDisabledException")
                                .exceptionBuilderSupplier(AcceleratorNotDisabledException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EndpointGroupAlreadyExistsException")
                                .exceptionBuilderSupplier(EndpointGroupAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPortRangeException")
                                .exceptionBuilderSupplier(InvalidPortRangeException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
