/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for the Amazon FSx for Lustre file system.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LustreFileSystemConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<LustreFileSystemConfiguration.Builder, LustreFileSystemConfiguration> {
    private static final SdkField<String> WEEKLY_MAINTENANCE_START_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("WeeklyMaintenanceStartTime")
            .getter(getter(LustreFileSystemConfiguration::weeklyMaintenanceStartTime))
            .setter(setter(Builder::weeklyMaintenanceStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WeeklyMaintenanceStartTime").build())
            .build();

    private static final SdkField<DataRepositoryConfiguration> DATA_REPOSITORY_CONFIGURATION_FIELD = SdkField
            .<DataRepositoryConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("DataRepositoryConfiguration")
            .getter(getter(LustreFileSystemConfiguration::dataRepositoryConfiguration))
            .setter(setter(Builder::dataRepositoryConfiguration))
            .constructor(DataRepositoryConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataRepositoryConfiguration")
                    .build()).build();

    private static final SdkField<String> DEPLOYMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeploymentType").getter(getter(LustreFileSystemConfiguration::deploymentTypeAsString))
            .setter(setter(Builder::deploymentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeploymentType").build()).build();

    private static final SdkField<Integer> PER_UNIT_STORAGE_THROUGHPUT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("PerUnitStorageThroughput")
            .getter(getter(LustreFileSystemConfiguration::perUnitStorageThroughput))
            .setter(setter(Builder::perUnitStorageThroughput))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PerUnitStorageThroughput").build())
            .build();

    private static final SdkField<String> MOUNT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MountName").getter(getter(LustreFileSystemConfiguration::mountName)).setter(setter(Builder::mountName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MountName").build()).build();

    private static final SdkField<String> DAILY_AUTOMATIC_BACKUP_START_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DailyAutomaticBackupStartTime")
            .getter(getter(LustreFileSystemConfiguration::dailyAutomaticBackupStartTime))
            .setter(setter(Builder::dailyAutomaticBackupStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DailyAutomaticBackupStartTime")
                    .build()).build();

    private static final SdkField<Integer> AUTOMATIC_BACKUP_RETENTION_DAYS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("AutomaticBackupRetentionDays")
            .getter(getter(LustreFileSystemConfiguration::automaticBackupRetentionDays))
            .setter(setter(Builder::automaticBackupRetentionDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutomaticBackupRetentionDays")
                    .build()).build();

    private static final SdkField<Boolean> COPY_TAGS_TO_BACKUPS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CopyTagsToBackups").getter(getter(LustreFileSystemConfiguration::copyTagsToBackups))
            .setter(setter(Builder::copyTagsToBackups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyTagsToBackups").build()).build();

    private static final SdkField<String> DRIVE_CACHE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DriveCacheType").getter(getter(LustreFileSystemConfiguration::driveCacheTypeAsString))
            .setter(setter(Builder::driveCacheType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DriveCacheType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            WEEKLY_MAINTENANCE_START_TIME_FIELD, DATA_REPOSITORY_CONFIGURATION_FIELD, DEPLOYMENT_TYPE_FIELD,
            PER_UNIT_STORAGE_THROUGHPUT_FIELD, MOUNT_NAME_FIELD, DAILY_AUTOMATIC_BACKUP_START_TIME_FIELD,
            AUTOMATIC_BACKUP_RETENTION_DAYS_FIELD, COPY_TAGS_TO_BACKUPS_FIELD, DRIVE_CACHE_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String weeklyMaintenanceStartTime;

    private final DataRepositoryConfiguration dataRepositoryConfiguration;

    private final String deploymentType;

    private final Integer perUnitStorageThroughput;

    private final String mountName;

    private final String dailyAutomaticBackupStartTime;

    private final Integer automaticBackupRetentionDays;

    private final Boolean copyTagsToBackups;

    private final String driveCacheType;

    private LustreFileSystemConfiguration(BuilderImpl builder) {
        this.weeklyMaintenanceStartTime = builder.weeklyMaintenanceStartTime;
        this.dataRepositoryConfiguration = builder.dataRepositoryConfiguration;
        this.deploymentType = builder.deploymentType;
        this.perUnitStorageThroughput = builder.perUnitStorageThroughput;
        this.mountName = builder.mountName;
        this.dailyAutomaticBackupStartTime = builder.dailyAutomaticBackupStartTime;
        this.automaticBackupRetentionDays = builder.automaticBackupRetentionDays;
        this.copyTagsToBackups = builder.copyTagsToBackups;
        this.driveCacheType = builder.driveCacheType;
    }

    /**
     * <p>
     * The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone. d is the weekday
     * number, from 1 through 7, beginning with Monday and ending with Sunday.
     * </p>
     * 
     * @return The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone. d is the
     *         weekday number, from 1 through 7, beginning with Monday and ending with Sunday.
     */
    public final String weeklyMaintenanceStartTime() {
        return weeklyMaintenanceStartTime;
    }

    /**
     * Returns the value of the DataRepositoryConfiguration property for this object.
     * 
     * @return The value of the DataRepositoryConfiguration property for this object.
     */
    public final DataRepositoryConfiguration dataRepositoryConfiguration() {
        return dataRepositoryConfiguration;
    }

    /**
     * <p>
     * The deployment type of the FSX for Lustre file system. <i>Scratch deployment type</i> is designed for temporary
     * storage and shorter-term processing of data.
     * </p>
     * <p>
     * <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need temporary
     * storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type provides in-transit
     * encryption of data and higher burst throughput capacity than <code>SCRATCH_1</code>.
     * </p>
     * <p>
     * The <code>PERSISTENT_1</code> deployment type is used for longer-term storage and workloads and encryption of
     * data in transit. To learn more about deployment types, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for Lustre Deployment
     * Options</a>. (Default = <code>SCRATCH_1</code>)
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deploymentType}
     * will return {@link LustreDeploymentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #deploymentTypeAsString}.
     * </p>
     * 
     * @return The deployment type of the FSX for Lustre file system. <i>Scratch deployment type</i> is designed for
     *         temporary storage and shorter-term processing of data.</p>
     *         <p>
     *         <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need
     *         temporary storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type
     *         provides in-transit encryption of data and higher burst throughput capacity than <code>SCRATCH_1</code>.
     *         </p>
     *         <p>
     *         The <code>PERSISTENT_1</code> deployment type is used for longer-term storage and workloads and
     *         encryption of data in transit. To learn more about deployment types, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for Lustre
     *         Deployment Options</a>. (Default = <code>SCRATCH_1</code>)
     * @see LustreDeploymentType
     */
    public final LustreDeploymentType deploymentType() {
        return LustreDeploymentType.fromValue(deploymentType);
    }

    /**
     * <p>
     * The deployment type of the FSX for Lustre file system. <i>Scratch deployment type</i> is designed for temporary
     * storage and shorter-term processing of data.
     * </p>
     * <p>
     * <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need temporary
     * storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type provides in-transit
     * encryption of data and higher burst throughput capacity than <code>SCRATCH_1</code>.
     * </p>
     * <p>
     * The <code>PERSISTENT_1</code> deployment type is used for longer-term storage and workloads and encryption of
     * data in transit. To learn more about deployment types, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for Lustre Deployment
     * Options</a>. (Default = <code>SCRATCH_1</code>)
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deploymentType}
     * will return {@link LustreDeploymentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #deploymentTypeAsString}.
     * </p>
     * 
     * @return The deployment type of the FSX for Lustre file system. <i>Scratch deployment type</i> is designed for
     *         temporary storage and shorter-term processing of data.</p>
     *         <p>
     *         <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need
     *         temporary storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type
     *         provides in-transit encryption of data and higher burst throughput capacity than <code>SCRATCH_1</code>.
     *         </p>
     *         <p>
     *         The <code>PERSISTENT_1</code> deployment type is used for longer-term storage and workloads and
     *         encryption of data in transit. To learn more about deployment types, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for Lustre
     *         Deployment Options</a>. (Default = <code>SCRATCH_1</code>)
     * @see LustreDeploymentType
     */
    public final String deploymentTypeAsString() {
        return deploymentType;
    }

    /**
     * <p>
     * Per unit storage throughput represents the megabytes per second of read or write throughput per 1 tebibyte of
     * storage provisioned. File system throughput capacity is equal to Storage capacity (TiB) *
     * PerUnitStorageThroughput (MB/s/TiB). This option is only valid for <code>PERSISTENT_1</code> deployment types.
     * </p>
     * <p>
     * Valid values for SSD storage: 50, 100, 200. Valid values for HDD storage: 12, 40.
     * </p>
     * 
     * @return Per unit storage throughput represents the megabytes per second of read or write throughput per 1
     *         tebibyte of storage provisioned. File system throughput capacity is equal to Storage capacity (TiB) *
     *         PerUnitStorageThroughput (MB/s/TiB). This option is only valid for <code>PERSISTENT_1</code> deployment
     *         types. </p>
     *         <p>
     *         Valid values for SSD storage: 50, 100, 200. Valid values for HDD storage: 12, 40.
     */
    public final Integer perUnitStorageThroughput() {
        return perUnitStorageThroughput;
    }

    /**
     * <p>
     * You use the <code>MountName</code> value when mounting the file system.
     * </p>
     * <p>
     * For the <code>SCRATCH_1</code> deployment type, this value is always "<code>fsx</code>". For
     * <code>SCRATCH_2</code> and <code>PERSISTENT_1</code> deployment types, this value is a string that is unique
     * within an AWS Region.
     * </p>
     * 
     * @return You use the <code>MountName</code> value when mounting the file system.</p>
     *         <p>
     *         For the <code>SCRATCH_1</code> deployment type, this value is always "<code>fsx</code>". For
     *         <code>SCRATCH_2</code> and <code>PERSISTENT_1</code> deployment types, this value is a string that is
     *         unique within an AWS Region.
     */
    public final String mountName() {
        return mountName;
    }

    /**
     * Returns the value of the DailyAutomaticBackupStartTime property for this object.
     * 
     * @return The value of the DailyAutomaticBackupStartTime property for this object.
     */
    public final String dailyAutomaticBackupStartTime() {
        return dailyAutomaticBackupStartTime;
    }

    /**
     * Returns the value of the AutomaticBackupRetentionDays property for this object.
     * 
     * @return The value of the AutomaticBackupRetentionDays property for this object.
     */
    public final Integer automaticBackupRetentionDays() {
        return automaticBackupRetentionDays;
    }

    /**
     * <p>
     * A boolean flag indicating whether tags on the file system should be copied to backups. If it's set to true, all
     * tags on the file system are copied to all automatic backups and any user-initiated backups where the user doesn't
     * specify any tags. If this value is true, and you specify one or more tags, only the specified tags are copied to
     * backups. If you specify one or more tags when creating a user-initiated backup, no tags are copied from the file
     * system, regardless of this value. (Default = false)
     * </p>
     * 
     * @return A boolean flag indicating whether tags on the file system should be copied to backups. If it's set to
     *         true, all tags on the file system are copied to all automatic backups and any user-initiated backups
     *         where the user doesn't specify any tags. If this value is true, and you specify one or more tags, only
     *         the specified tags are copied to backups. If you specify one or more tags when creating a user-initiated
     *         backup, no tags are copied from the file system, regardless of this value. (Default = false)
     */
    public final Boolean copyTagsToBackups() {
        return copyTagsToBackups;
    }

    /**
     * <p>
     * The type of drive cache used by PERSISTENT_1 file systems that are provisioned with HDD storage devices. This
     * parameter is required when storage type is HDD. Set to <code>READ</code>, improve the performance for frequently
     * accessed files and allows 20% of the total storage capacity of the file system to be cached.
     * </p>
     * <p>
     * This parameter is required when <code>StorageType</code> is set to HDD.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #driveCacheType}
     * will return {@link DriveCacheType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #driveCacheTypeAsString}.
     * </p>
     * 
     * @return The type of drive cache used by PERSISTENT_1 file systems that are provisioned with HDD storage devices.
     *         This parameter is required when storage type is HDD. Set to <code>READ</code>, improve the performance
     *         for frequently accessed files and allows 20% of the total storage capacity of the file system to be
     *         cached. </p>
     *         <p>
     *         This parameter is required when <code>StorageType</code> is set to HDD.
     * @see DriveCacheType
     */
    public final DriveCacheType driveCacheType() {
        return DriveCacheType.fromValue(driveCacheType);
    }

    /**
     * <p>
     * The type of drive cache used by PERSISTENT_1 file systems that are provisioned with HDD storage devices. This
     * parameter is required when storage type is HDD. Set to <code>READ</code>, improve the performance for frequently
     * accessed files and allows 20% of the total storage capacity of the file system to be cached.
     * </p>
     * <p>
     * This parameter is required when <code>StorageType</code> is set to HDD.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #driveCacheType}
     * will return {@link DriveCacheType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #driveCacheTypeAsString}.
     * </p>
     * 
     * @return The type of drive cache used by PERSISTENT_1 file systems that are provisioned with HDD storage devices.
     *         This parameter is required when storage type is HDD. Set to <code>READ</code>, improve the performance
     *         for frequently accessed files and allows 20% of the total storage capacity of the file system to be
     *         cached. </p>
     *         <p>
     *         This parameter is required when <code>StorageType</code> is set to HDD.
     * @see DriveCacheType
     */
    public final String driveCacheTypeAsString() {
        return driveCacheType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(weeklyMaintenanceStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(dataRepositoryConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(deploymentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(perUnitStorageThroughput());
        hashCode = 31 * hashCode + Objects.hashCode(mountName());
        hashCode = 31 * hashCode + Objects.hashCode(dailyAutomaticBackupStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(automaticBackupRetentionDays());
        hashCode = 31 * hashCode + Objects.hashCode(copyTagsToBackups());
        hashCode = 31 * hashCode + Objects.hashCode(driveCacheTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LustreFileSystemConfiguration)) {
            return false;
        }
        LustreFileSystemConfiguration other = (LustreFileSystemConfiguration) obj;
        return Objects.equals(weeklyMaintenanceStartTime(), other.weeklyMaintenanceStartTime())
                && Objects.equals(dataRepositoryConfiguration(), other.dataRepositoryConfiguration())
                && Objects.equals(deploymentTypeAsString(), other.deploymentTypeAsString())
                && Objects.equals(perUnitStorageThroughput(), other.perUnitStorageThroughput())
                && Objects.equals(mountName(), other.mountName())
                && Objects.equals(dailyAutomaticBackupStartTime(), other.dailyAutomaticBackupStartTime())
                && Objects.equals(automaticBackupRetentionDays(), other.automaticBackupRetentionDays())
                && Objects.equals(copyTagsToBackups(), other.copyTagsToBackups())
                && Objects.equals(driveCacheTypeAsString(), other.driveCacheTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LustreFileSystemConfiguration").add("WeeklyMaintenanceStartTime", weeklyMaintenanceStartTime())
                .add("DataRepositoryConfiguration", dataRepositoryConfiguration())
                .add("DeploymentType", deploymentTypeAsString()).add("PerUnitStorageThroughput", perUnitStorageThroughput())
                .add("MountName", mountName()).add("DailyAutomaticBackupStartTime", dailyAutomaticBackupStartTime())
                .add("AutomaticBackupRetentionDays", automaticBackupRetentionDays())
                .add("CopyTagsToBackups", copyTagsToBackups()).add("DriveCacheType", driveCacheTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "WeeklyMaintenanceStartTime":
            return Optional.ofNullable(clazz.cast(weeklyMaintenanceStartTime()));
        case "DataRepositoryConfiguration":
            return Optional.ofNullable(clazz.cast(dataRepositoryConfiguration()));
        case "DeploymentType":
            return Optional.ofNullable(clazz.cast(deploymentTypeAsString()));
        case "PerUnitStorageThroughput":
            return Optional.ofNullable(clazz.cast(perUnitStorageThroughput()));
        case "MountName":
            return Optional.ofNullable(clazz.cast(mountName()));
        case "DailyAutomaticBackupStartTime":
            return Optional.ofNullable(clazz.cast(dailyAutomaticBackupStartTime()));
        case "AutomaticBackupRetentionDays":
            return Optional.ofNullable(clazz.cast(automaticBackupRetentionDays()));
        case "CopyTagsToBackups":
            return Optional.ofNullable(clazz.cast(copyTagsToBackups()));
        case "DriveCacheType":
            return Optional.ofNullable(clazz.cast(driveCacheTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LustreFileSystemConfiguration, T> g) {
        return obj -> g.apply((LustreFileSystemConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LustreFileSystemConfiguration> {
        /**
         * <p>
         * The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone. d is the
         * weekday number, from 1 through 7, beginning with Monday and ending with Sunday.
         * </p>
         * 
         * @param weeklyMaintenanceStartTime
         *        The preferred start time to perform weekly maintenance, formatted d:HH:MM in the UTC time zone. d is
         *        the weekday number, from 1 through 7, beginning with Monday and ending with Sunday.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weeklyMaintenanceStartTime(String weeklyMaintenanceStartTime);

        /**
         * Sets the value of the DataRepositoryConfiguration property for this object.
         *
         * @param dataRepositoryConfiguration
         *        The new value for the DataRepositoryConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataRepositoryConfiguration(DataRepositoryConfiguration dataRepositoryConfiguration);

        /**
         * Sets the value of the DataRepositoryConfiguration property for this object.
         *
         * This is a convenience that creates an instance of the {@link DataRepositoryConfiguration.Builder} avoiding
         * the need to create one manually via {@link DataRepositoryConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link DataRepositoryConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #dataRepositoryConfiguration(DataRepositoryConfiguration)}.
         * 
         * @param dataRepositoryConfiguration
         *        a consumer that will call methods on {@link DataRepositoryConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataRepositoryConfiguration(DataRepositoryConfiguration)
         */
        default Builder dataRepositoryConfiguration(Consumer<DataRepositoryConfiguration.Builder> dataRepositoryConfiguration) {
            return dataRepositoryConfiguration(DataRepositoryConfiguration.builder().applyMutation(dataRepositoryConfiguration)
                    .build());
        }

        /**
         * <p>
         * The deployment type of the FSX for Lustre file system. <i>Scratch deployment type</i> is designed for
         * temporary storage and shorter-term processing of data.
         * </p>
         * <p>
         * <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need
         * temporary storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type provides
         * in-transit encryption of data and higher burst throughput capacity than <code>SCRATCH_1</code>.
         * </p>
         * <p>
         * The <code>PERSISTENT_1</code> deployment type is used for longer-term storage and workloads and encryption of
         * data in transit. To learn more about deployment types, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for Lustre
         * Deployment Options</a>. (Default = <code>SCRATCH_1</code>)
         * </p>
         * 
         * @param deploymentType
         *        The deployment type of the FSX for Lustre file system. <i>Scratch deployment type</i> is designed for
         *        temporary storage and shorter-term processing of data.</p>
         *        <p>
         *        <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need
         *        temporary storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type
         *        provides in-transit encryption of data and higher burst throughput capacity than
         *        <code>SCRATCH_1</code>.
         *        </p>
         *        <p>
         *        The <code>PERSISTENT_1</code> deployment type is used for longer-term storage and workloads and
         *        encryption of data in transit. To learn more about deployment types, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for Lustre
         *        Deployment Options</a>. (Default = <code>SCRATCH_1</code>)
         * @see LustreDeploymentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LustreDeploymentType
         */
        Builder deploymentType(String deploymentType);

        /**
         * <p>
         * The deployment type of the FSX for Lustre file system. <i>Scratch deployment type</i> is designed for
         * temporary storage and shorter-term processing of data.
         * </p>
         * <p>
         * <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need
         * temporary storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type provides
         * in-transit encryption of data and higher burst throughput capacity than <code>SCRATCH_1</code>.
         * </p>
         * <p>
         * The <code>PERSISTENT_1</code> deployment type is used for longer-term storage and workloads and encryption of
         * data in transit. To learn more about deployment types, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for Lustre
         * Deployment Options</a>. (Default = <code>SCRATCH_1</code>)
         * </p>
         * 
         * @param deploymentType
         *        The deployment type of the FSX for Lustre file system. <i>Scratch deployment type</i> is designed for
         *        temporary storage and shorter-term processing of data.</p>
         *        <p>
         *        <code>SCRATCH_1</code> and <code>SCRATCH_2</code> deployment types are best suited for when you need
         *        temporary storage and shorter-term processing of data. The <code>SCRATCH_2</code> deployment type
         *        provides in-transit encryption of data and higher burst throughput capacity than
         *        <code>SCRATCH_1</code>.
         *        </p>
         *        <p>
         *        The <code>PERSISTENT_1</code> deployment type is used for longer-term storage and workloads and
         *        encryption of data in transit. To learn more about deployment types, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/lustre-deployment-types.html"> FSx for Lustre
         *        Deployment Options</a>. (Default = <code>SCRATCH_1</code>)
         * @see LustreDeploymentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LustreDeploymentType
         */
        Builder deploymentType(LustreDeploymentType deploymentType);

        /**
         * <p>
         * Per unit storage throughput represents the megabytes per second of read or write throughput per 1 tebibyte of
         * storage provisioned. File system throughput capacity is equal to Storage capacity (TiB) *
         * PerUnitStorageThroughput (MB/s/TiB). This option is only valid for <code>PERSISTENT_1</code> deployment
         * types.
         * </p>
         * <p>
         * Valid values for SSD storage: 50, 100, 200. Valid values for HDD storage: 12, 40.
         * </p>
         * 
         * @param perUnitStorageThroughput
         *        Per unit storage throughput represents the megabytes per second of read or write throughput per 1
         *        tebibyte of storage provisioned. File system throughput capacity is equal to Storage capacity (TiB) *
         *        PerUnitStorageThroughput (MB/s/TiB). This option is only valid for <code>PERSISTENT_1</code>
         *        deployment types. </p>
         *        <p>
         *        Valid values for SSD storage: 50, 100, 200. Valid values for HDD storage: 12, 40.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder perUnitStorageThroughput(Integer perUnitStorageThroughput);

        /**
         * <p>
         * You use the <code>MountName</code> value when mounting the file system.
         * </p>
         * <p>
         * For the <code>SCRATCH_1</code> deployment type, this value is always "<code>fsx</code>". For
         * <code>SCRATCH_2</code> and <code>PERSISTENT_1</code> deployment types, this value is a string that is unique
         * within an AWS Region.
         * </p>
         * 
         * @param mountName
         *        You use the <code>MountName</code> value when mounting the file system.</p>
         *        <p>
         *        For the <code>SCRATCH_1</code> deployment type, this value is always "<code>fsx</code>". For
         *        <code>SCRATCH_2</code> and <code>PERSISTENT_1</code> deployment types, this value is a string that is
         *        unique within an AWS Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountName(String mountName);

        /**
         * Sets the value of the DailyAutomaticBackupStartTime property for this object.
         *
         * @param dailyAutomaticBackupStartTime
         *        The new value for the DailyAutomaticBackupStartTime property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime);

        /**
         * Sets the value of the AutomaticBackupRetentionDays property for this object.
         *
         * @param automaticBackupRetentionDays
         *        The new value for the AutomaticBackupRetentionDays property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder automaticBackupRetentionDays(Integer automaticBackupRetentionDays);

        /**
         * <p>
         * A boolean flag indicating whether tags on the file system should be copied to backups. If it's set to true,
         * all tags on the file system are copied to all automatic backups and any user-initiated backups where the user
         * doesn't specify any tags. If this value is true, and you specify one or more tags, only the specified tags
         * are copied to backups. If you specify one or more tags when creating a user-initiated backup, no tags are
         * copied from the file system, regardless of this value. (Default = false)
         * </p>
         * 
         * @param copyTagsToBackups
         *        A boolean flag indicating whether tags on the file system should be copied to backups. If it's set to
         *        true, all tags on the file system are copied to all automatic backups and any user-initiated backups
         *        where the user doesn't specify any tags. If this value is true, and you specify one or more tags, only
         *        the specified tags are copied to backups. If you specify one or more tags when creating a
         *        user-initiated backup, no tags are copied from the file system, regardless of this value. (Default =
         *        false)
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyTagsToBackups(Boolean copyTagsToBackups);

        /**
         * <p>
         * The type of drive cache used by PERSISTENT_1 file systems that are provisioned with HDD storage devices. This
         * parameter is required when storage type is HDD. Set to <code>READ</code>, improve the performance for
         * frequently accessed files and allows 20% of the total storage capacity of the file system to be cached.
         * </p>
         * <p>
         * This parameter is required when <code>StorageType</code> is set to HDD.
         * </p>
         * 
         * @param driveCacheType
         *        The type of drive cache used by PERSISTENT_1 file systems that are provisioned with HDD storage
         *        devices. This parameter is required when storage type is HDD. Set to <code>READ</code>, improve the
         *        performance for frequently accessed files and allows 20% of the total storage capacity of the file
         *        system to be cached. </p>
         *        <p>
         *        This parameter is required when <code>StorageType</code> is set to HDD.
         * @see DriveCacheType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DriveCacheType
         */
        Builder driveCacheType(String driveCacheType);

        /**
         * <p>
         * The type of drive cache used by PERSISTENT_1 file systems that are provisioned with HDD storage devices. This
         * parameter is required when storage type is HDD. Set to <code>READ</code>, improve the performance for
         * frequently accessed files and allows 20% of the total storage capacity of the file system to be cached.
         * </p>
         * <p>
         * This parameter is required when <code>StorageType</code> is set to HDD.
         * </p>
         * 
         * @param driveCacheType
         *        The type of drive cache used by PERSISTENT_1 file systems that are provisioned with HDD storage
         *        devices. This parameter is required when storage type is HDD. Set to <code>READ</code>, improve the
         *        performance for frequently accessed files and allows 20% of the total storage capacity of the file
         *        system to be cached. </p>
         *        <p>
         *        This parameter is required when <code>StorageType</code> is set to HDD.
         * @see DriveCacheType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DriveCacheType
         */
        Builder driveCacheType(DriveCacheType driveCacheType);
    }

    static final class BuilderImpl implements Builder {
        private String weeklyMaintenanceStartTime;

        private DataRepositoryConfiguration dataRepositoryConfiguration;

        private String deploymentType;

        private Integer perUnitStorageThroughput;

        private String mountName;

        private String dailyAutomaticBackupStartTime;

        private Integer automaticBackupRetentionDays;

        private Boolean copyTagsToBackups;

        private String driveCacheType;

        private BuilderImpl() {
        }

        private BuilderImpl(LustreFileSystemConfiguration model) {
            weeklyMaintenanceStartTime(model.weeklyMaintenanceStartTime);
            dataRepositoryConfiguration(model.dataRepositoryConfiguration);
            deploymentType(model.deploymentType);
            perUnitStorageThroughput(model.perUnitStorageThroughput);
            mountName(model.mountName);
            dailyAutomaticBackupStartTime(model.dailyAutomaticBackupStartTime);
            automaticBackupRetentionDays(model.automaticBackupRetentionDays);
            copyTagsToBackups(model.copyTagsToBackups);
            driveCacheType(model.driveCacheType);
        }

        public final String getWeeklyMaintenanceStartTime() {
            return weeklyMaintenanceStartTime;
        }

        @Override
        public final Builder weeklyMaintenanceStartTime(String weeklyMaintenanceStartTime) {
            this.weeklyMaintenanceStartTime = weeklyMaintenanceStartTime;
            return this;
        }

        public final void setWeeklyMaintenanceStartTime(String weeklyMaintenanceStartTime) {
            this.weeklyMaintenanceStartTime = weeklyMaintenanceStartTime;
        }

        public final DataRepositoryConfiguration.Builder getDataRepositoryConfiguration() {
            return dataRepositoryConfiguration != null ? dataRepositoryConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder dataRepositoryConfiguration(DataRepositoryConfiguration dataRepositoryConfiguration) {
            this.dataRepositoryConfiguration = dataRepositoryConfiguration;
            return this;
        }

        public final void setDataRepositoryConfiguration(DataRepositoryConfiguration.BuilderImpl dataRepositoryConfiguration) {
            this.dataRepositoryConfiguration = dataRepositoryConfiguration != null ? dataRepositoryConfiguration.build() : null;
        }

        public final String getDeploymentType() {
            return deploymentType;
        }

        @Override
        public final Builder deploymentType(String deploymentType) {
            this.deploymentType = deploymentType;
            return this;
        }

        @Override
        public final Builder deploymentType(LustreDeploymentType deploymentType) {
            this.deploymentType(deploymentType == null ? null : deploymentType.toString());
            return this;
        }

        public final void setDeploymentType(String deploymentType) {
            this.deploymentType = deploymentType;
        }

        public final Integer getPerUnitStorageThroughput() {
            return perUnitStorageThroughput;
        }

        @Override
        public final Builder perUnitStorageThroughput(Integer perUnitStorageThroughput) {
            this.perUnitStorageThroughput = perUnitStorageThroughput;
            return this;
        }

        public final void setPerUnitStorageThroughput(Integer perUnitStorageThroughput) {
            this.perUnitStorageThroughput = perUnitStorageThroughput;
        }

        public final String getMountName() {
            return mountName;
        }

        @Override
        public final Builder mountName(String mountName) {
            this.mountName = mountName;
            return this;
        }

        public final void setMountName(String mountName) {
            this.mountName = mountName;
        }

        public final String getDailyAutomaticBackupStartTime() {
            return dailyAutomaticBackupStartTime;
        }

        @Override
        public final Builder dailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime) {
            this.dailyAutomaticBackupStartTime = dailyAutomaticBackupStartTime;
            return this;
        }

        public final void setDailyAutomaticBackupStartTime(String dailyAutomaticBackupStartTime) {
            this.dailyAutomaticBackupStartTime = dailyAutomaticBackupStartTime;
        }

        public final Integer getAutomaticBackupRetentionDays() {
            return automaticBackupRetentionDays;
        }

        @Override
        public final Builder automaticBackupRetentionDays(Integer automaticBackupRetentionDays) {
            this.automaticBackupRetentionDays = automaticBackupRetentionDays;
            return this;
        }

        public final void setAutomaticBackupRetentionDays(Integer automaticBackupRetentionDays) {
            this.automaticBackupRetentionDays = automaticBackupRetentionDays;
        }

        public final Boolean getCopyTagsToBackups() {
            return copyTagsToBackups;
        }

        @Override
        public final Builder copyTagsToBackups(Boolean copyTagsToBackups) {
            this.copyTagsToBackups = copyTagsToBackups;
            return this;
        }

        public final void setCopyTagsToBackups(Boolean copyTagsToBackups) {
            this.copyTagsToBackups = copyTagsToBackups;
        }

        public final String getDriveCacheType() {
            return driveCacheType;
        }

        @Override
        public final Builder driveCacheType(String driveCacheType) {
            this.driveCacheType = driveCacheType;
            return this;
        }

        @Override
        public final Builder driveCacheType(DriveCacheType driveCacheType) {
            this.driveCacheType(driveCacheType == null ? null : driveCacheType.toString());
            return this;
        }

        public final void setDriveCacheType(String driveCacheType) {
            this.driveCacheType = driveCacheType;
        }

        @Override
        public LustreFileSystemConfiguration build() {
            return new LustreFileSystemConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
