/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.evs.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An ESXi host that runs on an Amazon EC2 bare metal instance. Four hosts are created in an Amazon EVS environment
 * during environment creation. You can add hosts to an environment using the <code>CreateEnvironmentHost</code>
 * operation. Amazon EVS supports 4-16 hosts per environment.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Host implements SdkPojo, Serializable, ToCopyableBuilder<Host.Builder, Host> {
    private static final SdkField<String> HOST_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("hostName").getter(getter(Host::hostName)).setter(setter(Builder::hostName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hostName").build()).build();

    private static final SdkField<String> IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ipAddress").getter(getter(Host::ipAddress)).setter(setter(Builder::ipAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ipAddress").build()).build();

    private static final SdkField<String> KEY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("keyName").getter(getter(Host::keyName)).setter(setter(Builder::keyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("keyName").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceType").getter(getter(Host::instanceTypeAsString)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceType").build()).build();

    private static final SdkField<String> PLACEMENT_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("placementGroupId").getter(getter(Host::placementGroupId)).setter(setter(Builder::placementGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("placementGroupId").build()).build();

    private static final SdkField<String> DEDICATED_HOST_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dedicatedHostId").getter(getter(Host::dedicatedHostId)).setter(setter(Builder::dedicatedHostId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dedicatedHostId").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(Host::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<Instant> MODIFIED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("modifiedAt").getter(getter(Host::modifiedAt)).setter(setter(Builder::modifiedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modifiedAt").build()).build();

    private static final SdkField<String> HOST_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("hostState").getter(getter(Host::hostStateAsString)).setter(setter(Builder::hostState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hostState").build()).build();

    private static final SdkField<String> STATE_DETAILS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("stateDetails").getter(getter(Host::stateDetails)).setter(setter(Builder::stateDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stateDetails").build()).build();

    private static final SdkField<String> EC2_INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ec2InstanceId").getter(getter(Host::ec2InstanceId)).setter(setter(Builder::ec2InstanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ec2InstanceId").build()).build();

    private static final SdkField<List<NetworkInterface>> NETWORK_INTERFACES_FIELD = SdkField
            .<List<NetworkInterface>> builder(MarshallingType.LIST)
            .memberName("networkInterfaces")
            .getter(getter(Host::networkInterfaces))
            .setter(setter(Builder::networkInterfaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkInterfaces").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NetworkInterface> builder(MarshallingType.SDK_POJO)
                                            .constructor(NetworkInterface::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HOST_NAME_FIELD,
            IP_ADDRESS_FIELD, KEY_NAME_FIELD, INSTANCE_TYPE_FIELD, PLACEMENT_GROUP_ID_FIELD, DEDICATED_HOST_ID_FIELD,
            CREATED_AT_FIELD, MODIFIED_AT_FIELD, HOST_STATE_FIELD, STATE_DETAILS_FIELD, EC2_INSTANCE_ID_FIELD,
            NETWORK_INTERFACES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String hostName;

    private final String ipAddress;

    private final String keyName;

    private final String instanceType;

    private final String placementGroupId;

    private final String dedicatedHostId;

    private final Instant createdAt;

    private final Instant modifiedAt;

    private final String hostState;

    private final String stateDetails;

    private final String ec2InstanceId;

    private final List<NetworkInterface> networkInterfaces;

    private Host(BuilderImpl builder) {
        this.hostName = builder.hostName;
        this.ipAddress = builder.ipAddress;
        this.keyName = builder.keyName;
        this.instanceType = builder.instanceType;
        this.placementGroupId = builder.placementGroupId;
        this.dedicatedHostId = builder.dedicatedHostId;
        this.createdAt = builder.createdAt;
        this.modifiedAt = builder.modifiedAt;
        this.hostState = builder.hostState;
        this.stateDetails = builder.stateDetails;
        this.ec2InstanceId = builder.ec2InstanceId;
        this.networkInterfaces = builder.networkInterfaces;
    }

    /**
     * <p>
     * The DNS hostname of the host. DNS hostnames for hosts must be unique across Amazon EVS environments and within
     * VCF.
     * </p>
     * 
     * @return The DNS hostname of the host. DNS hostnames for hosts must be unique across Amazon EVS environments and
     *         within VCF.
     */
    public final String hostName() {
        return hostName;
    }

    /**
     * <p>
     * The IP address of the host.
     * </p>
     * 
     * @return The IP address of the host.
     */
    public final String ipAddress() {
        return ipAddress;
    }

    /**
     * <p>
     * The name of the SSH key that is used to access the host.
     * </p>
     * 
     * @return The name of the SSH key that is used to access the host.
     */
    public final String keyName() {
        return keyName;
    }

    /**
     * <p>
     * The EC2 instance type of the host.
     * </p>
     * <note>
     * <p>
     * EC2 instances created through Amazon EVS do not support associating an IAM instance profile.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The EC2 instance type of the host.</p> <note>
     *         <p>
     *         EC2 instances created through Amazon EVS do not support associating an IAM instance profile.
     *         </p>
     * @see InstanceType
     */
    public final InstanceType instanceType() {
        return InstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The EC2 instance type of the host.
     * </p>
     * <note>
     * <p>
     * EC2 instances created through Amazon EVS do not support associating an IAM instance profile.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The EC2 instance type of the host.</p> <note>
     *         <p>
     *         EC2 instances created through Amazon EVS do not support associating an IAM instance profile.
     *         </p>
     * @see InstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * The unique ID of the placement group where the host is placed.
     * </p>
     * 
     * @return The unique ID of the placement group where the host is placed.
     */
    public final String placementGroupId() {
        return placementGroupId;
    }

    /**
     * <p>
     * The unique ID of the Amazon EC2 Dedicated Host.
     * </p>
     * 
     * @return The unique ID of the Amazon EC2 Dedicated Host.
     */
    public final String dedicatedHostId() {
        return dedicatedHostId;
    }

    /**
     * <p>
     * The date and time that the host was created.
     * </p>
     * 
     * @return The date and time that the host was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The date and time that the host was modified.
     * </p>
     * 
     * @return The date and time that the host was modified.
     */
    public final Instant modifiedAt() {
        return modifiedAt;
    }

    /**
     * <p>
     * The state of the host.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hostState} will
     * return {@link HostState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #hostStateAsString}.
     * </p>
     * 
     * @return The state of the host.
     * @see HostState
     */
    public final HostState hostState() {
        return HostState.fromValue(hostState);
    }

    /**
     * <p>
     * The state of the host.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hostState} will
     * return {@link HostState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #hostStateAsString}.
     * </p>
     * 
     * @return The state of the host.
     * @see HostState
     */
    public final String hostStateAsString() {
        return hostState;
    }

    /**
     * <p>
     * A detailed description of the <code>hostState</code> of a host.
     * </p>
     * 
     * @return A detailed description of the <code>hostState</code> of a host.
     */
    public final String stateDetails() {
        return stateDetails;
    }

    /**
     * <p>
     * The unique ID of the EC2 instance that represents the host.
     * </p>
     * 
     * @return The unique ID of the EC2 instance that represents the host.
     */
    public final String ec2InstanceId() {
        return ec2InstanceId;
    }

    /**
     * For responses, this returns true if the service returned a value for the NetworkInterfaces property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNetworkInterfaces() {
        return networkInterfaces != null && !(networkInterfaces instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The elastic network interfaces that are attached to the host.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNetworkInterfaces} method.
     * </p>
     * 
     * @return The elastic network interfaces that are attached to the host.
     */
    public final List<NetworkInterface> networkInterfaces() {
        return networkInterfaces;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hostName());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddress());
        hashCode = 31 * hashCode + Objects.hashCode(keyName());
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(placementGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(dedicatedHostId());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(modifiedAt());
        hashCode = 31 * hashCode + Objects.hashCode(hostStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateDetails());
        hashCode = 31 * hashCode + Objects.hashCode(ec2InstanceId());
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkInterfaces() ? networkInterfaces() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Host)) {
            return false;
        }
        Host other = (Host) obj;
        return Objects.equals(hostName(), other.hostName()) && Objects.equals(ipAddress(), other.ipAddress())
                && Objects.equals(keyName(), other.keyName())
                && Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(placementGroupId(), other.placementGroupId())
                && Objects.equals(dedicatedHostId(), other.dedicatedHostId()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(modifiedAt(), other.modifiedAt())
                && Objects.equals(hostStateAsString(), other.hostStateAsString())
                && Objects.equals(stateDetails(), other.stateDetails()) && Objects.equals(ec2InstanceId(), other.ec2InstanceId())
                && hasNetworkInterfaces() == other.hasNetworkInterfaces()
                && Objects.equals(networkInterfaces(), other.networkInterfaces());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Host").add("HostName", hostName()).add("IpAddress", ipAddress()).add("KeyName", keyName())
                .add("InstanceType", instanceTypeAsString()).add("PlacementGroupId", placementGroupId())
                .add("DedicatedHostId", dedicatedHostId()).add("CreatedAt", createdAt()).add("ModifiedAt", modifiedAt())
                .add("HostState", hostStateAsString()).add("StateDetails", stateDetails()).add("Ec2InstanceId", ec2InstanceId())
                .add("NetworkInterfaces", hasNetworkInterfaces() ? networkInterfaces() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "hostName":
            return Optional.ofNullable(clazz.cast(hostName()));
        case "ipAddress":
            return Optional.ofNullable(clazz.cast(ipAddress()));
        case "keyName":
            return Optional.ofNullable(clazz.cast(keyName()));
        case "instanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "placementGroupId":
            return Optional.ofNullable(clazz.cast(placementGroupId()));
        case "dedicatedHostId":
            return Optional.ofNullable(clazz.cast(dedicatedHostId()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "modifiedAt":
            return Optional.ofNullable(clazz.cast(modifiedAt()));
        case "hostState":
            return Optional.ofNullable(clazz.cast(hostStateAsString()));
        case "stateDetails":
            return Optional.ofNullable(clazz.cast(stateDetails()));
        case "ec2InstanceId":
            return Optional.ofNullable(clazz.cast(ec2InstanceId()));
        case "networkInterfaces":
            return Optional.ofNullable(clazz.cast(networkInterfaces()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("hostName", HOST_NAME_FIELD);
        map.put("ipAddress", IP_ADDRESS_FIELD);
        map.put("keyName", KEY_NAME_FIELD);
        map.put("instanceType", INSTANCE_TYPE_FIELD);
        map.put("placementGroupId", PLACEMENT_GROUP_ID_FIELD);
        map.put("dedicatedHostId", DEDICATED_HOST_ID_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("modifiedAt", MODIFIED_AT_FIELD);
        map.put("hostState", HOST_STATE_FIELD);
        map.put("stateDetails", STATE_DETAILS_FIELD);
        map.put("ec2InstanceId", EC2_INSTANCE_ID_FIELD);
        map.put("networkInterfaces", NETWORK_INTERFACES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Host, T> g) {
        return obj -> g.apply((Host) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Host> {
        /**
         * <p>
         * The DNS hostname of the host. DNS hostnames for hosts must be unique across Amazon EVS environments and
         * within VCF.
         * </p>
         * 
         * @param hostName
         *        The DNS hostname of the host. DNS hostnames for hosts must be unique across Amazon EVS environments
         *        and within VCF.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostName(String hostName);

        /**
         * <p>
         * The IP address of the host.
         * </p>
         * 
         * @param ipAddress
         *        The IP address of the host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * <p>
         * The name of the SSH key that is used to access the host.
         * </p>
         * 
         * @param keyName
         *        The name of the SSH key that is used to access the host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyName(String keyName);

        /**
         * <p>
         * The EC2 instance type of the host.
         * </p>
         * <note>
         * <p>
         * EC2 instances created through Amazon EVS do not support associating an IAM instance profile.
         * </p>
         * </note>
         * 
         * @param instanceType
         *        The EC2 instance type of the host.</p> <note>
         *        <p>
         *        EC2 instances created through Amazon EVS do not support associating an IAM instance profile.
         *        </p>
         * @see InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The EC2 instance type of the host.
         * </p>
         * <note>
         * <p>
         * EC2 instances created through Amazon EVS do not support associating an IAM instance profile.
         * </p>
         * </note>
         * 
         * @param instanceType
         *        The EC2 instance type of the host.</p> <note>
         *        <p>
         *        EC2 instances created through Amazon EVS do not support associating an IAM instance profile.
         *        </p>
         * @see InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceType
         */
        Builder instanceType(InstanceType instanceType);

        /**
         * <p>
         * The unique ID of the placement group where the host is placed.
         * </p>
         * 
         * @param placementGroupId
         *        The unique ID of the placement group where the host is placed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placementGroupId(String placementGroupId);

        /**
         * <p>
         * The unique ID of the Amazon EC2 Dedicated Host.
         * </p>
         * 
         * @param dedicatedHostId
         *        The unique ID of the Amazon EC2 Dedicated Host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dedicatedHostId(String dedicatedHostId);

        /**
         * <p>
         * The date and time that the host was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time that the host was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The date and time that the host was modified.
         * </p>
         * 
         * @param modifiedAt
         *        The date and time that the host was modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifiedAt(Instant modifiedAt);

        /**
         * <p>
         * The state of the host.
         * </p>
         * 
         * @param hostState
         *        The state of the host.
         * @see HostState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HostState
         */
        Builder hostState(String hostState);

        /**
         * <p>
         * The state of the host.
         * </p>
         * 
         * @param hostState
         *        The state of the host.
         * @see HostState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HostState
         */
        Builder hostState(HostState hostState);

        /**
         * <p>
         * A detailed description of the <code>hostState</code> of a host.
         * </p>
         * 
         * @param stateDetails
         *        A detailed description of the <code>hostState</code> of a host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateDetails(String stateDetails);

        /**
         * <p>
         * The unique ID of the EC2 instance that represents the host.
         * </p>
         * 
         * @param ec2InstanceId
         *        The unique ID of the EC2 instance that represents the host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2InstanceId(String ec2InstanceId);

        /**
         * <p>
         * The elastic network interfaces that are attached to the host.
         * </p>
         * 
         * @param networkInterfaces
         *        The elastic network interfaces that are attached to the host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces);

        /**
         * <p>
         * The elastic network interfaces that are attached to the host.
         * </p>
         * 
         * @param networkInterfaces
         *        The elastic network interfaces that are attached to the host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(NetworkInterface... networkInterfaces);

        /**
         * <p>
         * The elastic network interfaces that are attached to the host.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.evs.model.NetworkInterface.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.evs.model.NetworkInterface#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.evs.model.NetworkInterface.Builder#build()} is called immediately and
         * its result is passed to {@link #networkInterfaces(List<NetworkInterface>)}.
         * 
         * @param networkInterfaces
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.evs.model.NetworkInterface.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkInterfaces(java.util.Collection<NetworkInterface>)
         */
        Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces);
    }

    static final class BuilderImpl implements Builder {
        private String hostName;

        private String ipAddress;

        private String keyName;

        private String instanceType;

        private String placementGroupId;

        private String dedicatedHostId;

        private Instant createdAt;

        private Instant modifiedAt;

        private String hostState;

        private String stateDetails;

        private String ec2InstanceId;

        private List<NetworkInterface> networkInterfaces = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Host model) {
            hostName(model.hostName);
            ipAddress(model.ipAddress);
            keyName(model.keyName);
            instanceType(model.instanceType);
            placementGroupId(model.placementGroupId);
            dedicatedHostId(model.dedicatedHostId);
            createdAt(model.createdAt);
            modifiedAt(model.modifiedAt);
            hostState(model.hostState);
            stateDetails(model.stateDetails);
            ec2InstanceId(model.ec2InstanceId);
            networkInterfaces(model.networkInterfaces);
        }

        public final String getHostName() {
            return hostName;
        }

        public final void setHostName(String hostName) {
            this.hostName = hostName;
        }

        @Override
        public final Builder hostName(String hostName) {
            this.hostName = hostName;
            return this;
        }

        public final String getIpAddress() {
            return ipAddress;
        }

        public final void setIpAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        @Override
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final String getKeyName() {
            return keyName;
        }

        public final void setKeyName(String keyName) {
            this.keyName = keyName;
        }

        @Override
        public final Builder keyName(String keyName) {
            this.keyName = keyName;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(InstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final String getPlacementGroupId() {
            return placementGroupId;
        }

        public final void setPlacementGroupId(String placementGroupId) {
            this.placementGroupId = placementGroupId;
        }

        @Override
        public final Builder placementGroupId(String placementGroupId) {
            this.placementGroupId = placementGroupId;
            return this;
        }

        public final String getDedicatedHostId() {
            return dedicatedHostId;
        }

        public final void setDedicatedHostId(String dedicatedHostId) {
            this.dedicatedHostId = dedicatedHostId;
        }

        @Override
        public final Builder dedicatedHostId(String dedicatedHostId) {
            this.dedicatedHostId = dedicatedHostId;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getModifiedAt() {
            return modifiedAt;
        }

        public final void setModifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
        }

        @Override
        public final Builder modifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
            return this;
        }

        public final String getHostState() {
            return hostState;
        }

        public final void setHostState(String hostState) {
            this.hostState = hostState;
        }

        @Override
        public final Builder hostState(String hostState) {
            this.hostState = hostState;
            return this;
        }

        @Override
        public final Builder hostState(HostState hostState) {
            this.hostState(hostState == null ? null : hostState.toString());
            return this;
        }

        public final String getStateDetails() {
            return stateDetails;
        }

        public final void setStateDetails(String stateDetails) {
            this.stateDetails = stateDetails;
        }

        @Override
        public final Builder stateDetails(String stateDetails) {
            this.stateDetails = stateDetails;
            return this;
        }

        public final String getEc2InstanceId() {
            return ec2InstanceId;
        }

        public final void setEc2InstanceId(String ec2InstanceId) {
            this.ec2InstanceId = ec2InstanceId;
        }

        @Override
        public final Builder ec2InstanceId(String ec2InstanceId) {
            this.ec2InstanceId = ec2InstanceId;
            return this;
        }

        public final List<NetworkInterface.Builder> getNetworkInterfaces() {
            List<NetworkInterface.Builder> result = NetworkInterfaceListCopier.copyToBuilder(this.networkInterfaces);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNetworkInterfaces(Collection<NetworkInterface.BuilderImpl> networkInterfaces) {
            this.networkInterfaces = NetworkInterfaceListCopier.copyFromBuilder(networkInterfaces);
        }

        @Override
        public final Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces) {
            this.networkInterfaces = NetworkInterfaceListCopier.copy(networkInterfaces);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(NetworkInterface... networkInterfaces) {
            networkInterfaces(Arrays.asList(networkInterfaces));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces) {
            networkInterfaces(Stream.of(networkInterfaces).map(c -> NetworkInterface.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Host build() {
            return new Host(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
