/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.evs.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that represents an Amazon EVS environment.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Environment implements SdkPojo, Serializable, ToCopyableBuilder<Environment.Builder, Environment> {
    private static final SdkField<String> ENVIRONMENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("environmentId").getter(getter(Environment::environmentId)).setter(setter(Builder::environmentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environmentId").build()).build();

    private static final SdkField<String> ENVIRONMENT_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("environmentState").getter(getter(Environment::environmentStateAsString))
            .setter(setter(Builder::environmentState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environmentState").build()).build();

    private static final SdkField<String> STATE_DETAILS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("stateDetails").getter(getter(Environment::stateDetails)).setter(setter(Builder::stateDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stateDetails").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(Environment::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<Instant> MODIFIED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("modifiedAt").getter(getter(Environment::modifiedAt)).setter(setter(Builder::modifiedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modifiedAt").build()).build();

    private static final SdkField<String> ENVIRONMENT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("environmentArn").getter(getter(Environment::environmentArn)).setter(setter(Builder::environmentArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environmentArn").build()).build();

    private static final SdkField<String> ENVIRONMENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("environmentName").getter(getter(Environment::environmentName)).setter(setter(Builder::environmentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environmentName").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("vpcId")
            .getter(getter(Environment::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vpcId").build()).build();

    private static final SdkField<String> SERVICE_ACCESS_SUBNET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serviceAccessSubnetId").getter(getter(Environment::serviceAccessSubnetId))
            .setter(setter(Builder::serviceAccessSubnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceAccessSubnetId").build())
            .build();

    private static final SdkField<String> VCF_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("vcfVersion").getter(getter(Environment::vcfVersionAsString)).setter(setter(Builder::vcfVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vcfVersion").build()).build();

    private static final SdkField<Boolean> TERMS_ACCEPTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("termsAccepted").getter(getter(Environment::termsAccepted)).setter(setter(Builder::termsAccepted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("termsAccepted").build()).build();

    private static final SdkField<List<LicenseInfo>> LICENSE_INFO_FIELD = SdkField
            .<List<LicenseInfo>> builder(MarshallingType.LIST)
            .memberName("licenseInfo")
            .getter(getter(Environment::licenseInfo))
            .setter(setter(Builder::licenseInfo))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("licenseInfo").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LicenseInfo> builder(MarshallingType.SDK_POJO)
                                            .constructor(LicenseInfo::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SITE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("siteId")
            .getter(getter(Environment::siteId)).setter(setter(Builder::siteId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("siteId").build()).build();

    private static final SdkField<String> ENVIRONMENT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("environmentStatus").getter(getter(Environment::environmentStatusAsString))
            .setter(setter(Builder::environmentStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environmentStatus").build()).build();

    private static final SdkField<List<Check>> CHECKS_FIELD = SdkField
            .<List<Check>> builder(MarshallingType.LIST)
            .memberName("checks")
            .getter(getter(Environment::checks))
            .setter(setter(Builder::checks))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("checks").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Check> builder(MarshallingType.SDK_POJO)
                                            .constructor(Check::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ConnectivityInfo> CONNECTIVITY_INFO_FIELD = SdkField
            .<ConnectivityInfo> builder(MarshallingType.SDK_POJO).memberName("connectivityInfo")
            .getter(getter(Environment::connectivityInfo)).setter(setter(Builder::connectivityInfo))
            .constructor(ConnectivityInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("connectivityInfo").build()).build();

    private static final SdkField<VcfHostnames> VCF_HOSTNAMES_FIELD = SdkField.<VcfHostnames> builder(MarshallingType.SDK_POJO)
            .memberName("vcfHostnames").getter(getter(Environment::vcfHostnames)).setter(setter(Builder::vcfHostnames))
            .constructor(VcfHostnames::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vcfHostnames").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("kmsKeyId").getter(getter(Environment::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kmsKeyId").build()).build();

    private static final SdkField<ServiceAccessSecurityGroups> SERVICE_ACCESS_SECURITY_GROUPS_FIELD = SdkField
            .<ServiceAccessSecurityGroups> builder(MarshallingType.SDK_POJO)
            .memberName("serviceAccessSecurityGroups")
            .getter(getter(Environment::serviceAccessSecurityGroups))
            .setter(setter(Builder::serviceAccessSecurityGroups))
            .constructor(ServiceAccessSecurityGroups::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceAccessSecurityGroups")
                    .build()).build();

    private static final SdkField<List<Secret>> CREDENTIALS_FIELD = SdkField
            .<List<Secret>> builder(MarshallingType.LIST)
            .memberName("credentials")
            .getter(getter(Environment::credentials))
            .setter(setter(Builder::credentials))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("credentials").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Secret> builder(MarshallingType.SDK_POJO)
                                            .constructor(Secret::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENVIRONMENT_ID_FIELD,
            ENVIRONMENT_STATE_FIELD, STATE_DETAILS_FIELD, CREATED_AT_FIELD, MODIFIED_AT_FIELD, ENVIRONMENT_ARN_FIELD,
            ENVIRONMENT_NAME_FIELD, VPC_ID_FIELD, SERVICE_ACCESS_SUBNET_ID_FIELD, VCF_VERSION_FIELD, TERMS_ACCEPTED_FIELD,
            LICENSE_INFO_FIELD, SITE_ID_FIELD, ENVIRONMENT_STATUS_FIELD, CHECKS_FIELD, CONNECTIVITY_INFO_FIELD,
            VCF_HOSTNAMES_FIELD, KMS_KEY_ID_FIELD, SERVICE_ACCESS_SECURITY_GROUPS_FIELD, CREDENTIALS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String environmentId;

    private final String environmentState;

    private final String stateDetails;

    private final Instant createdAt;

    private final Instant modifiedAt;

    private final String environmentArn;

    private final String environmentName;

    private final String vpcId;

    private final String serviceAccessSubnetId;

    private final String vcfVersion;

    private final Boolean termsAccepted;

    private final List<LicenseInfo> licenseInfo;

    private final String siteId;

    private final String environmentStatus;

    private final List<Check> checks;

    private final ConnectivityInfo connectivityInfo;

    private final VcfHostnames vcfHostnames;

    private final String kmsKeyId;

    private final ServiceAccessSecurityGroups serviceAccessSecurityGroups;

    private final List<Secret> credentials;

    private Environment(BuilderImpl builder) {
        this.environmentId = builder.environmentId;
        this.environmentState = builder.environmentState;
        this.stateDetails = builder.stateDetails;
        this.createdAt = builder.createdAt;
        this.modifiedAt = builder.modifiedAt;
        this.environmentArn = builder.environmentArn;
        this.environmentName = builder.environmentName;
        this.vpcId = builder.vpcId;
        this.serviceAccessSubnetId = builder.serviceAccessSubnetId;
        this.vcfVersion = builder.vcfVersion;
        this.termsAccepted = builder.termsAccepted;
        this.licenseInfo = builder.licenseInfo;
        this.siteId = builder.siteId;
        this.environmentStatus = builder.environmentStatus;
        this.checks = builder.checks;
        this.connectivityInfo = builder.connectivityInfo;
        this.vcfHostnames = builder.vcfHostnames;
        this.kmsKeyId = builder.kmsKeyId;
        this.serviceAccessSecurityGroups = builder.serviceAccessSecurityGroups;
        this.credentials = builder.credentials;
    }

    /**
     * <p>
     * The unique ID for the environment.
     * </p>
     * 
     * @return The unique ID for the environment.
     */
    public final String environmentId() {
        return environmentId;
    }

    /**
     * <p>
     * The state of an environment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #environmentState}
     * will return {@link EnvironmentState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #environmentStateAsString}.
     * </p>
     * 
     * @return The state of an environment.
     * @see EnvironmentState
     */
    public final EnvironmentState environmentState() {
        return EnvironmentState.fromValue(environmentState);
    }

    /**
     * <p>
     * The state of an environment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #environmentState}
     * will return {@link EnvironmentState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #environmentStateAsString}.
     * </p>
     * 
     * @return The state of an environment.
     * @see EnvironmentState
     */
    public final String environmentStateAsString() {
        return environmentState;
    }

    /**
     * <p>
     * A detailed description of the <code>environmentState</code> of an environment.
     * </p>
     * 
     * @return A detailed description of the <code>environmentState</code> of an environment.
     */
    public final String stateDetails() {
        return stateDetails;
    }

    /**
     * <p>
     * The date and time that the environment was created.
     * </p>
     * 
     * @return The date and time that the environment was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The date and time that the environment was modified.
     * </p>
     * 
     * @return The date and time that the environment was modified.
     */
    public final Instant modifiedAt() {
        return modifiedAt;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that is associated with the environment.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that is associated with the environment.
     */
    public final String environmentArn() {
        return environmentArn;
    }

    /**
     * <p>
     * The name of the environment.
     * </p>
     * 
     * @return The name of the environment.
     */
    public final String environmentName() {
        return environmentName;
    }

    /**
     * <p>
     * The VPC associated with the environment.
     * </p>
     * 
     * @return The VPC associated with the environment.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * The subnet that is used to establish connectivity between the Amazon EVS control plane and VPC. Amazon EVS uses
     * this subnet to perform validations and create the environment.
     * </p>
     * 
     * @return The subnet that is used to establish connectivity between the Amazon EVS control plane and VPC. Amazon
     *         EVS uses this subnet to perform validations and create the environment.
     */
    public final String serviceAccessSubnetId() {
        return serviceAccessSubnetId;
    }

    /**
     * <p>
     * The VCF version of the environment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #vcfVersion} will
     * return {@link VcfVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #vcfVersionAsString}.
     * </p>
     * 
     * @return The VCF version of the environment.
     * @see VcfVersion
     */
    public final VcfVersion vcfVersion() {
        return VcfVersion.fromValue(vcfVersion);
    }

    /**
     * <p>
     * The VCF version of the environment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #vcfVersion} will
     * return {@link VcfVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #vcfVersionAsString}.
     * </p>
     * 
     * @return The VCF version of the environment.
     * @see VcfVersion
     */
    public final String vcfVersionAsString() {
        return vcfVersion;
    }

    /**
     * <p>
     * Customer confirmation that the customer has purchased and will continue to maintain the required number of VCF
     * software licenses to cover all physical processor cores in the Amazon EVS environment. Information about your VCF
     * software in Amazon EVS will be shared with Broadcom to verify license compliance. Amazon EVS does not validate
     * license keys. To validate license keys, visit the Broadcom support portal.
     * </p>
     * 
     * @return Customer confirmation that the customer has purchased and will continue to maintain the required number
     *         of VCF software licenses to cover all physical processor cores in the Amazon EVS environment. Information
     *         about your VCF software in Amazon EVS will be shared with Broadcom to verify license compliance. Amazon
     *         EVS does not validate license keys. To validate license keys, visit the Broadcom support portal.
     */
    public final Boolean termsAccepted() {
        return termsAccepted;
    }

    /**
     * For responses, this returns true if the service returned a value for the LicenseInfo property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLicenseInfo() {
        return licenseInfo != null && !(licenseInfo instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The license information that Amazon EVS requires to create an environment. Amazon EVS requires two license keys:
     * a VCF solution key and a vSAN license key. The VCF solution key must cover a minimum of 256 cores. The vSAN
     * license key must provide at least 110 TiB of vSAN capacity.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLicenseInfo} method.
     * </p>
     * 
     * @return The license information that Amazon EVS requires to create an environment. Amazon EVS requires two
     *         license keys: a VCF solution key and a vSAN license key. The VCF solution key must cover a minimum of 256
     *         cores. The vSAN license key must provide at least 110 TiB of vSAN capacity.
     */
    public final List<LicenseInfo> licenseInfo() {
        return licenseInfo;
    }

    /**
     * <p>
     * The Broadcom Site ID that is associated with your Amazon EVS environment. Amazon EVS uses the Broadcom Site ID
     * that you provide to meet Broadcom VCF license usage reporting requirements for Amazon EVS.
     * </p>
     * 
     * @return The Broadcom Site ID that is associated with your Amazon EVS environment. Amazon EVS uses the Broadcom
     *         Site ID that you provide to meet Broadcom VCF license usage reporting requirements for Amazon EVS.
     */
    public final String siteId() {
        return siteId;
    }

    /**
     * <p>
     * Reports impaired functionality that stems from issues internal to the environment, such as impaired reachability.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #environmentStatus}
     * will return {@link CheckResult#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #environmentStatusAsString}.
     * </p>
     * 
     * @return Reports impaired functionality that stems from issues internal to the environment, such as impaired
     *         reachability.
     * @see CheckResult
     */
    public final CheckResult environmentStatus() {
        return CheckResult.fromValue(environmentStatus);
    }

    /**
     * <p>
     * Reports impaired functionality that stems from issues internal to the environment, such as impaired reachability.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #environmentStatus}
     * will return {@link CheckResult#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #environmentStatusAsString}.
     * </p>
     * 
     * @return Reports impaired functionality that stems from issues internal to the environment, such as impaired
     *         reachability.
     * @see CheckResult
     */
    public final String environmentStatusAsString() {
        return environmentStatus;
    }

    /**
     * For responses, this returns true if the service returned a value for the Checks property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasChecks() {
        return checks != null && !(checks instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A check on the environment to identify instance health and VMware VCF licensing issues.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasChecks} method.
     * </p>
     * 
     * @return A check on the environment to identify instance health and VMware VCF licensing issues.
     */
    public final List<Check> checks() {
        return checks;
    }

    /**
     * <p>
     * The connectivity configuration for the environment. Amazon EVS requires that you specify two route server peer
     * IDs. During environment creation, the route server endpoints peer with the NSX uplink VLAN for connectivity to
     * the NSX overlay network.
     * </p>
     * 
     * @return The connectivity configuration for the environment. Amazon EVS requires that you specify two route server
     *         peer IDs. During environment creation, the route server endpoints peer with the NSX uplink VLAN for
     *         connectivity to the NSX overlay network.
     */
    public final ConnectivityInfo connectivityInfo() {
        return connectivityInfo;
    }

    /**
     * <p>
     * The DNS hostnames to be used by the VCF management appliances in your environment.
     * </p>
     * <p>
     * For environment creation to be successful, each hostname entry must resolve to a domain name that you've
     * registered in your DNS service of choice and configured in the DHCP option set of your VPC. DNS hostnames cannot
     * be changed after environment creation has started.
     * </p>
     * 
     * @return The DNS hostnames to be used by the VCF management appliances in your environment.</p>
     *         <p>
     *         For environment creation to be successful, each hostname entry must resolve to a domain name that you've
     *         registered in your DNS service of choice and configured in the DHCP option set of your VPC. DNS hostnames
     *         cannot be changed after environment creation has started.
     */
    public final VcfHostnames vcfHostnames() {
        return vcfHostnames;
    }

    /**
     * <p>
     * The Amazon Web Services KMS key ID that Amazon Web Services Secrets Manager uses to encrypt secrets that are
     * associated with the environment. These secrets contain the VCF credentials that are needed to install vCenter
     * Server, NSX, and SDDC Manager.
     * </p>
     * <p>
     * By default, Amazon EVS use the Amazon Web Services Secrets Manager managed key <code>aws/secretsmanager</code>.
     * You can also specify a customer managed key.
     * </p>
     * 
     * @return The Amazon Web Services KMS key ID that Amazon Web Services Secrets Manager uses to encrypt secrets that
     *         are associated with the environment. These secrets contain the VCF credentials that are needed to install
     *         vCenter Server, NSX, and SDDC Manager.</p>
     *         <p>
     *         By default, Amazon EVS use the Amazon Web Services Secrets Manager managed key
     *         <code>aws/secretsmanager</code>. You can also specify a customer managed key.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The security groups that allow traffic between the Amazon EVS control plane and your VPC for service access. If a
     * security group is not specified, Amazon EVS uses the default security group in your account for service access.
     * </p>
     * 
     * @return The security groups that allow traffic between the Amazon EVS control plane and your VPC for service
     *         access. If a security group is not specified, Amazon EVS uses the default security group in your account
     *         for service access.
     */
    public final ServiceAccessSecurityGroups serviceAccessSecurityGroups() {
        return serviceAccessSecurityGroups;
    }

    /**
     * For responses, this returns true if the service returned a value for the Credentials property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCredentials() {
        return credentials != null && !(credentials instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The VCF credentials that are stored as Amazon EVS managed secrets in Amazon Web Services Secrets Manager.
     * </p>
     * <p>
     * Amazon EVS stores credentials that are needed to install vCenter Server, NSX, and SDDC Manager.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCredentials} method.
     * </p>
     * 
     * @return The VCF credentials that are stored as Amazon EVS managed secrets in Amazon Web Services Secrets
     *         Manager.</p>
     *         <p>
     *         Amazon EVS stores credentials that are needed to install vCenter Server, NSX, and SDDC Manager.
     */
    public final List<Secret> credentials() {
        return credentials;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(environmentId());
        hashCode = 31 * hashCode + Objects.hashCode(environmentStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateDetails());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(modifiedAt());
        hashCode = 31 * hashCode + Objects.hashCode(environmentArn());
        hashCode = 31 * hashCode + Objects.hashCode(environmentName());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(serviceAccessSubnetId());
        hashCode = 31 * hashCode + Objects.hashCode(vcfVersionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(termsAccepted());
        hashCode = 31 * hashCode + Objects.hashCode(hasLicenseInfo() ? licenseInfo() : null);
        hashCode = 31 * hashCode + Objects.hashCode(siteId());
        hashCode = 31 * hashCode + Objects.hashCode(environmentStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasChecks() ? checks() : null);
        hashCode = 31 * hashCode + Objects.hashCode(connectivityInfo());
        hashCode = 31 * hashCode + Objects.hashCode(vcfHostnames());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(serviceAccessSecurityGroups());
        hashCode = 31 * hashCode + Objects.hashCode(hasCredentials() ? credentials() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Environment)) {
            return false;
        }
        Environment other = (Environment) obj;
        return Objects.equals(environmentId(), other.environmentId())
                && Objects.equals(environmentStateAsString(), other.environmentStateAsString())
                && Objects.equals(stateDetails(), other.stateDetails()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(modifiedAt(), other.modifiedAt()) && Objects.equals(environmentArn(), other.environmentArn())
                && Objects.equals(environmentName(), other.environmentName()) && Objects.equals(vpcId(), other.vpcId())
                && Objects.equals(serviceAccessSubnetId(), other.serviceAccessSubnetId())
                && Objects.equals(vcfVersionAsString(), other.vcfVersionAsString())
                && Objects.equals(termsAccepted(), other.termsAccepted()) && hasLicenseInfo() == other.hasLicenseInfo()
                && Objects.equals(licenseInfo(), other.licenseInfo()) && Objects.equals(siteId(), other.siteId())
                && Objects.equals(environmentStatusAsString(), other.environmentStatusAsString())
                && hasChecks() == other.hasChecks() && Objects.equals(checks(), other.checks())
                && Objects.equals(connectivityInfo(), other.connectivityInfo())
                && Objects.equals(vcfHostnames(), other.vcfHostnames()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(serviceAccessSecurityGroups(), other.serviceAccessSecurityGroups())
                && hasCredentials() == other.hasCredentials() && Objects.equals(credentials(), other.credentials());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Environment").add("EnvironmentId", environmentId())
                .add("EnvironmentState", environmentStateAsString()).add("StateDetails", stateDetails())
                .add("CreatedAt", createdAt()).add("ModifiedAt", modifiedAt()).add("EnvironmentArn", environmentArn())
                .add("EnvironmentName", environmentName()).add("VpcId", vpcId())
                .add("ServiceAccessSubnetId", serviceAccessSubnetId()).add("VcfVersion", vcfVersionAsString())
                .add("TermsAccepted", termsAccepted()).add("LicenseInfo", hasLicenseInfo() ? licenseInfo() : null)
                .add("SiteId", siteId()).add("EnvironmentStatus", environmentStatusAsString())
                .add("Checks", hasChecks() ? checks() : null).add("ConnectivityInfo", connectivityInfo())
                .add("VcfHostnames", vcfHostnames()).add("KmsKeyId", kmsKeyId())
                .add("ServiceAccessSecurityGroups", serviceAccessSecurityGroups())
                .add("Credentials", hasCredentials() ? credentials() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "environmentId":
            return Optional.ofNullable(clazz.cast(environmentId()));
        case "environmentState":
            return Optional.ofNullable(clazz.cast(environmentStateAsString()));
        case "stateDetails":
            return Optional.ofNullable(clazz.cast(stateDetails()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "modifiedAt":
            return Optional.ofNullable(clazz.cast(modifiedAt()));
        case "environmentArn":
            return Optional.ofNullable(clazz.cast(environmentArn()));
        case "environmentName":
            return Optional.ofNullable(clazz.cast(environmentName()));
        case "vpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "serviceAccessSubnetId":
            return Optional.ofNullable(clazz.cast(serviceAccessSubnetId()));
        case "vcfVersion":
            return Optional.ofNullable(clazz.cast(vcfVersionAsString()));
        case "termsAccepted":
            return Optional.ofNullable(clazz.cast(termsAccepted()));
        case "licenseInfo":
            return Optional.ofNullable(clazz.cast(licenseInfo()));
        case "siteId":
            return Optional.ofNullable(clazz.cast(siteId()));
        case "environmentStatus":
            return Optional.ofNullable(clazz.cast(environmentStatusAsString()));
        case "checks":
            return Optional.ofNullable(clazz.cast(checks()));
        case "connectivityInfo":
            return Optional.ofNullable(clazz.cast(connectivityInfo()));
        case "vcfHostnames":
            return Optional.ofNullable(clazz.cast(vcfHostnames()));
        case "kmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "serviceAccessSecurityGroups":
            return Optional.ofNullable(clazz.cast(serviceAccessSecurityGroups()));
        case "credentials":
            return Optional.ofNullable(clazz.cast(credentials()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("environmentId", ENVIRONMENT_ID_FIELD);
        map.put("environmentState", ENVIRONMENT_STATE_FIELD);
        map.put("stateDetails", STATE_DETAILS_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("modifiedAt", MODIFIED_AT_FIELD);
        map.put("environmentArn", ENVIRONMENT_ARN_FIELD);
        map.put("environmentName", ENVIRONMENT_NAME_FIELD);
        map.put("vpcId", VPC_ID_FIELD);
        map.put("serviceAccessSubnetId", SERVICE_ACCESS_SUBNET_ID_FIELD);
        map.put("vcfVersion", VCF_VERSION_FIELD);
        map.put("termsAccepted", TERMS_ACCEPTED_FIELD);
        map.put("licenseInfo", LICENSE_INFO_FIELD);
        map.put("siteId", SITE_ID_FIELD);
        map.put("environmentStatus", ENVIRONMENT_STATUS_FIELD);
        map.put("checks", CHECKS_FIELD);
        map.put("connectivityInfo", CONNECTIVITY_INFO_FIELD);
        map.put("vcfHostnames", VCF_HOSTNAMES_FIELD);
        map.put("kmsKeyId", KMS_KEY_ID_FIELD);
        map.put("serviceAccessSecurityGroups", SERVICE_ACCESS_SECURITY_GROUPS_FIELD);
        map.put("credentials", CREDENTIALS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Environment, T> g) {
        return obj -> g.apply((Environment) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Environment> {
        /**
         * <p>
         * The unique ID for the environment.
         * </p>
         * 
         * @param environmentId
         *        The unique ID for the environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentId(String environmentId);

        /**
         * <p>
         * The state of an environment.
         * </p>
         * 
         * @param environmentState
         *        The state of an environment.
         * @see EnvironmentState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnvironmentState
         */
        Builder environmentState(String environmentState);

        /**
         * <p>
         * The state of an environment.
         * </p>
         * 
         * @param environmentState
         *        The state of an environment.
         * @see EnvironmentState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnvironmentState
         */
        Builder environmentState(EnvironmentState environmentState);

        /**
         * <p>
         * A detailed description of the <code>environmentState</code> of an environment.
         * </p>
         * 
         * @param stateDetails
         *        A detailed description of the <code>environmentState</code> of an environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateDetails(String stateDetails);

        /**
         * <p>
         * The date and time that the environment was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time that the environment was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The date and time that the environment was modified.
         * </p>
         * 
         * @param modifiedAt
         *        The date and time that the environment was modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifiedAt(Instant modifiedAt);

        /**
         * <p>
         * The Amazon Resource Name (ARN) that is associated with the environment.
         * </p>
         * 
         * @param environmentArn
         *        The Amazon Resource Name (ARN) that is associated with the environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentArn(String environmentArn);

        /**
         * <p>
         * The name of the environment.
         * </p>
         * 
         * @param environmentName
         *        The name of the environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentName(String environmentName);

        /**
         * <p>
         * The VPC associated with the environment.
         * </p>
         * 
         * @param vpcId
         *        The VPC associated with the environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The subnet that is used to establish connectivity between the Amazon EVS control plane and VPC. Amazon EVS
         * uses this subnet to perform validations and create the environment.
         * </p>
         * 
         * @param serviceAccessSubnetId
         *        The subnet that is used to establish connectivity between the Amazon EVS control plane and VPC. Amazon
         *        EVS uses this subnet to perform validations and create the environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceAccessSubnetId(String serviceAccessSubnetId);

        /**
         * <p>
         * The VCF version of the environment.
         * </p>
         * 
         * @param vcfVersion
         *        The VCF version of the environment.
         * @see VcfVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VcfVersion
         */
        Builder vcfVersion(String vcfVersion);

        /**
         * <p>
         * The VCF version of the environment.
         * </p>
         * 
         * @param vcfVersion
         *        The VCF version of the environment.
         * @see VcfVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VcfVersion
         */
        Builder vcfVersion(VcfVersion vcfVersion);

        /**
         * <p>
         * Customer confirmation that the customer has purchased and will continue to maintain the required number of
         * VCF software licenses to cover all physical processor cores in the Amazon EVS environment. Information about
         * your VCF software in Amazon EVS will be shared with Broadcom to verify license compliance. Amazon EVS does
         * not validate license keys. To validate license keys, visit the Broadcom support portal.
         * </p>
         * 
         * @param termsAccepted
         *        Customer confirmation that the customer has purchased and will continue to maintain the required
         *        number of VCF software licenses to cover all physical processor cores in the Amazon EVS environment.
         *        Information about your VCF software in Amazon EVS will be shared with Broadcom to verify license
         *        compliance. Amazon EVS does not validate license keys. To validate license keys, visit the Broadcom
         *        support portal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder termsAccepted(Boolean termsAccepted);

        /**
         * <p>
         * The license information that Amazon EVS requires to create an environment. Amazon EVS requires two license
         * keys: a VCF solution key and a vSAN license key. The VCF solution key must cover a minimum of 256 cores. The
         * vSAN license key must provide at least 110 TiB of vSAN capacity.
         * </p>
         * 
         * @param licenseInfo
         *        The license information that Amazon EVS requires to create an environment. Amazon EVS requires two
         *        license keys: a VCF solution key and a vSAN license key. The VCF solution key must cover a minimum of
         *        256 cores. The vSAN license key must provide at least 110 TiB of vSAN capacity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseInfo(Collection<LicenseInfo> licenseInfo);

        /**
         * <p>
         * The license information that Amazon EVS requires to create an environment. Amazon EVS requires two license
         * keys: a VCF solution key and a vSAN license key. The VCF solution key must cover a minimum of 256 cores. The
         * vSAN license key must provide at least 110 TiB of vSAN capacity.
         * </p>
         * 
         * @param licenseInfo
         *        The license information that Amazon EVS requires to create an environment. Amazon EVS requires two
         *        license keys: a VCF solution key and a vSAN license key. The VCF solution key must cover a minimum of
         *        256 cores. The vSAN license key must provide at least 110 TiB of vSAN capacity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenseInfo(LicenseInfo... licenseInfo);

        /**
         * <p>
         * The license information that Amazon EVS requires to create an environment. Amazon EVS requires two license
         * keys: a VCF solution key and a vSAN license key. The VCF solution key must cover a minimum of 256 cores. The
         * vSAN license key must provide at least 110 TiB of vSAN capacity.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.evs.model.LicenseInfo.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.evs.model.LicenseInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.evs.model.LicenseInfo.Builder#build()} is called immediately and its
         * result is passed to {@link #licenseInfo(List<LicenseInfo>)}.
         * 
         * @param licenseInfo
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.evs.model.LicenseInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #licenseInfo(java.util.Collection<LicenseInfo>)
         */
        Builder licenseInfo(Consumer<LicenseInfo.Builder>... licenseInfo);

        /**
         * <p>
         * The Broadcom Site ID that is associated with your Amazon EVS environment. Amazon EVS uses the Broadcom Site
         * ID that you provide to meet Broadcom VCF license usage reporting requirements for Amazon EVS.
         * </p>
         * 
         * @param siteId
         *        The Broadcom Site ID that is associated with your Amazon EVS environment. Amazon EVS uses the Broadcom
         *        Site ID that you provide to meet Broadcom VCF license usage reporting requirements for Amazon EVS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder siteId(String siteId);

        /**
         * <p>
         * Reports impaired functionality that stems from issues internal to the environment, such as impaired
         * reachability.
         * </p>
         * 
         * @param environmentStatus
         *        Reports impaired functionality that stems from issues internal to the environment, such as impaired
         *        reachability.
         * @see CheckResult
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CheckResult
         */
        Builder environmentStatus(String environmentStatus);

        /**
         * <p>
         * Reports impaired functionality that stems from issues internal to the environment, such as impaired
         * reachability.
         * </p>
         * 
         * @param environmentStatus
         *        Reports impaired functionality that stems from issues internal to the environment, such as impaired
         *        reachability.
         * @see CheckResult
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CheckResult
         */
        Builder environmentStatus(CheckResult environmentStatus);

        /**
         * <p>
         * A check on the environment to identify instance health and VMware VCF licensing issues.
         * </p>
         * 
         * @param checks
         *        A check on the environment to identify instance health and VMware VCF licensing issues.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder checks(Collection<Check> checks);

        /**
         * <p>
         * A check on the environment to identify instance health and VMware VCF licensing issues.
         * </p>
         * 
         * @param checks
         *        A check on the environment to identify instance health and VMware VCF licensing issues.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder checks(Check... checks);

        /**
         * <p>
         * A check on the environment to identify instance health and VMware VCF licensing issues.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.evs.model.Check.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.evs.model.Check#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.evs.model.Check.Builder#build()}
         * is called immediately and its result is passed to {@link #checks(List<Check>)}.
         * 
         * @param checks
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.evs.model.Check.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #checks(java.util.Collection<Check>)
         */
        Builder checks(Consumer<Check.Builder>... checks);

        /**
         * <p>
         * The connectivity configuration for the environment. Amazon EVS requires that you specify two route server
         * peer IDs. During environment creation, the route server endpoints peer with the NSX uplink VLAN for
         * connectivity to the NSX overlay network.
         * </p>
         * 
         * @param connectivityInfo
         *        The connectivity configuration for the environment. Amazon EVS requires that you specify two route
         *        server peer IDs. During environment creation, the route server endpoints peer with the NSX uplink VLAN
         *        for connectivity to the NSX overlay network.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectivityInfo(ConnectivityInfo connectivityInfo);

        /**
         * <p>
         * The connectivity configuration for the environment. Amazon EVS requires that you specify two route server
         * peer IDs. During environment creation, the route server endpoints peer with the NSX uplink VLAN for
         * connectivity to the NSX overlay network.
         * </p>
         * This is a convenience method that creates an instance of the {@link ConnectivityInfo.Builder} avoiding the
         * need to create one manually via {@link ConnectivityInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ConnectivityInfo.Builder#build()} is called immediately and its
         * result is passed to {@link #connectivityInfo(ConnectivityInfo)}.
         * 
         * @param connectivityInfo
         *        a consumer that will call methods on {@link ConnectivityInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #connectivityInfo(ConnectivityInfo)
         */
        default Builder connectivityInfo(Consumer<ConnectivityInfo.Builder> connectivityInfo) {
            return connectivityInfo(ConnectivityInfo.builder().applyMutation(connectivityInfo).build());
        }

        /**
         * <p>
         * The DNS hostnames to be used by the VCF management appliances in your environment.
         * </p>
         * <p>
         * For environment creation to be successful, each hostname entry must resolve to a domain name that you've
         * registered in your DNS service of choice and configured in the DHCP option set of your VPC. DNS hostnames
         * cannot be changed after environment creation has started.
         * </p>
         * 
         * @param vcfHostnames
         *        The DNS hostnames to be used by the VCF management appliances in your environment.</p>
         *        <p>
         *        For environment creation to be successful, each hostname entry must resolve to a domain name that
         *        you've registered in your DNS service of choice and configured in the DHCP option set of your VPC. DNS
         *        hostnames cannot be changed after environment creation has started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vcfHostnames(VcfHostnames vcfHostnames);

        /**
         * <p>
         * The DNS hostnames to be used by the VCF management appliances in your environment.
         * </p>
         * <p>
         * For environment creation to be successful, each hostname entry must resolve to a domain name that you've
         * registered in your DNS service of choice and configured in the DHCP option set of your VPC. DNS hostnames
         * cannot be changed after environment creation has started.
         * </p>
         * This is a convenience method that creates an instance of the {@link VcfHostnames.Builder} avoiding the need
         * to create one manually via {@link VcfHostnames#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VcfHostnames.Builder#build()} is called immediately and its
         * result is passed to {@link #vcfHostnames(VcfHostnames)}.
         * 
         * @param vcfHostnames
         *        a consumer that will call methods on {@link VcfHostnames.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vcfHostnames(VcfHostnames)
         */
        default Builder vcfHostnames(Consumer<VcfHostnames.Builder> vcfHostnames) {
            return vcfHostnames(VcfHostnames.builder().applyMutation(vcfHostnames).build());
        }

        /**
         * <p>
         * The Amazon Web Services KMS key ID that Amazon Web Services Secrets Manager uses to encrypt secrets that are
         * associated with the environment. These secrets contain the VCF credentials that are needed to install vCenter
         * Server, NSX, and SDDC Manager.
         * </p>
         * <p>
         * By default, Amazon EVS use the Amazon Web Services Secrets Manager managed key
         * <code>aws/secretsmanager</code>. You can also specify a customer managed key.
         * </p>
         * 
         * @param kmsKeyId
         *        The Amazon Web Services KMS key ID that Amazon Web Services Secrets Manager uses to encrypt secrets
         *        that are associated with the environment. These secrets contain the VCF credentials that are needed to
         *        install vCenter Server, NSX, and SDDC Manager.</p>
         *        <p>
         *        By default, Amazon EVS use the Amazon Web Services Secrets Manager managed key
         *        <code>aws/secretsmanager</code>. You can also specify a customer managed key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The security groups that allow traffic between the Amazon EVS control plane and your VPC for service access.
         * If a security group is not specified, Amazon EVS uses the default security group in your account for service
         * access.
         * </p>
         * 
         * @param serviceAccessSecurityGroups
         *        The security groups that allow traffic between the Amazon EVS control plane and your VPC for service
         *        access. If a security group is not specified, Amazon EVS uses the default security group in your
         *        account for service access.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceAccessSecurityGroups(ServiceAccessSecurityGroups serviceAccessSecurityGroups);

        /**
         * <p>
         * The security groups that allow traffic between the Amazon EVS control plane and your VPC for service access.
         * If a security group is not specified, Amazon EVS uses the default security group in your account for service
         * access.
         * </p>
         * This is a convenience method that creates an instance of the {@link ServiceAccessSecurityGroups.Builder}
         * avoiding the need to create one manually via {@link ServiceAccessSecurityGroups#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ServiceAccessSecurityGroups.Builder#build()} is called
         * immediately and its result is passed to {@link #serviceAccessSecurityGroups(ServiceAccessSecurityGroups)}.
         * 
         * @param serviceAccessSecurityGroups
         *        a consumer that will call methods on {@link ServiceAccessSecurityGroups.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serviceAccessSecurityGroups(ServiceAccessSecurityGroups)
         */
        default Builder serviceAccessSecurityGroups(Consumer<ServiceAccessSecurityGroups.Builder> serviceAccessSecurityGroups) {
            return serviceAccessSecurityGroups(ServiceAccessSecurityGroups.builder().applyMutation(serviceAccessSecurityGroups)
                    .build());
        }

        /**
         * <p>
         * The VCF credentials that are stored as Amazon EVS managed secrets in Amazon Web Services Secrets Manager.
         * </p>
         * <p>
         * Amazon EVS stores credentials that are needed to install vCenter Server, NSX, and SDDC Manager.
         * </p>
         * 
         * @param credentials
         *        The VCF credentials that are stored as Amazon EVS managed secrets in Amazon Web Services Secrets
         *        Manager.</p>
         *        <p>
         *        Amazon EVS stores credentials that are needed to install vCenter Server, NSX, and SDDC Manager.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder credentials(Collection<Secret> credentials);

        /**
         * <p>
         * The VCF credentials that are stored as Amazon EVS managed secrets in Amazon Web Services Secrets Manager.
         * </p>
         * <p>
         * Amazon EVS stores credentials that are needed to install vCenter Server, NSX, and SDDC Manager.
         * </p>
         * 
         * @param credentials
         *        The VCF credentials that are stored as Amazon EVS managed secrets in Amazon Web Services Secrets
         *        Manager.</p>
         *        <p>
         *        Amazon EVS stores credentials that are needed to install vCenter Server, NSX, and SDDC Manager.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder credentials(Secret... credentials);

        /**
         * <p>
         * The VCF credentials that are stored as Amazon EVS managed secrets in Amazon Web Services Secrets Manager.
         * </p>
         * <p>
         * Amazon EVS stores credentials that are needed to install vCenter Server, NSX, and SDDC Manager.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.evs.model.Secret.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.evs.model.Secret#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.evs.model.Secret.Builder#build()}
         * is called immediately and its result is passed to {@link #credentials(List<Secret>)}.
         * 
         * @param credentials
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.evs.model.Secret.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #credentials(java.util.Collection<Secret>)
         */
        Builder credentials(Consumer<Secret.Builder>... credentials);
    }

    static final class BuilderImpl implements Builder {
        private String environmentId;

        private String environmentState;

        private String stateDetails;

        private Instant createdAt;

        private Instant modifiedAt;

        private String environmentArn;

        private String environmentName;

        private String vpcId;

        private String serviceAccessSubnetId;

        private String vcfVersion;

        private Boolean termsAccepted;

        private List<LicenseInfo> licenseInfo = DefaultSdkAutoConstructList.getInstance();

        private String siteId;

        private String environmentStatus;

        private List<Check> checks = DefaultSdkAutoConstructList.getInstance();

        private ConnectivityInfo connectivityInfo;

        private VcfHostnames vcfHostnames;

        private String kmsKeyId;

        private ServiceAccessSecurityGroups serviceAccessSecurityGroups;

        private List<Secret> credentials = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Environment model) {
            environmentId(model.environmentId);
            environmentState(model.environmentState);
            stateDetails(model.stateDetails);
            createdAt(model.createdAt);
            modifiedAt(model.modifiedAt);
            environmentArn(model.environmentArn);
            environmentName(model.environmentName);
            vpcId(model.vpcId);
            serviceAccessSubnetId(model.serviceAccessSubnetId);
            vcfVersion(model.vcfVersion);
            termsAccepted(model.termsAccepted);
            licenseInfo(model.licenseInfo);
            siteId(model.siteId);
            environmentStatus(model.environmentStatus);
            checks(model.checks);
            connectivityInfo(model.connectivityInfo);
            vcfHostnames(model.vcfHostnames);
            kmsKeyId(model.kmsKeyId);
            serviceAccessSecurityGroups(model.serviceAccessSecurityGroups);
            credentials(model.credentials);
        }

        public final String getEnvironmentId() {
            return environmentId;
        }

        public final void setEnvironmentId(String environmentId) {
            this.environmentId = environmentId;
        }

        @Override
        public final Builder environmentId(String environmentId) {
            this.environmentId = environmentId;
            return this;
        }

        public final String getEnvironmentState() {
            return environmentState;
        }

        public final void setEnvironmentState(String environmentState) {
            this.environmentState = environmentState;
        }

        @Override
        public final Builder environmentState(String environmentState) {
            this.environmentState = environmentState;
            return this;
        }

        @Override
        public final Builder environmentState(EnvironmentState environmentState) {
            this.environmentState(environmentState == null ? null : environmentState.toString());
            return this;
        }

        public final String getStateDetails() {
            return stateDetails;
        }

        public final void setStateDetails(String stateDetails) {
            this.stateDetails = stateDetails;
        }

        @Override
        public final Builder stateDetails(String stateDetails) {
            this.stateDetails = stateDetails;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getModifiedAt() {
            return modifiedAt;
        }

        public final void setModifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
        }

        @Override
        public final Builder modifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
            return this;
        }

        public final String getEnvironmentArn() {
            return environmentArn;
        }

        public final void setEnvironmentArn(String environmentArn) {
            this.environmentArn = environmentArn;
        }

        @Override
        public final Builder environmentArn(String environmentArn) {
            this.environmentArn = environmentArn;
            return this;
        }

        public final String getEnvironmentName() {
            return environmentName;
        }

        public final void setEnvironmentName(String environmentName) {
            this.environmentName = environmentName;
        }

        @Override
        public final Builder environmentName(String environmentName) {
            this.environmentName = environmentName;
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final String getServiceAccessSubnetId() {
            return serviceAccessSubnetId;
        }

        public final void setServiceAccessSubnetId(String serviceAccessSubnetId) {
            this.serviceAccessSubnetId = serviceAccessSubnetId;
        }

        @Override
        public final Builder serviceAccessSubnetId(String serviceAccessSubnetId) {
            this.serviceAccessSubnetId = serviceAccessSubnetId;
            return this;
        }

        public final String getVcfVersion() {
            return vcfVersion;
        }

        public final void setVcfVersion(String vcfVersion) {
            this.vcfVersion = vcfVersion;
        }

        @Override
        public final Builder vcfVersion(String vcfVersion) {
            this.vcfVersion = vcfVersion;
            return this;
        }

        @Override
        public final Builder vcfVersion(VcfVersion vcfVersion) {
            this.vcfVersion(vcfVersion == null ? null : vcfVersion.toString());
            return this;
        }

        public final Boolean getTermsAccepted() {
            return termsAccepted;
        }

        public final void setTermsAccepted(Boolean termsAccepted) {
            this.termsAccepted = termsAccepted;
        }

        @Override
        public final Builder termsAccepted(Boolean termsAccepted) {
            this.termsAccepted = termsAccepted;
            return this;
        }

        public final List<LicenseInfo.Builder> getLicenseInfo() {
            List<LicenseInfo.Builder> result = LicenseInfoListCopier.copyToBuilder(this.licenseInfo);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLicenseInfo(Collection<LicenseInfo.BuilderImpl> licenseInfo) {
            this.licenseInfo = LicenseInfoListCopier.copyFromBuilder(licenseInfo);
        }

        @Override
        public final Builder licenseInfo(Collection<LicenseInfo> licenseInfo) {
            this.licenseInfo = LicenseInfoListCopier.copy(licenseInfo);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder licenseInfo(LicenseInfo... licenseInfo) {
            licenseInfo(Arrays.asList(licenseInfo));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder licenseInfo(Consumer<LicenseInfo.Builder>... licenseInfo) {
            licenseInfo(Stream.of(licenseInfo).map(c -> LicenseInfo.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getSiteId() {
            return siteId;
        }

        public final void setSiteId(String siteId) {
            this.siteId = siteId;
        }

        @Override
        public final Builder siteId(String siteId) {
            this.siteId = siteId;
            return this;
        }

        public final String getEnvironmentStatus() {
            return environmentStatus;
        }

        public final void setEnvironmentStatus(String environmentStatus) {
            this.environmentStatus = environmentStatus;
        }

        @Override
        public final Builder environmentStatus(String environmentStatus) {
            this.environmentStatus = environmentStatus;
            return this;
        }

        @Override
        public final Builder environmentStatus(CheckResult environmentStatus) {
            this.environmentStatus(environmentStatus == null ? null : environmentStatus.toString());
            return this;
        }

        public final List<Check.Builder> getChecks() {
            List<Check.Builder> result = ChecksListCopier.copyToBuilder(this.checks);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setChecks(Collection<Check.BuilderImpl> checks) {
            this.checks = ChecksListCopier.copyFromBuilder(checks);
        }

        @Override
        public final Builder checks(Collection<Check> checks) {
            this.checks = ChecksListCopier.copy(checks);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder checks(Check... checks) {
            checks(Arrays.asList(checks));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder checks(Consumer<Check.Builder>... checks) {
            checks(Stream.of(checks).map(c -> Check.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final ConnectivityInfo.Builder getConnectivityInfo() {
            return connectivityInfo != null ? connectivityInfo.toBuilder() : null;
        }

        public final void setConnectivityInfo(ConnectivityInfo.BuilderImpl connectivityInfo) {
            this.connectivityInfo = connectivityInfo != null ? connectivityInfo.build() : null;
        }

        @Override
        public final Builder connectivityInfo(ConnectivityInfo connectivityInfo) {
            this.connectivityInfo = connectivityInfo;
            return this;
        }

        public final VcfHostnames.Builder getVcfHostnames() {
            return vcfHostnames != null ? vcfHostnames.toBuilder() : null;
        }

        public final void setVcfHostnames(VcfHostnames.BuilderImpl vcfHostnames) {
            this.vcfHostnames = vcfHostnames != null ? vcfHostnames.build() : null;
        }

        @Override
        public final Builder vcfHostnames(VcfHostnames vcfHostnames) {
            this.vcfHostnames = vcfHostnames;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final ServiceAccessSecurityGroups.Builder getServiceAccessSecurityGroups() {
            return serviceAccessSecurityGroups != null ? serviceAccessSecurityGroups.toBuilder() : null;
        }

        public final void setServiceAccessSecurityGroups(ServiceAccessSecurityGroups.BuilderImpl serviceAccessSecurityGroups) {
            this.serviceAccessSecurityGroups = serviceAccessSecurityGroups != null ? serviceAccessSecurityGroups.build() : null;
        }

        @Override
        public final Builder serviceAccessSecurityGroups(ServiceAccessSecurityGroups serviceAccessSecurityGroups) {
            this.serviceAccessSecurityGroups = serviceAccessSecurityGroups;
            return this;
        }

        public final List<Secret.Builder> getCredentials() {
            List<Secret.Builder> result = SecretListCopier.copyToBuilder(this.credentials);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCredentials(Collection<Secret.BuilderImpl> credentials) {
            this.credentials = SecretListCopier.copyFromBuilder(credentials);
        }

        @Override
        public final Builder credentials(Collection<Secret> credentials) {
            this.credentials = SecretListCopier.copy(credentials);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder credentials(Secret... credentials) {
            credentials(Arrays.asList(credentials));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder credentials(Consumer<Secret.Builder>... credentials) {
            credentials(Stream.of(credentials).map(c -> Secret.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Environment build() {
            return new Environment(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
