/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticloadbalancingv2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateLoadBalancerRequest extends ElasticLoadBalancingV2Request implements
        ToCopyableBuilder<CreateLoadBalancerRequest.Builder, CreateLoadBalancerRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CreateLoadBalancerRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<String>> SUBNETS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Subnets")
            .getter(getter(CreateLoadBalancerRequest::subnets))
            .setter(setter(Builder::subnets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subnets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<SubnetMapping>> SUBNET_MAPPINGS_FIELD = SdkField
            .<List<SubnetMapping>> builder(MarshallingType.LIST)
            .memberName("SubnetMappings")
            .getter(getter(CreateLoadBalancerRequest::subnetMappings))
            .setter(setter(Builder::subnetMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SubnetMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(SubnetMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SECURITY_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroups")
            .getter(getter(CreateLoadBalancerRequest::securityGroups))
            .setter(setter(Builder::securityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SCHEME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Scheme")
            .getter(getter(CreateLoadBalancerRequest::schemeAsString)).setter(setter(Builder::scheme))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Scheme").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateLoadBalancerRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(CreateLoadBalancerRequest::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> IP_ADDRESS_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IpAddressType").getter(getter(CreateLoadBalancerRequest::ipAddressTypeAsString))
            .setter(setter(Builder::ipAddressType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddressType").build()).build();

    private static final SdkField<String> CUSTOMER_OWNED_IPV4_POOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomerOwnedIpv4Pool").getter(getter(CreateLoadBalancerRequest::customerOwnedIpv4Pool))
            .setter(setter(Builder::customerOwnedIpv4Pool))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerOwnedIpv4Pool").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, SUBNETS_FIELD,
            SUBNET_MAPPINGS_FIELD, SECURITY_GROUPS_FIELD, SCHEME_FIELD, TAGS_FIELD, TYPE_FIELD, IP_ADDRESS_TYPE_FIELD,
            CUSTOMER_OWNED_IPV4_POOL_FIELD));

    private final String name;

    private final List<String> subnets;

    private final List<SubnetMapping> subnetMappings;

    private final List<String> securityGroups;

    private final String scheme;

    private final List<Tag> tags;

    private final String type;

    private final String ipAddressType;

    private final String customerOwnedIpv4Pool;

    private CreateLoadBalancerRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.subnets = builder.subnets;
        this.subnetMappings = builder.subnetMappings;
        this.securityGroups = builder.securityGroups;
        this.scheme = builder.scheme;
        this.tags = builder.tags;
        this.type = builder.type;
        this.ipAddressType = builder.ipAddressType;
        this.customerOwnedIpv4Pool = builder.customerOwnedIpv4Pool;
    }

    /**
     * <p>
     * The name of the load balancer.
     * </p>
     * <p>
     * This name must be unique per region per account, can have a maximum of 32 characters, must contain only
     * alphanumeric characters or hyphens, must not begin or end with a hyphen, and must not begin with "internal-".
     * </p>
     * 
     * @return The name of the load balancer.</p>
     *         <p>
     *         This name must be unique per region per account, can have a maximum of 32 characters, must contain only
     *         alphanumeric characters or hyphens, must not begin or end with a hyphen, and must not begin with
     *         "internal-".
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Subnets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSubnets() {
        return subnets != null && !(subnets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the public subnets. You can specify only one subnet per Availability Zone. You must specify either
     * subnets or subnet mappings, but not both. To specify an Elastic IP address, specify subnet mappings instead of
     * subnets.
     * </p>
     * <p>
     * [Application Load Balancers] You must specify subnets from at least two Availability Zones.
     * </p>
     * <p>
     * [Application Load Balancers on Outposts] You must specify one Outpost subnet.
     * </p>
     * <p>
     * [Application Load Balancers on Local Zones] You can specify subnets from one or more Local Zones.
     * </p>
     * <p>
     * [Network Load Balancers] You can specify subnets from one or more Availability Zones.
     * </p>
     * <p>
     * [Gateway Load Balancers] You can specify subnets from one or more Availability Zones.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubnets} method.
     * </p>
     * 
     * @return The IDs of the public subnets. You can specify only one subnet per Availability Zone. You must specify
     *         either subnets or subnet mappings, but not both. To specify an Elastic IP address, specify subnet
     *         mappings instead of subnets.</p>
     *         <p>
     *         [Application Load Balancers] You must specify subnets from at least two Availability Zones.
     *         </p>
     *         <p>
     *         [Application Load Balancers on Outposts] You must specify one Outpost subnet.
     *         </p>
     *         <p>
     *         [Application Load Balancers on Local Zones] You can specify subnets from one or more Local Zones.
     *         </p>
     *         <p>
     *         [Network Load Balancers] You can specify subnets from one or more Availability Zones.
     *         </p>
     *         <p>
     *         [Gateway Load Balancers] You can specify subnets from one or more Availability Zones.
     */
    public final List<String> subnets() {
        return subnets;
    }

    /**
     * For responses, this returns true if the service returned a value for the SubnetMappings property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSubnetMappings() {
        return subnetMappings != null && !(subnetMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the public subnets. You can specify only one subnet per Availability Zone. You must specify either
     * subnets or subnet mappings, but not both.
     * </p>
     * <p>
     * [Application Load Balancers] You must specify subnets from at least two Availability Zones. You cannot specify
     * Elastic IP addresses for your subnets.
     * </p>
     * <p>
     * [Application Load Balancers on Outposts] You must specify one Outpost subnet.
     * </p>
     * <p>
     * [Application Load Balancers on Local Zones] You can specify subnets from one or more Local Zones.
     * </p>
     * <p>
     * [Network Load Balancers] You can specify subnets from one or more Availability Zones. You can specify one Elastic
     * IP address per subnet if you need static IP addresses for your internet-facing load balancer. For internal load
     * balancers, you can specify one private IP address per subnet from the IPv4 range of the subnet. For
     * internet-facing load balancer, you can specify one IPv6 address per subnet.
     * </p>
     * <p>
     * [Gateway Load Balancers] You can specify subnets from one or more Availability Zones. You cannot specify Elastic
     * IP addresses for your subnets.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubnetMappings} method.
     * </p>
     * 
     * @return The IDs of the public subnets. You can specify only one subnet per Availability Zone. You must specify
     *         either subnets or subnet mappings, but not both.</p>
     *         <p>
     *         [Application Load Balancers] You must specify subnets from at least two Availability Zones. You cannot
     *         specify Elastic IP addresses for your subnets.
     *         </p>
     *         <p>
     *         [Application Load Balancers on Outposts] You must specify one Outpost subnet.
     *         </p>
     *         <p>
     *         [Application Load Balancers on Local Zones] You can specify subnets from one or more Local Zones.
     *         </p>
     *         <p>
     *         [Network Load Balancers] You can specify subnets from one or more Availability Zones. You can specify one
     *         Elastic IP address per subnet if you need static IP addresses for your internet-facing load balancer. For
     *         internal load balancers, you can specify one private IP address per subnet from the IPv4 range of the
     *         subnet. For internet-facing load balancer, you can specify one IPv6 address per subnet.
     *         </p>
     *         <p>
     *         [Gateway Load Balancers] You can specify subnets from one or more Availability Zones. You cannot specify
     *         Elastic IP addresses for your subnets.
     */
    public final List<SubnetMapping> subnetMappings() {
        return subnetMappings;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroups property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroups() {
        return securityGroups != null && !(securityGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * [Application Load Balancers] The IDs of the security groups for the load balancer.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroups} method.
     * </p>
     * 
     * @return [Application Load Balancers] The IDs of the security groups for the load balancer.
     */
    public final List<String> securityGroups() {
        return securityGroups;
    }

    /**
     * <p>
     * The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an Internet-facing load
     * balancer is publicly resolvable to the public IP addresses of the nodes. Therefore, Internet-facing load
     * balancers can route requests from clients over the internet.
     * </p>
     * <p>
     * The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal load balancer
     * is publicly resolvable to the private IP addresses of the nodes. Therefore, internal load balancers can route
     * requests only from clients with access to the VPC for the load balancer.
     * </p>
     * <p>
     * The default is an Internet-facing load balancer.
     * </p>
     * <p>
     * You cannot specify a scheme for a Gateway Load Balancer.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scheme} will
     * return {@link LoadBalancerSchemeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #schemeAsString}.
     * </p>
     * 
     * @return The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an
     *         Internet-facing load balancer is publicly resolvable to the public IP addresses of the nodes. Therefore,
     *         Internet-facing load balancers can route requests from clients over the internet.</p>
     *         <p>
     *         The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal load
     *         balancer is publicly resolvable to the private IP addresses of the nodes. Therefore, internal load
     *         balancers can route requests only from clients with access to the VPC for the load balancer.
     *         </p>
     *         <p>
     *         The default is an Internet-facing load balancer.
     *         </p>
     *         <p>
     *         You cannot specify a scheme for a Gateway Load Balancer.
     * @see LoadBalancerSchemeEnum
     */
    public final LoadBalancerSchemeEnum scheme() {
        return LoadBalancerSchemeEnum.fromValue(scheme);
    }

    /**
     * <p>
     * The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an Internet-facing load
     * balancer is publicly resolvable to the public IP addresses of the nodes. Therefore, Internet-facing load
     * balancers can route requests from clients over the internet.
     * </p>
     * <p>
     * The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal load balancer
     * is publicly resolvable to the private IP addresses of the nodes. Therefore, internal load balancers can route
     * requests only from clients with access to the VPC for the load balancer.
     * </p>
     * <p>
     * The default is an Internet-facing load balancer.
     * </p>
     * <p>
     * You cannot specify a scheme for a Gateway Load Balancer.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scheme} will
     * return {@link LoadBalancerSchemeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #schemeAsString}.
     * </p>
     * 
     * @return The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an
     *         Internet-facing load balancer is publicly resolvable to the public IP addresses of the nodes. Therefore,
     *         Internet-facing load balancers can route requests from clients over the internet.</p>
     *         <p>
     *         The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal load
     *         balancer is publicly resolvable to the private IP addresses of the nodes. Therefore, internal load
     *         balancers can route requests only from clients with access to the VPC for the load balancer.
     *         </p>
     *         <p>
     *         The default is an Internet-facing load balancer.
     *         </p>
     *         <p>
     *         You cannot specify a scheme for a Gateway Load Balancer.
     * @see LoadBalancerSchemeEnum
     */
    public final String schemeAsString() {
        return scheme;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to assign to the load balancer.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags to assign to the load balancer.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The type of load balancer. The default is <code>application</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link LoadBalancerTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of load balancer. The default is <code>application</code>.
     * @see LoadBalancerTypeEnum
     */
    public final LoadBalancerTypeEnum type() {
        return LoadBalancerTypeEnum.fromValue(type);
    }

    /**
     * <p>
     * The type of load balancer. The default is <code>application</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link LoadBalancerTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of load balancer. The default is <code>application</code>.
     * @see LoadBalancerTypeEnum
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The type of IP addresses used by the subnets for your load balancer. The possible values are <code>ipv4</code>
     * (for IPv4 addresses) and <code>dualstack</code> (for IPv4 and IPv6 addresses).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link IpAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return The type of IP addresses used by the subnets for your load balancer. The possible values are
     *         <code>ipv4</code> (for IPv4 addresses) and <code>dualstack</code> (for IPv4 and IPv6 addresses).
     * @see IpAddressType
     */
    public final IpAddressType ipAddressType() {
        return IpAddressType.fromValue(ipAddressType);
    }

    /**
     * <p>
     * The type of IP addresses used by the subnets for your load balancer. The possible values are <code>ipv4</code>
     * (for IPv4 addresses) and <code>dualstack</code> (for IPv4 and IPv6 addresses).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link IpAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return The type of IP addresses used by the subnets for your load balancer. The possible values are
     *         <code>ipv4</code> (for IPv4 addresses) and <code>dualstack</code> (for IPv4 and IPv6 addresses).
     * @see IpAddressType
     */
    public final String ipAddressTypeAsString() {
        return ipAddressType;
    }

    /**
     * <p>
     * [Application Load Balancers on Outposts] The ID of the customer-owned address pool (CoIP pool).
     * </p>
     * 
     * @return [Application Load Balancers on Outposts] The ID of the customer-owned address pool (CoIP pool).
     */
    public final String customerOwnedIpv4Pool() {
        return customerOwnedIpv4Pool;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnets() ? subnets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetMappings() ? subnetMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroups() ? securityGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(schemeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddressTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(customerOwnedIpv4Pool());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateLoadBalancerRequest)) {
            return false;
        }
        CreateLoadBalancerRequest other = (CreateLoadBalancerRequest) obj;
        return Objects.equals(name(), other.name()) && hasSubnets() == other.hasSubnets()
                && Objects.equals(subnets(), other.subnets()) && hasSubnetMappings() == other.hasSubnetMappings()
                && Objects.equals(subnetMappings(), other.subnetMappings()) && hasSecurityGroups() == other.hasSecurityGroups()
                && Objects.equals(securityGroups(), other.securityGroups())
                && Objects.equals(schemeAsString(), other.schemeAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(ipAddressTypeAsString(), other.ipAddressTypeAsString())
                && Objects.equals(customerOwnedIpv4Pool(), other.customerOwnedIpv4Pool());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateLoadBalancerRequest").add("Name", name()).add("Subnets", hasSubnets() ? subnets() : null)
                .add("SubnetMappings", hasSubnetMappings() ? subnetMappings() : null)
                .add("SecurityGroups", hasSecurityGroups() ? securityGroups() : null).add("Scheme", schemeAsString())
                .add("Tags", hasTags() ? tags() : null).add("Type", typeAsString()).add("IpAddressType", ipAddressTypeAsString())
                .add("CustomerOwnedIpv4Pool", customerOwnedIpv4Pool()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Subnets":
            return Optional.ofNullable(clazz.cast(subnets()));
        case "SubnetMappings":
            return Optional.ofNullable(clazz.cast(subnetMappings()));
        case "SecurityGroups":
            return Optional.ofNullable(clazz.cast(securityGroups()));
        case "Scheme":
            return Optional.ofNullable(clazz.cast(schemeAsString()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "IpAddressType":
            return Optional.ofNullable(clazz.cast(ipAddressTypeAsString()));
        case "CustomerOwnedIpv4Pool":
            return Optional.ofNullable(clazz.cast(customerOwnedIpv4Pool()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateLoadBalancerRequest, T> g) {
        return obj -> g.apply((CreateLoadBalancerRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ElasticLoadBalancingV2Request.Builder, SdkPojo,
            CopyableBuilder<Builder, CreateLoadBalancerRequest> {
        /**
         * <p>
         * The name of the load balancer.
         * </p>
         * <p>
         * This name must be unique per region per account, can have a maximum of 32 characters, must contain only
         * alphanumeric characters or hyphens, must not begin or end with a hyphen, and must not begin with "internal-".
         * </p>
         * 
         * @param name
         *        The name of the load balancer.</p>
         *        <p>
         *        This name must be unique per region per account, can have a maximum of 32 characters, must contain
         *        only alphanumeric characters or hyphens, must not begin or end with a hyphen, and must not begin with
         *        "internal-".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The IDs of the public subnets. You can specify only one subnet per Availability Zone. You must specify either
         * subnets or subnet mappings, but not both. To specify an Elastic IP address, specify subnet mappings instead
         * of subnets.
         * </p>
         * <p>
         * [Application Load Balancers] You must specify subnets from at least two Availability Zones.
         * </p>
         * <p>
         * [Application Load Balancers on Outposts] You must specify one Outpost subnet.
         * </p>
         * <p>
         * [Application Load Balancers on Local Zones] You can specify subnets from one or more Local Zones.
         * </p>
         * <p>
         * [Network Load Balancers] You can specify subnets from one or more Availability Zones.
         * </p>
         * <p>
         * [Gateway Load Balancers] You can specify subnets from one or more Availability Zones.
         * </p>
         * 
         * @param subnets
         *        The IDs of the public subnets. You can specify only one subnet per Availability Zone. You must specify
         *        either subnets or subnet mappings, but not both. To specify an Elastic IP address, specify subnet
         *        mappings instead of subnets.</p>
         *        <p>
         *        [Application Load Balancers] You must specify subnets from at least two Availability Zones.
         *        </p>
         *        <p>
         *        [Application Load Balancers on Outposts] You must specify one Outpost subnet.
         *        </p>
         *        <p>
         *        [Application Load Balancers on Local Zones] You can specify subnets from one or more Local Zones.
         *        </p>
         *        <p>
         *        [Network Load Balancers] You can specify subnets from one or more Availability Zones.
         *        </p>
         *        <p>
         *        [Gateway Load Balancers] You can specify subnets from one or more Availability Zones.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(Collection<String> subnets);

        /**
         * <p>
         * The IDs of the public subnets. You can specify only one subnet per Availability Zone. You must specify either
         * subnets or subnet mappings, but not both. To specify an Elastic IP address, specify subnet mappings instead
         * of subnets.
         * </p>
         * <p>
         * [Application Load Balancers] You must specify subnets from at least two Availability Zones.
         * </p>
         * <p>
         * [Application Load Balancers on Outposts] You must specify one Outpost subnet.
         * </p>
         * <p>
         * [Application Load Balancers on Local Zones] You can specify subnets from one or more Local Zones.
         * </p>
         * <p>
         * [Network Load Balancers] You can specify subnets from one or more Availability Zones.
         * </p>
         * <p>
         * [Gateway Load Balancers] You can specify subnets from one or more Availability Zones.
         * </p>
         * 
         * @param subnets
         *        The IDs of the public subnets. You can specify only one subnet per Availability Zone. You must specify
         *        either subnets or subnet mappings, but not both. To specify an Elastic IP address, specify subnet
         *        mappings instead of subnets.</p>
         *        <p>
         *        [Application Load Balancers] You must specify subnets from at least two Availability Zones.
         *        </p>
         *        <p>
         *        [Application Load Balancers on Outposts] You must specify one Outpost subnet.
         *        </p>
         *        <p>
         *        [Application Load Balancers on Local Zones] You can specify subnets from one or more Local Zones.
         *        </p>
         *        <p>
         *        [Network Load Balancers] You can specify subnets from one or more Availability Zones.
         *        </p>
         *        <p>
         *        [Gateway Load Balancers] You can specify subnets from one or more Availability Zones.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(String... subnets);

        /**
         * <p>
         * The IDs of the public subnets. You can specify only one subnet per Availability Zone. You must specify either
         * subnets or subnet mappings, but not both.
         * </p>
         * <p>
         * [Application Load Balancers] You must specify subnets from at least two Availability Zones. You cannot
         * specify Elastic IP addresses for your subnets.
         * </p>
         * <p>
         * [Application Load Balancers on Outposts] You must specify one Outpost subnet.
         * </p>
         * <p>
         * [Application Load Balancers on Local Zones] You can specify subnets from one or more Local Zones.
         * </p>
         * <p>
         * [Network Load Balancers] You can specify subnets from one or more Availability Zones. You can specify one
         * Elastic IP address per subnet if you need static IP addresses for your internet-facing load balancer. For
         * internal load balancers, you can specify one private IP address per subnet from the IPv4 range of the subnet.
         * For internet-facing load balancer, you can specify one IPv6 address per subnet.
         * </p>
         * <p>
         * [Gateway Load Balancers] You can specify subnets from one or more Availability Zones. You cannot specify
         * Elastic IP addresses for your subnets.
         * </p>
         * 
         * @param subnetMappings
         *        The IDs of the public subnets. You can specify only one subnet per Availability Zone. You must specify
         *        either subnets or subnet mappings, but not both.</p>
         *        <p>
         *        [Application Load Balancers] You must specify subnets from at least two Availability Zones. You cannot
         *        specify Elastic IP addresses for your subnets.
         *        </p>
         *        <p>
         *        [Application Load Balancers on Outposts] You must specify one Outpost subnet.
         *        </p>
         *        <p>
         *        [Application Load Balancers on Local Zones] You can specify subnets from one or more Local Zones.
         *        </p>
         *        <p>
         *        [Network Load Balancers] You can specify subnets from one or more Availability Zones. You can specify
         *        one Elastic IP address per subnet if you need static IP addresses for your internet-facing load
         *        balancer. For internal load balancers, you can specify one private IP address per subnet from the IPv4
         *        range of the subnet. For internet-facing load balancer, you can specify one IPv6 address per subnet.
         *        </p>
         *        <p>
         *        [Gateway Load Balancers] You can specify subnets from one or more Availability Zones. You cannot
         *        specify Elastic IP addresses for your subnets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetMappings(Collection<SubnetMapping> subnetMappings);

        /**
         * <p>
         * The IDs of the public subnets. You can specify only one subnet per Availability Zone. You must specify either
         * subnets or subnet mappings, but not both.
         * </p>
         * <p>
         * [Application Load Balancers] You must specify subnets from at least two Availability Zones. You cannot
         * specify Elastic IP addresses for your subnets.
         * </p>
         * <p>
         * [Application Load Balancers on Outposts] You must specify one Outpost subnet.
         * </p>
         * <p>
         * [Application Load Balancers on Local Zones] You can specify subnets from one or more Local Zones.
         * </p>
         * <p>
         * [Network Load Balancers] You can specify subnets from one or more Availability Zones. You can specify one
         * Elastic IP address per subnet if you need static IP addresses for your internet-facing load balancer. For
         * internal load balancers, you can specify one private IP address per subnet from the IPv4 range of the subnet.
         * For internet-facing load balancer, you can specify one IPv6 address per subnet.
         * </p>
         * <p>
         * [Gateway Load Balancers] You can specify subnets from one or more Availability Zones. You cannot specify
         * Elastic IP addresses for your subnets.
         * </p>
         * 
         * @param subnetMappings
         *        The IDs of the public subnets. You can specify only one subnet per Availability Zone. You must specify
         *        either subnets or subnet mappings, but not both.</p>
         *        <p>
         *        [Application Load Balancers] You must specify subnets from at least two Availability Zones. You cannot
         *        specify Elastic IP addresses for your subnets.
         *        </p>
         *        <p>
         *        [Application Load Balancers on Outposts] You must specify one Outpost subnet.
         *        </p>
         *        <p>
         *        [Application Load Balancers on Local Zones] You can specify subnets from one or more Local Zones.
         *        </p>
         *        <p>
         *        [Network Load Balancers] You can specify subnets from one or more Availability Zones. You can specify
         *        one Elastic IP address per subnet if you need static IP addresses for your internet-facing load
         *        balancer. For internal load balancers, you can specify one private IP address per subnet from the IPv4
         *        range of the subnet. For internet-facing load balancer, you can specify one IPv6 address per subnet.
         *        </p>
         *        <p>
         *        [Gateway Load Balancers] You can specify subnets from one or more Availability Zones. You cannot
         *        specify Elastic IP addresses for your subnets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetMappings(SubnetMapping... subnetMappings);

        /**
         * <p>
         * The IDs of the public subnets. You can specify only one subnet per Availability Zone. You must specify either
         * subnets or subnet mappings, but not both.
         * </p>
         * <p>
         * [Application Load Balancers] You must specify subnets from at least two Availability Zones. You cannot
         * specify Elastic IP addresses for your subnets.
         * </p>
         * <p>
         * [Application Load Balancers on Outposts] You must specify one Outpost subnet.
         * </p>
         * <p>
         * [Application Load Balancers on Local Zones] You can specify subnets from one or more Local Zones.
         * </p>
         * <p>
         * [Network Load Balancers] You can specify subnets from one or more Availability Zones. You can specify one
         * Elastic IP address per subnet if you need static IP addresses for your internet-facing load balancer. For
         * internal load balancers, you can specify one private IP address per subnet from the IPv4 range of the subnet.
         * For internet-facing load balancer, you can specify one IPv6 address per subnet.
         * </p>
         * <p>
         * [Gateway Load Balancers] You can specify subnets from one or more Availability Zones. You cannot specify
         * Elastic IP addresses for your subnets.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.elasticloadbalancingv2.model.SubnetMapping.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.elasticloadbalancingv2.model.SubnetMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.elasticloadbalancingv2.model.SubnetMapping.Builder#build()} is called
         * immediately and its result is passed to {@link #subnetMappings(List<SubnetMapping>)}.
         * 
         * @param subnetMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.elasticloadbalancingv2.model.SubnetMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #subnetMappings(java.util.Collection<SubnetMapping>)
         */
        Builder subnetMappings(Consumer<SubnetMapping.Builder>... subnetMappings);

        /**
         * <p>
         * [Application Load Balancers] The IDs of the security groups for the load balancer.
         * </p>
         * 
         * @param securityGroups
         *        [Application Load Balancers] The IDs of the security groups for the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(Collection<String> securityGroups);

        /**
         * <p>
         * [Application Load Balancers] The IDs of the security groups for the load balancer.
         * </p>
         * 
         * @param securityGroups
         *        [Application Load Balancers] The IDs of the security groups for the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(String... securityGroups);

        /**
         * <p>
         * The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an Internet-facing
         * load balancer is publicly resolvable to the public IP addresses of the nodes. Therefore, Internet-facing load
         * balancers can route requests from clients over the internet.
         * </p>
         * <p>
         * The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal load
         * balancer is publicly resolvable to the private IP addresses of the nodes. Therefore, internal load balancers
         * can route requests only from clients with access to the VPC for the load balancer.
         * </p>
         * <p>
         * The default is an Internet-facing load balancer.
         * </p>
         * <p>
         * You cannot specify a scheme for a Gateway Load Balancer.
         * </p>
         * 
         * @param scheme
         *        The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an
         *        Internet-facing load balancer is publicly resolvable to the public IP addresses of the nodes.
         *        Therefore, Internet-facing load balancers can route requests from clients over the internet.</p>
         *        <p>
         *        The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal
         *        load balancer is publicly resolvable to the private IP addresses of the nodes. Therefore, internal
         *        load balancers can route requests only from clients with access to the VPC for the load balancer.
         *        </p>
         *        <p>
         *        The default is an Internet-facing load balancer.
         *        </p>
         *        <p>
         *        You cannot specify a scheme for a Gateway Load Balancer.
         * @see LoadBalancerSchemeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LoadBalancerSchemeEnum
         */
        Builder scheme(String scheme);

        /**
         * <p>
         * The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an Internet-facing
         * load balancer is publicly resolvable to the public IP addresses of the nodes. Therefore, Internet-facing load
         * balancers can route requests from clients over the internet.
         * </p>
         * <p>
         * The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal load
         * balancer is publicly resolvable to the private IP addresses of the nodes. Therefore, internal load balancers
         * can route requests only from clients with access to the VPC for the load balancer.
         * </p>
         * <p>
         * The default is an Internet-facing load balancer.
         * </p>
         * <p>
         * You cannot specify a scheme for a Gateway Load Balancer.
         * </p>
         * 
         * @param scheme
         *        The nodes of an Internet-facing load balancer have public IP addresses. The DNS name of an
         *        Internet-facing load balancer is publicly resolvable to the public IP addresses of the nodes.
         *        Therefore, Internet-facing load balancers can route requests from clients over the internet.</p>
         *        <p>
         *        The nodes of an internal load balancer have only private IP addresses. The DNS name of an internal
         *        load balancer is publicly resolvable to the private IP addresses of the nodes. Therefore, internal
         *        load balancers can route requests only from clients with access to the VPC for the load balancer.
         *        </p>
         *        <p>
         *        The default is an Internet-facing load balancer.
         *        </p>
         *        <p>
         *        You cannot specify a scheme for a Gateway Load Balancer.
         * @see LoadBalancerSchemeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LoadBalancerSchemeEnum
         */
        Builder scheme(LoadBalancerSchemeEnum scheme);

        /**
         * <p>
         * The tags to assign to the load balancer.
         * </p>
         * 
         * @param tags
         *        The tags to assign to the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags to assign to the load balancer.
         * </p>
         * 
         * @param tags
         *        The tags to assign to the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags to assign to the load balancer.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.elasticloadbalancingv2.model.Tag.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.elasticloadbalancingv2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.elasticloadbalancingv2.model.Tag.Builder#build()} is called
         * immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.elasticloadbalancingv2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The type of load balancer. The default is <code>application</code>.
         * </p>
         * 
         * @param type
         *        The type of load balancer. The default is <code>application</code>.
         * @see LoadBalancerTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LoadBalancerTypeEnum
         */
        Builder type(String type);

        /**
         * <p>
         * The type of load balancer. The default is <code>application</code>.
         * </p>
         * 
         * @param type
         *        The type of load balancer. The default is <code>application</code>.
         * @see LoadBalancerTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LoadBalancerTypeEnum
         */
        Builder type(LoadBalancerTypeEnum type);

        /**
         * <p>
         * The type of IP addresses used by the subnets for your load balancer. The possible values are
         * <code>ipv4</code> (for IPv4 addresses) and <code>dualstack</code> (for IPv4 and IPv6 addresses).
         * </p>
         * 
         * @param ipAddressType
         *        The type of IP addresses used by the subnets for your load balancer. The possible values are
         *        <code>ipv4</code> (for IPv4 addresses) and <code>dualstack</code> (for IPv4 and IPv6 addresses).
         * @see IpAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpAddressType
         */
        Builder ipAddressType(String ipAddressType);

        /**
         * <p>
         * The type of IP addresses used by the subnets for your load balancer. The possible values are
         * <code>ipv4</code> (for IPv4 addresses) and <code>dualstack</code> (for IPv4 and IPv6 addresses).
         * </p>
         * 
         * @param ipAddressType
         *        The type of IP addresses used by the subnets for your load balancer. The possible values are
         *        <code>ipv4</code> (for IPv4 addresses) and <code>dualstack</code> (for IPv4 and IPv6 addresses).
         * @see IpAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpAddressType
         */
        Builder ipAddressType(IpAddressType ipAddressType);

        /**
         * <p>
         * [Application Load Balancers on Outposts] The ID of the customer-owned address pool (CoIP pool).
         * </p>
         * 
         * @param customerOwnedIpv4Pool
         *        [Application Load Balancers on Outposts] The ID of the customer-owned address pool (CoIP pool).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerOwnedIpv4Pool(String customerOwnedIpv4Pool);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ElasticLoadBalancingV2Request.BuilderImpl implements Builder {
        private String name;

        private List<String> subnets = DefaultSdkAutoConstructList.getInstance();

        private List<SubnetMapping> subnetMappings = DefaultSdkAutoConstructList.getInstance();

        private List<String> securityGroups = DefaultSdkAutoConstructList.getInstance();

        private String scheme;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String type;

        private String ipAddressType;

        private String customerOwnedIpv4Pool;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateLoadBalancerRequest model) {
            super(model);
            name(model.name);
            subnets(model.subnets);
            subnetMappings(model.subnetMappings);
            securityGroups(model.securityGroups);
            scheme(model.scheme);
            tags(model.tags);
            type(model.type);
            ipAddressType(model.ipAddressType);
            customerOwnedIpv4Pool(model.customerOwnedIpv4Pool);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Collection<String> getSubnets() {
            if (subnets instanceof SdkAutoConstructList) {
                return null;
            }
            return subnets;
        }

        public final void setSubnets(Collection<String> subnets) {
            this.subnets = SubnetsCopier.copy(subnets);
        }

        @Override
        public final Builder subnets(Collection<String> subnets) {
            this.subnets = SubnetsCopier.copy(subnets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnets(String... subnets) {
            subnets(Arrays.asList(subnets));
            return this;
        }

        public final List<SubnetMapping.Builder> getSubnetMappings() {
            List<SubnetMapping.Builder> result = SubnetMappingsCopier.copyToBuilder(this.subnetMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSubnetMappings(Collection<SubnetMapping.BuilderImpl> subnetMappings) {
            this.subnetMappings = SubnetMappingsCopier.copyFromBuilder(subnetMappings);
        }

        @Override
        public final Builder subnetMappings(Collection<SubnetMapping> subnetMappings) {
            this.subnetMappings = SubnetMappingsCopier.copy(subnetMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetMappings(SubnetMapping... subnetMappings) {
            subnetMappings(Arrays.asList(subnetMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetMappings(Consumer<SubnetMapping.Builder>... subnetMappings) {
            subnetMappings(Stream.of(subnetMappings).map(c -> SubnetMapping.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getSecurityGroups() {
            if (securityGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroups;
        }

        public final void setSecurityGroups(Collection<String> securityGroups) {
            this.securityGroups = SecurityGroupsCopier.copy(securityGroups);
        }

        @Override
        public final Builder securityGroups(Collection<String> securityGroups) {
            this.securityGroups = SecurityGroupsCopier.copy(securityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroups(String... securityGroups) {
            securityGroups(Arrays.asList(securityGroups));
            return this;
        }

        public final String getScheme() {
            return scheme;
        }

        public final void setScheme(String scheme) {
            this.scheme = scheme;
        }

        @Override
        public final Builder scheme(String scheme) {
            this.scheme = scheme;
            return this;
        }

        @Override
        public final Builder scheme(LoadBalancerSchemeEnum scheme) {
            this.scheme(scheme == null ? null : scheme.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(LoadBalancerTypeEnum type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getIpAddressType() {
            return ipAddressType;
        }

        public final void setIpAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
        }

        @Override
        public final Builder ipAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
            return this;
        }

        @Override
        public final Builder ipAddressType(IpAddressType ipAddressType) {
            this.ipAddressType(ipAddressType == null ? null : ipAddressType.toString());
            return this;
        }

        public final String getCustomerOwnedIpv4Pool() {
            return customerOwnedIpv4Pool;
        }

        public final void setCustomerOwnedIpv4Pool(String customerOwnedIpv4Pool) {
            this.customerOwnedIpv4Pool = customerOwnedIpv4Pool;
        }

        @Override
        public final Builder customerOwnedIpv4Pool(String customerOwnedIpv4Pool) {
            this.customerOwnedIpv4Pool = customerOwnedIpv4Pool;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateLoadBalancerRequest build() {
            return new CreateLoadBalancerRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
