/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticloadbalancing.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the parameters for DescribeInstanceHealth.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribeInstanceHealthRequest extends ElasticLoadBalancingRequest implements
        ToCopyableBuilder<DescribeInstanceHealthRequest.Builder, DescribeInstanceHealthRequest> {
    private static final SdkField<String> LOAD_BALANCER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeInstanceHealthRequest::loadBalancerName)).setter(setter(Builder::loadBalancerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoadBalancerName").build()).build();

    private static final SdkField<List<Instance>> INSTANCES_FIELD = SdkField
            .<List<Instance>> builder(MarshallingType.LIST)
            .getter(getter(DescribeInstanceHealthRequest::instances))
            .setter(setter(Builder::instances))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Instances").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Instance> builder(MarshallingType.SDK_POJO)
                                            .constructor(Instance::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOAD_BALANCER_NAME_FIELD,
            INSTANCES_FIELD));

    private final String loadBalancerName;

    private final List<Instance> instances;

    private DescribeInstanceHealthRequest(BuilderImpl builder) {
        super(builder);
        this.loadBalancerName = builder.loadBalancerName;
        this.instances = builder.instances;
    }

    /**
     * <p>
     * The name of the load balancer.
     * </p>
     * 
     * @return The name of the load balancer.
     */
    public String loadBalancerName() {
        return loadBalancerName;
    }

    /**
     * Returns true if the Instances property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasInstances() {
        return instances != null && !(instances instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the instances.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasInstances()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The IDs of the instances.
     */
    public List<Instance> instances() {
        return instances;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(loadBalancerName());
        hashCode = 31 * hashCode + Objects.hashCode(instances());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribeInstanceHealthRequest)) {
            return false;
        }
        DescribeInstanceHealthRequest other = (DescribeInstanceHealthRequest) obj;
        return Objects.equals(loadBalancerName(), other.loadBalancerName()) && Objects.equals(instances(), other.instances());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DescribeInstanceHealthRequest").add("LoadBalancerName", loadBalancerName())
                .add("Instances", instances()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LoadBalancerName":
            return Optional.ofNullable(clazz.cast(loadBalancerName()));
        case "Instances":
            return Optional.ofNullable(clazz.cast(instances()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DescribeInstanceHealthRequest, T> g) {
        return obj -> g.apply((DescribeInstanceHealthRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ElasticLoadBalancingRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, DescribeInstanceHealthRequest> {
        /**
         * <p>
         * The name of the load balancer.
         * </p>
         * 
         * @param loadBalancerName
         *        The name of the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadBalancerName(String loadBalancerName);

        /**
         * <p>
         * The IDs of the instances.
         * </p>
         * 
         * @param instances
         *        The IDs of the instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instances(Collection<Instance> instances);

        /**
         * <p>
         * The IDs of the instances.
         * </p>
         * 
         * @param instances
         *        The IDs of the instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instances(Instance... instances);

        /**
         * <p>
         * The IDs of the instances.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Instance>.Builder} avoiding the need to
         * create one manually via {@link List<Instance>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Instance>.Builder#build()} is called immediately and its
         * result is passed to {@link #instances(List<Instance>)}.
         * 
         * @param instances
         *        a consumer that will call methods on {@link List<Instance>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instances(List<Instance>)
         */
        Builder instances(Consumer<Instance.Builder>... instances);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ElasticLoadBalancingRequest.BuilderImpl implements Builder {
        private String loadBalancerName;

        private List<Instance> instances = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DescribeInstanceHealthRequest model) {
            super(model);
            loadBalancerName(model.loadBalancerName);
            instances(model.instances);
        }

        public final String getLoadBalancerName() {
            return loadBalancerName;
        }

        @Override
        public final Builder loadBalancerName(String loadBalancerName) {
            this.loadBalancerName = loadBalancerName;
            return this;
        }

        public final void setLoadBalancerName(String loadBalancerName) {
            this.loadBalancerName = loadBalancerName;
        }

        public final Collection<Instance.Builder> getInstances() {
            return instances != null ? instances.stream().map(Instance::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder instances(Collection<Instance> instances) {
            this.instances = InstancesCopier.copy(instances);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instances(Instance... instances) {
            instances(Arrays.asList(instances));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instances(Consumer<Instance.Builder>... instances) {
            instances(Stream.of(instances).map(c -> Instance.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setInstances(Collection<Instance.BuilderImpl> instances) {
            this.instances = InstancesCopier.copyFromBuilder(instances);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public DescribeInstanceHealthRequest build() {
            return new DescribeInstanceHealthRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
